/*
Copyright (C) 2003, Nik Reiman - nik@aboleo.net

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

/****
  Stops either a particular UBS daemon, or stops all known running daemons.  This will NOT clean up after the daemons; they should do that on their own.
****/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <signal.h>
#include <dirent.h>
#include <sys/types.h>
#include <errno.h>

#include "ubs-main.h"

/*+
  Stop a daemon

  int cmd_stop Returns OK on success, NO_DIR if bin/ can't be opened, or otherwise whatever sub_stop_proc() returns.

  int cmdc Number of arguments

  char **cmdv Argument vector
+*/
int cmd_stop(int cmdc, char **cmdv) {
 DIR *dp;
 struct dirent *ds;
 int ch, format = ENGLISH;

 while((ch = getopt(cmdc, cmdv, "hwces")) != -1) {
  switch(ch) {
   case 'e':
    format = ENGLISH;
    break;
   case 'c':
    format = CSV;
    break;
   case 'w':
    format = HTML;
    break;
   case 's':
    format = SILENT;
    break;
   case 'h':
    sub_help_cmd("stop");
    return OK;
    break;
   default:
    sub_help_cmd("stop");
    return OK;
    break;
  }
 }

 cmdc -= optind;
 cmdv += optind;

 if(cmdc == 0 || !strcasecmp(cmdv[0], "all")) {
  // Stop all.  To do this, we basically find all the files
  // in the bin directory that match some form of ubs-*, and kill
  // each one of those.  Kind of weird.  But it works, so shut up.
  if((dp = opendir("bin")) == NULL) {
   printf("Unable to open bin/ directory\n");
   return NO_DIR;
  }

  while((ds = readdir(dp)) != NULL) {
   if(!strncmp(ds->d_name, "ubs-", 4)) {
    sub_stop_proc(ds->d_name, format);
   }
  }
  closedir(dp);
 }
 // just stop a process as specified on the command line
 else {
  return sub_stop_proc(cmdv[0], format);
 }
 
 return OK;
}

/*+
  Start up an individual daemon by name, where name is "PREFIX/bin/NAME"

  int sub_stop_proc Returns OK on success, NO_FILE if the .pid file cannot be read, NO_SIGNAL if the process cannot be killed, and NO_PERMS if kill reports it does not have sufficient privileges.

  char *proc The process name
+*/
int sub_stop_proc(char *proc, int format) {
 FILE *fp;
 char buf[STRBUF];

 snprintf(buf, STRBUF, "var/%s.pid", proc);

 if((fp = fopen(buf, "r")) == NULL) {
  if(format == ENGLISH) {
   printf("%s: Not Running\n", proc);
  }
  else if(format == CSV) {
   printf("%s,Not Running\n", proc);
  }
  else if(format == HTML) {
   printf("<tr><td>%s</td><td>Not Running</td></tr>\n", proc);
  }
  return NO_FILE;
 }
 if(fgets(buf, BUFSIZ, fp) == NULL) {
  printf("Error reading from %s\n", buf);
  fclose(fp);
  return NO_FILE;
 }
 if(kill((pid_t)atoi(buf), SIGQUIT) < 0) {
  if(errno == ESRCH) {
   if(format == ENGLISH) {
    printf("%s: Not Running\n", proc);
   }
   else if(format == CSV) {
    printf("%s,Not Running\n", proc);
   }
   else if(format == HTML) {
    printf("<tr><td>%s</td><td>Not Running</td></tr>\n", proc);
   }
   fclose(fp);
   return NO_SIGNAL;
  }
  else if(errno == EPERM) {
   if(format == ENGLISH) {
    printf("%s: Permission Denied\n", proc);
   }
   else if(format == CSV) {
    printf("%s,Permission Denied\n", proc);
   }
   else if(format == HTML) {
    printf("<tr><td>%s</td><td>Permission Denied</td></tr>\n", proc);
   }
   fclose(fp);
   return NO_PERMS;
  }
 }
 else {
  if(format == ENGLISH) {
   printf("%s: Stopped\n", proc);
  }
  else if(format == CSV) {
   printf("%s,Stopped\n", proc);
  }
  else if(format == HTML) {
   printf("<tr><td>%s</td><td>Stopped</td></tr>\n", proc);
  }
  fclose(fp);
  return OK;
 }
 return OK;
}
