/*
Copyright (C) 2003, Nik Reiman - nik@aboleo.net

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

/****
  Takes a string, splits up the arguments into an array, and launches the resulting array via exec().
****/

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <signal.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/wait.h>

#include "ubs.h"

/*+
  Frontend to execvp(), which is the preferred exec() variant used by the UBS.

  int exec_proc Returns NO_MEM on a memory allocation error, or NO_FORK, if something else goes wrong.  Since this function calls exec (but never actually forks itself; the caller must take care of that), then it will never return.

  char *command The string to be executed.  Each argument must be separated by some whitespace.

  char *subarg Optionally, the *command string may contain an argument which may be substituted in for another string.  If this is the case, then the command string must contain '%s' somewhere in it, and the *subarg string will be inserted in place of the '%s'.
+*/
int exec_proc(char *command, char *subarg) {
 int ret, arg_count = 0;
 char run_string[STRBUF], *p, *run_cmd_args[MAXARGS];

  strncpy(run_string, command, STRBUF);

  // this is all ugly and needs to die, probably.  it seems to work, so
  // maybe keep it.  who knows.  life is full of hard choices.
  while((p = strchr(run_string, ' ')) != NULL) {
   if((run_cmd_args[arg_count] = (char *)malloc(sizeof(char) * BIGSTR)) == NULL) {
    log_error_msg(LOG_ERROR, "Unable to allocate memory for run_cmd_args");
    return NO_MEM;
   }
   bzero(run_cmd_args[arg_count], BIGSTR);
   strncpy(run_cmd_args[arg_count], run_string, p - run_string);
   if(!strcmp(run_cmd_args[arg_count], "%s")) {
    strncpy(run_cmd_args[arg_count], subarg, BIGSTR);
   }
   arg_count++;
   strncpy(run_string, p + 1, STRBUF);
  }
  run_cmd_args[arg_count] = (char *)malloc(sizeof(char) * BIGSTR);
  bzero(run_cmd_args[arg_count], BIGSTR);
  strncpy(run_cmd_args[arg_count], run_string, BIGSTR);
  if(!strcmp(run_cmd_args[arg_count], "%s")) {
   strncpy(run_cmd_args[arg_count], subarg, BIGSTR);
  }
  run_cmd_args[++arg_count] = (char *)0;

  ret = execvp(run_cmd_args[0], run_cmd_args);

  // should never reach here.  If so, we exit, so we don't screw up
  // whoever called us with fork().
  if(ret < 0) {
   log_error_msg(LOG_ERROR, "Failed to exec process");
   exit(FAIL);
  }

 return NO_FORK;
}

/*+
  Sends a unix signal "ping" to a UBS module that will be intercepted.

  int ping_pid Returns OK if the process is running, NO_SIGNAL if the process is not running, and NO_FILE if the .pid file is not present or malformed.

  char *target The process name to ping
+*/
int ping_pid(char *target) {
 FILE *fp;
 char buf[STRBUF];

 // absolute path is needed because daemons might have to check this before they chdir
 snprintf(buf, STRBUF, "%s/var/%s.pid", ubs_table_data(&GLOBAL, "prefix"), target);
 if((fp = fopen(buf, "r")) == NULL) {
  return NO_FILE;
 }
 if(fgets(buf, STRBUF, fp) == NULL) {
  fclose(fp);
  return NO_FILE;
 }
 if(kill(atoi(buf), SIGURG) == 0) {
  fclose(fp);
  return OK;
 }
 else {
  fclose(fp);
  return NO_SIGNAL;
 }

 return OK;
}
