
/*
  
  FACILITY:  NMMTL
  
  MODULE DESCRIPTION:
  
  Contatins nmmtl_evaluate_conductors function.
  
  AUTHORS:
  
  Kevin J. Buchs
  
  CREATION DATE:  Fri Nov 22 11:54:00 1991
  
  COPYRIGHT:   Copyright (C) 1991 by Mayo Foundation. All rights reserved.
  
  PORTABILITY ISSUES:
  
  should be none.
  
  */


/*
 *******************************************************************
 **  INCLUDE FILES
 *******************************************************************
 */

#include "nmmtl.h"

/*
 *******************************************************************
 **  STRUCTURE DECLARATIONS AND TYPE DEFINTIONS
 *******************************************************************
 */
/*
 *******************************************************************
 **  MACRO DEFINITIONS
 *******************************************************************
 */
/*
 *******************************************************************
 **  PREPROCESSOR CONSTANTS
 *******************************************************************
 */
/*
 *******************************************************************
 **  GLOBALS
 *******************************************************************
 */
/*
 *******************************************************************
 **  FUNCTION DECLARATIONS
 *******************************************************************
 */
/*
 *******************************************************************
 **  FUNCTION DEFINITIONS
 *******************************************************************
 */


/*
  
  FUNCTION NAME: nmmtl_evaluate_conductors
  
  FUNCTIONAL DESCRIPTION:
  
  Breaks up the conductors into appropriate line segments and circle
  segmeents depending upon whether they are rectangles, polygons, or
  circles.  Determines the intersections of these segments with
  dielectric-dielectric interface segments, and generates all the
  needed sub-segments for both types.
  
  
  FORMAL PARAMETERS:
  
  struct dielectric *dielectrics
  the raw dielectric rectangles
  
  double air_starts,
  Y coordinate of highest dielectric - where air starts
  
  int cntr_seg
  number of segments to break a contour into
  
  -- not used --
  float half_minimum_dimension
  half of the smallest geometric dimension - used to determine if
  segments are broken small enough.
  --  --
  
  EXTENT_DATA_P extent_data
  data on extent of conductor region (and whole cross section)

  int *conductor_counter
  keep count of the conductors
  
  CONTOURS_P *signals
  list of the signal contours
  
  CONTOURS_P *groundwires
  list of the groundwire contours
  
  LINE_SEGMENTS_P *conductor_ls
  list of conductor line segments generated by this routine
  
  CIRCLE_SEGMENTS_P *conductor_cs
  list of conductor circle segments generated by this routine.
  
  -- NOT USED --
  COND_PROJ_LIST_P *cond_projections
  develop a list of projects on the upper ground place of all conductor
  edges.
  -- --
  
  DIELECTRIC_SEGMENTS_P *dielectric_segments
  pointer to a list of segments formed by dielectric interfaces.
  
  
  RETURN VALUE:
  
  SUCCESS or FAIL
  
  
  CALLING SEQUENCE:
  
  status = nmmtl_evaluate_conductors(dielectrics,
  air_starts,
  cntr_seg,
  &extent_data,
  &conductor_counter,
  &signals,
  &conductor_ls,
  &conductor_cs,
  -- NOT USED -- &cond_projections,
  &dielectric_segments);
  
  */


int nmmtl_evaluate_conductors(struct dielectric *dielectrics,
			      double air_starts,
			      int cntr_seg,
			      EXTENT_DATA_P extent_data,
			      int *conductor_counter,
			      CONTOURS_P *signals,
			      CONTOURS_P *groundwires,
			      LINE_SEGMENTS_P *conductor_ls,
			      CIRCLE_SEGMENTS_P *conductor_cs,
			      DIELECTRIC_SEGMENTS_P *dielectric_segments)
{
  
  int status;
  CONTOURS_P contour;
#ifdef NMMTL_DUMP_DIAG
  extern FILE *dump_file;
#endif
  
  /* Process the signal conductors */
  contour = *signals;
  
  while(contour != NULL)
  {
    
    /* another valid conductor is evaluated */
    *conductor_counter = *conductor_counter + 1;
    
    switch(contour->primitive)
    {
    case POLYGON:
      status = nmmtl_evaluate_polygons(cntr_seg,
				       *conductor_counter,contour,conductor_ls,extent_data);
      /* nmmtl_project_polygon(cond_projections,contour); */
      
#ifdef NMMTL_DUMP_DIAG
      fprintf(dump_file,"\nprocessed polygon:\n");
      nmmtl_dump_polygon(contour);
      fprintf(dump_file,"\n\n");
#endif
      break;
    case RECTANGLE:
      status = nmmtl_evaluate_rectangles(cntr_seg,
#ifndef NO_HALF_MIN_CHECKING
                                         half_minimum_dimension,
#endif
					 *conductor_counter,contour,
                                         conductor_ls,extent_data);
      /* nmmtl_project_rectangle(cond_projections,contour); */
#ifdef NMMTL_DUMP_DIAG
      fprintf(dump_file,"\nprocessed rectangle:\n");
      nmmtl_dump_rectangle(contour);
      fprintf(dump_file,"\n\n");
#endif
      break;
    case CIRCLE:
      status = nmmtl_evaluate_circles(cntr_seg,
#ifndef NO_HALF_MIN_CHECKING
				      half_minimum_dimension,
#endif
				      *conductor_counter,contour,conductor_cs,extent_data);
      /* nmmtl_project_circle(cond_projections,contour); */
#ifdef NMMTL_DUMP_DIAG
      fprintf(dump_file,"\nprocessed circle:\n");
      nmmtl_dump_circle(contour);
      fprintf(dump_file,"\n\n");
#endif
      break;
    }
    
    if(status != SUCCESS) return(status);
    
    contour = contour->next;
    
  }
  
  /* Process the groundwire conductors */
  contour = *groundwires;
  
  while(contour != NULL)
  {
    
    /* pass in conductor number of zero for all these groundwires */
    
    switch(contour->primitive)
    {
    case POLYGON:
      status = nmmtl_evaluate_polygons(cntr_seg,
				       0,contour,conductor_ls,extent_data);
      /* nmmtl_project_polygon(cond_projections,contour); */
#ifdef NMMTL_DUMP_DIAG
      fprintf(dump_file,"\nprocessed polygon:\n");
      nmmtl_dump_polygon(contour);
      fprintf(dump_file,"\n\n");
#endif
      break;
    case RECTANGLE:
      status = nmmtl_evaluate_rectangles(cntr_seg,
#ifndef NO_HALF_MIN_CHECKING
	     half_minimum_dimension,
#endif
             0,contour,conductor_ls,extent_data);
      /* nmmtl_project_rectangle(cond_projections,contour); */
#ifdef NMMTL_DUMP_DIAG
      fprintf(dump_file,"\nprocessed rectangle:\n");
      nmmtl_dump_rectangle(contour);
      fprintf(dump_file,"\n\n");
#endif
      break;
    case CIRCLE:
      status = nmmtl_evaluate_circles(cntr_seg,
#ifndef NO_HALF_MIN_CHECKING
				      half_minimum_dimension,
#endif
				      0,contour,conductor_cs,extent_data);
      /* nmmtl_project_circle(cond_projections,contour); */
#ifdef NMMTL_DUMP_DIAG
      fprintf(dump_file,"\nprocessed circle:\n");
      nmmtl_dump_circle(contour);
      fprintf(dump_file,"\n\n");
#endif
      break;
    }
    
    if(status != SUCCESS) return(status);
    contour = contour->next;
  }
  
  status = nmmtl_determine_intersections(conductor_ls,
					 dielectric_segments);
  if(status != SUCCESS) return(status);
  
  status = nmmtl_determine_arc_intersectio(conductor_cs,
					   dielectric_segments);
  if(status != SUCCESS) return(status);
  
  status = nmmtl_orphans(conductor_cs,conductor_ls,dielectrics,
			 air_starts,dielectric_segments);
  
  if(status != SUCCESS) return(status);
  
#ifdef NMMTL_DUMP_DIAG
  status = nmmtl_dump_segments(*dielectric_segments,*conductor_ls,
			       *conductor_cs);
#endif
  
  
#ifdef DIAG_DUMP_PROJECTIONS
  /* dump projections list */
{
  COND_PROJ_LIST_P list;
#ifdef NMMTL_DUMP_DIAG
  fprintf(dump_file,"\n\nConductor Projections:\n  ");
  for(list = *cond_projections;list != NULL;list = list->next)
  {
    fprintf(dump_file,"%g,",list->key);
  }
  fprintf(dump_file,"\n");
#endif
  
}		      
#endif

return(status);


}
