/****************************************************************************
    Copyright (C) 1987-2001 by Jeffery P. Hansen

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    Last edit by hansen on Mon Dec 18 11:30:18 2000
****************************************************************************/
#include <stdlib.h>
#include <string.h>
#include "tkgate.h"

#define ENGLISH_LINESPACE	12	/* Line spacing for English text */
#define KANJI_LINESPACE		15	/* Line spacing for lines with kanji characters */

static char *psComment[] = {
  "%",
  "% An COMMENT gate",
  "%",
  "/pscomment {",
  "  0 startgate",
  "  12 cfont",
  "  newpath 0 0 moveto",
  "  { {} forall exch neg 0 exch rmoveto gsave show grestore } forall",
  "  grestore",
  "} bind def",
  0
};

GCElement *Comment_Make(EditState **es,GModuleDef *env,int GType,
		      int x,int y,int r,char *Name,int noWire,char**,int);
void Comment_Delete(GCElement *g,GModuleDef *env,int drawp);
void Comment_GetExtents(GCElement *g,int *minx,int *miny,int *maxx,int *maxy,int *bd);
int Comment_HitDistance(GCElement *g,int x,int y);
void Comment_Draw(GCElement *g,int md);
void Comment_PSWrite(FILE *f,GModLayout *L,GCElement *g);
void Comment_VerSave(FILE *f,GCElement *g);
void Comment_EditProps(GCElement *g,int isLoadDialog);
void Comment_SetProp(GCElement*,char*,void*);
GCElement *Comment_Replicate(GModuleDef *M,GCElement *g,int x,int y,unsigned);
void Comment_Init(GCElement*g);

GGateInfo gate_comment_info = {
  COMMENT,
  "COMMENT",
  "comment",0x0,
  "pscomment",psComment,

  {{"C",	{0,0},			{"gmcomment",0,0},		"gat_make comment"},
   {0}},
  0,

  0,{{0}},
  {{0,-12,CT},{12,0,LJ},{0,-12,CT},{12,0,LJ}},	/* Are these even used? */
  {1},

  {0},
  
  Comment_Make,
  Comment_Init,
  Comment_Delete,
  Comment_GetExtents,
  Comment_HitDistance,
  Comment_Draw,
  Generic_Move,
  Comment_Replicate,
  Err_AddInput,
  Err_AddOutput,
  Err_AddInOut,
  Err_ChangePin,
  Nop_SimStateFunc,
  Nop_SimHitFunc,
  Comment_PSWrite,
  Comment_EditProps,
  Comment_VerSave,
  Comment_SetProp
};

static int hasKanji(char *s)
{
  for (;*s;s++)
    if ((*s & 0x80)) return 1;

  return 0;
}

static void Comment_updateSize(GCElement *g)
{
  ComLine *L;
  int y; 
  
  if (!XGate.textF) return;

  y = 0;
  g->u.comment.width = 10; 
  g->u.comment.height = 0; 
  for (L = g->u.comment.first;L;L = L->next) {
    int w = GKTextWidth(XGate.textF,L->text,strlen(L->text));
    if (w > g->u.comment.width) g->u.comment.width = w;

#if TKGATE_JSUPPORT
    if (hasKanji(L->text))
      y += KANJI_LINESPACE;
    else
      y += ENGLISH_LINESPACE;
#else
    y += ENGLISH_LINESPACE;
#endif
  }
  g->u.comment.height = y; 
  if (g->u.comment.height < 10) g->u.comment.height = 10;
}

void Comment_Init(GCElement*g)
{
  Generic_Init(g);
  g->u.comment.width = g->u.comment.height = 0;
  g->u.comment.doLink = 0;
  g->u.comment.link = strdup("");
}

void Comment_flushLines(GCElement *g)
{
  ComLine *L,*N;

  for (L = g->u.comment.first;L;L = N) {
    N = L->next;
    free(L->text);
    free(L);
  }
  g->u.comment.first = g->u.comment.last = 0;
}

void Comment_addLine(GCElement *g,char *text)
{
  ComLine *L = (ComLine*) malloc(sizeof(ComLine)); 
  L->text = strdup(text);
  L->next = 0;

  if (g->u.comment.last) {
    g->u.comment.last->next = L;
    g->u.comment.last = L;
  } else
    g->u.comment.first = g->u.comment.last = L;
}

GCElement *Comment_Make(EditState **es,GModuleDef *env,int GType,
		      int x,int y,int r,char *Name,int noWire,char **options,int nOptions)
{
  GCElement *g;

  if (!(g = Generic_Make(es,env,GType,x,y,r,Name,noWire,options,nOptions)))
    return NULL;

  g->u.comment.first = g->u.comment.last = 0;
  g->u.comment.doLink = 0;
  g->u.comment.link = strdup("");

  if (es) {
    GGateInfo *gi = g->typeinfo;
    int ok;
    char *buf;

    (*gi->EditProps)(g,1);
    DoTcl("tkg_editGate");
    if ((buf = Tcl_GetVar(XGate.tcl,"edgat_ok",TCL_GLOBAL_ONLY)) && sscanf(buf,"%d",&ok) == 1 && ok)
      (*gi->EditProps)(g,0);
    else {
      gate_delete(g,(*es)->env,1);
      return 0;
    }

  }

  return g;
}

void Comment_Delete(GCElement *g,GModuleDef *M,int drawp)
{
  if (M)     gate_remove(M,g);
  if (drawp) gate_draw(g,0);

  Comment_flushLines(g);
}

void Comment_GetExtents(GCElement *g,int *minx,int *miny,int *maxx,int *maxy,int *bd)
{
  if (!g->u.comment.width) Comment_updateSize(g);

  *minx = g->xpos;
  *miny = g->ypos;
  *maxx = g->xpos + g->u.comment.width;
  *maxy = g->ypos + g->u.comment.height;

  if (bd) *bd = 20;
}

int Comment_HitDistance(GCElement *g,int x,int y)
{
  int dy;

  if (x < g->xpos || x > g->xpos + g->u.comment.width)
    return GATERANGE+1;

  if (y < g->ypos || y > g->ypos + g->u.comment.height)
    return GATERANGE+1;

  dy = y-(g->ypos + g->u.comment.height/2);
  if (dy < 0) dy = -dy;

  if (dy < GATERANGE-1)
    return dy;

  return GATERANGE-1;
}

GCElement *Comment_Replicate(GModuleDef *M,GCElement *g,int x,int y,unsigned flags)
{
  GCElement *ng;
  ComLine *L;

  ng = Generic_Replicate(M,g,x,y,flags);
  ng->u.comment.first = ng->u.comment.last = 0;
  for (L = g->u.comment.first;L;L = L->next) {
    Comment_addLine(ng,L->text);
  }

  ng->u.comment.doLink = g->u.comment.doLink;
  ng->u.comment.link = strdup(g->u.comment.link);

  return ng;
}

void Comment_Draw(GCElement *g,int md)
{
  ComLine *L;
  int x,y; 
  GC gc;

  x = g->xpos;
  y = g->ypos;

  if (!g->u.comment.width) Comment_updateSize(g);

  if (g->u.comment.doLink) {
    gc = XGate.hyperlinkGC;
#if TKGATE_JSUPPORT
    Tkg_changeColor(XGate.kanjiGC, GXxor, XGate.hyperlink_pixel);
#endif
  } else
    gc = XGate.commentGC;

  if (g->selected)
    XSetFont(XGate.D,gc,XGate.textbXF);
  else
    XSetFont(XGate.D,gc,XGate.textXF);


  for (L = g->u.comment.first;L;L = L->next) {
#if TKGATE_JSUPPORT
    if (hasKanji(L->text))
      y += KANJI_LINESPACE;
    else
      y += ENGLISH_LINESPACE;
#else
    y += ENGLISH_LINESPACE;
#endif
    dce_DrawString(gc,x,y,LJ,L->text);
  }

#if TKGATE_JSUPPORT
  if (g->u.comment.doLink)
    Tkg_changeColor(XGate.kanjiGC, GXxor, XGate.comment_pixel);
#endif

  if (!g->u.comment.first) {
    ZDrawRectangle(XGate.D,XGate.W,gc,
		   g->xpos+XGate.org_x-5,g->ypos+XGate.org_y-5,
		   g->u.comment.width+10,g->u.comment.height+10);
  }
}

void Comment_PSWrite(FILE *f,GModLayout *L,GCElement *g)
{
  ComLine *Li;
  char buf[STRMAX];

  fprintf(f,"[\n");
  for (Li = g->u.comment.first;Li;Li = Li->next) {
    int lspace;

    if (hasKanji(Li->text))
      lspace = KANJI_LINESPACE;
    else
      lspace = ENGLISH_LINESPACE;
    fprintf(f,"   [%d (%s)]\n",lspace,filterParen(buf,Li->text));
  }
  fprintf(f,"] %d %d pscomment\n",g->xpos,g->ypos);
}

void Comment_VerSave(FILE *f,GCElement *g)
{
  char buf[STRMAX];
  ComLine *L;

  fprintf(f,"  //: comment %s /dolink:%d /link:\"%s\"",g->ename,g->u.comment.doLink,g->u.comment.link);

  VerilogBasicGateComment(f,g,0);
  fprintf(f,"\n");

  for (L = g->u.comment.first;L;L = L->next) {
    fprintf(f,"  //: /line:\"%s\"\n",quoteChars(buf,L->text,"\"\\"));
  }
  fprintf(f,"  //: /end\n");
}

void Comment_EditProps(GCElement *g,int isLoadDialog)
{
  char buf[STRMAX];

  Generic_EditProps(g,isLoadDialog);

  if (isLoadDialog) {
    ComLine *L;
    int n = 0;

    for (L = g->u.comment.first;L;L = L->next) {
      DoTcl("set edgat_commentLines(%d) \"%s\"",n++,quoteChars(buf,L->text,"\"[]\\"));
    }
    DoTcl("set edgat_commentLen %d",n);

    DoTcl("set edgat_dolink %d",g->u.comment.doLink);
    DoTcl("set edgat_link \"%s\"",g->u.comment.link);
  } else {
    char *p;
    int n = 0;
    int i;
    Tcl_Interp *tcl = XGate.tcl;

    Comment_flushLines(g);

    if ((p = Tcl_GetVar(tcl,"edgat_commentLen",TCL_GLOBAL_ONLY)))
      sscanf(p,"%d",&n);
    for (i = 0;i < n;i++) {
      sprintf(buf,"edgat_commentLines(%d)",i);
      p = Tcl_GetVar(tcl,buf,TCL_GLOBAL_ONLY);
      Comment_addLine(g,p);
    }
    Comment_updateSize(g);

    p = Tcl_GetVar(tcl,"edgat_dolink",TCL_GLOBAL_ONLY);
    sscanf(p,"%d",&g->u.comment.doLink);
    free(g->u.comment.link);
    p = Tcl_GetVar(tcl,"edgat_link",TCL_GLOBAL_ONLY);
    g->u.comment.link = strdup(p);
  }
}

void Comment_SetProp(GCElement *g,char *prop,void *value)
{
  char *s = (char*)value;

  if (strcmp(prop,"/line") == 0) {
    Comment_addLine(g,s);
    Comment_updateSize(g);
  }
  if (strcmp(prop,"/dolink") == 0) {
    g->u.comment.doLink = *(int*)value;
  }
  if (strcmp(prop,"/link") == 0) {
    free(g->u.comment.link);
    g->u.comment.link = strdup(s);
  }
}

void init_comment()
{
  RegisterGate(&gate_comment_info);
}
