/*	$Id: masterpty.c,v 1.1 2004/09/12 19:20:28 steve Exp $	*/

/*-
 * Copyright (c) 2004 Steve C. Woodford.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *        This product includes software developed by Steve C. Woodford.
 * 4. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
#include <sys/types.h>

#include <errno.h>
#include <fcntl.h>
#include <libgen.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <syslog.h>
#include <unistd.h>

#include "context.h"
#include "buffer.h"
#include "config.h"
#include "masterpty.h"

static const char *cf_masterpty_init(void *, char **, int, int, void *);
static int	masterpty_init(struct client_ctx *, const void *);
static void	masterpty_destroy(struct client_ctx *);
static int	masterpty_event(struct client_ctx *, int);
static int	masterpty_read_pending(struct client_ctx *);
static int	masterpty_ioctl(struct client_ctx *, int, void *,
		    struct client_ctx *);

struct client_ops masterpty_ops = {
	"masterpty",
	cf_masterpty_init, masterpty_init, masterpty_destroy, masterpty_event,
	NULL, masterpty_read_pending, NULL, masterpty_ioctl
};

static const char *cf_masterpty_symlinkdir(void *, char **, int, int, void *);
static const char *cf_masterpty_symlinkname(void *, char **, int, int, void *);
static const char *cf_masterpty_readonly(void *, char **, int, int, void *);
static const char *cf_masterpty_allowbreak(void *, char **, int, int, void *);

static struct config_tokens cf_masterpty_tokens[] = {
	{"symlinkdir",	1,	cf_masterpty_symlinkdir},
	{"symlinkname",	1,	cf_masterpty_symlinkname},
	{"readonly",	1,	cf_masterpty_readonly},
	{"allowbreak",	1,	cf_masterpty_allowbreak},
	{NULL, 0, NULL}
};

struct masterpty_args {
	char	ma_linkdir[FILENAME_MAX];
	char	ma_linkname[FILENAME_MAX];
	char	*ma_name;
	struct client_options ma_copts;
};

struct masterpty_ctx {
	struct client_ctx *mc_slave;
	char *mc_slavename;
	char *mc_symlink;
	int mc_fd;
};


static int	slavepty_init(struct client_ctx *, const void *);
static void	slavepty_destroy(struct client_ctx *);
static int	slavepty_event(struct client_ctx *, int);
static int	slavepty_output(struct client_ctx *, const char *, size_t,
		    struct client_ctx *);
static int	slavepty_write_pending(struct client_ctx *);

static struct client_ops slavepty_ops = {
	"slavepty",
	NULL, slavepty_init, slavepty_destroy, slavepty_event,
	slavepty_output, NULL, slavepty_write_pending, NULL
};

struct slavepty_ctx {
	struct buffer_ctx *sc_bc;
};


static int
masterpty_init(struct client_ctx *cc, const void *arg)
{
	const struct masterpty_args *ma = arg;
	struct masterpty_ctx *mc;
	char *slavename;
	char buff[1024];

	if ((cc->cc_fd = posix_openpt(O_RDWR | O_NONBLOCK)) < 0)
		return (-1);

	if (unlockpt(cc->cc_fd) < 0 || grantpt(cc->cc_fd) < 0) {
		close(cc->cc_fd);
		return (-1);
	}

	if ((slavename = ptsname(cc->cc_fd)) == NULL) {
		close(cc->cc_fd);
		return (-1);
	}

	if ((mc = calloc(1, sizeof(*mc))) == NULL) {
		close(cc->cc_fd);
		return (-1);
	}
	cc->cc_data = (void *) mc;
	cc->cc_name = NULL;
	mc->mc_fd = cc->cc_fd;

	if ((mc->mc_slavename = strdup(slavename)) == NULL) {
		(void) free(mc);
		close(cc->cc_fd);
		return (-1);
	}

	if (strlen(ma->ma_linkname) == 0)
		snprintf(buff, sizeof(buff), "%s/%s", ma->ma_linkdir,
		    basename(mc->mc_slavename));
	else
		snprintf(buff, sizeof(buff), "%s/%s", ma->ma_linkdir,
		    ma->ma_linkname);

	if ((mc->mc_symlink = strdup(buff)) == NULL) {
		(void) free(mc);
		close(cc->cc_fd);
		return (-1);
	}

	if (symlink(mc->mc_slavename, mc->mc_symlink) < 0) {
		(void) free(mc->mc_symlink);
		(void) free(mc);
		close(cc->cc_fd);
		return (-1);
	}

	return (0);
}

static void
masterpty_destroy(struct client_ctx *cc)
{
	struct masterpty_ctx *mc = cc->cc_data;

	if (mc->mc_slave)
		context_del_client(cc->cc_ctx, mc->mc_slave);

	(void) close(cc->cc_fd);
	(void) free(mc->mc_symlink);
	(void) free(mc->mc_slavename);
	(void) free(mc);
}

static int
masterpty_event(struct client_ctx *cc, int events)
{
	struct masterpty_ctx *mc = cc->cc_data;

	if ((events & POLLRDNORM) == 0 || mc->mc_slave != NULL)
		return (0);

	if (client_init(&mc->mc_slave, cc, &slavepty_ops,
	    &cc->cc_options, mc) < 0)
		return (-1);

	if (context_add_client(cc->cc_ctx, mc->mc_slave) < 0) {
		client_destroy(mc->mc_slave);
		mc->mc_slave = NULL;
		return (-1);
	}

	return (0);
}

static int
masterpty_read_pending(struct client_ctx *cc)
{
	struct masterpty_ctx *mc = cc->cc_data;

	return (mc->mc_slave == NULL);
}

static int
masterpty_ioctl(struct client_ctx *cc, int cmd, void *arg,
    struct client_ctx *sender)
{
	struct masterpty_ctx *mc = cc->cc_data;

	switch (cmd) {
	case CLIENT_IOCTL_CHILD_DELETED:
		mc->mc_slave = NULL;
		break;
	}

	return (0);
}

static int
slavepty_init(struct client_ctx *cc, const void *arg)
{
	const struct masterpty_ctx *mc = arg;
	struct slavepty_ctx *sc;

	if ((cc->cc_fd = dup(mc->mc_fd)) < 0)
		return (-1);

	if ((sc = calloc(1, sizeof(*sc))) == NULL) {
		close(cc->cc_fd);
		return (-1);
	}

	if (buffer_init(&sc->sc_bc, cc->cc_fd) < 0) {
		(void) free(sc);
		close(cc->cc_fd);
		return (-1);
	}

	cc->cc_canhup = 1;
	cc->cc_data = (void *) sc;
	cc->cc_name = mc->mc_slavename;

	return (0);
}

static void
slavepty_destroy(struct client_ctx *cc)
{
	struct slavepty_ctx *sc = cc->cc_data;

	buffer_destroy(sc->sc_bc);
	(void) close(cc->cc_fd);
	(void) free(sc);
}

static int
slavepty_event(struct client_ctx *cc, int events)
{
	struct slavepty_ctx *sc = cc->cc_data;
	char buff[256];
	ssize_t rv = 0;

	if ((events & POLLRDNORM) != 0) {
		while ((rv = read(cc->cc_fd, buff, sizeof(buff))) != 0) {
			if (rv < 0) {
				if (errno == EINTR || errno == EWOULDBLOCK)
					break;
				return (-1);
			}

			context_client_output(cc->cc_ctx, buff, (size_t)rv, cc);
		}
	}

	if (rv >= 0 && (events & POLLWRNORM) != 0)
		rv = buffer_drain(sc->sc_bc);

	if (rv >= 0 && (events & POLLHUP) != 0)
		rv = -1;

	return (rv);
}

/* ARGSUSED */
static int
slavepty_output(struct client_ctx *cc, const char *buff, size_t len,
    struct client_ctx *sender)
{
	struct slavepty_ctx *sc = cc->cc_data;

	return buffer_fill(sc->sc_bc, buff, len);
}

static int
slavepty_write_pending(struct client_ctx *cc)
{
	struct slavepty_ctx *sc = cc->cc_data;

	return (buffer_length(sc->sc_bc) != 0);
}

/* ARGSUSED */
static const char *
cf_masterpty_init(void *cs, char **argv, int argc, int is_compound, void *arg)
{
	struct context *ctx = arg;
	struct client_ctx *cc;
	struct masterpty_args ma;
	const char *errstr;

	memset(&ma, 0, sizeof(ma));
	ma.ma_copts.co_allowbreak = 1;
	strcpy(ma.ma_linkdir, TITS_DEFAULT_LINKDIR);

	errstr = config_parse(cs, cf_masterpty_tokens, (void *)&ma);
	if (errstr)
		return (errstr);

	if (client_init(&cc, NULL, &masterpty_ops, &ma.ma_copts, &ma) < 0)
		errstr = config_err(cs, "masterpty: %s", strerror(errno));
	else
	if (context_add_client(ctx, cc) < 0) {
		errstr = config_err(cs, "masterpty: %s", strerror(errno));
		client_destroy(cc);
	}

	return (errstr);
}

/* ARGSUSED */
static const char *
cf_masterpty_symlinkdir(void *cs, char **argv, int argc, int cmpnd, void *arg)
{
	struct masterpty_args *ma = arg;

	strncpy(ma->ma_linkdir, argv[1], sizeof(ma->ma_linkdir));

	return (NULL);
}

/* ARGSUSED */
static const char *
cf_masterpty_symlinkname(void *cs, char **argv, int argc, int cmpnd, void *arg)
{
	struct masterpty_args *ma = arg;

	strncpy(ma->ma_linkname, argv[1], sizeof(ma->ma_linkname));

	return (NULL);
}

/* ARGSUSED */
static const char *
cf_masterpty_readonly(void *cs, char **argv, int argc, int cmpnd, void *arg)
{
	struct masterpty_args *ma = arg;

	return (config_boolean(cs, argv[1], &ma->ma_copts.co_readonly));
}

/* ARGSUSED */
static const char *
cf_masterpty_allowbreak(void *cs, char **argv, int argc, int cmpnd, void *arg)
{
	struct masterpty_args *ma = arg;

	return (config_boolean(cs, argv[1], &ma->ma_copts.co_allowbreak));
}
