/************************************************************************/
/*									*/
/*  Print Postscript.							*/
/*									*/
/************************************************************************/

#   include	"utilPsConfig.h"

#   include	<stddef.h>
#   include	<stdio.h>
#   include	<stdlib.h>
#   include	<string.h>
#   include	<time.h>

#   include	<psPrint.h>
#   include	<geo2DInteger.h>
#   include	<sioGeneral.h>

#   include	<appDebugon.h>

/************************************************************************/
/*									*/
/*  Initialisation for 'nup' printing.					*/
/*									*/
/************************************************************************/

int psSetNupSchema(	PrintingState *			ps,
			const DocumentGeometry *	dgPage,
			const PrintGeometry *		pg,
			int				hasPageHeader,
			int				hasPageFooter )
    {
    const double		fac= 0.05;
    int				rotatePages;

    AffineTransform2D		at1Page;

    if  ( utilNupGetBaseTranform( &at1Page, &rotatePages, pg, dgPage, fac ) )
	{ LDEB(1); return -1;	}

    ps->psRotateSheetGrid= rotatePages;
    ps->psPrintGeometry= *pg;

    if  ( utilNupSetSchema( &(ps->psNupSchema),
			ps->psRotateSheetGrid, &at1Page, pg, fac, dgPage ) )
	{ LLDEB(pg->pgGridRows,pg->pgGridCols); return -1;	}

    if  ( ps->psRotateSheetGrid )
	{ ps->psOrientation= "Landscape";	}
    else{ ps->psOrientation= "Portrait";	}

    psNupSheetBoundingBox( &(ps->psSheetBoundingBox), &(ps->psNupSchema),
					dgPage, hasPageHeader, hasPageFooter );

    return 0;
    }

void psRefreshNupSchema(
			PrintingState *			ps,
			const DocumentGeometry *	dgPage )
    {
    const double		fac= 0.05;
    const PrintGeometry *	pg= &(ps->psPrintGeometry);
    AffineTransform2D		at1Page;

    int				rotatePages; /* ignored */

    if  ( utilNupGetBaseTranform( &at1Page, &rotatePages, pg, dgPage, fac ) )
	{ LDEB(1); return;	}

    if  ( utilNupSetSchema( &(ps->psNupSchema),
			ps->psRotateSheetGrid, &at1Page, pg, fac, dgPage ) )
	{ LLDEB(pg->pgGridRows,pg->pgGridCols); return;	}

    /* ??
    psNupSheetBoundingBox( &(ps->psSheetBoundingBox), &(ps->psNupSchema),
					dgPage, hasPageHeader, hasPageFooter );
    */

    return;
    }

/************************************************************************/
/*									*/
/*  Issue a character string.						*/
/*									*/
/************************************************************************/

void psPrintString(	SimpleOutputStream *	sos,
			const unsigned char *	s,
			int			len,
			int			sevenBits )
    {
    int				i;

    for ( i= 0; i < len; s++, i++ )
	{
	if  ( *s == '(' || *s == ')' || *s == '\\' )
	    {
	    sioOutPutByte( '\\', sos );
	    sioOutPutByte( *s, sos );
	    continue;
	    }

	if  ( *s == '\r' )
	    {
	    sioOutPutByte( '\\', sos );
	    sioOutPutByte( 'r', sos );
	    continue;
	    }

	if  ( *s == '\n' )
	    {
	    sioOutPutByte( '\\', sos );
	    sioOutPutByte( 'n', sos );
	    continue;
	    }

	if  ( (   isascii( *s ) && ! isprint( *s )	)	||
	      ( ! isascii( *s ) && sevenBits		)	)
	    {
	    sioOutPrintf( sos, "\\%03o", *s );
	    continue;
	    }

	sioOutPutByte( *s, sos );
	}

    return;
    }

void psPrintStringValue(	PrintingState *		ps,
				const unsigned char *	s,
				int			len )
    {
    SimpleOutputStream *	sos= ps->psSos;

    sioOutPutByte( '(', sos );
    psPrintString( ps->psSos, s, len, ps->ps7Bits );
    sioOutPutByte( ')', sos );

    return;
    }

void psMoveShowString(	PrintingState *		ps,
			const unsigned char *	s,
			int			len,
			int			x,
			int			y )
    {
    SimpleOutputStream *	sos= ps->psSos;

    sioOutPutByte( '(', sos );
    psPrintString( ps->psSos, s, len, ps->ps7Bits );
    sioOutPrintf( sos, ") %d %d mvs\n", x, y );

    ps->psLastPageMarked= ps->psPagesPrinted;

    return;
    }

void psShowString(	PrintingState *		ps,
			const unsigned char *	s,
			int			len )
    {
    SimpleOutputStream *	sos= ps->psSos;

    sioOutPutByte( '(', sos );
    psPrintString( ps->psSos, s, len, ps->ps7Bits );
    sioOutPrintf( sos, ") utf8show " );

    ps->psLastPageMarked= ps->psPagesPrinted;

    return;
    }

/************************************************************************/
/*									*/
/*  Set the font, this depends on the assumption that fonts have been	*/
/*  defined for the different fonts in the document.			*/
/*									*/
/*  NOTE that 'smallcaps' is never set for characters that are not to	*/
/*  be scaled.								*/
/*									*/
/************************************************************************/

void psSetFont(	PrintingState *			ps,
		const AfmFontInfo *		afi,
		const TextAttribute *		ta )
    {
    SimpleOutputStream *	sos= ps->psSos;
    int				fontSizeTwips= 10* ta->taFontSizeHalfPoints;
    char			fontName[40];

    if  ( ta->taSuperSub == DOCfontSUPERSCRIPT	||
	  ta->taSuperSub == DOCfontSUBSCRIPT	)
	{ fontSizeTwips= SUPERSUB_SIZE( fontSizeTwips ); }

    if  ( ta->taSmallCaps )
	{ fontSizeTwips= ( 8* fontSizeTwips )/ 10;	}

    if  ( fontSizeTwips == 0 )
	{ LDEB(fontSizeTwips); fontSizeTwips= 1;	}

    psSetFontName( fontName, afi );
    sioOutPrintf( sos, "%d %s\n", fontSizeTwips, fontName );

    ps->psLastPageMarked= ps->psPagesPrinted;

    return;
    }

/************************************************************************/
/*									*/
/*  Initialise a printing session.					*/
/*									*/
/************************************************************************/

void psInitPrintingState(	PrintingState *	ps )
    {
    ps->psSos= (SimpleOutputStream *)0;

    ps->psLastPageMarked= -1;
    ps->psLastSheetMarked= -1;
    ps->psPagesPrinted= 0;
    ps->psSheetsPrinted= 0;
    ps->psSheetsStarted= 0;

    psInitPrintGeometry( &(ps->psPrintGeometry) );

    psInitNupSchema( &(ps->psNupSchema) );
    ps->psRotateSheetGrid= 0;
    geoIdentityAffineTransform2D( &(ps->psCurrentTransform) );

    ps->psOrientation= (char *)0;

    geoInitRectangle( &(ps->psSheetBoundingBox) );

    ps->psInsideLink= 0;

    ps->psLinkParticulesDone= 0;
    ps->psLinkRectLeft= -1;
    ps->psLinkFile= (const char *)0;
    ps->psLinkFileSize= 0;
    ps->psLinkMark= (const char *)0;
    ps->psLinkMarkSize= 0;

    ps->psUsePostScriptFilters= 1;
    ps->psUsePostScriptIndexedImages= 1;
    ps->ps7Bits= 0;

    return;
    }

void psCleanPrintingState(	PrintingState *	ps )
    {
    psCleanNupSchema( &(ps->psNupSchema) );

    psCleanPrintGeometry( &(ps->psPrintGeometry) );

    return;
    }

/************************************************************************/
/*									*/
/*  Set the color for subsequent drawing operations.			*/
/*									*/
/************************************************************************/

void psSetRgb8Color(	PrintingState *		ps,
			const RGB8Color *	rgb8 )
    {
    if  ( rgb8->rgb8Red == rgb8->rgb8Green	&&
	  rgb8->rgb8Green == rgb8->rgb8Blue	)
	{
	sioOutPrintf( ps->psSos, "%g setgray\n", rgb8->rgb8Red/ 255.0 );
	}
    else{
	sioOutPrintf( ps->psSos, "%g %g %g setrgbcolor\n",
						rgb8->rgb8Red/ 255.0,
						rgb8->rgb8Green/ 255.0,
						rgb8->rgb8Blue/ 255.0 );
	}

    ps->psLastPageMarked= ps->psPagesPrinted;

    return;
    }

/************************************************************************/
/*									*/
/*  Issue a 2D affine transform in PS notation.				*/
/*									*/
/************************************************************************/

void psTransformMatrix(	SimpleOutputStream *		sos,
			const AffineTransform2D *	at )
    {
    sioOutPrintf( sos, "[ %g %g %g %g %g %g ]",
				    at->at2Axx, at->at2Axy,
				    at->at2Ayx, at->at2Ayy,
				    at->at2Tx, at->at2Ty );
    return;
    }

/************************************************************************/
/*									*/
/*  Define a procedure from an array of lines of PostScript code.	*/
/*  Empty lines and lines completely consisting of a comment are	*/
/*  skipped.								*/
/*									*/
/************************************************************************/

void psDefineProcedure(	SimpleOutputStream *	sos,
				const char **		lines,
				int			count )
    {
    int		i;

    for ( i= 0; i < count; lines++, i++ )
	{
	const char *	s= *lines;

	while( isspace( *s ) )
	    { s++;	}

	if  ( ! *s || *s == '%' )
	    { continue;	}

	sioOutPrintf( sos, "%s\n", *lines );
	}

    sioOutPrintf( sos, "\n" );
    }

/************************************************************************/
/*									*/
/*  Fill a rectangle							*/
/*									*/
/************************************************************************/

void psFillRectangle(		PrintingState *		ps,
				int			x,
				int			y,
				int			wide,
				int			high )
    {
    sioOutPrintf( ps->psSos, "%d %d %d %d rectfill\n", x, y, wide, high );

    ps->psLastPageMarked= ps->psPagesPrinted;

    return;
    }

void psRectanglePath(		PrintingState *		ps,
				int			x,
				int			y,
				int			wide,
				int			high )
    {
    sioOutPrintf( ps->psSos, "newpath %d %d moveto ", x, y );
    sioOutPrintf( ps->psSos, "%d 0 rlineto ", wide );
    sioOutPrintf( ps->psSos, "0 %d rlineto ", high );
    sioOutPrintf( ps->psSos, "%d 0 rlineto\n", -wide );
    sioOutPrintf( ps->psSos, "0 %d rlineto closepath\n", -high );

    ps->psLastPageMarked= ps->psPagesPrinted;

    return;
    }

/************************************************************************/
/*									*/
/*  Insert the destination of a pdfmark jump in the Printout.		*/
/*									*/
/************************************************************************/

int psDestPdfmark(		PrintingState *		ps,
				int			lineTop,
				const char *		refName,
				int			refSize )
    {
    psEmitDestPdfmark( ps->psSos, &(ps->psCurrentTransform),
						lineTop, refName, refSize );
    return 0;
    }

/************************************************************************/
/*									*/
/*  Write links to be picked up by the distiller.			*/
/*									*/
/************************************************************************/

void psFlushLink(		PrintingState *		ps,
				int			x0,
				int			wide,
				int			lineTop,
				int			lineHeight )
    {
    if  ( ps->psLinkParticulesDone > 0 )
	{
	DocumentRectangle	drLink;

	drLink.drX0= ps->psLinkRectLeft;
	drLink.drY0= lineTop+ lineHeight;
	drLink.drX1= x0+ wide;
	drLink.drY1= lineTop;

	psSourcePdfmark( ps->psSos, &drLink,
		    (const unsigned char *)ps->psLinkFile, ps->psLinkFileSize,
		    (const unsigned char *)ps->psLinkMark, ps->psLinkMarkSize );

	ps->psLinkParticulesDone= 0;
	ps->psLinkRectLeft= x0;
	}

    return;
    }

