/************************************************************************/
/*									*/
/*  Print Postscript.							*/
/*									*/
/************************************************************************/

#   include	"utilPsConfig.h"

#   include	<stddef.h>
#   include	<stdio.h>
#   include	<stdlib.h>
#   include	<string.h>
#   include	<time.h>

#   include	<psPrint.h>
#   include	<psGlyphs.h>
#   include	<utilTree.h>
#   include	<geo2DInteger.h>
#   include	<utilFontmap.h>
#   include	<appSystem.h>
#   include	<psFace.h>
#   include	<sioStdio.h>
#   include	<sioPfb.h>
#   include	<psTtf.h>
#   include	<ucd.h>

#   include	<appDebugon.h>

/************************************************************************/
/*									*/
/*  List the font names that appear in a document.			*/
/*									*/
/************************************************************************/

static int utilPsGetEmbedFromFilename(	const char *		fontFile )
    {
    const char *	extension= appFileExtensionOfName( fontFile );

    if  ( extension &&
	  ( ! strcmp( extension, "pfb" ) 	||
	    ! strcmp( extension, "PFB" ) 	)	)
	{ return PSembedBTOA;	}

    if  ( extension &&
	  ( ! strcmp( extension, "pfa" ) 	||
	    ! strcmp( extension, "PFA" ) 	)	)
	{ return PSembedCOPY;	}

    if  ( extension &&
	  ( ! strcmp( extension, "ttf" ) 	||
	    ! strcmp( extension, "TTF" ) 	)	)
	{ return PSembedTTFTO42;	}

    if  ( extension &&
	  ( ! strcmp( extension, "ttc" ) 	||
	    ! strcmp( extension, "TTC" ) 	)	)
	{ return PSembedTTCTO42;	}

    return PSembedNO;
    }

static int utilPsGetEmbed(		PostScriptFace *	psf,
					const char *		fontDirectory )
    {
    int		res;

    if  ( psf->psfAfi->afiFontFileName )
	{
	if  ( appTestFileExists( psf->psfAfi->afiFontFileName ) )
	    { return PSembedNO;	}

	res= utilPsGetEmbedFromFilename( psf->psfAfi->afiFontFileName );
	if  ( res == PSembedNO )
	    { return res;	}

	psf->psfFontFileName= strdup( psf->psfAfi->afiFontFileName );
	if  ( ! psf->psfFontFileName )
	    { XDEB(psf->psfFontFileName); return PSembedNO;	}
	psf->psfFontFileNameLength= strlen( psf->psfFontFileName );
	psf->psfFontFileIndex= psf->psfAfi->afiFontFileIndex;

	psf->psfEmbed= res;
	return res;
	}
    else{
	const char *	fontFile;

	const int	relativeIsFile= 0;
	char		absolute[1000+1];
	int		l;

	fontFile= utilFontmapGetEntry( psf->psfAfi->afiFontName );
	if  ( ! fontFile )
	    { return PSembedNO;	}

	l= appAbsoluteName( absolute, sizeof(absolute)-1, fontFile,
					    relativeIsFile, fontDirectory );
	if  ( l <= 0 )
	    { return PSembedNO;	}
	if  ( appTestFileExists( absolute ) )
	    { return PSembedNO;	}

	res= utilPsGetEmbedFromFilename( absolute );
	if  ( res == PSembedNO )
	    { return res;	}

	psf->psfFontFileName= strdup( absolute );
	if  ( ! psf->psfFontFileName )
	    { XDEB(psf->psfFontFileName); return PSembedNO;	}
	psf->psfFontFileNameLength= l;
	psf->psfFontFileIndex= 0;

	psf->psfEmbed= res;
	return res;
	}
    }

/************************************************************************/
/*									*/
/*  Emit DSC comments about the fonts in the document.			*/
/*									*/
/*  1)  List the fonts that are used in the document.			*/
/*  2)  List the fonts that are embedded in the document.		*/
/*  3)  List 1-2: The fonts that are still needed.			*/
/*									*/
/************************************************************************/

void psDscListFontNames(	SimpleOutputStream *		sos,
				int				embedFonts,
				const PostScriptTypeList *	pstl )
    {
    PostScriptFace *	psf;
    const char *	fontName;
    int			done;

    if  ( ! pstl->pstlFaceTree )
	{ return;	}

    done= 0;
    psf= (PostScriptFace *)utilTreeGetFirst( pstl->pstlFaceTree, &fontName );
    /*  1  */
    while( psf )
	{
	if  ( done > 0 )
	    {
	    sioOutPrintf( sos, "%%%%+ %s\n", psf->psfAfi->afiFontName );
	    }
	else{
	    sioOutPrintf( sos, "%%%%DocumentFonts: %s\n",
					    psf->psfAfi->afiFontName );
	    }

	done++;
	psf= (PostScriptFace *)utilTreeGetNext( pstl->pstlFaceTree, &fontName );
	}

    /*  2  */
    if  ( embedFonts )
	{
	done= 0;
	psf= (PostScriptFace *)utilTreeGetFirst( pstl->pstlFaceTree,
								&fontName );
	while( psf )
	    {
	    if  ( psf->psfEmbed == PSembedUNKNOWN )
		{ utilPsGetEmbed( psf, pstl->pstlFontDirectory );	}

	    if  ( psf->psfEmbed > 0 )
		{
		if  ( done > 0 )
		    {
		    sioOutPrintf( sos, "%%%%+ %s\n", psf->psfAfi->afiFontName );
		    }
		else{
		    sioOutPrintf( sos, "%%%%DocumentSuppliedFonts: %s\n",
						    psf->psfAfi->afiFontName );
		    }

		done++;
		}

	    psf= (PostScriptFace *)utilTreeGetNext( pstl->pstlFaceTree,
								&fontName );
	    }
	}

    /*  3  */
    done= 0;
    psf= (PostScriptFace *)utilTreeGetFirst( pstl->pstlFaceTree, &fontName );
    while( psf )
	{
	if  ( ! embedFonts || psf->psfEmbed <= 0 )
	    {
	    if  ( done > 0 )
		{
		sioOutPrintf( sos, "%%%%+ %s\n", psf->psfAfi->afiFontName );
		}
	    else{
		sioOutPrintf( sos, "%%%%DocumentNeededFonts: %s\n",
						psf->psfAfi->afiFontName );
		}

	    done++;
	    }

	psf= (PostScriptFace *)utilTreeGetNext( pstl->pstlFaceTree, &fontName );
	}

    return;
    }

/************************************************************************/
/*									*/
/*  Make function names for setting the fonts in a list.		*/
/*									*/
/*  1)  Define all standard encodings.					*/
/*  2)  Define all encoded fonts.					*/
/*									*/
/************************************************************************/

static void psDefineEncodedFont(
				SimpleOutputStream *	sos,
				const AfmFontInfo *	afi,
				const char *		encodedFontName,
				const char *		encodingArrayName )
    {
    sioOutPrintf( sos, "/%s findfont dup length dict begin\n",
							afi->afiFontName );
    sioOutPrintf( sos, "  {\n" );
    sioOutPrintf( sos, "    1 index /FID ne\n" );
    sioOutPrintf( sos, "      { def } { pop pop } ifelse\n" );
    sioOutPrintf( sos, "  } forall\n");
    sioOutPrintf( sos, "  /Encoding %s def currentdict\n", encodingArrayName );
    sioOutPrintf( sos, "end " );

    sioOutPrintf( sos, "/%s exch definefont pop\n\n", encodedFontName );
    return;
    }

int psIncludeFonts(	SimpleOutputStream *		sos,
			const PostScriptTypeList *	pstl )
    {
    const char *		fontName;
    PostScriptFace *		psf;

    psf= (PostScriptFace *)utilTreeGetFirst( pstl->pstlFaceTree, &fontName );
    while( psf )
	{
	if  ( psf->psfEmbed == PSembedUNKNOWN )
	    { utilPsGetEmbed( psf, pstl->pstlFontDirectory );	}

	if  ( psf->psfEmbed > 0 )
	    {
	    SimpleInputStream *	sisFile= (SimpleInputStream *)0;
	    SimpleInputStream *	sisPfb= (SimpleInputStream *)0;
	    SimpleInputStream *	sis= (SimpleInputStream *)0;

	    const AfmFontInfo *	afi= psf->psfAfi;

	    sisFile= sioInStdioOpen( psf->psfFontFileName );
	    if  ( ! sisFile )
		{ SXDEB(psf->psfFontFileName,sisFile); return -1;	}
	    sis= sisFile;

	    sioOutPrintf( sos, "\n" );
	    sioOutPrintf( sos, "%%%%%%%%{   F_%d:\n", afi->afiFaceNumber );
	    sioOutPrintf( sos, "%%%%%%%%    %s\n", afi->afiFullName );
	    sioOutPrintf( sos, "%%%%%%%% :: %s\n", psf->psfFontFileName );
	    sioOutPrintf( sos, "%%%%%%%%\n" );

	    sioOutPrintf( sos, "%%%%BeginFont %s\n",
					    afi->afiFontName );

	    switch( psf->psfEmbed )
		{
		int		c;
		int		atBol;

		case PSembedBTOA:
		    sisPfb= sioInPfbOpen( sisFile );
		    if  ( ! sisPfb )
			{
			SXDEB(psf->psfFontFileName,sisPfb);
			sioInClose( sisFile );
			return -1;
			}

		    sis= sisPfb;

		    /*FALLTHROUGH*/
		case PSembedCOPY:
		    atBol= 1;
		    while( ( c= sioInGetByte( sis ) ) != EOF )
			{
			sioOutPutByte( c, sos );
			atBol= c == '\n';
			}

		    if  ( ! atBol )
			{ sioOutPutByte( '\n', sos );	}

		    break;

		case PSembedTTFTO42:
		    if  ( psTtfToPt42( sos, psf->psfFontFileName, sisFile ) )
			{
			SDEB(psf->psfFontFileName);
			sioInClose( sisFile );
			return -1;
			}
		    break;

		case PSembedTTCTO42:
		    if  ( psTtcToPt42( sos,
				    psf->psfFontFileName,
				    psf->psfFontFileIndex,
				    sisFile ) )
			{
			SDEB(psf->psfFontFileName);
			sioInClose( sisFile );
			return -1;
			}
		    break;

		default:
		    LDEB(psf->psfEmbed);
		    sioInClose( sisFile );
		    return -1;
		}

	    sioOutPrintf( sos, "%%%%EndFont %s\n", afi->afiFontName );
	    sioOutPrintf( sos, "%%%%%%%%}   F_%d:\n", afi->afiFaceNumber );

	    if  ( sisPfb )
		{ sioInClose( sisPfb );	}
	    if  ( sisFile )
		{ sioInClose( sisFile );	}
	    }

	psf= (PostScriptFace *)utilTreeGetNext( pstl->pstlFaceTree, &fontName );
	}

    return 0;
    }

/************************************************************************/
/*									*/
/*  Emit a standard font encoding page: Mapping all positions to their	*/
/*  respective standard unicode names.					*/
/*									*/
/*  1)  Store an array with the desired name in the current dictionary.	*/
/*  2)  Store '.notdef' in all positions of the array.			*/
/*  3)  Store glyph names in the array.					*/
/*									*/
/************************************************************************/

static void psEmitStandardFontEncodingPage(	SimpleOutputStream *	sos,
						int			page )
    {
    int			i;
    char		name[20];

    sprintf( name, "CP_%d", page );

    /*  1  */
    sioOutPrintf( sos, "/%s 256 array def\n", name );

    /*  2  */
    sioOutPrintf( sos, "0 1 255 { %s exch /.notdef put } for %s\n",
							    name, name );

    /*  3  */
    for ( i= 0; i < UPP; i++ )
	{
	int	code= page* UPP+ i;

	if  ( ucdIsUtf16( code ) )
	    {
	    sioOutPrintf( sos, "  dup %3d /%s put\n",
					i, psUnicodeToGlyphName( code ) );
	    }
	}

    sioOutPrintf( sos, "readonly\n\n" );
    }

/************************************************************************/
/*									*/
/*  Overwrite te non-standard glyphs in a page of the font encoding.	*/
/*									*/
/************************************************************************/

static void psListNonStdPage(	SimpleOutputStream *		sos,
				const AfmFontInfo *		afi,
				int				page )
    {
    int		i;

    for ( i= 0; i < UPP; i++ )
	{
	int		code= page* UPP+ i;

	if  ( ucdIsUtf16( code ) )
	    {
	    int				glyph;

	    glyph= utilIndexMappingGet( &(afi->afiUnicodeToGlyphMapping), code );
	    if  ( glyph < 0 )
		{ /*LDEB(glyph);*/	}
	    else{
		const AfmCharMetric *	acm= afi->afiMetrics[glyph];
		const char *		stdName= psUnicodeToGlyphName( code );

		if  ( acm && acm->acmN && acm->acmN[0]	&&
		      strcmp( stdName, acm->acmN )		)
		    {
		    const AfmCodeList *	acl= &(acm->acmUnicodeCodeList);
		    int			c;

		    for ( c= 0; c < acl->aclCodeCount; c++ )
			{
			if  ( acl->aclCodes[c] == code )
			    {
			    sioOutPrintf( sos, "  dup %d /%s put\n",
							    i,  acm->acmN );
			    break;
			    }
			}
		    }
		}
	    }
	}
    }

static void psSetEncodedFontName(	char *			encodedFontName,
					const PostScriptFace *	psf,
					int			page )
    {
    const AfmFontInfo *		afi= psf->psfAfi;

    sprintf( encodedFontName, "F_%d_%02X", afi->afiFaceNumber, page );
    }

void psSetFontName(		char *			fontName,
				const AfmFontInfo *	afi )
    { sprintf( fontName, "F_%d", afi->afiFaceNumber ); }

static void psListFontEncodingPages(
				SimpleOutputStream *		sos,
				const PostScriptFace *		psf )
    {
    const AfmFontInfo *		afi= psf->psfAfi;
    int				page;
    int				done= 0;

    for ( page= 0; page < UNPAGE; page++ )
	{
	if  ( psf->psfPageUsed[page] )
	    {
	    char		encodingArrayName[20];
	    char		encodedFontName[20];

	    psSetEncodedFontName( encodedFontName, psf, page );

	    if  ( psf->psfPageNonStd[page] )
		{
		sprintf( encodingArrayName, "FCP_%d_%02X",
						afi->afiFaceNumber, page );

		sioOutPrintf( sos, "/%s 256 array def\n", encodingArrayName );
		sioOutPrintf( sos, "CP_%d %s copy\n", page, encodingArrayName );

		psListNonStdPage( sos, afi, page );

		sioOutPrintf( sos, "readonly\n\n" );
		}
	    else{ sprintf( encodingArrayName, "CP_%d", page );	}

	    psDefineEncodedFont( sos, afi,
				    encodedFontName, encodingArrayName );
	    done++;
	    }
	}

    sioOutPrintf( sos, "\n" );
    }

void psSelectFontProcedures(	SimpleOutputStream *		sos,
				const PostScriptTypeList *	pstl,
				int				allFonts )
    {
    const char *		fontName;
    PostScriptFace *		psf;
    int				page;

    int				faceNumber= 1;

    for ( page= 0; page < UNPAGE; page++ )
	{
	if  ( pstl->pstlPageUsed[page] )
	    { psEmitStandardFontEncodingPage( sos, page );	}
	}

    /*  2  */
    psf= (PostScriptFace *)utilTreeGetFirst( pstl->pstlFaceTree, &fontName );
    while( psf )
	{
	if  ( psf->psfAppearsInText || allFonts )
	    {
	    char		compositeName[20];
	    const AfmFontInfo *	afi= psf->psfAfi;
	    int			childCount;

	    int			encoding[UNPAGE];

	    for ( page= 0; page < UNPAGE; page++ )
		{ encoding[page]= 0;	}

	    sprintf( compositeName, "CF_%d", afi->afiFaceNumber );

	    sioOutPrintf( sos, "\n" );
	    sioOutPrintf( sos, "%%%%%%%%    %s:\n", compositeName );
	    sioOutPrintf( sos, "%%%%%%%%    %s\n", afi->afiFullName );
	    sioOutPrintf( sos, "%%%%%%%% -> %s\n", afi->afiFontName );
	    sioOutPrintf( sos, "%%%%%%%%\n\n" );

	    psListFontEncodingPages( sos, psf );

	    sioOutPrintf( sos, "/%s dup <<\n", compositeName );
	    sioOutPrintf( sos, "  /FontType 0\n" );
	    sioOutPrintf( sos, "  /FMapType 2\n" ); /* 8/8 */
	    sioOutPrintf( sos, "  /FontMatrix [ 1 0 0 1 0 0 ]\n" );
	    sioOutPrintf( sos, "  /FDepVector [\n" );

	    childCount= 0;
	    for ( page= 0; page < UNPAGE; page++ )
		{
		if  ( psf->psfPageUsed[page] )
		    {
		    char	encodedFontName[20];

		    psSetEncodedFontName( encodedFontName, psf, page );
		    sioOutPrintf( sos, "    /%s findfont %% %d\n",
					    encodedFontName, childCount );
		    encoding[page]= childCount++;
		    }
		}

	    sioOutPrintf( sos, "  ]\n" );
	    sioOutPrintf( sos, "  /Encoding [" );

	    sioOutPrintf( sos, "\n%%  " );
	    for ( page= 0; page < 16; page++ )
		{ sioOutPrintf( sos, "  +%X", page );	}

	    for ( page= 0; page < UNPAGE; page++ )
		{
		if  ( page % 16 == 0 )
		    {
		    if  ( page > 0 )
			{ sioOutPrintf( sos, "   %% %X+..", (page- 1)/16 ); }

		    sioOutPrintf( sos, "\n   " );
		    }

		sioOutPrintf( sos, "%4d", encoding[page] );
		}

	    sioOutPrintf( sos, " ] %% %X+..\n", (page- 1)/16  );
	    sioOutPrintf( sos, ">> definefont\n" );

	    sioOutPrintf( sos, "/F_%d\t{\n", afi->afiFaceNumber );
	    sioOutPrintf( sos, "  6 array identmatrix\n" );
	    sioOutPrintf( sos, "  dup 0 3 index put\n" );
	    sioOutPrintf( sos, "  dup 3 3 index neg put\n" );
	    sioOutPrintf( sos, "  /%s exch selectfont\n", compositeName );
	    sioOutPrintf( sos, "  } bind def\n" );
	    }

	psf= (PostScriptFace *)utilTreeGetNext( pstl->pstlFaceTree, &fontName );
	faceNumber++;
	}

    sioOutPrintf( sos, "\n" );

    return;
    }

