/************************************************************************/
/*									*/
/*  Utility routines to build font configuration.			*/
/*									*/
/************************************************************************/

#   include	"utilPsConfig.h"

#   include	<stdio.h>
#   include	<stdlib.h>
#   include	<string.h>
#   include	<ctype.h>

#   include	<locale.h>

#   include	<reg.h>

#   include	"sioStdio.h"
#   include	"sioPfb.h"
#   include	"sioPipe.h"
#   include	"appSystem.h"
#   include	"psTtf.h"
#   include	"psBuildConfigFiles.h"

#   include	<appDebugon.h>

#   ifdef __VMS
    static const char * const		afmExtension= "AFM";
    static const char * const		pfbExtension= "PFB";
    static const char * const		pfaExtension= "PFA";
#   else
    static const char * const		afmExtension= "afm";
    static const char * const		pfbExtension= "pfb";
    static const char * const		pfaExtension= "pfa";
#   endif
    static const char * const		ttfExtension= "ttf";
    static const char * const		TTFExtension= "TTF";

static const char * psFontNamePattern=
			"^/FontName[ \t]*/([^ \t]*)[ \t][ \t]*def";

static const char * PS_LocalAfmDir= "localfonts";

/************************************************************************/
/*									*/
/*  Find the font name for a Postcript type 1 font.			*/
/*									*/
/************************************************************************/

static char * psFontNameFromPf(	SimpleInputStream *	sisFont,
				regProg *		prog,
				const char *		filename )
    {
    char *		rval= (char *)0;
    char		scratch[550+1];

    while( sioInGetString( scratch, sizeof(scratch)-1, sisFont ) )
	{
	ExpressionMatch		em;

	scratch[sizeof(scratch)-1]= '\0';

	if  ( regFindLeftToRight( &em, prog, scratch, 0, strlen( scratch ) ) )
	    {
	    int		from;
	    int		past;
	    int		l;

	    regGetMatch( &from, &past, &em, 0 );

	    l= past- from;

	    rval= malloc( l+ 1 );
	    if  ( ! rval )
		{ LXDEB(l,rval);					}
	    else{ strncpy( rval, scratch+ from, l )[l]= '\0'; }

	    break;
	    }
	}

    if  ( ! rval )
	{ SXDEB(filename,rval);	}

    return rval;
    }

static int psFillAfiForPf(	AfmFontInfo *		afi,
				SimpleInputStream *	sisFont,
				regProg *		prog,
				const char *		filename,
				const char *		afmCommand )
    {
    int		rval= 0;
    char *	fontname= (char *)0;
    char *	command= (char *)0;

    const int	deferMetrics= 0;

    SimpleInputStream *	sisCmd= (SimpleInputStream *)0;

    fontname= psFontNameFromPf( sisFont, prog, filename );
    if  ( ! fontname )
	{ XDEB(fontname); rval= -1; goto ready;	}

    if  ( strlen( afmCommand )+ strlen( fontname ) > sizeof(command)- 1 )
	{ SDEB(fontname); rval= -1; goto ready;	}

    command= malloc( strlen( afmCommand )+ strlen( fontname )+ 1 );
    if  ( ! command )
	{ PDEB(command); rval= -1; goto ready;	}
    sprintf( command, afmCommand, fontname );
    sisCmd= sioInPipeOpen( command );
    if  ( ! sisCmd )
	{ SXDEB(command,sisCmd); rval= -1; goto ready;	}

    rval= psAfmReadAfm( sisCmd, afi, deferMetrics );
    if  ( rval  )
	{ SLDEB(command,rval); rval= -1; }

    if  ( afi->afiFontFileName )
	{ free( afi->afiFontFileName );	}
    afi->afiFontFileName= strdup( filename );

    if  ( psGetUnicodesFromGlyphNames( afi ) )
	{ SDEB(afi->afiFullName);	}
    if  ( psResolveFallbackGlyph( afi ) )
	{ SDEB(afi->afiFullName);	}

  ready:

    if  ( command )
	{ free( command );	}
    if  ( sisCmd )
	{ sioInClose( sisCmd );	}
    if  ( fontname )
	{ free( fontname );	}

    return rval;
    }

static char * psFontNameFromPfa(	regProg *		prog,
					const char *		filename )
    {
    char *		rval= (char *)0;
    SimpleInputStream *	sisFile= (SimpleInputStream *)0;

    sisFile= sioInStdioOpen( filename );
    if  ( ! sisFile )
	{ SXDEB(filename,sisFile); goto ready;	}

    rval= psFontNameFromPf( sisFile, prog, filename );
    if  ( ! rval )
	{ SXDEB(filename,rval);		}

  ready:
    if  ( sisFile )
	{ sioInClose( sisFile );	}

    return rval;
    }

static int psFillAfiForPfa(	AfmFontInfo *		afi,
				regProg *		prog,
				const char *		filename,
				const char *		afmCommand )
    {
    int			rval= 0;
    SimpleInputStream *	sisFile= (SimpleInputStream *)0;

    sisFile= sioInStdioOpen( filename );
    if  ( ! sisFile )
	{ SXDEB(filename,sisFile); rval= -1; goto ready;	}

    rval= psFillAfiForPf( afi, sisFile, prog, filename, afmCommand );
    if  ( rval )
	{ SLDEB(filename,rval);		}

  ready:
    if  ( sisFile )
	{ sioInClose( sisFile );	}

    return rval;
    }

static char * psFontNameFromPfb(	regProg *		prog,
					const char *		filename )
    {
    char *		rval= (char *)0;
    SimpleInputStream *	sisFile= (SimpleInputStream *)0;
    SimpleInputStream *	sisFont= (SimpleInputStream *)0;

    sisFile= sioInStdioOpen( filename );
    if  ( ! sisFile )
	{ SXDEB(filename,sisFile); goto ready;	}

    sisFont= sioInPfbOpen( sisFile );
    if  ( ! sisFont )
	{ SXDEB(filename,sisFont); goto ready;	}

    rval= psFontNameFromPf( sisFont, prog, filename );
    if  ( ! rval )
	{ SXDEB(filename,rval);		}

  ready:
    if  ( sisFont )
	{ sioInClose( sisFont );	}
    if  ( sisFile )
	{ sioInClose( sisFile );	}

    return rval;
    }

static int psFillAfiForPfb(	AfmFontInfo *		afi,
				regProg *		prog,
				const char *		filename,
				const char *		afmCommand )
    {
    int			rval= 0;
    SimpleInputStream *	sisFile= (SimpleInputStream *)0;
    SimpleInputStream *	sisFont= (SimpleInputStream *)0;

    sisFile= sioInStdioOpen( filename );
    if  ( ! sisFile )
	{ SXDEB(filename,sisFile); rval= -1; goto ready;	}

    sisFont= sioInPfbOpen( sisFile );
    if  ( ! sisFont )
	{ SXDEB(filename,sisFont); rval= -1; goto ready;	}

    rval= psFillAfiForPf( afi, sisFont, prog, filename, afmCommand );
    if  ( rval )
	{ SLDEB(filename,rval);		}

  ready:
    if  ( sisFont )
	{ sioInClose( sisFont );	}
    if  ( sisFile )
	{ sioInClose( sisFile );	}

    return rval;
    }

/************************************************************************/
/*									*/
/*  Find the font name for a TTF file. This is an expensive way. That	*/
/*  is not a problem as we rarely do this.				*/
/*									*/
/************************************************************************/

static int psFillAfiForTtf(	AfmFontInfo *		afi,
				const char *		filename )
    {
    int			rval= 0;
    SimpleInputStream *	sisFile= (SimpleInputStream *)0;

    sisFile= sioInStdioOpen( filename );
    if  ( ! sisFile )
	{ SXDEB(filename,sisFile); rval= -1; goto ready;	}

    if  ( psTtfToAfi( afi, filename, sisFile ) )
	{ LDEB(1); rval= -1; goto ready;	}

  ready:

    if  ( sisFile )
	{ sioInClose( sisFile );	}

    return rval;
    }

static char * psFontNameFromTtf(	const char *		filename )
    {
    char *		rval= (char *)0;
    AfmFontInfo		afi;

    psInitAfmFontInfo( &afi );

    if  ( psFillAfiForTtf( &afi, filename ) )
	{ LDEB(1); goto ready;	}

    rval= strdup( afi.afiFontName );
    if  ( ! rval )
	{ XDEB(rval);	}

  ready:
    psCleanAfmFontInfo( &afi );

    return rval;
    }

/************************************************************************/
/*									*/
/*  Scan a list of files and issue GhostScript compatible Fontmap lines	*/
/*  that can be added to a GhostScript Fontmap file. The purpose of	*/
/*  this function is to make the fonts known to GhostScript. This means	*/
/*  that we cannot use GhostScript yet.					*/
/*									*/
/************************************************************************/

int psFontmapForFiles(		SimpleOutputStream *	sosOut,
				int			fileCount,
				char **			fileNames )
    {
    int			rval= 0;
    int			file;

    regProg *		prog= (regProg *)0;
    char *		fontname= (char *)0;
    const int		options= 0;

    prog= regCompile( psFontNamePattern, options );
    if  ( ! prog )
	{ XDEB(prog); rval= -1; goto ready;	}

    for ( file= 0; file < fileCount; file++ )
	{
	char *		filename= fileNames[file];
	const char *	extension= appFileExtensionOfName( filename );

	if  ( ! extension )
	    { XDEB(extension); rval= -1; continue;	}

	if  ( ! strcmp( extension, pfaExtension ) )
	    { fontname= psFontNameFromPfa( prog, filename );	}

	if  ( ! strcmp( extension, pfbExtension ) )
	    { fontname= psFontNameFromPfb( prog, filename );	}

	if  ( ! strcmp( extension, ttfExtension ) ||
	      ! strcmp( extension, TTFExtension ) )
	    { fontname= psFontNameFromTtf( filename );		}

	if  ( fontname )
	    {
	    sioOutPrintf( sosOut, "/%s (%s) ;\n", fontname, filename );
	    free( fontname ); fontname= (char *)0;
	    }
	else{ SDEB(filename); rval= -1;	}
	}

  ready:

    if  ( fontname )
	{ free( fontname );	}
    if  ( prog )
	{ free( prog );	}

    return rval;
    }

/************************************************************************/

typedef struct AfmDirBuilder
    {
    char			adbAfmDirectory[300+1];
    regProg *			adbFontNameProg;
    char			adbAfmCommand[300+1];
    PostScriptFontList *	adbPostScriptFontList;
    } AfmDirBuilder;

static int psSaveAfmFile(	AfmFontInfo *		afi,
				AfmDirBuilder *		adb )
    {
    int				rval= 0;
    char *			relative= (char *)0;
    char			absolute[400];
    const int			relativeIsFile= 0;
    char *			s;

    int				rellen=  strlen( afi->afiFontName );

    relative= malloc( rellen+ 4+ 1 );
    if  ( ! relative )
	{ LPDEB(rellen,relative); rval= -1; goto ready;	}

    sprintf( relative, "%s.afm", afi->afiFontName );
    s= relative;
    while( *s )
	{
	if  ( isspace( *s ) )
	    { *s= '-';	}
	s++;
	}
    if  ( appAbsoluteName( absolute, sizeof(absolute)- 1, relative,
				relativeIsFile, adb->adbAfmDirectory ) < 0 )
	{ SDEB(relative); rval= -1; goto ready;	}

    afi->afiAfmFileName= strdup( absolute );
    if  ( ! afi->afiAfmFileName )
	{ XDEB(afi->afiAfmFileName); rval= -1; goto ready;	}

    if  ( psPostScriptFontListAddInfo( adb->adbPostScriptFontList, afi ) )
	{ LDEB(1); rval= -1; goto ready;	}

  ready:

    if  ( relative )
	{ free( relative );	}

    return rval;
    }

static int psAfmFromPfaFile(	const char *	filename,
				void *		through )
    {
    int			rval= 0;
    AfmDirBuilder *	adb= (AfmDirBuilder *)through;

    AfmFontInfo *	afi= (AfmFontInfo *)0;

    afi= malloc( sizeof(AfmFontInfo) );
    if  ( ! afi )
	{ XDEB(afi); rval= -1; goto ready;	}
    psInitAfmFontInfo( afi );

    /* Broken symlinks */
    if  ( appTestFileExists( filename ) )
	{ goto ready;	}

    if  ( psFillAfiForPfa( afi, adb->adbFontNameProg,
					filename, adb->adbAfmCommand ) )
	{ SDEB(filename); goto ready;	} /* do not fail */

    if  ( psSaveAfmFile( afi, adb ) )
	{ SDEB(adb->adbAfmDirectory); rval= -1; goto ready;	}

    afi= (AfmFontInfo *)0;

  ready:

    if  ( afi )
	{ psFreeAfmFontInfo( afi );	}

    return rval;
    }

static int psAfmFromPfbFile(	const char *	filename,
				void *		through )
    {
    int			rval= 0;
    AfmDirBuilder *	adb= (AfmDirBuilder *)through;

    AfmFontInfo *	afi= (AfmFontInfo *)0;

    afi= malloc( sizeof(AfmFontInfo) );
    if  ( ! afi )
	{ XDEB(afi); rval= -1; goto ready;	}
    psInitAfmFontInfo( afi );

    /* Broken symlinks */
    if  ( appTestFileExists( filename ) )
	{ goto ready;	}

    if  ( psFillAfiForPfb( afi, adb->adbFontNameProg,
					filename, adb->adbAfmCommand ) )
	{ SDEB(filename); goto ready;	} /* do not fail */

    if  ( psSaveAfmFile( afi, adb ) )
	{ SDEB(adb->adbAfmDirectory); rval= -1; goto ready;	}

    afi= (AfmFontInfo *)0;

  ready:

    if  ( afi )
	{ psFreeAfmFontInfo( afi );	}

    return rval;
    }

static int psAfmFromTtfFile(	const char *	filename,
				void *		through )
    {
    int			rval= 0;
    AfmDirBuilder *	adb= (AfmDirBuilder *)through;

    AfmFontInfo *	afi= (AfmFontInfo *)0;

    afi= malloc( sizeof(AfmFontInfo) );
    if  ( ! afi )
	{ XDEB(afi); rval= -1; goto ready;	}
    psInitAfmFontInfo( afi );

    /* Broken symlinks */
    if  ( appTestFileExists( filename ) )
	{ goto ready;	}

    if  ( psFillAfiForTtf( afi, filename ) )
	{ SDEB(filename); goto ready;	} /* do not fail */

    if  ( psSaveAfmFile( afi, adb ) )
	{ SDEB(adb->adbAfmDirectory); rval= -1; goto ready;	}

    afi= (AfmFontInfo *)0;

  ready:

    if  ( afi )
	{ psFreeAfmFontInfo( afi );	}

    return rval;
    }

/************************************************************************/

static int psStartAfmDirBuilder(	AfmDirBuilder *		adb,
					PostScriptFontList *	psfl,
					const char *		afmDirectory,
					const char *		psDirectory )
    {
    const int		relativeIsFile= 0;
    const int		options= 0;

    const char *	afmBase= "gs -sthefont=%s -q %s/gsfontmetrics.ps -c quit";
    adb->adbFontNameProg= (regProg *)0;
    adb->adbPostScriptFontList= psfl;

    if  ( appAbsoluteName( adb->adbAfmDirectory,
			sizeof(adb->adbAfmDirectory)- 1,
			PS_LocalAfmDir, relativeIsFile, afmDirectory ) < 0 )
	{ SDEB(PS_LocalAfmDir); return -1;	}

    adb->adbFontNameProg= regCompile( psFontNamePattern, options );
    if  ( ! adb->adbFontNameProg )
	{ XDEB(adb->adbFontNameProg); return -1;	}

    if  ( strlen( psDirectory )+ strlen( afmBase ) >
					    sizeof(adb->adbAfmCommand)- 1 )
	{ SDEB(psDirectory); return -1;	}
    sprintf( adb->adbAfmCommand, afmBase, "%s", psDirectory );

    return 0;
    }

/************************************************************************/
/*									*/
/*  Build an afm directory: Include all fonts that are on the		*/
/*  GhostScript library path.						*/
/*									*/
/************************************************************************/

int psGSLibAfmDirectory(	PostScriptFontList *	psfl,
				const char *		afmDirectory,
				const char *		psDirectory )
    {
    int				rval= 0;
    SimpleInputStream *		sisCmd= (SimpleInputStream *)0;
    char *			libCommand= (char *)0;

    AfmDirBuilder		adb;

    const char *	libBase= "gs -q %s/gslibpath.ps -c quit";

    if  ( psStartAfmDirBuilder( &adb, psfl, afmDirectory, psDirectory ) )
	{ LDEB(1); rval= -1; goto ready;	}

    libCommand= malloc( strlen( psDirectory )+ strlen( libBase )+ 1 );
    if  ( ! libCommand )
	{ PDEB(libCommand); rval= -1; goto ready;	}
    sprintf( libCommand, libBase, psDirectory );

    sisCmd= sioInPipeOpen( libCommand );
    if  ( ! sisCmd )
	{ SXDEB(libCommand,sisCmd); rval= -1; goto ready;	}

    setlocale( LC_NUMERIC, "C" );

    while( sioInGetString( libCommand, sizeof(libCommand)-1, sisCmd ) )
	{
	int		l;

	libCommand[sizeof(libCommand)-1]= '\0';
	l= strlen( libCommand );
	if  ( l < 2 )
	    { LDEB(l); rval= -1; goto ready;	}
	if  ( libCommand[l-1] != '\n' )
	    { SDEB(libCommand); rval= -1; goto ready;	}
	libCommand[l-1]= '\0';

	if  ( ! appFileNameIsAbsolute( libCommand ) )
	    { continue;	}
	if  ( appTestDirectory( libCommand ) )
	    { continue;	}

	l= appForAllFiles( libCommand, pfaExtension, &adb, psAfmFromPfaFile );
	if  ( l )
	    { SDEB(libCommand); rval= -1;	}
	l= appForAllFiles( libCommand, pfbExtension, &adb, psAfmFromPfbFile );
	if  ( l )
	    { SDEB(libCommand); rval= -1;	}
	l= appForAllFiles( libCommand, ttfExtension, &adb, psAfmFromTtfFile );
	if  ( l )
	    { SDEB(libCommand); rval= -1;	}
	l= appForAllFiles( libCommand, TTFExtension, &adb, psAfmFromTtfFile );
	if  ( l )
	    { SDEB(libCommand); rval= -1;	}
	}

  ready:

    setlocale( LC_NUMERIC, "" );

    if  ( libCommand )
	{ free( libCommand );	}
    if  ( sisCmd )
	{ sioInClose( sisCmd );	}

    if  ( adb.adbFontNameProg )
	{ regFree( adb.adbFontNameProg );	}

    return rval;
    }

/************************************************************************/
/*									*/
/*  Build an afm directory file for a particular font file.		*/
/*									*/
/************************************************************************/

int psAfmForFontFiles(	PostScriptFontList *	psfl,
			int			fileCount,
			char **			fileNames,
			const char *		afmDirectory,
			const char *		psDirectory )
    {
    int				rval= 0;
    int				file;
    char			pwd[500+1];

    char			absolute[1000+1];
    const int			relativeIsFile= 0;

    AfmDirBuilder		adb;

    if  ( psStartAfmDirBuilder( &adb, psfl, afmDirectory, psDirectory ) )
	{ LDEB(1); rval= -1; goto ready;	}

    if  ( appCurrentDirectory( pwd, sizeof(pwd)- 1 ) < 0 )
	{ LDEB(1); rval= -1; goto ready;	}

    setlocale( LC_NUMERIC, "C" );

    for ( file= 0; file < fileCount; file++ )
	{
	char *		filename= fileNames[file];
	const char *	extension= appFileExtensionOfName( filename );

	if  ( ! appFileNameIsAbsolute( filename ) )
	    {
	    int		l;

	    l= appAbsoluteName( absolute, sizeof(absolute)-1, filename,
							relativeIsFile, pwd );
	    if  ( l < 0 )
		{ LDEB(l); rval= -1; continue;	}

	    filename= absolute;
	    }

	if  ( ! extension )
	    { XDEB(extension); rval= -1; continue;	}

	if  ( ! strcmp( extension, pfaExtension ) )
	    {
	    if  ( psAfmFromPfaFile( filename, &adb ) )
		{ SDEB(filename); rval= -1;	}

	    continue;
	    }

	if  ( ! strcmp( extension, pfbExtension ) )
	    {
	    if  ( psAfmFromPfbFile( filename, &adb ) )
		{ SDEB(filename); rval= -1;	}

	    continue;
	    }

	if  ( ! strcmp( extension, ttfExtension ) ||
	      ! strcmp( extension, TTFExtension ) )
	    {
	    if  ( psAfmFromTtfFile( filename, &adb ) )
		{ SDEB(filename); rval= -1;	}

	    continue;
	    }

	SDEB(filename); rval= -1; continue;
	}

  ready:

    setlocale( LC_NUMERIC, "" );

    if  ( adb.adbFontNameProg )
	{ regFree( adb.adbFontNameProg );	}

    return 0;
    }

int psAfmToGSFontmap(	SimpleOutputStream *	sosFontDir,
			const char *		afmFileName,
			SimpleInputStream *	sisAfm )
    {
    int			rval= 0;
    AfmFontInfo		afi;
    const char *	filename= "FontFileName";

    const int		deferMetrics= 1;

    psInitAfmFontInfo( &afi );

    setlocale( LC_NUMERIC, "C" );

    if  ( psAfmReadAfm( sisAfm, &afi, deferMetrics ) )
	{ SDEB(afmFileName); rval= -1; goto ready;	}

    if  ( afi.afiFontFileName )
	{ filename= afi.afiFontFileName;	}

    sioOutPrintf( sosFontDir, "/%s (%s) ;\n", afi.afiFontName, filename );

  ready:

    setlocale( LC_NUMERIC, "" );

    psCleanAfmFontInfo( &afi );

    return rval;
    }
