/************************************************************************/
/*									*/
/*  Evaluate TOC fields.						*/
/*									*/
/************************************************************************/

#   include	"docBufConfig.h"

#   include	<stdlib.h>
#   include	<string.h>
#   include	<stdio.h>

#   include	<appDebugon.h>

#   include	<appUnit.h>

#   include	"docBuf.h"
#   include	"docParaString.h"
#   include	"docPageGrid.h"
#   include	"docEvalField.h"
#   include	"docEdit.h"
#   include	"docTocField.h"

typedef struct TocEntry
    {
    int				teLevel;
    const DocumentField *	teField;
    DocumentSelection		teDsInside;
    DocumentSelection		teDsAround;
    int				tePart0;
    int				tePart1;
    const char *		teMarkName;
    int				teMarkSize;
    int				teNumbered;
    } TocEntry;

static void docInitTocEntry(	TocEntry *	te )
    {
    te->teLevel= PPoutlineBODYTEXT;
    te->teField= (const DocumentField *)0;
    docInitDocumentSelection( &(te->teDsInside) );
    docInitDocumentSelection( &(te->teDsAround) );
    te->tePart0= 0;
    te->tePart1= 0;
    te->teMarkName= (const char *)0;
    te->teMarkSize= 0;
    te->teNumbered= 1;
    }

/************************************************************************/

typedef struct CalculateToc
    {
    BufferDocument *		ctBdToc;
    BufferDocument *		ctBdDoc;
    BufferItem *		ctSectBi;
    DocumentField *		ctDfTocTo;
    TocField			ctTocField;
    BlockFrame			ctBlockFrame;
    ParagraphProperties		ctRefPP;
    TextAttribute		ctTextAttribute;
    int				ctDefaultTextAttributeNumber;
    const DocumentStyle *	ctLevelStyles[PPoutline_COUNT];
    int				ctLevelAttributeNumbers[PPoutline_COUNT];

    TocEntry *			ctEntries;
    int				ctEntryCount;

    unsigned char *		ctStyleLevels;
    int				ctStyleLevelCount;
    } CalculateToc;

static void docInitCalculateToc(	CalculateToc *	ct )
    {
    int		i;

    ct->ctBdToc= (BufferDocument *)0;
    ct->ctBdDoc= (BufferDocument *)0;
    ct->ctSectBi= (BufferItem *)0;
    ct->ctDfTocTo= (DocumentField *)0;

    docInitTocField( &(ct->ctTocField) );
    docLayoutInitBlockFrame( &(ct->ctBlockFrame) );
    docInitParagraphProperties( &(ct->ctRefPP) );
    utilInitTextAttribute( &(ct->ctTextAttribute) );
    ct->ctDefaultTextAttributeNumber= -1;
    for ( i= 0; i < PPoutline_COUNT; i++ )
	{
	ct->ctLevelStyles[i]= (const DocumentStyle *)0;
	ct->ctLevelAttributeNumbers[i]= -1;
	}

    ct->ctEntries= (TocEntry *)0;
    ct->ctEntryCount= 0;

    ct->ctStyleLevels= (unsigned char *)0;
    ct->ctStyleLevelCount= 0;
    return;
    }

static void docCleanCalculateToc(	CalculateToc *	ct )
    {
    if  ( ct->ctBdToc )
	{ docFreeDocument( ct->ctBdToc );	}

    docCleanTocField( &(ct->ctTocField) );
    docCleanParagraphProperties( &(ct->ctRefPP) );

    if  ( ct->ctEntries )
	{ free( ct->ctEntries );	}

    if  ( ct->ctStyleLevels )
	{ free( ct->ctStyleLevels );	}

    return;
    }

/************************************************************************/
/*									*/
/*  Sort Toc entries. Sort by increasing paragraph number,		*/
/*  decreasing size and increasing position.				*/
/*									*/
/************************************************************************/

static int docCompareTocEntries(	const void *	voidte1,
					const void *	voidte2	)
    {
    const TocEntry *	te1= (const TocEntry *)voidte1;
    const TocEntry *	te2= (const TocEntry *)voidte2;
    int			parts1;
    int			parts2;

    if  ( te1->teField->dfHeadPosition.epParaNr >
	  te2->teField->dfHeadPosition.epParaNr )
	{ return  1;	}

    if  ( te1->teField->dfHeadPosition.epParaNr <
	  te2->teField->dfHeadPosition.epParaNr )
	{ return -1;	}

    parts1= te1->tePart1- te1->tePart0;
    parts2= te2->tePart1- te2->tePart0;
    if  ( parts1 < parts2 )
	{ return  1;	}
    if  ( parts1 > parts2 )
	{ return -1;	}

    if  ( te1->tePart0 > te2->tePart0 )
	{ return  1;	}
    if  ( te1->tePart0 < te2->tePart0 )
	{ return -1;	}

    return 0;
    }

/************************************************************************/
/*									*/
/*  Evaluate TOC fields.						*/
/*  This is a two level process: First build the TOC from REF and	*/
/*  PAGEREF fields, then let those be evaluated by the text field	*/
/*  evaluator.								*/
/*									*/
/*  Expertimenting with MS-Word shows that the format of the table of	*/
/*  contents is controlled by the format of the first entry: the	*/
/*  paragraph in which the TOC field starts. Things are not THAT simple	*/
/*  however: To be investigated!					*/
/*									*/
/************************************************************************/

static int docParaInsertTocEntry(	CalculateToc *		ct,
					int			makeHyperlinks,
					const TocEntry *	te,
					const SelectionScope *	ss,
					BufferItem *		paraBi )
    {
    int			textAttrNr= ct->ctLevelAttributeNumbers[te->teLevel];
    DocumentField *	dfParent= ct->ctDfTocTo;
    DocumentField *	dfHyper= (DocumentField *)0;
    DocumentField *	dfRef= (DocumentField *)0;
    DocumentField *	dfPageref= (DocumentField *)0;

    int			stroffShift;

    TextParticule *	tp;

    DocumentFieldList *	dfl= &(ct->ctBdToc->bdFieldList);

    int			part= paraBi->biParaParticuleCount;
    EditPosition	ep;

    ep.epParaNr= docNumberOfParagraph( paraBi );
    ep.epStroff= docParaStrlen( paraBi );

    /* HYPERLINK */
    if  ( makeHyperlinks )
	{
	const char *	fileName= (const char *)0;
	int		fileSize= 0;

	dfHyper= docClaimField( dfl );
	if  ( ! dfHyper )
	    { XDEB(dfHyper); return -1;	}

	tp= docMakeSpecialParticule( paraBi, part, ep.epStroff,
					    DOCkindFIELDSTART, textAttrNr );
	if  ( ! tp )
	    { XDEB(tp); return -1;	}
	if  ( part >= 0 )
	    { part++;	}
	ep.epStroff += tp->tpStrlen;

	tp->tpObjectNumber= dfHyper->dfFieldNumber;
	dfHyper->dfHeadPosition= ep;
	dfHyper->dfSelectionScope= *ss;

	dfHyper->dfKind= DOCfkHYPERLINK;
	if  ( docFieldSetHyperlink( dfHyper, fileName, fileSize,
					    te->teMarkName, te->teMarkSize ) )
	    { LDEB(te->teMarkSize); return -1;	}

	dfParent= dfHyper;
	}

    /* REF */
    dfRef= docClaimField( dfl );
    if  ( ! dfRef )
	{ XDEB(dfRef); return -1;	}

    tp= docMakeSpecialParticule( paraBi, part, ep.epStroff,
					    DOCkindFIELDSTART, textAttrNr );
    if  ( ! tp )
	{ XDEB(tp); return -1;	}
    if  ( part >= 0 )
	{ part++;	}
    ep.epStroff += tp->tpStrlen;

    tp->tpObjectNumber= dfRef->dfFieldNumber;
    dfRef->dfHeadPosition= ep;
    dfRef->dfSelectionScope= *ss;

    dfRef->dfKind= DOCfkREF;
    if  ( docFieldSetRef( dfRef, te->teMarkName, te->teMarkSize ) )
	{ LDEB(te->teMarkSize); return -1;	}

    if  ( docParaStringReplace( &stroffShift, paraBi, ep.epStroff, ep.epStroff,
								    "x", 1 ) )
	{ LDEB(1); return -1; }
    tp= docInsertTextParticule( paraBi, part, ep.epStroff, stroffShift,
						DOCkindSPAN, textAttrNr );
    if  ( ! tp )
	{ XDEB(tp); return -1;	}
    if  ( part >= 0 )
	{ part++;	}
    ep.epStroff += stroffShift;

    tp= docMakeSpecialParticule( paraBi, part, ep.epStroff,
					    DOCkindFIELDEND, textAttrNr );
    if  ( ! tp )
	{ XDEB(tp); return -1;	}
    if  ( part >= 0 )
	{ part++;	}
    ep.epStroff += tp->tpStrlen;

    tp->tpObjectNumber= dfRef->dfFieldNumber;
    docSetFieldEnd( dfRef, &ep );
    if  ( docAddChildToField( dfRef, dfParent ) )
	{ LDEB(1); return -1;	}

    if  ( te->teNumbered )
	{
	/* SEPARATOR */
	if  ( docParaStringReplace( &stroffShift, paraBi,
					ep.epStroff, ep.epStroff, " ", 1 ) )
	    { LDEB(1); return -1; }
	tp= docInsertTextParticule( paraBi, part, ep.epStroff, stroffShift,
						DOCkindTAB, textAttrNr );
	if  ( ! tp )
	    { XDEB(tp); return -1;	}
	if  ( part >= 0 )
	    { part++;	}
	ep.epStroff += stroffShift;

	/* PAGEREF */
	dfPageref= docClaimField( dfl );
	if  ( ! dfPageref )
	    { XDEB(dfPageref); return -1;	}

	tp= docMakeSpecialParticule( paraBi, part, ep.epStroff,
					    DOCkindFIELDSTART, textAttrNr );
	if  ( ! tp )
	    { XDEB(tp); return -1;	}
	if  ( part >= 0 )
	    { part++;	}
	ep.epStroff += tp->tpStrlen;

	tp->tpObjectNumber= dfPageref->dfFieldNumber;
	dfPageref->dfHeadPosition= ep;
	dfPageref->dfSelectionScope= *ss;

	dfPageref->dfKind= DOCfkPAGEREF;
	if  ( docFieldSetPageref( dfPageref, te->teMarkName, te->teMarkSize ) )
	    { LDEB(te->teMarkSize); return -1;	}

	if  ( docParaStringReplace( &stroffShift, paraBi,
					ep.epStroff, ep.epStroff, "x", 1 ) )
	    { LDEB(1); return -1; }
	tp= docInsertTextParticule( paraBi, part, ep.epStroff, stroffShift,
						DOCkindSPAN, textAttrNr );
	if  ( ! tp )
	    { XDEB(tp); return -1;	}
	if  ( part >= 0 )
	    { part++;	}
	ep.epStroff += stroffShift;

	tp= docMakeSpecialParticule( paraBi, part, ep.epStroff,
					    DOCkindFIELDEND, textAttrNr );
	if  ( ! tp )
	    { XDEB(tp); return -1;	}
	if  ( part >= 0 )
	    { part++;	}
	ep.epStroff += tp->tpStrlen;

	tp->tpObjectNumber= dfPageref->dfFieldNumber;
	docSetFieldEnd( dfPageref, &ep );
	if  ( docAddChildToField( dfPageref, dfParent ) )
	    { LDEB(1); return -1;	}
	}

    if  ( dfHyper )
	{
	tp= docMakeSpecialParticule( paraBi, part, ep.epStroff,
					    DOCkindFIELDEND, textAttrNr );
	if  ( ! tp )
	    { XDEB(tp); return -1;	}
	if  ( part >= 0 )
	    { part++;	}
	ep.epStroff += tp->tpStrlen;

	tp->tpObjectNumber= dfHyper->dfFieldNumber;
	docSetFieldEnd( dfHyper, &ep );
	if  ( docAddChildToField( dfHyper, ct->ctDfTocTo ) )
	    { LDEB(1); return -1;	}
	}

    /************************************************************/
    /*  Delete an empty particule at the end of the paragraph.	*/
    /************************************************************/
    part= paraBi->biParaParticuleCount- 1;
    tp= paraBi->biParaParticules+ part;
    if  ( tp->tpKind == DOCkindSPAN	&&
	  tp->tpStrlen == 0		)
	{ paraBi->biParaParticuleCount--;	}

    return 0;
    }

/************************************************************************/
/*									*/
/*  Does a field go to the TOC?						*/
/*									*/
/************************************************************************/

static int docGetTocEntry(	TocEntry *		te,
				const CalculateToc *	ct )
    {
    char			flag[2];

    if  ( te->teField->dfKind == DOCfkBOOKMARK )
	{
	BufferItem *	paraBi;

	if  ( ! ct->ctTocField.tfUseStyles		&&
	      ! ct->ctTocField.tfUseOutlineLevels	)
	    { return 1;	}

	if  ( docDelimitFieldInDoc( &(te->teDsInside), &(te->teDsAround),
					&(te->tePart0), &(te->tePart1),
					ct->ctBdDoc, te->teField ) )
	    { LDEB(te->teField->dfFieldNumber); return 1; }

	if  ( docIsIBarSelection( &(te->teDsInside) ) )
	    { return 1;	}

	paraBi= te->teDsInside.dsHead.dpBi;
	if  ( te->teDsInside.dsTail.dpBi != paraBi )
	    { return 1;	}

	/*  Holds any text? */
	{
	int			part;
	const TextParticule *	tp;

	tp= te->teDsInside.dsHead.dpBi->biParaParticules+ te->tePart0+ 1;
	for ( part= te->tePart0+ 1; part < te->tePart1; tp++, part++ )
	    {
	    if  ( tp->tpKind == DOCkindSPAN && tp->tpStrlen > 0 )
		{ break;	}
	    }

	if  ( part >= te->tePart1 )
	    { return 1;	}
	}

	if  ( docFieldGetBookmark( te->teField,
				    &(te->teMarkName), &(te->teMarkSize) ) )
	    { LDEB(te->teField->dfFieldNumber); return -1;	}

	if  ( ct->ctTocField.tfUseStyles			&&
	      paraBi->biParaStyle > 0				&&
	      paraBi->biParaStyle < ct->ctStyleLevelCount	)
	    {
	    te->teLevel= ct->ctStyleLevels[paraBi->biParaStyle];

	    if  ( te->teLevel <= ct->ctTocField.tfLevel1	&&
		  te->teLevel >= ct->ctTocField.tfLevel0	)
		{
		te->teNumbered= te->teLevel < ct->ctTocField.tfNLevel0	||
				te->teLevel > ct->ctTocField.tfNLevel1	;
		return 0;
		}
	    }

	if  ( ct->ctTocField.tfUseOutlineLevels )
	    {
	    te->teLevel= paraBi->biParaOutlineLevel;

	    if  ( te->teLevel <= ct->ctTocField.tfLevel1	&&
		  te->teLevel >= ct->ctTocField.tfLevel0	)
		{
		te->teNumbered= te->teLevel < ct->ctTocField.tfNLevel0	||
				te->teLevel > ct->ctTocField.tfNLevel1	;
		return 0;
		}
	    }

	return 1;
	}

    if  ( te->teField->dfKind == DOCfkTC	||
	  te->teField->dfKind == DOCfkTCN	)
	{
	DocumentField *	dfBookmark;

	if  ( ! ct->ctTocField.tfUseTcEntries )
	    { return 1;	}

	flag[1]= '\0';
	if  ( docFieldGetTc( te->teField, flag,
				    &(te->teLevel), &(te->teNumbered) ) )
	    { LDEB(te->teField->dfFieldNumber); return -1;	}

	dfBookmark= docFindTypedChildField( &(ct->ctBdDoc->bdBody.eiRootFields),
			    &(te->teField->dfHeadPosition), DOCfkBOOKMARK );
	if  ( ! dfBookmark )
	    { /*XDEB(dfBookmark);*/ return 1;	}

	if  ( docDelimitFieldInDoc( &(te->teDsInside), &(te->teDsAround),
					&(te->tePart0), &(te->tePart1),
					ct->ctBdDoc, dfBookmark ) )
	    { LDEB(dfBookmark->dfFieldNumber); return 1; }

	if  ( docIsIBarSelection( &(te->teDsInside) ) )
	    { return 1;	}

	if  ( docFieldGetBookmark( dfBookmark,
				    &(te->teMarkName), &(te->teMarkSize) ) )
	    { LDEB(te->teField->dfFieldNumber); return -1;	}

	if  ( te->teLevel <= ct->ctTocField.tfLevel1	&&
	      te->teLevel >= ct->ctTocField.tfLevel0	)
	    {
	    te->teNumbered= te->teLevel < ct->ctTocField.tfNLevel0	||
			    te->teLevel > ct->ctTocField.tfNLevel1	;
	    return 0;
	    }

	return 1;
	}

    return 1;
    }

/************************************************************************/
/*									*/
/*  Create a paragraph to hold one TOC entry.				*/
/*									*/
/************************************************************************/

static BufferItem * docMakeTocParagraph(
				CalculateToc *			ct,
				BufferItem *			refBi,
				int				level,
				int				numbered )

    {
    const DocumentStyle *	ds= ct->ctLevelStyles[level];
    const DocumentProperties *	dp= &(ct->ctBdToc->bdProperties);
    int				textAttrNr= ct->ctLevelAttributeNumbers[level];

    BufferItem *		paraBiToc;
    ParagraphProperties *	pp;


    paraBiToc= docInsertEmptyParagraph( ct->ctBdToc, refBi, textAttrNr );
    if  ( ! paraBiToc )
	{ XDEB(paraBiToc); return paraBiToc;	}

    if  ( ds && ! utilPropMaskIsEmpty( &(ds->dsParaMask) ) )
	{
	PropertyMask			ppChgMask;
	const ParagraphProperties *	ppFrom= &(ds->dsParagraphProperties);

	/*  No mapping needed: The style is in the target document */
	if  ( docUpdParaProperties( &ppChgMask,
					&(paraBiToc->biParaProperties),
					&(ds->dsParaMask), ppFrom,
					(const DocumentAttributeMap *)0 ) )
	    { LDEB(1);		}
	}
    else{
	if  ( docCopyParagraphProperties( &(paraBiToc->biParaProperties),
							    &(ct->ctRefPP) ) )
	    { LDEB(1);	}

	paraBiToc->biParaLeftIndentTwips= level* dp->dpTabIntervalTwips;
	}

    pp= &(paraBiToc->biParaProperties);
    if  ( numbered					&&
	  pp->ppTabStopList.tslTabStopCount == 0	)
	{
	ParagraphFrame		pf;
	TabStop			ts;

	docParagraphFrameTwips( &pf, &(ct->ctBlockFrame), paraBiToc );

	docInitTabStop( &ts );

	ts.tsAlignment= DOCtaRIGHT;
	ts.tsTwips= pf.pfCellContentRect.drX1- pf.pfCellContentRect.drX0;
	ts.tsLeader= DOCtlDOTS;

	if  ( docParaAddTab( pp, &ts ) )
	    { LDEB(1);	}
	}

    return paraBiToc;
    }

/************************************************************************/
/*									*/
/*  Clone the source TOC field to hold all TOC entries in the		*/
/*  intermediary toc document.						*/
/*									*/
/************************************************************************/

static int docMakeTocField(	CalculateToc *		ct,
				const DocumentField *	dfToc,
				BufferItem *		paraBiToc )
    {
    EditPosition	epStart;
    const int		stroff= 0;
    const int		part= 0;
    TextParticule *	tp;
    int			textAttrNr= ct->ctDefaultTextAttributeNumber;

    docSetEditPosition( &epStart, paraBiToc, stroff );

    ct->ctDfTocTo= docClaimFieldCopy( &(ct->ctBdToc->bdFieldList),
		    dfToc, &(ct->ctSectBi->biSectSelectionScope), &epStart );
    if  ( ! ct->ctDfTocTo )
	{ XDEB(ct->ctDfTocTo); return -1;	}

    tp= docMakeSpecialParticule( paraBiToc, part, epStart.epStroff,
					    DOCkindFIELDSTART, textAttrNr );
    if  ( ! tp )
	{ XDEB(tp); return -1;	}

    tp->tpObjectNumber= ct->ctDfTocTo->dfFieldNumber;

    return 0;
    }

/************************************************************************/

static int docSetupTocDocument(		CalculateToc *		ct )
    {
    int			level;
    TextAttribute	ta;

    ct->ctBdToc= docNewDocument();
    if  ( ! ct->ctBdToc )
	{ XDEB(ct->ctBdToc); return -1;	}

    if  ( docCopyDocumentProperties( &(ct->ctBdToc->bdProperties),
					    &(ct->ctBdDoc->bdProperties) ) )
	{ LDEB(1); return -1;	}

    if  ( docCopyStyleSheet( &(ct->ctBdToc->bdStyleSheet),
					    &(ct->ctBdDoc->bdStyleSheet) ) )
	{ LDEB(1); return -1;	}

    ct->ctSectBi= docInsertItem( ct->ctBdToc, ct->ctBdToc->bdBody.eiRoot,
							    -1, DOClevSECT );
    if  ( ! ct->ctSectBi )
	{ XDEB(ct->ctSectBi); return -1;	}

    ct->ctDefaultTextAttributeNumber= utilTextAttributeNumber(
					&(ct->ctBdToc->bdTextAttributeList),
					&(ct->ctTextAttribute) );

    for ( level= 0; level <= PPoutlineDEEPEST; level++ )
	{
	char			scratch[20+1];
	const DocumentStyle *	ds;

	sprintf( scratch, "toc %d", level+ 1 );
	ds= docGetStyleByName( &(ct->ctBdToc->bdStyleSheet), scratch );
	ct->ctLevelStyles[level]= ds;

	ct->ctLevelAttributeNumbers[level]= ct->ctDefaultTextAttributeNumber;
	if  ( ds && ! utilPropMaskIsEmpty( &(ds->dsTextMask) ) )
	    {
	    PropertyMask	doneMask;

	    docPlainTextAttribute( &ta, ct->ctBdToc );

	    utilUpdateTextAttribute( &doneMask, &ta,
				&(ds->dsTextAttribute), &(ds->dsTextMask) );
	    ct->ctLevelAttributeNumbers[level]= utilTextAttributeNumber(
				    &(ct->ctBdToc->bdTextAttributeList), &ta );
	    }
	}

    return 0;
    }

/************************************************************************/

static int docFinishTocField(	CalculateToc *		ct,
				BufferItem *		paraBiToc )
    {
    EditPosition	epEnd;
    TextParticule *	tp;
    int			textAttrNr= ct->ctDefaultTextAttributeNumber;

    docSetEditPosition( &epEnd, paraBiToc, docParaStrlen( paraBiToc ) );

    tp= docMakeSpecialParticule( paraBiToc, paraBiToc->biParaParticuleCount,
			    epEnd.epStroff, DOCkindFIELDEND, textAttrNr );
    if  ( ! tp )
	{ XDEB(tp); return -1;	}
    tp->tpObjectNumber= ct->ctDfTocTo->dfFieldNumber;

    if  ( docExternalItemAddRootField( &(ct->ctBdToc->bdBody), ct->ctDfTocTo ) )
	{ LDEB(1); return -1;	}

    return 0;
    }

/************************************************************************/

static int docGetTocParaProperties(
				CalculateToc *			ct,
				const BufferItem *		bodySectBi,
				const DocumentSelection *	dsToc,
				int				part0,
				int				part1 )
    {
    docBlockFrameTwips( &(ct->ctBlockFrame),
				dsToc->dsHead.dpBi, bodySectBi, ct->ctBdDoc,
				dsToc->dsHead.dpBi->biTopPosition.lpPage,
				dsToc->dsHead.dpBi->biTopPosition.lpColumn );


    if  ( docCopyParagraphProperties( &(ct->ctRefPP),
				&(dsToc->dsHead.dpBi->biParaProperties ) ) )
	{ LDEB(1); return -1;	}

    {
    int				part;
    const TextParticule *	tp= (const TextParticule *)0;

    tp= dsToc->dsHead.dpBi->biParaParticules+ part0;
    utilGetTextAttributeByNumber( &(ct->ctTextAttribute),
					&(ct->ctBdDoc->bdTextAttributeList),
					tp->tpTextAttrNr );

    part= part0;
    while( part < dsToc->dsHead.dpBi->biParaParticuleCount )
	{
	if  ( tp->tpKind == DOCkindSPAN )
	    {
	    utilGetTextAttributeByNumber( &(ct->ctTextAttribute),
					&(ct->ctBdDoc->bdTextAttributeList),
					tp->tpTextAttrNr );

	    break;
	    }
	part++; tp++;
	}
    }

    return 0;
    }

/************************************************************************/
/*									*/
/*  Collect TOC entries. Relies on the bookmarks that should already	*/
/*  have been made.							*/
/*									*/
/*  Sorting takes care of selecting the most appropriate ones.		*/
/*									*/
/************************************************************************/

static int docCollectTocEntries(CalculateToc *			ct )
    {
    const DocumentFieldList *	dflDoc= &(ct->ctBdDoc->bdFieldList);
    const int			fieldCount= dflDoc->dflPagedList.plItemCount;
    int				fieldNr;

    for ( fieldNr= 0; fieldNr < fieldCount; fieldNr++ )
	{
	int			res;
	const DocumentField *	df= docGetFieldByNumber( dflDoc, fieldNr );

	if  ( ! df )
	    { continue;	}
	if  ( df->dfSelectionScope.ssInExternalItem != DOCinBODY )
	    { continue;	}

	if  ( df->dfKind != DOCfkBOOKMARK	&&
	      df->dfKind != DOCfkTC		&&
	      df->dfKind != DOCfkTCN		)
	    { continue;	}

	docInitTocEntry( ct->ctEntries+ ct->ctEntryCount );
	ct->ctEntries[ct->ctEntryCount].teField= df;

	res= docGetTocEntry( ct->ctEntries+ ct->ctEntryCount, ct );
	if  ( res < 0 )
	    { LLDEB(fieldNr,res); return -1;	}
	if  ( res > 0 )
	    { continue;	}

	ct->ctEntryCount++;
	}

    qsort( ct->ctEntries, ct->ctEntryCount,
				    sizeof(TocEntry), docCompareTocEntries );

    return 0;
    }

/************************************************************************/

static int docTocCollectTocStyles(	CalculateToc *		ct )
    {
    const BufferDocument *	bdDoc= ct->ctBdDoc;
    const DocumentStyle *	ds= bdDoc->bdStyleSheet.dssStyles;
    int				i;

    ct->ctStyleLevels= malloc( bdDoc->bdStyleSheet.dssStyleCount );
    if  ( ! ct->ctStyleLevels )
	{
	LXDEB(bdDoc->bdStyleSheet.dssStyleCount,ct->ctStyleLevels);
	return -1;
	}

    for ( i= 0; i < bdDoc->bdStyleSheet.dssStyleCount; ds++, i++ )
	{
	int			j;
	const StyleLevel *	sl;

	ct->ctStyleLevels[i]= PPoutlineBODYTEXT;

	sl= ct->ctTocField.tfStyleLevels;
	for ( j= 0; j < ct->ctTocField.tfStyleLevelCount; sl++, j++ )
	    {
	    if  ( docStyleHasName( ds,
				sl->slStyleName, sl->slStyleNameLength ) )
		{ ct->ctStyleLevels[i]= sl->slLevel;	}
	    }
	}

    ct->ctStyleLevelCount= bdDoc->bdStyleSheet.dssStyleCount;

    return 0;
    }

static int docTocSetStyleBookmarks(	CalculateToc *		ct )
    {
    BufferDocument *		bdDoc= ct->ctBdDoc;
    DocumentPosition		dp;

    if  ( docFirstPosition( &dp, bdDoc->bdBody.eiRoot ) )
	{ LDEB(1); return 0;	}
    while( dp.dpBi )
	{
	if  ( dp.dpBi->biParaStyle > 0				&&
	      dp.dpBi->biParaStyle < ct->ctStyleLevelCount	)
	    {
	    int	level= ct->ctStyleLevels[dp.dpBi->biParaStyle];

	    if  ( level <= ct->ctTocField.tfLevel1	&&
		  level >= ct->ctTocField.tfLevel0	)
		{
		if  ( docSetParaTocBookmark( bdDoc,
						&(bdDoc->bdBody), dp.dpBi ) )
		    { LDEB(1); return -1;	}
		}
	    }

	dp.dpBi= docNextParagraph( dp.dpBi );
	}

    return 0;
    }

/************************************************************************/
/*									*/
/*  Recalculate TOC field.						*/
/*									*/
/************************************************************************/

static int docCalculateTocField( CalculateToc *			ct,
				BufferDocument *		x_bdDoc,
				const BufferItem *		bodySectBi,
				const DocumentSelection *	dsToc,
				int				part0,
				int				part1,
				const DocumentField *		dfToc )
    {
    BufferItem *		paraBiToc= (BufferItem *)0;
    BufferItem *		refBi= (BufferItem *)0;

    int				fieldNr;
    const DocumentFieldList *	dflDoc;
    int				fieldCount;

    int				entryNr= 0;
    int				entryParaNr= -1;
    TocEntry *			te;

    ct->ctBdDoc= x_bdDoc;
    dflDoc= &(ct->ctBdDoc->bdFieldList);
    fieldCount= dflDoc->dflPagedList.plItemCount;
    docFieldGetToc( &(ct->ctTocField), dfToc );

    if  ( ct->ctTocField.tfUseStyles )
	{
	if  ( docTocCollectTocStyles( ct ) )
	    { LDEB(ct->ctTocField.tfUseStyles); return -1;	}
	if  ( docTocSetStyleBookmarks( ct ) )
	    { LDEB(ct->ctTocField.tfUseStyles); return -1;	}
	}

    if  ( docGetTocParaProperties( ct, bodySectBi, dsToc, part0, part1 ) )
	{ LDEB(1); return -1;	}

    ct->ctEntries= malloc( fieldCount* sizeof(TocEntry) );
    if  ( ! ct->ctEntries )
	{ LXDEB(fieldCount,ct->ctEntries); return -1;	}
    for ( fieldNr= 0; fieldNr < fieldCount; fieldNr++ )
	{ docInitTocEntry( ct->ctEntries+ fieldNr ); }

    if  ( docSetupTocDocument( ct ) )
	{ LDEB(1); return -1;	}


    if  ( docCollectTocEntries( ct ) )
	{ LDEB(1); return -1;	}

    refBi= ct->ctSectBi;
    entryParaNr= -1;
    te= ct->ctEntries;
    for ( entryNr= 0; entryNr < ct->ctEntryCount; te++, entryNr++ )
	{
	if  ( te->teField->dfHeadPosition.epParaNr == entryParaNr )
	    { continue;	}
	entryParaNr= te->teField->dfHeadPosition.epParaNr;

	paraBiToc= docMakeTocParagraph( ct, refBi,
					    te->teLevel, te->teNumbered );
	if  ( ! paraBiToc )
	    { XDEB(paraBiToc); return -1;	}

	if  ( ! ct->ctDfTocTo )
	    {
	    if  ( docMakeTocField( ct, dfToc, paraBiToc ) )
		{ LDEB(1); return -1;	}
	    }

	if  ( docParaInsertTocEntry( ct, ct->ctTocField.tfHyperlinks, te,
				    &(dfToc->dfSelectionScope), paraBiToc ) )
	    { LDEB(entryNr); return -1;	}

	refBi= paraBiToc;
	}

    if  ( ! paraBiToc )
	{
	const int		stroff= 0;
	int			stroffShift;
	const int		level= 0;
	const int		numbered= 0;

	const char *		text= "TOC";
	const int		len= 3;

	paraBiToc= docMakeTocParagraph( ct, refBi, level, numbered );
	if  ( ! paraBiToc )
	    { XDEB(paraBiToc); return -1;	}

	if  ( docParaStringReplace( &stroffShift, paraBiToc,
						stroff, stroff, text, len ) )
	    { LDEB(len); return -1;	}

	if  ( docRedivideStringInParticules( paraBiToc, stroff, len,
				    paraBiToc->biParaParticuleCount, 0, 
				    ct->ctDefaultTextAttributeNumber ) < 0 )
	    { LDEB(1); return -1;	}

	if  ( docMakeTocField( ct, dfToc, paraBiToc ) )
	    { LDEB(1); return -1;	}
	}

    if  ( docFinishTocField( ct, paraBiToc ) )
	{ LDEB(1); return -1;	}

    return 0;
    }

/************************************************************************/
/*									*/
/*  Recalculate one TOC field in a document.				*/
/*									*/
/************************************************************************/

int docRecalculateOneTocField(	BufferDocument *	bdDoc,
				const DocumentField *	df )
    {
    int				rval= 0;

    const char *		refFileName= (const char *)0;

    DocumentSelection		dsInsideToc;
    DocumentSelection		dsAroundToc;
    int				part0;
    int				part1;

    EditOperation		eo;
    DocumentCopyJob		dcj;
    CalculateToc		ct;

    docInitDocumentCopyJob( &dcj );
    docInitEditOperation( &eo );
    docInitCalculateToc( &ct );

    if  ( docDelimitFieldInDoc( &dsInsideToc, &dsAroundToc,
						&part0, &part1, bdDoc, df ) )
	{ LDEB(1); rval= -1; goto ready;	}

#   if 0
    /*  MS-Word does something like this: */
    docFirstPosition( &(dsAroundToc.dsHead), dsAroundToc.dsHead.dpBi );
    docLastPosition( &(dsAroundToc.dsTail), dsAroundToc.dsTail.dpBi );
#   endif

    eo.eoBodySectBi= docGetBodySectBiOfScope( &(df->dfSelectionScope), bdDoc );

    if  ( docCalculateTocField( &ct, bdDoc, eo.eoBodySectBi,
					    &dsInsideToc, part0, part1, df ) )
	{ LDEB(1); rval= -1; goto ready;	}

    if  ( docStartEditOperation( &eo, &dsAroundToc, bdDoc ) )
	{ LDEB(1); rval= -1; goto ready;	}

    if  ( docSet2DocumentCopyJob( &dcj, &eo, ct.ctBdToc, refFileName ) )
	{ LDEB(1); rval= -1; goto ready;	}

    if  ( docIncludeDocument( &dcj ) )
	{ LDEB(1); rval= -1; goto ready;	}

  ready:

    docCleanDocumentCopyJob( &dcj );
    docCleanCalculateToc( &ct );

    return rval;
    }
