/************************************************************************/
/*									*/
/*  Management of bookmarks related to TOC fields.			*/
/*									*/
/************************************************************************/

#   include	"docBufConfig.h"

#   include	<stdlib.h>
#   include	<string.h>
#   include	<stdio.h>

#   include	<appDebugon.h>

#   include	<appUnit.h>

#   include	"docBuf.h"
#   include	"docTocField.h"

/************************************************************************/
/*									*/
/*  Recalculate TOC fields in a document.				*/
/*									*/
/************************************************************************/

static int docIsTocBookmark(	long *				pId,
				const DocumentField *		df )
    {
    const char *	markName;
    int			markSize;

    char *		past;
    long		id;


    if  ( docFieldGetBookmark( df, &markName, &markSize ) )
	{ LDEB(df->dfFieldNumber); return 0;	}

    if  ( markSize < 4 || strncmp( markName, "_Toc", 4 ) )
	{ return 0;	}

    id= strtod( markName+ 4, &past );
    if  ( past == markName+ 4 )
	{ return 0;	}
    if  ( past- markName < markSize )
	{ return 0;	}

    if  ( pId )
	{ *pId= id;	}
    return 1;
    }

static int docSetTocBookmark(	BufferDocument *	bd,
				DocumentField *		dfBookmark )
    {
    DocumentFieldList *	dfl= &(bd->bdFieldList);
    const int		fieldCount= dfl->dflPagedList.plItemCount;
    int			fieldNr;

    char		markName[DOCmaxBOOKMARK+1];
    int			markSize= 0;

    long		id0= 0;
    long		id1= 0;

    for ( fieldNr= 0; fieldNr < fieldCount; fieldNr++ )
	{
	DocumentField *		df= docGetFieldByNumber( dfl, fieldNr );
	long			id;

	if  ( ! df )
	    { continue;	}
	if  ( df->dfKind != DOCfkBOOKMARK	||
	      ! docIsTocBookmark( &id, df )	)
	    { continue;	}

	if  ( id0 <= 0 || id0 > id )
	    { id0= id;	}
	if  ( id1 <= 0 || id1 < id )
	    { id1= id;	}
	}

    if  ( id0 > 1 )
	{ sprintf( markName, "_Toc%ld", id0- 1 );	}
    else{ sprintf( markName, "_Toc%ld", id1+ 1 );	}
    markSize= strlen( markName );

    if  ( docFieldSetBookmark( dfBookmark, markName, markSize ) )
	{ SDEB(markName); return -1;	}

    return 0;
    }

void docRemoveUnbalancedTocBookmarks(	BufferDocument *	bdDoc )
    {
    DocumentFieldList *	dfl= &(bdDoc->bdFieldList);
    const int		fieldCount= dfl->dflPagedList.plItemCount;
    int			fieldNr;

    for ( fieldNr= 0; fieldNr < fieldCount; fieldNr++ )
	{
	DocumentField *		df= docGetFieldByNumber( dfl, fieldNr );

	if  ( ! df )
	    { continue;		}
	if  ( df->dfSelectionScope.ssInExternalItem != DOCinBODY	)
	    { continue;	}

	if  ( df->dfKind == DOCfkBOOKMARK				&&
	      df->dfHeadPosition.epParaNr !=
				      df->dfTailPosition.epParaNr	&&
	      docIsTocBookmark( (long *)0, df )				)
	    {
	    DocumentSelection	dsInside;
	    DocumentSelection	dsAround;
	    int			headPart;
	    int			tailPart;

	    if  ( docDelimitFieldInDoc( &dsInside, &dsAround,
					    &headPart, &tailPart, bdDoc, df ) )
		{ LDEB(fieldNr); continue; }

	    if  ( docDeleteField( dsInside.dsHead.dpBi, dsInside.dsTail.dpBi,
						&headPart, &tailPart,
						headPart, tailPart,
						&(bdDoc->bdBody), bdDoc, df ) )
		{ LDEB(fieldNr);	}
	    }
	}
    }

/************************************************************************/
/*									*/
/*  Set a TOC bookmark that contains the whole paragraph.		*/
/*									*/
/************************************************************************/

int docSetParaTocBookmark(	BufferDocument *	bd,
				DocumentTree *		ei,
				BufferItem *		paraBi )
    {
    DocumentPosition	dpHead;
    DocumentPosition	dpTail;

    DocumentField *	dfHead= (DocumentField *)0;
    DocumentField *	dfTail= (DocumentField *)0;

    if  ( docFirstPosition( &dpHead, paraBi ) )
	{ LDEB(1); return -1;	}
    docAvoidParaHeadField( &dpHead, (int *)0, bd );

    {
    const int			lastOne= 1;
    int				part;
    const TextParticule *	tp;

    docFindParticuleOfPosition( &part, &dpHead, lastOne );

    tp= dpHead.dpBi->biParaParticules+ part;
    while( part < paraBi->biParaParticuleCount		&&
	   ( tp->tpKind == DOCkindPAGEBREAK	||
	     tp->tpKind == DOCkindLINEBREAK	)	)
	{
	dpHead.dpStroff= tp->tpStrlen+ tp->tpStroff;
	tp++; part++;
	}
    }

    if  ( docLastPosition( &dpTail, paraBi ) )
	{ LDEB(1); return -1;	}

    dfHead= docFindTypedFieldForPosition( bd, &dpHead, DOCfkBOOKMARK );
    dfTail= docFindTypedFieldForPosition( bd, &dpTail, DOCfkBOOKMARK );
    if  ( ! dfHead || ! dfTail || dfHead != dfTail )
	{
	DocumentSelection	dsBookmark;
	DocumentSelection	dsInsideMark;
	DocumentSelection	dsAroundMark;
	int			partHead= -1;
	int			partTail= -1;

	DocumentField *		dfBookmark= (DocumentField *)0;

	TextAttribute	taSet;
	PropertyMask	taSetMask;

	utilInitTextAttribute( &taSet );
	utilPropMaskClear( &taSetMask );

	if  ( dpTail.dpStroff < dpHead.dpStroff )
	    { LLDEB(dpTail.dpStroff,dpHead.dpStroff);	}

	if  ( dpTail.dpStroff > dpHead.dpStroff )
	    {
	    const int	singlePara= 1;

	    docSetRangeSelection( &dsBookmark, &dpHead, &dpTail, 1, -1, -1 );

	    if  ( docSurroundTextSelectionByField( &dfBookmark,
						&dsInsideMark, &dsAroundMark,
						&partHead, &partTail, bd, ei,
						singlePara, &dsBookmark, 
						&taSetMask, &taSet ) )
		{ LDEB(1); return -1;	}

	    if  ( docSetTocBookmark( bd, dfBookmark ) )
		{ LDEB(1); return -1;	}
	    }
	}

    return 0;
    }

/************************************************************************/
/*									*/
/*  Make sure that all leaves in the outlinelevel tree have a bookmark	*/
/*  that can be used in the table of contents.				*/
/*									*/
/************************************************************************/

static int docSetTocBookmarksForLevel(	BufferDocument *		bd,
					DocumentTree *			ei,
					const ListNumberTreeNode *	lntn )
    {
    int		i;
    int		rval= 0;

    if  ( lntn->lntnIsLeaf && lntn->lntnParaNr > 0 )
	{
	BufferItem *	paraBi;

	paraBi= docGetParagraphByNumber( ei, lntn->lntnParaNr );
	if  ( ! paraBi )
	    { LXDEB(lntn->lntnParaNr,paraBi); rval= -1;	}
	else{
	    if  ( docSetParaTocBookmark( bd, ei, paraBi ) )
		{ LDEB(lntn->lntnParaNr); rval= -1;	}
	    }
	}

    for ( i= 0; i < lntn->lntnChildCount; i++ )
	{
	if  ( docSetTocBookmarksForLevel( bd, ei, &(lntn->lntnChildren[i]) ) )
	    { LDEB(i); rval= -1;	}
	}

    return rval;
    }

int docSetTocBookmarks(	BufferDocument *		bd )
    {
    DocumentTree *	ei= &(bd->bdBody);

    DocumentFieldList *	dfl= &(bd->bdFieldList);
    const int		fieldCount= dfl->dflPagedList.plItemCount;
    int			fieldNr;

    if  ( docSetTocBookmarksForLevel( bd, ei, &(ei->eiOutlineTree) ) )
	{ LDEB(1); return -1;	}

    for ( fieldNr= 0; fieldNr < fieldCount; fieldNr++ )
	{
	DocumentField *		dfTc= docGetFieldByNumber( dfl, fieldNr );
	DocumentField *		dfBookmark;
	DocumentSelection	dsInsideTc;
	DocumentSelection	dsAroundTc;
	DocumentSelection	dsInsideMark;
	DocumentSelection	dsAroundMark;
	int			headPart;
	int			tailPart;

	if  ( ! dfTc )
	    { continue;	}
	if  (  ( dfTc->dfKind != DOCfkTC && dfTc->dfKind != DOCfkTCN )	||
	      dfTc->dfSelectionScope.ssInExternalItem != DOCinBODY	)
	    { continue;	}

	if  ( docDelimitFieldInDoc( &dsInsideTc, &dsAroundTc,
					    &headPart, &tailPart, bd, dfTc ) )
	    { LDEB(1); return -1;	}

	dfBookmark= docFindTypedFieldForPosition( bd, &(dsInsideTc.dsHead),
							    DOCfkBOOKMARK );
	if  ( ! dfBookmark )
	    {
	    TextAttribute	taSet;
	    PropertyMask	taSetMask;
	    const int		singlePara= 1;

	    utilInitTextAttribute( &taSet );
	    utilPropMaskClear( &taSetMask );

	    if  ( docSurroundTextSelectionByField( &dfBookmark,
						&dsInsideMark, &dsAroundMark,
						&headPart, &tailPart, bd, ei,
						singlePara, &dsAroundTc, 
						&taSetMask, &taSet ) )
		{ LDEB(1); return -1;	}

	    if  ( docSetTocBookmark( bd, dfBookmark ) )
		{ LDEB(1); return -1;	}
	    }
	}

    return 0;
    }

