/************************************************************************/
/*									*/
/*  Manage the actual string content while reading an RTF document.	*/
/*									*/
/************************************************************************/

#   include	"docBufConfig.h"

#   include	<stdlib.h>
#   include	<string.h>
#   include	<stdio.h>
#   include	<ctype.h>

#   include	<appDebugon.h>

#   include	<utilMatchFont.h>
#   include	<uniUtf8.h>
#   include	<ucd.h>

#   include	"docRtf.h"
#   include	"docParaString.h"

/************************************************************************/
/*									*/
/*  Handle text..							*/
/*									*/
/*  1)  Ignore it.							*/
/*  2)  Refuse it.							*/
/*  3)  Save it for later use. (Convert it to UTF-8 on the fly).	*/
/*									*/
/************************************************************************/

/*  1  */
int docRtfIgnoreText(	RtfReadingContext *	rrc,
			const char *		text,
			int			len )
    { return 0; }

/*  2  */
int docRtfRefuseText(	RtfReadingContext *	rrc,
			const char *		text,
			int			len )
    { LDEB(1); return -1; }

/************************************************************************/
/*									*/
/*  Save text: It is not encoded.					*/
/*									*/
/************************************************************************/

int docRtfSaveRawBytes(		RtfReadingContext *	rrc,
				const char *		text,
				int			len )
    {
    RtfReadingState *	rrs= rrc->rrcState;

    if  ( utilAddToMemoryBuffer( &(rrs->rrsSavedText),
					(const unsigned char *)text, len ) )
	{ LDEB(len); return -1;	}

    return 0;
    }

/************************************************************************/
/*									*/
/*  Convert text from some encoding to UTF-8.				*/
/*									*/
/************************************************************************/

static int docRtfSaveBytes(	void *		vmb,
				int		offset,
				const char *	bytes,
				int		count )
    {
    MemoryBuffer *	mb= (MemoryBuffer *)vmb;

    if  ( utilAddToMemoryBuffer( mb, (const unsigned char *)bytes, count ) )
	{ LDEB(mb->mbSize); return -1;	}

    return count;
    }

/************************************************************************/
/*									*/
/*  Save text: It is in the document encoding.				*/
/*									*/
/************************************************************************/

int docRtfSaveDocEncodedText(	RtfReadingContext *	rrc,
				const char *		text,
				int			len )
    {
    RtfReadingState *	rrs= rrc->rrcState;
    int			upto;
    int			consumed= 0;

    upto= utilTextConverterConvertToUtf8(
				    &(rrc->rrcTextConverters.rtcRtfConverter),
				    (void *)&(rrs->rrsSavedText),
				    docRtfSaveBytes, &consumed,
				    rrs->rrsSavedText.mbSize, text, len );
    if  ( upto < 0 )
	{ LDEB(upto); return -1;	}

    return 0;
    }

/************************************************************************/
/*									*/
/*  Store the text bytes that we collected from the rtf file in some	*/
/*  location. Use realloc() to rezize the target location and flush	*/
/*  the collected text bytes.						*/
/*									*/
/************************************************************************/

int docRtfStoreSavedText(	char **			pTarget,
				int *			pSize,
				RtfReadingContext *	rrc,
				int			removeSemicolon )
    {
    RtfReadingState *	rrs= rrc->rrcState;

    char *	fresh;
    int		size;

    if  ( rrs->rrsSavedText.mbSize == 0 )
	{ *pSize= 0; return 0;	}

    size= rrs->rrsSavedText.mbSize;
    fresh= realloc( *pTarget, size+ 1 );
    if  ( ! fresh )
	{ LXDEB(size,fresh); return -1;	}

    memcpy( fresh, rrs->rrsSavedText.mbBytes, size );
    fresh[size]= '\0';

    if  ( removeSemicolon		&&
	  size > 0			&&
	  fresh[size- 1] == ';'		)
	{ fresh[--size]= '\0';	}

    utilEmptyMemoryBuffer( &(rrs->rrsSavedText) );

    *pTarget= fresh;
    *pSize= size;
    return 0;
    }

int docRtfMemoryBufferSetText(	MemoryBuffer *		mb,
				RtfReadingContext *	rrc,
				int			removeSemicolon )
    {
    char *	text= (char *)0;
    int		size;

    if  ( docRtfStoreSavedText( &text, &size, rrc, removeSemicolon ) )
	{ LDEB(1); return -1;	}

    if  ( utilSetMemoryBuffer( mb, (const unsigned char *)text, size ) )
	{ LDEB(1); return -1;	}

    if  ( text )
	{ free( text );		}

    return 0;
    }

int docRtfMemoryBufferAppendText(
				MemoryBuffer *		mb,
				RtfReadingContext *	rrc )
    {
    const int	removeSemicolon= 0;
    char *	text= (char *)0;
    int		size;

    if  ( docRtfStoreSavedText( &text, &size, rrc, removeSemicolon ) )
	{ LDEB(1); return -1;	}

    if  ( utilAddToMemoryBuffer( mb, (const unsigned char *)text, size ) )
	{ LDEB(1); return -1;	}

    if  ( text )
	{ free( text );		}

    return 0;
    }

/************************************************************************/

static const DocumentFont * docRtfGetCurrentFont(
				BufferDocument *		bd,
				TextAttribute *			ta )
    {
    DocumentProperties *	dp= &(bd->bdProperties);
    DocumentFontList *		dfl= &(dp->dpFontList);
    const DocumentFont *	df;

    if  ( ta->taFontNumber < 0 )
	{
	/*LDEB(ta->taFontNumber);*/
	ta->taFontNumber= docGetDefaultFont( bd );
	}

    df= docFontListGetFontByNumber( dfl, ta->taFontNumber );
    if  ( ! df )
	{
	if  ( ta->taFontNumber >= 0 )
	    { LXDEB(ta->taFontNumber,df);	}
	else{
	    ta->taFontNumber= docGetDefaultFont( bd );
	    df= docFontListGetFontByNumber( dfl, ta->taFontNumber );
	    }
	}

    return df;
    }

static const char * docRtfGetEncodingName(
				BufferDocument *		bd,
				TextAttribute *			ta,
				int				charset )
    {
    const char *		encodingName= (const char *)0;

    const DocumentFont *	df;

    df= docRtfGetCurrentFont( bd, ta );
    if  ( df )
	{ encodingName= utilGetEncodingName( df->dfName, charset );	}

    return encodingName;
    }

static int docRtfReadAdaptToFontEncoding(
				RtfReadingContext *		rrc,
				RtfReadingState *		rrs )
    {
    const char *		encodingName= (const char *)0;

    if  ( rrs->rrsTextCharset >= 0 )
	{
	encodingName= docRtfGetEncodingName(
		    rrc->rrcBd, &(rrs->rrsTextAttribute), rrs->rrsTextCharset );
	}

    if  ( ! encodingName )
	{ encodingName= docRtfReadGetRtfEncodingName( rrc );	}

    docRtfReadSetTextEncodingName( rrc, encodingName );

    return 0;
    }

/************************************************************************/
/*									*/
/*  Insert particules from the input in the document.			*/
/*									*/
/************************************************************************/

int docRtfTextParticule(	RtfReadingContext *	rrc,
				const char *		text,
				int			len )
    {
    RtfReadingState *		rrs= rrc->rrcState;
    BufferDocument *		bd= rrc->rrcBd;
    BufferItem *		paraBi;

    if  ( rrc->rrcInIgnoredGroup )
	{ return 0;	}

    paraBi= docRtfGetParaItem( rrc );
    if  ( ! paraBi )
	{ XDEB(paraBi); return -1; }

    if  ( rrc->rrcAtParaHead )
	{
	int		mindTable= 1;

	if  ( docRtfSetParaProperties(  &(paraBi->biParaProperties),
							bd, mindTable, rrs ) )
	    { LDEB(1); return -1;	}

	rrc->rrcAtParaHead= 0;
	}

    if  ( docRtfReadAdaptToFontEncoding( rrc, rrs ) )
	{ LDEB(1);	}

    if  ( rrs->rrsTextShadingChanged )
	{ docRtfRefreshTextShading( rrc, rrs );	}

    if  ( docSaveParticules( bd, paraBi, &(rrs->rrsTextAttribute),
		    &(rrc->rrcTextConverters.rtcTextConverter), text, len ) )
	{ LDEB(1); return -1;	}

    rrc->rrcAfterNoteref= 0;
    rrc->rrcAtParaHead= 0;
    
    return 0;
    }

/************************************************************************/
/*									*/
/*  Handle an explicit unicode.						*/
/*  Special characters.							*/
/*									*/
/************************************************************************/

static int docRtfTextUnicodeValue(	const RtfControlWord *	rcw,
					int			arg,
					RtfReadingContext *	rrc )
    {
    RtfReadingState *	rrs= rrc->rrcState;

    unsigned char	bytes[7];
    int			count;

    if  ( arg < 0 )
	{ arg += 65536;	}

    /* Dirty HACK */
    if  ( ucdIsPrivate( arg ) )
	{
	bytes[0]= arg & 0xff;
	bytes[1]= '\0';;

	return docRtfSaveDocEncodedText( rrc, (char *)bytes, 1 );
	}

    count= uniPutUtf8( bytes, arg );
    if  ( count < 1 )
	{ LDEB(count); return 0;	}

    if  ( rrc->rrcAddParticule == docRtfSaveRawBytes )
	{ XXDEB(rrc->rrcAddParticule,docRtfSaveRawBytes); return 0; }

    if  ( rrc->rrcAddParticule == docRtfTextParticule )
	{
	int			stroffShift= 0;
	int			stroff;
	BufferItem *		paraBi= docRtfGetParaItem( rrc );
	BufferDocument *	bd= rrc->rrcBd;
	int			textAttributeNumber;

	if  ( ! paraBi )
	    { SXDEB(rcw->rcwWord,paraBi); return -1; }

	if  ( rrs->rrsTextShadingChanged )
	    { docRtfRefreshTextShading( rrc, rrs );	}

	textAttributeNumber= docTextAttributeNumber( bd,
						&(rrs->rrsTextAttribute) );
	if  ( textAttributeNumber < 0 )
	    { LDEB(textAttributeNumber); return -1;	}

	stroff= docParaStrlen( paraBi );

	if  ( docParaStringReplace( &stroffShift, paraBi, stroff, stroff,
						    (char *)bytes, count ) )
	    { LDEB(docParaStrlen(paraBi)); return -1;	}

	if  ( docParaDivideAppendedText( paraBi, textAttributeNumber,
						    stroff, stroff+ count ) )
	    { LLDEB(count,paraBi->biParaParticuleCount); return -1; }
	}
    else{
	if  ( utilAddToMemoryBuffer( &(rrs->rrsSavedText), bytes, count ) )
	    { LDEB(count); return -1;	}
	}

    return 0;
    }

int docRtfTextUnicode(		const RtfControlWord *	rcw,
				int			arg,
				RtfReadingContext *	rrc )
    {
    RtfReadingState *	rrs= rrc->rrcState;

    if  ( docRtfTextUnicodeValue( rcw, arg, rrc ) )
	{ SXDEB(rcw->rcwWord,arg); return -1;	}

    rrs->rrsUnicodeBytesToSkip= rrs->rrsBytesPerUnicode;
    return 0;
    }

int docRtfTextSpecialChar(	const RtfControlWord *	rcw,
				int			arg,
				RtfReadingContext *	rrc )
    {
    /* docRtfTextParticule() adjusts level */

    if  ( docRtfTextUnicodeValue( rcw, rcw->rcwID, rrc ) )
	{ SXDEB(rcw->rcwWord,arg); return -1;	}

    return 0;
    }

int docRtfTextSpecialParticule(	const RtfControlWord *	rcw,
				int			arg,
				RtfReadingContext *	rrc )
    {
    RtfReadingState *	rrs= rrc->rrcState;
    BufferItem *	paraBi;

    if  ( rrc->rrcInIgnoredGroup > 0 )
	{ return 0;	}

    if  ( rrs->rrsTextShadingChanged )
	{ docRtfRefreshTextShading( rrc, rrs );	}

    paraBi= docRtfGetParaItem( rrc );
    if  ( ! paraBi )
	{ SXDEB(rcw->rcwWord,paraBi); return -1; }

    switch( rcw->rcwID )
	{
	case DOCkindTAB:
	case DOCkindLINEBREAK:
	case DOCkindCHFTNSEP:
	case DOCkindCHFTNSEPC:
	    if  ( docSaveSpecialParticule( rrc->rrcBd, paraBi,
				&(rrs->rrsTextAttribute), rcw->rcwID ) )
		{ LDEB(1); return -1;	}

	    rrc->rrcAfterNoteref= 0;
	    rrc->rrcAtParaHead= 0;

	    break;

	case DOCkindPAGEBREAK:
	case DOCkindCOLUMNBREAK:
	    {
	    int				done= 0;

	    if  ( rrc->rrcAtParaHead && rcw->rcwID == DOCkindPAGEBREAK )
		{ rrs->rrsParagraphBreakOverride= DOCibkPAGE; done= 1;	}

	    if  ( rrc->rrcAtParaHead && rcw->rcwID == DOCkindCOLUMNBREAK )
		{ rrs->rrsParagraphBreakOverride= DOCibkCOL; done= 1;	}

	    if  ( ! done						&&
		  docSaveSpecialParticule( rrc->rrcBd, paraBi,
				&(rrs->rrsTextAttribute), rcw->rcwID ) )
		{ LDEB(1); return -1;	}

	    rrc->rrcAfterNoteref= 0;
	    rrc->rrcAtParaHead= 0;

	    break;
	    }

	case DOCkindOPT_HYPH:
	    if  ( docSaveSpecialParticule( rrc->rrcBd, paraBi,
				&(rrs->rrsTextAttribute), DOCkindOPT_HYPH ) )
		{ LDEB(1); return -1;	}

	    rrc->rrcAfterNoteref= 0;
	    rrc->rrcAtParaHead= 0;
	    break;

	default:
	    SLDEB(rcw->rcwWord,rcw->rcwID);
	    break;
	}

    return 0;
    }

/************************************************************************/
/*									*/
/*  Emit a string to the RTF output stream.				*/
/*									*/
/************************************************************************/

static int docRtfEscapeString(	void *			vrwc,
				int			produced, /* ignored */
				const char *		ss,
				int			n )
    {
    RtfWritingContext *		rwc= (RtfWritingContext *)vrwc;
    const unsigned char *	us= (const unsigned char *)ss;
    SimpleOutputStream *	sos= rwc->rwcSosOut;
    int				i;
    int				addSpace= 0;

    if  ( n == 0 )
	{ return n;	}

    switch( rwc->rwcAfter )
	{
	case RTFafterTAG:
	    if  ( ss[0] == ' ' || isalnum( ss[0] ) || ss[0] == '-' )
		{ addSpace= 1;	}
	    break;

	case RTFafterARG:
	    if  ( ss[0] == ' ' || isdigit( ss[0] ) )
		{ addSpace= 1;	}
	    break;

	case RTFafterTEXT:
	    break;

	default:
	    CDEB(rwc->rwcAfter);
	}


    if  ( addSpace )
	{ sioOutPutByte( ' ', rwc->rwcSosOut ); rwc->rwcCol += 1;	}
    rwc->rwcAfter= RTFafterTEXT;

    i= 0;
    while( i < n )
	{
	int		c= *us;

	switch( c )
	    {
	    case '{': case '\\': case '}':
		sioOutPutByte( '\\', sos );
		sioOutPutByte( c, sos );
		rwc->rwcCol += 2;
		break;
	    default:
		if  ( c < 32 || c > 127 )
		    {
		    static char	hexdigits[]= "0123456789abcdef";

		    sioOutPutByte( '\\', sos );
		    sioOutPutByte( '\'', sos );
		    sioOutPutByte( hexdigits[ ( c >> 4 ) & 0x0f ], sos );
		    sioOutPutByte( hexdigits[ ( c >> 0 ) & 0x0f ], sos );
		    rwc->rwcCol += 4;
		    }
		else{
		    sioOutPutByte( c, sos );
		    rwc->rwcCol += 1;
		    }
		break;
	    }

	i++; us++;
	}

    return n;
    }

/************************************************************************/
/*									*/
/*  Emit a unicode (UTF-16) character as a tag.				*/
/*									*/
/************************************************************************/

static int docRtfEmitUnicode(		RtfWritingContext *	rwc,
					int			symbol )
    {
    if  ( symbol > 32767 && symbol < 65536 )
	{ symbol -= 65536;	}

    docRtfWriteArgTag( rwc, "\\u", symbol );
    docRtfEscapeString( (void *)rwc, 0, "?", 1 );

    return 0;
    }

/************************************************************************/
/*									*/
/*  Emit an utf-8 string in some legacy encoding.			*/
/*  The string might very well contain unicodes that cannot be		*/
/*  represented in the legacy encoding. The encoder stops on those	*/
/*  bytes and we emit them in \u1234 format.				*/
/*									*/
/************************************************************************/

static int docRtfWriteEncodedString(	RtfWritingContext *	rwc,
					TextConverter *		tc,
					int			fontEncoded,
					const char *		ss,
					int			n )
    {
    int		produced= 0;

    while( n > 0 )
	{
	int		consumed= 0;

	produced= utilTextConverterConvertFromUtf8(
				    tc, (void *)rwc, docRtfEscapeString,
				    &consumed, produced, ss, n );
	if  ( produced < 0 )
	    { LDEB(produced); return -1;	}

	ss += consumed; n -= consumed;

	if  ( n > 0 )
	    {
	    unsigned short	symbol;

	    consumed= uniGetUtf8( &symbol, (unsigned char *)ss );
	    if  ( consumed < 1 )
		{ LDEB(consumed); return -1;	}

	    if  ( fontEncoded )
		{
		BufferDocument *	bd= rwc->rwcBd;
		const DocumentFont *	df;
		TextAttribute *		ta= &(rwc->rwcTextAttribute);

		df= docRtfGetCurrentFont( bd, ta );
		if  ( df )
		    {
		    int			fontNumber;
		    int			charset= FONTcharsetDEFAULT;
		    const char *	encodingName;

		    fontNumber= docRtfWriteGetCharset( rwc, &charset,
								df, symbol );
		    if  ( fontNumber >= 0 && rwc->rwcTextCharset != charset )
			{
			docRtfWriteArgTag( rwc, "\\f", fontNumber );

			encodingName= utilGetEncodingName( df->dfName,
								    charset );
			docRtfWriteSetTextEncodingName( rwc, encodingName );
			rwc->rwcTextCharset= charset;
			continue;
			}
		    }
		}

	    docRtfEmitUnicode( rwc, symbol );

	    ss += consumed; n -= consumed;
	    }
	}

    return 0;
    }

void docRtfWriteDocEncodedString(	RtfWritingContext *	rwc,
					const char *		ss,
					int			n )
    {
    const int	fontEncoded= 0;

    docRtfWriteEncodedString( rwc, &(rwc->rwcTextConverters.rtcRtfConverter),
							fontEncoded, ss, n );
    }

void docRtfWriteFontEncodedString(	RtfWritingContext *	rwc,
					const char *		ss,
					int			n )
    {
    const int			fontEncoded= 1;
    const char *		encodingName= (const char *)0;

    encodingName= docRtfGetEncodingName( rwc->rwcBd,
				&(rwc->rwcTextAttribute), rwc->rwcTextCharset );
    if  ( ! encodingName )
	{ encodingName= docRtfWriteGetRtfEncodingName( rwc );	}

    docRtfWriteSetTextEncodingName( rwc, encodingName );

    docRtfWriteEncodedString( rwc, &(rwc->rwcTextConverters.rtcTextConverter),
							fontEncoded, ss, n );
    }

void docRtfWriteRawBytes(	RtfWritingContext *	rwc,
				const char *		ss,
				int			n )
    {
    docRtfEscapeString( (void *)rwc, 0, ss, n );
    }
