/************************************************************************/
/*									*/
/*  Save a BufferDocument into an RTF file.				*/
/*									*/
/************************************************************************/

#   include	"docBufConfig.h"

#   include	<stdlib.h>
#   include	<string.h>
#   include	<stdio.h>
#   include	<ctype.h>

#   include	<bitmap.h>

#   include	<appDebugon.h>

#   include	<charnames.h>
#   include	<utilMatchFont.h>
#   include	"docRtf.h"
#   include	"docRtfReadWrite.h"

/************************************************************************/
/*									*/
/*  Save a paragraph in RTF format.					*/
/*									*/
/*  Column counting is approximate as it is only for cosmetic reasons.	*/
/*									*/
/************************************************************************/

static void docRtfParaSaveProperties(
				RtfWritingContext *		rwc,
				int				fromDefault,
				int				saveIntbl,
				int *				pPropertyChange,
				int *				pTabsSaved,
				const ParagraphProperties *	newPP,
				const ParagraphProperties *	prevPP )
    {
    ParagraphProperties		ppp;
    PropertyMask		updMask;

    utilPropMaskClear( &updMask );

    if  ( ! fromDefault )
	{
	utilPropMaskClear( &updMask );
	utilPropMaskFill( &updMask, PPprop_FULL_COUNT );
	PROPmaskUNSET( &updMask, PPpropTABLE_NESTING );

	docParaPropertyDifference( &updMask, newPP, prevPP, &updMask );

	if  ( utilPropMaskIsEmpty( &updMask ) )
	    {
	    if  ( saveIntbl						&&
		  rwc->rwcTableNestingWritten != newPP->ppTableNesting	)
		{ docRtfSaveParaTableNesting( rwc, newPP->ppTableNesting ); }

	    return;
	    }
	}

    docInitParagraphProperties( &ppp );

    /*  1  */
    docRtfWriteTag( rwc, "\\pard" );
    *pPropertyChange= 1;

    if  ( saveIntbl )
	{ docRtfSaveParaTableNesting( rwc, newPP->ppTableNesting );	}

    utilPropMaskClear( &updMask );
    utilPropMaskFill( &updMask, PPprop_FULL_COUNT );
    PROPmaskUNSET( &updMask, PPpropTABLE_NESTING );

    docParaPropertyDifference( &updMask, newPP, &ppp, &updMask );

    docCleanParagraphProperties( &ppp );

    docRtfSaveParagraphProperties( rwc, &updMask, newPP );

    if  ( PROPmaskISSET( &updMask, PPpropTAB_STOPS )	&&
	  newPP->ppTabStopList.tslTabStopCount > 0	)
	{ *pTabsSaved= 1;	}

    return;
    }

static int docRtfSaveObject(	RtfWritingContext *		rwc,
				const BufferItem *		paraBi,
				const TextParticule *		tp )
    {
    InsertedObject *	io;
    PictureProperties *	pip;

    io= docGetObject( rwc->rwcBd, tp->tpObjectNumber );
    if  ( ! io )
	{ LXDEB(tp->tpObjectNumber,io); return -1;	}

    pip= &(io->ioPictureProperties);

    docRtfWriteNextLine( rwc );

    switch( io->ioKind )
	{
	case DOCokPICTWMETAFILE:
	    docRtfWriteDestinationBegin( rwc, "\\pict" );
	    docRtfWriteArgTag( rwc, "\\wmetafile", pip->pipMapMode );

	    if  ( pip->pipMetafileIsBitmap )
		{
		docRtfWriteTag( rwc, "\\picbmp" );

		if  ( pip->pipMetafileBitmapBpp > 0 )
		    {
		    docRtfWriteArgTag( rwc, "\\picbpp",
						pip->pipMetafileBitmapBpp );
		    }
		}

	    docRtfSavePictureTags( rwc, &(io->ioPictureProperties) );

	    docRtfWriteMemoryBuffer( rwc, &(io->ioObjectData) );
	    docRtfWriteNextLine( rwc );
	    docRtfWriteDestinationEnd( rwc );
	    return 0;

	case DOCokMACPICT:
	    docRtfWriteDestinationBegin( rwc, "\\pict" );
	    docRtfWriteTag( rwc, "\\macpict" );

	    docRtfSavePictureTags( rwc, &(io->ioPictureProperties) );

	    docRtfWriteMemoryBuffer( rwc, &(io->ioObjectData) );
	    docRtfWriteNextLine( rwc );
	    docRtfWriteDestinationEnd( rwc );

	    return 0;

	case DOCokPICTPNGBLIP:
	    docRtfWriteDestinationBegin( rwc, "\\pict" );
	    docRtfWriteTag( rwc, "\\pngblip" );

	    docRtfSavePictureTags( rwc, &(io->ioPictureProperties) );

	    docRtfWriteMemoryBuffer( rwc, &(io->ioObjectData) );
	    docRtfWriteNextLine( rwc );
	    docRtfWriteDestinationEnd( rwc );
	    return 0;

	case DOCokPICTJPEGBLIP:
	    docRtfWriteDestinationBegin( rwc, "\\pict" );
	    docRtfWriteTag( rwc, "\\jpegblip" );

	    docRtfSavePictureTags( rwc, &(io->ioPictureProperties) );

	    docRtfWriteMemoryBuffer( rwc, &(io->ioObjectData) );
	    docRtfWriteNextLine( rwc );
	    docRtfWriteDestinationEnd( rwc );
	    return 0;

	case DOCokPICTEMFBLIP:
	    docRtfWriteDestinationBegin( rwc, "\\pict" );
	    docRtfWriteTag( rwc, "\\emfblip" );

	    docRtfSavePictureTags( rwc, &(io->ioPictureProperties) );

	    docRtfWriteMemoryBuffer( rwc, &(io->ioObjectData) );
	    docRtfWriteNextLine( rwc );
	    docRtfWriteDestinationEnd( rwc );
	    return 0;

	case DOCokOLEOBJECT:
	    docRtfWriteDestinationBegin( rwc, "\\object" );


	    switch( io->ioRtfEmbedKind )
		{
		case EMBEDkindOBJEMB:
		    docRtfWriteTag( rwc, "\\objemb" );
		    break;
		case EMBEDkindOBJLINK:
		    docRtfWriteTag( rwc, "\\objlink" );
		    break;
		case EMBEDkindOBJAUTLINK:
		    docRtfWriteTag( rwc, "\\objautlink" );
		    break;
		case EMBEDkindOBJSUB:
		    docRtfWriteTag( rwc, "\\objsub" );
		    break;
		case EMBEDkindOBJPUB:
		    docRtfWriteTag( rwc, "\\objpub" );
		    break;
		case EMBEDkindOBJICEMB:
		    docRtfWriteTag( rwc, "\\objicemb" );
		    break;
		case EMBEDkindOBJHTML:
		    docRtfWriteTag( rwc, "\\objhtml" );
		    break;
		case EMBEDkindOBJOCX:
		    docRtfWriteTag( rwc, "\\objocx" );
		    break;
		default:
		    LDEB(io->ioRtfEmbedKind);
		    break;
		}

	    switch( io->ioRtfResultKind )
		{
		case RESULTkindUNKNOWN:
		    break;
		case RESULTkindRTF:
		    docRtfWriteTag( rwc, "\\rsltrtf" );
		    break;
		case RESULTkindTXT:
		    docRtfWriteTag( rwc, "\\rslttxt" );
		    break;
		case RESULTkindPICT:
		    docRtfWriteTag( rwc, "\\rsltpict" );
		    break;
		case RESULTkindBMP:
		    docRtfWriteTag( rwc, "\\rsltbmp" );
		    break;
		case RESULTkindHTML:
		    docRtfWriteTag( rwc, "\\rslthtml" );
		    break;
		default:
		    LDEB(io->ioRtfResultKind);
		    break;
		}

	    if  ( io->ioObjectClass )
		{
		const int	addSemicolon= 0;

		docRtfWriteDocEncodedStringDestination( rwc, "\\*\\objclass",
				io->ioObjectClass, strlen( io->ioObjectClass ),
				addSemicolon );
		}

	    if  ( io->ioObjectName )
		{
		const int	addSemicolon= 0;

		docRtfWriteDocEncodedStringDestination( rwc, "\\*\\objname",
				io->ioObjectName, strlen( io->ioObjectName ),
				addSemicolon );
		}

	    docRtfWriteArgTag( rwc, "\\objw", io->ioTwipsWide );
	    docRtfWriteArgTag( rwc, "\\objh", io->ioTwipsHigh );
	    if  ( io->ioScaleXSet != 100 )
		{ docRtfWriteArgTag( rwc, "\\objscalex", io->ioScaleXSet ); }
	    if  ( io->ioScaleYSet != 100 )
		{ docRtfWriteArgTag( rwc, "\\objscaley", io->ioScaleYSet ); }

	    docRtfWriteDestinationBegin( rwc, "\\*\\objdata" );
	    docRtfWriteMemoryBuffer( rwc, &io->ioObjectData );
	    docRtfWriteNextLine( rwc );
	    docRtfWriteDestinationEnd( rwc );

	    if  ( io->ioResultKind == DOCokPICTWMETAFILE )
		{
		docRtfWriteDestinationBegin( rwc, "\\result" );
		docRtfWriteDestinationBegin( rwc, "\\pict" );
		docRtfWriteArgTag( rwc, "\\wmetafile", pip->pipMapMode );

		docRtfSavePictureTags( rwc, &(io->ioPictureProperties) );

		docRtfWriteMemoryBuffer( rwc, &io->ioResultData );
		docRtfWriteNextLine( rwc );
		docRtfWriteDestinationEnd( rwc );

		docRtfWriteDestinationEnd( rwc );
		}

	    docRtfWriteDestinationEnd( rwc );
	    return 0;

	case DOCokEPS_FILE:
	    return 0;

	case DOCokDRAWING_SHAPE:
	    LSDEB(io->ioKind,docObjectKindStr(io->ioKind));
	    return 0;

	default:
	    LSDEB(io->ioKind,docObjectKindStr(io->ioKind));
	    return -1;
	}

    /*  Not reached ..
    return 0;
    */
    }

/************************************************************************/
/*									*/
/*  Finish/Begin writing a hyperlink.					*/
/*									*/
/************************************************************************/

static void docRtfFinishFldrslt(	RtfWritingContext *	rwc )
    {
    docRtfWriteDestinationEnd( rwc );
    docRtfWriteDestinationEnd( rwc );
    docRtfWriteDestinationEnd( rwc );

    rwc->rwcInFldrslt--;

    if  ( docRtfPopAttribute( rwc ) )
	{ LDEB(1);	}

    return;
    }

static void docRtfWriteStartField(	RtfWritingContext *	rwc,
					const DocumentField *	df )
    {
    const char *		bytes= (char *)df->dfInstructions.mbBytes;
    int				byteCount= df->dfInstructions.mbSize;

    docRtfWriteDestinationBegin( rwc, "\\field" );
    docRtfWriteDestinationBegin( rwc, "\\*\\fldinst" );
    docRtfWriteDestinationBegin( rwc, "" );

    while( byteCount > 1 && bytes[byteCount- 1] == ' ' )
	{ byteCount--;	}

    docRtfReserveColumns( rwc, byteCount );

    docRtfWriteDocEncodedString( rwc, bytes, byteCount );
    docRtfWriteDocEncodedString( rwc, " ", 1 );

    docRtfWriteDestinationEnd( rwc );
    docRtfWriteDestinationEnd( rwc );

    docRtfWriteDestinationBegin( rwc, "\\fldrslt" );
    docRtfWriteDestinationBegin( rwc, "" );

    rwc->rwcInFldrslt++;

    if  ( docRtfPushAttribute( rwc ) )
	{ LDEB(1);	}

    return;
    }

static void docRtfBookmarkTag(	RtfWritingContext *	rwc,
				const char *		tag,
				const DocumentField *	df )
    {
    const char *		markName= (char *)0;
    int				markSize;
    const int			addSemicolon= 0;

    if  ( ! docFieldGetBookmark( df, &markName, &markSize ) )
	{
	docRtfWriteDocEncodedStringDestination( rwc, tag,
			    (const char *)markName, markSize, addSemicolon );
	}

    return;
    }

static void docRtfStartTc(	RtfWritingContext *	rwc,
				const DocumentField *	df )
    {
    const FieldKindInformation *	fki= DOC_FieldKinds+ df->dfKind;
    SimpleOutputStream *		sos= rwc->rwcSosOut;

    docRtfWriteDestinationBegin( rwc, fki->fkiRtfTag );

    sioOutWriteBytes( sos, df->dfInstructions.mbBytes,
					    df->dfInstructions.mbSize );
    rwc->rwcCol += df->dfInstructions.mbSize;

    /* The field instructions actually contain tags */
    rwc->rwcAfter= RTFafterTAG;

    return;
    }

/************************************************************************/
/*									*/
/*  Save a footnote.							*/
/*									*/
/************************************************************************/

static int docRtfSaveNote(		RtfWritingContext *	rwc,
					const DocumentNote *	dn )
    {
    if  ( dn->dnDocumentTree.eiRoot )
	{
	const int		evenIfAbsent= 0;
	const int		forcePar= 0;
	const NoteProperties *	np= &(dn->dnNoteProperties);

	switch( np->npExternalItemKind )
	    {
	    case DOCinFOOTNOTE:
		if  ( docRtfSaveExternalItem( rwc, "\\footnote",
			dn->dnDocumentTree.eiRoot, evenIfAbsent, forcePar ) )
		    { LDEB(np->npExternalItemKind); return -1;	}
		break;

	    case DOCinENDNOTE:
		if  ( docRtfSaveExternalItem( rwc, "\\footnote\\ftnalt",
			dn->dnDocumentTree.eiRoot, evenIfAbsent, forcePar ) )
		    { LDEB(np->npExternalItemKind); return -1;	}
		break;

	    default:
		LDEB(np->npExternalItemKind);
		break;
	    }
	}

    return 0;
    }

/************************************************************************/
/*									*/
/*  Save a paragraph.							*/
/*									*/
/*  1)  When we are saving a selection, and the selection is inside a	*/
/*	table cell, do not set the \intbl flag.				*/
/*  2)  To make WP 8 happy, always save 'intbl' for the first paragraph	*/
/*	in a table row.							*/
/*									*/
/************************************************************************/

static int docRtfSaveNoteField(	RtfWritingContext *	rwc,
				int *			pCount,
				const BufferItem *	paraBi,
				const int		part,
				const TextParticule *	tp,
				const DocumentField *	df )
    {
    const DocumentNote *	dn;
    const NoteProperties *	np;
    int				count;
    int				closed;

    const char *		first= (const char *)0;
    int				attrNr= -1;
    const int			paralen= docParaStrlen( paraBi );
    int				stroff= paralen;

    count= docCountParticulesInField( paraBi, &closed, part,
						paraBi->biParaParticuleCount );

    /*  Prefer the attributes of the field result */
    if  ( count > 0 && tp[1].tpKind == DOCkindSPAN )
	{
	stroff= tp[1].tpStroff;
	attrNr= tp[1].tpTextAttrNr;
	}
    else{
	stroff= tp[0].tpStroff+ tp[0].tpStrlen;
	attrNr= tp[0].tpTextAttrNr;
	}

    if  ( stroff < paralen )
	{ first= (const char *)docParaString( paraBi, stroff );	}

    docRtfWriteSwitchTextAttributes( rwc, attrNr, first );

    dn= docGetNoteOfField( df, rwc->rwcBd );
    if  ( ! dn )
	{ XDEB(dn); return -1;	}
    np= &(dn->dnNoteProperties);

    if  ( np->npAutoNumber )
	{ docRtfWriteTag( rwc, "\\chftn" );	}
    else{
	docRtfWriteFontEncodedString( rwc,
				(const char *)np->npFixedText.mbBytes,
				np->npFixedText.mbSize );
	}

    if  ( df->dfNoteIndex >= 0		&&
	  docRtfSaveNote( rwc, dn )	)
	{ LDEB(df->dfKind); return -1;	}

    *pCount= count;
    return 0;
    }

/************************************************************************/
/*									*/
/*  Save a field head particule						*/
/*									*/
/************************************************************************/

static int docRtfSaveFieldHead(		RtfWritingContext *		rwc,
					const BufferItem *		paraBi,
					int				part,
					const TextParticule *		tp )
    {
    const DocumentField *		df;
    const FieldKindInformation *	fki;

    df= docGetFieldByNumber( &(rwc->rwcBd->bdFieldList), tp->tpObjectNumber );
    if  ( ! df )
	{ LPDEB(tp->tpObjectNumber,df); return 0;	}

    fki= DOC_FieldKinds+ df->dfKind;

    if  ( fki->fkiIsFieldInRtf )
	{
	if  ( df->dfKind != DOCfkMERGEFIELD				||
	      ! ( rwc->rwcSaveFlags & RTFflagNO_MERGEFIELDS )		)
	    { docRtfWriteStartField( rwc, df );	}
	}

    if  ( df->dfKind == DOCfkBOOKMARK			&&
	  ! ( rwc->rwcSaveFlags & RTFflagNO_BOOKMARKS )	)
	{ docRtfBookmarkTag( rwc, "\\*\\bkmkstart", df );	}

    if  ( df->dfKind == DOCfkTC || df->dfKind == DOCfkTCN )
	{ docRtfStartTc( rwc, df ); }

    if  ( df->dfKind == DOCfkCHFTN )
	{
	int		count= 0;

	if  ( docRtfSaveNoteField( rwc, &count, paraBi, part, tp, df ) )
	    { LDEB(df->dfKind); return -1;	}

	return count;
	}

    if  ( df->dfKind == DOCfkLISTTEXT	&&
	  paraBi->biParaListOverride > 0	)
	{
	if  ( docRtfPushAttribute( rwc ) )
	    { LDEB(1);	}

	docRtfWriteDestinationBegin( rwc, "\\listtext" );
	}

    if  ( fki->fkiIsDestinationInRtf	&&
	  fki->fkiRtfTag		)
	{
	docRtfWriteDestinationBegin( rwc, fki->fkiRtfTag );

	docRtfWriteDocEncodedString( rwc,
		(char *)df->dfData.mbBytes, df->dfData.mbSize );

	docRtfWriteDestinationBegin( rwc, "" );

	docRtfWriteDocEncodedString( rwc,
			(char *)df->dfInstructions.mbBytes,
			df->dfInstructions.mbSize );

	docRtfWriteDestinationEnd( rwc );
	docRtfWriteDestinationEnd( rwc );
	}

    return 0;
    }

/************************************************************************/
/*									*/
/*  Save a field tail particule						*/
/*									*/
/************************************************************************/

static int docRtfSaveFieldTail(		RtfWritingContext *		rwc,
					const BufferItem *		paraBi,
					const TextParticule *		tp )
    {
    const DocumentField *		df;
    const FieldKindInformation *	fki;

    if  ( tp->tpObjectNumber < 0 )
	{ LDEB(tp->tpObjectNumber); return 0; }

    df= docGetFieldByNumber( &(rwc->rwcBd->bdFieldList), tp->tpObjectNumber );
    if  ( ! df )
	{ LPDEB(tp->tpObjectNumber,df); return 0;	}

    fki= DOC_FieldKinds+ df->dfKind;

    if  ( df->dfKind == DOCfkBOOKMARK			&&
	  ! ( rwc->rwcSaveFlags & RTFflagNO_BOOKMARKS )	)
	{ docRtfBookmarkTag( rwc, "\\*\\bkmkend", df ); }

    if  ( df->dfKind == DOCfkTC || df->dfKind == DOCfkTCN )
	{ docRtfWriteDestinationEnd( rwc ); }

    if  ( fki->fkiIsFieldInRtf )
	{
	if  ( df->dfKind != DOCfkMERGEFIELD		    	||
	      ! ( rwc->rwcSaveFlags & RTFflagNO_MERGEFIELDS )	)
	    {
	    if  ( rwc->rwcInFldrslt )
		{ docRtfFinishFldrslt( rwc );	}
	    }
	}

    if  ( df->dfKind == DOCfkLISTTEXT		&&
	  paraBi->biParaListOverride > 0	)
	{
	docRtfWriteDestinationEnd( rwc );

	if  ( docRtfPopAttribute( rwc ) )
	    { LDEB(1);	}
	}

    return 0;
    }

/************************************************************************/

static int docRtfSaveTagParticule(	RtfWritingContext *		rwc,
					int				kind )
    {
    switch( kind )
	{
	case DOCkindCHFTNSEP:
	    docRtfWriteTag( rwc, "\\chftnsep" );
	    break;

	case DOCkindCHFTNSEPC:
	    docRtfWriteTag( rwc, "\\chftnsepc" );
	    break;

	case DOCkindOPT_HYPH:
	    docRtfWriteTag( rwc, "\\-" );
	    rwc->rwcAfter= RTFafterTEXT;
	    break;

	case DOCkindTAB:
	    docRtfWriteTag( rwc, "\\tab" );
	    break;

	case DOCkindLINEBREAK:
	    docRtfWriteTag( rwc, "\\line" );
	    docRtfWriteNextLine( rwc );
	    break;

	case DOCkindPAGEBREAK:
	    docRtfWriteTag( rwc, "\\page" );
	    docRtfWriteNextLine( rwc );
	    break;

	case DOCkindCOLUMNBREAK:
	    docRtfWriteTag( rwc, "\\column" );
	    docRtfWriteNextLine( rwc );
	    break;

	default:
	    LDEB(kind); return -1;
	}

    return 0;
    }

/************************************************************************/
/*									*/
/*  Save a paragraph to rtf.						*/
/*									*/
/*  8)  MS-Word has a tendency to only pick up certain paragraph	*/
/*	properties, such as the tabs, at the paragraph mark. On the	*/
/*	other do superfluous paragraph marks result in empty paragraphs.*/
/*	Try to find a compromise: The \cell or \row count as a paragraph*/
/*	mark, so no need to save a paragraph mark in a table. This	*/
/*	construct was necessary to activate tabs in headers and footers.*/
/*									*/
/************************************************************************/

int docRtfSaveParaItem(		RtfWritingContext *		rwc,
				const BufferItem *		paraBi,
				const DocumentSelection *	ds )
    {
    TextParticule *		tp;

    const int			paralen= docParaStrlen( paraBi );
    int				stroffFrom= -1;
    int				stroffUpto;
    int				part= 0;

    int				saveIntbl;
    int				restartFromDefault= 0;
    int				paraPropChange= 0;
    int				tabsSaved= 0;

    PropertyMask		ppChgMask;
    PropertyMask		ppUpdMask;

    DocumentPosition		dpLast;

    saveIntbl= ! ds || ! docSelectionInsideCell( ds );

    if  ( saveIntbl )
	{
	/*  2  */
	if  ( paraBi->biParaTableNesting > 0		&&
	      paraBi->biNumberInParent == 0		&&
	      paraBi->biParent				&&
	      paraBi->biParent->biNumberInParent == 0	)
	    { restartFromDefault= 1; }
	}

    docRtfParaSaveProperties( rwc, restartFromDefault, saveIntbl,
					&paraPropChange, &tabsSaved,
					&(paraBi->biParaProperties),
					&(rwc->rwcParagraphProperties) );

    if  ( restartFromDefault || paraPropChange )
	{ docRtfWriteSwitchToPlain( rwc );	}

    if  ( paraBi->biParaParticuleCount == 0 )
	{ LDEB(paraBi->biParaParticuleCount);	}

    if  ( ds && ds->dsHead.dpBi == paraBi )
	{
	const int	lastOne= 1;

	stroffFrom= ds->dsHead.dpStroff;

	if  ( docFindParticuleOfPosition( &part, &(ds->dsHead), lastOne ) )
	    { LDEB(stroffFrom); return -1;	}
	}

    tp= paraBi->biParaParticules+ part;

    if  ( ds && ds->dsTail.dpBi == paraBi )
	{ stroffUpto= ds->dsTail.dpStroff;	}
    else{ stroffUpto= -1;			}

    for ( ; part < paraBi->biParaParticuleCount; part++, tp++ )
	{
	if  ( tp->tpKind == DOCkindSPAN		||
	      tp->tpStrlen > 0			)
	    {
	    const char *	first= (const char *)0;

	    if  ( tp->tpStroff < paralen )
		{ first= (const char *)docParaString( paraBi, tp->tpStroff ); }

	    docRtfWriteSwitchTextAttributes( rwc, tp->tpTextAttrNr, first );
	    }

	switch( tp->tpKind )
	    {
	    int		extra;

	    case DOCkindCHFTNSEP:
	    case DOCkindCHFTNSEPC:
	    case DOCkindOPT_HYPH:
	    case DOCkindTAB:
	    case DOCkindLINEBREAK:
	    case DOCkindPAGEBREAK:
	    case DOCkindCOLUMNBREAK:
		if  ( stroffUpto >= 0 && tp->tpStroff >= stroffUpto )
		    { break;	}

		if  ( docRtfSaveTagParticule( rwc, tp->tpKind ) )
		    { LSDEB(tp->tpKind,docKindStr(tp->tpKind)); return -1; }

		continue;

	    case DOCkindSPAN:
		if  ( stroffUpto >= 0 && tp->tpStroff >= stroffUpto )
		    { break;	}

		{
		int		n;
		int		stroff= tp->tpStroff;

		if  ( stroff < stroffFrom )
		    { stroff=  stroffFrom;	}

		if  ( stroffUpto >= 0				&&
		      tp->tpStroff+ tp->tpStrlen > stroffUpto	)
		    { n= stroffUpto- stroff;			}
		else{ n= tp->tpStroff+ tp->tpStrlen- stroff;	}

		if  ( rwc->rwcCol > 0				&&
		      rwc->rwcCol+ n > 72			&&
		      n < 72					)
		    { docRtfWriteNextLine( rwc ); }

		docRtfWriteFontEncodedString( rwc,
				(char *)docParaString( paraBi, stroff ), n );
		}

		continue;

	    case DOCkindOBJECT:
		if  ( stroffUpto >= 0 && tp->tpStroff >= stroffUpto )
		    { break;	}

		if  ( docRtfSaveObject( rwc, paraBi, tp ) )
		    { LDEB(tp->tpKind); }
		continue;

	    case DOCkindFIELDSTART:
		if  ( stroffUpto >= 0 && tp->tpStroff >= stroffUpto )
		    { break;	}

		extra= docRtfSaveFieldHead( rwc, paraBi, part, tp );
		if  ( extra < 0 )
		    { LDEB(extra); return -1;	}
		tp += extra; part += extra;
		continue;

	    case DOCkindFIELDEND:
		extra= docRtfSaveFieldTail( rwc, paraBi, tp );
		if  ( extra != 0 )
		    { LDEB(extra); return -1;	}
		continue;

	    default:
		LDEB(tp->tpKind);
		continue;
	    }

	break;
	}

    docInitDocumentPosition( &dpLast );
    docLastPosition( &dpLast, (BufferItem *)paraBi );

    if  ( ( rwc->rwcSaveFlags & RTFflagCLOSE_LAST )		||
	    stroffUpto < 0					||
	  ( dpLast.dpBi && stroffUpto == dpLast.dpStroff )	)
	{
	/*  8  */
	if  ( ( rwc->rwcSaveFlags & RTFflagCLOSE_LAST )			||
	      ( paraBi->biParaTableNesting == 0			&&
	        paraBi->biInExternalItem == DOCinBODY		)	||
	      ( tabsSaved && paraBi->biParaTableNesting == 0	)	||
	      paraBi->biNumberInParent < paraBi->biParent->biChildCount- 1 )
	    {
	    docRtfWriteTag( rwc, "\\par" );
	    docRtfWriteNextLine( rwc );
	    }
	}

    utilPropMaskClear( &ppChgMask );
    utilPropMaskClear( &ppUpdMask );
    utilPropMaskFill( &ppUpdMask, PPprop_FULL_COUNT );

    if  ( docUpdParaProperties( &ppChgMask, &(rwc->rwcParagraphProperties),
				    &ppUpdMask, &(paraBi->biParaProperties),
				    (const DocumentAttributeMap *)0 ) )
	{ LDEB(1); return -1;	}

    return 0;
    }

