/************************************************************************/
/*									*/
/*  Read the list table of a document.					*/
/*									*/
/************************************************************************/

#   include	"docBufConfig.h"

#   include	<stdlib.h>
#   include	<string.h>
#   include	<stdio.h>
#   include	<ctype.h>

#   include	<appDebugon.h>

#   include	"docRtf.h"

/************************************************************************/
/*									*/
/*  Remember the 'listlevel' related strings.				*/
/*									*/
/*  NOTE that some experimentation with MS-Word shows that the		*/
/*	level texts are in the document encoding. Not in the encoding	*/
/*	of the font of the bullet.					*/
/*									*/
/************************************************************************/

static int docRtfCommitListLevelText(	const RtfControlWord *	rcw,
					RtfReadingContext *	rrc )
    {
    const int	removeSemicolon= 1;

    if  ( docRtfMemoryBufferSetText( &(rrc->rrcListLevelText),
						    rrc, removeSemicolon ) )
	{ LDEB(1); return -1;	}

    return 0;
    }

static int docRtfCommitListLevelNumbers(	const RtfControlWord *	rcw,
						RtfReadingContext *	rrc )
    {
    const int	removeSemicolon= 1;

    if  ( docRtfMemoryBufferSetText( &(rrc->rrcListLevelNumbers),
						    rrc, removeSemicolon ) )
	{ LDEB(1); return -1;	}

    return 0;
    }

/************************************************************************/
/*									*/
/*  Remember the name of a list.					*/
/*									*/
/************************************************************************/

static int docRtfCommitListName(	const RtfControlWord *	rcw,
					RtfReadingContext *	rrc )
    {
    const int	removeSemicolon= 1;
    int		size= 0;

    if  ( docRtfStoreSavedText( &(rrc->rrcDocumentList.dlListName), &size,
						    rrc, removeSemicolon ) )
	{ LDEB(1); return -1; }

    return 0;
    }

static int docRtfCommitListStyleName(	const RtfControlWord *	rcw,
					RtfReadingContext *	rrc )
    {
    const int	removeSemicolon= 1;
    int		size= 0;

    if  ( docRtfStoreSavedText( &(rrc->rrcDocumentList.dlListStyleName), &size,
						    rrc, removeSemicolon ) )
	{ LDEB(1); return -1; }

    return 0;
    }

int docRtfRememberListLevelProperty(
				const RtfControlWord *	rcw,
				int			arg,
				RtfReadingContext *	rrc )
    {
    DocumentListLevel *		dll= &(rrc->rrcDocumentListLevel);

    switch( rcw->rcwID )
	{
	case LLpropSTARTAT:
	    dll->dllStartAt= arg;
	    break;

	case LLpropSTYLE:
	    dll->dllNumberStyle= arg;
	    break;

	case LLpropJUSTIFY:
	    dll->dllJustification= arg;
	    break;

	case LLpropFOLLOW:
	    dll->dllFollow= arg;
	    break;

	case LLpropLEGAL:
	    dll->dllPrevToDigits= arg != 0;
	    break;

	case LLpropNORESTART:
	    dll->dllNoRestart= arg != 0;
	    break;

	case LLpropPICTURE:
	    dll->dllPictureNumber= arg;
	    break;

	case LLpropFBIAS:
	    dll->dllFontBias= arg != 0;
	    break;

	/* LLpropTEXT */

	case LLpropTEMPLATEID:
	    dll->dllTemplateID= arg;
	    break;

	/* LLpropNUMBERS */

	case LLpropOLD:
	    dll->dllFromOld= arg != 0;
	    break;

	case LLpropPREV:
	    dll->dllUsePrevText= arg != 0;
	    break;

	case LLpropPREVSPACE:
	    dll->dllUsePrevSpace= arg != 0;
	    break;

	case LLpropINDENT:
	    dll->dllIndent= arg;
	    break;

	case LLpropSPACE:
	    dll->dllSpace= arg;
	    break;

	default:
	    SDEB(rcw->rcwWord);
	    break;
	}

    return 0;
    }


/************************************************************************/
/*									*/
/*  Remember list properties.						*/
/*									*/
/*  1)  The same tag is used in the listoverrides, so set the value in	*/
/*	both structures. (Only one is used)				*/
/*									*/
/************************************************************************/

int docRtfRememberListProperty(	const RtfControlWord *	rcw,
				int			arg,
				RtfReadingContext *	rrc )
    {
    DocumentList *	dl= &(rrc->rrcDocumentList);

    switch( rcw->rcwID )
	{
	case DLpropLISTID:
	    /*  1  */
	    dl->dlListID= arg;
	    rrc->rrcListOverride.loListID= arg;
	    break;

	case DLpropTEMPLATEID:
	    dl->dlListTemplateID= arg;
	    break;

	case DLpropHYBRID:
	    dl->dlListHybrid= arg;
	    break;

	case DLpropRESTARTHDN:
	    dl->dlRestartPerSect= arg != 0;
	    break;

	case DLpropSTYLEID:
	    dl->dlListStyleID= arg;
	    break;

	/* DLpropNAME */
	/* DLpropLEVELS */

	case RTFid_NOT_SUPPORTED:
	    break;

	default:
	    SDEB(rcw->rcwWord);
	    break;
	}

    return 0;
    }

static RtfControlWord docRtfListLevelGroups[]=
{
    RTF_TEXT_GROUP( "leveltext",	LLpropTEXT,
					    docRtfCommitListLevelText ),
    RTF_BYTE_GROUP( "levelnumbers",	LLpropNUMBERS,
					    docRtfCommitListLevelNumbers ),

    { (char *)0, 0, 0 }
};

static int docRtfCommitListLevelProperties(	const RtfControlWord *	rcw,
						RtfReadingContext *	rrc )
    {
    BufferDocument *		bd= rrc->rrcBd;
    RtfReadingState *		rrs= rrc->rrcState;

    if  ( docListLevelFromRtfStrings( &(rrc->rrcDocumentListLevel),
		    &(rrc->rrcListLevelText), &(rrc->rrcListLevelNumbers) ) )
	{ LDEB(1); return -1;	}

    rrs->rrsParagraphProperties.ppShadingNumber= docItemShadingNumber(
					&(rrc->rrcBd->bdItemShadingList),
					&(rrs->rrsParagraphShading) );

    if  ( rrs->rrsTextShadingChanged )
	{
	docRtfRefreshTextShading( rrc, rrs );
	PROPmaskADD( &(rrc->rrcDocumentStyle.dsTextMask), TApropSHADING );
	}

    if  ( documentListLevelSetStyle( &(rrc->rrcDocumentListLevel),
					&(rrc->rrcDocumentStyle.dsParaMask),
					&(rrs->rrsParagraphProperties),
					&(rrc->rrcDocumentStyle.dsTextMask),
					&(rrs->rrsTextAttribute) ) )
	{ LDEB(1); return -1;	}

    docRtfResetParagraphProperties( rrs );
    docRtfResetTextAttribute( rrs, bd );

    utilPropMaskClear( &(rrc->rrcDocumentStyle.dsParaMask) );
    utilPropMaskClear( &(rrc->rrcDocumentStyle.dsTextMask) );

    return 0;
    }

int docRtfReadListLevelGroup(	const RtfControlWord *	rcw,
				RtfReadingContext *	rrc )
    {
    RtfReadingState *		rrs= rrc->rrcState;

    docCleanDocumentListLevel( &(rrc->rrcDocumentListLevel) );
    docInitDocumentListLevel( &(rrc->rrcDocumentListLevel) );
    utilEmptyMemoryBuffer( &(rrc->rrcListLevelText) );
    utilEmptyMemoryBuffer( &(rrc->rrcListLevelNumbers) );

    docRtfResetParagraphProperties( rrs );
    docRtfResetTextAttribute( rrs, rrc->rrcBd );

    utilPropMaskClear( &(rrc->rrcDocumentStyle.dsParaMask) );
    utilPropMaskClear( &(rrc->rrcDocumentStyle.dsTextMask) );

    if  ( docRtfReadGroup( rcw, 0, 0, rrc,
			    docRtfListLevelGroups, docRtfRefuseText,
			    docRtfCommitListLevelProperties ) )
	{ SDEB(rcw->rcwWord); return -1;	}

    return 0;
    }

static int docRtfListLevel(	const RtfControlWord *	rcw,
				int			arg,
				RtfReadingContext *	rrc )
    {
    RtfReadingState *		rrs= rrc->rrcState;

    docCleanDocumentListLevel( &(rrc->rrcDocumentListLevel) );
    docInitDocumentListLevel( &(rrc->rrcDocumentListLevel) );
    utilEmptyMemoryBuffer( &(rrc->rrcListLevelText) );
    utilEmptyMemoryBuffer( &(rrc->rrcListLevelNumbers) );

    docRtfResetParagraphProperties( rrs );
    docRtfResetTextAttribute( rrs, rrc->rrcBd );

    utilPropMaskClear( &(rrc->rrcDocumentStyle.dsParaMask) );
    utilPropMaskClear( &(rrc->rrcDocumentStyle.dsTextMask) );

    if  ( docRtfReadListLevelGroup( rcw, rrc ) )
	{ SLDEB(rcw->rcwWord,arg); return -1;	}

    if  ( docDocumentListAddLevel( &(rrc->rrcDocumentList),
			&(rrc->rrcDocumentListLevel), (int *)0, (int *)0 ) )
	{ LDEB(1); return -1;	}

    docCleanDocumentListLevel( &(rrc->rrcDocumentListLevel) );
    docInitDocumentListLevel( &(rrc->rrcDocumentListLevel) );

    return 0;
    }

static RtfControlWord	docRtfListGroups[]=
{
    RTF_TEXT_GROUP( "listname",	DLpropNAME, docRtfCommitListName ),
    RTF_TEXT_GROUP( "liststylename", DLpropSTYLENAME, docRtfCommitListStyleName ),
    RTF_DEST_XX( "listlevel",	DLpropLEVELS,	docRtfListLevel ),

    { (char *)0, 0, 0 }
};

static int docRtfList(		const RtfControlWord *	rcw,
				int			arg,
				RtfReadingContext *	rrc )
    {
    DocumentProperties *	dp= &(rrc->rrcBd->bdProperties);

    docCleanDocumentList( &(rrc->rrcDocumentList) );
    docInitDocumentList( &(rrc->rrcDocumentList) );

    if  ( docRtfReadGroup( rcw, 0, 0, rrc,
		    docRtfListGroups, docRtfRefuseText, (RtfCommitGroup)0 ) )
	{ SLDEB(rcw->rcwWord,arg); return -1;	}

    if  ( docDocumentListTableAddList( &(dp->dpListTable),
				&(rrc->rrcDocumentList), (int *)0, (int *)0 ) )
	{ LDEB(1); return -1;	}

    docCleanDocumentList( &(rrc->rrcDocumentList) );
    docInitDocumentList( &(rrc->rrcDocumentList) );

    return 0;
    }

static RtfControlWord	docRtfListtableGroups[]=
    {
	RTF_DEST_XX( "list",	DPpropLISTTABLE,	docRtfList ),

	{ (char *)0, 0, 0 }
    };

int docRtfListTable(		const RtfControlWord *	rcw,
				int			arg,
				RtfReadingContext *	rrc )
    {
    if  ( docRtfReadGroup( rcw, 0, 0, rrc,
				docRtfListtableGroups,
				docRtfIgnoreText, (RtfCommitGroup)0 ) )
	{ SLDEB(rcw->rcwWord,arg); return -1;	}

    return 0;
    }

/************************************************************************/
/*									*/
/*  Write a list level to rtf.						*/
/*									*/
/************************************************************************/

void docRtfWriteListLevel(	RtfWritingContext *		rwc,
				const DocumentListLevel *	dll )
    {
    MemoryBuffer		mbtext;
    MemoryBuffer		mbnumbers;

    utilInitMemoryBuffer( &mbtext );
    utilInitMemoryBuffer( &mbnumbers );

    docRtfWriteDestinationBegin( rwc, "\\listlevel" );

    docRtfWriteArgTag( rwc, "\\levelnfc", dll->dllNumberStyle );
    docRtfWriteArgTag( rwc, "\\levelnfcn", dll->dllNumberStyle );
    docRtfWriteArgTag( rwc, "\\leveljc", dll->dllJustification );
    docRtfWriteArgTag( rwc, "\\leveljcn", dll->dllJustification );

    docRtfWriteArgTag( rwc, "\\levelfollow", dll->dllFollow );
    docRtfWriteArgTag( rwc, "\\levelstartat", dll->dllStartAt );

    if  ( dll->dllPrevToDigits )
	{ docRtfWriteTag( rwc, "\\levellegal" );	}
    if  ( dll->dllNoRestart )
	{ docRtfWriteTag( rwc, "\\levelnorestart" );	}

    if  ( dll->dllFromOld )
	{ docRtfWriteTag( rwc, "\\levelold" );	}
    if  ( dll->dllUsePrevText )
	{ docRtfWriteTag( rwc, "\\levelprev" );	}
    if  ( dll->dllUsePrevSpace )
	{ docRtfWriteTag( rwc, "\\levelprevspace" );	}

    if  ( dll->dllIndent != 0 )
	{ docRtfWriteArgTag( rwc, "\\levelindent", dll->dllIndent ); }
    if  ( dll->dllSpace != 0 )
	{ docRtfWriteArgTag( rwc, "\\levelspace", dll->dllSpace ); }

    if  ( docListLevelToRtfStrings( &mbtext, &mbnumbers, dll ) )
	{ LDEB(1);	}
    else{
	int			size= 0;
	const unsigned char *	bytes;

	bytes= utilMemoryBufferGetBytes( &size, &mbtext );
	if  ( size > 0 )
	    {
	    const int	addSemicolon= 1;
	    char	tag[50];

	    strcpy( tag, "\\leveltext" );

	    if  ( dll->dllTemplateID < -5 || dll->dllTemplateID > -1 )
		{
		sprintf( tag, "\\leveltext\\leveltemplateid%ld",
						    dll->dllTemplateID );
		}

	    docRtfWriteDocEncodedStringDestination( rwc, tag,
					(char *)bytes, size, addSemicolon );
	    }

	bytes= utilMemoryBufferGetBytes( &size, &mbnumbers );
	if  ( size > 0 )
	    {
	    docRtfWriteRawBytesDestination( rwc, "\\levelnumbers",
							(char *)bytes, size );
	    }
	}

    if  ( dll->dllPictureNumber >= 0 )
	{ docRtfWriteArgTag( rwc, "\\levelpicture", dll->dllPictureNumber ); }

    docRtfSaveTextAttribute( rwc, &(dll->dllTextAttributeMask),
						    &(dll->dllTextAttribute) );

    if  ( ! utilPropMaskIsEmpty( &(dll->dllParaPropertyMask) ) )
	{
	ParagraphProperties	pp;

	docInitParagraphProperties( &pp );

	pp.ppTabStopList= dll->dllTabStopList; /* HACK */
	pp.ppLeftIndentTwips= dll->dllLeftIndentTwips;
	pp.ppFirstIndentTwips= dll->dllFirstIndentTwips;

	docRtfSaveParagraphProperties( rwc, &(dll->dllParaPropertyMask), &pp );
	}

    docRtfWriteDestinationEnd( rwc );

    utilCleanMemoryBuffer( &mbtext );
    utilCleanMemoryBuffer( &mbnumbers );

    return;
    }

/************************************************************************/
/*									*/
/*  Write a listtable to rtf.						*/
/*									*/
/************************************************************************/

void docRtfWriteListTable(	RtfWritingContext *		rwc,
				const DocumentListTable *	dlt )
    {
    int				i;
    const DocumentList *	dl= dlt->dltLists;

    docRtfWriteDestinationBegin( rwc, "\\*\\listtable" );
    docRtfWriteNextLine( rwc );

    for ( i= 0; i < dlt->dltListCount; dl++, i++ )
	{
	const DocumentListLevel *	dll;
	int				lev;

	docRtfWriteDestinationBegin( rwc, "\\list" );

	if  ( dl->dlListTemplateID != -1 )
	    {
	    docRtfWriteArgTag( rwc, "\\listtemplateid", dl->dlListTemplateID );
	    }

	if  ( dl->dlListHybrid )
	    { docRtfWriteTag( rwc, "\\listhybrid" );	}
	else{ docRtfWriteTag( rwc, "\\listsimple" );	}

	if  ( dl->dlLevelCount > 0 )
	    { docRtfWriteNextLine( rwc );	}

	dll= dl->dlLevels;
	for ( lev= 0; lev < dl->dlLevelCount; dll++, lev++ )
	    {
	    docRtfWriteListLevel( rwc, dll );
	    docRtfWriteNextLine( rwc );
	    }

	if  ( dl->dlListID != -1 )
	    { docRtfWriteArgTag( rwc, "\\listid", dl->dlListID );	}

	{
	const int	addSemicolon= 1;
	const char *	listname= "";

	if  ( dl->dlListName )
	    { listname= dl->dlListName;	}

	docRtfWriteDocEncodedStringDestination( rwc, "\\listname",
				listname, strlen( listname ), addSemicolon );
	}

	if  ( dl->dlListStyleID != -1 )
	    {
	    docRtfWriteArgTag( rwc, "\\liststyleid", dl->dlListStyleID );
	    }

	docRtfWriteDestinationEnd( rwc );
	if  ( i+ 1 < dlt->dltListCount )
	    { docRtfWriteNextLine( rwc );	}
	}

    docRtfWriteDestinationEnd( rwc );
    docRtfWriteNextLine( rwc );

    return;
    }

/************************************************************************/
/*									*/
/*  Word 95 type paragraph numbers: Just remember the properties.	*/
/*									*/
/************************************************************************/

int docRtfPnProperty(		const RtfControlWord *	rcw,
				int			arg,
				RtfReadingContext *	rrc )
    {
    ParagraphNumber *		pn= &(rrc->rrcParagraphNumber);

    switch( rcw->rcwID )
	{
	case LLpropSTARTAT:
	    pn->pnStartAt= arg;
	    break;

	case LLpropSTYLE:
	    pn->pnNumberStyle= arg;
	    break;

	case LLpropJUSTIFY:
	    pn->pnJustification= arg;
	    break;

	case LLpropINDENT:
	    pn->pnIndent= arg;
	    break;

	case LLpropPREV:
	    pn->pnUsePrevText= arg != 0;
	    break;

	case LLpropSPACE:
	    pn->pnSpace= arg;
	    break;

	default:
	    SDEB(rcw->rcwWord);
	    break;
	}

    return 0;
    }


