/************************************************************************/
/*									*/
/*  Calculate page number and ref fields.				*/
/*									*/
/************************************************************************/

#   include	"docBufConfig.h"

#   include	<stdlib.h>
#   include	<string.h>
#   include	<stdio.h>

#   include	<appDebugon.h>

#   include	<utilBase26.h>
#   include	<utilRoman.h>
#   include	"docBuf.h"
#   include	"docEvalField.h"

/************************************************************************/
/*									*/
/*  Evaluate page number related fields.				*/
/*									*/
/************************************************************************/

# define PAGEREF_FIC_COUNT 8
static int docFieldGetPageref(		const DocumentField *	df,
					const char **		pMarkName,
					int *			pMarkSize )
    {
    FieldInstructionsComponent	fic[PAGEREF_FIC_COUNT];
    int				n;
    int				comp;
    unsigned char *		bytes= df->dfInstructions.mbBytes;

    char *			markName= (char *)0;
    int				markSize= 0;

    if  ( df->dfKind != DOCfkPAGEREF )
	{ return -1;	}

    n= docSplitFieldInstructions( &(df->dfInstructions),
						fic, PAGEREF_FIC_COUNT );
    if  ( n < 2 )
	{ LDEB(n); return -1;	}

    comp= 0;
    if  ( fic[comp].ficSize != 7					||
	  memcmp( bytes+ fic[comp].ficOffset, "PAGEREF", 7 )	)
	{ SDEB((char *)bytes); return -1;	}
    comp++;

    markName= (char *)bytes+ fic[comp].ficOffset;
    markSize= fic[comp].ficSize;
    comp++;

    if  ( comp < n						&&
	  fic[comp].ficSize == 2					&&
	  ! memcmp( bytes+ fic[comp].ficOffset, "\\h", 2 )	)
	{ comp++;	}

    while( comp < n )
	{ LSDEB(comp,(char *)bytes+ fic[comp].ficOffset); comp++; }

    *pMarkName= markName;
    *pMarkSize= markSize;
    return 0;
    }

/************************************************************************/
/*									*/
/*  Format a page number to be used in a certain paragraph.		*/
/*									*/
/************************************************************************/

static int docGetPageNumberOffset(	const BufferItem *	sectBi )
    {
    if  ( ! sectBi->biParent )
	{ XDEB(sectBi->biParent); return 0;	}

    while( sectBi->biNumberInParent > 0 )
	{
	if  ( sectBi->biSectRestartPageNumbers )
	    { break;	}

	sectBi= sectBi->biParent->biChildren[sectBi->biNumberInParent- 1];
	}

    if  ( sectBi->biSectRestartPageNumbers )
	{
	return sectBi->biTopPosition.lpPage- sectBi->biSectStartPageNumber;
	}
    else{ return sectBi->biTopPosition.lpPage; }
    }

static void docFormatPageNumber(	char *			target,
					int			targetSize,
					const BufferItem *	sectBi,
					int			pageNumber )
    {
    int			style= sectBi->biSectPageNumberStyle;

    if  ( targetSize < 20 )
	{ LDEB(targetSize); strcpy( target, "?" ); return;	}

    pageNumber -= docGetPageNumberOffset( sectBi );

    switch( style )
	{
	default:
	    LDEB(style);
	    /*FALLTHROUGH*/

	case DOCpgnDEC:
	    sprintf( target, "%d", pageNumber+ 1 );
	    break;

	case DOCpgnUCLTR:
	    if  ( utilBase26String( target, targetSize, pageNumber+ 1, 1 ) )
		{ LDEB(pageNumber); return ;	}
	    break;

	case DOCpgnLCLTR:
	    if  ( utilBase26String( target, targetSize, pageNumber+ 1, 0 ) )
		{ LDEB(pageNumber); return ;	}
	    break;

	case DOCpgnUCRM:
	    if  ( utilRomanString( target, targetSize, pageNumber+ 1, 1 ) )
		{ sprintf( target, "UCRM:%d", pageNumber+ 1 );	}
	    break;

	case DOCpgnLCRM:
	    if  ( utilRomanString( target, targetSize, pageNumber+ 1, 0 ) )
		{ sprintf( target, "lcrm:%d", pageNumber+ 1 );	}
	    break;
	}

    return;
    }

/************************************************************************/
/*									*/
/*  Return the value of a pageref field.				*/
/*									*/
/************************************************************************/

int docCalculatePagerefFieldString(
				int *				pCalculated,
				MemoryBuffer *			mbResult,
				const BufferItem *		paraBi,
				int				part,
				int				partCount,
				const DocumentField *		dfRef,
				const RecalculateFields *	rf )
    {
    char			scratch[100+1];
    BufferItem *		bodyBi= rf->rfBd->bdBody.eiRoot;
    DocumentField *		dfMark;
    int				pageNumber;
    int				i;
    const BufferItem *		sectBi= (const BufferItem *)0;

    const char *		markName;
    int				markSize;

    int				n;

    if  ( docFieldGetPageref( dfRef, &markName, &markSize ) )
	{ LDEB(1); *pCalculated= 0; return 0;	}

    n= docGetBookmarkField( &dfMark, &(rf->rfBd->bdFieldList),
							markName, markSize );
    if  ( n < 0 )
	{ /* SLLDEB(markName,markSize,n); */ *pCalculated= 0; return 0;	}

    pageNumber= dfMark->dfPage;

    for ( i= 0; i < bodyBi->biChildCount; i++ )
	{
	sectBi= bodyBi->biChildren[i];
	if  ( sectBi->biBelowPosition.lpPage >= pageNumber )
	    { break;	}
	}

    if  ( i >= bodyBi->biChildCount )
	{ LDEB(1); *pCalculated= 0; return 0;	}

    docFormatPageNumber( scratch, sizeof(scratch)-1, sectBi, pageNumber );

    utilAddToMemoryBuffer( mbResult,
			    (unsigned char *)scratch, strlen( scratch ) );
    *pCalculated= 1;
    return 0;
    }

/************************************************************************/
/*									*/
/*  Return the number of the current section.				*/
/*									*/
/************************************************************************/

int docCalculateSectionFieldString(
				int *				pCalculated,
				MemoryBuffer *			mbResult,
				const BufferItem *		paraBi,
				int				part,
				int				partCount,
				const DocumentField *		df,
				const RecalculateFields *	rf )
    {
    char		scratch[100+1];
    const BufferItem *	sectBi;

    sectBi= docGetSectItem( (BufferItem *)paraBi );
    if  ( ! sectBi )
	{ XDEB(sectBi); *pCalculated= 0; return 0;	}

    if  ( sectBi->biInExternalItem != DOCinBODY )
	{
	if  ( ! sectBi->biSectExternalUseForSectBi )
	    {
	    XDEB(sectBi->biSectExternalUseForSectBi);
	    *pCalculated= 0; return 0;
	    }

	sectBi= sectBi->biSectExternalUseForSectBi;
	}

    sprintf( scratch, "%d", sectBi->biNumberInParent+ 1 );
    utilAddToMemoryBuffer( mbResult,
			    (unsigned char *)scratch, strlen( scratch ) );

    *pCalculated= 1; return 0;
    }

int docCalculateSectionPagesFieldString(
				int *				pCalculated,
				MemoryBuffer *			mbResult,
				const BufferItem *		paraBi,
				int				part,
				int				partCount,
				const DocumentField *		df,
				const RecalculateFields *	rf )
    {
    char		scratch[100+1];
    const BufferItem *	sectBi;

    sectBi= docGetSectItem( (BufferItem *)paraBi );
    if  ( ! sectBi )
	{ XDEB(sectBi); *pCalculated= 0; return 0;	}

    if  ( sectBi->biInExternalItem != DOCinBODY )
	{
	if  ( ! sectBi->biSectExternalUseForSectBi )
	    {
	    XDEB(sectBi->biSectExternalUseForSectBi);
	    *pCalculated= 0; return 0;
	    }

	sectBi= sectBi->biSectExternalUseForSectBi;
	}

    sprintf( scratch, "%d",
	sectBi->biBelowPosition.lpPage- sectBi->biTopPosition.lpPage+ 1 );
    utilAddToMemoryBuffer( mbResult,
			    (unsigned char *)scratch, strlen( scratch ) );

    *pCalculated= 1; return 0;
    }

/************************************************************************/
/*									*/
/*  Return the value of a page/numpages field.				*/
/*									*/
/************************************************************************/

int docCalculatePageFieldString(
				int *				pCalculated,
				MemoryBuffer *			mbResult,
				const BufferItem *		paraBi,
				int				part,
				int				partCount,
				const DocumentField *		df,
				const RecalculateFields *	rf )
    {
    char		scratch[100+1];
    const BufferItem *	headerFooterBi;

    headerFooterBi= paraBi->biParent;
    headerFooterBi= headerFooterBi->biParent;
    headerFooterBi= headerFooterBi->biParent;

    if  ( ! headerFooterBi->biSectExternalUseForSectBi )
	{ XDEB(headerFooterBi->biSectExternalUseForSectBi); return -1; }

    docFormatPageNumber( scratch, sizeof(scratch)-1,
			    headerFooterBi->biSectExternalUseForSectBi,
			    headerFooterBi->biSectExternalUseForPage );
    utilAddToMemoryBuffer( mbResult,
			    (unsigned char *)scratch, strlen( scratch ) );

    *pCalculated= 1; return 0;
    }

int docCalculateNumpagesFieldString(
				int *				pCalculated,
				MemoryBuffer *			mbResult,
				const BufferItem *		paraBi,
				int				part,
				int				partCount,
				const DocumentField *		df,
				const RecalculateFields *	rf )
    {
    char		scratch[100+1];
    BufferItem *	bodyBi= rf->rfBd->bdBody.eiRoot;

    sprintf( scratch, "%d", bodyBi->biBelowPosition.lpPage+ 1 );
    utilAddToMemoryBuffer( mbResult,
			    (unsigned char *)scratch, strlen( scratch ) );

    *pCalculated= 1; return 0;
    }

