/************************************************************************/
/*									*/
/*  Manage notes.							*/
/*									*/
/************************************************************************/

#   include	"docBufConfig.h"

#   include	<stdlib.h>
#   include	<string.h>
#   include	<stdio.h>

#   include	<appDebugon.h>

#   include	<appUnit.h>
#   include	"docBuf.h"
#   include	"docParaString.h"

/************************************************************************/
/*									*/
/*  Manage Footnotes/Endnotes.						*/
/*									*/
/************************************************************************/

void docInitNoteProperties(	NoteProperties *	np )
    {
    utilInitMemoryBuffer( &(np->npFixedText) );
    np->npExternalItemKind= DOCinFOOTNOTE;
    np->npAutoNumber= 1;
    }

void docCleanNoteProperties(	NoteProperties *	np )
    {
    utilCleanMemoryBuffer( &(np->npFixedText) );
    }

void docInitNote(		DocumentNote *		dn )
    {
    docInitExternalItem( &(dn->dnDocumentTree) );

    dn->dnNoteNumber= -1;
    dn->dnReferringPage= -1;
    dn->dnReferringColumn= -1;

    docInitNoteProperties( &(dn->dnNoteProperties) );

    return;
    }

int docCopyNoteProperties(	NoteProperties *	to,
				const NoteProperties *	from )
    {
    if  ( utilCopyMemoryBuffer( &(to->npFixedText), &(from->npFixedText) ) )
	{ LDEB(1); return -1;	}

    to->npExternalItemKind= from->npExternalItemKind;
    to->npAutoNumber= from->npAutoNumber;

    return 0;
    }

int docEqualNoteProperties(	const NoteProperties *	np1,
				const NoteProperties *	np2 )
    {
    if  ( np1->npExternalItemKind != np2->npExternalItemKind	||
	  np1->npAutoNumber != np2->npAutoNumber		)
	{ return 0;	}

    return utilEqualMemoryBuffer( &(np1->npFixedText), &(np2->npFixedText) );
    }

void docCleanNote(		BufferDocument *	bd,
				DocumentNote *		dn )
    {
    docCleanExternalItem( bd, &(dn->dnDocumentTree) );
    docCleanNoteProperties( &(dn->dnNoteProperties) );

    return;
    }

/************************************************************************/
/*									*/
/*  Insert a Footnote/Endnote in a BufferItem.				*/
/*									*/
/*  1)  Look for a note or the place to insert a new one.		*/
/*	NOTE that positions with a negative paragraph number are always	*/
/*	skipped.							*/
/*	NOTE that when a position for a new note is found, its index	*/
/*	is that of its successor, that usually will be shifted away.	*/
/*  2)  When the previous position is an empty hole, use it in stead	*/
/*	of shifting the successors away.				*/
/*  3)  Make a guess about the note number. If the guess is wrong, it	*/
/*	fixed later. Fixing can involve a reformat, so making not too	*/
/*	bad a guess here does help.					*/
/*									*/
/************************************************************************/

static void docSetNote(		DocumentNote *		fresh,
				int			autoNumber )
    {
    docInitNote( fresh );

    /*  3  */
    if  ( autoNumber )
	{ fresh->dnNoteNumber= autoNumber;	}
    else{ fresh->dnNoteNumber= 0;		}

    fresh->dnNoteProperties.npAutoNumber= autoNumber;

    return;
    }

/************************************************************************/
/*									*/
/*  Insert a note or an annotation.					*/
/*									*/
/*  1)  More than one annotation can be attached to the same location.	*/
/*									*/
/************************************************************************/

int docInsertNote(			DocumentNote **		pDn,
					BufferDocument *	bd,
					DocumentField *		dfNote,
					int			autoNumber )
    {
    NotesList *		nl= &(bd->bdNotesList);
    DocumentNote *	fresh;

    int			i;

    fresh= realloc( nl->nlNotes, ( nl->nlNoteCount+ 1 )* sizeof(DocumentNote) );
    if  ( ! fresh )
	{ LXDEB(nl->nlNoteCount,fresh); return -1;	}
    nl->nlNotes= fresh;

    for ( i= 0; i < nl->nlNoteCount; i++ )
	{
	if  ( NOTE_IS_DELETED( &(fresh[i]) ) )
	    { break;	}
	}

    docSetNote( fresh+ i, autoNumber );
    
    if  ( i >= nl->nlNoteCount )
	{ nl->nlNoteCount++;	}

    dfNote->dfNoteIndex= i;
    *pDn= fresh+ i; return i;
    }

/************************************************************************/
/*									*/
/*  Count the number of notes in the document.				*/
/*									*/
/************************************************************************/

int docCountNotes(		const BufferDocument *	bd )
    {
    const NotesList *	nl= &(bd->bdNotesList);
    int			count= 0;

    int			i;
    DocumentNote *	dn;

    dn= nl->nlNotes;
    for ( i= 0; i < nl->nlNoteCount; dn++, i++ )
	{
	if  ( NOTE_IS_DELETED( dn ) )
	    { continue;	}

	count++;
	}

    return count;
    }

/************************************************************************/
/*									*/
/*  Return the first note on a page.					*/
/*									*/
/************************************************************************/

DocumentField * docGetFirstNoteFromColumn(
				DocumentNote **		pDn,
				const BufferDocument *	bd,
				int			page,
				int			column,
				int			extItKind )
    {
    DocumentField *	dfNote;
    DocumentNote *	dn;

    dfNote= docGetFirstNoteOfDocument( &dn, bd, extItKind );
    while( dfNote )
	{
	if  ( dn->dnReferringPage > page )
	    { break;	}

	if  ( dn->dnReferringPage == page )
	    {
	    if  ( dn->dnReferringColumn > column )
		{ break;	}

	    if  ( dn->dnReferringColumn == column )
		{ *pDn= dn; return dfNote;	}
	    }

	dfNote= docGetNextNoteInDocument( &dn, bd, dfNote, extItKind );
	}

    return (DocumentField *)0;
    }

DocumentField * docGetFirstNoteInColumn(
				DocumentNote **		pDn,
				const BufferDocument *	bd,
				int			page,
				int			column,
				int			extItKind )
    {
    DocumentField *	dfNote;
    DocumentNote *	dn;

    dfNote= docGetFirstNoteOfDocument( &dn, bd, extItKind );
    while( dfNote )
	{
	DocumentTree *	ei= &(dn->dnDocumentTree);

	if  ( ei->eiPageFormattedFor > page )
	    { break;	}

	if  ( ei->eiPageFormattedFor == page )
	    {
	    if  ( ei->eiColumnFormattedFor > column )
		{ break;	}

	    if  ( ei->eiColumnFormattedFor == column )
		{ *pDn= dn; return dfNote;	}
	    }

	dfNote= docGetNextNoteInDocument( &dn, bd, dfNote, extItKind );
	}

    return (DocumentField *)0;
    }

DocumentField * docGetLastNoteInColumn(
				DocumentNote **		pDn,
				const BufferDocument *	bd,
				int			page,
				int			column,
				int			extItKind )
    {
    DocumentField *	dfRet= (DocumentField *)0;
    DocumentNote *	dnRet= (DocumentNote *)0;

    DocumentField *	dfNote;
    DocumentNote *	dn;

    dfNote= docGetFirstNoteOfDocument( &dn, bd, extItKind );
    while( dfNote )
	{
	DocumentTree *	ei= &(dn->dnDocumentTree);

	if  ( ei->eiPageFormattedFor > page )
	    { break;	}

	if  ( ei->eiPageFormattedFor == page )
	    {
	    if  ( ei->eiColumnFormattedFor > column )
		{ break;	}

	    if  ( ei->eiColumnFormattedFor == column )
		{ dnRet= dn; dfRet= dfNote;	}
	    }

	dfNote= docGetNextNoteInDocument( &dn, bd, dfNote, extItKind );
	}

    *pDn= dnRet;
    return dfRet;
    }

DocumentField * docGetFirstNoteOnPage(
				DocumentNote **		pDn,
				const BufferDocument *	bd,
				int			page,
				int			extItKind )
    {
    DocumentField *	dfNote;
    DocumentNote *	dn;

    dfNote= docGetFirstNoteOfDocument( &dn, bd, extItKind );
    while( dfNote )
	{
	DocumentTree *	ei= &(dn->dnDocumentTree);

	if  ( ei->eiPageFormattedFor > page )
	    { break;	}

	if  ( ei->eiPageFormattedFor == page )
	    { *pDn= dn; return dfNote;	}

	dfNote= docGetNextNoteInDocument( &dn, bd, dfNote, extItKind );
	}

    return (DocumentField *)0;
    }

static DocumentField * docGetNextSectionNote(
				DocumentField *		df,
				DocumentNote **		pDn,
				const BufferDocument *	bd,
				int			sect,
				int			extItKind )
    {
    const ChildFields *		rootFields= &(bd->bdBody.eiRootFields);

    while( df )
	{
	if  ( df->dfKind == DOCfkCHFTN )
	    {
	    DocumentNote *	dn= docGetNoteOfField( df, bd );

	    if  ( ! dn )
		{ XDEB(dn); 	}
	    else{
		if  ( extItKind < 0					||
		      dn->dnNoteProperties.npExternalItemKind ==
							    extItKind	)
		    { *pDn= dn; return df;	}
		}
	    }

	df= docGetNextFieldInSection( rootFields, sect, df );
	}

    return (DocumentField *)0;
    }

DocumentField * docGetFirstNoteOfSection(
				DocumentNote **		pDn,
				const BufferDocument *	bd,
				int			sect,
				int			extItKind )
    {
    const ChildFields *		rootFields= &(bd->bdBody.eiRootFields);
    DocumentField *		df;

    if  ( sect < 0 )
	{ return docGetFirstNoteOfDocument(  pDn, bd, extItKind ); }

    df= docGetFirstFieldOfSection( rootFields, sect );
    return docGetNextSectionNote( df, pDn, bd, sect, extItKind );
    }

DocumentField * docGetNextNoteInSection(
				DocumentNote **		pDn,
				const BufferDocument *	bd,
				int			sect,
				DocumentField *		df,
				int			extItKind )
    {
    const ChildFields *		rootFields= &(bd->bdBody.eiRootFields);

    if  ( sect < 0 )
	{ return docGetNextNoteInDocument(  pDn, bd, df, extItKind ); }

    df= docGetNextFieldInSection( rootFields, sect, df );
    return docGetNextSectionNote( df, pDn, bd, sect, extItKind );
    }

static DocumentField * docGetNextDocumentNote(
					DocumentField *		df,
					DocumentNote **		pDn,
					const BufferDocument *	bd,
					int			extItKind )
    {
    const ChildFields *		rootFields= &(bd->bdBody.eiRootFields);

    while( df )
	{
	if  ( df->dfKind == DOCfkCHFTN )
	    {
	    DocumentNote *	dn= docGetNoteOfField( df, bd );

	    if  ( ! dn )
		{ XDEB(dn);	}
	    else{
		if  ( extItKind < 0					||
		      dn->dnNoteProperties.npExternalItemKind ==
							    extItKind	)
		    { *pDn= dn; return df;	}
		}
	    }

	df= docGetNextField( rootFields, df );
	}

    return (DocumentField *)0;
    }

DocumentField * docGetFirstNoteOfDocument(
				DocumentNote **		pDn,
				const BufferDocument *	bd,
				int			extItKind )
    {
    const ChildFields *		rootFields= &(bd->bdBody.eiRootFields);
    DocumentField *		df;

    df= docGetFirstField( rootFields );

    return docGetNextDocumentNote( df, pDn, bd, extItKind );
    }

DocumentField * docGetNextNoteInDocument(
				DocumentNote **		pDn,
				const BufferDocument *	bd,
				DocumentField *		df,
				int			extItKind )
    {
    const ChildFields *		rootFields= &(bd->bdBody.eiRootFields);

    df= docGetNextField( rootFields, df );

    return docGetNextDocumentNote( df, pDn, bd, extItKind );
    }

static DocumentField * docGetPrevDocumentNote(
					DocumentField *		df,
					DocumentNote **		pDn,
					const BufferDocument *	bd,
					int			extItKind )
    {
    const ChildFields *		rootFields= &(bd->bdBody.eiRootFields);

    while( df )
	{
	if  ( df->dfKind == DOCfkCHFTN )
	    {
	    DocumentNote *	dn= docGetNoteOfField( df, bd );

	    if  ( ! dn )
		{ XDEB(dn); 	}
	    else{
		if  ( extItKind < 0					||
		      dn->dnNoteProperties.npExternalItemKind ==
							    extItKind	)
		    { *pDn= dn; return df;	}
		}
	    }

	df= docGetPrevField( rootFields, df );
	}

    return (DocumentField *)0;
    }

DocumentField * docGetLastNoteOfDocument(
				DocumentNote **		pDn,
				const BufferDocument *	bd,
				int			extItKind )
    {
    const ChildFields *		rootFields= &(bd->bdBody.eiRootFields);
    DocumentField *		df;

    df= docGetLastField( rootFields );

    return docGetPrevDocumentNote( df, pDn, bd, extItKind );
    }

DocumentField * docGetPrevNoteInDocument(
				DocumentNote **		pDn,
				const BufferDocument *	bd,
				DocumentField *		df,
				int			extItKind )
    {
    const ChildFields *		rootFields= &(bd->bdBody.eiRootFields);

    df= docGetPrevField( rootFields, df );

    return docGetPrevDocumentNote( df, pDn, bd, extItKind );
    }

/************************************************************************/
/*									*/
/*  Renumber the notes of a document.					*/
/*									*/
/************************************************************************/

typedef struct RenumberNotes
    {
    int				rnChanged;

    int				rnFootN;
    int				rnEndN;
    int				rnSect;
    int				rnPage;

    const NotesProperties *	rnFootnoteProperties;
    const NotesProperties *	rnEndnoteProperties;
    } RenumberNotes;

static void docRenumberNote(	DocumentNote *		dn,
				const DocumentField *	df,
				RenumberNotes *		rn )
    {
    const NotesProperties *	np;

    if  ( NOTE_IS_DELETED( dn ) )
	{ LDEB(df->dfFieldNumber); return;	}

    if  ( ! dn->dnNoteProperties.npAutoNumber )
	{
	if  ( dn->dnNoteNumber != 0 )
	    { rn->rnChanged= 1;	}

	dn->dnNoteNumber= 0;
	return;
	}

    /****/
    np= rn->rnFootnoteProperties;

    if  ( np->npRestart == FTN_RST_PER_SECTION		&&
	  df->dfSelectionScope.ssSectNr != rn->rnSect	)
	{ rn->rnFootN= 0;	}

    if  ( np->npRestart == FTN_RST_PER_PAGE		&&
	  dn->dnReferringPage != rn->rnPage		)
	{ rn->rnFootN= 0;	}

    /****/
    np= rn->rnEndnoteProperties;

    if  ( np->npRestart == FTN_RST_PER_SECTION		&&
	  df->dfSelectionScope.ssSectNr != rn->rnSect	)
	{ rn->rnEndN= 0;	}

    if  ( np->npRestart == FTN_RST_PER_PAGE		&&
	  dn->dnReferringPage != rn->rnPage		)
	{ rn->rnEndN= 0;	}

    /****/

    switch( dn->dnNoteProperties.npExternalItemKind )
	{
	case DOCinFOOTNOTE:
	    if  ( dn->dnNoteNumber != rn->rnFootN )
		{ dn->dnNoteNumber=   rn->rnFootN; rn->rnChanged= 1;	}
	    rn->rnFootN++;
	    break;

	case DOCinENDNOTE:
	    if  ( dn->dnNoteNumber != rn->rnEndN )
		{ dn->dnNoteNumber=   rn->rnEndN; rn->rnChanged= 1;	}
	    rn->rnEndN++;
	    break;

	default:
	    LDEB(dn->dnNoteProperties.npExternalItemKind);
	    break;
	}

    rn->rnSect= df->dfSelectionScope.ssSectNr;
    rn->rnPage= dn->dnReferringPage;
    return;
    }

/************************************************************************/
/*									*/
/*  Renumber notes: Scan the fields in preorder and change the numbers.	*/
/*									*/
/************************************************************************/

static void docRenumberNotesInChildFields(	const ChildFields *	cf,
						const BufferDocument *	bd,
						RenumberNotes *		rn )
    {
    int			i;

    for ( i= 0; i < cf->cfChildCount; i++ )
	{
	const DocumentField *	df= cf->cfChildren[i];

	if  ( df->dfKind == DOCfkCHFTN )
	    {
	    DocumentNote *	dn= docGetNoteOfField( df, bd );

	    if  ( ! dn )
		{ XDEB(dn); continue;	}

	    docRenumberNote( dn, df, rn );
	    }

	docRenumberNotesInChildFields( &(df->dfChildFields), bd, rn );
	}
    }

void docRenumberNotes(		int *			pChanged,
				BufferDocument *	bd )
    {
    const DocumentProperties *	dp= &(bd->bdProperties);
    RenumberNotes		rn;

    rn.rnChanged= 0;
    rn.rnFootN= 0;
    rn.rnEndN= 0;
    rn.rnPage= -1;
    rn.rnSect= -1;
    rn.rnFootnoteProperties= &(dp->dpFootnoteProperties);
    rn.rnEndnoteProperties= &(dp->dpEndnoteProperties);

    docRenumberNotesInChildFields( &(bd->bdBody.eiRootFields), bd, &rn );

    *pChanged= rn.rnChanged;
    return;
    }

/************************************************************************/
/*									*/
/*  Make a note at the given position.					*/
/*									*/
/*  1)  Claim memory for a note.					*/
/*  2)  Insert a particule that refers to the note.			*/
/*  4)  Insert an empty paragraph in the note to begin with.		*/
/*  5)  Make a chftn field to represent the note in the external	*/
/*	paragraph.							*/
/*  6)  Insert a space after the note number field. This also is to	*/
/*	work around a bug in the editing around fields.			*/
/*									*/
/*  !)  The chftn field that precedes the note that was inserted by the	*/
/*	caller.								*/
/*									*/
/************************************************************************/

int docMakeNote(	DocumentNote **			pDn,
			BufferDocument *		bd,
			DocumentField *			dfNote,
			const BufferItem *		bodyBi,
			int				extItKind,
			int				fieldKind )
    {
    TextAttribute	ta;
    int			textAttributeNumberPlain;
    int			textAttributeNumberSuper;

    DocumentNote *	dn;

    int			noteIndex;

    int			stroffShift= 0;
    const int		autoNumber= 1;

    BufferItem *	noteParaBi;

    DocumentField *	dfHead;
    DocumentSelection	dsInsideHead;
    DocumentSelection	dsAroundHead;
    int			partHead= -1;
    int			partTail= -1;

    TextParticule *	tp;

    /*  1  */
    noteIndex= docInsertNote( &dn, bd, dfNote, autoNumber );
    if  ( noteIndex < 0 )
	{ LDEB(noteIndex); return -1;	}

    dn->dnNoteProperties.npExternalItemKind= extItKind;

    docPlainTextAttribute( &ta, bd );
    ta.taFontSizeHalfPoints= 20;

    textAttributeNumberPlain=
		utilTextAttributeNumber( &(bd->bdTextAttributeList), &ta );

    /*  4  */
    noteParaBi= docMakeExternalParagraph( bd, &(dn->dnDocumentTree),
					bodyBi, textAttributeNumberPlain,
					dfNote->dfFieldNumber, extItKind );
    if  ( ! noteParaBi )
	{ XDEB(noteParaBi); return -1;	}

    {
    TextAttribute	taSet;

    taSet= ta;

    taSet.taSuperSub= DOCfontSUPERSCRIPT;

    textAttributeNumberSuper=
		utilTextAttributeNumber( &(bd->bdTextAttributeList), &taSet );
    }

    /*  5  */
    if  ( docInsertParaHeadField( &dfHead, &dsInsideHead, &dsAroundHead,
		    &partHead, &partTail,
		    noteParaBi, bd, &(dn->dnDocumentTree),
		    fieldKind, textAttributeNumberSuper ) )
	{ LDEB(1); return -1;	}
    noteParaBi->biParaParticules[partTail].tpTextAttrNr=
						    textAttributeNumberPlain;

    /*  6  */
    if  ( docParaStringReplace( &stroffShift, noteParaBi,
					    dsAroundHead.dsTail.dpStroff,
					    dsAroundHead.dsTail.dpStroff,
					    " ", 1 ) )
	{ LDEB(dsAroundHead.dsTail.dpStroff); return -1; }

    tp= docInsertTextParticule( noteParaBi, noteParaBi->biParaParticuleCount,
				    docParaStrlen( noteParaBi )- 1, 1,
				    DOCkindSPAN, textAttributeNumberPlain );
    if  ( ! tp )
	{ LDEB(noteParaBi->biParaParticuleCount); return -1; }

    *pDn= dn; return 0;
    }

/************************************************************************/
/*									*/
/*  Get the note corresponding to a selection.				*/
/*  Always return the field that owns the note.				*/
/*									*/
/************************************************************************/

DocumentField * docGetSelectedNote(
				DocumentNote **			pDn,
				BufferDocument *		bd,
				const DocumentSelection *	ds )
    {
    DocumentField *		dfNote= (DocumentField *)0;
    DocumentNote *		dn= (DocumentNote *)0;

    /*  2  */
    if  ( ds->dsSelectionScope.ssInExternalItem == DOCinFOOTNOTE	||
	  ds->dsSelectionScope.ssInExternalItem == DOCinENDNOTE		)
	{
	BufferItem *		bi= ds->dsHead.dpBi;
	int			ownerNumber;

	bi= docGetSectItem( ds->dsHead.dpBi );
	if  ( ! bi )
	    { XDEB(bi); return (DocumentField *)0;		}

	ownerNumber= bi->biSectSelectionScope.ssOwnerNumber;
	dfNote= docGetFieldByNumber( &(bd->bdFieldList), ownerNumber );
	dn= docGetNoteOfField( dfNote, bd );
	if  ( ! dn )
	    { XDEB(dn); return (DocumentField *)0;	}
	}

    if  ( ds->dsSelectionScope.ssInExternalItem == DOCinBODY )
	{
	if  ( ! docIsParaSelection( ds ) )
	    { return (DocumentField *)0;	}

	dfNote= docFindTypedFieldForPosition( bd, &(ds->dsHead), DOCfkCHFTN );
	if  ( ! dfNote )
	    { return (DocumentField *)0;	}
	else{
	    dn= docGetNoteOfField( dfNote, bd );
	    if  ( ! dn )
		{ XDEB(dn); return (DocumentField *)0;	}
	    }
	}

    if  ( ! dn || ! dfNote )
	{ /* XXDEB(dn,dfNote); */ return (DocumentField *)0;	}

    *pDn= dn;
    return dfNote;
    }

/************************************************************************/
/*									*/
/*  Verify that a certain kind of notes separator exists. Make it if	*/
/*  not.								*/
/*									*/
/*  1)  Correctly handeled by docTextAttributeNumber().			*/
/*									*/
/************************************************************************/

int docCheckNoteSeparatorItem(		BufferDocument *	bd,
					int			extItKind )
    {
    DocumentTree *	ei;
    int			particuleKind;

    switch( extItKind )
	{
	case DOCinFTNSEP:
	case DOCinAFTNSEP:
	    particuleKind= DOCkindCHFTNSEP;
	    break;

	case DOCinFTNSEPC:
	case DOCinAFTNSEPC:
	    particuleKind= DOCkindCHFTNSEPC;
	    break;

	default:
	    LDEB(extItKind);
	    particuleKind= DOCkindCHFTNSEP;
	    break;
	}

    ei= docDocumentNoteSeparator( bd, extItKind );
    if  ( ! ei )
	{ LXDEB(extItKind,ei); return -1;	}

    if  ( ! ei->eiRoot )
	{
	const int		ownerNumber= -1;
	int			attributeNumber;

	BufferItem *		paraBi;
	TextAttribute		ta;

	utilInitTextAttribute( &ta );
	ta.taFontNumber= -1; /*  1  */
	ta.taFontSizeHalfPoints= 24;

	attributeNumber= docTextAttributeNumber( bd, &ta );
	if  ( attributeNumber < 0 )
	    { LDEB(attributeNumber); return -1;	}

	paraBi= docMakeExternalParagraph( bd, ei, bd->bdBody.eiRoot,
				    attributeNumber, ownerNumber, extItKind );
	if  ( ! paraBi )
	    { XDEB(paraBi);	}
	else{
	    int		stroffShift= 0;
	    int		off= docParaStrlen( paraBi );

	    if  ( docParaStringReplace( &stroffShift, paraBi,
							off, off, "-", 1 ) )
		{ LDEB(1);	}
	    else{
		TextParticule *	tp= paraBi->biParaParticules;

		tp->tpKind= particuleKind;
		tp->tpStrlen= 1;
		}
	    }
	}

    return 0;
    }

/************************************************************************/

DocumentNote *	docGetNoteOfField(	const DocumentField *	dfNote,
					const BufferDocument *	bd )
    {
    const NotesList *	nl= &(bd->bdNotesList);

    if  ( dfNote->dfKind != DOCfkCHFTN	&&
	  dfNote->dfKind != DOCfkCHATN	)
	{
	LSDEB(dfNote->dfKind,docFieldKindStr(dfNote->dfKind));
	return (DocumentNote *)0;
	}

    if  ( dfNote->dfSelectionScope.ssInExternalItem == DOCinBODY )
	{
	if  ( dfNote->dfNoteIndex < 0			||
	      dfNote->dfNoteIndex >= nl->nlNoteCount	)
	    {
	    LLLDEB(dfNote->dfFieldNumber,dfNote->dfNoteIndex,nl->nlNoteCount);
	    return (DocumentNote *)0;
	    }

	return nl->nlNotes+ dfNote->dfNoteIndex;
	}

    if  ( dfNote->dfSelectionScope.ssInExternalItem == DOCinFOOTNOTE	||
	  dfNote->dfSelectionScope.ssInExternalItem == DOCinANNOTATION	||
	  dfNote->dfSelectionScope.ssInExternalItem == DOCinENDNOTE	)
	{
	DocumentField *	dfOwner;

	dfOwner= docGetFieldByNumber( &(bd->bdFieldList),
				dfNote->dfSelectionScope.ssOwnerNumber );

	if  ( dfOwner->dfNoteIndex < 0			||
	      dfOwner->dfNoteIndex >= nl->nlNoteCount	)
	    {
	    LLLDEB(dfOwner->dfFieldNumber,dfOwner->dfNoteIndex,nl->nlNoteCount);
	    return (DocumentNote *)0;
	    }

	return nl->nlNotes+ dfOwner->dfNoteIndex;
	}

    LDEB(dfNote->dfSelectionScope.ssInExternalItem);
    return (DocumentNote *)0;
    }

