/************************************************************************/
/*									*/
/*  Management of the list table of a document.				*/
/*  Individual list levels.						*/
/*									*/
/*  O)	Word 7.0 == Word 95 compatibility				*/
/*									*/
/************************************************************************/

#   include	"docBufConfig.h"

#   include	<stdlib.h>
#   include	<string.h>
#   include	<stdio.h>

#   include	<utilBase26.h>
#   include	<utilRoman.h>
#   include	<uniUtf8.h>

#   include	<appDebugon.h>

#   include	"docListLevel.h"

void docInitDocumentListLevel(	DocumentListLevel *	dll )
    {
    dll->dllStartAt= 1;
    dll->dllNumberStyle= 0;
    dll->dllJustification= DOCllaLEFT;
    dll->dllFollow= DOCllfTAB;
    dll->dllPrevToDigits= 0;
    dll->dllNoRestart= 0;
    dll->dllPictureNumber= -1;
    dll->dllFontBias= 0;

    dll->dllFormatString= (char *)0;
    dll->dllLevelNumbers= (LevelNumber *)0;
    dll->dllLevelNumberCount= 0;
    dll->dllTemplateID= -1;

    dll->dllFromOld= 0;			/*  O  */
    dll->dllUsePrevText= 0;		/*  O  */
    dll->dllUsePrevSpace= 0;		/*  O  */
    dll->dllIndent= 0;			/*  O  */
    dll->dllSpace= 0;			/*  O  */

    docInitTabStopList( &(dll->dllTabStopList) );
    dll->dllLeftIndentTwips= 0;
    dll->dllFirstIndentTwips= 0;
    utilPropMaskClear( &(dll->dllParaPropertyMask) );

    utilInitTextAttribute( &(dll->dllTextAttribute) );
    utilPropMaskClear( &(dll->dllTextAttributeMask) );
    }

void docCleanDocumentListLevel(	DocumentListLevel *	dll )
    {
    if  ( dll->dllFormatString )
	{ free( dll->dllFormatString ); }
    if  ( dll->dllLevelNumbers )
	{ free( dll->dllLevelNumbers ); }

    docCleanTabStopList( &(dll->dllTabStopList) );
    }

int docCopyDocumentListLevel(	DocumentListLevel *		to,
				const DocumentListLevel *	from,
				const int *			fontMap,
				const int *			colorMap )
    {
    int			rval= 0;

    char *		formatString= (char *)0;
    LevelNumber *	levelNumbers= (LevelNumber *)0;
    TabStopList		tsl;
    const int		pixels= 0;

    int			changed= 0;

    docInitTabStopList( &tsl );

    if  ( docCopyTabStopList( &changed, &tsl,
					&(from->dllTabStopList), pixels ) )
	{ LDEB(1); rval= -1; goto ready;	}

    if  ( from->dllFormatString )
	{
	int		size;

	if  ( ! from->dllLevelNumbers )
	    { PDEB(from->dllLevelNumbers); rval= -1; goto ready;	}

	size= from->dllLevelNumbers[from->dllLevelNumberCount].lnOffsetBytes;
	formatString= (char *)malloc( size+ 1 );
	if  ( ! formatString )
	    { LPDEB(size,formatString); rval= -1; goto ready;	}

	memcpy( formatString, from->dllFormatString, size );
	formatString[size]= '\0';
	}

    if  ( from->dllLevelNumbers )
	{
	int		i;
	int		count= from->dllLevelNumberCount+ 1; /* NOTE +1 */

	levelNumbers= (LevelNumber *)malloc( count* sizeof(LevelNumber) );
	if  ( ! levelNumbers )
	    { LPDEB(count,levelNumbers); rval= -1; goto ready;	}

	for ( i= 0; i < count; i++ )
	    { levelNumbers[i]= from->dllLevelNumbers[i];	}
	}

    if  ( to->dllFormatString )
	{ free( to->dllFormatString ); }
    if  ( to->dllLevelNumbers )
	{ free( to->dllLevelNumbers ); }
    docCleanTabStopList( &(to->dllTabStopList) );

    *to= *from;

    if  ( fontMap							&&
	  PROPmaskISSET( &(to->dllTextAttributeMask), TApropDOC_FONT_NUMBER ) )
	{
	to->dllTextAttribute.taFontNumber=
				fontMap[from->dllTextAttribute.taFontNumber];
	}
    if  ( colorMap							&&
	  PROPmaskISSET( &(to->dllTextAttributeMask), TApropDOC_FONT_NUMBER ) &&
	  from->dllTextAttribute.taTextColorNumber > 0			)
	{
	to->dllTextAttribute.taTextColorNumber=
			    colorMap[from->dllTextAttribute.taTextColorNumber];
	}

    /* steal */
    to->dllFormatString= formatString; formatString= (char *)0;
    to->dllLevelNumbers= levelNumbers; levelNumbers= (LevelNumber *)0;
    to->dllLevelNumberCount= from->dllLevelNumberCount;
    to->dllTabStopList= tsl; docInitTabStopList( &tsl );

  ready:

    if  ( formatString )
	{ free( formatString );	}
    if  ( levelNumbers )
	{ free( levelNumbers );	}

    docCleanTabStopList( &tsl );

    return rval;
    }

/************************************************************************/
/*									*/
/*  Set the relevant style attributes in a list level.			*/
/*									*/
/*  1)  Paragraph Properties.						*/
/*  2)  Text Attributes.						*/
/*									*/
/************************************************************************/

int documentListLevelSetStyle(	DocumentListLevel *		dll,
				const PropertyMask *		paraMask,
				const ParagraphProperties *	pp,
				const PropertyMask *		textMask,
				const TextAttribute *		ta )
    {
    /*  1  */
    utilPropMaskClear( &(dll->dllParaPropertyMask) );

    if  ( PROPmaskISSET( paraMask, PPpropTAB_STOPS ) )
	{
	const int	pixels= 0;
	int		changed= 0;

	if  ( docCopyTabStopList( &changed, &(dll->dllTabStopList),
					    &(pp->ppTabStopList), pixels ) )
	    { LDEB(1); return -1;	}

	PROPmaskADD( &(dll->dllParaPropertyMask), PPpropTAB_STOPS );
	}

    if  ( PROPmaskISSET( paraMask, PPpropLEFT_INDENT ) )
	{
	dll->dllLeftIndentTwips= pp->ppLeftIndentTwips;
	PROPmaskADD( &(dll->dllParaPropertyMask), PPpropLEFT_INDENT );
	}

    if  ( PROPmaskISSET( paraMask, PPpropFIRST_INDENT ) )
	{
	dll->dllFirstIndentTwips= pp->ppFirstIndentTwips;
	PROPmaskADD( &(dll->dllParaPropertyMask), PPpropFIRST_INDENT );
	}

    /*  2  */
    {
    PropertyMask	doneMask;

    utilPropMaskClear( &doneMask );

    dll->dllTextAttributeMask= *textMask;
    utilUpdateTextAttribute( &doneMask, &(dll->dllTextAttribute),
								ta, textMask );
    }

    return 0;
    }

/************************************************************************/
/*									*/
/*  Return the constant text that precedes field 'idx' in the listtext	*/
/*  format. (Or for idx == dll->dllNumberSizeBytes, return the trailing	*/
/*  piece.)								*/
/*									*/
/************************************************************************/

int docListLevelGetField(	int *				pConstOff,
				int *				pConstLen,
				int *				pConstChars,
				int *				pLevel,
				const DocumentListLevel *	dll,
				int				field )
    {
    int		constHead= 0;
    int		constTail= 0;
    int		level= -1;

    if  ( field < 0 || field > dll->dllLevelNumberCount )
	{ LLDEB(field,dll->dllLevelNumberCount); return -1;	}

    if  ( field > 0 )
	{ constHead= dll->dllLevelNumbers[field-1].lnOffsetBytes;	}

    constTail= dll->dllLevelNumbers[field].lnOffsetBytes;

    if  ( field < dll->dllLevelNumberCount )
	{ level= dll->dllLevelNumbers[field].lnLevel;	}

    *pConstOff= constHead;
    *pConstLen= constTail- constHead;
    *pConstChars= dll->dllLevelNumbers[field].lnConstChars;
    *pLevel= level;

    return 0;
    }

/************************************************************************/
/*									*/
/*  Get the (constant) text before a component in the level text of a	*/
/*  list. (Or for idx == dll->dllNumberSizeBytes, return the trailing	*/
/*  piece.								*/
/*									*/
/************************************************************************/

int docListLevelGetText(	char *				to,
				int				maxsize,
				const DocumentListLevel *	dll,
				int				field )
    {
    int			constOff;
    int			constLen;
    int			constChars;
    int			level;

    if  ( docListLevelGetField( &constOff, &constLen, &constChars,
							&level, dll, field ) )
	{ LDEB(field); return -1;	}
    if  ( constLen > maxsize )
	{ LLDEB(constLen,maxsize); return -1;	}

    memcpy( to, dll->dllFormatString+ constOff, constLen );
    to[constLen]= '\0';

    return constLen;
    }

/************************************************************************/
/*									*/
/*  Update a component in the level text of a list.			*/
/*									*/
/*  idx is the index in dll->dllNumberIndices. The text before the	*/
/*  number is changed. The trailing text after the last number is	*/
/*  updated if idx == dll->dllNumberSizeBytes.				*/
/*									*/
/*  1)  Determine stretch in dll->dllTextFormat				*/
/*  2)  More memory needed?						*/
/*  3)  Insert new value.						*/
/*  4)  Shift indirection via dll->dllNumberIndices.			*/
/*  5)  Remember size in dll->dllTextFormat.				*/
/*									*/
/************************************************************************/

int docListLevelSetText(	int *				pChanged,
				DocumentListLevel *		dll,
				const char *			text,
				int				field )
    {
    int			constOff;
    int			nbytes;
    int			ochars;
    int			nchars;
    int			obytes;
    int			level;

    int			sizeBytes;

    char *		from;
    int			i;

    if  ( ! dll->dllLevelNumbers )
	{ PDEB(dll->dllLevelNumbers); return -1;	}
    sizeBytes= dll->dllLevelNumbers[dll->dllLevelNumberCount].lnOffsetBytes;

    nbytes= 0; nchars= 0;
    from= (char *)text;
    while( *from )
	{
	unsigned short	symbol;
	int		step;

	step= uniGetUtf8( &symbol, (unsigned char *)from );
	if  ( step < 1 )
	    { LDEB(step); break;	}

	nchars++; nbytes += step; from += step;
	}

    /*  1  */
    if  ( docListLevelGetField( &constOff, &obytes, &ochars, &level,
								dll, field ) )
	{ LDEB(field); return -1;	}

    from= dll->dllFormatString+ constOff;
    if  ( nbytes == obytes && ! memcmp( from, text, obytes ) )
	{ *pChanged= 0; return 0;	}

    /*  2  */
    if  ( obytes < nbytes )
	{
	char *	fresh;

	fresh= realloc( dll->dllFormatString, sizeBytes+ nbytes- obytes+ 1 );
	if  ( ! fresh )
	    { LXDEB(sizeBytes+nbytes-obytes,fresh); return -1;	}
	dll->dllFormatString= fresh;
	from= dll->dllFormatString+ constOff;
	}

    /*  3  */
    memmove( from+ nbytes, from+ obytes, sizeBytes- constOff- obytes+ 1 );
    memcpy( from, text, nbytes );

    dll->dllLevelNumbers[field].lnConstChars= nchars;

    /*  4  */
    for ( i= field; i <= dll->dllLevelNumberCount; i++ ) /* NOTE: <= */
	{ dll->dllLevelNumbers[i].lnOffsetBytes += nbytes- obytes;	}

    *pChanged= 1; return 0;
    }

int docListLevelSetNumber(	int *				pChanged,
				DocumentListLevel *		dll,
				int				level,
				int				field )
    {
    if  ( field < 0 || field > dll->dllLevelNumberCount )
	{ LLDEB(field,dll->dllLevelNumberCount); return -1;	}

    if  ( field < dll->dllLevelNumberCount		&&
    	  dll->dllLevelNumbers[field].lnLevel == level	)
	{ pChanged= 0; return 0;	}

    if  ( field < dll->dllLevelNumberCount )
	{ dll->dllLevelNumbers[field].lnLevel= level;	}
    else{
	if  ( docListLevelInsertNumber( dll, level, field ) )
	    { LDEB(field); return -1;	}
	}

    *pChanged= 1; return 0;
    }

/************************************************************************/
/*									*/
/*  Insert a reference to a level number in the format string of a list	*/
/*  level.								*/
/*									*/
/************************************************************************/

int docListLevelInsertNumber(	DocumentListLevel *		dll,
				int				level,
				int				field )
    {
    LevelNumber *	fresh;
    int			count= dll->dllLevelNumberCount+ 2; /* NOTE: +2 */
    int			i;

    /*  1  */
    if  ( field < 0 || field > dll->dllLevelNumberCount )
	{ LLDEB(field,dll->dllLevelNumberCount); return -1;	}

    /*  2  */
    fresh= realloc( dll->dllLevelNumbers, count* sizeof(LevelNumber) );
    if  ( ! fresh )
	{ LPDEB(count,fresh); return -1;	}

    dll->dllLevelNumbers= fresh;
    dll->dllLevelNumberCount++;

    for ( i= dll->dllLevelNumberCount; i > field; i-- )
	{ dll->dllLevelNumbers[i]= dll->dllLevelNumbers[i-1];	}

    /* dll->dllLevelNumbers[field].lnOffsetBytes:= unchanged */
    dll->dllLevelNumbers[field].lnConstChars= 0;
    dll->dllLevelNumbers[field].lnLevel= level;

    return 0;
    }

/************************************************************************/
/*									*/
/*  Delete a reference to a level number from the format string of a	*/
/*  list level.								*/
/*									*/
/************************************************************************/

int docListLevelDeleteNumber(	DocumentListLevel *		dll,
				int				field )
    {
    int			i;

    /*  1  */
    if  ( field < 0 || field >= dll->dllLevelNumberCount )
	{ LLDEB(field,dll->dllLevelNumberCount); return -1;	}

    for ( i= field; i < dll->dllLevelNumberCount; i++ )
	{ dll->dllLevelNumbers[i]= dll->dllLevelNumbers[i+1];	}

    dll->dllLevelNumberCount--;

    return 0;
    }

/************************************************************************/
/*									*/
/*  Format the value of a 'listtext' field.				*/
/*									*/
/*  I.E. the text of the bullet or paragraph number at the beginning	*/
/*	of a paragraph that gets one because it has a list reference.	*/
/*									*/
/************************************************************************/

int docListLevelFormatLevelNumber(	char *	target,
					int	maxsize,
					int	val,
					int	format )
    {
    int		step= 0;

    if  ( maxsize < 1 )
	{ LDEB(maxsize); return -1;	}

    switch( format )
	{
	case DOCpnDEC:
	    if  ( maxsize < 11 )
		{ LDEB(maxsize); return -1;	}
	    sprintf( target, "%d", val );
	    step= strlen( target );
	    break;

	case DOCpnUCRM:
	    if  ( utilRomanString( target, maxsize, val, 1 ) )
		{ LDEB(val); return -1;	}
	    step= strlen( target );
	    break;

	case DOCpnLCRM:
	    if  ( utilRomanString( target, maxsize, val, 0 ) )
		{ LDEB(val); return -1;	}
	    step= strlen( target );
	    break;

	case DOCpnUCLTR:
	    if  ( val == 0 )
		{ strcpy( target, "+" ); }
	    else{
		if  ( utilBase26String( target, maxsize, val, 1 ) )
		    { LDEB(val); return -1;	}
		}
	    step= strlen( target );
	    break;

	case DOCpnLCLTR:
	    if  ( val == 0 )
		{ strcpy( target, "+" ); }
	    else{
		if  ( utilBase26String( target, maxsize, val, 0 ) )
		    { LDEB(val); return -1;	}
		}
	    step= strlen( target );
	    break;

	case 22:
	    if  ( maxsize < 11 )
		{ LDEB(maxsize); return -1;	}
	    sprintf( target, "%02d", val );
	    step= strlen( target );
	    break;

	case 23:
	    step= uniPutUtf8( (unsigned char *)target, 0xb7 );
	    target[step]= '\0';
	    break;

	case 255:
	    break;

	default:
	    LLDEB(format,val);
	    if  ( maxsize < 13 )
		{ LDEB(maxsize); return -1;	}
	    sprintf( target, "(%d)", val );
	    step= strlen( target );
	    break;
	}

    return step;
    }

/************************************************************************/
/*									*/
/*  Format a paragraph number: A listtext field.			*/
/*									*/
/************************************************************************/

static int docListLevelFormatConst(
				int *				pConstLen,
				int *				pLevel,
				char *				to,
				int				maxbytes,
				const DocumentListLevel *	dll,
				int				field )
    {
    int			constOff;
    int			constLen;
    int			constChars;
    int			level;

    if  ( docListLevelGetField( &constOff, &constLen, &constChars, &level,
								dll, field ) )
	{ LDEB(field); return -1;	}
    if  ( constLen > maxbytes )
	{ LLDEB(constLen,maxbytes); return -1;	}

    if  ( constLen > 0 )
	{ memcpy( to, dll->dllFormatString+ constOff, constLen );	}

    to[constLen]= '\0';

    *pConstLen= constLen;
    *pLevel= level;

    return 0;
    }

int docListLevelFormatParagraphNumber(
				char *				to,
				int				maxbytes,
				int *				byteOffsets,
				int				maxoffsets,
				int				ilvl,
				const int *			numberPath,
				const int *			startPath,
				const int *			formatPath,
				const DocumentListLevel *	dll )
    {
    int			bytesDone= 0;
    int			field;

    int			constLen;
    int			level;

    if  ( ! dll->dllFormatString || ! dll->dllLevelNumbers )
	{ PPDEB(dll->dllFormatString,dll->dllLevelNumbers); return -1; }

    for ( field= 0; field < dll->dllLevelNumberCount; field++ )
	{
	int		varLen;

	level= -1;
	if  ( docListLevelFormatConst( &constLen, &level,
					to, maxbytes- bytesDone, dll, field ) )
	    { LDEB(field); return -1;	}

	if  ( 2* field+ 0 < maxoffsets )
	    { byteOffsets[2* field+ 0]= bytesDone;	}
	if  ( constLen > 0 )
	    { to += constLen; bytesDone += constLen;	}

	if  ( level > ilvl )
	    { LLDEB(ilvl,level); return -1;	}

	if  ( 2* field+ 1 < maxoffsets )
	    { byteOffsets[2* field+ 1]= bytesDone;	}

	varLen= docListLevelFormatLevelNumber( to, maxbytes- bytesDone,
				startPath[level]+ numberPath[level],
				formatPath[level] );
	if  ( varLen < 0 )
	    { LLDEB(level,varLen); return -1;	}

	to += varLen; bytesDone += varLen;
	}

    level= -1;
    if  ( docListLevelFormatConst( &constLen, &level,
					to, maxbytes- bytesDone, dll, field ) )
	{ LDEB(field); return -1;	}

    if  ( 2* field+ 0 < maxoffsets )
	{ byteOffsets[2* field+ 0]= bytesDone;	}
    if  ( constLen > 0 )
	{ to += constLen; bytesDone += constLen;	}

    if  ( 2* field+ 1 < maxoffsets )
	{ byteOffsets[2* field+ 1]= bytesDone;	}

    to[0]= '\0';
    return bytesDone;
    }

int docDefaultListLevel(	DocumentListLevel *		dllTo,
				int				level,
				int				deftab,
				int				levelTemplateID,
				const PropertyMask *		taSetMask,
				const TextAttribute *		taSet )
    {
    int			rval= 0;

    char *		formatString= (char *)0;
    LevelNumber *	levelNumbers= (LevelNumber *)0;

    int			i;

    docCleanDocumentListLevel( dllTo );
    docInitDocumentListLevel( dllTo );

    /*  1  */
    dllTo->dllTemplateID= levelTemplateID;

    formatString= malloc( level+ 1 );
    if  ( ! formatString )
	{ LPDEB(level,formatString); rval= -1; goto ready;	}

    levelNumbers= malloc( (level+ 2)* sizeof(LevelNumber) ); /* NOTE: +2 */
    if  ( ! levelNumbers )
	{ LPDEB(level,levelNumbers); rval= -1; goto ready;	}

    memset( formatString, '.', level );
    formatString[level]= '\0';

    for ( i= 0; i <= level; i++ )
	{
	levelNumbers[i].lnOffsetBytes= i;
	levelNumbers[i].lnLevel= i;
	levelNumbers[i].lnConstChars= ( i > 0 );
	}
    levelNumbers[i].lnOffsetBytes= i- 1;
    levelNumbers[i].lnLevel= -1;
    levelNumbers[i].lnConstChars= 0;

    dllTo->dllFormatString= formatString;
    formatString= (char *)0; /* steal */
    dllTo->dllLevelNumbers= levelNumbers;
    levelNumbers= (LevelNumber *)0; /* steal */
    dllTo->dllLevelNumberCount= level+ 1;

    dllTo->dllTextAttribute= *taSet;
    dllTo->dllTextAttributeMask= *taSetMask;

    dllTo->dllFirstIndentTwips= -deftab/ 2;
    dllTo->dllLeftIndentTwips= ( level+ 1 )* deftab;
    PROPmaskADD( &(dllTo->dllParaPropertyMask), PPpropFIRST_INDENT );
    PROPmaskADD( &(dllTo->dllParaPropertyMask), PPpropLEFT_INDENT );

  ready:

    if  ( formatString )
	{ free( formatString );		}
    if  ( levelNumbers )
	{ free( levelNumbers );	}

    return rval;
    }

int docListLevelToRtfStrings(	MemoryBuffer *			mbtext,
				MemoryBuffer *			mbnumbers,
				const DocumentListLevel *	dll )
    {
    unsigned char	uc;
    int			chars= 1;
    int			field;

    int			constOff;
    int			constLen;
    int			constChars;
    int			level;

    utilEmptyMemoryBuffer( mbtext );
    utilEmptyMemoryBuffer( mbnumbers );

    if  ( ! dll->dllFormatString || ! dll->dllLevelNumbers )
	{ PPDEB(dll->dllFormatString,dll->dllLevelNumbers); return 0; }

    uc= 0;
    if  ( utilAddToMemoryBuffer( mbtext, &uc, 1 ) )
	{ LDEB(1); return -1;	}

    for ( field= 0; field < dll->dllLevelNumberCount; field++ )
	{
	if  ( docListLevelGetField( &constOff, &constLen, &constChars, &level,
								dll, field ) )
	    { LDEB(field); return -1;	}

	if  ( utilAddToMemoryBuffer( mbtext,
		(unsigned char *)dll->dllFormatString+ constOff, constLen ) )
	    { LDEB(constLen); return -1;	}

	chars += constChars;

	uc= chars;
	if  ( utilAddToMemoryBuffer( mbnumbers, &uc, 1 ) )
	    { LDEB(1); return -1;	}

	uc= level;
	if  ( utilAddToMemoryBuffer( mbtext, &uc, 1 ) )
	    { LDEB(1); return -1;	}

	chars++;
	}

    if  ( docListLevelGetField( &constOff, &constLen, &constChars, &level,
								dll, field ) )
	{ LDEB(field); return -1;	}

    if  ( utilAddToMemoryBuffer( mbtext,
		(unsigned char *)dll->dllFormatString+ constOff, constLen ) )
	    { LDEB(constLen); return -1;	}

    chars += constChars;
    mbtext->mbBytes[0]= chars- 1;

    return 0;
    }

static int docListLevelConstFromRtfStrings(
					const unsigned char *	from,
					int			constChars )
    {
    int		c;
    int		constLen= 0;

    for ( c= 0; c < constChars; c++ )
	{
	unsigned short	symbol;
	int		step;

	step= uniGetUtf8( &symbol, from );
	if  ( step < 1 )
	    { CLDEB(*from,step); return -1;	}

	constLen += step; from += step;
	}

    return constLen;
    }

int docListLevelFromRtfStrings(	DocumentListLevel *		dll,
				const MemoryBuffer *		mbtext,
				const MemoryBuffer *		mbnumbers )
    {
    int			rval= 0;

    char *		formatString= (char *)0;
    LevelNumber *	levelNumbers= (LevelNumber *)0;

    const unsigned char *	fbytes;
    int			fsize= 0;
    const unsigned char *	lbytes;
    int			lsize= 0;

    int			field;
    int			offsetChars= 1; /* in rtf */
    int			constChars;
    int			constOff= 0;
    int			constLen= 0;

    const unsigned char * from;

    fbytes= utilMemoryBufferGetBytes( &fsize, mbtext );
    lbytes= utilMemoryBufferGetBytes( &lsize, mbnumbers );

    formatString= malloc( fsize );
    if  ( ! formatString )
	{ LPDEB(fsize,formatString); rval= -1; goto ready;	}
    formatString[0]= '\0';

    levelNumbers= malloc( (lsize+ 1)*sizeof(LevelNumber) );
    if  ( ! levelNumbers )
	{ LPDEB(lsize,levelNumbers); rval= -1; goto ready;	}

    from= fbytes+ 1;

    for ( field= 0; field < lsize; field++ )
	{
	int	fieldOffsetChars= lbytes[field]; /* in rtf */

	constChars= fieldOffsetChars- offsetChars;
	constLen= docListLevelConstFromRtfStrings( from, constChars );

	if  ( constLen < 0 )
	    { LLDEB(field,constLen); rval= -1; goto ready;	}

	memcpy( formatString+ constOff, from, constLen );
	from += constLen;

	levelNumbers[field].lnOffsetBytes= constOff+ constLen;
	levelNumbers[field].lnConstChars= constChars;
	levelNumbers[field].lnLevel= *(from++);

	offsetChars= fieldOffsetChars+ 1;
	constOff += constLen;
	}

    constChars= fbytes[0]+ 1- offsetChars;
    constLen= docListLevelConstFromRtfStrings( from, constChars );
    if  ( constLen < 0 )
	{ LDEB(constLen); rval= -1; goto ready;	}

    memcpy( formatString+ constOff, from, constLen );

    levelNumbers[field].lnOffsetBytes= constOff+ constLen;
    levelNumbers[field].lnConstChars= constChars;
    levelNumbers[field].lnLevel= -1;

    constOff += constLen;
    formatString[constOff]= '\0';

    dll->dllFormatString= formatString;
    formatString= (char *)0; /* steal */
    dll->dllLevelNumbers= levelNumbers;
    levelNumbers= (LevelNumber *)0; /* steal */
    dll->dllLevelNumberCount= lsize;

#   if 0
    SDEB(dll->dllFormatString);
    for ( field= 0; field <= dll->dllLevelNumberCount; field++ )
	{
	appDebug( "%d: ob= %2d, cg= %2d, lev= %2d\n", field,
			dll->dllLevelNumbers[field].lnOffsetBytes,
			dll->dllLevelNumbers[field].lnConstChars,
			dll->dllLevelNumbers[field].lnLevel );
	}
    docListDocumentListLevel(-1,dll);
#   endif

  ready:

    if  ( formatString )
	{ free( formatString );	}
    if  ( levelNumbers )
	{ free( levelNumbers );	}

    return rval;
    }

