/************************************************************************/
/*									*/
/*  Replace the selection in a document with another document.		*/
/*  ( Used with 'paste', 'insert file', 'undo', 'redo'. )		*/
/*									*/
/************************************************************************/

#   include	"docBufConfig.h"

#   include	<stddef.h>
#   include	<stdlib.h>
#   include	<stdio.h>
#   include	<ctype.h>

#   include	"docEdit.h"

#   include	<appDebugon.h>

#   define	SHOW_SELECTION_RANGE	0

/************************************************************************/
/*									*/
/*  Replace the selection in a document with another document.		*/
/*  ( Used with 'paste', 'insert file', 'TOC' )				*/
/*									*/
/*  B)  The first particule of the line was split, probably, part fits	*/
/*	on the previous line. Reformat from the previous particule.	*/
/*	If paragraphs were merged, redraw the whole neighbourhood.	*/
/*									*/
/*  1)  Replace the selection of the target with the text of those	*/
/*	particules at the beginning of the source that have the same	*/
/*	attributes.							*/
/*  2)  Insert the rest of the first paragraph of the source into the	*/
/*	target.								*/
/*  4)  If the insertion consists of more than one paragraph, split the	*/
/*	target paragraph.						*/
/*  5)  Insert the last particule of the insertion as text.		*/
/*									*/
/*  z)  Copy all paragraphs between the first and last (exclusive) of	*/
/*	the source to the target.					*/
/*									*/
/************************************************************************/

/************************************************************************/
/*									*/
/*  Determine the split level of the inserted document.			*/
/*									*/
/************************************************************************/

static int docInsGetSplitLevel(		const BufferDocument *	bdFrom )
    {
    const BufferItem *	biSplit= bdFrom->bdBody.eiRoot;

    while( biSplit->biChildCount == 1 )
	{ biSplit= biSplit->biChildren[0];	}

    if  ( biSplit->biChildCount == 0 )
	{ LLDEB(biSplit->biChildCount,biSplit->biLevel); return -1; }

    return  biSplit->biChildren[0]->biLevel;
    }

/************************************************************************/
/*									*/
/*  Include a single paragraph in the current document. It will		*/
/*  replace the  current selection.					*/
/*									*/
/************************************************************************/

static int docInsertSingleParagraph(
				DocumentCopyJob *		dcjInsert,
				int				paraNrBegin )
    {
    EditOperation *		eo= dcjInsert->dcjEditOperation;

    int				rval= 0;
    DocumentCopyJob		dcjTail;

    DocumentPosition		dpFrom;
    const int			partFrom= 0;

    docInitDocumentCopyJob( &dcjTail );

    if  ( docFirstPosition( &dpFrom, dcjInsert->dcjBdFrom->bdBody.eiRoot ) )
	{ LDEB(1); rval= -1; goto ready;	}

    /*  1  */
    if  ( eo->eoHeadDp.dpBi != eo->eoTailDp.dpBi )
	{
	EditPosition		epSave;
	int			headPartCount;

	DocumentPosition	tailDp= eo->eoTailDp;
	int			tailParticule= eo->eoTailParticule;

	if  ( eo->eoHeadDp.dpBi->biParaTableNesting > 0		&&
	      eo->eoTailDp.dpBi->biParaTableNesting == 0	)
	    { docFlattenRow( eo->eoHeadDp.dpBi );	}

	/*  2  */
	headPartCount= eo->eoHeadDp.dpBi->biParaParticuleCount;
	if  ( docParaDeleteText( eo,
			&tailDp, &tailParticule,
			paraNrBegin, &(eo->eoHeadDp), eo->eoHeadParticule,
			docParaStrlen( eo->eoHeadDp.dpBi ), headPartCount ) )
	    { LDEB(1); rval= -1; goto ready;	}

	if  ( docParaInsertTail( dcjInsert, &tailDp, &tailParticule,
			    paraNrBegin, &tailDp, tailParticule,
			    &dpFrom, partFrom ) )
	    { LDEB(partFrom); rval= -1; goto ready;	}

	epSave= eo->eoSelectedRange.erTail;
	epSave.epParaNr= paraNrBegin;
	epSave.epStroff= docParaStrlen( eo->eoHeadDp.dpBi );

	/*  3  */
	if  ( eo->eoTailParticule < eo->eoTailDp.dpBi->biParaParticuleCount )
	    {
	    DocumentPosition	afterDp;
	    int			afterParticule;

	    if  ( docSet1DocumentCopyJob( &dcjTail, eo ) )
		{ LDEB(1); rval= -1; goto ready;	}

	    if  ( docParaInsertTail( &dcjTail,
				&afterDp, &afterParticule,
				paraNrBegin, &tailDp, tailParticule,
				&(eo->eoTailDp), eo->eoTailParticule ) )
		{ LDEB(eo->eoTailParticule); rval= -1; goto ready;	}
	    }

	eo->eoTailDp= tailDp;
	eo->eoTailParticule= tailParticule;

	eo->eoSelectedRange.erTail= epSave;

	/*  4  */
	if  ( docRemoveSelectionTail( eo ) )
	    { LDEB(1); rval= -1; goto ready;	}
	}
    else{
	/*  5  */
	if  ( docParaDeleteText( eo,
		&(eo->eoTailDp), &(eo->eoTailParticule),
		paraNrBegin, &(eo->eoHeadDp), eo->eoHeadParticule,
		eo->eoTailDp.dpStroff, eo->eoTailParticule ) )
	    { LDEB(1); rval= -1; goto ready;	}

	if  ( docParaInsertTail( dcjInsert,
			&(eo->eoTailDp), &(eo->eoTailParticule),
			paraNrBegin, &(eo->eoTailDp), eo->eoTailParticule,
			&dpFrom, partFrom ) )
	    { LDEB(partFrom); rval= -1; goto ready;	}
	}

    docEditFinishStep( eo );

  ready:

    docCleanDocumentCopyJob( &dcjTail );

    return rval;
    }

/************************************************************************/
/*									*/
/*  Include a multi paragraph document in the current document. It will	*/
/*  replace the  current selection.					*/
/*									*/
/*  1)  Delete the selection from the target document.			*/
/*  2)  Split the current paragraph upto the desired level.		*/
/*  3)  Append the first paragraph of the source to the current		*/
/*	paragraph.							*/
/*  4)  Insert the source document tree.				*/
/*	Append the items to the parents at the same level in the	*/
/*	target.								*/
/*									*/
/************************************************************************/

static int docInsertMultipleParagraphs(
				DocumentCopyJob *		dcjInsert,
				int				paraNrBegin )
    {
    int				rval= 0;

    EditOperation *		eo= dcjInsert->dcjEditOperation;
    const BufferDocument *	bdFrom= dcjInsert->dcjBdFrom;

    DocumentCopyJob		dcjTail;

    DocumentPosition		dpFrom;
    DocumentPosition		dpLast;
    const int			partFrom= 0;
    const BufferItem *		biFrom;

    BufferItem *		newParaBi;

    BufferItem *		biTo;
    int				n;

    DocumentPosition		tailDp;
    int				tailParticule;

    int				splitLevel= docInsGetSplitLevel( bdFrom );

    int				sectShift;
    int				lastInserted;

    int				mergeHead= 1;
    int				mergeTail= 1;
    int				mergeTailItems= 1;

    docInitDocumentCopyJob( &dcjTail );

    if  ( splitLevel < 0 )
	{ LDEB(splitLevel); rval= -1; goto ready;	}

    if  ( docFirstPosition( &dpFrom, bdFrom->bdBody.eiRoot ) )
	{ LDEB(1); rval= -1; goto ready;	}
    if  ( docLastPosition( &dpLast, bdFrom->bdBody.eiRoot ) )
	{ LDEB(1); rval= -1; goto ready;	}

    if  ( dpFrom.dpBi->biParaTableNesting > 0 )
	{
	mergeHead= 0;
	if  ( splitLevel > DOClevROW )
	    { splitLevel=  DOClevROW;	}
	}
    if  ( dpLast.dpBi->biParaTableNesting > 0 )
	{
	mergeTail= 0; mergeTailItems= mergeTail;
	if  ( splitLevel > DOClevROW )
	    { splitLevel=  DOClevROW;	}
	}

    if  ( eo->eoHeadDp.dpBi->biParaTableNesting > 0 && splitLevel < DOClevPARA )
	{ rval= -1; goto ready; }
    if  ( eo->eoTailDp.dpBi->biParaTableNesting > 0 && splitLevel < DOClevPARA )
	{ rval= -1; goto ready; }

    /*  1  */
    if  ( docDeleteSelection( eo ) )
	{ LDEB(1); rval= -1; goto ready;	}

    /*  2  */
    if  ( docSplitParaItem( &newParaBi, eo, splitLevel ) )
	{ LDEB(splitLevel); return -1;	}

    /*  3  */
    docLastPosition( &tailDp, eo->eoHeadDp.dpBi );
    tailParticule= eo->eoHeadDp.dpBi->biParaParticuleCount;

    if  ( mergeHead )
	{
	if  ( docParaStrlen( tailDp.dpBi ) == 0 )
	    {
	    PropertyMask	ppUpdMask;
	    PropertyMask	ppDoneMask;

	    utilPropMaskClear( &ppUpdMask );
	    utilPropMaskFill( &ppUpdMask, PPprop_COUNT );
	    PROPmaskUNSET( &ppUpdMask, PPpropTABLE_NESTING );

	    if  ( docUpdParaProperties( &ppDoneMask,
			&(tailDp.dpBi->biParaProperties),
			&ppUpdMask, &(dpFrom.dpBi->biParaProperties),
			&(dcjInsert->dcjAttributeMap) ) )
		{ LDEB(1); rval= -1; goto ready; }
	    }

	if  ( docParaInsertTail( dcjInsert, &tailDp, &tailParticule,
		    paraNrBegin, &tailDp, tailParticule, &dpFrom, partFrom ) )
	    { LDEB(partFrom); rval= -1; goto ready;	}
	}

    /*  4  */
    biFrom= dpFrom.dpBi;
    biTo= tailDp.dpBi;

    n= biTo->biNumberInParent;
    biFrom= biFrom->biParent;
    biTo= biTo->biParent;

    for (;;)
	{
	int	child;
	int	from= 1;

	if  ( biTo->biChildren[0]->biLevel == DOClevROW )
	    { mergeHead= 1; mergeTailItems= 1; from= 0;	}

	if  ( mergeHead )
	    {
	    for ( child= from; child < biFrom->biChildCount; child++ )
		{
		BufferItem *	insBi;

		n++;
		insBi= docCopyItem( dcjInsert, biTo,
						n, biFrom->biChildren[child] );
		if  ( ! insBi )
		    { XDEB(insBi); rval= -1; goto ready;	}
		}
	    }

	if  ( biTo->biChildren[0]->biLevel == splitLevel )
	    { break;	}

	n= biTo->biNumberInParent;
	biFrom= biFrom->biParent;
	biTo= biTo->biParent;
	}

    if  ( docLastPosition( &tailDp, biTo->biChildren[n] ) )
	{ LDEB(1); rval= -1; goto ready;	}

    lastInserted= docNumberOfParagraph( tailDp.dpBi );
    sectShift= 0;

    if  ( biTo->biLevel == DOClevROW )
	{ mergeTailItems= mergeTail;	}
    while( biTo->biChildren[n]->biLevel != DOClevPARA )
	{
	BufferItem *	biParent= biTo;
	int		nextN;

	biTo= biParent->biChildren[n];
	nextN= biTo->biChildCount- 1;

	if  ( biTo->biLevel == DOClevROW )
	    { mergeTailItems= mergeTail;	}

	if  ( n+ 1 < biParent->biChildCount && mergeTailItems )
	    {
	    int			sectionsDeleted;
	    BufferItem *	lastParent;

	    lastParent= biParent->biChildren[n+ 1];

	    if  ( docMergeGroupItems( biTo, lastParent ) )
		{ LDEB(1); rval= -1; goto ready;	}

	    sectionsDeleted= 0;
	    docDeleteEmptyParents( eo, &sectionsDeleted, lastParent );
	    sectShift -= sectionsDeleted;
	    }

	n= nextN;
	}

    if  ( sectShift != 0 )
	{
	const int	paraNr= lastInserted+ 1;
	const int	stroffFrom= 0;
	const int	stroffShift= 0;
	const int	paraShift= 0;

	eo->eoSectionsDeleted -= sectShift;

	docEditShiftReferences( eo, &(eo->eoSelectionScope),
				    paraNr, stroffFrom,
				    sectShift, paraShift, stroffShift );
	}

    if  ( mergeTail && n+ 1 < biTo->biChildCount )
	{
	const int		partTailFrom= 0;
	DocumentPosition	dpTailFrom;

	int			partTailTo= 0;
	DocumentPosition	dpTailTo;

	int			paraNrTail;

	if  ( docLastPosition( &(eo->eoTailDp), biTo->biChildren[n] ) )
	    { LDEB(1); rval= -1; goto ready;	}
	eo->eoTailParticule= eo->eoTailDp.dpBi->biParaParticuleCount;
	paraNrTail= docNumberOfParagraph( biTo->biChildren[n] );

	if  ( docFirstPosition( &dpTailFrom, biTo->biChildren[n+1] ) )
	    { LDEB(1); rval= -1; goto ready;	}

	if  ( docLastPosition( &(eo->eoLastDp), biTo->biChildren[n+1] ) )
	    { LDEB(1); rval= -1; goto ready;	}

	if  ( docSet1DocumentCopyJob( &dcjTail, eo ) )
	    { LDEB(1); rval= -1; goto ready;	}

	if  ( docParaInsertTail( &dcjTail, &dpTailTo, &partTailTo,
			    paraNrTail, &(eo->eoTailDp), eo->eoTailParticule,
			    &dpTailFrom, partTailFrom ) )
	    { LDEB(partTailFrom); rval= -1; goto ready;	}

	if  ( docRemoveSelectionTail( eo ) )
	    { LDEB(1); rval= -1; goto ready;	}
	}

  ready:

    docCleanDocumentCopyJob( &dcjTail );

    return rval;
    }

/************************************************************************/
/*									*/
/*  Include a document in the current document. It will replace the	*/
/*  current selection.							*/
/*									*/
/*  1)  Find out what to insert.					*/
/*  2)  Insert text belonging to a single paragraph.			*/
/*  3)  Insert text belonging to a multiple paragraphs.			*/
/*									*/
/************************************************************************/

int docIncludeDocument(		DocumentCopyJob *		dcjInsert )
    {
    EditOperation *		eo= dcjInsert->dcjEditOperation;
    BufferDocument *		bdFrom= dcjInsert->dcjBdFrom;

    DocumentPosition		dpHead;
    DocumentPosition		dpTail;

    int				paraNrBegin;

    paraNrBegin= docNumberOfParagraph( eo->eoHeadDp.dpBi );

    if  ( docEditDeleteReplacedFields( eo ) )
	{ LDEB(1);	}

#   if SHOW_SELECTION_RANGE
    appDebug( "INCLUDE.. %3d(%3d) .. %3d(%3d)\n",
			    eo->eoSelectedRange.erHead.epParaNr,
			    eo->eoSelectedRange.erHead.epStroff,
			    eo->eoSelectedRange.erTail.epParaNr,
			    eo->eoSelectedRange.erTail.epStroff );
#   endif

    /*  1  */
    if  ( docFirstPosition( &dpHead, bdFrom->bdBody.eiRoot ) )
	{ LDEB(1); return -1;	}
    if  ( docLastPosition(  &dpTail, bdFrom->bdBody.eiRoot ) )
	{ LDEB(1); return -1;	}

    /*  2,3  */
    if  ( dpHead.dpBi == dpTail.dpBi )
	{
	if  ( docInsertSingleParagraph( dcjInsert, paraNrBegin ) )
	    { LDEB(1); return -1;	}
	}
    else{
	if  ( docInsertMultipleParagraphs( dcjInsert, paraNrBegin ) )
	    { LDEB(1); return -1;	}
	}

    eo->eoNotesAdded += dcjInsert->dcjNotesCopied;
    eo->eoBulletsChanged += dcjInsert->dcjBulletsCopied;

    return 0;
    }

