/************************************************************************/
/*									*/
/*  Substitute values into a document based on a font.			*/
/*									*/
/************************************************************************/

#   include	"docBufConfig.h"

#   include	<stdlib.h>
#   include	<string.h>
#   include	<stdio.h>
#   include	<ctype.h>

#   include	<appDebugon.h>

#   include	<sioStdio.h>
#   include	<appSystem.h>
#   include	<utilMatchFont.h>
#   include	<ucdBlock.h>
#   include	<psGlyphs.h>
#   include	<uniUtf8.h>
#   include	<ucd.h>

#   include	"docBuf.h"
#   include	"docEdit.h"
#   include	"docDocumentCopyJob.h"
#   include	"docRecalculateFields.h"
#   include	"docEvalField.h"
#   include	"docRtfReadWrite.h"

/************************************************************************/

typedef struct FontReport
    {
    const AfmFontInfo *	frAfi;
    BufferDocument *	frBd;

    int			frBlockIndex;
    unsigned int	frPageFirst;
    unsigned int	frPageLast;
    int			frCurrentSection;
    } FontReport;

/************************************************************************/
/*									*/
/*  Evaluate a single font attribute.					*/
/*									*/
/************************************************************************/

static int fontReportProvideData(	int *		pCalculated,
					MemoryBuffer *	mbResult,
					const char *	value )
    {
    utilAddToMemoryBuffer( mbResult, (unsigned char *)value, strlen( value ) );
    *pCalculated= 1;
    return 0;
    }

static int fontReportProvideOptional(	int *		pCalculated,
					MemoryBuffer *	mbResult,
					const char *	value )
    {
    if  ( value )
	{
	return fontReportProvideData( pCalculated, mbResult, value );
	}
    else{
	return fontReportProvideData( pCalculated, mbResult, "(None)" );
	}

    return 0;
    }

static int docFontReportGetUnicode(	const FontReport *	fr,
					const char *		fieldName,
					int			off )
    {
    char *		past= (char *)fieldName+ off;
    long		l= strtol( fieldName+ off, &past, 10 );

    if  ( past == fieldName+ off || *past )
	{ SDEB(fieldName); return -1;	}

    return fr->frPageFirst+ l;
    }

static int fontReportFieldDataProvider(	int *		pCalculated,
					MemoryBuffer *	mbResult,
					const char *	fieldName,
					void *		through )
    {
    FontReport *	fr= (FontReport *)through;
    const AfmFontInfo *	afi= fr->frAfi;

    static char		scratch[20];

    int			past= 65536;
    const UcdBlock *	ub= UCD_Blocks+ fr->frBlockIndex;

    if  ( past > ub->ubLast+ 1 )
	{ past=  ub->ubLast+ 1;	}

    if  ( ! strcmp( fieldName, "FullName" ) )
	{
	return fontReportProvideOptional( pCalculated, mbResult,
						    afi->afiFullName );
	}

    if  ( ! strcmp( fieldName, "FontName" ) )
	{
	return fontReportProvideOptional( pCalculated, mbResult,
						    afi->afiFontName );
	}

    if  ( ! strcmp( fieldName, "FamilyName" ) )
	{
	return fontReportProvideOptional( pCalculated, mbResult,
						    afi->afiFamilyName );
	}

    if  ( ! strcmp( fieldName, "Width" ) )
	{
	return fontReportProvideOptional( pCalculated, mbResult,
						    afi->afiWidthStr );
	}

    if  ( ! strcmp( fieldName, "Weight" ) )
	{
	return fontReportProvideOptional( pCalculated, mbResult,
						    afi->afiWeightStr );
	}

    if  ( ! strcmp( fieldName, "Slant" ) )
	{
	return fontReportProvideData( pCalculated, mbResult,
						    afi->afiItalicAngle < -1?
						    "Italic":"Roman" );
	}

    if  ( ! strcmp( fieldName, "IsFixed" ) )
	{
	return fontReportProvideData( pCalculated, mbResult,
						    afi->afiIsFixedPitch?
						    "Fixed":"Proportional" );
	}

    if  ( ! strcmp( fieldName, "PageFirst" ) )
	{
	sprintf( scratch, "uni%04X", fr->frPageFirst );

	return fontReportProvideOptional( pCalculated, mbResult,
						    scratch );
	}

    if  ( ! strcmp( fieldName, "PageLast" ) )
	{
	sprintf( scratch, "uni%04X", fr->frPageLast );

	return fontReportProvideOptional( pCalculated, mbResult,
						    scratch );
	}

    if  ( ! strcmp( fieldName, "BlockName" ) )
	{
	return fontReportProvideOptional( pCalculated, mbResult,
				UCD_Blocks[fr->frBlockIndex].ubDescription );
	}

    if  ( ! strcmp( fieldName, "AfmFile" ) )
	{
	return fontReportProvideOptional( pCalculated, mbResult,
						    afi->afiAfmFileName );
	}

    if  ( ! strcmp( fieldName, "FontFile" ) )
	{
	return fontReportProvideOptional( pCalculated, mbResult,
						    afi->afiFontFileName );
	}

    if  ( ! strncmp( fieldName, "Nr", 2 ) && isdigit( fieldName[2] ) )
	{
	int		uni= docFontReportGetUnicode( fr, fieldName, 2 );

	if  ( uni < past && ucdIsUtf16( uni ) )
	    {
	    if  ( uni <= 0xff )
		{ sprintf( scratch, "0x%02x", (unsigned)uni );	}
	    else{ sprintf( scratch, "0x%04x", (unsigned)uni );	}

	    return fontReportProvideData( pCalculated, mbResult, scratch );
	    }

	return fontReportProvideOptional( pCalculated, mbResult, " " );
	}

    if  ( ! strncmp( fieldName, "N", 1 ) && isdigit( fieldName[1] ) )
	{
	int		uni= docFontReportGetUnicode( fr, fieldName, 1 );

	if  ( uni > 0							    &&
	      uni < past						    &&
	      utilIndexMappingGet( &(afi->afiUnicodeToGlyphMapping), uni ) >= 0 )
	    {
	    const char *	N= psUnicodeToGlyphName( uni );

	    return fontReportProvideOptional( pCalculated, mbResult, N );
	    }

	return fontReportProvideOptional( pCalculated, mbResult, " " );
	}

    if  ( ! strncmp( fieldName, "G", 1 ) && isdigit( fieldName[1] ) )
	{
	int		uni= docFontReportGetUnicode( fr, fieldName, 1 );

	if  ( uni > 0							    &&
	      uni < past						    &&
	      utilIndexMappingGet( &(afi->afiUnicodeToGlyphMapping), uni ) >= 0 )
	    {
	    int		step= uniPutUtf8( (unsigned char *)scratch, uni );

	    if  ( step < 1 )
		{ LDEB(step);	}
	    else{
		scratch[step]= '\0';

		return fontReportProvideData( pCalculated, mbResult, scratch );
		}
	    }

	return fontReportProvideData( pCalculated, mbResult, " " );
	}

    SDEB(fieldName);
    *pCalculated= 0;
    return 0;
    }

/************************************************************************/

static int docSubstituteFontPage(	int *			pUsed,
					RecalculateFields *	rf,
					const FontReport *	fr )
    {
    int			uni;
    int			rval= 0;
    int			used= 0;

    DocumentCopyJob	dcj;
    EditOperation	eo;

    const AfmFontInfo *	afi= fr->frAfi;
    BufferDocument *	bd= fr->frBd;

    docInitDocumentCopyJob( &dcj );
    docInitEditOperation( &eo );

    for ( uni= fr->frPageFirst; uni <= fr->frPageLast; uni++ )
	{
	if  ( utilIndexMappingGet( &(afi->afiUnicodeToGlyphMapping), uni ) >= 0 )
	    { used= 1; break;	}
	}

    if  ( used )
	{
	BufferItem *	bodySectBi= bd->bdBody.eiRoot->biChildren[0];

	if  ( fr->frCurrentSection < bd->bdBody.eiRoot->biChildCount )
	    {
	    const int	page= 0;

	    bodySectBi= bd->bdBody.eiRoot->biChildren[fr->frCurrentSection];

	    if  ( docRecalculateTextLevelFieldsInExternalItem(
				    rf, &(bd->bdBody), bodySectBi, page ) )
		{ LDEB(1); return -1;	}
	    }
	else{
	    DocumentPosition		dp;
	    DocumentSelection		ds;

	    if  ( docLastPosition( &dp, bd->bdBody.eiRoot ) )
		{ LDEB(1); rval= -1; goto ready;	}
	    docSetRangeSelection( &ds, &dp, &dp, 1, -1, -1 );

	    if  ( docStartEditOperation( &eo, &ds, bd ) )
		{ LDEB(1); rval= -1; goto ready;	}

	    if  ( docSet1DocumentCopyJob( &dcj, &eo ) )
		{ LDEB(1); rval= -1; goto ready;	}
	    dcj.dcjCopyFields= 1;

	    bodySectBi= docCopyItem( &dcj, bd->bdBody.eiRoot,
					fr->frCurrentSection, bodySectBi );
	    if  ( ! bodySectBi )
		{ XDEB(bodySectBi); rval= -1; goto ready;	}

	    if  ( docRecalculateTextLevelFields( rf, bodySectBi ) )
		{ LDEB(1); rval= -1; goto ready;	}
	    }
	}

    *pUsed= used;

  ready:

    docCleanDocumentCopyJob( &dcj );

    return rval;
    }

/************************************************************************/
/*									*/
/*  Substitute values into a document based on a font.			*/
/*									*/
/************************************************************************/

static int docSubstituteFontProperties(	const AfmFontInfo *	afi,
					BufferDocument *	bd )
    {
    FontReport		fr;
    RecalculateFields	rf;
    const UcdBlock *	ub;

    fr.frAfi= afi;
    fr.frBd= bd;
    fr.frBlockIndex= 0;
    fr.frPageFirst= 0;
    fr.frPageLast= 0xff;
    fr.frCurrentSection= 0;

    docInitRecalculateFields( &rf );
    rf.rfBd= bd;
    rf.rfUpdateFlags= FIELDdoDOC_INFO;
    rf.rfFieldDataProvider= fontReportFieldDataProvider;
    rf.rfMergeThrough= &fr;

    ub= UCD_Blocks;
    while( fr.frBlockIndex < UCD_BlockCount )
	{
	int		used= 0;

	/* Stop at the specials */
	if  ( ub->ubFirst >= 0xfff0 )
	    { break;	}

	fr.frPageFirst= ub->ubFirst;

	while( fr.frPageFirst <= ub->ubLast )
	    {
	    fr.frPageLast= fr.frPageFirst+ 255;

	    if  ( fr.frPageLast > ub->ubLast )
		{ fr.frPageLast=  ub->ubLast;	}

	    if  ( docSubstituteFontPage( &used, &rf, &fr ) )
		{ LDEB(1); return -1;	}
	    if  ( used )
		{ fr.frCurrentSection++;	}

	    fr.frPageFirst= fr.frPageLast+ 1;
	    }

	fr.frBlockIndex++;
	ub++;
	}

    return 0;
    }

/************************************************************************/
/*									*/
/*  Substitute values into a document based on a font.			*/
/*									*/
/************************************************************************/

static int docMakeFontExample(	const AfmFontInfo *		afi,
				const PostScriptFontList *	psfl,
				DocumentFontList *		dfl,
				const char *			template,
				const char *			outputDir )
    {
    int				rval= 0;

    char			relative[200+1];
    char			absolute[200+1];

    SimpleInputStream *		sis= (SimpleInputStream *)0;
    SimpleOutputStream *	sos= (SimpleOutputStream *)0;
    BufferDocument *		bd= (BufferDocument *)0;

    const int			relativeIsFile= 0;
    char *			s;

    if  ( strlen( afi->afiFontName )+ 5 > sizeof(relative)- 1 )
	{ SDEB(afi->afiFontName); rval= -1; goto ready;	}
    sprintf( relative, "%s.rtf", afi->afiFontName );
    s= relative;
    while( *s )
	{
	if  ( isspace( *s ) )
	    { *s= '-';	}
	s++;
	}
    if  ( appAbsoluteName( absolute, sizeof(absolute)- 1,
			    relative, relativeIsFile, outputDir ) < 0 )
	{ SDEB(relative); rval= -1; goto ready;	}

    sis= sioInStdioOpen( template );
    if  ( ! sis )
	{ SXDEB(template,sis); rval= -1; goto ready;	}

    bd= docRtfReadFile( sis, psfl );
    if  ( ! bd )
	{ SDEB(template); rval= -1; goto ready;	}

    if  ( docSubstituteFontProperties( afi, bd ) )
	{ SDEB(afi->afiFontName); rval= -1; goto ready;	}

    /*  HACK internals of document font list. */
    {
    int				n;
    DocumentFont *		dfTo;

    n= docGetFontByName( &(bd->bdProperties.dpFontList), "Times" );
    if  ( n < 0 )
	{ SLDEB("Times",n); rval= -1; goto ready;	}

    dfTo= docFontListGetFontByNumber( &(bd->bdProperties.dpFontList), n );
    if  ( ! dfTo )
	{ LXDEB(n,dfTo); goto ready;	}

    if  ( docFontSetFamilyName( dfTo, afi->afiFamilyName ) )
	{ LDEB(1); rval= -1; goto ready;	}
    dfTo->dfUsed= 1;
    }

    sos= sioOutStdioOpen( absolute );
    if  ( ! sos )
	{ SXDEB(absolute,sos); rval= -1; goto ready;	}

    if  ( docRtfSaveDocument( sos, bd,
			(const DocumentSelection *)0, RTFflagNO_MERGEFIELDS ) )
	{ SDEB(absolute); rval= -1; goto ready;	}

  ready:

    if  ( sos )
	{ sioOutClose( sos );		}
    if  ( bd )
	{ docFreeDocument( bd );	}
    if  ( sis )
	{ sioInClose( sis );		}

    return rval;
    }

int docFontsDocuments(		const PostScriptFontList *	psfl,
				const char *			templDir,
				const char *			outputDir )
    {
    int				rval= 0;
    int				fam;

    DocumentFontList		dfl;

    docInitFontList( &dfl );

    if  ( appTestDirectory( outputDir )		&&
	  appMakeDirectories( outputDir )	)
	{ SDEB(outputDir); rval= -1; goto ready;	}

    if  ( utilAddPsFontsToDocList( &dfl, psfl ) )
	{ LDEB(1); rval= -1; goto ready;	}

    for ( fam= 0; fam < psfl->psflFamilyCount; fam++ )
	{
	PsFontFamily *	aff= psfl->psflFamilies[fam];
	int		face;

	for ( face= 0; face < aff->affFaceCount; face++ )
	    {
	    const char *	relative;
	    AfmFontInfo *	afi= aff->affFaces[face];
	    char		absolute[200+1];
	    const int		relativeIsFile= 0;

	    if  ( afi->afiMetricsDeferred )
		{ continue;	}

	    if  ( afi->afiWeightInt >= FONTweightDEMIBOLD )
		{
		if  ( afi->afiItalicAngle < -1.0 )
		    { relative= "font-bi.rtf";	}
		else{ relative= "font-b.rtf";	}
		}
	    else{
		if  ( afi->afiItalicAngle < -1.0 )
		    { relative= "font-i.rtf";	}
		else{ relative= "font.rtf";		}
		}

	    if  ( appAbsoluteName( absolute, sizeof(absolute)- 1,
				    relative, relativeIsFile, templDir ) < 0 )
		{ SDEB(relative); rval= -1; goto ready;	}

	    if  ( afi->afiMetricsDeferred		&&
		  psGetDeferredMetrics( afi )		)
		{ SDEB(afi->afiAfmFileName); rval= -1; goto ready;	}

	    if  ( docMakeFontExample( afi, psfl, &dfl, absolute, outputDir ) )
		{ SDEB(afi->afiFontName); rval= -1; goto ready;	}
	    }
	}

  ready:

    docCleanFontList( &dfl );

    return rval;
    }

