/************************************************************************/
/*									*/
/*  Buffer administration routines.					*/
/*									*/
/************************************************************************/

#   include	"docBufConfig.h"

#   include	<stdlib.h>
#   include	<string.h>
#   include	<stdio.h>
#   include	<limits.h>

#   include	<appDebugon.h>

#   include	<appUnit.h>
#   include	"docDocumentField.h"

/************************************************************************/
/*									*/
/*  Manage fields.							*/
/*									*/
/************************************************************************/

#   define	FIC_COUNT	15

int docFieldHasMergeformat(		const unsigned char *		bytes,
					FieldInstructionsComponent *	fic,
					int				n,
					int				comp )
    {
    if  ( comp < n-1						&&
	  fic[comp].ficSize == 2				&&
	  ! memcmp( bytes+ fic[comp].ficOffset, "\\*", 2 )	&&
	  fic[comp+ 1].ficSize == 11				)
	{
	const char *		mergeformat= "mergeformat";
	int			i;

	for ( i= 0; i < 11; i++ )
	    {
	    if  ( tolower( bytes[fic[comp+1].ficOffset+ i] ) != mergeformat[i] )
		{ break;	}
	    }

	if  ( i == 11 )
	    { return 1;	}
	}

    return 0;
    }

/************************************************************************/
/*									*/
/*  Extract a numeric rtf tag from the field data.			*/
/*									*/
/************************************************************************/

static void docLookupGetTag(	int *			pVal,
				const DocumentField *	df,
				const char *		tag )
    {
    int			taglen= strlen( tag );
    int			pos;

    const char *	bytes= (const char *)df->dfData.mbBytes;
    int			size= df->dfData.mbSize;

    if  ( tag[0] != '\\' )
	{ SDEB(tag); return;	}

    pos= 0;
    while( pos < size && bytes[pos] != '\\' )
	{ pos++;	}

    while( pos < size )
	{
	int		past= pos+ 1;
	int		end= past;

	while( past < size && isalnum( bytes[past] ) )
	    {
	    if  ( isalpha( bytes[past] ) )
		{ end= past+ 1;	}

	    past++;
	    }

	if  ( end- pos == taglen			&&
	      ! strncmp( bytes+ pos, tag, taglen )	)
	    {
	    if  ( end < past )
		{
		char *	p; 
		*pVal= strtol( bytes+ end, &p, 10 );
		}

	    return;
	    }

	pos= past;
	while( pos < size && bytes[pos] != '\\' )
	    { pos++;	}
	}

    return;
    }

int docFieldGetTc(		const DocumentField *	df,
				char *			pFlag,
				int *			pLevel,
				int *			pNumbered )
    {
    int		level= -1;
    int		flag= -1;

    if  ( df->dfKind != DOCfkTC && df->dfKind != DOCfkTCN )
	{ return -1;	}

    docLookupGetTag( &level, df, "\\tcl" ); level--;
    docLookupGetTag( &flag, df, "\\tcf" );

    if  ( level < 0 )
	{ level= 0;	}
    if  ( flag < 0 )
	{ flag= 'C';	}

    *pLevel= level;
    *pFlag= flag;
    *pNumbered= df->dfKind == DOCfkTC;
    return 0;
    }

/************************************************************************/
/*									*/
/*  Extract the name from a bookmark field.				*/
/*									*/
/*  Return the bookmark name.						*/
/*  The string is NOT '\0' terminated.					*/
/*									*/
/************************************************************************/

int docFieldGetBookmark(	const DocumentField *	df,
				const char **		pMarkName,
				int *			pMarkSize )
    {
    FieldInstructionsComponent	fic[FIC_COUNT];
    int				n;
    int				comp;
    unsigned char *		bytes= df->dfInstructions.mbBytes;

    char *			markName= (char *)0;
    int				markSize= 0;

    if  ( df->dfKind != DOCfkBOOKMARK )
	{ return -1;	}

    n= docSplitFieldInstructions( &(df->dfInstructions), fic, FIC_COUNT );
    if  ( n < 2 )
	{ LDEB(n); return -1;	}

    comp= 0;
    if  ( fic[comp].ficSize != 8					||
	  memcmp( bytes+ fic[comp].ficOffset, "BOOKMARK", 8 )	)
	{ SDEB((char *)bytes); return -1;	}
    comp++;

    markName= (char *)bytes+ fic[comp].ficOffset;
    markSize= fic[comp].ficSize;
    comp++;

    while( comp < n )
	{ LSDEB(comp,(char *)bytes+ fic[comp].ficOffset); comp++; }

    *pMarkName= markName;
    *pMarkSize= markSize;
    return 0;
    }

int docFieldMatchesBookmark(	const DocumentField *		df,
				const char *			markName,
				int				markSize )
    {
    const char *	foundName;
    int			foundSize;

    if  ( df->dfKind == DOCfkBOOKMARK				&&
	  ! docFieldGetBookmark( df, &foundName, &foundSize )	&&
	  foundSize == markSize					&&
	  ! memcmp( foundName, markName, markSize )		)
	{ return 1;	}

    return 0;
    }

static int docFieldSetSingleValueField(	DocumentField *		df,
					const char *		kindName,
					int			kindSize,
					const char *		valueName,
					int			valueSize )
    {
    int			size;
    char *		fresh;
    char *		to;

    size= 1+ kindSize+ 1 + valueSize+ 2;

    fresh= to= (char *)malloc( size+ 1 );
    if  ( ! fresh )
	{ LXDEB(size,fresh); return -1;	}

    *(to++)= ' ';
    memcpy( to, kindName, kindSize ); to += kindSize;
    *(to++)= ' ';
    *(to++)= '"';
    memcpy( to, valueName, valueSize ); to += valueSize;
    *(to++)= '"';

    *(to)= '\0';

    if  ( docSetFieldInst( df, fresh, size ) )
	{ LDEB(size); free( fresh ); return -1;	}

    free( fresh ); return 0;
    }

int docFieldSetBookmark(	DocumentField *		df,
				const char *		markName,
				int			markSize )
    {
    const char		kindName[]= "BOOKMARK";
    int			kindSize= sizeof(kindName)- 1;

    df->dfKind= DOCfkBOOKMARK;

    return docFieldSetSingleValueField( df,
				    kindName, kindSize, markName, markSize );
    }

int docFieldSetRef(		DocumentField *		df,
				const char *		markName,
				int			markSize )
    {
    const char		kindName[]= "REF";
    int			kindSize= sizeof(kindName)- 1;

    df->dfKind= DOCfkREF;

    return docFieldSetSingleValueField( df,
				    kindName, kindSize, markName, markSize );
    }

int docFieldSetPageref(		DocumentField *		df,
				const char *		markName,
				int			markSize )
    {
    const char		kindName[]= "PAGEREF";
    int			kindSize= sizeof(kindName)- 1;

    df->dfKind= DOCfkPAGEREF;

    return docFieldSetSingleValueField( df,
				    kindName, kindSize, markName, markSize );
    }

/************************************************************************/
/*									*/
/*  Split field instructions in a vector of arguments.			*/
/*									*/
/************************************************************************/

int docSplitFieldInstructions(	const MemoryBuffer *		mb,
				FieldInstructionsComponent *	fic,
				int				maxComponent )
    {
    unsigned char *	bytes= mb->mbBytes;

    int			offset= 0;
    int			n= 0;

    for (;;)
	{
	int		size= 0;

	while( offset < mb->mbSize && *bytes == ' ' )
	    { offset++; bytes++;	}

	if  ( offset >= mb->mbSize )
	    { break;	}

	if  ( n >= maxComponent )
	    { LLDEB(n,maxComponent); return 0;	}

	if  ( bytes[0] == '\\'		&&
	      offset < mb->mbSize- 1	&&
	      bytes[1] != ' '		)
	    {
	    fic->ficOffset= offset;

	    offset += 2;  bytes += 2;
	    fic->ficSize= 2;
	    n++; fic++;
	    continue;
	    }

	if  ( bytes[0] == '"' )
	    {
	    offset++; bytes++;

	    if  ( bytes[0] == '"' )
		{ offset++; bytes++; continue; }

	    fic->ficOffset= offset;
	    while( offset < mb->mbSize && *bytes != '"' )
		{ size++; offset++; bytes++;	}

	    if  ( offset >= mb->mbSize )
		{ SDEB((char *)mb->mbBytes); return 0;	}

	    offset++; bytes++;
	    }
	else{
	    fic->ficOffset= offset;
	    while( offset < mb->mbSize && *bytes != ' ' )
		{ size++; offset++; bytes++;	}
	    }

	fic->ficSize= size;
	n++; fic++;
	}

    return n;
    }

/************************************************************************/
/*									*/
/*  Bring a bookmark in a generally accepted format.			*/
/*  I.E: Replace everything that is not isalnum() and truncate to	*/
/*  DOCmaxBOOKMARK.							*/
/*									*/
/************************************************************************/

void docAdaptBookmarkName(	int *		pLength,
				char *		markName )
    {
    int		len= 0;

    while( *markName )
	{
	if  ( len >= DOCmaxBOOKMARK )
	    { *markName= '\0'; break;	}

	if  ( ! isalnum( *markName ) )
	    { *markName= '_';	}

	len++; markName++;
	}

    *pLength= len; return;
    }


