/************************************************************************/
/*									*/
/*  Manage the nesting of fields inside the text.			*/
/*									*/
/************************************************************************/

#   include	"docBufConfig.h"

#   include	<stdlib.h>
#   include	<string.h>
#   include	<stdio.h>
#   include	<limits.h>

#   include	<appDebugon.h>

#   include	<appUnit.h>
#   include	"docBuf.h"
#   include	"docParaString.h"
#   include	"docEdit.h"

/************************************************************************/
/*									*/
/*  Insert the particule that terminates a field.			*/
/*									*/
/*  1)  Find the particule.						*/
/*  2)  Should it be split?						*/
/*  3)  Insert end particule.						*/
/*  4)  Change its attributes. (Superscript for notes!)			*/
/*									*/
/************************************************************************/

static int docFieldInsertEndParticule(
				BufferDocument *		bd,
				const DocumentPosition *	dpEnd,
				int				part1,
				int *				pTailPart,
				int				fieldNumber,
				const PropertyMask *		taSetMask,
				const TextAttribute *		taSet )
    {
    TextParticule *	tp;
    int			textAttrNr;

    BufferItem *	bi= dpEnd->dpBi;

    /*  1  */
    tp= bi->biParaParticules+ part1;
    if  ( tp->tpStrlen > 0				&&
	  dpEnd->dpStroff == tp->tpStroff+ tp->tpStrlen	)
	{ part1++; tp++;	}

    if  ( part1 < bi->biParaParticuleCount )
	{ textAttrNr= tp[ 0].tpTextAttrNr; }
    else{ textAttrNr= tp[-1].tpTextAttrNr; }

    /*  2  */
    if  ( part1 < bi->biParaParticuleCount	&&
	  tp->tpStroff != dpEnd->dpStroff	)
	{
	int	len= tp->tpStroff+ tp->tpStrlen- dpEnd->dpStroff;

	tp->tpStrlen= dpEnd->dpStroff- tp->tpStroff;

	tp= docInsertTextParticule( bi, part1+ 1,
		    dpEnd->dpStroff, len, tp->tpKind, textAttrNr );
	if  ( ! tp )
	    { XDEB(tp); return -1;	}

	part1++;
	}

    /*  3  */
    tp= docMakeSpecialParticule( bi, part1,
		    dpEnd->dpStroff, DOCkindFIELDEND, textAttrNr );
    if  ( ! tp )
	{ XDEB(tp); return -1;	}

    tp->tpObjectNumber= fieldNumber;
    docShiftParticuleOffsets( bd, bi, part1+ 1,
				bi->biParaParticuleCount, tp->tpStrlen );

    /*  4  */
    if  ( taSetMask && ! utilPropMaskIsEmpty( taSetMask ) )
	{
	if  ( docChangeParticuleAttributes( (PropertyMask *)0,
						bd, bi, part1, part1+ 1,
						taSet, taSetMask ) )
	    { LDEB(part1); return -1;	}
	}

    *pTailPart= part1;
    return 0;
    }

/************************************************************************/
/*									*/
/*  Insert the particule that starts a field.				*/
/*									*/
/*  1)  Find the particule.						*/
/*  2)  Should it be split?						*/
/*  3)  Insert start particule.						*/
/*  4)  Change its attributes. (Superscript for notes!)			*/
/*									*/
/************************************************************************/

static int docFieldInsertStartParticule(
				BufferDocument *		bd,
				const DocumentPosition *	dpStart,
				int *				pStroffShift,
				int				part0,
				int *				pHeadPart,
				int *				pTailPart,
				int				fieldNumber,
				const PropertyMask *		taSetMask,
				const TextAttribute *		taSet )
    {
    TextParticule *	tp;
    int			textAttrNr;

    int			part1= *pTailPart;

    BufferItem *	bi= dpStart->dpBi;

    /*  1  */
    tp= bi->biParaParticules+ part0;
    if  ( tp->tpStrlen > 0					&&
	  dpStart->dpStroff == tp->tpStroff+ tp->tpStrlen	)
	{ part0++; tp++;	}

    if  ( part0 < bi->biParaParticuleCount )
	{ textAttrNr= tp[ 0].tpTextAttrNr; }
    else{ textAttrNr= tp[-1].tpTextAttrNr; }

    /*  2  */
    if  ( part0 < bi->biParaParticuleCount	&&
	  tp->tpStroff != dpStart->dpStroff	)
	{
	int		stroff= tp->tpStroff;
	int		len= dpStart->dpStroff- tp->tpStroff;

	tp->tpStrlen= tp->tpStroff+ tp->tpStrlen- dpStart->dpStroff;
	tp->tpStroff= dpStart->dpStroff;

	tp= docInsertTextParticule( bi, part0,
				stroff, len, tp->tpKind, textAttrNr );
	if  ( ! tp )
	    { XDEB(tp); return -1;	}

	if  ( part1 >= part0 )
	    { part1++;	}

	part0++;
	}

    /*  3  */
    tp= docMakeSpecialParticule( bi, part0,
		    dpStart->dpStroff, DOCkindFIELDSTART, textAttrNr );
    if  ( ! tp )
	{ XDEB(tp); return -1;	}

    tp->tpObjectNumber= fieldNumber;

    docShiftParticuleOffsets( bd, bi, part0+ 1,
				bi->biParaParticuleCount, tp->tpStrlen );
    /*  4  */
    if  ( taSetMask && ! utilPropMaskIsEmpty( taSetMask ) )
	{
	if  ( docChangeParticuleAttributes( (PropertyMask *)0, bd,
						bi, part0, part0+ 1,
						taSet, taSetMask ) )
	    { LDEB(part1); return -1;	}
	}

    if  ( part1 >= part0 )
	{ part1++;	}

    *pStroffShift= tp->tpStrlen;
    *pHeadPart= part0; *pTailPart= part1; return 0;
    }

/************************************************************************/
/*									*/
/*  Surround a selected range of text by a text level field.		*/
/*									*/
/************************************************************************/

int docSurroundTextSelectionByField(
				DocumentField **		pDf,
				DocumentSelection *		dsInside,
				DocumentSelection *		dsAround,
				int *				pHeadPart,
				int *				pTailPart,
				BufferDocument *		bd,
				DocumentTree *			ei,
				int				singlePara,
				const DocumentSelection *	ds,
				const PropertyMask *		taSetMask,
				const TextAttribute *		taSet )
    {
    int				headPart;
    int				tailPart;
    int				startShift= 0;

    DocumentSelection		dsBalanced= *ds;

    DocumentField *		df;
    DocumentField *		dfLeft;
    DocumentField *		dfRight;

    int				headMoved= 0;
    int				tailMoved= 0;

    const BufferItem *		sectBi;
    int				headParaNr;
    int				tailParaNr;

    if  ( singlePara && dsBalanced.dsHead.dpBi != dsBalanced.dsTail.dpBi )
	{
	LXXDEB(singlePara,dsBalanced.dsHead.dpBi,dsBalanced.dsTail.dpBi);
	return -1;
	}

    sectBi= docGetSectItem( dsBalanced.dsHead.dpBi );
    if  ( ! sectBi )
	{ XDEB(sectBi); return -1;	}

    if  ( docFindParticuleOfPosition( &headPart, &(dsBalanced.dsHead), 1 ) )
	{ LDEB(dsBalanced.dsHead.dpStroff); return -1;	}
    if  ( docFindParticuleOfPosition( &tailPart, &(dsBalanced.dsTail), 0 ) )
	{ LDEB(dsBalanced.dsTail.dpStroff); return -1;	}

    if  ( docBalanceFieldSelection( &dfLeft, &dfRight,
			&headMoved, &tailMoved,
			&headPart, &tailPart, 
			&(dsBalanced.dsHead), &(dsBalanced.dsTail), ei, bd ) )
	{ LDEB(1); return -1;	}

#   if 0
    if  ( headMoved || tailMoved )
	{ LLDEB(headMoved,tailMoved); /*return -1;*/	}
#   endif

    df= docClaimField( &(bd->bdFieldList) );
    if  ( ! df )
	{ XDEB(df); return -1;	}

    /*  find end, split?, insert end particule */
    if  ( docFieldInsertEndParticule( bd, &(dsBalanced.dsTail),
					tailPart, &tailPart,
					df->dfFieldNumber, taSetMask, taSet ) )
	{ LDEB(1); return -1;	}

    /*  find begin, split?, insert start particule */
    if  ( docFieldInsertStartParticule( bd, &(dsBalanced.dsHead), &startShift,
					headPart, &headPart, &tailPart,
					df->dfFieldNumber, taSetMask, taSet ) )
	{ LDEB(1); return -1;	}

    if  ( dsBalanced.dsHead.dpBi == dsBalanced.dsTail.dpBi )
	{ dsBalanced.dsTail.dpStroff += startShift;	}

    headParaNr= docNumberOfParagraph( dsBalanced.dsHead.dpBi );
    tailParaNr= docNumberOfParagraph( dsBalanced.dsTail.dpBi );
    df->dfSelectionScope= sectBi->biSectSelectionScope;
    df->dfHeadPosition.epParaNr= headParaNr;
    df->dfHeadPosition.epStroff= dsBalanced.dsHead.dpStroff;
    df->dfTailPosition.epParaNr= tailParaNr;
    df->dfTailPosition.epStroff= dsBalanced.dsTail.dpStroff;

    if  ( ! utilPropMaskIsEmpty( taSetMask ) )
	{
	PropertyMask	taDoneMask;

	utilPropMaskClear( &taDoneMask );

	if  ( docChangeParticuleAttributes( &taDoneMask, bd,
						dsBalanced.dsHead.dpBi,
						headPart, tailPart,
						taSet, taSetMask ) )
	    { LDEB(1);	}
	}

    if  ( docInsertFieldInTree( &(ei->eiRootFields), df ) )
	{ LDEB(1); return -1;	}

    if  ( docDelimitFieldInDoc( dsInside, dsAround,
					    pHeadPart, pTailPart, bd, df ) )
	{ LDEB(df->dfFieldNumber); return -1; }

    *pDf= df;

    return 0;
    }

DocumentField * docMakeField(	BufferDocument *		bd,
				DocumentTree *			ei,
				const DocumentSelection *	dsInput,
				int				part0,
				int				part1,
				int				attr0,
				int				attr1 )
    {
    int			paraNr0= docNumberOfParagraph( dsInput->dsHead.dpBi );
    int			paraNr1;
    const BufferItem *	sectBi0;

    DocumentField *	rval= (DocumentField *)0;
    DocumentField *	df;
    int			singleParagraph= 0;

    int			stroff0;
    int			stroff1;

    df= docClaimField( &(bd->bdFieldList) );
    if  ( ! df )
	{ XDEB(df); goto ready;	}
    df->dfKind= DOCfkUNKNOWN;

    sectBi0= docGetSectItem( dsInput->dsHead.dpBi );
    if  ( ! sectBi0 )
	{ XDEB(sectBi0); goto ready;	}

    if  ( dsInput->dsTail.dpBi == dsInput->dsHead.dpBi )
	{ singleParagraph= 1; paraNr1= paraNr0;	}
    else{
	const BufferItem *	sectBi1;

	singleParagraph= 0;
	paraNr1= docNumberOfParagraph( dsInput->dsTail.dpBi );

	sectBi1= docGetSectItem( dsInput->dsTail.dpBi );
	if  ( ! sectBi1 )
	    { XDEB(sectBi1); goto ready;	}

	if  ( sectBi1 != sectBi0 )
	    { XXDEB(sectBi0,sectBi1); goto ready;	}
	}

    {
    TextParticule *	tp;

    tp= docMakeSpecialParticule( dsInput->dsTail.dpBi, part1,
		    dsInput->dsTail.dpStroff, DOCkindFIELDEND, attr1 );
    if  ( ! tp )
	{ XDEB(tp); goto ready;	}
    tp->tpObjectNumber= df->dfFieldNumber;
    stroff1= tp->tpStroff;

    docShiftParticuleOffsets( bd, dsInput->dsTail.dpBi, part1+ 1,
		dsInput->dsTail.dpBi->biParaParticuleCount, tp->tpStrlen );
    }

    {
    TextParticule *	tp;

    tp= docMakeSpecialParticule( dsInput->dsHead.dpBi, part0,
		    dsInput->dsHead.dpStroff, DOCkindFIELDSTART, attr0 );
    if  ( ! tp )
	{ XDEB(tp); goto ready;	}
    tp->tpObjectNumber= df->dfFieldNumber;
    stroff0= tp->tpStroff;

    docShiftParticuleOffsets( bd, dsInput->dsHead.dpBi, part0+ 1,
		dsInput->dsHead.dpBi->biParaParticuleCount, tp->tpStrlen );

    if  ( singleParagraph )
	{ stroff1 += tp->tpStrlen;	}
    }

    df->dfSelectionScope= sectBi0->biSectSelectionScope;
    df->dfHeadPosition.epParaNr= paraNr0;
    df->dfHeadPosition.epStroff= stroff0;
    df->dfTailPosition.epParaNr= paraNr1;
    df->dfTailPosition.epStroff= stroff1;

    if  ( docInsertFieldInTree( &(ei->eiRootFields), df ) )
	{ LDEB(1); goto ready;	}

    rval= df; df= (DocumentField *)0; /* steal */

  ready:

    if  ( df )
	{ docDeleteFieldFromList( &(bd->bdFieldList), df );	}

    return rval;
    }

/************************************************************************/
/*									*/
/*  Find the special field at the head of a paragraph in a list.	*/
/*									*/
/************************************************************************/

int docParaHeadFieldKind(	const BufferItem *	paraBi,
				const BufferDocument *	bd )
    {
    int		fieldKind= -1;

    if  ( paraBi->biInExternalItem == DOCinFOOTNOTE	||
	  paraBi->biInExternalItem == DOCinENDNOTE	)
	{
	BufferItem *		bi= docGetSectItem( paraBi->biParent );
	if  ( ! bi )
	    { XDEB(bi); return -1;		}
	else{
	    DocumentPosition	dp;

	    if  ( docFirstPosition( &dp, bi ) )
		{ LDEB(1); return -1;	}

	    if  ( dp.dpBi == paraBi )
		{ fieldKind= DOCfkCHFTN;	}
	    }
	}

    if  ( paraBi->biParaListOverride > 0 )
	{ fieldKind= DOCfkLISTTEXT;	}

    return fieldKind;
    }

int docDelimitParaHeadField(	DocumentField **	pDfHead,
				DocumentSelection *	dsInsideHead,
				DocumentSelection *	dsAroundHead,
				int *			pHeadPart,
				int *			pTailPart,
				const BufferItem *	paraBi,
				const BufferDocument *	bd )
    {
    int				part;
    const TextParticule *	tp;
    int				fieldKind;

    fieldKind= docParaHeadFieldKind( paraBi, bd );
    if  ( fieldKind < 0 )
	{ return 1;	}

    tp= paraBi->biParaParticules;
    for ( part= 0; part < paraBi->biParaParticuleCount; tp++, part++ )
	{
	DocumentField *	df;

	if  ( tp->tpKind == DOCkindFIELDSTART )
	    {
	    df= docGetFieldByNumber( &(bd->bdFieldList), tp->tpObjectNumber );
	    if  ( ! df )
		{ LXDEB(tp->tpObjectNumber,df); continue;	}

	    if  ( df->dfKind == fieldKind )
		{
		if  ( docDelimitFieldInDoc( dsInsideHead, dsAroundHead,
					    pHeadPart, pTailPart, bd, df ) )
		    { LDEB(1); return -1; }

		*pDfHead= df; return 0;
		}
	    }
	}

    return 1;
    }

/************************************************************************/
/*									*/
/*  Insert the special field at the head of a numbered paragraph.	*/
/*  (buller), or of a foot/endnote.					*/
/*									*/
/*  0)  Insert a text particule at the head of the paragraph as a	*/
/*	temporary field value.						*/
/*  1)  Allocate a field.						*/
/*  2)  Insert start particule.						*/
/*  3)  Insert end particule.						*/
/*  4)  Make sure there is at least one particule after the field.	*/
/*									*/
/************************************************************************/

int docInsertParaHeadField(	DocumentField **	pDfHead,
				DocumentSelection *	dsInsideHead,
				DocumentSelection *	dsAroundHead,
				int *			pHeadPart,
				int *			pTailPart,
				BufferItem *		paraBi,
				BufferDocument *	bd,
				DocumentTree *		ei,
				int			fieldKind,
				int			textAttrNr )
    {
    DocumentField *		df;
    TextParticule *		tpText;
    int				stroffShift= 0;
    int				head= 0;
    const int			stroffHead= 0;
    int				wasEmpty= ( docParaStrlen( paraBi ) == 0 );
    DocumentFieldList *		dfl= &(bd->bdFieldList);

    DocumentSelection		dsField;

    while( head+ 1 < paraBi->biParaParticuleCount			&&
	   paraBi->biParaParticules[head+ 1].tpStroff == stroffHead	&&
	   paraBi->biParaParticules[head+ 1].tpKind == DOCkindFIELDSTART &&
	   docGetFieldKindByNumber( dfl,
	     paraBi->biParaParticules[head+ 1].tpObjectNumber==DOCfkBOOKMARK ) )
	{ head++;	}

    /*  4  */
    if  ( docParaStringReplace( &stroffShift, paraBi,
			    stroffHead, stroffHead, (const char *)"?", 1 ) )
	{ LDEB(docParaStrlen(paraBi)); return -1; }

    if  ( paraBi->biParaParticuleCount > 0 && wasEmpty )
	{
	tpText= paraBi->biParaParticules;
	if  ( tpText->tpKind != DOCkindSPAN )
	    { SDEB(docKindStr(tpText->tpKind)); return -1;	}

	tpText->tpStrlen= 1;

	if  ( docShiftParticuleOffsets( bd, paraBi, head+ 1,
				paraBi->biParaParticuleCount, stroffShift ) )
	    { LDEB(stroffShift); }
	}
    else{
	tpText= docInsertTextParticule( paraBi, head,
			    stroffHead, 1, DOCkindSPAN, textAttrNr );
	if  ( ! tpText )
	    { LXDEB(paraBi->biParaParticuleCount,tpText); return -1; }

	if  ( docShiftParticuleOffsets( bd, paraBi, head+ 1,
				paraBi->biParaParticuleCount, stroffShift ) )
	    { LDEB(stroffShift); }
	}

    docSetParaSelection( &dsField, paraBi, 1, stroffHead, stroffShift );

    /*  2,3  */
    df= docMakeField( bd, ei, &dsField,
		    head, head+ 1, textAttrNr, textAttrNr );
    if  ( ! df )
	{ XDEB(df); return -1;	}

    df->dfKind= fieldKind;

    /*  4  */
    if  ( paraBi->biParaParticuleCount == head+ 3 )
	{
	tpText= docInsertTextParticule( paraBi, head+ 3,
					docParaStrlen( paraBi ), 0,
					DOCkindSPAN, textAttrNr );
	if  ( ! tpText )
	    { LXDEB(paraBi->biParaParticuleCount,tpText); return -1; }
	}

    if  ( docDelimitFieldInDoc( dsInsideHead, dsAroundHead,
					    pHeadPart, pTailPart, bd, df ) )
	{ LDEB(1); return -1; }

    *pDfHead= df;
    return 0;
    }

/************************************************************************/
/*									*/
/*  Delete a field from a document and remove all references.		*/
/*									*/
/*  NOTE The shortcut that optimizes a lot of superfluous calls.	*/
/*									*/
/************************************************************************/

int docDeleteFieldFromParent(	DocumentTree *			ei,
				DocumentField *			df )
    {
    ChildFields *	cf;

    if  ( df->dfParent )
	{ cf= &(df->dfParent->dfChildFields);	}
    else{ cf= &(ei->eiRootFields);		}

    if  ( df->dfChildFields.cfChildCount == 0	&&
	  cf->cfChildCount == 1			)
	{ cf->cfChildCount--;	}
    else{
	if  ( docDeleteChildField( cf, df ) )
	    { LDEB(1); return -1;	}
	}

    df->dfNumberInParent= -1;

    return 0;
    }

/************************************************************************/

static int docDeleteFieldParticule(	BufferItem *		paraBi,
					int			part,
					BufferDocument *	bd )
    {
    int				textAttrNr;
    int				stroffShift= 0;
    const TextParticule *	tp= paraBi->biParaParticules+ part;

    textAttrNr= paraBi->biParaParticules[0].tpTextAttrNr;
    docParaStringReplace( &stroffShift, paraBi,
				tp->tpStroff, tp->tpStroff+ tp->tpStrlen,
				(const char *)0, 0 );
    docDeleteParticules( paraBi, part, 1 );

    docShiftParticuleOffsets(bd,paraBi,
			part,paraBi->biParaParticuleCount, -1 );

    if  ( paraBi->biParaParticuleCount == 0 )
	{
	if  ( ! docInsertTextParticule( paraBi,
					0, 0, 0, DOCkindSPAN, textAttrNr ) )
	    { LDEB(1); return -1;	}
	}

    return 0;
    }

/************************************************************************/
/*									*/
/*  Delete the begin and start particules of a field and the field	*/
/*  itself.								*/
/*									*/
/*  *pPartHead and *pPartTail are set to values that are appropriate to	*/
/*  delete the contents of the field afterward.				*/
/*									*/
/************************************************************************/

int docDeleteField(		BufferItem *			paraBiHead,
				BufferItem *			paraBiTail,
				int *				pPartHead,
				int *				pPartTail,
				int				partHead,
				int				partTail,
				DocumentTree *			ei,
				BufferDocument *		bd,
				DocumentField *			df )
    {
    int		rval= 0;

    if  ( partTail < 0						||
	  partTail >= paraBiTail->biParaParticuleCount		||
	  docDeleteFieldParticule( paraBiTail, partTail, bd )	)
	{ LLDEB(partTail,paraBiTail->biParaParticuleCount); return -1; }

    if  ( partHead < 0						||
	  partHead >= paraBiHead->biParaParticuleCount		||
	  docDeleteFieldParticule( paraBiHead, partHead, bd )	)
	{ LLDEB(partHead,paraBiHead->biParaParticuleCount); return -1; }

    if  ( docDeleteFieldFromParent( ei, df ) )
	{ LDEB(1); rval= -1;	}

    docDeleteFieldFromDocument( bd, df );

    *pPartHead= partHead;

    if  ( paraBiTail == paraBiHead )
	{ *pPartTail= partTail- 2;	}
    else{ *pPartTail= partTail- 1;	}

    return rval;
    }

/************************************************************************/

int docDeleteFieldsInRange(	EditOperation *			eo,
				int				headPart,
				int				tailPart,
				DocumentSelection *		dsBalanced )
    {
    int			bulletsDeleted= 0;
    int			notesDeleted= 0;

    EditRange		erBalanced;

    DocumentField *	dfLeft;
    DocumentField *	dfRight;

    int			headMoved= 0;
    int			tailMoved= 0;

    if  ( docBalanceFieldSelection( &dfLeft, &dfRight,
			    &headMoved, &tailMoved, &headPart, &tailPart, 
			    &(dsBalanced->dsHead), &(dsBalanced->dsTail),
			    eo->eoEi, eo->eoBd ) )
	{ LDEB(1); return -1;	}

    docSetEditRange( &erBalanced, dsBalanced );

    if  ( docDeleteFieldRange( &bulletsDeleted, &notesDeleted, eo->eoBd,
		    &erBalanced, &(eo->eoEi->eiRootFields), dfLeft, dfRight ) )
	{ LDEB(1);	}

    eo->eoNotesDeleted += notesDeleted;
    if  ( bulletsDeleted )
	{ eo->eoFieldUpdate |= FIELDdoLISTTEXT;	}

    return 0;
    }

/************************************************************************/
/*									*/
/*  A set of branches in the document tree, rather than a balanced	*/
/*  selection is deleted. Make sure that no partial fields remain.	*/
/*									*/
/*  This code relies on the fact that a field is deleted if its		*/
/*  contents are fully selected.					*/
/*									*/
/************************************************************************/

int docEditDeleteFieldsForBlockDelete(	EditOperation *			eo,
					const DocumentSelection *	ds )
    {
    int			rval= 0;
    DocumentSelection	dsBalanced= *ds;
    int			headPart;
    int			tailPart;

    headPart= 0;
    tailPart= dsBalanced.dsTail.dpBi->biParaParticuleCount- 1;

    dsBalanced.dsHead.dpStroff= 0;
    dsBalanced.dsTail.dpStroff= docParaStrlen( dsBalanced.dsTail.dpBi );

    if  ( docDeleteFieldsInRange( eo, headPart, tailPart, &dsBalanced ) )
	{ LDEB(1); return -1;	}

    return rval;
    }

/************************************************************************/
/*									*/
/*  Delete all fields that completely fall inside a range in the	*/
/*  document.								*/
/*									*/
/************************************************************************/

static void docDeleteFieldChildren(	int *			pBulletsDeleted,
					int *			pNotesDeleted,
					BufferDocument *	bd,
					DocumentField *		dfp )
    {
    if  ( dfp->dfKind == DOCfkLISTTEXT )
	{ (*pBulletsDeleted)++;	}
    if  ( dfp->dfKind == DOCfkCHFTN && dfp->dfNoteIndex >= 0 )
	{ (*pNotesDeleted)++;	}

    docDeleteChildFields( pBulletsDeleted, pNotesDeleted, bd,
						    &(dfp->dfChildFields) );
    return;
    }

void docDeleteChildFields(	int *			pBulletsDeleted,
				int *			pNotesDeleted,
				BufferDocument *	bd,
				ChildFields *		cf )
    {
    int		i;

    for ( i= 0; i < cf->cfChildCount; i++ )
	{
	DocumentField *	dfc= cf->cfChildren[i];

	docDeleteFieldChildren( pBulletsDeleted, pNotesDeleted, bd, dfc );
	docDeleteFieldFromDocument( bd, dfc );
	}

    docCleanChildFields( cf );
    docInitChildFields( cf );
    }

/************************************************************************/
/*									*/
/*  Delete all fields that overlap a range.				*/
/*									*/
/************************************************************************/

int docDeleteFieldRange(	int *			pBulletsDeleted,
				int *			pNotesDeleted,
				BufferDocument *	bd,
				const EditRange *	er,
				ChildFields *		rootFields,
				DocumentField *		dfLeft,
				DocumentField *		dfRight )
    {
    int			rval= 0;
    int			i;

    int			count= rootFields->cfChildCount;
    DocumentField **	fields= (DocumentField **)0;

    if  ( rootFields->cfChildCount == 0 )
	{ goto ready;	}

    fields= malloc( count* sizeof(DocumentField *) );
    if  ( ! fields )
	{ LXDEB(count,fields); rval= -1; goto ready;	}

    for ( i= 0; i < count; i++ )
	{ fields[i]= rootFields->cfChildren[i];	}

    /*  1  */
    for ( i= 0; i < count; i++ )
	{
	DocumentField *		df= fields[i];
	int			bcm;
	int			ecm;

	if  ( docCompareEditPositions( &(df->dfTailPosition),
						    &(er->erHead) ) <= 0 )
	    { continue;	}

	if  ( docCompareEditPositions( &(df->dfHeadPosition),
						    &(er->erTail) ) >= 0 )
	    { break;	}

	bcm= docCompareEditPositions( &(df->dfHeadPosition), &(er->erHead) );
	ecm= docCompareEditPositions( &(df->dfTailPosition), &(er->erTail) );

	if  ( bcm < 0 || ecm > 0 )
	    {
	    docDeleteFieldRange( pBulletsDeleted, pNotesDeleted, bd, er,
						    &(df->dfChildFields),
						    dfLeft, dfRight );
	    }
	else{
	    if  ( df->dfKind == DOCfkLISTTEXT )
		{ (*pBulletsDeleted)++;	}
	    if  ( df->dfKind == DOCfkCHFTN && df->dfNoteIndex >= 0 )
		{ (*pNotesDeleted)++;	}

	    docDeleteFieldChildren( pBulletsDeleted, pNotesDeleted, bd, df );
	    docDeleteChildField( rootFields, df );
	    docDeleteFieldFromDocument( bd, df );
	    }
	}

  ready:

    if  ( fields )
	{ free( fields );	}

    return rval;
    }

