/************************************************************************/
/*									*/
/*  Buffer administration routines.					*/
/*									*/
/************************************************************************/

#   include	"docBufConfig.h"

#   include	<stdlib.h>
#   include	<string.h>
#   include	<stdio.h>

#   include	<utilMatchFont.h>

#   include	<appDebugon.h>

#   include	"docBuf.h"

/************************************************************************/

void docDeleteFieldFromDocument(	BufferDocument *	bd,
					DocumentField *		df )
    {
    if  ( df->dfNoteIndex >= 0 )
	{
	docCleanNote( bd, bd->bdNotesList.nlNotes+ df->dfNoteIndex );
	docInitNote( bd->bdNotesList.nlNotes+ df->dfNoteIndex );
	}

    docDeleteFieldFromList( &(bd->bdFieldList), df );

    return;
    }

/************************************************************************/
/*									*/
/*  Clean and free a whole document.					*/
/*									*/
/************************************************************************/

void docFreeDocument( BufferDocument *	bd )
    {
    int		i;

    docCleanExternalItem( bd, &(bd->bdBody) );

    for ( i= 0; i < bd->bdNotesList.nlNoteCount; i++ )
	{ docCleanNote( bd, &(bd->bdNotesList.nlNotes[i]) );	}

    docCleanExternalItem( bd, &(bd->bdEiFtnsep) );
    docCleanExternalItem( bd, &(bd->bdEiFtnsepc) );
    docCleanExternalItem( bd, &(bd->bdEiFtncn) );

    docCleanExternalItem( bd, &(bd->bdEiAftnsep) );
    docCleanExternalItem( bd, &(bd->bdEiAftnsepc) );
    docCleanExternalItem( bd, &(bd->bdEiAftncn) );

    docCleanStyleSheet( &(bd->bdStyleSheet) );
    docCleanFieldList( &(bd->bdFieldList) );
    docCleanShapeList( &(bd->bdShapeList) );
    utilCleanPagedList( &(bd->bdObjectList) );

    docCleanDocumentProperties( &(bd->bdProperties) );

    if  ( bd->bdNotesList.nlNotes )
	{ free( bd->bdNotesList.nlNotes );	}

    utilCleanNumberedPropertiesList( &(bd->bdTextAttributeList) );
    utilCleanNumberedPropertiesList( &(bd->bdBorderPropertyList) );
    utilCleanNumberedPropertiesList( &(bd->bdItemShadingList) );
    utilCleanNumberedPropertiesList( &(bd->bdFramePropertyList) );

    free( bd );
    }

/************************************************************************/

int docGetDefaultFont(		BufferDocument *	bd )
    {
    int				i;
    const DocumentFontList *	dfl= &(bd->bdProperties.dpFontList);

    if  ( bd->bdDefaultFont >= 0 )
	{ return bd->bdDefaultFont;	}

    if  ( bd->bdProperties.dpDefaultFont >= 0 )
	{
	bd->bdDefaultFont= bd->bdProperties.dpDefaultFont;
	return bd->bdDefaultFont;
	}

    for ( i= 0; i < dfl->dflFontCount; i++ )
	{
	if  ( docFontListGetFontByNumber( dfl, i ) )
	    {
	    bd->bdDefaultFont= i;
	    return bd->bdDefaultFont;
	    }
	}

    /*LDEB(bd->bdDefaultFont);*/
    return bd->bdDefaultFont;
    }

/************************************************************************/
/*									*/
/*  Initialise a BufferDocument.					*/
/*									*/
/************************************************************************/

static void docInitDocument(	BufferDocument *	bd )
    {
    utilInitTextAttributeList( &(bd->bdTextAttributeList) );
    docInitBorderPropertyList( &(bd->bdBorderPropertyList) );
    docInitItemShadingList( &(bd->bdItemShadingList) );
    docInitFramePropertyList( &(bd->bdFramePropertyList) );

    utilInitPagedList( &(bd->bdObjectList) );
    utilStartPagedList( &(bd->bdObjectList),
			    sizeof(InsertedObject),
			    (InitPagedListItem)docInitInsertedObject,
			    (CleanPagedListItem)docCleanInsertedObject );

    docInitDocumentProperties( &(bd->bdProperties) );
    bd->bdDefaultFont= -1;

    docInitExternalItem( &(bd->bdBody) );

    docInitStyleSheet( &(bd->bdStyleSheet) );
    docInitFieldList( &(bd->bdFieldList) );
    docInitShapeList( &(bd->bdShapeList) );

    bd->bdNotesList.nlNotes= (DocumentNote *)0;
    bd->bdNotesList.nlNoteCount= 0;

    docInitExternalItem( &(bd->bdEiFtnsep) );
    docInitExternalItem( &(bd->bdEiFtnsepc) );
    docInitExternalItem( &(bd->bdEiFtncn) );

    docInitExternalItem( &(bd->bdEiAftnsep) );
    docInitExternalItem( &(bd->bdEiAftnsepc) );
    docInitExternalItem( &(bd->bdEiAftncn) );

    bd->bdAutoHyphenate= 0;

    return;
    }

/************************************************************************/
/*									*/
/*  Insert the body of the document tree in a fresh document.		*/
/*									*/
/************************************************************************/

static int docSetupDocumentBody(	BufferDocument *	bd )
    {
    bd->bdBody.eiRoot= malloc(sizeof(BufferItem));
    if  ( ! bd->bdBody.eiRoot )
	{ XDEB(bd->bdBody.eiRoot); return -1;	}

    docInitItem( bd->bdBody.eiRoot, (BufferItem *)0, bd,
						0, DOClevBODY, DOCinBODY );

    return 0;
    }

/************************************************************************/
/*									*/
/*  Create a fresh document.						*/
/*									*/
/************************************************************************/

BufferDocument * docNewDocument( void )
    {
    BufferDocument *	bd= (BufferDocument *)malloc( sizeof(BufferDocument) );
    if  ( ! bd )
	{ XDEB(bd); return bd;	}

    docInitDocument( bd );

    if  ( docSetupDocumentBody( bd ) )
	{ LDEB(1); docFreeDocument( bd ); return (BufferDocument *)0;	}

    return bd;
    }

/************************************************************************/
/*									*/
/*  Make a new document consisting of one paragraph with one empty	*/
/*  particule. This is the starting point for editing and for reading	*/
/*  plain text documents.						*/
/*									*/
/************************************************************************/

BufferDocument * docNewFile(	TextAttribute *			taDefault,
				const char *			defaultFontName,
				int				defaultFontSize,
				const PostScriptFontList *	psfl,
				const DocumentGeometry *	dg )
    {
    BufferDocument *		bd;
    DocumentProperties *	dp;

    PropertyMask		taNewMask;
    TextAttribute		taNew;
    int				textAttributeNumber;

    BufferItem *		bi;

    utilPropMaskClear( &taNewMask );

    bd= docNewDocument();

    dp= &(bd->bdProperties);
    dp->dpGeometry= *dg;

    /*  3  */
    if  ( psfl )
	{
	if  ( utilAddPsFontsToDocList( &(dp->dpFontList), psfl ) )
	    { LDEB(psfl->psflFamilyCount); goto failed;	}
	}
    else{
	if  ( utilAddBase35FontsToDocList( &(dp->dpFontList) ) )
	    { LDEB(35); goto failed;	}
	}

    utilInitTextAttribute( &taNew );
    taNew.taFontNumber= docGetFontByName( &(dp->dpFontList), defaultFontName );
    if  ( taNew.taFontNumber < 0 )
	{ LDEB(taNew.taFontNumber); goto failed;	}
    PROPmaskADD( &taNewMask, TApropDOC_FONT_NUMBER );

    if  ( defaultFontSize < 6 || defaultFontSize > 100 )
	{ LDEB(defaultFontSize); defaultFontSize= 24;	}
    taNew.taFontSizeHalfPoints= defaultFontSize;
    PROPmaskADD( &taNewMask, TApropFONTSIZE );

    dp->dpDefaultFont= taNew.taFontNumber;
    docRememberFontUsed( bd, &taNew );

    textAttributeNumber= utilTextAttributeNumber(
				    &(bd->bdTextAttributeList), &taNew );
    if  ( textAttributeNumber < 0 )
	{ SLDEB(defaultFontName,textAttributeNumber); goto failed;	}

    bi= docInsertItem( bd, bd->bdBody.eiRoot, -1, DOClevSECT );
    if  ( ! bi )
	{ XDEB(bi); goto failed;	}

    bi->biSectDocumentGeometry= dp->dpGeometry;

    bi= docInsertEmptyParagraph( bd, bi, textAttributeNumber );
    if  ( ! bi )
	{ XDEB(bi); goto failed;	}

    *taDefault= taNew;
    return bd;

  failed:

    docFreeDocument( bd );

    return (BufferDocument *)0;
    }

/************************************************************************/
/*									*/
/*  Statistics about a document. Used in the 'Document Properties'	*/
/*  dialog.								*/
/*									*/
/************************************************************************/

static void docCollectItemStatistics(	DocumentStatistics *	ds,
					const BufferItem *	bi )
    {
    int		i;

    switch( bi->biLevel )
	{
	case DOClevSECT:
	case DOClevBODY:
	case DOClevCELL:
	case DOClevROW:
	    for ( i= 0; i < bi->biChildCount; i++ )
		{ docCollectItemStatistics( ds, bi->biChildren[i] ); }
	    break;

	case DOClevPARA:
	    ds->dsParagraphCount++;
	    ds->dsCharacterCount += docParaStrlen( bi );
	    ds->dsLineCount += bi->biParaLineCount;

	    for ( i= 0; i < bi->biParaLineCount; i++ )
		{ ds->dsWordCount += bi->biParaLines[i].tlWordCount;	}

	    break;

	default:
	    LDEB(bi->biLevel); return;
	}

    return;
    }

void docCollectDocumentStatistics(	DocumentStatistics *	ds,
					const BufferDocument *	bd )
    {
    docInitDocumentStatistics( ds );

    docCollectItemStatistics( ds, bd->bdBody.eiRoot );

    ds->dsPageCount= bd->bdBody.eiRoot->biBelowPosition.lpPage+ 1;

    return;
    }

/************************************************************************/
/*									*/
/*  Remember that a font is used.					*/
/*									*/
/************************************************************************/

void docRememberFontUsed(		BufferDocument *	bd,
					const TextAttribute *	ta )
    {
    DocumentProperties *	dp= &(bd->bdProperties);

    if  ( ta->taFontNumber >= 0					&&
	  ta->taFontNumber < dp->dpFontList.dflFontCount	)
	{
	DocumentFont *	df;

	df= docFontListGetFontByNumber( &dp->dpFontList, ta->taFontNumber );
	if  ( df )
	    { df->dfUsed= 1;	}
	}

    return;
    }

/************************************************************************/
/*									*/
/*  Determine a text attribute number for a particule.			*/
/*									*/
/************************************************************************/

int docTextAttributeNumber(	BufferDocument *	bd,
				const TextAttribute *	ta )
    {
    const DocumentProperties *	dp= &(bd->bdProperties);
    const DocumentFontList *	dfl= &(dp->dpFontList);

    int				textAttributeNumber;

    TextAttribute		taCopy= *ta;

    if  ( taCopy.taFontNumber < 0			||
	  taCopy.taFontNumber >= dfl->dflFontCount	)
	{
	if  ( taCopy.taFontNumber >= dfl->dflFontCount )
	    { LLDEB(taCopy.taFontNumber,dfl->dflFontCount);	}

	taCopy.taFontNumber= docGetDefaultFont( bd );

	if  ( taCopy.taFontNumber < 0 )
	    { taCopy.taFontNumber= 0;	}

	if  ( taCopy.taFontNumber >= dfl->dflFontCount )
	    { LLDEB(taCopy.taFontNumber,dfl->dflFontCount);	}
	}

    textAttributeNumber= utilTextAttributeNumber(
				    &(bd->bdTextAttributeList), &taCopy );
    if  ( textAttributeNumber < 0 )
	{ LDEB(textAttributeNumber); return -1;	}

    return textAttributeNumber;
    }

/************************************************************************/
/*									*/
/*  Determine the plain text attribute for this document.		*/
/*									*/
/************************************************************************/

void docPlainTextAttribute(		TextAttribute *		ta,
					BufferDocument *	bd )
    {
    utilInitTextAttribute( ta );
    if  ( bd )
	{ ta->taFontNumber= docGetDefaultFont( bd );	}
    ta->taFontSizeHalfPoints= 24;

    return;
    }

/************************************************************************/
/*									*/
/*  Determine the number of the attribute belonging to the scaps	*/
/*  font size. Note that this is used for the shifted lower case	*/
/*  letters only.							*/
/*									*/
/************************************************************************/

int docScapsAttributeNumber(		BufferDocument *	bd,
					const TextAttribute *	ta )
    {
    TextAttribute	taScaps= *ta;

    taScaps.taFontSizeHalfPoints= SCAPS_SIZE( ta->taFontSizeHalfPoints );
    taScaps.taSmallCaps= 0;

    return utilTextAttributeNumber( &(bd->bdTextAttributeList), &taScaps );
    }

/************************************************************************/

void docDeleteDrawingShape(	struct BufferDocument *	bd,
				DrawingShape *		ds )
    {
    int		i;

    for ( i= 0; i < ds->dsChildCount; i++ )
	{
	if  ( ds->dsChildren[i] )
	    {
	    docDeleteDrawingShape( bd, ds->dsChildren[i] );
	    ds->dsChildren[i]= (DrawingShape *)0;
	    }
	}

    docCleanExternalItem( bd, &(ds->dsDocumentTree) );
    docInitExternalItem( &(ds->dsDocumentTree) ); /* Checked in clean */

    /* Done from the paged list: docCleanDrawingShape( bd, ds ); */
    docDeleteShapeFromList( &(bd->bdShapeList), ds );
    }

