/************************************************************************/
/*									*/
/*  Drawing primitives for X11/Motif.					*/
/*									*/
/************************************************************************/

#   include	"appFrameConfig.h"

#   include	<stddef.h>
#   include	<stdio.h>
#   include	<math.h>

#   include	<geo2DInteger.h>
#   include	"appDraw.h"
#   include	"appScreenFont.h"
#   include	"guiPixmapImpl.h"

#   include	<appDebugon.h>

#   ifdef USE_MOTIF

/************************************************************************/
/*									*/
/*  Collect exposures for optimised drawing.				*/
/*									*/
/************************************************************************/

#   define	LOG_EXPOSES	0

void appCollectExposures(	DocumentRectangle *	drClip,
				AppDrawingData *	add,
				int			ox,
				int			oy,
				XEvent *		event )
    {
    Region		region;

    region= XCreateRegion();
    XtAddExposureToRegion( event, region );

    drClip->drX0= ox+ (int)event->xexpose.x;
    drClip->drY0= oy+ (int)event->xexpose.y;
    drClip->drX1= ox+ (int)event->xexpose.x+ (int)event->xexpose.width;
    drClip->drY1= oy+ (int)event->xexpose.y+ (int)event->xexpose.height;

#   if LOG_EXPOSES
    appDebug( "E=====: [%4d+%4d]x[%4d+%4d]\n",
		    drClip->drX0,
		    drClip->drX1- drClip->drX0+ 1,
		    drClip->drY0,
		    drClip->drY1- drClip->drY0+ 1 );
#   endif

    while( QLength( add->addDisplay ) > 0 )
	{
	XEvent			nextEvent;
	DocumentRectangle	drMore;

	XPeekEvent( add->addDisplay, &nextEvent );

	if  ( nextEvent.type != Expose && nextEvent.type != GraphicsExpose )
	    { break;	}
	if  ( nextEvent.xexpose.window != add->addDrawable )
	    { break;	}

	XNextEvent( add->addDisplay, &nextEvent );
	XtAddExposureToRegion( &nextEvent, region );
	drMore.drX0= ox+ (int)nextEvent.xexpose.x;
	drMore.drY0= oy+ (int)nextEvent.xexpose.y;
	drMore.drX1= ox+ (int)nextEvent.xexpose.x+ (int)nextEvent.xexpose.width;
	drMore.drY1= oy+ (int)nextEvent.xexpose.y+ (int)nextEvent.xexpose.height;

#	if LOG_EXPOSES
	appDebug( "E+++++: [%4d+%4d]x[%4d+%4d]\n",
			drMore.drX0,
			drMore.drX1- drMore.drX0+ 1,
			drMore.drY0,
			drMore.drY1- drMore.drY0+ 1 );
#	endif

	geoUnionRectangle( drClip, drClip, &drMore );

#	if LOG_EXPOSES
	appDebug( "E.....: [%4d+%4d]x[%4d+%4d]\n",
			drClip->drX0,
			drClip->drX1- drClip->drX0+ 1,
			drClip->drY0,
			drClip->drY1- drClip->drY0+ 1 );
#	endif
	}

    XSetRegion( add->addDisplay, add->addGc, region );

    XDestroyRegion( region );

    return;
    }

void appExposeDrawingData(	const AppDrawingData *	add )
    {
    if  ( ! add->addDrawable )
	{ XDEB(add->addDrawable); return; }

    XClearArea( add->addDisplay, add->addDrawable, 0, 0, 0, 0, True );
    }

void appExposeRectangle(	const AppDrawingData *		add,
				const DocumentRectangle *	drExpose )
    {
    int		x= drExpose->drX0;
    int		y= drExpose->drY0;
    int		wide= drExpose->drX1- drExpose->drX0+ 1;
    int		high= drExpose->drY1- drExpose->drY0+ 1;

    XEvent	xev;

    if  ( ! add->addDrawable )
	{ XDEB(add->addDrawable); return; }

    xev.type= Expose;
    xev.xexpose.serial= 0;
    xev.xexpose.send_event= True;
    xev.xexpose.display= add->addDisplay;
    xev.xexpose.window= add->addDrawable;
    xev.xexpose.x= x;
    xev.xexpose.y= y;
    xev.xexpose.width= wide;
    xev.xexpose.height= high;
    xev.xexpose.count= 0;

    if  ( XSendEvent( add->addDisplay, add->addDrawable,
					0, ExposureMask, &xev ) != Success )
	{ /*LDEB(1);*/	}

    return;
    }

/************************************************************************/

#   ifdef USE_XFT

static void appMotifXftDrawCreate(	AppDrawingData *	add )
    {
    Display *		display= add->addDisplay;
    int			screen= add->addScreen;
    Colormap		cmap= DefaultColormap( display, screen );
    Visual *		vis= DefaultVisual( display, screen );

    if  ( add->addXftDrawable )
	{ return;				}

    add->addXftColorList.axclDisplay= display;
    add->addXftColorList.axclVisual= vis;
    add->addXftColorList.axclColormap= cmap;

    add->addXftDrawable= XftDrawCreate( add->addXftColorList.axclDisplay,
					    add->addDrawable,
					    add->addXftColorList.axclVisual,
					    add->addXftColorList.axclColormap );

    if  ( ! add->addXftDrawable )
	{ XDEB(add->addXftDrawable);	}

    return;
    }

#   endif


void appCleanDrawingData(	AppDrawingData *	add )
    {
    if  ( add->addDisplay )
	{ appDrawCleanScreenFontList( add );	}

#   ifdef USE_XFT
    appCleanDrawingDataXft( add );
#   endif

    if  ( add->addGc )
	{ XFreeGC( add->addDisplay, add->addGc );	}

    return ;
    }

void appSetScreenDrawingEnvironment(
			AppDrawingData *		add,
			AppColors *			colors,
			double				magnification,
			double				xfac,
			double				screenPixelsPerMM,
			const PostScriptFontList *	psfl )
    {
    add->addMagnification= magnification;
    add->addMagnifiedPixelsPerTwip= xfac;
    add->addScreenPixelsPerMM= screenPixelsPerMM;

    appDrawSetPostScriptFontList( add, psfl );
    add->addForScreenDrawing= 1;

    return;
    }

void appCloneDrawingEnvironment(	AppDrawingData *	add,
					const AppDrawingData *	parent_add,
					double			magnification,
					double			xfac,
					APP_WINDOW		drawable )
    {
    if  ( drawable )
	{ add->addForScreenDrawing= 1;	}

    add->addMagnification= magnification;
    add->addMagnifiedPixelsPerTwip= xfac;
    add->addScreenPixelsPerMM= parent_add->addScreenPixelsPerMM;

    appDrawSetPostScriptFontList( add, parent_add->addPostScriptFontList );

    add->addDisplay= parent_add->addDisplay;
    add->addScreen= parent_add->addScreen;
    add->addDrawable= drawable;

    add->addAvoidFontconfig= parent_add->addAvoidFontconfig;
#   ifdef USE_XFT
    if  ( ! add->addAvoidFontconfig )
	{ appMotifXftDrawCreate( add );	}
#   endif

    if  ( add->addDisplay )
	{
	add->addGc= XCreateGC( add->addDisplay, add->addDrawable,
							0L, (XGCValues *)0 );
	}

    add->addForeColor= parent_add->addForeColor;
    add->addBackColor= parent_add->addBackColor;
    add->addTopColor= parent_add->addTopColor;
    add->addBottomColor= parent_add->addBottomColor;

    return;
    }


int appSetDrawingDataForWidget(	APP_WIDGET		w,
				AppColors *		colors,
				double			magnification,
				AppDrawingData *	add )
    {
    double		horPixPerMM;
    double		verPixPerMM;
    double		xfac;
    double		yfac;

    Colormap		cmap;

    add->addForScreenDrawing= 1;
    add->addColors= colors;

    add->addDisplay= XtDisplay( w );
    add->addScreen= DefaultScreen( add->addDisplay );
    add->addDrawable= XtWindow( w );
    add->addGc= XCreateGC( add->addDisplay, add->addDrawable, 0, NULL );

#   ifdef USE_XFT
    if  ( ! add->addAvoidFontconfig )
	{ appMotifXftDrawCreate( add );	}
#   endif

    horPixPerMM= ( (double)DisplayWidth( add->addDisplay, add->addScreen ) )/
			    DisplayWidthMM( add->addDisplay, add->addScreen );
    verPixPerMM= ( (double)DisplayHeight( add->addDisplay, add->addScreen ) )/
			    DisplayHeightMM( add->addDisplay, add->addScreen );

    xfac=  ( 25.4/ ( 20.0* 72.0 ) )* horPixPerMM;
    yfac=  ( 25.4/ ( 20.0* 72.0 ) )* verPixPerMM;

    xfac *= magnification;
    yfac *= magnification;

    add->addMagnification= magnification;
    add->addMagnifiedPixelsPerTwip= sqrt( xfac* yfac );
    add->addScreenPixelsPerMM= sqrt( horPixPerMM* verPixPerMM );

    XtVaGetValues( w,
			XmNforeground,		&(add->addForeColor.pixel),
			XmNbackground,		&(add->addBackColor.pixel),
			XmNtopShadowColor,	&(add->addTopColor.pixel),
			XmNbottomShadowColor,	&(add->addBottomColor.pixel),
			NULL );

    cmap= DefaultColormap( add->addDisplay, add->addScreen );

    XQueryColor( add->addDisplay, cmap, &(add->addForeColor) );
    XQueryColor( add->addDisplay, cmap, &(add->addBackColor) );
    XQueryColor( add->addDisplay, cmap, &(add->addTopColor) );
    XQueryColor( add->addDisplay, cmap, &(add->addBottomColor) );

    return 0;
    }

int appDrawWhiteColor(		AppDrawingData *		add,
				XColor *			xc )
    {
    xc->red= xc->green= xc->blue= 0xffff;
    xc->pixel= WhitePixel( add->addDisplay, add->addScreen );
    return 0;
    }

int appDrawBlackColor(		AppDrawingData *		add,
				XColor *			xc )
    {
    xc->red= xc->green= xc->blue= 0x0000;
    xc->pixel= BlackPixel( add->addDisplay, add->addScreen );
    return 0;
    }

void appDrawSetForegroundBlack(	AppDrawingData *	add )
    {
    XColor		xc;

    appDrawBlackColor( add, &xc );

    appDrawSetForegroundColor( add, &xc );
    }

void appDrawSetForegroundWhite(	AppDrawingData *	add )
    {
    XColor		xc;

    appDrawWhiteColor( add, &xc );

    appDrawSetForegroundColor( add, &xc );
    }

void appDrawSetBackgroundWhite(	AppDrawingData *	add )
    {
    XSetBackground( add->addDisplay, add->addGc,
			    WhitePixel( add->addDisplay, add->addScreen ) );
    }

void appDrawSetForegroundColor(	AppDrawingData *	add,
				XColor *		xc )
    {
    XSetForeground( add->addDisplay, add->addGc, xc->pixel );

#   ifdef USE_XFT
    appSolidXftColor( &(add->addXftCurrentColor), xc );
#   endif
    }

void appDrawSetBackgroundColor(	AppDrawingData *	add,
				XColor *		xc )
    { XSetBackground( add->addDisplay, add->addGc, xc->pixel ); }

void appDrawFillRectangle(	AppDrawingData *	add,
				int			x,
				int			y,
				int			wide,
				int			high )
    {
#   ifdef USE_XFT

    if  ( ! appDrawFillRectangleXft( add->addXftDrawable,
			&(add->addXftCurrentColor), &(add->addXftColorList),
			x, y, wide, high ) )
	{ return;	}

#   endif

    XFillRectangle( add->addDisplay, add->addDrawable, add->addGc,
							    x, y, wide, high );
    }

void appDrawDrawRectangle(	AppDrawingData *	add,
				int			x,
				int			y,
				int			wide,
				int			high )
    {
    XDrawRectangle( add->addDisplay, add->addDrawable, add->addGc,
							    x, y, wide, high );
    }

void appDrawDrawLine(		AppDrawingData *	add,
				int			x0,
				int			y0,
				int			x1,
				int			y1 )
    {
    XDrawLine( add->addDisplay, add->addDrawable, add->addGc, x0, y0, x1, y1 );
    }

void appDrawMoveArea(		AppDrawingData *	add,
				int			src_x,
				int			src_y,
				int			wide,
				int			high,
				int			dest_x,
				int			dest_y )
    {
    XCopyArea( add->addDisplay, add->addDrawable, add->addDrawable, add->addGc,
				    src_x, src_y, wide, high, dest_x, dest_y );
    }

void appDrawNoClipping(		AppDrawingData *	add )
    {
#   if LOG_EXPOSES
    appDebug( "C=====: OFF\n" );
#   endif

#   ifdef USE_XFT
    if  ( add->addXftDrawable )
	{
	XftDrawSetClip( add->addXftDrawable, None );
	}
#   endif

    XSetClipMask( add->addDisplay, add->addGc, None );

    return;
    }

extern void appDrawSetClipRect(	AppDrawingData *		add,
				const DocumentRectangle *	drClip )
    {
    XRectangle		xRect;

    xRect.x= drClip->drX0;
    xRect.y= drClip->drY0;
    xRect.width= drClip->drX1- drClip->drX0+ 1;
    xRect.height= drClip->drY1- drClip->drY0+ 1;

    if  ( xRect.width < 1 || xRect.height < 1 )
	{ LLDEB(xRect.width,xRect.height); return; }

#   if LOG_EXPOSES
    appDebug( "C=====: [%4d+%4d]x[%4d+%4d]\n",
		    drClip->drX0,
		    drClip->drX1- drClip->drX0+ 1,
		    drClip->drY0,
		    drClip->drY1- drClip->drY0+ 1 );
#   endif

#   ifdef USE_XFT
    if  ( add->addXftDrawable )
	{
	XftDrawSetClipRectangles( add->addXftDrawable, 0, 0, &xRect, 1 );
	}
#   endif

    XSetClipRectangles( add->addDisplay, add->addGc,
						0, 0, &xRect, 1, Unsorted );

    return;
    }

void appDrawFillPolygon(		AppDrawingData *	add,
					const APP_POINT *	points,
					int			count )
    {
    XFillPolygon( add->addDisplay, add->addDrawable, add->addGc,
			(APP_POINT *)points, count, Complex, CoordModeOrigin );
    }

void appDrawDrawLines(			AppDrawingData *	add,
					const APP_POINT *	points,
					int			count,
					int			close )
    {
    int		i;

    /*
    XDrawLines( add->addDisplay, add->addDrawable, add->addGc,
				(APP_POINT *)points, count, CoordModeOrigin );
    */

    count--;

    for ( i= 0; i < count; i++ )
	{
	XDrawLine( add->addDisplay, add->addDrawable, add->addGc,
					    points[i+0].x, points[i+0].y,
					    points[i+1].x, points[i+1].y );
	}

    if  ( close )
	{
	XDrawLine( add->addDisplay, add->addDrawable, add->addGc,
					    points[i].x, points[i].y,
					    points[0].x, points[0].y );
	}

    return;
    }

void appDrawDrawPixmap(			AppDrawingData *	add,
					const ScreenPixmap	sp,
					int			xSrc,
					int			ySrc,
					int			wide,
					int			high,
					int			xDest,
					int			yDest )
    {
    if  ( xSrc < 0 || ySrc < 0 )
	{ LLDEB(xSrc,ySrc); return;	}

    if  ( xDest < 0 )
	{ xSrc -= xDest; wide += xDest; xDest= 0; }
    if  ( yDest < 0 )
	{ ySrc -= yDest; high += yDest; yDest= 0; }

    if  ( wide > 0 && high > 0 )
	{
	XCopyArea( add->addDisplay, sp->spBitmapImage,
				    add->addDrawable, add->addGc,
				    xSrc, ySrc, wide, high, xDest, yDest );
	}

    return;
    }

int appDrawGetSizeOfWidget(		int *		pWide,
					int *		pHigh,
					APP_WIDGET	w )
    {
    Dimension		wide;
    Dimension		high;

    XtVaGetValues( w,	XmNwidth,	&wide,
			XmNheight,	&high,
			NULL );

    *pWide= wide; *pHigh= high; return 0;
    }

void appDrawFreePixmap(	AppDrawingData *	add,
			APP_BITMAP_IMAGE	pixmap )
    {
    XFreePixmap( add->addDisplay, pixmap );
    }

APP_BITMAP_MASK appMakeBitmap(	AppDrawingData *	add,
				int			wide,
				int			high )
    {
    return XCreatePixmap( add->addDisplay, add->addDrawable,
						    wide, high, 1 );
    }

APP_BITMAP_IMAGE appMakePixmap(	AppDrawingData *	add,
				int			wide,
				int			high )
    {
    int			depth= DefaultDepth( add->addDisplay, add->addScreen );
    APP_BITMAP_IMAGE	rval;

    rval= XCreatePixmap( add->addDisplay, add->addDrawable,
						    wide, high, depth );
    if  ( ! rval )
	{ XDEB(rval);	}

    XSetForeground( add->addDisplay, add->addGc,
			    WhitePixel( add->addDisplay, add->addScreen ) );

    XFillRectangle( add->addDisplay, rval, add->addGc, 0, 0, wide, high );

    return rval;
    }

void appDrawDrawSegments(	AppDrawingData *	add,
				APP_SEGMENT *		segments,
				int			count )
    {
    XDrawSegments( add->addDisplay, add->addDrawable, add->addGc,
							    segments, count );
    }

/************************************************************************/
/*									*/
/*  Draw an arc. Angles are in X11 style:				*/
/*									*/
/*  Unit:		1/64 degree.					*/
/*  Origin:		At the right of the circle.			*/
/*  Orientation:	Counterclockwise.				*/
/*									*/
/************************************************************************/

void appDrawDrawArc(		AppDrawingData *	add,
				int			x,
				int			y,
				int			wide,
				int			high,
				int			alpha0,
				int			alpha_step )
    {
    XDrawArc( add->addDisplay, add->addDrawable, add->addGc,
					x, y, wide, high, alpha0, alpha_step );
    }

void appDrawFillArc(		AppDrawingData *	add,
				int			x,
				int			y,
				int			wide,
				int			high,
				int			alpha0,
				int			alpha_step )
    {
    XFillArc( add->addDisplay, add->addDrawable, add->addGc,
					x, y, wide, high, alpha0, alpha_step );
    }

void appDrawDrawArcs(		AppDrawingData *	add,
				const APP_ARC *		arcs,
				int			count )
    {
    XDrawArcs( add->addDisplay, add->addDrawable, add->addGc,
						    (APP_ARC *)arcs, count );
    }

void appDrawFillArcs(		AppDrawingData *	add,
				const APP_ARC *		arcs,
				int			count )
    {
    XFillArcs( add->addDisplay, add->addDrawable, add->addGc,
						    (APP_ARC *)arcs, count );
    }

int appDrawSetLineAttributes(	AppDrawingData *	add,
				int			lineWidth,
				int			lineStyle,
				int			capStyle,
				int			joinStyle,
				const unsigned char *	dashList,
				int			dashCount )
    {
    XSetLineAttributes( add->addDisplay, add->addGc,
				lineWidth, lineStyle, capStyle, joinStyle );

    if  ( lineStyle != LINEstyleSOLID )
	{
	if  ( ! dashList || dashCount == 0 )
	    { XLDEB(dashList,dashCount);	}
	else{
	    XSetDashes( add->addDisplay, add->addGc, 0,
						(char *)dashList, dashCount );
	    }
	}

    return 0;
    }

#   endif
