/************************************************************************/
/*									*/
/*  Ted: Management of the selection and the current position.		*/
/*									*/
/************************************************************************/

#   include	"tedConfig.h"

#   include	<stddef.h>
#   include	<stdlib.h>
#   include	<stdio.h>
#   include	<ctype.h>
#   include	<limits.h>

#   include	"docPageGrid.h"
#   include	"tedApp.h"
#   include	"docScreenLayout.h"
#   include	"tedRuler.h"
#   include	"tedLayout.h"

#   include	<appDebugon.h>

/************************************************************************/
/*									*/
/*  Is the selection exactly an object.					*/
/*									*/
/************************************************************************/

int tedGetObjectSelection(	TedDocument *			td,
				const BufferItem **		pBodySectBi,
				int *				pPart,
				DocumentPosition *		dpObject,
				InsertedObject **		pIo )
    {
    const BufferDocument *	bd= td->tdDocument;
    int				rval;
    BufferItem *		bodyBi= bd->bdBody.eiRoot;
    int				bodySectNr;

    if  ( td->tdBodySectionNumber < 0				||
	  td->tdBodySectionNumber >= bodyBi->biChildCount	)
	{ bodySectNr= 0;			}
    else{ bodySectNr= td->tdBodySectionNumber;	}

    rval= docGetObjectSelection( &(td->tdDocumentSelection), bd,
							pPart, dpObject, pIo );
    if  ( rval )
	{ return rval;	}

    *pBodySectBi= bodyBi->biChildren[bodySectNr];

    return 0;
    }


static void tedScrollToPositions(EditDocument *			ed,
				const PositionGeometry *	pgB,
				const PositionGeometry *	pgE,
				const LayoutContext *		lc,
				int *				pScrolledX,
				int *				pScrolledY )
    {
    DocumentRectangle		drPixels;

    docPixelRectangleForPositions( &drPixels, pgB, pgE, lc );

    appScrollToRectangle( ed,
			drPixels.drX0, drPixels.drY0,
			drPixels.drX1, drPixels.drY1,
			pScrolledX, pScrolledY );

    return;
    }

/************************************************************************/
/*									*/
/*  Extend a selection that begins outside a table cell and ends inside	*/
/*  that cell.								*/
/*									*/
/*  The speccial cases that occur when the selection begins in a table	*/
/*  are covered by tedExtendSelectionFromTable(), that has priority	*/
/*  over tedExtendSelectionIntoTable().					*/
/*									*/
/************************************************************************/

static int tedExtendSelectionIntoTable(	EditDocument *		ed,
					DocumentPosition *	dpEnd )
    {
    BufferItem *	cellBi= dpEnd->dpBi->biParent;
    BufferItem *	rowBi=	cellBi->biParent;

    int			col1= rowBi->biChildCount -1;

    docLastPosition( dpEnd, rowBi->biChildren[col1] );

    return 0;
    }

/************************************************************************/
/*									*/
/*  Extend a selection that begins inside a table cell and ends outside	*/
/*  that cell.								*/
/*									*/
/*  1)  Not in same section.						*/
/*  2)  Look for selection end in same table.				*/
/*  3)  Found?								*/
/*									*/
/************************************************************************/

static int tedExtendSelectionFromTable(	EditDocument *		ed,
					DocumentPosition *	dpBegin,
					DocumentPosition *	dpEnd,
					int *			pCol0,
					int *			pCol1 )
    {
    BufferItem *	cellBi= dpBegin->dpBi->biParent;
    BufferItem *	rowBi=	cellBi->biParent;
    BufferItem *	sectBi= rowBi->biParent;

    BufferItem *	endRowBi= dpEnd->dpBi->biParent->biParent;

    int			row1= rowBi->biNumberInParent;

    int			col0= -1;
    int			col1= -1;

    /*  1  */
    if  ( dpEnd->dpBi->biParent->biParent->biParent != sectBi )
	{
	docFirstPosition( dpBegin, rowBi->biChildren[0] );

	if  ( dpEnd->dpBi->biParaTableNesting > 0 )
	    { tedExtendSelectionIntoTable( ed, dpEnd ); }

	return 0;
	}

    /*  2  */
    while( row1 < rowBi->biRowTablePast- 1 )
	{
	if  ( endRowBi == sectBi->biChildren[row1] )
	    { break;	}

	row1++;
	}

    if  ( endRowBi == sectBi->biChildren[row1] )
	{
	if  ( dpEnd->dpBi->biParent->biNumberInParent <
					    cellBi->biNumberInParent )
	    {
	    col0= dpEnd->dpBi->biParent->biNumberInParent;
	    col1= dpBegin->dpBi->biParent->biNumberInParent;

	    docFirstPosition( dpBegin, rowBi->biChildren[col0] );
	    rowBi= dpEnd->dpBi->biParent->biParent;
	    docLastPosition( dpEnd, rowBi->biChildren[col1] );
	    }
	else{
	    docFirstPosition( dpBegin, dpBegin->dpBi->biParent );
	    docLastPosition( dpEnd, dpEnd->dpBi->biParent );
	    }

	col0= dpBegin->dpBi->biParent->biNumberInParent;
	col1= dpEnd->dpBi->biParent->biNumberInParent;
	}
    else{
	docFirstPosition( dpBegin, rowBi->biChildren[0] );

	if  ( dpEnd->dpBi->biParaTableNesting > 0 )
	    { tedExtendSelectionIntoTable( ed, dpEnd ); }

	return 0;
	}

    *pCol0= col0; *pCol1= col1; return 0;
    }

/************************************************************************/
/*									*/
/*  Extend the selection upon subsequent MotionNotify events.		*/
/*									*/
/*  1)  Do not completely describe the selection now. This is enough	*/
/*	for the redraws. [The mouse up handler will fully describe!]	*/
/*									*/
/************************************************************************/

static void tedSetExtendedSelection(
				EditDocument *			ed,
				int				exposeSelection,
				const DocumentRectangle *	drOldSel,
				int				col0,
				int				col1,
				int				direction,
				const BufferItem *		bodySectBi,
				const DocumentPosition *	dpExposeBegin,
				const DocumentPosition *	dpExposeEnd,
				const DocumentPosition *	dpTo,
				const DocumentPosition *	dpBegin,
				const DocumentPosition *	dpEnd )
    {
    TedDocument *		td= (TedDocument *)ed->edPrivateData;

    DocumentSelection		dsExpose;

    DocumentSelection		dsDoc;
    SelectionGeometry		sgDoc;
    SelectionDescription	sdDoc;

    PositionGeometry		pgTo;

    int				scrolledX= 0;
    int				scrolledY= 0;
    const int			lastLine= 0;

    LayoutContext		lc;

    layoutInitContext( &lc );
    tedSetLayoutContext( &lc, ed );

    docSetRangeSelection( &dsExpose, dpExposeBegin, dpExposeEnd, 1, -1, -1 );

    tedPositionGeometry( &pgTo, dpTo, bodySectBi, direction > 0, &lc );

    tedScrollToPositions( ed, &pgTo, &pgTo, &lc, &scrolledX, &scrolledY );

    if  ( exposeSelection )
	{
	appDocExposeRectangle( ed, drOldSel, scrolledX, scrolledY );
	}

    tedExposeSelection( ed, &dsExpose, bodySectBi, scrolledX, scrolledY );

    docSetRangeSelection( &(td->tdDocumentSelection), dpBegin, dpEnd,
						    direction, col0, col1 );
    tedSelectionGeometry( &(td->tdSelectionGeometry),
				    &(td->tdDocumentSelection), bodySectBi,
				    lastLine, &lc );

    /*  1  */
    td->tdSelectionDescription.sdIsIBarSelection=
			docIsIBarSelection( &(td->tdDocumentSelection) );

    if  ( exposeSelection )
	{
	if  ( tedGetSelection( &dsDoc, &sgDoc, &sdDoc,
			    (DocumentTree **)0, (const BufferItem **)0, td ) )
	    { LDEB(1); return;	}

	appDocExposeRectangle( ed, &(sgDoc.sgRectangle), scrolledX, scrolledY );
	}

    tedManagePrimarySelection( ed );

    return;
    }

static void tedBalanceFieldSelection(	BufferDocument *	bd,
					DocumentTree *		ei,
					int *			pBalanced,
					DocumentPosition *	dpBegin,
					DocumentPosition *	dpEnd )
    {
    DocumentField *		dfLeft;
    DocumentField *		dfRight;
    int				beginMoved= 0;
    int				endMoved= 0;
    int				headPart= -1;
    int				tailPart= -1;

    docBalanceFieldSelection( &dfLeft, &dfRight, &beginMoved, &endMoved,
						    &headPart, &tailPart,
						    dpBegin, dpEnd, ei, bd );

    if  ( beginMoved )
	{ *pBalanced= 1;	}
    if  ( endMoved )
	{ *pBalanced= 1;	}

    return;
    }

int tedExtendSelectionToPosition(
				EditDocument *			ed,
				const DocumentPosition *	dpAnchor,
				const DocumentPosition *	dpFound )
    {
    TedDocument *		td= (TedDocument *)ed->edPrivateData;

    int				col0= -1;
    int				col1= -1;

    DocumentPosition		dpFrom;
    DocumentPosition		dpTo;

    int				directionToAnchor;
    int				directionBeginAnchor;
    int				directionEndAnchor;
    int				dir;

    int				scrolledX= 0;
    int				scrolledY= 0;

    int				cellChanged= 0;
    int				balanced= 0;
    int				exposeSelection;

    DocumentSelection		dsDoc;
    SelectionGeometry		sgDoc;
    SelectionDescription	sdDoc;

    DocumentTree *		ei;
    const BufferItem *		bodySectBi;

    dpFrom= *dpAnchor;
    dpTo= *dpFound;

    if  ( tedGetSelection( &dsDoc, &sgDoc, &sdDoc, &ei, &bodySectBi, td ) )
	{ LDEB(1); return -1;	}

    directionToAnchor= docComparePositions( &dpTo, dpAnchor );
    directionBeginAnchor= docComparePositions( &(dsDoc.dsHead), dpAnchor );
    directionEndAnchor= docComparePositions( &(dsDoc.dsTail), dpAnchor );

    /********************/
    /*  Before anchor.	*/
    /********************/
    if  ( directionToAnchor < 0 )
	{
	int		directionToBegin;

	if  ( docPositionsInsideCell( &dpTo, &dpFrom ) )
	    {
	    if  ( dpFrom.dpBi->biParaTableNesting > 0 )
		{ col0= col1= dpFrom.dpBi->biParent->biNumberInParent;	}
	    }
	else{
	    if  ( dpTo.dpBi->biParaTableNesting > 0 )
		{
		if  ( tedExtendSelectionFromTable( ed, &dpTo, &dpFrom,
							    &col0, &col1 ) )
		    { LDEB(1); return -1;	}
		}
	    else{
		if  ( dpFrom.dpBi->biParaTableNesting > 0 )
		    {
		    if  ( tedExtendSelectionIntoTable( ed, &dpFrom ) )
			{ LDEB(1); return -1;	}
		    }
		}
	    }

	tedBalanceFieldSelection( td->tdDocument, ei, &balanced, &dpTo, &dpFrom );

	directionToBegin= docComparePositions( &dpTo, &(dsDoc.dsHead) );
	cellChanged= ! docPositionsInsideCell( &dpTo, &(dsDoc.dsHead) );

	/****************************************/
	/*  Undraw selection after the anchor.	*/
	/****************************************/
	if  ( directionEndAnchor > 0	||
	      balanced			||
	      cellChanged		)
	    {
	    appDocExposeRectangle( ed, &(sgDoc.sgRectangle),
						    scrolledX, scrolledY );
	    }

	/************************/
	/*  Extended Left.	*/
	/************************/
	if  ( directionToBegin < 0 )
	    {
	    dir= -1;
	    exposeSelection=	directionEndAnchor > 0	||
				balanced		||
				cellChanged		;

	    tedSetExtendedSelection( ed, exposeSelection, &(sgDoc.sgRectangle),
				col0, col1, dir, bodySectBi,
				&dpTo, &(dsDoc.dsHead),
				&dpTo, &dpTo, &dpFrom );

	    return 0;
	    }

	/************************/
	/*  Shrunk Left.	*/
	/************************/
	if  ( directionToBegin > 0 )
	    {
	    dir= -1;
	    exposeSelection=	directionEndAnchor > 0	||
				balanced		||
				cellChanged		;

	    tedSetExtendedSelection( ed, exposeSelection, &(sgDoc.sgRectangle),
				    col0, col1, dir, bodySectBi,
				    &(dsDoc.dsHead), &dpTo,
				    &dpTo, &dpTo, &dpFrom );

	    return 0;
	    }

	return 0;
	}

    /********************/
    /*  After anchor.	*/
    /********************/
    if  ( directionToAnchor > 0 )
	{
	int		directionToEnd;

	if  ( docPositionsInsideCell( &dpTo, &dpFrom ) )
	    {
	    if  ( dpFrom.dpBi->biParaTableNesting > 0 )
		{ col0= col1= dpFrom.dpBi->biParent->biNumberInParent;	}
	    }
	else{
	    if  ( dpFrom.dpBi->biParaTableNesting > 0 )
		{
		if  ( tedExtendSelectionFromTable( ed, &dpFrom, &dpTo,
							    &col0, &col1 ) )
		    { LDEB(1); return -1;	}
		}
	    else{
		if  ( dpTo.dpBi->biParaTableNesting > 0 )
		    {
		    if  ( tedExtendSelectionIntoTable( ed, &dpTo ) )
			{ LDEB(1); return -1;	}
		    }
		}
	    }

	tedBalanceFieldSelection( td->tdDocument, ei, &balanced, &dpFrom, &dpTo );

	directionToEnd= docComparePositions( &dpTo, &(dsDoc.dsTail) );
	cellChanged= ! docPositionsInsideCell( &dpTo, &(dsDoc.dsTail) );

	/****************************************/
	/*  Undraw selection before the anchor.	*/
	/****************************************/
	if  ( directionBeginAnchor < 0	||
	      balanced			||
	      cellChanged		)
	    {
	    appDocExposeRectangle( ed, &(sgDoc.sgRectangle),
						    scrolledX, scrolledY );
	    }

	/************************/
	/*  Extended Right.	*/
	/************************/
	if  ( directionToEnd > 0 )
	    {
	    dir= 1;
	    exposeSelection=	directionBeginAnchor < 0	||
				balanced			||
				cellChanged			;

	    tedSetExtendedSelection( ed, exposeSelection, &(sgDoc.sgRectangle),
				    col0, col1, dir, bodySectBi,
				    &(dsDoc.dsTail), &dpTo,
				    &dpTo, &dpFrom, &dpTo );

	    return 0;
	    }

	/************************/
	/*  Shrunk Right.	*/
	/************************/
	if  ( directionToEnd < 0 )
	    {
	    dir= 1;
	    exposeSelection=	directionBeginAnchor < 0	||
				balanced			||
				cellChanged			;

	    tedSetExtendedSelection( ed, exposeSelection, &(sgDoc.sgRectangle),
				    col0, col1, dir, bodySectBi,
				    &dpTo, &(dsDoc.dsTail),
				    &dpTo, &dpFrom, &dpTo );

	    return 0;
	    }

	return 0;
	}

    /********************/
    /*  At anchor.	*/
    /********************/
	{
	dpFrom= *dpAnchor;
	dpTo= *dpAnchor;

	tedBalanceFieldSelection( td->tdDocument, ei, &balanced, &dpFrom, &dpTo );

	/****************************************/
	/*  Undraw selection before the anchor.	*/
	/*  Undraw selection after the anchor.	*/
	/****************************************/
	if  ( directionBeginAnchor < 0	||
	      directionEndAnchor > 0	)
	    {
	    appDocExposeRectangle( ed, &(sgDoc.sgRectangle),
						    scrolledX, scrolledY );
	    }

	dir= 0;
	exposeSelection=	directionBeginAnchor < 0	||
				directionEndAnchor > 0		||
				balanced			;

	tedSetExtendedSelection( ed, exposeSelection, &(sgDoc.sgRectangle),
				col0, col1, dir, bodySectBi,
				&dpFrom, &dpTo,
				&dpFrom, &dpFrom, &dpTo );
	}

    return 0;
    }

/************************************************************************/
/*									*/
/*  Move to the next/previous position.					*/
/*									*/
/************************************************************************/

static int tedArrowFindPositionInLine(	DocumentPosition *	dp,
					const LayoutContext *	lc,
					int			line,
					int			docXPixels )
    {
    const BufferItem *	bodySectBi;
    PositionGeometry	pg;

    docInitPositionGeometry( &pg );

    bodySectBi= docGetBodySectBi( dp->dpBi, lc->lcDocument );
    if  ( ! bodySectBi )
	{ XDEB(bodySectBi); return -1;	}

    if  ( tedFindPositionInLine( dp, &pg, lc,
				dp->dpBi, line, bodySectBi, docXPixels ) )
	{ LDEB(docXPixels); return -1;	}

    return 0;
    }

int tedArrowUp(		DocumentPosition *		dp,
			const PositionGeometry *	pg,
			const LayoutContext *		lc )
    {
    TextLine *		tl;
    int			line;

    if  ( docLineUp( &tl, &line, dp, pg->pgLine ) )
	{ return -1;	}

    if  ( tedArrowFindPositionInLine( dp, lc, line, pg->pgXPixels ) )
	{ LDEB(pg->pgXPixels); return -1;	}

    return 0;
    }

int tedArrowDown(	DocumentPosition *		dp,
			const PositionGeometry *	pg,
			const LayoutContext *		lc )
    {
    TextLine *		tl;
    int			line;

    if  ( docLineDown( &tl, &line, dp, pg->pgLine ) )
	{ return -1;	}

    if  ( tedArrowFindPositionInLine( dp, lc, line, pg->pgXPixels ) )
	{ LDEB(pg->pgXPixels); return -1;	}

    return 0;
    }

/************************************************************************/
/*									*/
/*  Scroll a selection into view.					*/
/*									*/
/************************************************************************/

void tedScrollToSelection(	EditDocument *		ed,
				int *			pScrolledX,
				int *			pScrolledY )
    {
    TedDocument *		td= (TedDocument *)ed->edPrivateData;

    DocumentSelection		dsDoc;
    SelectionGeometry		sgDoc;
    SelectionDescription	sdDoc;

    LayoutContext		lc;

    int				scrolledX= 0;
    int				scrolledY= 0;

    layoutInitContext( &lc );
    tedSetLayoutContext( &lc, ed );

    if  ( tedGetSelection( &dsDoc, &sgDoc, &sdDoc,
			    (DocumentTree **)0, (const BufferItem **)0, td ) )
	{ LDEB(1); return;	}

    tedScrollToPositions( ed, &(sgDoc.sgBegin), &(sgDoc.sgEnd),
						&lc, &scrolledX, &scrolledY );
    return;
    }

/************************************************************************/
/*									*/
/*  A selection was made, do bookkeeping and provoke drawing.		*/
/*									*/
/*  0)  If an object (Picture) was selected, remove the special window.	*/
/*  1)  Provoke the old selection to be redrawn. (Before scrolling)	*/
/*  2)  Stop Cursor blinking.						*/
/*  3)  Scroll to the new selection.					*/
/*  4)  Provoke the old selection to be redrawn. (It might have moved)	*/
/*  5)  Bookkeeping.							*/
/*  6)  Provoke the new selection to be redrawn.			*/
/*									*/
/*  NOTE:	As scrolling to the new selection can provoke a redraw	*/
/*		the selection needs to be set before the scrollbar is	*/
/*		adapted. Otherwise the selection administration can	*/
/*		still refer to deleted text.				*/
/*									*/
/*  A)  If the selection is partially inside and partially outside a	*/
/*	text level field, force it to contain the whole field. This is	*/
/*	to enforce proper text field nesting.				*/
/*									*/
/************************************************************************/

void tedSetSelection(	EditDocument *			ed,
			const DocumentSelection *	dsSet,
			int				lastLine,
			int *				pScrolledX,
			int *				pScrolledY )
    {
    TedDocument *		td= (TedDocument *)ed->edPrivateData;
    BufferDocument *		bd= td->tdDocument;
    AppDrawingData *		add= &(ed->edDocumentWidget.dwDrawingData);
    int				hadSelection;

    DocumentSelection		dsOld;
    SelectionGeometry		sgOld;
    SelectionDescription	sdOld;

    DocumentSelection		dsNew;
    SelectionGeometry		sgNew;

    int				balanced= 0;

    DocumentRectangle		drExpose;

    int				redrawExternalItemOld= 0;

    DocumentTree *		eiSet= (DocumentTree *)0;
    BufferItem *		bodySectBiSet= (BufferItem *)0;

    BufferItem *		rootBiSet;
    DocumentRectangle		drExternalSet;
    int				redrawExternalItemSet= 0;

    int				sameRoot;

    const int			justUsed= 0;
    int				bodySectNr= -1;

    LayoutContext		lc;

    layoutInitContext( &lc );
    tedSetLayoutContext( &lc, ed );

    if  ( tedGetSelection( &dsOld, &sgOld, &sdOld,
			    (DocumentTree **)0, (const BufferItem **)0, td ) )
	{ LDEB(1); return;	}

    /*  0  */
    if  ( sdOld.sdIsObjectSelection )
	{ tedHideObjectWindows( ed );	}

    rootBiSet= docGetSelectionRoot( &eiSet, &bodySectBiSet, bd, dsSet );
    if  ( ! rootBiSet )
	{ XDEB(rootBiSet); return;	}

    sameRoot= docSelectionSameRoot( &dsOld, rootBiSet );

    /******/
    switch( dsOld.dsSelectionScope.ssInExternalItem )
	{
	case DOCinBODY:
	    break;

	case DOCinFIRST_HEADER:
	case DOCinLEFT_HEADER:
	case DOCinRIGHT_HEADER:

	case DOCinFIRST_FOOTER:
	case DOCinLEFT_FOOTER:
	case DOCinRIGHT_FOOTER:
	    if  ( ! sameRoot )
		{ redrawExternalItemOld= 1;	}
	    break;

	case DOCinFOOTNOTE:
	case DOCinENDNOTE:
	    if  ( ! sameRoot )
		{ redrawExternalItemOld= 1;	}
	    break;

	case DOCinFTNSEP:
	case DOCinFTNSEPC:
	case DOCinFTNCN:
	case DOCinAFTNSEP:
	case DOCinAFTNSEPC:
	case DOCinAFTNCN:
	    if  ( ! sameRoot )
		{ redrawExternalItemOld= 1;	}
	    break;

	default:
	    LDEB(dsOld.dsSelectionScope.ssInExternalItem);
	    break;
	}

    /******/
    switch( rootBiSet->biInExternalItem )
	{
	case DOCinBODY:
	    bodySectBiSet= docGetSectItem( dsSet->dsHead.dpBi );
	    if  ( ! bodySectBiSet )
		{ XDEB(bodySectBiSet); return;	}

	    bodySectNr= bodySectBiSet->biNumberInParent;
	    break;

	case DOCinFIRST_HEADER:
	case DOCinLEFT_HEADER:
	case DOCinRIGHT_HEADER:

	case DOCinFIRST_FOOTER:
	case DOCinLEFT_FOOTER:
	case DOCinRIGHT_FOOTER:

	    {
	    const int		page= eiSet->eiPageSelectedUpon;
	    const int		column= 0; /* irrelevant */
	    int			changed= 0;
	    DocumentTree *	ei= (DocumentTree *)0;

	    if  ( ! sameRoot )
		{ redrawExternalItemSet= 1;	}

	    if  ( docGetHeaderFooter( &ei, &bodySectBiSet, dsSet, bd,
					    rootBiSet->biInExternalItem ) )
		{ LDEB(rootBiSet->biInExternalItem);	}

	    /*
	    page= docSectionHeaderFooterFirstPage( &usedInDoc, bodySectBiSet,
					rootBiSet->biInExternalItem, dp );

	    if  ( page < 0 || ! usedInDoc )
		{ LLDEB(page,usedInDoc); return;	}
	    */

	    if  ( docCheckPageOfSelectedExtItem( &changed, &bodySectBiSet,
				&drExternalSet, eiSet, &lc,
				docInitScreenLayoutExternalItem ) )
		{ LDEB(1);	}

	    if  ( changed )
		{ redrawExternalItemSet= 1;	}

	    if  ( redrawExternalItemSet )
		{
		if  ( docGetExternalItemBox( &drExternalSet, bodySectBiSet,
				    eiSet, justUsed, page, column, &lc ) )
		    {
		    LDEB(rootBiSet->biInExternalItem);
		    redrawExternalItemSet= 0;
		    }
		}

	    bodySectNr= bodySectBiSet->biNumberInParent;
	    break;
	    }

	case DOCinFOOTNOTE:
	case DOCinENDNOTE:
	    if  ( ! sameRoot )
		{
		DocumentTree *		ei= (DocumentTree *)0;
		const int		page= eiSet->eiPageFormattedFor;
		const int		column= eiSet->eiColumnFormattedFor;

		redrawExternalItemSet= 1;

		if  ( docGetRootForItem( &ei, &bodySectBiSet, bd, rootBiSet ) )
		    { LDEB(1); return;	}

		if  ( docGetExternalItemBox( &drExternalSet, bodySectBiSet,
				    eiSet, justUsed, page, column, &lc ) )
		    {
		    LSDEB(rootBiSet->biInExternalItem,
			docExternalKindStr(rootBiSet->biInExternalItem));
		    redrawExternalItemSet= 0;
		    }

		bodySectNr= bodySectBiSet->biNumberInParent;
		}
	    break;

	case DOCinFTNSEP:
	case DOCinFTNSEPC:
	case DOCinFTNCN:
	case DOCinAFTNSEP:
	case DOCinAFTNSEPC:
	case DOCinAFTNCN:
	    {
	    const int		page= eiSet->eiPageSelectedUpon;
	    const int		column= eiSet->eiColumnSelectedIn;
	    int			changed= 0;

	    if  ( ! sameRoot )
		{ redrawExternalItemSet= 1;	}

	    if  ( docCheckPageOfSelectedExtItem( &changed, &bodySectBiSet,
				&drExternalSet, eiSet, &lc,
				docInitScreenLayoutExternalItem ) )
		{ LDEB(1);	}

	    if  ( changed )
		{ redrawExternalItemSet= 1;	}

	    if  ( docGetExternalItemBox( &drExternalSet, bodySectBiSet,
				    eiSet, justUsed, page, column, &lc ) )
		{
		LDEB(rootBiSet->biInExternalItem);
		redrawExternalItemSet= 0;
		}

	    bodySectNr= bodySectBiSet->biNumberInParent;
	    }
	    break;

	default:
	    LDEB(rootBiSet->biInExternalItem);
	    break;
	}

    /*  A  */
    dsNew= *dsSet;
    tedBalanceFieldSelection( td->tdDocument, eiSet, &balanced,
					    &(dsNew.dsHead), &(dsNew.dsTail) );

    hadSelection= tedHasSelection( td );
    if  ( hadSelection )
	{
	/*  1  */
	drExpose= sgOld.sgRectangle;

	if  ( redrawExternalItemOld )
	    { geoUnionRectangle( &drExpose, &drExpose, &(add->addBackRect) ); }

	/*  2  */
	if  ( sdOld.sdIsIBarSelection )
	    { tedStopCursorBlink( ed ); }

	appDocExposeRectangle( ed, &drExpose, *pScrolledX, *pScrolledY );
	}

    /*  5  */
    if  ( dsNew.dsHead.dpBi->biParaTableNesting > 0		&&
	  dsNew.dsTail.dpBi->biParaTableNesting > 0		&&
	  dsNew.dsHead.dpBi->biParent->biParent ==
			dsNew.dsTail.dpBi->biParent->biParent	)
	{
	dsNew.dsCol0= dsNew.dsHead.dpBi->biParent->biNumberInParent;
	dsNew.dsCol1= dsNew.dsTail.dpBi->biParent->biNumberInParent;
	}

    /*  5  */
    if  ( ! bodySectBiSet )
	{ XDEB(bodySectBiSet);	}

    tedSelectionGeometry( &sgNew, &dsNew, bodySectBiSet, lastLine, &lc );
    td->tdVisibleSelectionCopied= 0;
    td->tdDocumentSelection= dsNew;
    td->tdSelectionGeometry= sgNew;

    /*  3  */
    tedScrollToSelection( ed, pScrolledX, pScrolledY );

    if  ( hadSelection )
	{
	/*  4  */
	drExpose= sgOld.sgRectangle;

	if  ( redrawExternalItemOld )
	    { geoUnionRectangle( &drExpose, &drExpose, &(add->addBackRect) ); }

	appDocExposeRectangle( ed, &drExpose, *pScrolledX, *pScrolledY );
	}

    /*  6  */
    drExpose= sgNew.sgRectangle;

    if  ( redrawExternalItemSet )
	{ geoUnionRectangle( &drExpose, &drExpose, &drExternalSet ); }

    appDocExposeRectangle( ed, &drExpose, *pScrolledX, *pScrolledY );

    tedDescribeSelection( ed );

    if  ( td->tdSelectionDescription.sdIsObjectSelection )
	{ tedMoveObjectWindows( ed );	}

    td->tdBodySectionNumber= bodySectNr;

    tedManagePrimarySelection( ed );

    return;
    }

void tedDescribeSelection(	EditDocument *			ed )
    {
    TedDocument *		td= (TedDocument *)ed->edPrivateData;
    BufferDocument *		bd= td->tdDocument;

    tedSetCurrentTextAttribute( td, tedHasIBarSelection( td ),
					&(td->tdDocumentSelection.dsHead) );

    docDescribeSelection( &(td->tdSelectionDescription),
			&(td->tdDocumentSelection), &(td->tdSelectionGeometry),
			bd, ed->edDocumentId, ed->edIsReadonly );
    }

void tedSetSelectedPosition(	EditDocument *			ed,
				const DocumentPosition *	dpSet,
				int				lastLine,
				int *				pScrolledX,
				int *				pScrolledY )
    {
    DocumentSelection		dsNew;

    docInitDocumentSelection( &dsNew );

    docSetIBarSelection( &dsNew, dpSet );

    tedSetSelection( ed, &dsNew, lastLine, pScrolledX, pScrolledY );

    return;
    }

int tedSelectItemHome(		EditDocument *			ed,
				BufferItem *			rootBi,
				int *				pScrolledX,
				int *				pScrolledY )
    {
    TedDocument *	td= (TedDocument *)ed->edPrivateData;
    BufferDocument *	bd= td->tdDocument;

    DocumentPosition	dp;
    const int		lastLine= 0;

    if  ( docFirstPosition( &dp, rootBi ) )
	{ LDEB(1); return -1;	}

    docAvoidParaHeadField( &dp, (int *)0, bd );
    tedSetSelectedPosition( ed, &dp, lastLine, pScrolledX, pScrolledY );

    return 0;
    }

/************************************************************************/
/*									*/
/*  Set an I Bar selection and cause the I Bar to be drawn.		*/
/*									*/
/************************************************************************/

int tedSetIBarSelection(		EditDocument *		ed,
					const DocumentPosition * dp,
					int			lastLine,
					int *			pScrolledX,
					int *			pScrolledY )
    {
    TedDocument *	td= (TedDocument *)ed->edPrivateData;

    td->tdVisibleSelectionCopied= 0;

    if  ( td->tdSelectionDescription.sdIsObjectSelection )
	{ tedHideObjectWindows( ed );	}

    tedSetSelectedPosition( ed, dp, lastLine, pScrolledX, pScrolledY );

    tedAdaptToolsToSelection( ed );

    return 0;
    }

/************************************************************************/
/*									*/
/*  Change the selection to cover whole paragraphs.			*/
/*									*/
/*  Depending on the 'direction' argument:				*/
/*	<  0:	The previous one.					*/
/*	== 0:	The current one.					*/
/*	>  0:	The next one.						*/
/*									*/
/************************************************************************/

int tedSelectWholeParagraph(	EditApplication *	ea,
				int			direction )
    {
    EditDocument *		ed= ea->eaCurrentDocument;
    TedDocument *		td= (TedDocument *)ed->edPrivateData;
    BufferItem *		bi;

    DocumentSelection		dsNew;
    SelectionGeometry		sg;
    SelectionDescription	sd;

    int				scrolledX= 0;
    int				scrolledY= 0;

    const int			lastLine= 0;

    if  ( tedGetSelection( &dsNew, &sg, &sd,
			    (DocumentTree **)0, (const BufferItem **)0, td ) )
	{ LDEB(1); return -1;	}

    bi= dsNew.dsHead.dpBi;
    if  ( ! bi )
	{ XDEB(bi); return -1;	}

    if  ( direction > 0 )
	{ bi= docNextParagraph( bi );	}
    if  ( direction < 0 )
	{ bi= docPrevParagraph( bi );	}

    if  ( ! bi )
	{ return -1;	}

    if  ( direction == 0 )
	{ direction= 1;	}

    docInitDocumentSelection( &dsNew );

    docSetParaSelection( &dsNew, bi, direction, 0, docParaStrlen( bi ) );

    tedSetSelection( ed, &dsNew, lastLine, &scrolledX, &scrolledY );

    tedAdaptToolsToSelection( ed );

    return 0;
    }

/************************************************************************/
/*									*/
/*  Change the selection to cover whole sections.			*/
/*									*/
/*  Depending on the 'direction' argument:				*/
/*	<  0:	The previous one.					*/
/*	== 0:	The current one.					*/
/*	>  0:	The next one.						*/
/*									*/
/************************************************************************/

int tedSelectWholeSection(	EditApplication *	ea,
				int			direction )
    {
    EditDocument *		ed= ea->eaCurrentDocument;
    TedDocument *		td= (TedDocument *)ed->edPrivateData;
    BufferDocument *		bd= td->tdDocument;

    DocumentSelection		dsNew;
    SelectionGeometry		sg;
    SelectionDescription	sd;

    int				ret;

    int				scrolledX= 0;
    int				scrolledY= 0;

    const int			lastLine= 0;

    if  ( tedGetSelection( &dsNew, &sg, &sd,
			    (DocumentTree **)0, (const BufferItem **)0, td ) )
	{ LDEB(1); return -1;	}

    ret= docSelectWholeSection( &dsNew, bd, direction );
    if  ( ret < 0 )
	{ LDEB(ret); return -1;	}
    if  ( ret > 0 )
	{ return -1;	}

    tedSetSelection( ed, &dsNew, lastLine, &scrolledX, &scrolledY );

    tedAdaptToolsToSelection( ed );

    return 0;
    }

