/************************************************************************/
/*									*/
/*  Ted: Field,Bookmark,Hyperlink related editing functionality.	*/
/*									*/
/************************************************************************/

#   include	"tedConfig.h"

#   include	<stddef.h>
#   include	<stdlib.h>
#   include	<stdio.h>
#   include	<string.h>

#   include	"tedEdit.h"
#   include	"tedLayout.h"
#   include	"docScreenLayout.h"
#   include	"docEvalField.h"

#   include	<appDebugon.h>

/************************************************************************/
/*									*/
/*  Determine the area covered by a field or a bookmark.		*/
/*									*/
/************************************************************************/

static void tedIncludeFieldInRedraw(
				TedEditOperation *		teo,
				const DocumentSelection *	dsField,
				int				headPart,
				int				tailPart )
    {
    EditOperation *		eo= &(teo->teoEo);

    SelectionGeometry		sgNew;

    const int			lastLine= 1;

    tedSelectionGeometry( &sgNew, dsField, eo->eoBodySectBi, lastLine,
						    &(teo->teoLayoutContext) );

    tedIncludeRectangleInChange( teo, &(sgNew.sgRectangle) );

    return;
    }

/************************************************************************/
/*									*/
/*  Convert the current selection to a field or a bookmark.		*/
/*									*/
/************************************************************************/

int tedLayoutFieldItem(		TedEditOperation *	teo,
				DocumentSelection *	dsField,
				unsigned int		whenMask )
    {
    const LayoutContext *	lc= &(teo->teoLayoutContext);
    EditDocument *		ed= teo->teoEditDocument;
    TedDocument *		td= (TedDocument *)ed->edPrivateData;
    BufferDocument *		bd= td->tdDocument;

    BufferItem *		bi;
    DocumentTree *		ei;
    BufferItem *		bodySectBi;

    bi= docGetSelectionRoot( &ei, &bodySectBi, bd, dsField );
    if  ( ! bi )
	{ XDEB(bi); return -1;	}

    if  ( whenMask )
	{
	RecalculateFields		rf;

	docInitRecalculateFields( &rf );

	rf.rfBd= bd;
	rf.rfEi= ei;
	rf.rfCloseObject= lc->lcCloseObject;
	rf.rfUpdateFlags= whenMask;
	rf.rfFieldsUpdated= 0;

	if  ( docRecalculateTextLevelFields( &rf, bi ) )
	    { XDEB(whenMask);	}
	}

    docEditIncludeItemInReformatRange( &(teo->teoEo), bi );

    return 0;
    }

int tedFinishSetField(		DocumentSelection *		dsField,
				TedEditOperation *		teo,
				unsigned int			whenMask )
    {
    if  ( tedLayoutFieldItem( teo, dsField, whenMask ) )
	{ LDEB(1); return -1;	}

    tedEditFinishSelection( teo, dsField );

    tedAdaptToolsToSelection( teo->teoEditDocument );

    appDocumentChanged( teo->teoEditDocument, 1 );

    return 0;
    }

/************************************************************************/
/*									*/
/*  Remove the particules that delimit a link or bookmark and adjust	*/
/*  the selection to changed particule numbers.				*/
/*									*/
/*  0)  First clean the field.						*/
/*									*/
/*  1)  Determine the rectangle that is to be redrawn.			*/
/*  2)  Delete the particules from the paragraph.			*/
/*  3)  Adjust the selection.						*/
/*  4)  Redraw.								*/
/*									*/
/************************************************************************/

void tedFlattenField(		EditDocument *			ed,
				const DocumentSelection *	dsAroundField,
				int				headPart,
				int				tailPart,
				DocumentField *			df )
    {
    TedDocument *		td= (TedDocument *)ed->edPrivateData;
    BufferDocument *		bd= td->tdDocument;

    SelectionGeometry		sg;
    SelectionDescription	sd;

    TedEditOperation		teo;

    tedStartEditOperation( &teo, &sg, &sd, ed, 0 );

    /*  1  */
    tedIncludeFieldInRedraw( &teo, dsAroundField, headPart, tailPart );

    /*  0,2  */
    docDeleteField( dsAroundField->dsHead.dpBi, dsAroundField->dsTail.dpBi,
				&headPart, &tailPart, headPart, tailPart,
				teo.teoEo.eoEi, bd, df );

    docEditIncludeItemInReformatRange( &(teo.teoEo),
					    dsAroundField->dsHead.dpBi );
    docEditIncludeItemInReformatRange( &(teo.teoEo),
					    dsAroundField->dsTail.dpBi );

    /*  3,4  */
    if  ( tedEditFinishOldSelection( &teo ) )
	{ LDEB(1);	}

    appDocumentChanged( ed, 1 );

    return;
    }

/************************************************************************/
/*									*/
/*  Replace the current selection of the document with a field.		*/
/*									*/
/*  2)  Replace the current selection with something arbitrary.		*/
/*  3)  Make the footnote number field.					*/
/*  4)  Finish by surrounding the replacement with the field.		*/
/*  5)  Final bookkeeping and redrawing.				*/
/*									*/
/************************************************************************/

int tedDocReplaceSelectionWithField(	DocumentField **	pDf,
					int *			pHeadPart,
					int *			pTailPart,
					DocumentSelection *	dsField,
					TedEditOperation *	teo,
					const MemoryBuffer *	mbFieldInst,
					int			fieldKind,
					const PropertyMask *	taSetMask,
					const TextAttribute *	taSet )
    {
    EditOperation *		eo= &(teo->teoEo);
    EditDocument *		ed= teo->teoEditDocument;
    TedDocument *		td= (TedDocument *)ed->edPrivateData;
    BufferDocument *		bd= td->tdDocument;

    DocumentField *		df;
    DocumentSelection		dsInput;
    DocumentSelection		dsInside;
    DocumentSelection		dsAround;
    DocumentSelection		dsRep;

    const int			singlePara= 1;

    /*  2  */
    dsInput.dsHead= eo->eoHeadDp;

    if  ( tedEditReplaceSelection( teo, (const char *)"?", 1 ) )
	{ LDEB(1); return -1;	}

    dsInput.dsTail= eo->eoTailDp;

    docSetRangeSelection( &dsRep, &(dsInput.dsHead), &(dsInput.dsTail),
								1, -1, -1 );

    /*  3,4  */
    if  ( docSurroundTextSelectionByField( &df, &dsInside, &dsAround,
						pHeadPart, pTailPart,
						bd, teo->teoEo.eoEi,
						singlePara, &dsRep,
						taSetMask, taSet ) )
	{ LDEB(1); return -1;	}

    if  ( utilCopyMemoryBuffer( &(df->dfInstructions), mbFieldInst ) )
	{ LDEB(mbFieldInst->mbSize); return -1;	}

    *dsField= dsRep;
    df->dfKind= fieldKind;
    *pDf= df;

    return 0;
    }

/************************************************************************/
/*									*/
/*  Insert a page number in the document.				*/
/*									*/
/*  1)  Force the header or footer that will receive the page number	*/
/*	to be reformatted the next time that it is drawn.		*/
/*  2)  Replace the selection with a PAGE field. As the replecement is	*/
/*	a page number that is recalculated every time the		*/
/*	header/footer is drawn, no field calculation is required.	*/
/*									*/
/************************************************************************/

void tedInsertPageNumber(	EditDocument *	ed )
    {
    MemoryBuffer		mbFieldInst;
    DocumentSelection		dsField;

    TedEditOperation		teo;
    EditOperation *		eo= &(teo.teoEo);

    SelectionGeometry		sg;
    SelectionDescription	sd;

    TextAttribute		taSet;
    PropertyMask		taSetMask;

    int				headPart;
    int				tailPart;

    DocumentField *		df;

    utilInitMemoryBuffer( &mbFieldInst );

    if  ( utilSetMemoryBuffer( &mbFieldInst, (unsigned char *)"PAGE", 4 ) )
	{ LDEB(4); goto ready;	}

    utilInitTextAttribute( &taSet );
    utilPropMaskClear( &taSetMask );

    tedStartEditOperation( &teo, &sg, &sd, ed, 0 );

    /*  1  */
    eo->eoEi->eiPageFormattedFor= -1;
    eo->eoEi->eiColumnFormattedFor= -1;

    /*  2  */
    if  ( tedDocReplaceSelectionWithField( &df,
					&headPart, &tailPart, &dsField,
					&teo, &mbFieldInst, DOCfkPAGE,
					&taSetMask, &taSet ) )
	{ LDEB(4); goto ready;	}

    /*  5  */
    if  ( tedFinishSetField( &dsField, &teo, FIELDdoNOTHING ) )
	{ LDEB(1); goto ready;	}

  ready:
    utilCleanMemoryBuffer( &mbFieldInst );

    return;
    }

/************************************************************************/
/*									*/
/*  Remove the bookmark that holds the head of the selection.		*/
/*  The contents of the bookmark are retained: Only the field is	*/
/*  removed.								*/
/*									*/
/************************************************************************/

int tedDocFlattenTypedField(	EditDocument *		ed,
				int			fieldType )
    {
    TedDocument *		td= (TedDocument *)ed->edPrivateData;
    BufferDocument *		bd= td->tdDocument;

    DocumentField *		df;
    DocumentSelection		dsInsideField;
    DocumentSelection		dsAroundField;
    int				headPart;
    int				tailPart;

    DocumentSelection		ds;
    SelectionGeometry		sg;
    SelectionDescription	sd;

    tedGetSelection( &ds, &sg, &sd,
			    (DocumentTree **)0, (const BufferItem **)0, td );

    df= docFindTypedFieldForPosition( bd, &(ds.dsHead), fieldType );
    if  ( ! df )
	{ XDEB(df); return -1;	}
    if  ( docDelimitFieldInDoc( &dsInsideField, &dsAroundField,
					    &headPart, &tailPart, bd, df ) )
	{ LDEB(1); return -1; }

    tedFlattenField( ed, &dsAroundField, headPart, tailPart, df );

    return 0;
    }

