/************************************************************************/
/*									*/
/*  Editor functionality.						*/
/*									*/
/************************************************************************/

#   include	"tedConfig.h"

#   include	<stddef.h>
#   include	<stdlib.h>
#   include	<stdio.h>
#   include	<ctype.h>

#   include	"tedEdit.h"
#   include	"tedLayout.h"

#   include	<appDebugon.h>

/************************************************************************/
/*									*/
/*  Replace the selection in the document with new text.		*/
/*									*/
/*  b)  Replace the tail of the beginning paragraph with the new text.	*/
/*  c)  Merge the two paragraphs.					*/
/*  d)  Erase all paragraphs after the beginning of the selection	*/
/*	to, including the end.						*/
/*  e)  Update the paragraph buffer and the particule administration.	*/
/*									*/
/*  B)  The first particule of the line was split, also reformat the	*/
/*	previous line.							*/
/*	If paragraphs were merged, redraw the whole neighbourhood.	*/
/*  C)  - Recalculate the geometry from the start of the selection to	*/
/*	  the end of the paragraph.					*/
/*	- Redraw the affected part of the paragraph.			*/
/*  6)  Update the notion of current particule and current line.	*/
/*  7)  Redraw the I-Bar.						*/
/*									*/
/************************************************************************/

int tedEditReplaceSelection(	TedEditOperation *	teo,
				const char *		addedText,
				int			addedLength )
    {
    EditDocument *		ed= teo->teoEditDocument;
    EditOperation *		eo= &(teo->teoEo);
    BufferDocument *		bd= eo->eoBd;
    TedDocument *		td= (TedDocument *)ed->edPrivateData;

    /*  b,c,d,e  */
    if  ( docReplaceSelection( eo, addedText, addedLength,
				    td->tdCurrentTextAttributeNumber ) )
	{ LDEB(addedLength); return -1;	}

    if  ( addedLength > 0 )
	{ docRememberFontUsed( bd, &(td->tdCurrentTextAttribute) );	}

    return 0;
    }

int tedEditDeleteSelection(	TedEditOperation *	teo )
    {
    EditOperation *		eo= &(teo->teoEo);

    /*  b,c,d,e  */
    if  ( docDeleteSelection( eo ) )
	{ LDEB(1); return -1;	}

    return 0;
    }

/************************************************************************/
/*									*/
/*  Merge the selection into one paragraph.				*/
/*									*/
/************************************************************************/

void tedMergeParagraphsInSelection(	EditDocument *	ed )
    {
    int				beforeEnd;

    TedEditOperation		teo;
    EditOperation *		eo= &(teo.teoEo);
    SelectionGeometry		sg;
    SelectionDescription	sd;

    tedStartEditOperation( &teo, &sg, &sd, ed, 1 );

    beforeEnd= docParaStrlen( eo->eoTailDp.dpBi )-
					eo->eoSelectedRange.erTail.epStroff;

    if  ( docMergeParagraphsInSelection( eo ) )
	{ LDEB(1); goto ready;	}

    if  ( ed->edFormat == TEDdockindTEXT_SAVE_FOLDED )
	{ ed->edFormat= TEDdockindTEXT_SAVE_WIDE;	}

    eo->eoSelectedRange.erTail.epParaNr= eo->eoSelectedRange.erHead.epParaNr;
    eo->eoSelectedRange.erTail.epStroff=
				docParaStrlen( eo->eoHeadDp.dpBi )- beforeEnd;

    if  ( tedEditFinishOldSelection( &teo ) )
	{ LDEB(1);	}

  ready:

    return;
    }

/************************************************************************/
/*									*/
/*  Split a paragraph.							*/
/*									*/
/*  1)  Split in the buffer administration.				*/
/*  2)  Recalculate the geometry from the start of the selection to the	*/
/*	end of the paragraph.						*/
/*  3)  Redraw the affected part of the paragraph.			*/
/*  4)  Cause the new paragraph to be reformatted in this edit		*/
/*	operation.							*/
/*  5)  If the new paragraph is part of a numbered/bulleted list,	*/
/*	insert a list text field at the head.				*/
/*									*/
/************************************************************************/

int tedSplitParaItem(		BufferItem **			pNewParaBi,
				TedEditOperation *		teo,
				int				splitLevel,
				int				onNewPage )
    {
    BufferItem *		newParaBi;
    const LayoutContext *	lc= &(teo->teoLayoutContext);

    /*  1  */
    if  ( docSplitParaItem( &newParaBi, &(teo->teoEo), splitLevel ) )
	{ LDEB(1); return -1;	}

    newParaBi->biParaBreakKind= DOCibkNONE;
    if  ( newParaBi->biParaTableNesting == 0 && onNewPage )
	{ newParaBi->biParaBreakKind= DOCibkPAGE;	}
    
    /*  3  */
    teo->teoChangedRect.drX0= lc->lcAdd->addBackRect.drX0;

    *pNewParaBi= newParaBi;
    return 0;
    }

void tedSplitParagraph(		EditDocument *		ed,
				int			onNewPage )
    {
    TedEditOperation		teo;
    SelectionGeometry		sg;
    SelectionDescription	sd;

    BufferItem *		newParaBi;
    DocumentPosition		dpNew;

    tedStartEditOperation( &teo, &sg, &sd, ed, 1 );

    if  ( tedEditDeleteSelection( &teo ) )
	{ return;	}

    if  ( tedSplitParaItem( &newParaBi, &teo, DOClevPARA, onNewPage ) )
	{ LDEB(1); return;	}

    if  ( docFirstPosition( &dpNew, newParaBi ) )
	{ LDEB(1); return;	}

    docAvoidParaHeadField( &dpNew, (int *)0, teo.teoEo.eoBd );
    tedEditFinishIBarSelection( &teo, dpNew.dpBi, dpNew.dpStroff );

    return;
    }

/************************************************************************/
/*									*/
/*  Insert an 'Object'.							*/
/*									*/
/************************************************************************/

static int tedReplaceSelectionWithObject(
					TedEditOperation *	teo,
					int			objectNr,
					InsertedObject *	io )
    {
    EditDocument *		ed= teo->teoEditDocument;
    TextParticule *		tp;

    int				stroffShift;
    int				partShift;

    EditOperation *		eo= &(teo->teoEo);
    /*
    TedEditOperation		teo;
    SelectionGeometry		sg;
    SelectionDescription	sd;

    tedStartEditOperation( &teo, &sg, &sd, ed, 0 );
    */

    if  ( tedEditDeleteSelection( teo ) )
	{ LDEB(1); return -1;	}

    tp= docParaSpecialParticule( eo, DOCkindOBJECT, &partShift, &stroffShift );
    if  ( ! tp )
	{ LXDEB(eo->eoTailParticule,tp); return -1;	}
    tp->tpObjectNumber= objectNr;

    /*  3,4,5  */
    docExtendParagraphAdjust( eo, eo->eoTailDp.dpBi, stroffShift );

    /*  6,7  */
    eo->eoTailDp.dpStroff += stroffShift;
    {
    const int		direction= 1;
    const int		col0= -1;
    const int		col1= -1;

    DocumentSelection	dsObj;
    DocumentSelection	dsNew;

    docInitDocumentSelection( &dsObj );
    docInitDocumentSelection( &dsNew );

    docSetDocumentPosition( &(dsObj.dsHead),
				eo->eoTailDp.dpBi, eo->eoTailDp.dpStroff- 1 );
    docSetDocumentPosition( &(dsObj.dsTail),
				eo->eoTailDp.dpBi, eo->eoTailDp.dpStroff- 0 );

    docSetRangeSelection( &dsNew, &(dsObj.dsHead), &(dsObj.dsTail),
						    direction, col0, col1 );

    if  ( tedEditFinishSelection( teo, &dsNew ) )
	{ LDEB(1);	}
    }

    appDocumentChanged( ed, 1 );

    return 0;
    }

/************************************************************************/
/*									*/
/*  An object has been resized: Redo layout.				*/
/*									*/
/************************************************************************/

int tedEditReopenObject(	EditDocument *			ed,
				int				part,
				const DocumentPosition *	dpObj,
				const PositionGeometry *	pgObj )
    {
    TedDocument *		td= (TedDocument *)ed->edPrivateData;
    BufferDocument *		bd= td->tdDocument;
    AppDrawingData *		add= &(ed->edDocumentWidget.dwDrawingData);
    double			xfac= add->addMagnifiedPixelsPerTwip;

    BufferItem *		bi= dpObj->dpBi;
    TextParticule *		tp= bi->biParaParticules+ part;
    InsertedObject *		io;

    const int			stroffShift= 0;

    TedEditOperation		teo;
    SelectionGeometry		sg;
    SelectionDescription	sd;

    io= docGetObject( bd, tp->tpObjectNumber );
    if  ( ! io )
	{ LPDEB(tp->tpObjectNumber,io); return -1;	}

    tp->tpTwipsWide= ( io->ioScaleXUsed* io->ioTwipsWide )/ 100;

    io->ioPixelsWide= COORDtoGRID( xfac, tp->tpTwipsWide );
    io->ioPixelsHigh= COORDtoGRID( xfac,
				( io->ioScaleYUsed* io->ioTwipsHigh )/ 100 );
    if  ( io->ioPixelsWide < 1 )
	{ io->ioPixelsWide=  1;	}
    if  ( io->ioPixelsHigh < 1 )
	{ io->ioPixelsHigh=  1;	}

    tedStartEditOperation( &teo, &sg, &sd, ed, 0 );

    if  ( tedReopenObject( tp, &(teo.teoLayoutContext) ) )
	{ LDEB(1); return -1;	}

    docSetParagraphAdjust( &(teo.teoEo), bi, stroffShift,
					dpObj->dpStroff+ 1+ stroffShift );

    if  ( tedEditFinishOldSelection( &teo ) )
	{ LDEB(1);	}

    appDocumentChanged( ed, 1 );

    return 0;
    }

/************************************************************************/
/*									*/
/*  Replace the selection with something new.				*/
/*									*/
/*  1)  No selection... Just refuse.					*/
/*  2)  Replace nothing with nothing: Do nothing.			*/
/*	(Saves a lot of work, and avoids some combinations of special	*/
/*	cases)								*/
/*									*/
/************************************************************************/

void tedDocReplaceSelection(	EditDocument *	ed,
				const char *	word,
				int		len )
    {
    TedDocument *		td= (TedDocument *)ed->edPrivateData;

    TedEditOperation		teo;
    SelectionGeometry		sg;
    SelectionDescription	sd;

    /*  1  */
    if  ( ! tedHasSelection( td ) )
	{ LDEB(1); return; }

    tedStartEditOperation( &teo, &sg, &sd, ed, 0 );

    /*  2  */
    if  ( teo.teoEo.eoIBarSelectionOld && len == 0 )
	{ return;	}

    if  ( tedEditReplaceSelection( &teo, word, len ) )
	{ return;	}

    /*  6,7  */
    tedEditFinishOldSelectionEnd( &teo );

    appDocumentChanged( ed, 1 );

    return;
    }

/************************************************************************/
/*									*/
/*  Delete the current paragraph from a document.			*/
/*									*/
/************************************************************************/

static int tedEditDeleteSelectedParagaphs(	TedEditOperation *	teo )
    {
    int				rval= 0;
    EditOperation *		eo= &(teo->teoEo);

    BufferItem *		paraBi0;
    BufferItem *		paraBi1;

    DocumentPosition		dpNew;
    BufferItem *		parentBi;
    int				para0;
    int				para1;
    int				child0;
    int				child1;

    DocumentSelection		dsParas;

    paraBi0= eo->eoHeadDp.dpBi;
    paraBi1= eo->eoTailDp.dpBi;

    docInitDocumentSelection( &dsParas );
    if  ( docFirstPosition( &dsParas.dsHead, paraBi0 ) )
	{ LDEB(1); rval= -1; goto ready;	}
    if  ( docLastPosition( &dsParas.dsTail, paraBi1 ) )
	{ LDEB(1); rval= -1; goto ready;	}

    if  ( docEditDeleteFieldsForBlockDelete( eo, &dsParas ) )
	{ LDEB(1); rval= -1; goto ready;	}

    {
    BufferItem *	newBi= docNextParagraph( paraBi1 );

    if  ( ! newBi || docFirstPosition( &dpNew, newBi ) )
	{
	newBi= docPrevParagraph( paraBi0 );
	if  ( ! newBi || docLastPosition( &dpNew, newBi ) )
	    { LDEB(1); return -1;	}
	}
    }

    parentBi= paraBi0->biParent;
    para0= child0= paraBi0->biNumberInParent;
    para1= child1= paraBi1->biNumberInParent;
    while( child0 == 0					&&
	   child1 == parentBi->biChildCount- 1		&&
	   parentBi->biParent				)
	{
	if  ( docIsRowItem( parentBi->biParent ) )
	    { break;	}

	if  ( parentBi->biLevel == DOClevSECT )
	    { break;	}

	child0= child1= parentBi->biNumberInParent;
	parentBi= parentBi->biParent;
	}

    if  ( child0 == 0					&&
	  child1 == parentBi->biChildCount- 1		)
	{
	tedEditIncludeItemInRedraw( teo, paraBi0 );

	if  ( docEditMakeParagraphEmpty( paraBi0, eo ) )
	    { LDEB(1); rval= -1; goto ready;	}

	docEditIncludeItemInReformatRange( eo, paraBi0 );

	if  ( docFirstPosition( &dpNew, paraBi0 ) )
	    { LDEB(1); rval= -1; goto ready;	}

	child0= para0+ 1;
	child1= para1;
	parentBi= paraBi0->biParent;
	}

    if  ( child0 <= child1 )
	{
	int		firstParaDeleted= -1;
	int		sectionsDeleted= 0;
	int		paragraphsDeleted= 0;

	tedEditIncludeItemInRedraw( teo, paraBi0 );

	docEditDeleteItems( eo, &sectionsDeleted,
				    &firstParaDeleted, &paragraphsDeleted,
				    parentBi, child0, child1- child0+ 1 );

	docEditIncludeItemInReformatRange( eo, parentBi );
	}

    tedEditFinishIBarSelection( teo, dpNew.dpBi, dpNew.dpStroff );

    appDocumentChanged( teo->teoEditDocument, 1 );

  ready:

    return rval;
    }

int tedDocDeleteSelectedParagraphs(	EditDocument *	ed )
    {
    TedEditOperation		teo;
    SelectionGeometry		sg;
    SelectionDescription	sd;

    tedStartEditOperation( &teo, &sg, &sd, ed, 1 );

    if  ( ! sd.sdIsSingleCell )
	{ LDEB(sd.sdIsSingleCell); return -1;	}

    return tedEditDeleteSelectedParagaphs( &teo );
    }

int tedDocDeleteCurrentFrame(	EditDocument *	ed )
    {
    if  ( tedDocSelectCurrentFrame( ed ) )
	{ LDEB(1); return -1;	}

    return tedDocDeleteSelectedParagraphs( ed );
    }

/************************************************************************/
/*									*/
/*  Delete the current section.						*/
/*									*/
/************************************************************************/

int tedDeleteCurrentSection(	EditApplication *	ea )
    {
    int				rval= 0;

    EditDocument *		ed= ea->eaCurrentDocument;
    BufferItem *		paraBi;
    BufferItem *		sectBi;

    DocumentPosition		dpNew;
    BufferItem *		parentBi;

    TedEditOperation		teo;
    EditOperation *		eo= &(teo.teoEo);
    SelectionGeometry		sg;
    SelectionDescription	sd;

    int				newPositionIsPast= 1;

    int				paraNrOld= -1;
    DocumentSelection		dsSect;

    tedStartEditOperation( &teo, &sg, &sd, ed, 1 );
    paraBi= eo->eoHeadDp.dpBi;

    sectBi= paraBi->biParent->biParent->biParent;
    parentBi= sectBi->biParent;

    docInitDocumentSelection( &dsSect );
    if  ( docFirstPosition( &dsSect.dsHead, sectBi ) )
	{ LDEB(1); rval= -1; goto ready;	}
    if  ( docLastPosition( &dsSect.dsTail, sectBi ) )
	{ LDEB(1); rval= -1; goto ready;	}

    if  ( docEditDeleteFieldsForBlockDelete( eo, &dsSect ) )
	{ LDEB(1); rval= -1; goto ready;	}

    dpNew= dsSect.dsTail;
    if  ( docNextPosition( &dpNew ) )
	{
	newPositionIsPast= 0;

	dpNew= dsSect.dsTail;
	if  ( docPrevPosition( &dpNew ) )
	    { docInitDocumentPosition( &dpNew ); }
	}

    if  ( parentBi->biInExternalItem == DOCinBODY	&&
	  newPositionIsPast				)
	{ paraNrOld= docNumberOfParagraph( dpNew.dpBi );	}

    if  ( parentBi->biChildCount < 2 )
	{
	ParagraphProperties	pp;
	TextParticule *		tp= paraBi->biParaParticules;
	int			textAttributeNumber= tp->tpTextAttrNr;

	PropertyMask		ppChgMask;
	PropertyMask		ppUpdMask;

	int			firstParaDeleted= -1;
	int			sectionsDeleted= 0;
	int			paragraphsDeleted= 0;
	const int		sectShift= 0;

	docInitParagraphProperties( &pp );

	utilPropMaskClear( &ppChgMask );

	utilPropMaskClear( &ppUpdMask );
	utilPropMaskFill( &ppUpdMask, PPprop_FULL_COUNT );
	PROPmaskUNSET( &ppUpdMask, PPpropTABLE_NESTING );

	if  ( docUpdParaProperties( &ppChgMask, &pp,
		    &ppUpdMask, &(paraBi->biParaProperties),
		    (const DocumentAttributeMap *)0 ) )
	    { LDEB(1); rval= -1; goto ready;	}

	tedEditIncludeItemInRedraw( &teo, sectBi );

	docEditDeleteItems( eo, &sectionsDeleted,
					&firstParaDeleted, &paragraphsDeleted,
					sectBi, 0, sectBi->biChildCount );

	if  ( docSectionParagraph( eo, &paraBi, sectBi, sectShift,
						&pp, textAttributeNumber ) )
	    { LDEB(1); rval= -1; goto ready;	}

	docEditIncludeItemInReformatRange( eo, sectBi );

	docCleanParagraphProperties( &pp );

	if  ( docFirstPosition( &dpNew, paraBi ) )
	    { LDEB(1);	}
	}
    else{
	int		sectionsDeleted= 0;
	int		firstParaDeleted= -1;
	int		paragraphsDeleted= 0;

	tedEditIncludeItemInRedraw( &teo, sectBi );

	docEditDeleteItems( eo, &sectionsDeleted,
				    &firstParaDeleted, &paragraphsDeleted,
				    parentBi, sectBi->biNumberInParent, 1 );

	docEditIncludeItemInReformatRange( eo, sectBi );
	}

    docEditIncludeItemInReformatRange( eo, parentBi );

    tedEditFinishIBarSelection( &teo, dpNew.dpBi, dpNew.dpStroff );

    appDocumentChanged( ed, 1 );

  ready:

    return rval;
    }

/************************************************************************/
/*									*/
/*  Insert a paragraph Before/After the selection, depending on the	*/
/*  value of the 'after' argument.					*/
/*									*/
/************************************************************************/

int tedInsertParagraph(	EditApplication *	ea,
			int			after )
    {
    EditDocument *		ed= ea->eaCurrentDocument;
    TedDocument *		td= (TedDocument *)ed->edPrivateData;
    BufferDocument *		bd= td->tdDocument;

    BufferItem *		paraBi;
    BufferItem *		newBi;

    int				textAttributeNumber;
    BufferItem *		parentBi;
    int				pos;

    PropertyMask		ppChgMask;
    PropertyMask		ppUpdMask;

    TedEditOperation		teo;
    EditOperation *		eo= &(teo.teoEo);
    SelectionGeometry		sg;
    SelectionDescription	sd;

    DocumentPosition		dpRef;

    int				paraNr;
    const int			paraShift= 1;

    tedStartEditOperation( &teo, &sg, &sd, ed, 1 );

    if  ( after )
	{
	paraBi= eo->eoTailDp.dpBi;
	if  ( ! paraBi )
	    { XDEB(paraBi); return -1;	}

	if  ( docLastPosition( &dpRef, paraBi ) )
	    { LDEB(1); docInitDocumentPosition( &dpRef );	}

	pos= paraBi->biNumberInParent+ 1;
	textAttributeNumber= paraBi->biParaParticules[
			paraBi->biParaParticuleCount-1].tpTextAttrNr;
	}
    else{
	paraBi= eo->eoHeadDp.dpBi;
	if  ( ! paraBi )
	    { XDEB(paraBi); return -1;	}

	if  ( docFirstPosition( &dpRef, paraBi ) )
	    { LDEB(1); docInitDocumentPosition( &dpRef );	}

	pos= paraBi->biNumberInParent;
	textAttributeNumber= paraBi->biParaParticules[0].tpTextAttrNr;
	}

    tedEditIncludeItemInRedraw( &teo, paraBi );

    parentBi= paraBi->biParent;

    newBi= docInsertItem( bd, parentBi, pos, DOClevPARA );
    if  ( ! newBi )
	{ XDEB(newBi); return -1;	}

    paraNr= docNumberOfParagraph( newBi );

    {
    const int	sectShift= 0;
    const int	stroffFrom= 0;
    const int	stroffShift= 0;

    docEditShiftReferences( eo, &(eo->eoSelectionScope),
				    paraNr+ 1, stroffFrom,
				    sectShift, paraShift, stroffShift );
    }

    utilPropMaskClear( &ppChgMask );

    utilPropMaskClear( &ppUpdMask );
    utilPropMaskFill( &ppUpdMask, PPprop_FULL_COUNT );

    if  ( docEditUpdParaProperties( eo, &ppChgMask, newBi,
		    &ppUpdMask, &(paraBi->biParaProperties),
		    (const DocumentAttributeMap *)0 ) )
	{ LDEB(1);	}

    if  ( ! docInsertTextParticule( newBi, 0, 0, 0,
					DOCkindSPAN, textAttributeNumber ) )
	{ LDEB(1); return -1;	}

    docEditIncludeItemInReformatRange( eo, paraBi );
    docEditIncludeItemInReformatRange( eo, newBi );

    tedEditFinishIBarSelection( &teo, newBi, 0 );

    appDocumentChanged( ed, 1 );

    return 0;
    }

int tedInsertSection(	EditApplication *	ea,
			int			after )
    {
    EditDocument *		ed= ea->eaCurrentDocument;
    TedDocument *		td= (TedDocument *)ed->edPrivateData;
    BufferDocument *		bd= td->tdDocument;

    BufferItem *		paraBi;
    BufferItem *		sectBi;
    BufferItem *		newSectBi;
    BufferItem *		newParaBi;
    int				paraNr;

    int				textAttributeNumber;
    BufferItem *		parentBi;
    int				pos;

    TedEditOperation		teo;
    EditOperation *		eo= &(teo.teoEo);
    SelectionGeometry		sg;
    SelectionDescription	sd;

    DocumentPosition		dpRef;

    const int			sectShift= 1;

    tedStartEditOperation( &teo, &sg, &sd, ed, 1 );

    if  ( after )
	{
	paraBi= eo->eoTailDp.dpBi;
	if  ( ! paraBi || paraBi->biInExternalItem != DOCinBODY )
	    { XDEB(paraBi); return -1;	}

	sectBi= docGetSectItem( paraBi );
	if  ( ! sectBi )
	    { XDEB(sectBi); return -1;	}

	if  ( docLastPosition( &dpRef, sectBi ) )
	    { LDEB(1); docInitDocumentPosition( &dpRef );	}

	pos= sectBi->biNumberInParent+ 1;
	textAttributeNumber= paraBi->biParaParticules[
			paraBi->biParaParticuleCount-1].tpTextAttrNr;

	paraNr= docNumberOfParagraph( dpRef.dpBi )+ 1;
	}
    else{
	paraBi= eo->eoHeadDp.dpBi;
	if  ( ! paraBi || paraBi->biInExternalItem != DOCinBODY )
	    { XDEB(paraBi); return -1;	}

	sectBi= docGetSectItem( paraBi );
	if  ( ! sectBi )
	    { XDEB(sectBi); return -1;	}

	if  ( docFirstPosition( &dpRef, sectBi ) )
	    { LDEB(1); docInitDocumentPosition( &dpRef );	}

	pos= sectBi->biNumberInParent;
	textAttributeNumber= paraBi->biParaParticules[0].tpTextAttrNr;

	paraNr= docNumberOfParagraph( dpRef.dpBi );
	}

    tedEditIncludeItemInRedraw( &teo, sectBi );

    parentBi= sectBi->biParent;

    newSectBi= docInsertItem( bd, parentBi, pos, DOClevSECT );
    if  ( ! newSectBi )
	{ XDEB(newSectBi); return -1;	}
    eo->eoSectionsAdded++;

    if  ( docCopySectionProperties( &(newSectBi->biSectProperties),
						&(sectBi->biSectProperties) ) )
	{ LDEB(1);	}

    if  ( docSectionParagraph( eo, &newParaBi, newSectBi, sectShift,
			&(paraBi->biParaProperties), textAttributeNumber ) )
	{ LDEB(1); return -1;	}

    docEditIncludeItemInReformatRange( eo, sectBi );
    docEditIncludeItemInReformatRange( eo, newSectBi );

    tedEditFinishIBarSelection( &teo, newParaBi, 0 );

    appDocumentChanged( ed, 1 );

    return 0;
    }

/************************************************************************/
/*									*/
/*  Insert a (line/page) break particule in the document.		*/
/*									*/
/*  TABs could have been inserted with this routine, but against	*/
/*  flashing, a more subtle approach is used for TABs.			*/
/*									*/
/************************************************************************/

void tedEditInsertSpecialParticule(	EditDocument *	ed,
					int		kind,
					int		redoLayout )
    {
    TextParticule *		tp;

    TedEditOperation		teo;
    EditOperation *		eo= &(teo.teoEo);
    SelectionGeometry		sg;
    SelectionDescription	sd;

    tedStartEditOperation( &teo, &sg, &sd, ed, 1 );

    if  ( tedEditDeleteSelection( &teo ) )
	{ return;	}

    tp= docEditParaSpecialParticule( eo, kind );
    if  ( ! tp )
	{ LXDEB(eo->eoTailParticule,tp); return;	}

    if  ( docCheckNoBreakAsLast( eo, eo->eoTailDp.dpBi ) )
	{ LDEB(1);	}

    /*  3,4,5  */
    if  ( redoLayout )
	{ docEditIncludeItemInReformatRange( eo, eo->eoTailDp.dpBi );	}

    /*  6,7  */
    tedEditFinishIBarSelection( &teo,
			teo.teoEo.eoTailDp.dpBi, teo.teoEo.eoTailDp.dpStroff );

    appDocumentChanged( ed, 1 );

    return;
    }

/************************************************************************/
/*									*/
/*  Insert a 'Tab'.							*/
/*									*/
/************************************************************************/

void tedEditReplaceSelectionWithTab( EditDocument *	ed )
    {
    const int		redoLayout= 0;

    tedEditInsertSpecialParticule( ed, DOCkindTAB, redoLayout );

    return;
    }

/************************************************************************/
/*									*/
/*  Insert a section break in the document.				*/
/*									*/
/************************************************************************/

APP_MENU_CALLBACK_H( tedDocInsertSectBreak, option, voided, e )
    {
    EditDocument *		ed= (EditDocument *)voided;

    const int			onNewPage= 0;

    TedEditOperation		teo;
    EditOperation *		eo= &(teo.teoEo);
    SelectionGeometry		sg;
    SelectionDescription	sd;

    DocumentPosition		dpNew;

    tedStartEditOperation( &teo, &sg, &sd, ed, 1 );

    if  ( tedEditDeleteSelection( &teo ) )
	{ return;	}

    if  ( eo->eoTailDp.dpStroff > 0 )
	{
	BufferItem *		newParaBi;

	if  ( tedSplitParaItem( &newParaBi, &teo, DOClevSECT, onNewPage ) )
	    { LDEB(1); return;	}

	if  ( docFirstPosition( &dpNew, newParaBi ) )
	    { LDEB(1); return;	}

	eo->eoSectionsAdded++;
	}
    else{
	if  ( docEditInsertSection( &dpNew, eo ) )
	    { LDEB(1); return;	}
	}

    /*  7,8  */
    docAvoidParaHeadField( &dpNew, (int *)0, eo->eoBd );
    tedEditFinishIBarSelection( &teo, dpNew.dpBi, dpNew.dpStroff );

    appDocumentChanged( ed, 1 );

    return;
    }

/************************************************************************/
/*									*/
/*  Replace the current selection in a document with a bitmap image.	*/
/*									*/
/************************************************************************/

int tedReplaceSelectionWithBitmapImage(	EditDocument *		ed,
					AppBitmapImage *	abi )
    {
    InsertedObject *		io;
    int				objectNumber;

    SelectionGeometry		sg;
    SelectionDescription	sd;

    TedEditOperation		teo;

    tedStartEditOperation( &teo, &sg, &sd, ed, 1 );

    io= tedObjectMakeBitmapObject( &objectNumber, ed,
					teo.teoEo.eoHeadDp.dpBi,
					&(teo.teoLayoutContext), abi );
    if  ( ! io )
	{ XDEB(io); return -1; }

    if  ( tedReplaceSelectionWithObject( &teo, objectNumber, io ) )
	{ LDEB(1); return -1;	}

    return 0;
    }

/************************************************************************/
/*									*/
/*  Callback of the 'Insert Picture' menu option.			*/
/*									*/
/************************************************************************/

int tedObjectOpenPicture(	EditApplication *	ea,
				void *			voided,
				APP_WIDGET		relative,
				APP_WIDGET		option,
				const char *		filename )
    {
    EditDocument *		ed= (EditDocument *)voided;
    TedDocument *		td= (TedDocument *)ed->edPrivateData;
    BufferDocument *		bd= td->tdDocument;

    TedEditOperation		teo;
    SelectionGeometry		sg;
    SelectionDescription	sd;

    int				objectNumber;
    InsertedObject *		io;
    PictureProperties *		pip;

    tedStartEditOperation( &teo, &sg, &sd, ed, 1 );

    io= docClaimObject( &objectNumber, bd );
    if  ( ! io )
	{ XDEB(io); return -1;	}

    pip= &(io->ioPictureProperties);

    if  ( docReadFileObject( filename, strlen( filename ), io ) )
	{ SDEB(filename); return -1;;	}

    tedScaleObjectToSelectedParagraph( ed, io );

    /*  9
    io->ioBliptag= appGetTimestamp();
    */

    if  ( tedOpenObject( io, &(teo.teoLayoutContext) ) )
	{ LDEB(1);	}

    io->ioScaleXSet= io->ioScaleXUsed;
    io->ioScaleYSet= io->ioScaleYUsed;
    pip->pipScaleXSet= pip->pipScaleXUsed;
    pip->pipScaleYSet= pip->pipScaleYUsed;

    if  ( tedReplaceSelectionWithObject( &teo, objectNumber, io ) )
	{ SDEB(filename); return -1;	}

    return 0;
    }

