/************************************************************************/
/*									*/
/*  Ted, Screen drawing and forcing drawing through			*/
/*									*/
/************************************************************************/

#   include	"tedConfig.h"

#   include	<stddef.h>
#   include	<stdlib.h>
#   include	<stdio.h>

#   include	"tedApp.h"
#   include	"docScreenLayout.h"
#   include	"docPixels.h"
#   include	"tedDraw.h"
#   include	"tedLayout.h"

#   include	<appDebugon.h>

/************************************************************************/
/*									*/
/*  Draw a text item.							*/
/*									*/
/*  1)  Tabs need not to be drawn.					*/
/*									*/
/************************************************************************/

static int tedDrawTab(	const BufferItem *	paraBi,
			const TextParticule *	tp,
			int			x0FrameShifted,
			int			baseLinePixels,
			ScreenDrawingData *	sdd,
			DrawingContext *	dc )
    {
    const LayoutContext *	lc= &(dc->dcLayoutContext);
    AppDrawingData *		add= lc->lcAdd;
    double			xfac= add->addMagnifiedPixelsPerTwip;
    const BufferDocument *	bd= lc->lcDocument;
    const TabStopList *		tsl= &(paraBi->biParaTabStopList);
    const TabStop *		ts= tsl->tslTabStops+ tp->tpObjectNumber;

    int				x0;
    int				x1;

    int				textAttr;
    int				screenFont;
    TextAttribute		ta;

    int				pixelsWide;
    int				pixelSize;

    pixelsWide= COORDtoGRID( xfac, tp->tpTwipsWide );
    x0= x0FrameShifted+ tp->tpXContentXPixels;
    x1= x0FrameShifted+ tp->tpXContentXPixels+ pixelsWide;

    textAttr= tp->tpTextAttrNr;
    screenFont= utilIndexMappingGet( lc->lcAttributeToScreenFont, textAttr );

    utilGetTextAttributeByNumber( &ta, &(bd->bdTextAttributeList), textAttr );
    pixelSize= COORDtoGRID( xfac, 2* ta.taFontSizeHalfPoints );

    switch( ts->tsLeader )
	{
	case DOCtlNONE:
	    break;

	case DOCtlDOTS:

	    if  ( screenFont < 0 )
		{ LLDEB(textAttr,screenFont); return -1;	}

	    x0 += pixelSize/ 4;
	    x1 -= pixelSize/ 2;
	    x0= 3* ( ( x0+ 2 )/ 3 );

	    if  ( x1 <= x0 )
		{ return 0;	}

	    if  ( ta.taFontIsBold )
		{
		static const unsigned char	boldDot[]= { 2, 1 };

		appDrawSetLineAttributes( add,
			2, LINEstyleON_OFF_DASH, LINEcapBUTT, LINEjoinMITER,
			boldDot, sizeof( boldDot ) );
		}
	    else{
		static const unsigned char	dot[]= { 1, 2 };

		appDrawSetLineAttributes( add,
			1, LINEstyleON_OFF_DASH, LINEcapBUTT, LINEjoinMITER,
			dot, sizeof( dot ) );
		}

	    docDrawSetColorNumber( dc, sdd, ta.taTextColorNumber );

	    appDrawDrawLine( add,
			    x0- lc->lcOx, baseLinePixels- lc->lcOy,
			    x1- lc->lcOx, baseLinePixels- lc->lcOy );

	    break;

	case DOCtlUNDERLINE:
	    if  ( screenFont < 0 )
		{ LLDEB(textAttr,screenFont); return -1;	}

	    x0 += pixelSize/ 4;
	    x1 -= pixelSize/ 2;
	    if  ( x1 <= x0 )
		{ return 0;	}

	    if  ( ta.taFontIsBold )
		{
		appDrawSetLineAttributes( add,
				2, LINEstyleSOLID, LINEcapBUTT, LINEjoinMITER,
				(const unsigned char *)0, 0 );
		}
	    else{
		appDrawSetLineAttributes( add,
				1, LINEstyleSOLID, LINEcapBUTT, LINEjoinMITER,
				(const unsigned char *)0, 0 );
		}

	    docDrawSetColorNumber( dc, sdd, ta.taTextColorNumber );

	    appDrawDrawLine( add,
			    x0- lc->lcOx, baseLinePixels- lc->lcOy,
			    x1- lc->lcOx, baseLinePixels- lc->lcOy );

	    break;

	case DOCtlHYPH:
	    if  ( screenFont < 0 )
		{ LLDEB(textAttr,screenFont); return -1;	}

	    x0 += pixelSize/ 4;
	    x1 -= pixelSize/ 2;
	    x0= 7* ( ( x0+ 6 )/ 7 );

	    if  ( x1 <= x0 )
		{ return 0;	}

	    if  ( ta.taFontIsBold )
		{
		static const unsigned char	boldDash[]= { 4, 3 };

		appDrawSetLineAttributes( add,
			2, LINEstyleON_OFF_DASH, LINEcapBUTT, LINEjoinMITER,
			boldDash, sizeof( boldDash ) );
		}
	    else{
		static const unsigned char	dash[]= { 3, 4 };

		appDrawSetLineAttributes( add,
			1, LINEstyleON_OFF_DASH, LINEcapBUTT, LINEjoinMITER,
			dash, sizeof( dash ) );
		}

	    docDrawSetColorNumber( dc, sdd, ta.taTextColorNumber );

	    appDrawDrawLine( add,
			    x0- lc->lcOx, baseLinePixels- lc->lcOy,
			    x1- lc->lcOx, baseLinePixels- lc->lcOy );

	    break;

	    break;

	case DOCtlTHICK:
	    LDEB(ts->tsLeader);
	    break;
	case DOCtlEQUAL:
	    LDEB(ts->tsLeader);
	    break;
	default:
	    LDEB(ts->tsLeader);
	    break;
	}

    return 0;
    }

static void tedDrawParticuleLine(	DrawingContext *	dc,
					ScreenDrawingData *	sdd,
					const BufferItem *	paraBi,
					const TextParticule *	tp,
					int			textColorNr,
					int			x0,
					int			y0,
					int			h )
    {
    const LayoutContext *	lc= &(dc->dcLayoutContext);
    AppDrawingData *		add= lc->lcAdd;
    const int			lastOne= 0;
    DocumentPosition		dp;
    PositionGeometry		pg;
    int				x1;

    docSetDocumentPosition( &dp, (BufferItem *)paraBi,
						tp->tpStroff+ tp->tpStrlen );

    tedPositionGeometry( &pg, &dp, dc->dcBodySectBi, lastOne, lc );
    x1= pg.pgXPixels;
    if  ( x1 > x0 )
	{
	docDrawSetColorNumber( dc, (void *)sdd, textColorNr );

	appDrawFillRectangle( add,
			    x0- lc->lcOx, y0- lc->lcOy, x1- x0, h );
	}

    return;
    }

static void tedDrawParticuleUnderlines(	DrawingContext *	dc,
					ScreenDrawingData *	sdd,
					const BufferItem *	paraBi,
					const TextParticule *	tp,
					int			drawn,
					int			screenFont,
					int			x0FrameShifted,
					int			baseLinePixels )
    {
    const LayoutContext *	lc= &(dc->dcLayoutContext);
    AppDrawingData *		add= lc->lcAdd;
    const BufferDocument *	bd= lc->lcDocument;
    int				i;

    i= 0;
    while( i < drawn )
	{
	TextAttribute	ta;
	int		x0;
	int		y0;
	int		h;
	int		from;

	utilGetTextAttributeByNumber( &ta, &(bd->bdTextAttributeList),
						tp[i].tpTextAttrNr );
	if  ( ! ta.taTextIsUnderlined )
	    { i++; continue;	}

	x0= x0FrameShifted+ tp[i].tpXContentXPixels;
	appDrawUnderlineGeometry( &h, &y0,
					add, screenFont, baseLinePixels );

	from= i;
	while( i < drawn )
	    {
	    utilGetTextAttributeByNumber( &ta, &(bd->bdTextAttributeList),
						tp[i].tpTextAttrNr );
	    if  ( ! ta.taTextIsUnderlined )
		{ break;	}

	    i++;
	    }

	if  ( i > from )
	    {
	    tedDrawParticuleLine( dc, sdd, paraBi, &(tp[i-1]),
					    ta.taTextColorNumber, x0, y0, h );
	    }
	}

    return;
    }

static void tedDrawParticuleStrikethrough(	DrawingContext *	dc,
					ScreenDrawingData *	sdd,
					const BufferItem *	paraBi,
					const TextParticule *	tp,
					int			drawn,
					int			screenFont,
					int			x0FrameShifted,
					int			baseLinePixels )
    {
    const LayoutContext *	lc= &(dc->dcLayoutContext);
    AppDrawingData *		add= lc->lcAdd;
    const BufferDocument *	bd= lc->lcDocument;
    int				i;

    i= 0;
    while( i < drawn )
	{
	TextAttribute	ta;
	int		x0;
	int		y0;
	int		h;
	int		from;

	utilGetTextAttributeByNumber( &ta, &(bd->bdTextAttributeList),
						tp[i].tpTextAttrNr );
	if  ( ! ta.taHasStrikethrough )
	    { i++; continue;	}

	x0= x0FrameShifted+ tp[i].tpXContentXPixels;
	appDrawStrikethroughGeometry( &h, &y0,
					add, screenFont, baseLinePixels );

	from= i;
	while( i < drawn )
	    {
	    utilGetTextAttributeByNumber( &ta, &(bd->bdTextAttributeList),
						tp[i].tpTextAttrNr );
	    if  ( ! ta.taHasStrikethrough )
		{ break;	}

	    i++;
	    }

	if  ( i > from )
	    {
	    tedDrawParticuleLine( dc, sdd, paraBi, &(tp[i-1]),
					    ta.taTextColorNumber, x0, y0, h );
	    }
	}

    return;
    }

/************************************************************************/
/*									*/
/*  Draw a footnote separator.						*/
/*									*/
/************************************************************************/

static void tedDrawChftnsep(	DrawingContext *	dc,
				int			x0FrameShifted,
				int			baseLinePixels,
				void *			vsdd,
				const TextParticule *	tp,
				const TextLine *	tl )
    {
    const LayoutContext *	lc= &(dc->dcLayoutContext);
    AppDrawingData *		add= lc->lcAdd;
    const BufferDocument *	bd= lc->lcDocument;
    double			xfac= add->addMagnifiedPixelsPerTwip;
    ScreenDrawingData *		sdd= (ScreenDrawingData *)vsdd;

    int				y0;
    int				h;
    int				pixelsWide;
    int				x0;

    int				screenFont;
    TextAttribute		ta;

    appDrawSetLineAttributes( add,
			    1, LINEstyleSOLID, LINEcapBUTT, LINEjoinMITER,
			    (const unsigned char *)0, 0 );

    screenFont= utilIndexMappingGet( lc->lcAttributeToScreenFont, tp->tpTextAttrNr );
    if  ( screenFont < 0 )
	{ LLDEB(tp->tpTextAttrNr,screenFont); return;	}

    appDrawStrikethroughGeometry( &h, &y0, add, screenFont, baseLinePixels );

    utilGetTextAttributeByNumber( &ta, &(bd->bdTextAttributeList),
							    tp->tpTextAttrNr );

    pixelsWide= COORDtoGRID( xfac, tp->tpTwipsWide );
    x0= x0FrameShifted+ tp->tpXContentXPixels;

    docDrawSetColorNumber( dc, (void *)sdd, ta.taTextColorNumber );

    appDrawFillRectangle( add, x0- lc->lcOx, y0- lc->lcOy, pixelsWide, h );

    return;
    }

/************************************************************************/
/*									*/
/*  Draw the individual segments of a smallcaps string.			*/
/*									*/
/************************************************************************/

static void tedDrawSegments(	DrawingContext *	dc,
				AppDrawingData *	add,
				int			x,
				int			y,
				int			fullScreenFont,
				int			scapsScreenFont,
				const char *		s,
				const int *		segments,
				int			segmentCount )
    {
    int				seg;

    DocumentRectangle		drText;

    for ( seg= 0; seg < segmentCount; seg++ )
	{
	if  ( segments[2* seg+ 0] > 0 )
	    {
	    appDrawSetFont( add, fullScreenFont );

	    appDrawDrawString( add, x, y, (char *)s, segments[2* seg+ 0] );

	    appDrawTextExtents( &drText, x, y, add, fullScreenFont,
				    (const char *)s, segments[2* seg+ 0] );

	    s += segments[2* seg+ 0];
	    x= drText.drX1+ 1;
	    }

	if  ( segments[2* seg+ 1] > 0 )
	    {
	    appDrawSetFont( add, scapsScreenFont );

	    appDrawDrawString( add, x, y, (char *)s, segments[2* seg+ 1] );

	    appDrawScapsTextExtents( &drText, x, y,
				    add, fullScreenFont, scapsScreenFont,
				    (const char *)s, segments[2* seg+ 1] );

	    s += segments[2* seg+ 1];
	    x= drText.drX1+ 1;
	    }
	}

    return;
    }

/************************************************************************/

static int tedDrawSpan(		DrawingContext *	dc,
				void *			vsdd,
				int			x0,
				int			baseLinePixels,
				const TextAttribute *	ta,
				int			textAttr,
				int			fullScreenFont,
				const char *		printString,
				int			len )
    {
    const LayoutContext *	lc= &(dc->dcLayoutContext);
    AppDrawingData *		add= lc->lcAdd;

    char *			upperString= (char *)0;
    int *			segments= (int *)0;
    int				segmentCount= 0;

    int				y= baseLinePixels;

    if  ( ta->taSmallCaps || ta->taCapitals )
	{
	if  ( docMakeCapsString( &upperString, &segments, &segmentCount,
						    ta, printString, len ) )
	    { LDEB(len); return -1;	}

	printString= upperString;
	}

    if  ( ta->taSuperSub == DOCfontSUPERSCRIPT )
	{
	appDrawGetSuperBaseline( &y, baseLinePixels, add, fullScreenFont );
	}

    if  ( ta->taSuperSub == DOCfontSUBSCRIPT )
	{
	appDrawGetSubBaseline( &y, baseLinePixels, add, fullScreenFont );
	}

    docDrawSetFont( dc, vsdd, textAttr, ta );
    docDrawSetColorNumber( dc, vsdd, ta->taTextColorNumber );

    if  ( ta->taSmallCaps && ! ta->taCapitals )
	{
	int	scapsScreenFont= docLayoutScapsScreenFont( lc, ta );
	if  ( scapsScreenFont < 0 )
	    { LDEB(scapsScreenFont); return -1;	}

	tedDrawSegments( dc, add, x0- lc->lcOx, y- lc->lcOy,
					fullScreenFont, scapsScreenFont,
					printString, segments, segmentCount );
	}
    else{
	appDrawDrawString( add,
		x0- lc->lcOx, y- lc->lcOy, (char *)printString, len );
	}

    /*
    appDebug( "[%3d,%3d] -- %.*s\n", y, x0, len, printString );
    */

    if  ( upperString )
	{ free( upperString );	}
    if  ( segments )
	{ free( segments );	}

    return 0;
    }

/************************************************************************/
/*									*/
/*  Draw a series of particules.					*/
/*									*/
/*  1)  Only explicit tabs need to be drawn: Implicit ones have no	*/
/*	leader.								*/
/*  2)  MS-Word draws underlined tabs!					*/
/*									*/
/************************************************************************/

static int tedDrawParticules(	DrawingContext *		dc,
				void *				vsdd,
				const BufferItem *		paraBi,
				int				part,
				int				count,
				const TextLine *		tl,
				const DocumentRectangle *	drLine,
				int				x0FrameShifted,
				int				baseLinePixels )
    {
    const LayoutContext *	lc= &(dc->dcLayoutContext);
    const BufferDocument *	bd= lc->lcDocument;
    ScreenDrawingData *		sdd= (ScreenDrawingData *)vsdd;
    const TextParticule *	tp= paraBi->biParaParticules+ part;
    const TabStopList *		tsl= &(paraBi->biParaTabStopList);

    int				drawn;
    int				len;

    int				fullScreenFont;
    TextAttribute		ta;

    DocumentRectangle		drSpan;

    drSpan= *drLine;
    drSpan.drX0= x0FrameShifted+ tp->tpXContentXPixels;

    /*  1  */
    switch( tp->tpKind )
	{ 
	case DOCkindTAB:
	    /*  1  */
	    if  ( tp->tpObjectNumber >= 0			&&
		  tp->tpObjectNumber < tsl->tslTabStopCount	)
		{
		if  ( tedDrawTab( paraBi, tp,
			    x0FrameShifted, baseLinePixels, sdd, dc ) )
		    { LDEB(1);	}
		}

	    drawn= 1;
	    if  ( part+ drawn < tl->tlFirstParticule+ tl->tlParticuleCount )
		{ drSpan.drX1= x0FrameShifted+ tp[drawn].tpXContentXPixels; }

	    /*  2  */
	    fullScreenFont= utilIndexMappingGet( lc->lcAttributeToScreenFont,
							    tp->tpTextAttrNr );
	    if  ( fullScreenFont < 0 )
		{ LLDEB(tp->tpTextAttrNr,fullScreenFont); return -1;	}

	    tedDrawParticuleUnderlines( dc, sdd, paraBi, tp, drawn,
			    fullScreenFont, x0FrameShifted, baseLinePixels );
	    tedDrawParticuleStrikethrough( dc, sdd, paraBi, tp, drawn,
			    fullScreenFont, x0FrameShifted, baseLinePixels );

	    return drawn;

	case DOCkindSPAN:
	    break;

	case DOCkindFIELDSTART:
	case DOCkindFIELDEND:
	case DOCkindLINEBREAK:
	case DOCkindPAGEBREAK:
	case DOCkindCOLUMNBREAK:
	    return drawn= 1;

	case DOCkindOBJECT:

	    drawn= 1;
	    if  ( part+ drawn < tl->tlFirstParticule+ tl->tlParticuleCount )
		{ drSpan.drX1= x0FrameShifted+ tp[drawn].tpXContentXPixels; }

	    if  ( tedDrawObject( tp,
			x0FrameShifted+ tp->tpXContentXPixels, baseLinePixels,
			dc->dcClipRect, lc ) )
		{ LDEB(1); return -1;	}

	    return drawn;

	case DOCkindCHFTNSEP:
	case DOCkindCHFTNSEPC:
	    drawn= 1;
	    if  ( part+ drawn < tl->tlFirstParticule+ tl->tlParticuleCount )
		{ drSpan.drX1= x0FrameShifted+ tp[drawn].tpXContentXPixels; }

	    tedDrawChftnsep( dc, x0FrameShifted, baseLinePixels, vsdd, tp, tl );

	    return drawn;

	case DOCkindOPT_HYPH:
	    if  ( tp->tpTwipsWide > 0 )
		{ LLDEB(tp->tpKind,tp->tpTwipsWide);	}
	    return drawn= 1;

	default:
	    LDEB(tp->tpKind); return -1;
	}

    {
    const int	separate= 1; /*PS: paraBi->biParaAlignment == DOCiaJUSTIFIED;*/
    int		x0Twips= 0;
    int		x1Twips;

    drawn= docLayoutDelimitSpan( &len, &x1Twips, x0Twips, &ta,
				    bd, paraBi, part, part+ count, separate );
    }

    if  ( part+ drawn < tl->tlFirstParticule+ tl->tlParticuleCount )
	{ drSpan.drX1= x0FrameShifted+ tp[drawn].tpXContentXPixels; }

    if  ( dc->dcClipRect						&&
	  ! geoIntersectRectangle( (DocumentRectangle *)0,
					    dc->dcClipRect, &drSpan )	)
	{ return drawn;	}

    fullScreenFont= utilIndexMappingGet( lc->lcAttributeToScreenFont,
							    tp->tpTextAttrNr );
    if  ( fullScreenFont < 0 )
	{ LLDEB(tp->tpTextAttrNr,fullScreenFont); return -1;	}

    if  ( len > 0 )
	{
	const char *	printString= (char *)docParaString( paraBi, tp->tpStroff );
	int		x0= x0FrameShifted+ tp->tpXContentXPixels;

	if  ( tedDrawSpan( dc, vsdd, x0, baseLinePixels, 
					&ta, tp->tpTextAttrNr, fullScreenFont,
					printString, len ) )
	    { LDEB(len); return -1;	}
	}

    tedDrawParticuleUnderlines( dc, sdd, paraBi, tp, drawn, fullScreenFont,
					x0FrameShifted, baseLinePixels );
    tedDrawParticuleStrikethrough( dc, sdd, paraBi, tp, drawn, fullScreenFont,
					x0FrameShifted, baseLinePixels );

    return drawn;
    }

int tedDrawTextLine(		BufferItem *			paraBi,
				int				line,
				const ParagraphFrame *		pf,
				const DocumentRectangle *	drLine,
				void *				vsdd,
				DrawingContext *		dc,
				const BlockOrigin *		bo )
    {
    const LayoutContext *	lc= &(dc->dcLayoutContext);
    AppDrawingData *		add= lc->lcAdd;

    const TextLine *		tl= paraBi->biParaLines+ line;
    int				part= tl->tlFirstParticule;
    int				done;

    int				baseLinePixels;

    LayoutPosition		lpTop;
    int				x0FrameShifted;

    if  ( dc->dcClipRect						&&
	  ! geoIntersectRectangle( (DocumentRectangle *)0,
					    dc->dcClipRect, drLine )	)
	{ return tl->tlParticuleCount;	}

    docShiftPosition( &lpTop, bo, &(tl->tlTopPosition) );

    baseLinePixels= LP_Y1PIXELS( add, &lpTop, TL_BASELINE( tl ) );
    x0FrameShifted= X_PIXELS( add, pf->pfParaContentRect.drX0+ bo->boXShift );

    done= 0;
    while( done < tl->tlParticuleCount )
	{
	int		drawn;

	drawn= tedDrawParticules( dc, vsdd, paraBi,
				part, tl->tlParticuleCount- done, tl,
				drLine, x0FrameShifted, baseLinePixels );
	if  ( drawn < 1 )
	    { LDEB(drawn); return -1;	}

	done += drawn; part += drawn;
	}

    return done;
    }

/************************************************************************/
/*									*/
/*  Draw a rectangle to show that (part of) a line is selected.		*/
/*									*/
/*  1)  Delimit the line and calculate a preliminary rectangle for the	*/
/*	line.								*/
/*  2)  Table rectangle selections look nicer when we select the full	*/
/*	width of the table cell.					*/
/*  3)  Where to we start the selected rectangle? (left)		*/
/*  4)  Where to we end the selected rectangle? (right)			*/
/*  5)  Do the line and the selection overlap. The two separate tests	*/
/*	reject the situation where only the begin or end position of	*/
/*	the line is in the selection.					*/
/*	Either the begin of the selection must be before the end of the	*/
/*	line or the end of the selection must be after the end of the	*/
/*	line.								*/
/*  6)  Intersect with clipping rectangle.				*/
/*									*/
/************************************************************************/

static int tedLineRectangle(	DocumentRectangle *		drRedraw,
				int *				pInSelection,
				const BufferItem *		paraBi,
				DrawingContext *		dc,
				ScreenDrawingData *		sdd,
				int				line,
				const DocumentRectangle *	drParaContent )
    {
    const LayoutContext *	lc= &(dc->dcLayoutContext);
    AppDrawingData *		add= lc->lcAdd;
    const DocumentSelection *	ds= dc->dcDocumentSelection;
    const SelectionGeometry *	sg= dc->dcSelectionGeometry;

    DocumentSelection	dsLine;
    SelectionGeometry	sgLine;

    const int		lastLine= 0;
    DocumentRectangle	drLine;

    int			tableRectangle;
    int			cmp_SeLe;
    int			cmp_SbLb;
    int			cmp_SbLe;
    int			cmp_SeLb;

    int			partLineBegin;
    int			partLineEnd;

    int			inSelection;

    if  ( ! docSelectionSameRoot( ds, paraBi ) )
	{ return 0;	}

    /*  1  */
    docLineSelection( &dsLine, &partLineBegin, &partLineEnd, paraBi, line );
    tedSelectionGeometry( &sgLine, &dsLine, dc->dcBodySectBi, lastLine, lc );

    if  ( sgLine.sgBegin.pgLine != line )
	{ LLDEB(sgLine.sgBegin.pgLine,line);	}
    if  ( sgLine.sgEnd.pgLine != line )
	{ LLDEB(sgLine.sgEnd.pgLine,line);	}

    drLine= sgLine.sgRectangle;

    /*  2  */
    tableRectangle= 0;
    if  ( ds->dsCol0 >= 0						&&
	  ds->dsCol1 >= 0						&&
	  ! docPositionsInsideCell( &(ds->dsHead), &(ds->dsTail) )	)
	{ tableRectangle= 1; }

    cmp_SbLb= docComparePositions( &(ds->dsHead), &(dsLine.dsHead) );
    cmp_SeLb= docComparePositions( &(ds->dsTail), &(dsLine.dsHead) );
    cmp_SbLe= docComparePositions( &(ds->dsHead), &(dsLine.dsTail) );
    cmp_SeLe= docComparePositions( &(ds->dsTail), &(dsLine.dsTail) );

    /*  3  */
    if  ( cmp_SbLb <= 0 || ( cmp_SbLb == 0 && tableRectangle ) )
	{
	int	x0Twips= drParaContent->drX0;

	if  ( paraBi->biParaFirstIndentTwips < 0 )
	    { x0Twips += paraBi->biParaFirstIndentTwips;	}

	drLine.drX0= X_PIXELS( add, x0Twips );

	if  ( cmp_SbLb == 0 )
	    {
	    x0Twips= drParaContent->drX0;

	    if  ( line == 0 )
		{ x0Twips += paraBi->biParaFirstIndentTwips;	}

	    drLine.drX0= X_PIXELS( add, x0Twips );
	    }
	}
    else{ drLine.drX0= sg->sgBegin.pgXPixels;	}

    /*  4  */
    if  ( cmp_SeLe > 0 || ( cmp_SeLe == 0 && tableRectangle ) )
	{ drLine.drX1= X_PIXELS( add, drParaContent->drX1 );		}
    else{ drLine.drX1= sg->sgEnd.pgXPixels;				}

    /*  5  */
    inSelection= 0;
    if  ( cmp_SbLe < 0 && cmp_SeLb > 0 )
	{ inSelection= 1;	}

    if  ( docParaStrlen( ds->dsTail.dpBi ) == 0	&&
	  cmp_SeLb == 0				)
	{ cmp_SeLb=  1;	}
    if  ( docParaStrlen( ds->dsHead.dpBi ) == 0	&&
	  cmp_SbLe == 0				)
	{ cmp_SbLe= -1;	}
    if  ( cmp_SbLe < 0 && cmp_SeLb > 0 )
	{ inSelection= 1;	}

    /*  6  */
    if  ( ! dc->dcClipRect						||
	  geoIntersectRectangle( drRedraw, dc->dcClipRect, &drLine )	)
	{ *pInSelection= inSelection; return 1;	}
    else{ return 0;	}
    }

int tedDrawTextReverse(		BufferItem *			paraBi,
				int				line,
				const ParagraphFrame *		pf,
				const DocumentRectangle *	drLine,
				void *				vsdd,
				DrawingContext *		dc,
				const BlockOrigin *		bo )
    {
    const LayoutContext *	lc= &(dc->dcLayoutContext);
    AppDrawingData *		add= lc->lcAdd;
    ScreenDrawingData *		sdd= (ScreenDrawingData *)vsdd;

    DocumentRectangle		drRedraw;
    int				done;
    int				inSelection;

    if  ( tedLineRectangle( &drRedraw, &inSelection, paraBi, dc, sdd, line,
						    &(pf->pfParaContentRect) ) )
	{
	drRedraw.drX0 -= lc->lcOx;
	drRedraw.drX1 -= lc->lcOx;
	drRedraw.drY0 -= lc->lcOy;
	drRedraw.drY1 -= lc->lcOy;

	appDrawSetClipRect( add, &drRedraw );

	done= tedDrawTextLine( paraBi, line, pf, drLine, vsdd, dc, bo );

	tedOriginalClipping( dc, sdd );
	}
    else{
	TextLine *	tl= paraBi->biParaLines+ line;

	done= tl->tlParticuleCount;
	}

    return done;
    }

int tedDrawTextSelected(	BufferItem *			paraBi,
				int				line,
				const ParagraphFrame *		pf,
				const DocumentRectangle *	drLine,
				void *				vsdd,
				DrawingContext *		dc,
				const BlockOrigin *		bo )
    {
    const LayoutContext *	lc= &(dc->dcLayoutContext);
    AppDrawingData *		add= lc->lcAdd;
    ScreenDrawingData *		sdd= (ScreenDrawingData *)vsdd;

    DocumentRectangle		drRedraw;
    int				done;
    int				inSelection;

    if  ( tedLineRectangle( &drRedraw, &inSelection, paraBi, dc, sdd, line,
						    &(pf->pfParaContentRect) ) )
	{
	drRedraw.drX0 -= lc->lcOx;
	drRedraw.drX1 -= lc->lcOx;
	drRedraw.drY0 -= lc->lcOy;
	drRedraw.drY1 -= lc->lcOy;

	appDrawSetClipRect( add, &drRedraw );

	if  ( inSelection )
	    {
	    docDrawSetColorRgb( dc, (void *)sdd, &(sdd->sddBackColor) );

	    appDrawFillRectangle( add,
		    drRedraw.drX0,
		    drRedraw.drY0,
		    drRedraw.drX1- drRedraw.drX0+ 1,
		    drRedraw.drY1- drRedraw.drY0+ 1 );
	    }

	done= tedDrawTextLine( paraBi, line, pf, drLine, vsdd, dc, bo );

	tedOriginalClipping( dc, sdd );
	}
    else{
	TextLine *	tl= paraBi->biParaLines+ line;

	done= tl->tlParticuleCount;
	}

    return done;
    }

