/************************************************************************/
/*									*/
/*  Ted, Screen drawing and forcing drawing through			*/
/*									*/
/************************************************************************/

#   include	"tedConfig.h"

#   include	<stddef.h>
#   include	<stdlib.h>
#   include	<stdio.h>

#   include	"tedApp.h"
#   include	"docScreenLayout.h"
#   include	"docPixels.h"
#   include	"tedDraw.h"
#   include	"appMatchFont.h"
#   include	"tedLayout.h"

#   include	<appDebugon.h>

/************************************************************************/
/*									*/
/*  Cause the smallest rectangle that contains the selection to be	*/
/*  redrawn.								*/
/*									*/
/************************************************************************/

void tedExposeSelection(	EditDocument *			ed,
				const DocumentSelection *	ds,
				const BufferItem *		bodySectBi,
				int				scrolledX,
				int				scrolledY )
    {
    const int			lastLine= 0;
    SelectionGeometry		sg;
    LayoutContext		lc;

    layoutInitContext( &lc );
    tedSetLayoutContext( &lc, ed );

    tedSelectionGeometry( &sg, ds, bodySectBi, lastLine, &lc );

    appDocExposeRectangle( ed, &(sg.sgRectangle), scrolledX, scrolledY );

    return;
    }

/************************************************************************/
/*									*/
/*  Blinking cursor.							*/
/*									*/
/*  1)  Turned off when we are debugging exposures and redraws.		*/
/*									*/
/************************************************************************/

void tedUndrawIBar(	const EditDocument *	ed )
    {
#   if BLINK_IBAR

    TedDocument *		td= (TedDocument *)ed->edPrivateData;

    DocumentSelection		ds;
    SelectionGeometry		sg;
    SelectionDescription	sd;

    const int			scrolledX= 0;
    const int			scrolledY= 0;

    const BufferItem *		bodySectBi;

    if  ( tedGetSelection( &ds, &sg, &sd,
			    (DocumentTree **)0, &bodySectBi, td ) )
	{ LDEB(1); return;	}

    appDocExposeRectangle( ed, &(sg.sgRectangle), scrolledX, scrolledY );

#   endif

    return;
    }

/************************************************************************/
/*									*/
/*  Cause subsequent drawing to be done in a certain color.		*/
/*									*/
/*  NOTE that background drawing with color= 0 is against the RTF idea	*/
/*	of the default background color: transparent.			*/
/*									*/
/************************************************************************/

static int tedDrawSetColorRgb(	DrawingContext *	dc,
				void *			vsdd,
				const RGB8Color *	rgb8 )
    {
    const LayoutContext *	lc= &(dc->dcLayoutContext);

    drawSetForegroundColor( lc->lcDrawingSurface, rgb8 );
    appDrawSetForegroundColorRgb8( lc->lcAdd, rgb8 );

    return 0;
    }

static void tedDrawSetBorderColor(	DrawingContext *	dc,
					ScreenDrawingData *	sdd,
					int			colorNumber,
					int			asGrid )
    {
    if  ( asGrid )
	{ docDrawSetColorRgb( dc, (void *)sdd, &(sdd->sddGridColor) );	}
    else{ docDrawSetColorNumber( dc, (void *)sdd, colorNumber );	}

    return;
    }

static int tedDrawSetFont(	DrawingContext *	dc,
				void *			vsdd,
				int			textAttr,
				const TextAttribute *	ta )
    {
    const LayoutContext *	lc= &(dc->dcLayoutContext);
    AppDrawingData *		add= lc->lcAdd;

    int				screenFont;

    screenFont= utilIndexMappingGet( lc->lcAttributeToScreenFont, textAttr );
    if  ( screenFont < 0 )
	{ LLDEB(textAttr,screenFont); return -1;	}

    appDrawSetFont( add, screenFont );

    return 0;
    }

/************************************************************************/
/*									*/
/*  Draw background ornaments.						*/
/*									*/
/************************************************************************/

static int tedDrawOrnaments(	const BlockOrnaments *		bo,
				int				page,
				const DocumentRectangle *	drOutside,
				const DocumentRectangle *	drInside,
				void *				through,
				struct DrawingContext *		dc )
    {
    ScreenDrawingData *		sdd= (ScreenDrawingData *)through;
    const LayoutContext *	lc= &(dc->dcLayoutContext);
    AppDrawingData *		add= lc->lcAdd;
    const BufferDocument *	bd= lc->lcDocument;

    DocumentRectangle		drOutPixels;
    DocumentRectangle		drInPixels;

    int				topAsGrid= 0;
    int				leftAsGrid= 0;
    int				rightAsGrid= 0;
    int				bottomAsGrid= 0;

    docGetPixelRect( &drOutPixels, lc, drOutside, page );
    drInPixels= drOutPixels;

    if  ( dc->dcClipRect						&&
	  ! geoIntersectRectangle( (DocumentRectangle *)0,
					&drOutPixels, dc->dcClipRect )	)
	{ return 0;	}

    docGetPixelRect( &drInPixels, lc, drInside, page );

    if  ( drOutside->drX0 < drInside->drX0	&&
	  drOutPixels.drX0 >= drInPixels.drX0	)
	{ drInPixels.drX0= drOutPixels.drX0+ 1;	}

    if  ( drOutside->drY0 < drInside->drY0	&&
	  drOutPixels.drY0 >= drInPixels.drY0	)
	{ drInPixels.drY0= drOutPixels.drY0+ 1;	}

    if  ( drOutside->drY1 > drInside->drY1	&&
	  drOutPixels.drY1 <= drInPixels.drY1	)
	{ drInPixels.drY1= drOutPixels.drY1- 1;	}

    if  ( PROPmaskISSET( &(bo->boPropMask), ORNdrawTOP_GRID ) )
	{
	topAsGrid= 1;
	drInPixels.drY0= drOutPixels.drY0+ 1;
	}

    if  ( PROPmaskISSET( &(bo->boPropMask), ORNdrawLEFT_GRID ) )
	{
	leftAsGrid= 1;
	drInPixels.drX0= drOutPixels.drX0+ 1;
	}

    if  ( PROPmaskISSET( &(bo->boPropMask), ORNdrawRIGHT_GRID ) )
	{
	rightAsGrid= 1;
	drOutPixels.drX1++;
	}

    if  ( PROPmaskISSET( &(bo->boPropMask), ORNdrawBOTTOM_GRID ) )
	{
	bottomAsGrid= 1;
	drOutPixels.drY1++;
	}

    if  ( PROPmaskISSET( &(bo->boPropMask), ORNdrawTOP_BORDER )	)
	{
	if  ( drInPixels.drY0 == drOutPixels.drY0 )
	    { drInPixels.drY0++;	}
	}

    if  ( PROPmaskISSET( &(bo->boPropMask), ORNdrawLEFT_BORDER ) )
	{
	if  ( drInPixels.drX0 == drOutPixels.drX0 )
	    { drInPixels.drX0++;	}
	}

    if  ( PROPmaskISSET( &(bo->boPropMask), ORNdrawRIGHT_BORDER ) )
	{
	if  ( drInPixels.drX1 == drOutPixels.drX1 )
	    { drInPixels.drX1--;	}
	}

    if  ( PROPmaskISSET( &(bo->boPropMask), ORNdrawBOTTOM_BORDER ) )
	{
	if  ( drInPixels.drY1 == drOutPixels.drY1 )
	    { drInPixels.drY1--;	}
	}

    if  ( PROPmaskISSET( &(bo->boPropMask), ORNdrawSHADE ) )
	{
	DocumentRectangle	drShade= drInPixels;

	if  ( ! dc->dcClipRect						||
	      geoIntersectRectangle( &drShade, &drInPixels,
						    dc->dcClipRect )	)
	    {
	    ItemShading		is= bo->boShading;

	    int			isFilled= 0;
	    RGB8Color		rgb8;

	    if  ( docGetSolidRgbShadeOfItem( &isFilled, &rgb8, bd, &is ) )
		{ LDEB(1);	}

	    if  ( is.isPattern != DOCspSOLID )
		{
		int		level= 0;

		switch( is.isPattern )
		    {
		    case DOCspBGHORIZ:		level= 15; break;
		    case DOCspBGVERT:		level= 15; break;
		    case DOCspBGFDIAG:		level= 15; break;
		    case DOCspBGBDIAG:		level= 15; break;
		    case DOCspBGCROSS:		level= 15; break;
		    case DOCspBGDCROSS:		level= 15; break;

		    case DOCspBGDKHORIZ:	level= 30; break;
		    case DOCspBGDKVERT:		level= 30; break;
		    case DOCspBGDKFDIAG:	level= 30; break;
		    case DOCspBGDKBDIAG:	level= 30; break;
		    case DOCspBGDKCROSS:	level= 30; break;
		    case DOCspBGDKDCROSS:	level= 30; break;
		    }

		if  ( bo->boShading.isForeColor == 0 )
		    {
		    if  ( bo->boShading.isBackColor == 0 )
			{ bmRGB8SolidBlack( &rgb8 );	}

		    rgb8.rgb8Red= ( ( 100- level )* rgb8.rgb8Red )/ 100;
		    rgb8.rgb8Green= ( ( 100- level )* rgb8.rgb8Green )/ 100;
		    rgb8.rgb8Blue= ( ( 100- level )* rgb8.rgb8Blue )/ 100;
		    isFilled= 1;
		    }
		else{
		    is.isLevel= 100* level;
		    is.isPattern= DOCspSOLID;

		    if  ( docGetSolidRgbShadeOfItem( &isFilled, &rgb8,
								    bd, &is ) )
			{ LDEB(1);	}
		    }
		}

	    if  ( isFilled )
		{
		docDrawSetColorRgb( dc, (void *)sdd, &rgb8 );

		appDrawFillRectangle( add,
				    drShade.drX0- lc->lcOx,
				    drShade.drY0- lc->lcOy,
				    drShade.drX1- drShade.drX0+ 1,
				    drShade.drY1- drShade.drY0+ 1 );
		}
	    }
	}

    if  ( PROPmaskISSET( &(bo->boPropMask), ORNdrawTOP_BORDER )	||
	  PROPmaskISSET( &(bo->boPropMask), ORNdrawTOP_GRID )	)
	{
	DocumentRectangle	drBorder= drOutPixels;

	drBorder.drY1= drInPixels.drY0;
	drBorder.drY1= drInPixels.drY0- 1;
	if  ( drBorder.drY1 < drBorder.drY0 )
	    { drBorder.drY1=  drBorder.drY0;	}

	if  ( ! dc->dcClipRect						  ||
	      geoIntersectRectangle( &drBorder, &drBorder, dc->dcClipRect ) )
	    {
	    tedDrawSetBorderColor( dc, sdd,
				    bo->boTopBorder.bpColor, topAsGrid );

	    appDrawFillRectangle( add,
			drBorder.drX0- lc->lcOx,
			drBorder.drY0- lc->lcOy,
			drBorder.drX1- drBorder.drX0+ 1,
			drBorder.drY1- drBorder.drY0+ 1 );
	    }
	}

    if  ( PROPmaskISSET( &(bo->boPropMask), ORNdrawLEFT_BORDER )	||
	  PROPmaskISSET( &(bo->boPropMask), ORNdrawLEFT_GRID )		)
	{
	DocumentRectangle	drBorder= drOutPixels;

	drBorder.drY0= drInPixels.drY0;
	drBorder.drY1= drInPixels.drY1;
	drBorder.drX1= drInPixels.drX0- 1;

	if  ( drBorder.drX1 < drBorder.drX0 )
	    { drBorder.drX1=  drBorder.drX0;	}

	if  ( ! dc->dcClipRect						  ||
	      geoIntersectRectangle( &drBorder, &drBorder, dc->dcClipRect ) )
	    {
	    tedDrawSetBorderColor( dc, sdd,
				    bo->boLeftBorder.bpColor, leftAsGrid );

	    appDrawFillRectangle( add,
			drBorder.drX0- lc->lcOx,
			drBorder.drY0- lc->lcOy,
			drBorder.drX1- drBorder.drX0+ 1,
			drBorder.drY1- drBorder.drY0+ 1 );
	    }
	}

    if  ( PROPmaskISSET( &(bo->boPropMask), ORNdrawRIGHT_BORDER )	||
	  PROPmaskISSET( &(bo->boPropMask), ORNdrawRIGHT_GRID )		)
	{
	DocumentRectangle	drBorder= drOutPixels;

	drBorder.drY0= drInPixels.drY0;
	drBorder.drY1= drInPixels.drY1;
	drBorder.drX0= drInPixels.drX1+ 1;

	if  ( drBorder.drX0 > drBorder.drX1 )
	    { drBorder.drX0=  drBorder.drX1;	}

	/* Avoid overlaps with fills
	drBorder.drX0--;
	drBorder.drX1--;
	*/

	if  ( ! dc->dcClipRect						  ||
	      geoIntersectRectangle( &drBorder, &drBorder, dc->dcClipRect ) )
	    {
	    tedDrawSetBorderColor( dc, sdd,
				    bo->boRightBorder.bpColor, rightAsGrid );

	    appDrawFillRectangle( add,
			drBorder.drX0- lc->lcOx,
			drBorder.drY0- lc->lcOy,
			drBorder.drX1- drBorder.drX0+ 1,
			drBorder.drY1- drBorder.drY0+ 1 );
	    }
	}

    if  ( PROPmaskISSET( &(bo->boPropMask), ORNdrawBOTTOM_BORDER )	||
	  PROPmaskISSET( &(bo->boPropMask), ORNdrawBOTTOM_GRID )	)
	{
	DocumentRectangle	drBorder= drOutPixels;

	drBorder.drY0= drInPixels.drY1+ 1;

	if  ( drBorder.drY0 > drBorder.drY1 )
	    { drBorder.drY0=  drBorder.drY1;	}

	/* Avoid overlaps with fills
	drBorder.drY0--;
	drBorder.drY1--;
	*/

	if  ( ! dc->dcClipRect						  ||
	      geoIntersectRectangle( &drBorder, &drBorder, dc->dcClipRect ) )
	    {
	    tedDrawSetBorderColor( dc, sdd,
				    bo->boBottomBorder.bpColor, bottomAsGrid );

	    appDrawFillRectangle( add,
			drBorder.drX0- lc->lcOx,
			drBorder.drY0- lc->lcOy,
			drBorder.drX1- drBorder.drX0+ 1,
			drBorder.drY1- drBorder.drY0+ 1 );
	    }
	}

    return 0;
    }

/************************************************************************/
/*									*/
/*  Skip to the next page.						*/
/*									*/
/************************************************************************/

static void tedDrawPageGap(	ScreenDrawingData *		sdd,
				DrawingContext *		dc,
				int				page )
    {
    const LayoutContext *	lc= &(dc->dcLayoutContext);
    AppDrawingData *		add= lc->lcAdd;

    DocumentRectangle		drBetween;

    drBetween.drX0= add->addBackRect.drX0;
    drBetween.drX1= add->addBackRect.drX1;
    drBetween.drY0= page* add->addPageStepPixels- add->addPageGapPixels;
    drBetween.drY1= page* add->addPageStepPixels;

    if  ( dc->dcClipRect						   &&
	  ! geoIntersectRectangle( &drBetween, &drBetween, dc->dcClipRect ) )
	{ return;	}

    docDrawSetColorRgb( dc, (void *)sdd, &(sdd->sddPageColor) );

    appDrawFillRectangle( add,
					drBetween.drX0- lc->lcOx,
					drBetween.drY0- lc->lcOy,
					drBetween.drX1- drBetween.drX0+ 1,
					drBetween.drY1- drBetween.drY0+ 1 );

    return;
    }

/************************************************************************/
/*									*/
/*  Restore/instll original clip rect.					*/
/*									*/
/************************************************************************/

void tedOriginalClipping(	DrawingContext *		dc,
				ScreenDrawingData *		sdd )
    {
    const LayoutContext *	lc= &(dc->dcLayoutContext);
    AppDrawingData *	add= lc->lcAdd;

    if  ( dc->dcClipRect )
	{
	DocumentRectangle	drRestore;

	drRestore= *(dc->dcClipRect);
	drRestore.drX0 -= lc->lcOx;
	drRestore.drY0 -= lc->lcOy;
	drRestore.drX1 -= lc->lcOx;
	drRestore.drY1 -= lc->lcOy;

	appDrawSetClipRect( add, &drRestore );
	}
    else{
	appDrawNoClipping( add );
	}
    }

/************************************************************************/
/*									*/
/*  Draw an I Bar.							*/
/*									*/
/************************************************************************/

void tedGetIBarRect(	DocumentRectangle *		drPixels,
			const PositionGeometry *	pg,
			const LayoutContext *		lc )
    {
    docGetPixelRectForPos( drPixels, lc,
			    pg->pgXTwips, pg->pgXTwips,
			    &(pg->pgTopPosition), &(pg->pgBottomPosition) );

    drPixels->drX0= drPixels->drX1= pg->pgXPixels; /* Do not trust XTwips yet */

    return;
    }

int tedDrawIBar(	const DocumentRectangle *	drPixels,
			const LayoutContext *		lc )
    {
    DocumentRectangle	drShifted= *drPixels;

    drShifted.drX0 -= lc->lcOx;
    drShifted.drX1 -= lc->lcOx;
    drShifted.drY0 -= lc->lcOy;
    drShifted.drY1 -= lc->lcOy;

    appDrawFillRectangle( lc->lcAdd,
			    drShifted.drX0, drShifted.drY0,
			    drShifted.drX1- drShifted.drX0+ 1,
			    drShifted.drY1- drShifted.drY0+ 1 );
    return 0;
    }

/************************************************************************/
/*									*/
/*  Draw the eight blocks around a selected object.			*/
/*									*/
/************************************************************************/

static void tedDrawObjectBlocks(	const DocumentRectangle *	drObj,
					const APP_POINT *		xp,
					DrawingContext *		dc,
					ScreenDrawingData *		sdd )
    {
    const LayoutContext *	lc= &(dc->dcLayoutContext);
    AppDrawingData *	add= lc->lcAdd;

    int			i;

    docDrawSetColorRgb( dc, (void *)sdd, &(sdd->sddBackColor) );

    for ( i= 0; i < RESIZE_COUNT; i++ )
	{
	appDrawFillRectangle( add,
			xp[i].x- lc->lcOx+ 1, xp[i].y- lc->lcOy+ 1,
			RESIZE_BLOCK- 1, RESIZE_BLOCK- 1 );
	}

    docDrawSetColorRgb( dc, (void *)sdd, &(sdd->sddForeColor) );

    for ( i= 0; i < RESIZE_COUNT; i++ )
	{
	appDrawDrawRectangle( add,
			xp[i].x- lc->lcOx+ 1, xp[i].y- lc->lcOy+ 1,
			RESIZE_BLOCK- 2, RESIZE_BLOCK- 2 );
	}

    appDrawDrawRectangle( add,
		    drObj->drX0- lc->lcOx+ 1,
		    drObj->drY0- lc->lcOy+ 1,
		    drObj->drX1- drObj->drX0- 1,
		    drObj->drY1- drObj->drY0- 1 );

    return;
    }

/************************************************************************/
/*									*/
/*  Draw the box around an active footnote, header or footer.		*/
/*									*/
/************************************************************************/

static void tedDrawItemBox(	ScreenDrawingData *		sdd,
				DrawingContext *		dc,
				const DocumentRectangle *	drBox )
    {
    const LayoutContext *	lc= &(dc->dcLayoutContext);
    static unsigned char	dot[]= { 1, 2 };
    RGB8Color			rgb8;

    bmRGB8SolidBlack( &rgb8 );

    appDrawSetLineAttributes( lc->lcAdd,
		    1, LINEstyleON_OFF_DASH, LINEcapBUTT, LINEjoinMITER,
		    dot, sizeof( dot ) );

    docDrawSetColorRgb( dc, (void *)sdd, &rgb8 );

    appDrawDrawLine( lc->lcAdd,
			    drBox->drX0- lc->lcOx, drBox->drY0- lc->lcOy,
			    drBox->drX1- lc->lcOx, drBox->drY0- lc->lcOy );
    appDrawDrawLine( lc->lcAdd,
			    drBox->drX0- lc->lcOx, drBox->drY1- lc->lcOy,
			    drBox->drX1- lc->lcOx, drBox->drY1- lc->lcOy );
    appDrawDrawLine( lc->lcAdd,
			    drBox->drX0- lc->lcOx, drBox->drY0- lc->lcOy,
			    drBox->drX0- lc->lcOx, drBox->drY1- lc->lcOy );
    appDrawDrawLine( lc->lcAdd,
			    drBox->drX1- lc->lcOx, drBox->drY0- lc->lcOy,
			    drBox->drX1- lc->lcOx, drBox->drY1- lc->lcOy );

    }

static void tedDrawExternalItemBox(
			    const BufferItem *		bodySectBi,
			    const DocumentTree *	ei,
			    ScreenDrawingData *		sdd,
			    DrawingContext *		dc,
			    const DocumentRectangle *	drClip )
    {
    const LayoutContext *	lc= &(dc->dcLayoutContext);
    
    DocumentRectangle		drIntersect;
    DocumentRectangle		drBox;

    const int			justUsed= 0;

    if  ( ei->eiPageSelectedUpon < 0 || ei->eiColumnSelectedIn < 0 )
	{
	LLDEB(ei->eiPageSelectedUpon,ei->eiColumnSelectedIn);
	SDEB(docExternalKindStr(ei->eiRoot->biInExternalItem));
	return;
	}

    if  ( docGetExternalItemBox( &drBox, bodySectBi, ei, justUsed,
			    ei->eiPageSelectedUpon,
			    ei->eiColumnSelectedIn,
			    lc ) )
	{ LDEB(1); return;	}

    if  ( ! geoIntersectRectangle( &drIntersect, &drBox, drClip ) )
	{ return;	}

    tedDrawItemBox( sdd, dc, &drBox );

    return;
    }

static int tedDrawCheckPageOfSelectedExtItem(
				    SelectionGeometry *		sg,
				    const DocumentSelection *	ds,
				    DocumentTree *		selRootEi,
				    const LayoutContext *	lc )
    {
    int			changed= 0;
    DocumentRectangle	drChanged;
    const int		lastLine= 1;

    BufferItem *	bodySectBi;

    if  ( docCheckPageOfSelectedExtItem( &changed, &bodySectBi, &drChanged,
				selRootEi, lc,
				docInitScreenLayoutExternalItem ) )
	{ LDEB(1); return -1;	}

    if  ( changed )
	{
	tedSelectionGeometry( sg, ds, bodySectBi, lastLine, lc );
	}

    return 0;
    }

/************************************************************************/
/*									*/
/*  Start a page: Draw an external item box if needed.			*/
/*									*/
/*  1)  Selection not in an external item?				*/
/*  2)  In range? covers the case where both numbers are -1 because	*/
/*	nothing needs to be done.					*/
/*  3)  Calculate box on this page.					*/
/*  4)  Intersects clip rectangle?					*/
/*  5)  Draw!								*/
/*									*/
/************************************************************************/

static int tedDrawStartPage(	void *				vsdd,
				const DocumentGeometry *	dgPage,
				DrawingContext *		dc,
				int				page )
    {
    ScreenDrawingData *		sdd= (ScreenDrawingData *)vsdd;
    DocumentTree *		selRootEi= sdd->sddSelRootEi;
    BufferItem *		selRootBi= sdd->sddSelRootBi;
    BufferItem *		selRootBodySectBi= sdd->sddSelRootBodySectBi;

    const LayoutContext *	lc= &(dc->dcLayoutContext);

    const int			justUsed= 0;
    DocumentRectangle		drBox;
    DocumentRectangle		drIntersect;

    /*  1  */
    if  ( ! selRootEi || ! selRootBi )
	{ return 0;	}

    /*  2  */
    if  ( sdd->sddBoxFirstPage > page		||
	  sdd->sddBoxLastPage < page		)
	{ return 0;	}

    /*  3  */
    if  ( docGetExternalItemBox( &drBox, selRootBodySectBi, selRootEi,
					justUsed,
					selRootEi->eiPageSelectedUpon,
					selRootEi->eiColumnSelectedIn,
					lc ) )
	{ LDEB(1); return 0;	}

    /*  4  */
    if  ( dc->dcClipRect && ! geoIntersectRectangle( &drIntersect, &drBox, dc->dcClipRect ) )
	{ return 0;	}

    /*  5  */
    tedDrawItemBox( sdd, dc, &drBox );

    return 0;
    }

/************************************************************************/
/*									*/
/*  Expose handler for documents.					*/
/*									*/
/*  2)  Clear background.						*/
/*  2b) If the selected area overlaps with the exposed region, draw the	*/
/*	selection background.						*/
/*  3)  Draw text.							*/
/*  4)	Draw page separators.						*/
/*  5)  Draw I bar if necessary.					*/
/*									*/
/************************************************************************/

void tedDrawRectangle(		EditDocument *		ed,
				DocumentRectangle *	drClipPixels,
				int			ox,
				int			oy )
    {
    TedDocument *		td= (TedDocument *)ed->edPrivateData;
    DocumentWidget *		dw= &(ed->edDocumentWidget);
    AppDrawingData *		add= &(dw->dwDrawingData);

    BufferDocument *		bd= td->tdDocument;
    BufferItem * const		bodyBi= bd->bdBody.eiRoot;

    DrawingContext		dc;
    ScreenDrawingData		sdd;

    BufferItem *		selRootBi= (BufferItem *)0;

    RGB8Color			selColor;

    DocumentSelection		ds;
    SelectionGeometry		sg;
    SelectionDescription	sd;

    DocumentTree *		selRootEi= (DocumentTree *)0;
    BufferItem *		selRootBodySectBi= (BufferItem *)0;

    const BufferItem *		bodySectBi;

    LayoutPosition		lpBelow;

    DrawingSurface		drsf= ed->edDrawingSurface;
    DocumentRectangle		drClipTwips;

    if  ( tedGetSelection( &ds, &sg, &sd,
			    (DocumentTree **)0, &bodySectBi, td ) )
	{
	docInitDocumentSelection( &ds );
	docInitSelectionGeometry( &sg );
	}

    if  ( td->tdVisibleSelectionCopied )
	{ selColor= td->tdCopiedSelColor;	}
    else{ selColor= td->tdSelColor;		}

    drawTransformRectangleBackward( &drClipTwips, drsf, drClipPixels );

    docInitDrawingContext( &dc );

    tedSetLayoutContext( &(dc.dcLayoutContext), ed );
    dc.dcDrawTableGrid= ( td->tdDrawTableGrid >= 0 );
    dc.dcClipRect= drClipPixels;
    dc.dcSetColorRgb= tedDrawSetColorRgb;
    dc.dcSetFont= tedDrawSetFont;
    dc.dcDrawShape= tedDrawDrawingShape;

    sdd.sddForeColor.rgb8Red= 0;
    sdd.sddForeColor.rgb8Green= 0;
    sdd.sddForeColor.rgb8Blue= 0;

    sdd.sddBackColor.rgb8Red= 255;
    sdd.sddBackColor.rgb8Green= 255;
    sdd.sddBackColor.rgb8Blue= 255;

    sdd.sddGridColor= td->tdTableColor;

#   ifdef USE_MOTIF
    sdd.sddPageColor.rgb8Red= add->addBackColor.red;
    sdd.sddPageColor.rgb8Green= add->addBackColor.green;
    sdd.sddPageColor.rgb8Blue= add->addBackColor.blue;
#   endif

#   ifdef USE_GTK
    sdd.sddPageColor.rgb8Red= add->addBackColor.red;
    sdd.sddPageColor.rgb8Green= add->addBackColor.green;
    sdd.sddPageColor.rgb8Blue= add->addBackColor.blue;
#   endif

#   if LOG_REDRAWS
    docLogRectangle( "REDRAW", drClip );
#   endif

    tedOriginalClipping( &dc, &sdd );

    /*  2  */
    docDrawSetColorRgb( &dc, (void *)&sdd, &(sdd.sddBackColor) );

#   if 0
    drawFillRectangle( drsf, &drClipTwips );
#   else
    appDrawFillRectangle( add,
		drClipPixels->drX0- ox, drClipPixels->drY0- oy,
		drClipPixels->drX1- drClipPixels->drX0+ 1,
		drClipPixels->drY1- drClipPixels->drY0+ 1 );
#   endif

    /*  2a  */
    if  ( tedHasSelection( td ) )
	{
	selRootBi= docGetSelectionRoot(
				&selRootEi, &selRootBodySectBi, bd, &ds );
	if  ( ! selRootBi )
	    { XDEB(selRootBi);	}
	}

    /*  3  */
    sdd.sddForeColor.rgb8Red= 0;
    sdd.sddForeColor.rgb8Green= 0;
    sdd.sddForeColor.rgb8Blue= 0;

    sdd.sddBackColor.rgb8Red= 255;
    sdd.sddBackColor.rgb8Green= 255;
    sdd.sddBackColor.rgb8Blue= 255;

    sdd.sddSelRootBi= selRootBi;
    sdd.sddSelRootEi= selRootEi;
    sdd.sddSelRootBodySectBi= selRootBodySectBi;
    sdd.sddBoxFirstPage= -1;
    sdd.sddBoxLastPage= -1;

    if  ( selRootEi )
	{
	const BufferItem *	selRoolSectBi= selRootEi->eiRoot;

	if  ( selRoolSectBi->biInExternalItem == DOCinFOOTNOTE	||
	      selRoolSectBi->biInExternalItem == DOCinENDNOTE	)
	    {
	    sdd.sddBoxFirstPage= selRoolSectBi->biTopPosition.lpPage;
	    sdd.sddBoxLastPage= selRoolSectBi->biBelowPosition.lpPage;
	    }
	}

    dc.dcDrawTextLine= tedDrawTextLine;
    dc.dcDrawOrnaments= tedDrawOrnaments;

    dc.dcStartPage= tedDrawStartPage;

    dc.dcInitLayoutExternal= docInitScreenLayoutExternalItem;
    dc.dcDrawExternalItems= 0;
    dc.dcPostponeHeadersFooters= 0;

    dc.dcDocumentSelection= (DocumentSelection *)0;
    dc.dcSelectionGeometry= (SelectionGeometry *)0;

    dc.dcFirstPage= drClipPixels->drY0/ add->addPageStepPixels;
    dc.dcLastPage=  drClipPixels->drY1/ add->addPageStepPixels;
    if  ( dc.dcFirstPage > bodyBi->biBelowPosition.lpPage )
	{ dc.dcFirstPage=  bodyBi->biBelowPosition.lpPage; }

    appDrawSetBackgroundWhite( add );

    docDrawSetColorRgb( &dc, (void *)&sdd, &(sdd.sddForeColor) );

    dc.dcCurrentColorSet= 0;
    dc.dcCurrentTextAttributeSet= 0;

    docInitLayoutPosition( &lpBelow );

    if  ( docDrawItem( &lpBelow, bodyBi, (void *)&sdd, &dc ) )
	{ LDEB(1);	}

    {
    int		firstPage;
    int		lastPage;

    int		page;
    int		sectNr;

    int		y0= drClipPixels->drY0;
    int		y1= drClipPixels->drY1;

    y0= drClipPixels->drY0- add->addPageGapPixels;
    if  ( y0 < 0 )
	{ y0= 0;	}

    y1= drClipPixels->drY1+ add->addPageStepPixels- 1;
    if  ( y1 > BI_BELOW_PIXELS( add, bodyBi ) )
	{ y1=  BI_BELOW_PIXELS( add, bodyBi );	}

    firstPage= y0/ add->addPageStepPixels;
    lastPage=  y1/ add->addPageStepPixels;

    if  ( lastPage > bodyBi->biBelowPosition.lpPage )
	{ lastPage=  bodyBi->biBelowPosition.lpPage;	}

    for ( page= firstPage; page <= lastPage; page++ )
	{ tedDrawPageGap( &sdd, &dc, page );	}

    for ( sectNr= 0; sectNr < bodyBi->biChildCount; sectNr++ )
	{
	BufferItem *	bodySectBi= bodyBi->biChildren[sectNr];
	BufferItem *	prevSectBi= (BufferItem *)0;
	int		first= bodySectBi->biTopPosition.lpPage;
	int		last= bodySectBi->biBelowPosition.lpPage;

	if  ( first < dc.dcFirstPage )
	    { first=  dc.dcFirstPage;	}
	if  ( last >  dc.dcLastPage )
	    { last=   dc.dcLastPage;		}

	if  ( sectNr > 0 )
	    { prevSectBi= bodyBi->biChildren[sectNr- 1];	}
	/*
	Word draws the footer of the first section on the page.
	if  ( sectNr < bodyBi->biChildCount- 1 )
	    { nextSectBi= bodyBi->biChildren[sectNr+ 1];	}
	*/

	for ( page= first; page <= last; page++ )
	    {
	    int		column;
	    const int	belowText= 0;

	    if  ( ! prevSectBi					||
		  prevSectBi->biBelowPosition.lpPage < page	)
		{ docDrawPageHeader( bodySectBi, (void *)&sdd, &dc, page ); }

	    for ( column= 0; column < bodySectBi->biSectColumnCount; column++ )
		{
		docDrawFootnotesForColumn( page, column, (void *)&sdd, &dc );
		}

	    /* Word draws the footer of the first section on the page. */
	    if  ( ! prevSectBi					||
		  prevSectBi->biBelowPosition.lpPage < page	)
		{ docDrawPageFooter( bodySectBi, (void *)&sdd, &dc, page ); }

	    if  ( docDrawShapesForTree( &(bd->bdBody), bodyBi->biChildren[0],
					(void *)&sdd, &dc, belowText, page ) )
		{ LDEB(page);	}
	    }
	}
    }

    if  ( selRootBi					&&
	  ! sd.sdIsIBarSelection			)
	{
	if  ( td->tdDrawMonochrome )
	    {
	    bmRGB8SolidWhite( &(sdd.sddForeColor) );
	    dc.dcDrawTextLine= tedDrawTextReverse;
	    }
	else{
	    sdd.sddBackColor= selColor;
	    dc.dcDrawTextLine= tedDrawTextSelected;
	    }

	docDrawSetColorRgb( &dc, (void *)&sdd, &(sdd.sddForeColor) );

	dc.dcDrawOrnaments= (DRAW_ORNAMENTS)0;
	dc.dcFinishPage= (FINISH_PAGE)0;
	dc.dcStartPage= (START_PAGE)0;

	dc.dcInitLayoutExternal= (INIT_LAYOUT_EXTERNAL)0;
	dc.dcDrawExternalItems= 0;
	dc.dcPostponeHeadersFooters= 0;

	dc.dcDocumentSelection= &ds;
	dc.dcSelectionGeometry= &sg;

	if  ( selRootBi->biInExternalItem != DOCinBODY			&&
	      tedDrawCheckPageOfSelectedExtItem( &sg, &ds,
				    selRootEi, &(dc.dcLayoutContext) )	)
	    { LDEB(1); return; }

	docInitLayoutPosition( &lpBelow );

	dc.dcBodySectBi= sdd.sddSelRootBodySectBi;
	if  ( docDrawItem( &lpBelow, selRootBi, (void *)&sdd, &dc ) )
	    { LDEB(1);	}
	}

    if  ( selRootBi					&&
	  selRootBi->biInExternalItem != DOCinBODY	&&
	  selRootBi->biInExternalItem != DOCinENDNOTE	)
	{
	DocumentTree *	x_selRootEi;
	BufferItem *	bodySectBi;

	if  ( docGetRootOfSelectionScope( &x_selRootEi,
				&bodySectBi, bd, &(ds.dsSelectionScope) ) )
	    { LDEB(1);	}
	else{
	    if  ( sdd.sddSelRootEi != x_selRootEi )
		{ XXDEB(sdd.sddSelRootEi,x_selRootEi);	}

	    tedDrawExternalItemBox( bodySectBi, x_selRootEi,
						    &sdd, &dc, drClipPixels );
	    }
	}

    docDrawSetColorRgb( &dc, (void *)&sdd, &(sdd.sddForeColor) );

    /*  5  */
    if  ( selRootBi )
	{
	if  ( sd.sdIsIBarSelection )
	    {
	    DocumentRectangle		drIBarPixels;

	    if  ( selRootBi->biInExternalItem != DOCinBODY		&&
		  tedDrawCheckPageOfSelectedExtItem( &sg, &ds,
				selRootEi, &(dc.dcLayoutContext) )	)
		{ LDEB(1); return; }

	    tedGetIBarRect( &drIBarPixels,
				&(sg.sgBegin), &(dc.dcLayoutContext) );

	    if  ( geoIntersectRectangle( &drIBarPixels,
					&drIBarPixels, drClipPixels )	&&
		  ! td->tdShowIBarId					)
		{ tedDrawIBar( &drIBarPixels, &(dc.dcLayoutContext) ); }
	    }
	else{
	    InsertedObject *	io;
	    int			partO;
	    DocumentPosition	dpO;

	    docInitDocumentPosition( &dpO );

	    if  ( sd.sdIsObjectSelection				&&
		  ! docGetObjectSelection( &ds, bd, &partO, &dpO, &io  ) )
		{
		PositionGeometry	pgO;
		const int		lastOne= 1;

		APP_POINT		xp[RESIZE_COUNT];
		DocumentRectangle	drObject;

		if  ( selRootBi->biInExternalItem != DOCinBODY		&&
		      tedDrawCheckPageOfSelectedExtItem( &sg, &ds,
				    selRootEi, &(dc.dcLayoutContext) )	)
		    { LDEB(1); return; }

		sdd.sddForeColor.rgb8Red= 0;
		sdd.sddForeColor.rgb8Green= 0;
		sdd.sddForeColor.rgb8Blue= 0;

		docDrawSetColorRgb( &dc, (void *)&sdd, &(sdd.sddForeColor) );

		sdd.sddBackColor= selColor;

		tedPositionGeometry( &pgO, &dpO, bodySectBi,
					    lastOne, &(dc.dcLayoutContext) );

		tedGetObjectRectangle( &drObject, xp, io, &pgO,
					    &(dc.dcLayoutContext), td );

		if  ( geoIntersectRectangle( (DocumentRectangle *)0,
						&drObject, drClipPixels )	)
		    { tedDrawObjectBlocks( &drObject, xp, &dc, &sdd );	}
		}
	    }
	}

    if  ( ed->edSelectRectangle.srDirection != DOCselNONE )
	{
	DocumentRectangle	drHair;
	int			blackSet= 0;

	drHair= ed->edVisibleRect;
	drHair.drX0= ed->edSelectRectangle.srSelected.drX0+
					ed->edSelectRectangle.srLTM.drX0;
	drHair.drX1= ed->edSelectRectangle.srSelected.drX0+
					ed->edSelectRectangle.srLTM.drX0;

	if  ( geoIntersectRectangle( &drHair, &drHair, drClipPixels ) )
	    {
	    if  ( ! blackSet )
		{ appDrawSetForegroundBlack( add ); blackSet= 1;	}

	    appDrawFillRectangle( add,
			    drHair.drX0- ox, drHair.drY0- oy,
			    drHair.drX1- drHair.drX0+ 1,
			    drHair.drY1- drHair.drY0+ 1 );
	    }

	if  ( ed->edSelectRectangle.srLTM.drX1 !=
					    ed->edSelectRectangle.srLTM.drX0 )
	    {
	    drHair.drX0= ed->edSelectRectangle.srSelected.drX0+
					ed->edSelectRectangle.srLTM.drX1;
	    drHair.drX1= ed->edSelectRectangle.srSelected.drX0+
					ed->edSelectRectangle.srLTM.drX1;

	    if  ( geoIntersectRectangle( &drHair, &drHair, drClipPixels ) )
		{
		if  ( ! blackSet )
		    { appDrawSetForegroundBlack( add ); blackSet= 1;	}

		appDrawFillRectangle( add,
			    drHair.drX0- ox, drHair.drY0- oy,
			    drHair.drX1- drHair.drX0+ 1,
			    drHair.drY1- drHair.drY0+ 1 );
		}
	    }
	}

    appDrawNoClipping( add );

    return;
    }

/************************************************************************/
/*									*/
/*  Draw a shaded rectangle.						*/
/*									*/
/************************************************************************/

void tedDrawShadedRectangle(
			const LayoutContext *	lc,
			APP_BITMAP_IMAGE	shadingPixmaps[DOCsp_COUNT],
			int			pattern,
			int			x0,
			int			y0,
			int			x1,
			int			y1 )
    {
    if  ( ! shadingPixmaps[pattern] )
	{
	const int	wide= 8;
	const int	high= 8;
	const int	d= 4;
	int		t;
	int		s= 0;

	int		x;
	int		y;

	AppDrawingData	addPat;

	appInitDrawingData( &addPat );

	shadingPixmaps[pattern]= appMakePixmap( lc->lcAdd, wide, high );;
	if  ( ! shadingPixmaps[pattern] )
	    { XDEB(shadingPixmaps[pattern]); return;	}

	appCloneDrawingEnvironment( &addPat,
			lc->lcAdd, 1.0, 0.05, shadingPixmaps[pattern] );

	appDrawSetForegroundWhite( &addPat );
	appDrawFillRectangle( &addPat, 0, 0, wide, high );

	appDrawSetForegroundBlack( &addPat );

	switch( pattern )
	    {
	    case DOCspSOLID:
		LDEB(pattern);
		t= 0; s= 0;
		break;

	    case DOCspBGHORIZ:
	    case DOCspBGVERT:
	    case DOCspBGFDIAG:
	    case DOCspBGBDIAG:
	    case DOCspBGCROSS:
	    case DOCspBGDCROSS:
		t= 1; s= 0;
		break;

	    case DOCspBGDKHORIZ:
	    case DOCspBGDKVERT:
	    case DOCspBGDKFDIAG:
	    case DOCspBGDKBDIAG:
	    case DOCspBGDKCROSS:
	    case DOCspBGDKDCROSS:
		t= 2; s= 1;
		break;

	    default:
		LDEB(pattern);
		return;
	    }

	if  ( t > 0 )
	    {
	    appDrawSetLineAttributes( &addPat,
			t, LINEstyleSOLID, LINEcapPROJECTING, LINEjoinMITER,
			(unsigned char *)0, 0 );
	    }

	switch( pattern )
	    {
	    case DOCspSOLID:
		LDEB(pattern);
		break;

	    case DOCspBGHORIZ:
	    case DOCspBGDKHORIZ:
		y= 0;
		while( y < high )
		    {
		    appDrawDrawLine( &addPat, 0, y+ s, wide- 1, y+ s );
		    y += d;
		    }
		break;

	    case DOCspBGVERT:
	    case DOCspBGDKVERT:
		x= 0;
		while( x < wide )
		    {
		    appDrawDrawLine( &addPat, x+ s, 0, x+ s, high- 1 );
		    x += d;
		    }
		break;

	    case DOCspBGFDIAG:
	    case DOCspBGDKFDIAG:
		x= -high;
		while( x < wide )
		    {
		    appDrawDrawLine( &addPat, x+ s, 0, x+ high+ s, high );
		    x += d;
		    }
		break;

	    case DOCspBGBDIAG:
	    case DOCspBGDKBDIAG:
		x= 0;
		while( x < wide+ high )
		    {
		    appDrawDrawLine( &addPat, x, 0, x- high, high );
		    x += d;
		    }
		break;

	    case DOCspBGCROSS:
	    case DOCspBGDKCROSS:
		y= 0;
		while( y < high )
		    {
		    appDrawDrawLine( &addPat, 0, y+ s, wide- 1, y+ s );
		    y += d;
		    }
		x= 0;
		while( x < wide )
		    {
		    appDrawDrawLine( &addPat, x+ s, 0, x+ s, high- 1 );
		    x += d;
		    }
		break;

	    case DOCspBGDCROSS:
	    case DOCspBGDKDCROSS:
		x= -high;
		while( x < wide )
		    {
		    /*
		    appDrawDrawLine( &addPat, x+ s, 0, x+ high+ s, high );
		    */
		    appDrawDrawLine( &addPat, x, 0, x+ high, high );
		    x += d;
		    }
		x= 0;
		while( x < wide+ high )
		    {
		    appDrawDrawLine( &addPat, x, 0, x- high, high );
		    x += d;
		    }
		break;

	    default:
		LDEB(pattern);
		return;
	    }

	appCleanDrawingData( &addPat );
	}

#   ifdef USE_MOTIF
    XSetFillStyle( lc->lcAdd->addDisplay, lc->lcAdd->addGc, FillTiled );
    XSetTile( lc->lcAdd->addDisplay, lc->lcAdd->addGc, shadingPixmaps[pattern] );
#   endif
#   ifdef USE_GTK
    gdk_gc_set_fill( lc->lcAdd->addGc, GDK_TILED );
    gdk_gc_set_tile( lc->lcAdd->addGc, shadingPixmaps[pattern] );
#   endif

    appDrawFillRectangle( lc->lcAdd, x0, y0, x1- x0+ 1, y1- y0+ 1 );

#   ifdef USE_MOTIF
    XSetFillStyle( lc->lcAdd->addDisplay, lc->lcAdd->addGc, FillSolid );
#   endif
#   ifdef USE_GTK
    gdk_gc_set_fill( lc->lcAdd->addGc, GDK_SOLID );
#   endif

    return;
    }


/************************************************************************/
/*									*/
/*  Draw actual lines for a segment in a horizontal border.		*/
/*									*/
/************************************************************************/

static void tedBorderSetLineSolid(	AppDrawingData *	add,
					int			thick )
    {
    appDrawSetLineAttributes( add,
		    thick, LINEstyleSOLID, LINEcapBUTT, LINEjoinMITER,
		    (unsigned char *)0, 0 );
    }

static void tedBorderSetLineDashes(	AppDrawingData *	add,
					const unsigned char *	dashes,
					int			dashCount,
					int			thick )
    {
    appDrawSetLineAttributes( add,
		    thick, LINEstyleON_OFF_DASH, LINEcapBUTT, LINEjoinMITER,
		    dashes, dashCount );
    }

static const unsigned char TED_Border_DOT[]=	{ 1, 2 };
static const unsigned char TED_Border_DASH[]=	{ 3, 3 };
static const unsigned char TED_Border_DASHSM[]=	{ 3, 1 };
static const unsigned char TED_Border_DASHD[]=	{ 4, 2, 2, 2 };
static const unsigned char TED_Border_DASHDD[]=	{ 4, 2, 2, 2, 2, 2 };

void tedDrawHorizontalBorderLine(	AppDrawingData *	add,
					int			style,
					int			minThick,
					int			x0,
					int			x1,
					int			y )
    {
    int		yy;
    int		xx;

    switch( style )
	{
	case DOCbsNONE:
	    break;

	case DOCbsS:
	    tedBorderSetLineSolid( add, 1 );
	    appDrawDrawLine( add, x0, y, x1, y );
	    break;

	case DOCbsTH:
	    tedBorderSetLineSolid( add, 2 );
	    appDrawDrawLine( add, x0, y, x1, y );
	    break;

	case DOCbsSH:
	    tedBorderSetLineSolid( add, 2 );
	    appDrawDrawLine( add, x0, y, x1, y );
	    break;

	case DOCbsDB:
	    tedBorderSetLineSolid( add, 1 );
	    yy= y- 1;
	    appDrawDrawLine( add, x0, yy, x1, yy );
	    yy= y+ 1;
	    appDrawDrawLine( add, x0, yy, x1, yy );
	    break;

	case DOCbsHAIR:
	    tedBorderSetLineSolid( add, 1 );
	    appDrawDrawLine( add, x0, y, x1, y );
	    break;

	case DOCbsDOT:
	    tedBorderSetLineDashes( add,
			    TED_Border_DOT, sizeof( TED_Border_DOT ), 1 );
	    appDrawDrawLine( add, x0, y, x1, y );
	    break;

	case DOCbsDASH:
	    tedBorderSetLineDashes( add,
			    TED_Border_DASH, sizeof( TED_Border_DASH ), 1 );
	    appDrawDrawLine( add, x0, y, x1, y );
	    break;

	case DOCbsDASHD:
	    tedBorderSetLineDashes( add,
			TED_Border_DASHD, sizeof( TED_Border_DASHD ), 1 );
	    appDrawDrawLine( add, x0, y, x1, y );
	    break;

	case DOCbsDASHDD:
	    tedBorderSetLineDashes( add,
			TED_Border_DASHDD, sizeof( TED_Border_DASHDD ), 1 );
	    appDrawDrawLine( add, x0, y, x1, y );
	    break;

	case DOCbsTRIPLE:
	    tedBorderSetLineSolid( add, 1 );
	    yy= y- 2;
	    appDrawDrawLine( add, x0, yy, x1, yy );
	    yy= y;
	    appDrawDrawLine( add, x0, yy, x1, yy );
	    yy= y+ 2;
	    appDrawDrawLine( add, x0, yy, x1, yy );
	    break;

	case DOCbsTNTHSG:
	    tedBorderSetLineSolid( add, 1 );
	    yy= y- 2;
	    appDrawDrawLine( add, x0, yy, x1, yy );
	    tedBorderSetLineSolid( add, 2 );
	    yy= y+ 1;
	    appDrawDrawLine( add, x0, yy, x1, yy );
	    break;

	case DOCbsTHTNSG:
	    tedBorderSetLineSolid( add, 2 );
	    yy= y- 1;
	    appDrawDrawLine( add, x0, yy, x1, yy );
	    tedBorderSetLineSolid( add, 1 );
	    yy= y+ 1;
	    appDrawDrawLine( add, x0, yy, x1, yy );
	    break;

	case DOCbsTNTHTNSG:
	    tedBorderSetLineSolid( add, 1 );
	    yy= y- 3;
	    appDrawDrawLine( add, x0, yy, x1, yy );
	    tedBorderSetLineSolid( add, 2 );
	    yy= y+ 0;
	    appDrawDrawLine( add, x0, yy, x1, yy );
	    tedBorderSetLineSolid( add, 1 );
	    yy= y+ 2;
	    appDrawDrawLine( add, x0, yy, x1, yy );
	    break;

	case DOCbsTNTHMG:
	    tedBorderSetLineSolid( add, 1 );
	    yy= y- 2;
	    appDrawDrawLine( add, x0, yy, x1, yy );
	    tedBorderSetLineSolid( add, 3 );
	    yy= y+ 1;
	    appDrawDrawLine( add, x0, yy, x1, yy );
	    break;

	case DOCbsTHTNMG:
	    tedBorderSetLineSolid( add, 3 );
	    yy= y- 2;
	    appDrawDrawLine( add, x0, yy, x1, yy );
	    tedBorderSetLineSolid( add, 1 );
	    yy= y+ 1;
	    appDrawDrawLine( add, x0, yy, x1, yy );
	    break;

	case DOCbsTNTHTNMG:
	    tedBorderSetLineSolid( add, 1 );
	    yy= y- 3;
	    appDrawDrawLine( add, x0, yy, x1, yy );
	    tedBorderSetLineSolid( add, 3 );
	    yy= y+ 0;
	    appDrawDrawLine( add, x0, yy, x1, yy );
	    tedBorderSetLineSolid( add, 1 );
	    yy= y+ 3;
	    appDrawDrawLine( add, x0, yy, x1, yy );
	    break;

	case DOCbsTNTHLG:
	    tedBorderSetLineSolid( add, 1 );
	    yy= y- 3;
	    appDrawDrawLine( add, x0, yy, x1, yy );
	    tedBorderSetLineSolid( add, 4 );
	    yy= y+ 2;
	    appDrawDrawLine( add, x0, yy, x1, yy );
	    break;

	case DOCbsTHTNLG:
	    tedBorderSetLineSolid( add, 4 );
	    yy= y- 2;
	    appDrawDrawLine( add, x0, yy, x1, yy );
	    tedBorderSetLineSolid( add, 1 );
	    yy= y+ 2;
	    appDrawDrawLine( add, x0, yy, x1, yy );
	    break;

	case DOCbsTNTHTNLG:
	    tedBorderSetLineSolid( add, 1 );
	    yy= y- 4;
	    appDrawDrawLine( add, x0, yy, x1, yy );
	    tedBorderSetLineSolid( add, 4 );
	    yy= y+ 0;
	    appDrawDrawLine( add, x0, yy, x1, yy );
	    tedBorderSetLineSolid( add, 1 );
	    yy= y+ 3;
	    appDrawDrawLine( add, x0, yy, x1, yy );
	    break;

	case DOCbsWAVY:
	    tedBorderSetLineSolid( add, 1 );
	    xx= x0;
#	    define	W	2
#	    define	W2	4
	    appDrawDrawLine( add, xx, y, xx+ W, y+ W ); xx += W;
	    while( xx < x1 )
		{
		appDrawDrawLine( add, xx, y+ W, xx+ W2, y- W ); xx += W2;
		appDrawDrawLine( add, xx, y- W, xx+ W2, y+ W ); xx += W2;
		}
	    break;

	case DOCbsWAVYDB:
	    tedBorderSetLineSolid( add, 1 );
	    yy= y- 2;
	    xx= x0;
	    appDrawDrawLine( add, xx, yy, xx+ W, yy+ W ); xx += W;
	    while( xx < x1 )
		{
		appDrawDrawLine( add, xx, yy+ W, xx+ W2, yy- W ); xx += W2;
		appDrawDrawLine( add, xx, yy- W, xx+ W2, yy+ W ); xx += W2;
		}

	    yy= y+ 2;
	    xx= x0;
	    appDrawDrawLine( add, xx, yy, xx+ W, yy+ W ); xx += W;
	    while( xx < x1 )
		{
		appDrawDrawLine( add, xx, yy+ W, xx+ W2, yy- W ); xx += W2;
		appDrawDrawLine( add, xx, yy- W, xx+ W2, yy+ W ); xx += W2;
		}
	    break;

	case DOCbsDASHSM:
	    tedBorderSetLineDashes( add,
			TED_Border_DASHSM, sizeof( TED_Border_DASHSM ), 1 );
	    appDrawDrawLine( add, x0, y, x1, y );
	    break;

	case DOCbsDASHDOTSTR:
	    xx= x0- 2;
	    while( xx < x1 )
		{
		appDrawDrawLine( add, xx, y+ 1, xx+ 3, y- 2 ); xx += 2;
		appDrawDrawLine( add, xx, y+ 1, xx+ 3, y- 2 ); xx += 1;
		appDrawDrawLine( add, xx, y+ 1, xx+ 3, y- 2 ); xx += 2;
		}
	    break;

	case DOCbsEMBOSS:
	case DOCbsENGRAVE:
	case DOCbsFRAME:
	case DOCbsOUTSET:
	    tedBorderSetLineSolid( add, 1 );
	    appDrawDrawLine( add, x0, y, x1, y );
	    break;

	case DOCbsTBL:
	case DOCbsNIL:
	    /*no border*/
	    break;

	default:
	    LDEB(style);
	    break;
	}

    return;
    }

/************************************************************************/
/*									*/
/*  Drawing of OS/GUI images.						*/
/*									*/
/************************************************************************/

void tedDrawPixmap(		const LayoutContext *		lc,
				const ScreenPixmap		sp,
				int				xSrc,
				int				ySrc,
				int				wide,
				int				high,
				int				xDest,
				int				yDest )
    {
    if  ( sp )
	{
	appDrawDrawPixmap( lc->lcAdd, sp,
		xSrc, ySrc, /* src: x,y */
		wide, high, /* wide, high */
		xDest, yDest ); /* dest: x,y */
	}

    return;
    }
