/************************************************************************/
/*									*/
/*  Ted: Copy/Paste related functionality.				*/
/*									*/
/************************************************************************/

#   include	"tedConfig.h"

#   include	<stddef.h>
#   include	<stdlib.h>
#   include	<stdio.h>
#   include	<string.h>
#   include	<ctype.h>

#   include	<utilOfficeCharset.h>
#   include	"tedApp.h"
#   include	"tedEdit.h"
#   include	"tedLayout.h"
#   include	<appMatchFont.h>
#   include	"docRtfReadWrite.h"

#   include	<sioMemory.h>
#   include	<sioXprop.h>
#   include	<sioStdio.h>

#   include	<appDebugon.h>

/************************************************************************/
/*									*/
/*  Write copy/paste to file for debugging purposes.			*/
/*									*/
/************************************************************************/

static void tedSaveSelectionToFile(
				BufferDocument *		bd,
				const DocumentSelection *	ds,
				int				rtfFlags,
				const char *			filename )
    {
    SimpleOutputStream *	sos;

    sos= sioOutStdioOpen( filename );
    if  ( ! sos )
	{ SPDEB( filename, sos );	}
    else{
	if  ( docRtfSaveDocument( sos, bd, ds, rtfFlags ) )
	    { LDEB(1); 	}
	if  ( sioOutClose( sos ) )
	    { LDEB(1);	}
	}

    return;
    }

/************************************************************************/
/*									*/
/*  Edit menu actions.							*/
/*									*/
/*  8)  Also put a copied picture on the clipboard that 'xv' implements	*/
/*	on the root window.						*/
/*									*/
/************************************************************************/

static int tedTextCopyClosed(	TedDocument *			td,
				const LayoutContext *		lc,
				const DocumentSelection *	ds,
				const BufferItem *		bodySectBi )
    {
    PositionGeometry		pgBegin;
    PositionGeometry		pgEnd;
    const int			lastLine= 1;

    int				copyClosed= 1;

    tedPositionGeometry( &pgBegin, &(ds->dsHead), bodySectBi,
					ds->dsDirection > 0 || lastLine,
					lc );
    tedPositionGeometry( &pgEnd, &(ds->dsTail), bodySectBi,
					ds->dsDirection > 0 || lastLine,
					lc );

    if  ( ds->dsTail.dpStroff < docParaStrlen( ds->dsTail.dpBi ) )
	{ copyClosed= 0;	}

    if  ( docIsParaSelection( ds )		&&
	  pgBegin.pgLine == pgEnd.pgLine	)
	{ copyClosed= 0;	}

    return copyClosed;
    }

static int tedDocSaveSelection(	MemoryBuffer *		mb,
				int *			pClosed,
				DocumentSelection *	ds,
				SelectionDescription *	sd,
				const LayoutContext *	lc,
				TedDocument *		td )
    {
    SimpleOutputStream *	sos;

    SelectionGeometry		sg;
    const BufferItem *		bodySectBi;

    const int			rtfFlags= RTFflagNO_BOOKMARKS|RTFflagCLOSE_LAST;

    if  ( ! tedHasSelection( td ) || tedHasIBarSelection( td ) )
	{ return -1; }

    if  ( tedGetSelection( ds, &sg, sd, (DocumentTree **)0, &bodySectBi, td ) )
	{ return -1;	}

    sos= sioOutMemoryOpen( mb );
    if  ( ! sos )
	{ XDEB(sos); return -1;    }

    if  ( docRtfSaveDocument( sos, td->tdDocument, ds, rtfFlags ) )
	{ LDEB(1); sioOutClose( sos ); return -1;	}

    if  ( sioOutClose( sos ) )
	{ LDEB(1); return -1;	}

    if  ( getenv( "TED_SAVE_COPIES" ) )
	{
	tedSaveSelectionToFile( td->tdDocument, ds, rtfFlags,
							"/tmp/copied.rtf" );
	}

    *pClosed= tedTextCopyClosed( td, lc, ds, bodySectBi );
    return 0;
    }

static int tedDocCopySelection(	EditDocument *	ed )
    {
    TedDocument *		td= (TedDocument *)ed->edPrivateData;

    DocumentPosition		dpObject;
    int				partObject;
    InsertedObject *		io;

    DocumentSelection		ds;
    SelectionDescription	sd;

    LayoutContext		lc;

    layoutInitContext( &lc );
    tedSetLayoutContext( &lc, ed );

    if  ( tedDocSaveSelection( &(td->tdCopiedSelection),
				&(td->tdCopiedSelectionClosed),
				&ds, &sd, &lc, td ) )
	{ LDEB(1); return -1;	}

    bmCleanBitmapImage( &(td->tdCopiedImage) );
    bmInitBitmapImage( &(td->tdCopiedImage) );

    docInitDocumentPosition( &dpObject );

    if  ( sd.sdIsObjectSelection					&&
	  ! docGetObjectSelection( &ds, td->tdDocument,
				      &partObject, &dpObject, &io )	)
	{
	if  ( tedSaveObjectPicture( &(td->tdCopiedImage), io )	)
	    { LDEB(1);	}
	}

    if  ( appDocOwnSelection( ed, "CLIPBOARD",
					TedClipboardTextTargets,
					TedClipboardTextTargetCount ) )
	{ LDEB(1); return -1;	}

    return 0;
    }


void tedDocCut(			EditDocument *	ed )
    {
    TedDocument *		td= (TedDocument *)ed->edPrivateData;

    int				isTableSlice= 0;
    int				isRowSlice= 0;
    int				isColSlice= 0;
    TableRectangle		tr;

    DocumentSelection		ds;
    SelectionGeometry		sg;
    SelectionDescription	sd;

    if  ( tedGetSelection( &ds, &sg, &sd,
			    (DocumentTree **)0, (const BufferItem **)0, td ) )
	{ return;	}

    if  ( ! docGetTableSliceSelection( &isRowSlice, &isColSlice, &tr, &ds ) )
	{ isTableSlice= 1;	}

    if  ( ! isTableSlice && ! td->tdCanReplaceSelection )
	{ LDEB(td->tdCanReplaceSelection); return;	}

    if  ( tedDocCopySelection( ed ) )
	{ LDEB(1); return;	}

    if  ( isTableSlice )
	{ tedDeleteTableSliceSelection( ed );			}
    else{ tedDocReplaceSelection( ed, (const char *)0, 0 );	}

    return;
    }

void tedDocCopy(		EditDocument *	ed )
    {
    TedDocument *		td= (TedDocument *)ed->edPrivateData;

    DocumentSelection		ds;
    SelectionGeometry		sg;
    SelectionDescription	sd;

    const int			scrolledX= 0;
    const int			scrolledY= 0;

    if  ( tedGetSelection( &ds, &sg, &sd,
			    (DocumentTree **)0, (const BufferItem **)0, td ) )
	{ return;	}

    if  ( tedDocCopySelection( ed ) )
	{ return;	}

    td->tdVisibleSelectionCopied= 1;

    appDocExposeRectangle( ed, &(sg.sgRectangle), scrolledX, scrolledY );

    return;
    }

/************************************************************************/
/*									*/
/*  Ted, Copy/Paste functionality.					*/
/*									*/
/************************************************************************/

void tedClipboardLost(	APP_WIDGET			w,
			void *				voided,
			APP_EVENT *			event )
    {
    EditDocument *		ed= (EditDocument *)voided;
    TedDocument *		td= (TedDocument *)ed->edPrivateData;

    DocumentSelection		ds;
    SelectionGeometry		sg;
    SelectionDescription	sd;

    if  ( tedGetSelection( &ds, &sg, &sd,
			    (DocumentTree **)0, (const BufferItem **)0, td ) )
	{ return;	}

    td->tdVisibleSelectionCopied= 0;

    if  ( tedHasSelection( td ) && ! sd.sdIsIBarSelection )
	{
	const int	scrolledX= 0;
	const int	scrolledY= 0;

	appDocExposeRectangle( ed, &(sg.sgRectangle), scrolledX, scrolledY );
	}
    }

void tedPrimaryLost(	APP_WIDGET			w,
			void *				voided,
			APP_EVENT *			event )
    {
    EditDocument *		ed= (EditDocument *)voided;
    TedDocument *		td= (TedDocument *)ed->edPrivateData;

    td->tdOwnsPrimarySelection= 0;

    return;
    }

/************************************************************************/
/*									*/
/*  Return the selection to a requestor.				*/
/*									*/
/************************************************************************/

APP_GIVE_COPY( tedCopyClipboardRtf, w, event, voided )
    {
    EditDocument *	ed= (EditDocument *)voided;
    TedDocument *	td= (TedDocument *)ed->edPrivateData;

    SimpleOutputStream *	sos;

    if  ( td->tdCopiedSelection.mbSize == 0 )
	{ LDEB(td->tdCopiedSelection.mbSize); return; }

    sos= sioOutOpenCopy( w, event );
    if  ( ! sos )
	{ XDEB(sos); return;	}

    if  ( sioOutWriteBytes( sos, td->tdCopiedSelection.mbBytes,
				    td->tdCopiedSelection.mbSize ) <= 0 )
	{ LDEB(td->tdCopiedSelection.mbSize); sioOutClose( sos ); return; }

    sioOutClose( sos );

    if  ( getenv( "TED_SAVE_COPIES" ) )
	{
	sos= sioOutStdioOpen( "/tmp/copied.rtf" );
	XDEB(sos);
	if  ( sos )
	    {
	    if  ( sioOutWriteBytes( sos, td->tdCopiedSelection.mbBytes,
					td->tdCopiedSelection.mbSize ) <= 0 )
		{ LDEB(td->tdCopiedSelection.mbSize); }

	    sioOutClose( sos );
	    }
	}

    return;
    }

static void tedCopyString(	const MemoryBuffer *	mb,
				int			closed,
				APP_WIDGET		w,
				APP_SELECTION_EVENT *	event,
				EditDocument *		ed )
    {
    EditApplication *		ea= ed->edApplication;

    SimpleOutputStream *	sos;

    BufferDocument *		bd;
    SimpleInputStream *		sis;
    int				res;

    if  ( mb->mbSize == 0 )
	{ LDEB(mb->mbSize); return; }

    sos= sioOutOpenCopy( w, event );
    if  ( ! sos )
	{ XDEB(sos); return;	}

    sis= sioInMemoryOpen( mb );
    if  ( ! sis )
	{ XDEB(sis); sioOutClose( sos ); return;	}

    bd= docRtfReadFile( sis, &(ea->eaPostScriptFontList) );
    sioInClose( sis );

    if  ( ! bd )
	{ XDEB(bd); sioOutClose( sos ); return; }

    res= docPlainSaveDocument( sos, bd, (DocumentSelection *)0, 0, closed );

    docFreeDocument( bd );
    sioOutClose( sos );

    return;
    }

APP_GIVE_COPY( tedCopyPrimaryString, w, event, voided )
    {
    EditDocument *		ed= (EditDocument *)voided;
    TedDocument *		td= (TedDocument *)ed->edPrivateData;

    DocumentSelection		ds;
    SelectionDescription	sd;

    MemoryBuffer		mb;
    int				closed= 1;

    LayoutContext		lc;

    layoutInitContext( &lc );
    tedSetLayoutContext( &lc, ed );

    utilInitMemoryBuffer( &mb );

    if  ( ! tedDocSaveSelection( &mb, &closed, &ds, &sd, &lc, td ) )
	{ tedCopyString( &mb, closed, w, event, ed );	}

    utilCleanMemoryBuffer( &mb );

    return;
    }

APP_GIVE_COPY( tedCopyClipboardString, w, event, voided )
    {
    EditDocument *		ed= (EditDocument *)voided;
    TedDocument *		td= (TedDocument *)ed->edPrivateData;

    tedCopyString( &(td->tdCopiedSelection),
				td->tdCopiedSelectionClosed, w, event, ed );
    return;
    }

APP_GIVE_COPY( tedCopyClipboardPng, w, event, voided )
    {
    EditDocument *		ed= (EditDocument *)voided;
    TedDocument *		td= (TedDocument *)ed->edPrivateData;

    SimpleOutputStream *	sos;

    if  ( ! td->tdCopiedImage.abiBuffer )
	{ XDEB(td->tdCopiedImage.abiBuffer); return;	}

    sos= sioOutOpenCopy( w, event );
    if  ( ! sos )
	{ XDEB(sos); return;	}

    if  ( bmPngWritePng( &(td->tdCopiedImage.abiBitmap),
					td->tdCopiedImage.abiBuffer, sos ) )
	{ LDEB(1); sioOutClose( sos ); return;	}

    sioOutClose( sos );

    return;
    }

APP_GIVE_COPY( tedCopyClipboardPixmap, w, event, voided )
    {
    EditDocument *	ed= (EditDocument *)voided;
    AppDrawingData *	add= &(ed->edDocumentWidget.dwDrawingData);
    AppColors *		ac= &(ed->edDocumentWidget.dwColors);
    TedDocument *	td= (TedDocument *)ed->edPrivateData;

    APP_BITMAP_IMAGE	pixmapCopied;

    int			wide= td->tdCopiedImage.abiBitmap.bdPixelsWide;
    int			high= td->tdCopiedImage.abiBitmap.bdPixelsHigh;

    if  ( appImgMakePixmap( add, &pixmapCopied, wide, high,
		    ac, &(td->tdCopiedImage), (const DocumentRectangle *)0 ) )
	{
	bmCleanBitmapImage( &(td->tdCopiedImage) );
	bmInitBitmapImage( &(td->tdCopiedImage) );
	return;
	}

    appCopyPixmapValue( event, pixmapCopied );

    return;
    }

APP_GIVE_COPY( tedCopyFontTed, w, event, voided )
    {
    EditDocument *		ed= (EditDocument *)voided;
    TedDocument *		td= (TedDocument *)ed->edPrivateData;

    SimpleOutputStream *	sos;

    if  ( td->tdCopiedFont.mbSize == 0 )
	{ LDEB(td->tdCopiedFont.mbSize); return; }

    sos= sioOutOpenCopy( w, event );
    if  ( ! sos )
	{ XDEB(sos); return;	}

    if  ( sioOutWriteBytes( sos, td->tdCopiedFont.mbBytes,
					td->tdCopiedFont.mbSize ) <= 0 )
	{ LDEB(1); sioOutClose( sos ); return;	}

    sioOutClose( sos );

    return;
    }

/************************************************************************/
/*									*/
/*  Copy the ruler as it was saved on the document struct when the	*/
/*  menu option was clicked.						*/
/*									*/
/************************************************************************/

APP_GIVE_COPY( tedCopyRulerTed, w, event, voided )
    {
    EditDocument *		ed= (EditDocument *)voided;
    TedDocument *		td= (TedDocument *)ed->edPrivateData;

    SimpleOutputStream *	sos;

    if  ( td->tdCopiedRuler.mbSize == 0 )
	{ LDEB(td->tdCopiedRuler.mbSize); return; }

    sos= sioOutOpenCopy( w, event );
    if  ( ! sos )
	{ XDEB(sos); return;	}

    if  ( sioOutWriteBytes( sos, td->tdCopiedRuler.mbBytes,
					td->tdCopiedRuler.mbSize ) <= 0 )
	{ LDEB(1); sioOutClose( sos ); return;	}

    sioOutClose( sos );

    return;
    }

/************************************************************************/
/*									*/
/*  Finish a paste action.						*/
/*									*/
/************************************************************************/

typedef BufferDocument * (*TED_READ) (
				SimpleInputStream *		sis,
				EditApplication *		ea,
				const DocumentGeometry *	dg );

typedef int	(*TED_INCLUDE) (	EditDocument *		ed,
					BufferDocument *	bd );

static BufferDocument * tedPasteRtfReadFile(
				SimpleInputStream *		sis,
				EditApplication *		ea,
				const DocumentGeometry *	dg )
    {
    return docRtfReadFile( sis, &(ea->eaPostScriptFontList) );
    }

static BufferDocument * tedPastePlainReadFile(
				SimpleInputStream *		sis,
				EditApplication *		ea,
				const DocumentGeometry *	dg )
    {
    int		longestPara;

    return docPlainReadFile( sis, &longestPara, dg );
    }

static void tedPasteClipboardGeneric(	APP_WIDGET		w,
					EditDocument *		ed,
					APP_SELECTION_EVENT *	event,
					TED_READ		readDoc,
					TED_INCLUDE		includeDoc )
    {
    EditApplication *		ea= ed->edApplication;

    SimpleInputStream *		sis;

    BufferDocument *		bd;

    TedAppResources *		tar= (TedAppResources *)ea->eaResourceData;

    tedDetermineDefaultSettings( tar );

    sis= sioInOpenPaste( w, event );
    if  ( ! sis )
	{ XDEB(sis); return;	}

    bd= (*readDoc)( sis, ea, &(ea->eaDefaultDocumentGeometry) );

    sioInClose( sis );

    if  ( ! bd )
	{ XDEB( bd ); return; }

    if  ( getenv( "TED_SAVE_COPIES" ) )
	{
	const int			rtfFlags= 0;
	const DocumentSelection * const	ds= (const DocumentSelection *)0;

	tedSaveSelectionToFile( bd, ds, rtfFlags, "/tmp/pasted.rtf" );
	}

    if  ( (*includeDoc)( ed, bd ) )
	{ LDEB(1); docFreeDocument( bd ); return;	}

    docFreeDocument( bd );

    appDocumentChanged( ed, 1 );

    return;
    }

APP_PASTE_REPLY( tedPasteClipboardRtf, w, event, voided )
    {
    EditDocument *	ed= (EditDocument *)voided;

    tedPasteClipboardGeneric( w, ed, event,
			    tedPasteRtfReadFile, tedIncludeRtfDocument );
    }

APP_PASTE_REPLY( tedPasteClipboardString, w, event, voided )
    {
    EditDocument *	ed= (EditDocument *)voided;

    tedPasteClipboardGeneric( w, ed, event,
			    tedPastePlainReadFile, tedIncludePlainDocument );
    }

APP_PASTE_REPLY( tedPastePrimaryString, w, event, voided )
    {
    EditDocument *	ed= (EditDocument *)voided;

    tedPasteClipboardGeneric( w, ed, event,
			    tedPastePlainReadFile, tedIncludePlainDocument );
    }

APP_PASTE_REPLY( tedPasteClipboardPng, w, event, voided )
    {
    EditDocument *	ed= (EditDocument *)voided;
    TedDocument *	td= (TedDocument *)ed->edPrivateData;

    SimpleInputStream *	sis;

    AppBitmapImage *	abi;

    if  ( ! td->tdCanReplaceSelection )
	{ LDEB(td->tdCanReplaceSelection); return;	}

    abi= (AppBitmapImage *)malloc( sizeof(AppBitmapImage) );
    if  ( ! abi )
	{ XDEB(abi); return;	}
    bmInitBitmapImage( abi );

    sis= sioInOpenPaste( w, event );
    if  ( ! sis )
	{ XDEB(sis); return;	}

    if  ( bmPngReadPng( &(abi->abiBitmap), &(abi->abiBuffer), sis ) )
	{ LDEB(1); return;	}

    sioInClose( sis );

    if  ( tedReplaceSelectionWithBitmapImage( ed, abi ) )
	{ LDEB(1); return; }

    return;
    }

APP_PASTE_REPLY( tedPasteClipboardPixmap, w, event, voided )
    {
    EditDocument *	ed= (EditDocument *)voided;
    TedDocument *	td= (TedDocument *)ed->edPrivateData;

    AppBitmapImage *	abi;

    if  ( ! td->tdCanReplaceSelection )
	{ LDEB(td->tdCanReplaceSelection); return;	}

    abi= (AppBitmapImage *)malloc( sizeof(AppBitmapImage) );
    if  ( ! abi )
	{ XDEB(abi); return;	}
    bmInitBitmapImage( abi );

    if  ( appImgPastePixmap( abi, w, event ) )
	{ LDEB(1); return;	}

    if  ( tedReplaceSelectionWithBitmapImage( ed, abi ) )
	{ LDEB(1); return; }

    return;
    }

static int tedApplyPastedRuler(		EditDocument *		ed,
					BufferDocument *	bdFrom )
    {
    int				rval= 0;

    DocumentPosition		dp;

    TextAttribute		taSet;
    ParagraphProperties		ppSetMapped;

    PropertyMask		taSetMask;
    PropertyMask		ppUpdMask;
    PropertyMask		spUpdMask;
    PropertyMask		ppMappedMask;

    TedEditOperation		teo;
    EditOperation *		eo= &(teo.teoEo);
    SelectionGeometry		sg;
    SelectionDescription	sd;
    DocumentCopyJob		dcj;

    docInitDocumentCopyJob( &dcj );
    docInitParagraphProperties( &ppSetMapped );

    tedStartEditOperation( &teo, &sg, &sd, ed, 1 );

    if  ( docSet2DocumentCopyJob( &dcj, eo, bdFrom, ed->edFilename ) )
	{ LDEB(1); rval= -1; goto ready;	}

    if  ( docFirstDocumentPosition( &dp, bdFrom ) )
	{ LDEB(1); rval= -1; goto ready;	}

    utilPropMaskClear( &taSetMask );
    utilPropMaskClear( &spUpdMask );
    utilPropMaskClear( &ppMappedMask );

    utilInitTextAttribute( &taSet );

    utilPropMaskClear( &ppUpdMask );
    PROPmaskADD( &ppUpdMask, PPpropLEFT_INDENT );
    PROPmaskADD( &ppUpdMask, PPpropFIRST_INDENT );
    PROPmaskADD( &ppUpdMask, PPpropRIGHT_INDENT );
    PROPmaskADD( &ppUpdMask, PPpropALIGNMENT );
    PROPmaskADD( &ppUpdMask, PPpropTAB_STOPS );

    PROPmaskADD( &ppUpdMask, PPpropLISTOVERRIDE );
    PROPmaskADD( &ppUpdMask, PPpropOUTLINELEVEL );
    PROPmaskADD( &ppUpdMask, PPpropLISTLEVEL );

    if  ( docUpdParaProperties( &ppMappedMask, &ppSetMapped,
				    &ppUpdMask, &(dp.dpBi->biParaProperties),
						    &(dcj.dcjAttributeMap ) ) )
	{ LDEB(1); rval= -1; goto ready; }

    if  ( tedChangeSelectionProperties( ed,
				    &taSetMask, &taSet,
				    &ppUpdMask, &ppSetMapped,
				    &spUpdMask, (SectionProperties *)0 ) )
	{ LDEB(1); rval= -1; goto ready; }

  ready:

    docCleanParagraphProperties( &ppSetMapped );
    docCleanDocumentCopyJob( &dcj );

    return rval;
    }

APP_PASTE_REPLY( tedPasteRulerTed, w, event, voided )
    {
    EditDocument *	ed= (EditDocument *)voided;

    tedPasteClipboardGeneric( w, ed, event,
			    tedPasteRtfReadFile, tedApplyPastedRuler );
    return;
    }

/************************************************************************/

static int tedApplyPastedFont(		EditDocument *		ed,
					BufferDocument *	bdFrom )
    {
    int				rval= 0;

    DocumentSelection		dsAll;

    TextAttribute		taSetFrom;
    TextAttribute		taSetTo;
    ParagraphProperties		ppSet;

    PropertyMask		taSetMask;
    PropertyMask		ppUpdMask;
    PropertyMask		spUpdMask;
    PropertyMask		ppMappedMask;

    TedEditOperation		teo;
    EditOperation *		eo= &(teo.teoEo);
    SelectionGeometry		sg;
    SelectionDescription	sd;
    DocumentCopyJob		dcj;

    docInitDocumentSelection( &dsAll );

    docInitDocumentCopyJob( &dcj );
    docInitParagraphProperties( &ppSet );

    tedStartEditOperation( &teo, &sg, &sd, ed, 1 );

    if  ( docSet2DocumentCopyJob( &dcj, eo, bdFrom, ed->edFilename ) )
	{ LDEB(1); rval= -1; goto ready;	}

    if  ( docSelectWholeTree( &dsAll, bdFrom ) )
	{ LDEB(1); rval= -1; goto ready;	}

    utilPropMaskClear( &ppUpdMask );
    utilPropMaskClear( &taSetMask );
    utilPropMaskClear( &spUpdMask );
    utilPropMaskClear( &ppMappedMask );

    utilInitTextAttribute( &taSetFrom );
    utilInitTextAttribute( &taSetTo );

    docGetSelectionAttributes( bdFrom, &dsAll, &taSetMask, &taSetFrom );
    docMapTextAttribute( &taSetTo, &taSetFrom, &dcj );

    {
    PropertyMask		taOnlyMask;

    utilPropMaskClear( &taOnlyMask );

    PROPmaskADD( &taOnlyMask, TApropDOC_FONT_NUMBER );
    PROPmaskADD( &taOnlyMask, TApropFONTSIZE );
    PROPmaskADD( &taOnlyMask, TApropFONTBOLD );
    PROPmaskADD( &taOnlyMask, TApropFONTSLANTED );
    PROPmaskADD( &taOnlyMask, TApropTEXTUNDERLINED );
    PROPmaskADD( &taOnlyMask, TApropSTRIKETHROUGH );

    /* PROPmaskADD( &taOnlyMask, TApropSUPERSUB ); */
    /* PROPmaskADD( &taOnlyMask, TApropSMALLCAPS ); */
    /* PROPmaskADD( &taOnlyMask, TApropCAPITALS ); */

    PROPmaskADD( &taOnlyMask, TApropTEXT_COLOR );
    PROPmaskADD( &taOnlyMask, TApropTEXT_STYLE );
    PROPmaskADD( &taOnlyMask, TApropBORDER );
    PROPmaskADD( &taOnlyMask, TApropSHADING );

    utilPropMaskAnd( &taSetMask, &taSetMask, &taOnlyMask );
    }

    if  ( tedChangeSelectionProperties( ed,
				    &taSetMask, &taSetTo,
				    &ppUpdMask, &ppSet,
				    &spUpdMask, (SectionProperties *)0 ) )
	{ LDEB(1); rval= -1; goto ready; }

  ready:

    docCleanParagraphProperties( &ppSet );
    docCleanDocumentCopyJob( &dcj );

    return rval;
    }

APP_PASTE_REPLY( tedPasteFontTed, w, event, voided )
    {
    EditDocument *	ed= (EditDocument *)voided;

    tedPasteClipboardGeneric( w, ed, event,
			    tedPasteRtfReadFile, tedApplyPastedFont );
    return;
    }

/************************************************************************/
/*									*/
/*  Custom 'Paste' functionality.					*/
/*									*/
/************************************************************************/

APP_MENU_CALLBACK_H( tedDocEditPaste, option, voided, e )
    {
    EditDocument *			ed= (EditDocument *)voided;
    TedDocument *			td= (TedDocument *)ed->edPrivateData;

    if  ( ! td->tdCanReplaceSelection )
	{ LDEB(td->tdCanReplaceSelection); return;	}

    /*  1  */
    appDocAskForPaste( ed, "CLIPBOARD" );

    return;
    }

/************************************************************************/
/*									*/
/*  'Copy Font' menu option.						*/
/*									*/
/************************************************************************/

APP_MENU_CALLBACK_H( tedDocFontCopy, fontsOption, voided, e )
    {
    EditDocument *		ed= (EditDocument *)voided;
    TedDocument *		td= (TedDocument *)ed->edPrivateData;
    SimpleOutputStream *	sos;

    DocumentSelection		ds;
    SelectionGeometry		sg;
    SelectionDescription	sd;

    const int			rtfFlags= RTFflagNO_BOOKMARKS|RTFflagCLOSE_LAST;

    if  ( tedGetSelection( &ds, &sg, &sd,
			    (DocumentTree **)0, (const BufferItem **)0, td ) )
	{ LDEB(1); return;	}

    /*  2  */
    sos= sioOutMemoryOpen( &(td->tdCopiedFont) );
    if  ( ! sos )
	{ XDEB(sos); return;	}

    if  ( docRtfSaveDocument( sos, td->tdDocument, &ds, rtfFlags ) )
	{ LDEB(1); sioOutClose( sos ); return;	}

    if  ( sioOutClose( sos ) )
	{ LDEB(1); return;	}

    if  ( getenv( "TED_SAVE_COPIES" ) )
	{
	tedSaveSelectionToFile( td->tdDocument, &ds,
					    rtfFlags, "/tmp/font.rtf" );
	}

    appDocOwnSelection( ed, "RTFFONT", TedFontTargets, TedFontTargetCount );
    }

/************************************************************************/
/*									*/
/*  Callbacks for the Copy/Paste ruler menu options.			*/
/*									*/
/*  1)  The ruler paste is a selection paste with the selection		*/
/*	constrained to the last position of the first paragraph in the	*/
/*	selection.							*/
/*  2)  Remember the bytes expecting that someone will ask for them.	*/
/*  3)  Tell that we have a ruler paste available.			*/
/*									*/
/************************************************************************/

APP_MENU_CALLBACK_H( tedDocFormatCopyRul, fontsOption, voided, e )
    {
    EditDocument *		ed= (EditDocument *)voided;
    TedDocument *		td= (TedDocument *)ed->edPrivateData;
    SimpleOutputStream *	sos;

    DocumentSelection		ds;
    SelectionGeometry		sg;
    SelectionDescription	sd;

    DocumentSelection		dsPara;

    const int			rtfFlags= RTFflagNO_BOOKMARKS|RTFflagCLOSE_LAST;

    if  ( tedGetSelection( &ds, &sg, &sd,
			    (DocumentTree **)0, (const BufferItem **)0, td ) )
	{ LDEB(1); return;	}

    /*  1  */
    dsPara= ds;
    docLastPosition( &(dsPara.dsHead), dsPara.dsHead.dpBi );
    docSetIBarSelection( &dsPara, &(dsPara.dsHead) );

    /*  2  */
    sos= sioOutMemoryOpen( &(td->tdCopiedRuler) );
    if  ( ! sos )
	{ XDEB(sos); return;	}

    if  ( docRtfSaveDocument( sos, td->tdDocument, &dsPara, rtfFlags ) )
	{ LDEB(1); sioOutClose( sos ); return;	}

    if  ( sioOutClose( sos ) )
	{ LDEB(1); return;	}

    if  ( getenv( "TED_SAVE_COPIES" ) )
	{
	tedSaveSelectionToFile( td->tdDocument, &dsPara,
						rtfFlags, "/tmp/ruler.rtf" );
	}

    /*  2  */
    appDocOwnSelection( ed, "RTFRULER", TedRulerTargets, TedRulerTargetCount );
    }

APP_MENU_CALLBACK_H( tedDocFormatPasteRul, fontsOption, voided, e )
    {
    EditDocument *			ed= (EditDocument *)voided;

    appDocAskForPaste( ed, "RTFRULER" );
    }

/************************************************************************/

void tedManagePrimarySelection(	EditDocument *		ed )
    {
    TedDocument *		td= (TedDocument *)ed->edPrivateData;

    if  ( ! tedHasIBarSelection( td ) )
	{
	if  ( appDocOwnSelection( ed, "PRIMARY",
					    TedPrimaryTargets,
					    TedPrimaryTargetCount ) )
	    { LDEB(1);				}
	else{ td->tdOwnsPrimarySelection= 1;	}
	}
    else{
	if  ( td->tdOwnsPrimarySelection )
	    {
	    appDocReleaseSelection( ed, "PRIMARY" );
	    td->tdOwnsPrimarySelection= 0;
	    }
	}

    return;
    }

