/************************************************************************/
/*									*/
/*  Ted: Bookmark dialog.						*/
/*									*/
/************************************************************************/

#   include	"tedConfig.h"

#   include	<stddef.h>
#   include	<stdlib.h>
#   include	<stdio.h>
#   include	<string.h>

#   include	"tedApp.h"
#   include	"tedBookmarkTool.h"
#   include	<appInspector.h>

#   include	<appDebugon.h>

/************************************************************************/
/*									*/
/*  A)  A Bookmark was chosen in the list.				*/
/*  B)  Something was typed in the 'Destination' text widgets.		*/
/*									*/
/************************************************************************/

static void tedBookmarkAdaptToText(	BookmarkTool *		bt,
					int			adaptString,
					const char *		markName,
					int			exists )
    {
    int				gotChosen;
    int				gotSet;
    const int			isLocal= 1;
    int				changed;

    const char *		markSaved= bt->btMarkSet;

    if  ( ! markSaved )
	{ markSaved= "";	}

    if  ( ! markName )
	{ markName= "";	}

    gotSet= markSaved[0] != '\0';
    gotChosen= markName[0] != '\0';
    changed= strcmp( markName, markSaved ) != 0;

    appGuiEnableWidget( bt->btSetBookmarkButton,
		    ! exists				&&
		    ( ! bt->btIBarSelection || gotSet )	&&
		    ! bt->btDocumentReadonly		&&
		    changed && gotChosen		);
    appGuiEnableWidget( bt->btRemoveBookmarkButton,
		    ! bt->btDocumentReadonly		&&
		    ! changed && gotChosen		);

    appGuiEnableWidget( bt->btGoToBookmarkButton, exists && gotChosen );

    appGuiEnableWidget( bt->btMarkListWidget, isLocal );

    if  ( adaptString )
	{ appStringToTextWidget( bt->btMarkTextWidget, markName ); }

    return;
    }

/************************************************************************/
/*									*/
/*  'Revert' was pushed.						*/
/*									*/
/************************************************************************/

static APP_BUTTON_CALLBACK_H( tedBookmarkRevertPushed, w, voidbt )
    {
    BookmarkTool *		bt= (BookmarkTool *)voidbt;
    const int			adaptString= 1;

    if  ( bt->btMarkChosen )
	{
	free( bt->btMarkChosen );
	bt->btMarkChosen= (char *)0;
	}

    if  ( bt->btMarkSet )
	{
	bt->btMarkChosen= strdup( bt->btMarkSet );
	if  ( ! bt->btMarkChosen )
	    { XDEB(bt->btMarkChosen);	}
	}

    tedBookmarkAdaptToText( bt, adaptString,
				    bt->btMarkChosen, bt->btMarkExists );

    return;
    }

/************************************************************************/
/*									*/
/*  1)  Reset the document id to force the tool to refresh.		*/
/*									*/
/************************************************************************/

static APP_BUTTON_CALLBACK_H( tedBookmarkChangePushed, button, voidbt )
    {
    BookmarkTool *		bt= (BookmarkTool *)voidbt;
    char *			markName;

    /*  1  */
    bt->btCurrentDocumentId= 0;

    markName= appGetStringFromTextWidget( bt->btMarkTextWidget );
    if  ( ! markName )
	{ XDEB(markName); return;	}
    else{
	if  ( tedAppSetBookmark( bt->btApplication, markName ) )
	    { SDEB(markName);	}

	appFreeStringFromTextWidget( markName );
	}
    }

static APP_BUTTON_CALLBACK_H( tedBookmarkGotoPushed, button, voidbt )
    {
    BookmarkTool *		bt= (BookmarkTool *)voidbt;
    char *			markName;

    markName= appGetStringFromTextWidget( bt->btMarkTextWidget );
    if  ( ! markName )
	{ XDEB(markName); return;	}
    else{
	if  ( tedAppGoToBookmark( bt->btApplication,
					    markName, strlen( markName ) ) )
	    { SDEB(markName);	}

	appFreeStringFromTextWidget( markName );
	}
    }

/************************************************************************/
/*									*/
/*  Remove the selected bookmark.					*/
/*									*/
/*  1)  Reset the document id to force the tool to refresh.		*/
/*									*/
/************************************************************************/

static APP_BUTTON_CALLBACK_H( tedBookmarkRemovePushed, w, voidbt )
    {
    BookmarkTool *		bt= (BookmarkTool *)voidbt;

    /*  1  */
    bt->btCurrentDocumentId= 0;

    if  ( tedAppRemoveBookmark( bt->btApplication ) )
	{ LDEB(1);	}

    return;
    }

/************************************************************************/
/*									*/
/*  Typing happened in the 'Bookmark' text widget.			*/
/*									*/
/************************************************************************/

static APP_TXTYPING_CALLBACK_H( tedBookmarkDestinationChanged, w, voidbt )
    {
    BookmarkTool *	bt= (BookmarkTool *)voidbt;

    char *		markName;
    int			markSize;
    int			exists;

    const int		adaptString= 0;

    appGuiEnableWidget( bt->btRemoveBookmarkButton, 0 );

    markName= appGetStringFromTextWidget( bt->btMarkTextWidget );
    markSize= strlen( markName );

    if  ( markSize > 0 )
	{
	DocumentField *		df;

	if  ( tedAppGetBookmarkField( &df, bt->btApplication,
						markName, markSize ) < 0 )
	    { exists= 0;	}
	else{ exists= 1;	}
	}
    else{ exists= 0;		}

    tedBookmarkAdaptToText( bt, adaptString, markName, exists );

    appFreeStringFromTextWidget( markName );

    return;
    }

/************************************************************************/
/*									*/
/*  A Bookmark was chosen in the list.					*/
/*									*/
/************************************************************************/

static APP_LIST_CALLBACK_H( tedBookmarkChosen, w, voidbt, voidlcs )
    {
    BookmarkTool *	bt= (BookmarkTool *)voidbt;
    char *		markName;
    const int		exists= 1;
    const int		adaptString= 1;

    markName= appGuiGetStringFromListCallback( w, voidlcs );
    if  ( ! markName )
	{ XDEB(markName); return;	}

    tedBookmarkAdaptToText( bt, adaptString, markName, exists );

    appFreeStringFromListCallback( markName );

    return;
    }

/************************************************************************/
/*									*/
/*  Make a bookmark tool. I.E. the 'Bookmark' page on the format tool	*/
/*  to manage Hyperlinks.						*/
/*									*/
/************************************************************************/

void tedFillBookmarkTool(	BookmarkTool *			bt,
				const BookmarkPageResources *	bpr,
				InspectorSubject *		is,
				APP_WIDGET			pageWidget,
				const InspectorSubjectResources * isr )
    {
    APP_WIDGET		row;

    const int		listVisibleItems= 9;

    bt->btResources= bpr;

    /* FULL WIDTH */
    appMakeLabelInColumn( &(bt->btMarkLabelWidget),
				pageWidget, bpr->bprMarkText );
    appMakeTextInColumn( &(bt->btMarkTextWidget), pageWidget, 0, 1 );

    appGuiSetTypingCallbackForText( bt->btMarkTextWidget,
				tedBookmarkDestinationChanged, (void *)bt );

    appGuiMakeListInColumn( &(bt->btMarkListWidget),
		pageWidget, listVisibleItems,
		tedBookmarkChosen, (APP_BUTTON_CALLBACK_T)0, (void *)bt );

    appInspectorMakeButtonRow( &row, pageWidget,
	    &(bt->btGoToBookmarkButton), &(bt->btRemoveBookmarkButton),
	    isr->isrSelectButtonText, isr->isrDeleteButtonText,
	    tedBookmarkGotoPushed, tedBookmarkRemovePushed, (void *)bt );

    appInspectorMakeButtonRow( &row, pageWidget,
	    &(is->isRevertButton), &(is->isApplyButton),
	    isr->isrRevert, isr->isrApplyToSubject,
	    tedBookmarkRevertPushed, tedBookmarkChangePushed, (void *)bt );

    bt->btSetBookmarkButton= is->isApplyButton;

    return;
    }

/************************************************************************/
/*									*/
/*  Initialize/Clean a bookmark tool					*/
/*									*/
/************************************************************************/

void tedInitBookmarkTool(		BookmarkTool *		bt )
    {
    bt->btCurrentDocumentId= 0;

    bt->btMarkTextWidget= (APP_WIDGET)0;

    bt->btSetBookmarkButton= (APP_WIDGET)0;
    bt->btGoToBookmarkButton= (APP_WIDGET)0;
    bt->btRemoveBookmarkButton= (APP_WIDGET)0;

    bt->btMarkSet= (char *)0;
    bt->btMarkChosen= (char *)0;

    return;
    }

void tedFormatCleanBookmarkTool(		BookmarkTool *		bt )
    {
    if  ( bt->btMarkSet )
	{ free( bt->btMarkSet );	}

    if  ( bt->btMarkChosen )
	{ free( bt->btMarkChosen );	}

    return;
    }

/************************************************************************/
/*									*/
/*  Finish the bookmarks page.						*/
/*									*/
/************************************************************************/

void tedBookmarkToolFillChoosers( BookmarkTool *		bt )
    {
    return;
    }

void tedFinishBookmarkTool(	BookmarkTool *			bt )
    {
    return;
    }

/************************************************************************/
/*									*/
/*  Refresh the bookmark tool.						*/
/*									*/
/************************************************************************/

void tedRefreshBookmarkTool(	BookmarkTool *			bt,
				int *				pEnabled,
				int *				pPref,
				InspectorSubject *		is,
				const DocumentSelection *	ds,
				const SelectionDescription *	sd,
				DocumentTree *			ei,
				BufferDocument *		bd )
    {
    DocumentField *		dfBookmark;
    const int			adaptString= 1;

    if  ( bt->btCurrentDocumentId != sd->sdDocumentId )
	{
	tedFillBookmarkList( bt->btMarkListWidget, bd );

	bt->btCurrentDocumentId= sd->sdDocumentId;
	}

    if  ( bt->btMarkSet )
	{
	free( bt->btMarkSet );
	bt->btMarkSet= (char *)0;
	}

    if  ( bt->btMarkChosen )
	{
	free( bt->btMarkChosen );
	bt->btMarkChosen= (char *)0;
	}

    bt->btDocumentReadonly= sd->sdDocumentReadonly;
    bt->btIBarSelection= sd->sdIsIBarSelection;

    dfBookmark= docFindTypedFieldForPosition( bd, &(ds->dsHead),
							    DOCfkBOOKMARK );
    if  ( dfBookmark )
	{
	const char *		markName= (const char *)0;
	int			markSize= 0;
	char *			saved;

	if  ( docFieldGetBookmark( dfBookmark, &markName, &markSize ) )
	    { LDEB(1); *pEnabled= 0; return;	}

	saved= (char *)malloc( markSize+ 1 );
	if  ( ! saved )
	    { XDEB(saved);	}
	else{
	    strncpy( saved, markName, markSize )[markSize]= '\0';
	    appStringToTextWidget( bt->btMarkTextWidget, saved );
	    bt->btMarkSet= saved;

	    bt->btMarkChosen= strdup( bt->btMarkSet );
	    if  ( ! bt->btMarkChosen )
		{ XDEB(bt->btMarkChosen);	}
	    }

	bt->btMarkExists= 1;
	tedBookmarkAdaptToText( bt, adaptString, bt->btMarkSet, bt->btMarkExists );
	*pPref= 8;
	}
    else{
	int	suggested= 0;
	char	suggestion[DOCmaxBOOKMARK+ 1];

	suggestion[0]= '\0';
	if  ( ! bt->btIBarSelection )
	    { suggested= docSuggestNewBookmarkName( suggestion, bd, ds ); }

	bt->btMarkExists= 0;
	tedBookmarkAdaptToText( bt, adaptString, suggestion, bt->btMarkExists );
	}

    *pEnabled= ! sd->sdDocumentReadonly;
    return;
    }

/************************************************************************/
/*									*/
/*  Fill a listbox with the bookmarks in a document.			*/
/*									*/
/*  1)  Empty the list widget.						*/
/*  2)  Collect bookmarks.						*/
/*  3)  Sort them.							*/
/*  4)  And insert them in the list.					*/
/*									*/
/************************************************************************/

void tedFillBookmarkList(	APP_WIDGET		list,
				const BufferDocument *	bd )
    {
    int				i;

    int				bookmarkCount= 0;
    char **			bookmarks= (char **)0;
    const DocumentFieldList *	dfl= &(bd->bdFieldList);
    const int			fieldCount= dfl->dflPagedList.plItemCount;

    /*  1  */
    appGuiEmptyListWidget( list );

    if  ( fieldCount == 0 )
	{ goto ready;	}

    /*  2,3  */
    if  ( docMakeBookmarkList( &bookmarks, &bookmarkCount,
						    &(bd->bdFieldList) ) )
	{ goto ready;	}

    /*  4  */
    for ( i= 0; i < bookmarkCount; i++ )
	{ appGuiAddValueToListWidget( list, -1, bookmarks[i] );	}

  ready:

    for ( i= 0; i < bookmarkCount; i++ )
	{
	if  ( bookmarks[i] )
	    { free( bookmarks[i] );	}
	}

    if  ( bookmarks )
	{ free( bookmarks );	}

    return;
    }

/************************************************************************/
/*									*/
/*  Resource table for various texts.					*/
/*									*/
/************************************************************************/

static AppConfigurableResource TED_TedBookmarkSubjectResourceTable[]=
    {
    APP_RESOURCE( "formatToolBookmark",
	    offsetof(InspectorSubjectResources,isrSubjectName),
	    "Bookmark" ),

    APP_RESOURCE( "bookmarkSetBookmark",
	    offsetof(InspectorSubjectResources,isrApplyToSubject),
	    "Set Bookmark" ),
    APP_RESOURCE( "bookmarkRevert",
	    offsetof(InspectorSubjectResources,isrRevert),
	    "Revert" ),

    APP_RESOURCE( "bookmarkGoToBookmark",
	    offsetof(InspectorSubjectResources,isrSelectButtonText),
	    "Go To Bookmark" ),
    APP_RESOURCE( "bookmarkRemoveBookmark",
	    offsetof(InspectorSubjectResources,isrDeleteButtonText),
	    "Remove Bookmark" ),
    };

static AppConfigurableResource TED_TedBookmarkToolResourceTable[]=
    {
    APP_RESOURCE( "bookmarkBookmark",
	    offsetof(BookmarkPageResources,bprMarkText),
	    "Bookmark" ),

    APP_RESOURCE( "bookmarkCopyHeader",
	    offsetof(BookmarkPageResources,bprCopyHeaderText),
	    "Copy Bookmark" ),
    APP_RESOURCE( "bookmarkCopyButton",
	    offsetof(BookmarkPageResources,bprCopyButtonText),
	    "Copy" ),

    APP_RESOURCE( "bookmarkCopyAsHyperlink",
	    offsetof(BookmarkPageResources,
			    bprCopyAsItemTexts[LINKkindHYPERLINK]),
	    "As Hyperlink" ),
    APP_RESOURCE( "bookmarkCopyAsPageNumber",
	    offsetof(BookmarkPageResources,
			    bprCopyAsItemTexts[LINKkindPAGE_NUMBER]),
	    "As Page Number" ),
    APP_RESOURCE( "bookmarkCopyAsBookmarkText",
	    offsetof(BookmarkPageResources,
			    bprCopyAsItemTexts[LINKkindBOOKMARK_TEXT]),
	    "As Bookmark Text" ),
    APP_RESOURCE( "bookmarkCopyAsTextPlusPage",
	    offsetof(BookmarkPageResources,
			    bprCopyAsItemTexts[LINKkindTEXT_PLUS_PAGE]),
	    "As Text and Page Number" ),
    };

void tedBookmarkToolGetResourceTable(	EditApplication *		ea,
					BookmarkPageResources *		bpr,
					InspectorSubjectResources *	isr )
    {
    static int	gotToolResources= 0;
    static int	gotSubjectResources= 0;

    if  ( ! gotToolResources )
	{
	appGuiGetResourceValues( &gotToolResources, ea, (void *)bpr,
				TED_TedBookmarkToolResourceTable,
				sizeof(TED_TedBookmarkToolResourceTable)/
				sizeof(AppConfigurableResource) );
	}

    if  ( ! gotSubjectResources )
	{
	appGuiGetResourceValues( &gotSubjectResources, ea, (void *)isr,
				TED_TedBookmarkSubjectResourceTable,
				sizeof(TED_TedBookmarkSubjectResourceTable)/
				sizeof(AppConfigurableResource) );
	}

    return;
    }
