/************************************************************************/
/*									*/
/*  Print PostScript : print one line of text.				*/
/*									*/
/************************************************************************/

#   include	"tedConfig.h"

#   include	<stddef.h>
#   include	<stdio.h>
#   include	<stdlib.h>
#   include	<string.h>

#   include	<sioGeneral.h>

#   include	<psFontMetrics.h>
#   include	"docPsPrint.h"
#   include	"docDraw.h"
#   include	"docLayout.h"

#   include	<appDebugon.h>


/************************************************************************/
/*									*/
/*  Print a string.							*/
/*									*/
/************************************************************************/

static void docPsPrintString(	PrintingState *			ps,
				DrawingContext *		dc,
				int				xShift,
				int				yShift,
				int				baseline,
				const ParticuleData *		pd,
				int				textAttr,
				const TextAttribute *		ta,
				const unsigned char *		s,
				int				len )
    {
    int		fontSizeTwips;

    fontSizeTwips= 10* ta->taFontSizeHalfPoints;

    if  ( len > 0 )
	{
	int			y;

	docDrawSetFont( dc, (void *)ps, textAttr, ta );
	docDrawSetColorNumber( dc, (void *)ps, ta->taTextColorNumber );

	y= baseline;

	if  ( ta->taSuperSub == DOCfontSUPERSCRIPT )
	    { psGetSuperBaseline( &y, baseline, fontSizeTwips, pd->pdAfi ); }

	if  ( ta->taSuperSub == DOCfontSUBSCRIPT )
	    { psGetSubBaseline( &y, baseline, fontSizeTwips, pd->pdAfi ); }

	psMoveShowString( ps, s, len,
				    pd->pdX0+ pd->pdLeftBorderWidth+ xShift,
				    y+ yShift );

	ps->psLastPageMarked= ps->psPagesPrinted;
	}

    return;
    }

static void psPrintSegment(	PrintingState *			ps,
				DrawingContext *		dc,
				int				textAttr,
				const TextAttribute *		ta,
				const unsigned char *		s,
				int				len )
    {
    docDrawSetFont( dc, (void *)ps, textAttr, ta );
    psShowString( ps, s, len );
    return;
    }

static void psPrintSegments(	PrintingState *			ps,
				DrawingContext *		dc,
				int				xShift,
				int				yShift,
				int				baseLine,
				const ParticuleData *		pd,
				int				textAttr,
				const TextAttribute *		ta,
				const unsigned char *		s,
				const int *			segments,
				int				segmentCount )
    {
    TextAttribute		taN= *ta;
    int				seg;

    taN.taSmallCaps= 0;

    if  ( segments[0] > 0 )
	{
	docPsPrintString( ps, dc, xShift, yShift, baseLine, pd,
					    textAttr, &taN, s, segments[0] );
	s += segments[0];

	psPrintSegment( ps, dc, textAttr, ta, s, segments[1] );
	s += segments[1];
	}
    else{
	docPsPrintString( ps, dc, xShift, yShift, baseLine, pd,
					    textAttr, ta, s, segments[1] );
	s += segments[1];
	}

    for ( seg= 1; seg < segmentCount; seg++ )
	{
	if  ( segments[2* seg+ 0] > 0 )
	    {
	    psPrintSegment( ps, dc, textAttr, &taN, s, segments[2* seg+ 0] );
	    s += segments[2* seg+ 0];
	    }

	if  ( segments[2* seg+ 1] > 0 )
	    {
	    psPrintSegment( ps, dc, textAttr, ta, s, segments[2* seg+ 1] );
	    s += segments[2* seg+ 1];
	    }
	}

    return;
    }

static void psPrintParticuleUnderlines(	DrawingContext *	dc,
					int			xShift,
					int			yShift,
					PrintingState *		ps,
					const TextParticule *	tp,
					const ParticuleData *	pd,
					int			drawn,
					int			fontSizeTwips,
					int			baseLine )
    {
    const LayoutContext *	lc= &(dc->dcLayoutContext);
    const BufferDocument *	bd= lc->lcDocument;

    int				i;

    i= 0;
    while( i < drawn )
	{
	TextAttribute	ta;
	int		x0;
	int		x1;
	int		y0;
	int		h;

	utilGetTextAttributeByNumber( &ta, &(bd->bdTextAttributeList),
						tp[i].tpTextAttrNr );

	if  ( ! ta.taTextIsUnderlined )
	    { i++; continue;	}

	psUnderlineGeometry( &y0, &h, baseLine, fontSizeTwips, pd[i].pdAfi );

	x1= x0= pd[i].pdX0;

	while( i < drawn )
	    {
	    utilGetTextAttributeByNumber( &ta, &(bd->bdTextAttributeList),
						tp[i].tpTextAttrNr );

	    if  ( ! ta.taTextIsUnderlined )
		{ break;	}

	    x1= pd[i].pdX0+ tp[i].tpTwipsWide; i++;
	    }

	if  ( x1 > x0 )
	    {
	    docDrawSetColorNumber( dc, (void *)ps, ta.taTextColorNumber );
	    psFillRectangle( ps, x0+ xShift, y0+ yShift, x1- x0, h );
	    }
	}

    return;
    }

static void psPrintParticuleStrikethrough(
					DrawingContext *	dc,
					int			xShift,
					int			yShift,
					PrintingState *		ps,
					const TextParticule *	tp,
					const ParticuleData *	pd,
					int			drawn,
					int			fontSizeTwips,
					int			baseLine )
    {
    const LayoutContext *	lc= &(dc->dcLayoutContext);
    const BufferDocument *	bd= lc->lcDocument;

    int				i;

    i= 0;
    while( i < drawn )
	{
	TextAttribute	ta;
	int		x0;
	int		x1;
	int		y0;
	int		h;

	utilGetTextAttributeByNumber( &ta, &(bd->bdTextAttributeList),
						tp[i].tpTextAttrNr );

	if  ( ! ta.taHasStrikethrough )
	    { i++; continue;	}

	psStrikethroughGeometry( &y0, &h, baseLine, fontSizeTwips, pd[i].pdAfi );
	x1= x0= pd[i].pdX0;

	while( i < drawn )
	    {
	    utilGetTextAttributeByNumber( &ta, &(bd->bdTextAttributeList),
						tp[i].tpTextAttrNr );

	    if  ( ! ta.taHasStrikethrough )
		{ break;	}

	    x1= pd[i].pdX0+ tp[i].tpTwipsWide; i++;
	    }

	if  ( x1 > x0 )
	    {
	    docDrawSetColorNumber( dc, (void *)ps, ta.taTextColorNumber );
	    psFillRectangle( ps, x0+ xShift, y0+ yShift, x1- x0, h );
	    }
	}

    return;
    }

static void docPsPrintChftnsep(	DrawingContext *	dc,
				int			xShift,
				int			yShift,
				PrintingState *		ps,
				const TextParticule *	tp,
				const ParticuleData *	pd,
				int			baseLine )
    {
    const LayoutContext *	lc= &(dc->dcLayoutContext);
    const BufferDocument *	bd= lc->lcDocument;

    TextAttribute		ta;
    int				fontSizeTwips;

    int				xHeight;

    int				x0;
    int				x1;
    int				y0;
    int				h;

    utilGetTextAttributeByNumber( &ta, &(bd->bdTextAttributeList),
						tp->tpTextAttrNr );

    fontSizeTwips= 10* ta.taFontSizeHalfPoints;

    xHeight= ( fontSizeTwips+ 1 )/ 2;

    x0= pd->pdX0;
    x1= pd->pdX0+ 2800;
    y0= baseLine- xHeight/2- 15/2;
    h= 15;

    docDrawSetColorNumber( dc, (void *)ps, ta.taTextColorNumber );
    psFillRectangle( ps, x0+ xShift, y0+ yShift, x1- x0, h );

    return;
    }

/************************************************************************/
/*									*/
/*  Print a series of particules with the same attributes.		*/
/*									*/
/************************************************************************/

static int psPrintParticules(	PrintingState *			ps,
				DrawingContext *		dc,
				int				xShift,
				int				yShift,
				const BufferItem *		paraBi,
				int				part,
				const DocumentFontList *	dfl,
				const PostScriptFontList *	psfl,
				const TextParticule *		tp,
				ParticuleData *			pd,
				int				count,
				int				baseLine,
				int				lineTop,
				int				lineHeight )
    {
    const LayoutContext *	lc= &(dc->dcLayoutContext);
    const BufferDocument *	bd= lc->lcDocument;
    const TabStopList *		tsl= &(paraBi->biParaTabStopList);

    int				drawn;
    int				len;

    DocumentField *		df;

    TextAttribute		ta;
    int				fontSizeTwips;

    /*  1  */
    switch( tp->tpKind )
	{
	InsertedObject *	io;

	case DOCkindTAB:
	    if  ( pd->pdTabNumber >= 0				&&
		  pd->pdTabNumber < tsl->tslTabStopCount	)
		{
		if  ( docPsPrintTab( ps, dc, xShift, yShift, paraBi, tp, pd,
						    baseLine, lineHeight ) )
		    { LDEB(1);	}
		}

	    utilGetTextAttributeByNumber( &ta, &(bd->bdTextAttributeList),
						tp->tpTextAttrNr );

	    fontSizeTwips= 10* ta.taFontSizeHalfPoints;

	    drawn= 1;

	    psPrintParticuleUnderlines( dc, xShift, yShift, ps,
				    tp, pd, drawn, fontSizeTwips, baseLine );
	    psPrintParticuleStrikethrough( dc, xShift, yShift, ps,
				    tp, pd, drawn, fontSizeTwips, baseLine );
	    return drawn;

	case DOCkindSPAN:
	    break;

	case DOCkindFIELDSTART:
	    df= docGetFieldByNumber( &(bd->bdFieldList), tp->tpObjectNumber );
	    if  ( ! df )
		{ XDEB(df); return drawn= 1;	}

	    if  ( df->dfKind == DOCfkBOOKMARK )
		{
		const char *	refName;
		int		refSize;

		if  ( ! docFieldGetBookmark( df, &refName, &refSize ) )
		    {
		    psDestPdfmark( ps, lineTop, refName, refSize );
		    }

		return drawn= 1;
		}

	    if  ( df->dfKind == DOCfkCHFTN )
		{
		const char *	refName;
		int		refSize;
		const char *	markName;
		int		markSize;

		DocumentNote *	dn;

		char		ref[25+1];
		char		def[25+1];

		int		cnt;
		int		closed;

		cnt= docCountParticulesInField( paraBi, &closed, part,
						paraBi->biParaParticuleCount );
		if  ( cnt < 0 )
		    { LDEB(cnt); }

		dn= docGetNoteOfField( df, bd );
		if  ( ! dn )
		    {
		    LLXDEB(cnt,tp[cnt+1].tpStroff,dn);
		    SDEB(docExternalKindStr(paraBi->biInExternalItem));
		    }

		sprintf( ref, "_NREF_%d", df->dfFieldNumber+ 1 );
		sprintf( def, "_NDEF_%d", df->dfFieldNumber+ 1 );

		if  ( paraBi->biInExternalItem == DOCinBODY )
		    {
		    markName= def;
		    refName=  ref;
		    }
		else{
		    markName= ref;
		    refName=  def;
		    }

		markSize= strlen( markName );
		refSize= strlen( refName );

		psDestPdfmark( ps, lineTop, refName, refSize );

		ps->psLinkFile= (char *)0; ps->psLinkFileSize= 0;
		ps->psLinkMark= markName; ps->psLinkMarkSize= markSize;

		ps->psInsideLink= 1;
		ps->psLinkParticulesDone= 0;
		ps->psLinkRectLeft= pd->pdX0;

		return drawn= 1;
		}

	    if  ( df->dfKind == DOCfkHYPERLINK )
		{
		if  ( ! docFieldGetHyperlink( df,
				&(ps->psLinkFile), &(ps->psLinkFileSize),
				&(ps->psLinkMark), &(ps->psLinkMarkSize) ) )
		    {
		    ps->psInsideLink= 1;
		    ps->psLinkParticulesDone= 0;
		    ps->psLinkRectLeft= pd->pdX0;
		    }

		return drawn= 1;
		}

	    return drawn= 1;

	case DOCkindFIELDEND:
	    if  ( ps->psInsideLink )
		{
		psFlushLink( ps, pd->pdX0, pd->pdVisibleBBox.drX1,
						    lineTop, lineHeight );
		ps->psInsideLink= 0;
		}

	    return drawn= 1;

	case DOCkindLINEBREAK:
	case DOCkindPAGEBREAK:
	case DOCkindCOLUMNBREAK:
	    return drawn= 1;

	case DOCkindOBJECT:
	    io= docGetObject( bd, tp->tpObjectNumber );
	    if  ( ! io )
		{ LPDEB(tp->tpObjectNumber,io); return 0;	}
	    drawn= docPsPrintObject( ps, dc, xShift, yShift, io,
							psfl, pd, baseLine );

	    if  ( drawn < 1 )
		{ LDEB(drawn);	}
	    return drawn;

	case DOCkindCHFTNSEP:
	case DOCkindCHFTNSEPC:
	    docPsPrintChftnsep( dc, xShift, yShift, ps, tp, pd, baseLine );
	    return 1;

	case DOCkindOPT_HYPH:
	    if  ( tp->tpTwipsWide != 0 )
		{ LLDEB(tp->tpKind,tp->tpTwipsWide);	}
	    return drawn= 1;

	default:
	    LDEB(tp->tpKind); return -1;
	}

    {
    const int	separate= paraBi->biParaAlignment == DOCiaJUSTIFIED;
    int		x0Twips= pd->pdX0;
    int		x1Twips;

    drawn= docLayoutDelimitSpan( &len, &x1Twips, x0Twips, &ta, bd, paraBi,
						part, part+ count, separate );

#   if 0
    docDrawSetColorNumber( dc, (void *)ps, 0 );
    sioOutPrintf( ps->psSos, " %d %d moveto", x0Twips, lineTop );
    sioOutPrintf( ps->psSos, " %d %d lineto", x1Twips, lineTop );
    sioOutPrintf( ps->psSos, " %d %d lineto", x1Twips, lineTop+ lineHeight );
    sioOutPrintf( ps->psSos, " %d %d lineto", x0Twips, lineTop+ lineHeight );
    sioOutPrintf( ps->psSos, " closepath stroke\n" );
#   endif
    }

    fontSizeTwips= 10* ta.taFontSizeHalfPoints;

    if  ( len > 0 )
	{
	const char *	printString;
	char *		upperString= (char *)0;
	int *		segments= (int *)0;
	int		segmentCount= 0;
	int		textAttr= tp->tpTextAttrNr;

	printString= (char *)docParaString( paraBi, tp->tpStroff );

	if  ( ta.taSmallCaps || ta.taCapitals )
	    {
	    if  ( docMakeCapsString( &upperString, &segments, &segmentCount,
						&ta, printString, len ) )
		{ LDEB(len); return -1;	}

	    printString= upperString;
	    }

	if  ( ta.taSmallCaps && ! ta.taCapitals )
	    {
	    psPrintSegments(  ps, dc, xShift, yShift, baseLine, pd,
			    textAttr, &ta, (unsigned char *)printString,
			    segments, segmentCount );
	    }
	else{
	    docPsPrintString( ps, dc, xShift, yShift, baseLine, pd,
			    textAttr, &ta, (unsigned char *)printString, len );
	    }

	if  ( upperString )
	    { free( upperString );	}
	if  ( segments )
	    { free( segments );	}
	}

    psPrintParticuleUnderlines( dc, xShift, yShift, ps, tp, pd, drawn,
						    fontSizeTwips, baseLine );
    psPrintParticuleStrikethrough( dc, xShift, yShift, ps, tp, pd, drawn,
						    fontSizeTwips, baseLine );

    ps->psLinkParticulesDone += drawn;

    return drawn;
    }

/************************************************************************/
/*									*/
/*  Print one line of output.						*/
/*									*/
/************************************************************************/

static int psPrintTextLine(	PrintingState *			ps,
				DrawingContext *		dc,
				int				xShift,
				int				yShift,
				const DocumentFontList *	dfl,
				const PostScriptFontList *	psfl,
				const BufferItem *		bi,
				int				part,
				const TextParticule *		tp,
				int				particuleCount,
				ParticuleData *			pd,
				int				baseLine,
				int				lineTop,
				int				lineHeight )
    {
    int				done= 0;

    while( done < particuleCount )
	{
	int		drawn;

	drawn= psPrintParticules( ps, dc, xShift, yShift,
			bi, part, dfl, psfl, tp, pd,
			particuleCount- done, baseLine, lineTop, lineHeight );

	if  ( drawn < 1 )
	    { LDEB(drawn); return -1;	}

	done += drawn; tp += drawn; pd += drawn; part += drawn;
	}

    if  ( done > 0 && ps->psInsideLink )
	{
	const ParticuleData *	ppd= pd- 1;

	psFlushLink( ps, ppd->pdX0, ppd->pdVisibleBBox.drX1,
						    lineTop, lineHeight );
	}

    return 0;
    }

/************************************************************************/
/*									*/
/*  Layout and print successive lines of a paragraph.			*/
/*									*/
/************************************************************************/

int docPsPrintTextLine(		BufferItem *			paraBi,
				int				line,
				const ParagraphFrame *		pf,
				const DocumentRectangle *	drLine,
				void *				vps,
				DrawingContext *		dc,
				const BlockOrigin *		bo )
    {
    const LayoutContext *	lc= &(dc->dcLayoutContext);
    BufferDocument *		bd= lc->lcDocument;
    const DocumentProperties *	dp= &(bd->bdProperties);
    PrintingState *		ps= (PrintingState *)vps;
    const PostScriptFontList *	psfl= lc->lcPostScriptFontList;

    const TextLine *		tl= paraBi->biParaLines+ line;
    int				part= tl->tlFirstParticule;
    const TextParticule *	tp= paraBi->biParaParticules+ part;

    int				accepted;
    TextLine			boxLine;
    int				baseline;
    int				lineHeight;

    ParticuleData *		pd;

    if  ( docPsClaimParticuleData( paraBi->biParaParticuleCount, &pd ) )
	{ LDEB(paraBi->biParaParticuleCount); return -1;	}

    boxLine= *tl;
    accepted= docLayoutLineBox( bd, &boxLine, paraBi, part, psfl, pd, pf );

    if  ( accepted < 1 )
	{ LDEB(accepted); return -1;	}

    if  ( ps->psInsideLink )
	{ ps->psLinkRectLeft= pd->pdX0;	}

    baseline= tl->tlTopPosition.lpPageYTwips+ TL_BASELINE( tl );
    lineHeight= tl->tlDescY1- tl->tlAscY0;


    psPrintTextLine( ps, dc, bo->boXShift, bo->boYShift,
			&(dp->dpFontList), psfl,
			paraBi, part, tp, accepted, pd,
			baseline, tl->tlTopPosition.lpPageYTwips,
			lineHeight );

    return accepted;
    }

