/************************************************************************/
/*									*/
/*  Print PostScript, include PDF marks to preserve links when the	*/
/*  PostScript is converted to PDF.					*/
/*									*/
/*  Though the Adobe Technical Note #5150: "pdfmark Reference Manual"	*/
/*  exhaustively documents the pdfmark functionality, the book:		*/
/*  Merz, Thomas, "Postscript & Acrobat/PDF", Springer-Verlag, Berlin	*/
/*  &c, 1996, ISBN 3-540-60854-0. gives a summary on page 289 that is	*/
/*  much easier to understand.						*/
/*									*/
/************************************************************************/

#   include	"tedConfig.h"

#   include	<stddef.h>
#   include	<stdio.h>
#   include	<stdlib.h>
#   include	<string.h>

#   include	<sioStdio.h>
#   include	<appWinMeta.h>
#   include	<appMatchFont.h>

#   include	"docDraw.h"
#   include	"docLayout.h"
#   include	"docPsPrint.h"

#   include	<appDebugon.h>

#   define	SHOW_PAGE_GRID	0

/************************************************************************/
/*									*/
/*  Cause subsequent drawing to be done in a certain color.		*/
/*									*/
/*  NOTE that background drawing with color= 0 is against the RTF idea	*/
/*	of the default background color: transparent.			*/
/*									*/
/************************************************************************/

static int docPsSetColorRgb(	DrawingContext *	dc,
				void *			vps,
				const RGB8Color *	rgb8 )
    {
    PrintingState *		ps= (PrintingState *)vps;

    psSetRgb8Color( ps, rgb8 );

    return 0;
    }

static int docPsSetFont(	DrawingContext *	dc,
				void *			vps,
				int			textAttrNumber,
				const TextAttribute *	ta )
    {
    const LayoutContext *	lc= &(dc->dcLayoutContext);
    BufferDocument *		bd= lc->lcDocument;
    DocumentFontList *		dfl= &(bd->bdProperties.dpFontList);
    PrintingState *		ps= (PrintingState *)vps;
    const AfmFontInfo *		afi;

    const PostScriptFontList *	psfl= lc->lcPostScriptFontList;

    afi= appGetFontInfoForAttribute( ta, dfl, psfl );
    if  ( ! afi )
	{ LDEB(textAttrNumber); return -1; }

    psSetFont( ps, afi, ta );

    return 0;
    }

/************************************************************************/
/*									*/
/*  Printing of borders.						*/
/*									*/
/*  1)  The border is around paragraphs. Both the border and its	*/
/*      are outside the frame.						*/
/*									*/
/************************************************************************/

static void psSolidBorderProc(	SimpleOutputStream *		sos,
				const char *			name )
    {
    sioOutPrintf( sos, "%% x y w h\n" );

    sioOutPrintf( sos, "/%s\n", name );
    sioOutPrintf( sos, "  {\n" );
    sioOutPrintf( sos, "  rectfill\n" );
    sioOutPrintf( sos, "  } bind def\n" );
    return;
    }

static void psDashHorBorderProc( SimpleOutputStream *		sos,
				const char *			name,
				const unsigned char *		dashes,
				int				dashCount )
    {
    int		i;
    int		dashLength= 0;

    sioOutPrintf( sos, "%% x y w h\n" );

    sioOutPrintf( sos, "/%s\n", name );
    sioOutPrintf( sos, "  {\n" );

    sioOutPrintf( sos, "  gsave\n" );

    for ( i= 0; i < dashCount; i++ )
	{ dashLength += dashes[i];	}

    /* dx y th x0 */
    if  ( dashLength > 0 )
	{
	sioOutPrintf( sos, "  dup %d mod ", dashLength );
	/* x y w h offset */
	}
    else{
	sioOutPrintf( sos, "  " );
	}

    sioOutPrintf( sos, "[" );
    for ( i= 0; i < dashCount; i++ )
	{ sioOutPrintf( sos, " %d", dashes[i] ); }
    sioOutPrintf( sos, " ]" );

    if  ( dashLength > 0 )
	{
	/* x y w h offset [] */
	sioOutPrintf( sos, " exch setdash\n" );
	}
    else{
	/* x y w h [] */
	sioOutPrintf( sos, " 0 setdash\n" );
	}

    /* x y w h */
    sioOutPrintf( sos, "  dup setlinewidth 0 setlinecap\n" );
    /* x y w h -> x w y h -> x w y+h/2 ->  w x y+h/2 */
    sioOutPrintf( sos, "  3 -1 roll exch 1 add 2 div add 3 -1 roll exch\n" );

    /* dx y x0 */
    sioOutPrintf( sos, "  newpath moveto " );

    /* dx */
    sioOutPrintf( sos, " 0 rlineto stroke\n" );

    sioOutPrintf( sos, "  grestore\n" );
    sioOutPrintf( sos, "  } bind def\n" );
    }

static void psDashVerBorderProc( SimpleOutputStream *		sos,
				const char *			name,
				const unsigned char *		dashes,
				int				dashCount )
    {
    int		i;
    int		dashLength= 0;

    sioOutPrintf( sos, "%% x y w h\n" );

    sioOutPrintf( sos, "/%s\n", name );
    sioOutPrintf( sos, "  {\n" );

    sioOutPrintf( sos, "  gsave\n" );

    for ( i= 0; i < dashCount; i++ )
	{ dashLength += dashes[i];	}

    /* x y w h */
    if  ( dashLength > 0 )
	{
	sioOutPrintf( sos, "  dup %d mod ", dashLength );
	/* x y w h offset */
	}
    else{
	sioOutPrintf( sos, "  " );
	}

    sioOutPrintf( sos, "[" );
    for ( i= 0; i < dashCount; i++ )
	{ sioOutPrintf( sos, " %d", dashes[i] ); }
    sioOutPrintf( sos, " ]" );

    if  ( dashLength > 0 )
	{
	/* x y w h offset [] */
	sioOutPrintf( sos, " exch setdash\n" );
	}
    else{
	/* x y w h [] */
	sioOutPrintf( sos, " 0 setdash\n" );
	}

    /* x y w h */
    sioOutPrintf( sos, "  exch dup setlinewidth 0 setlinecap\n" );
    /* x y h w -> y h x w -> y h x+w/2 -> h x+w/2 y */
    sioOutPrintf( sos, "  4 -1 roll exch 1 add 2 div add 3 -1 roll\n" );

    /* h x+w/2 y */
    sioOutPrintf( sos, "  newpath moveto " );

    /* h */
    sioOutPrintf( sos, " 0 exch rlineto stroke\n" );

    sioOutPrintf( sos, "  grestore\n" );
    sioOutPrintf( sos, "  } bind def\n" );
    }

static const unsigned char PSborderDASH[]=	{ 60, 60 };
static const unsigned char PSborderDOT[]=	{ 20, 20 };
static const unsigned char PSborderDASHD[]=	{ 60, 40, 20, 40 };
static const unsigned char PSborderDASHDD[]=	{ 60, 40, 20, 40, 20, 40 };

static void psDefineBorderProcs(	SimpleOutputStream *	sos )
    {
    psSolidBorderProc( sos, "h-brdrs" );

    psDashHorBorderProc( sos, "h-brdrdash",
				    PSborderDASH, sizeof(PSborderDASH) );
    psDashHorBorderProc( sos, "h-brdrdot",
				    PSborderDOT, sizeof(PSborderDOT) );
    psDashHorBorderProc( sos, "h-brdrdashd",
				    PSborderDASHD, sizeof(PSborderDASHD) );
    psDashHorBorderProc( sos, "h-brdrdashdd",
				    PSborderDASHDD, sizeof(PSborderDASHDD) );

    psSolidBorderProc( sos, "v-brdrs" );

    psDashVerBorderProc( sos, "v-brdrdash",
				    PSborderDASH, sizeof(PSborderDASH) );
    psDashVerBorderProc( sos, "v-brdrdot",
				    PSborderDOT, sizeof(PSborderDOT) );
    psDashVerBorderProc( sos, "v-brdrdashd",
				    PSborderDASHD, sizeof(PSborderDASHD) );
    psDashVerBorderProc( sos, "v-brdrdashdd",
				    PSborderDASHDD, sizeof(PSborderDASHDD) );

    }

/************************************************************************/
/*									*/
/*  Draw a vertical border.						*/
/*									*/
/************************************************************************/

static int docPsPrintVerticalBorder(
				const BorderProperties *	bpVert,
				const DocumentRectangle *	dr,
				PrintingState *			ps,
				struct DrawingContext *		dc )
    {
    int		wide= dr->drX1- dr->drX0+ 1;
    int		high= dr->drY1- dr->drY0+ 1;

    docDrawSetColorNumber( dc, (void *)ps, bpVert->bpColor );

    switch( bpVert->bpStyle )
	{
	case DOCbsDASH:
	    sioOutPrintf( ps->psSos, "%d %d %d %d v-brdrdash ",
					     dr->drX0, dr->drY0, wide, high );
	    break;

	case DOCbsDOT:
	    sioOutPrintf( ps->psSos, "%d %d %d %d v-brdrdot ",
					     dr->drX0, dr->drY0, wide, high );
	    break;

	case DOCbsDASHD:
	    sioOutPrintf( ps->psSos, "%d %d %d %d v-brdrdashd ",
					     dr->drX0, dr->drY0, wide, high );
	    break;

	case DOCbsDASHDD:
	    sioOutPrintf( ps->psSos, "%d %d %d %d v-brdrdashdd ",
					     dr->drX0, dr->drY0, wide, high );
	    break;

	default:
	    sioOutPrintf( ps->psSos, "%d %d %d %d v-brdrs ",
					     dr->drX0, dr->drY0, wide, high );
	    break;
	}

    return 0;
    }

/************************************************************************/
/*									*/
/*  Draw an horizontal border.						*/
/*									*/
/************************************************************************/

static int docPsPrintHorizontalBorder(
				const BorderProperties *	bpHor,
				const DocumentRectangle *	dr,
				PrintingState *			ps,
				struct DrawingContext *		dc )
    {
    int		wide= dr->drX1- dr->drX0+ 1;
    int		high= dr->drY1- dr->drY0+ 1;

    docDrawSetColorNumber( dc, (void *)ps, bpHor->bpColor );

    switch( bpHor->bpStyle )
	{
	case DOCbsDASH:
	    sioOutPrintf( ps->psSos, "%d %d %d %d h-brdrdash ",
					     dr->drX0, dr->drY0, wide, high );
	    break;

	case DOCbsDOT:
	    sioOutPrintf( ps->psSos, "%d %d %d %d h-brdrdot ",
					     dr->drX0, dr->drY0, wide, high );
	    break;

	case DOCbsDASHD:
	    sioOutPrintf( ps->psSos, "%d %d %d %d h-brdrdashd ",
					     dr->drX0, dr->drY0, wide, high );
	    break;

	case DOCbsDASHDD:
	    sioOutPrintf( ps->psSos, "%d %d %d %d h-brdrdashdd ",
					     dr->drX0, dr->drY0, wide, high );
	    break;

	default:
	    sioOutPrintf( ps->psSos, "%d %d %d %d h-brdrs ",
					     dr->drX0, dr->drY0, wide, high );
	    break;
	}

    return 0;
    }

/************************************************************************/
/*									*/
/*  Draw background ornaments.						*/
/*									*/
/************************************************************************/

static int docPsPrintOrnaments(	const BlockOrnaments *		bo,
				int				page,
				const DocumentRectangle *	drOutside,
				const DocumentRectangle *	drInside,
				void *				through,
				struct DrawingContext *		dc )
    {
    PrintingState *		ps= (PrintingState *)through;
    const LayoutContext *	lc= &(dc->dcLayoutContext);
    const BufferDocument *	bd= lc->lcDocument;
    const DocumentProperties *	dp= &(bd->bdProperties);

    int				done= 0;

    if  ( PROPmaskISSET( &(bo->boPropMask), ORNdrawSHADE ) )
	{
	int			isFilled= 0;
	RGB8Color		rgb8;

	if  ( docGetSolidRgbShadeOfItem( &isFilled, &rgb8,
						bd, &(bo->boShading) ) )
	    { LDEB(1);	}

	if  ( isFilled )
	    {
	    docDrawSetColorRgb( dc, (void *)ps, &rgb8 );

	    psFillRectangle( ps, 
			    drInside->drX0, drInside->drY0,
			    drInside->drX1- drInside->drX0+ 1,
			    drInside->drY1- drInside->drY0+ 1 );

	    dc->dcCurrentColorSet= 0;
	    }

	if  ( bo->boShading.isPattern != DOCspSOLID )
	    {
	    RGB8Color			cf;
	    SimpleOutputStream *	sos= ps->psSos;

	    static const char * const procs[]=
		{
		(const char *)0, /* solid */
		"f-horiz",
		"f-vert",
		"f-fdiag",
		"f-bdiag",
		"f-cross",
		"f-dcross",
		"f-dkhoriz",
		"f-dkvert",
		"f-dkfdiag",
		"f-dkbdiag",
		"f-dkcross",
		"f-dkdcross",
		};

	    bmInitRGB8Color( &cf );
	    cf.rgb8Red= cf.rgb8Green= cf.rgb8Blue= 0;

	    if  ( bo->boShading.isForeColor > 0			&&
		  bo->boShading.isForeColor < dp->dpColorCount	)
		{ cf= dp->dpColors[bo->boShading.isForeColor];	}

	    docDrawSetColorRgb( dc, (void *)ps, &cf );

	    psRectanglePath( ps, 
			    drInside->drX0, drInside->drY0,
			    drInside->drX1- drInside->drX0+ 1,
			    drInside->drY1- drInside->drY0+ 1 );

	    sioOutPrintf( sos, "%s\n", procs[bo->boShading.isPattern] );
	    }
	}

    if  ( PROPmaskISSET( &(bo->boPropMask), ORNdrawTOP_BORDER ) )
	{
	DocumentRectangle	drBorder= *drOutside;

	drBorder.drY1= drInside->drY0- 1;

	docPsPrintHorizontalBorder( &(bo->boTopBorder), &drBorder, ps, dc );
	done= 1;
	}

    if  ( PROPmaskISSET( &(bo->boPropMask), ORNdrawLEFT_BORDER ) )
	{
	DocumentRectangle	drBorder= *drOutside;

	drBorder.drY0= drInside->drY0;
	drBorder.drY1= drInside->drY1;
	drBorder.drX1= drInside->drX0- 1;

	docPsPrintVerticalBorder( &(bo->boLeftBorder), &drBorder, ps, dc );
	done= 1;
	}

    if  ( PROPmaskISSET( &(bo->boPropMask), ORNdrawRIGHT_BORDER ) )
	{
	DocumentRectangle	drBorder= *drOutside;

	drBorder.drY0= drInside->drY0;
	drBorder.drY1= drInside->drY1;
	drBorder.drX0= drInside->drX1+ 1;

	docPsPrintVerticalBorder( &(bo->boRightBorder), &drBorder, ps, dc );
	done= 1;
	}

    if  ( PROPmaskISSET( &(bo->boPropMask), ORNdrawBOTTOM_BORDER ) )
	{
	DocumentRectangle	drBorder= *drOutside;

	drBorder.drY0= drInside->drY1+ 1;

	docPsPrintHorizontalBorder( &(bo->boBottomBorder), &drBorder, ps, dc );
	done= 1;
	}

    if  ( done )
	{ sioOutPrintf( ps->psSos, "\n" ); }

    return 0;
    }

/************************************************************************/

#   if  SHOW_PAGE_GRID

static void docPsPageBoxes(	DrawingContext *		dc,
				void *				vps,
				const DocumentGeometry *	dg )
    {
    PrintingState *		ps= (PrintingState *)vps;

    DocumentRectangle		drBBox;
    DocumentRectangle		drBody;
    DocumentRectangle		drHead;
    DocumentRectangle		drFoot;

    RGB8Color			rgb8;

    rgb8.rgb8Alpha= 255;

    utilDocumentGeometryGetPageBoundingBox( &drBBox, dg, 1, 1 );
    utilDocumentGeometryGetBodyRect( &drBody, dg );
    utilDocumentGeometryGetHeaderRect( &drHead, dg );
    utilDocumentGeometryGetFooterRect( &drFoot, dg );

    rgb8.rgb8Red= 255;
    rgb8.rgb8Green= 199;
    rgb8.rgb8Blue= 199;
    docPsSetColorRgb( dc, vps, &rgb8 );
    sioOutPrintf( ps->psSos, "newpath" );
    sioOutPrintf( ps->psSos, " %d %d moveto", drHead.drX0, drHead.drY0 );
    sioOutPrintf( ps->psSos, " %d %d lineto", drHead.drX1, drHead.drY0 );
    sioOutPrintf( ps->psSos, " %d %d lineto", drHead.drX1, drHead.drY1 );
    sioOutPrintf( ps->psSos, " %d %d lineto", drHead.drX0, drHead.drY1 );
    sioOutPrintf( ps->psSos, " %d %d lineto", drHead.drX0, drHead.drY0 );
    sioOutPrintf( ps->psSos, " fill\n" );

    rgb8.rgb8Red= 199;
    rgb8.rgb8Green= 199;
    rgb8.rgb8Blue= 255;
    docPsSetColorRgb( dc, vps, &rgb8 );
    sioOutPrintf( ps->psSos, "newpath" );
    sioOutPrintf( ps->psSos, " %d %d moveto", drFoot.drX0, drFoot.drY0 );
    sioOutPrintf( ps->psSos, " %d %d lineto", drFoot.drX1, drFoot.drY0 );
    sioOutPrintf( ps->psSos, " %d %d lineto", drFoot.drX1, drFoot.drY1 );
    sioOutPrintf( ps->psSos, " %d %d lineto", drFoot.drX0, drFoot.drY1 );
    sioOutPrintf( ps->psSos, " %d %d lineto", drFoot.drX0, drFoot.drY0 );
    sioOutPrintf( ps->psSos, " fill\n" );

    rgb8.rgb8Red= 199;
    rgb8.rgb8Green= 255;
    rgb8.rgb8Blue= 199;
    docPsSetColorRgb( dc, vps, &rgb8 );
    sioOutPrintf( ps->psSos, "newpath" );
    sioOutPrintf( ps->psSos, " %d %d moveto", drBody.drX0, drBody.drY0 );
    sioOutPrintf( ps->psSos, " %d %d lineto", drBody.drX1, drBody.drY0 );
    sioOutPrintf( ps->psSos, " %d %d lineto", drBody.drX1, drBody.drY1 );
    sioOutPrintf( ps->psSos, " %d %d lineto", drBody.drX0, drBody.drY1 );
    sioOutPrintf( ps->psSos, " %d %d lineto", drBody.drX0, drBody.drY0 );
    sioOutPrintf( ps->psSos, " fill\n" );

    rgb8.rgb8Red= 0;
    rgb8.rgb8Green= 255;
    rgb8.rgb8Blue= 0;
    docPsSetColorRgb( dc, vps, &rgb8 );
    sioOutPrintf( ps->psSos, "newpath" );
    sioOutPrintf( ps->psSos, " %d %d moveto", drBBox.drX0, drBBox.drY0 );
    sioOutPrintf( ps->psSos, " %d %d lineto", drBBox.drX1, drBBox.drY1 );
    sioOutPrintf( ps->psSos, " %d %d moveto", drBBox.drX0, drBBox.drY1 );
    sioOutPrintf( ps->psSos, " %d %d lineto", drBBox.drX1, drBBox.drY0 );
    sioOutPrintf( ps->psSos, " stroke\n" );
    }

#   endif

/************************************************************************/
/*									*/
/*  Skip to the next page.						*/
/*									*/
/*  1)  If anything is printed on the page, and header/footer printing	*/
/*      is postponed, print headers and footers for this page.		*/
/*  2)  If omitHeadersOnEmptyPages is set, dc->dcPostponeHeadersFooters	*/
/*	is derived from it. So any header printing is covered by (1)	*/
/*  3)  If the page is completely empty, skip it.			*/
/*  4)  if the page is empty, but not have been so if headers and	*/
/*	footers were printed, emit it. Otherwise skip it. This sounds a	*/
/*	bit strange, but the customer is always right. Actually, it has	*/
/*	a purpose: In simple documents, you do not want to waste paper.	*/
/*	So you should not print empty pages. In more complicated	*/
/*	documents, you do not want to print pages with only a header	*/
/*	and a footer to prevent fraud. For pagination purposes, you do	*/
/*	want to emit the page however. Now in stead of make the user	*/
/*	say what she wants with a multitude of parameters, we		*/
/*	distinguish between simple and complicated documents by		*/
/*	assuming that a complicated document has headers and footers.	*/
/*									*/
/************************************************************************/

static int docPsFinishPage(	void *				vps,
				DrawingContext *		dc,
				BufferItem *			bodySectBi,
				int				page,
				int				asLast )
    {
    PrintingState *	ps= (PrintingState *)vps;

    int			pageWasMarked;
    int			pageIsMarked;
    int			pageHasHeader= dc->dcDocHasPageHeaders;
    int			pageHasFooter= dc->dcDocHasPageFooters;
    int			skip= 0;

#   if 0
    docPsPageBoxes( dc, vps,
	&(dc->dcLayoutContext.lcDocument->bdProperties.dpGeometry) );
#   endif

    pageWasMarked= ps->psLastPageMarked >= ps->psPagesPrinted;

    /*
    sioOutPrintf( ps->psSos, "%% pageWasMarked= %d\n", pageWasMarked );
    sioOutPrintf( ps->psSos, "%% dcDocHasPageHeaders= %d\n",
					    dc->dcDocHasPageHeaders );
    sioOutPrintf( ps->psSos, "%% dcDocHasPageFooters= %d\n",
					    dc->dcDocHasPageFooters );
    */

    /*  1  */
    if  ( pageWasMarked			&&
	  dc->dcPostponeHeadersFooters	)
	{
	if  ( dc->dcDocHasPageHeaders				&&
	      docDrawPageHeader( bodySectBi, vps, dc, page )	)
	    { LLDEB(dc->dcDocHasPageHeaders,page);	}

	if  ( dc->dcDocHasPageFooters				&&
	      docDrawPageFooter( bodySectBi, vps, dc, page )	)
	    { LLDEB(dc->dcDocHasPageFooters,page);	}
	}

    /*  2  */

    pageIsMarked= ps->psLastPageMarked >= ps->psPagesPrinted;
    /*
    sioOutPrintf( ps->psSos, "%% pageIsMarked= %d\n", pageIsMarked );
    */

    /*  3  */
    if  ( ! pageIsMarked && ps->psPrintGeometry.pgSkipBlankPages )
	{ skip= 1;	}

    /*  4  */
    if  ( ! pageIsMarked )
	{
	const LayoutContext *	lc= &(dc->dcLayoutContext);
	const BufferDocument *	bd= lc->lcDocument;

	int				inExternalItem;
	DocumentTree *			ei;
	int				isEmpty;

	if  ( dc->dcDocHasPageHeaders )
	    {
	    inExternalItem= docWhatPageHeader( &ei, &isEmpty,
							bodySectBi, page, bd );
	    if  ( ! ei || ! ei->eiRoot || isEmpty )
		{ pageHasHeader= 0;	}
	    }

	if  ( dc->dcDocHasPageFooters )
	    {
	    inExternalItem= docWhatPageFooter( &ei, &isEmpty,
							bodySectBi, page, bd );
	    if  ( ! ei || ! ei->eiRoot || isEmpty )
		{ pageHasFooter= 0;	}
	    }
	}

    /*
    sioOutPrintf( ps->psSos, "%% pageHasHeader= %d\n", pageHasHeader );
    sioOutPrintf( ps->psSos, "%% pageHasFooter= %d\n", pageHasFooter );
    */

    if  ( ! pageIsMarked			&&
	  ps->psPrintGeometry.pgSkipEmptyPages	&&
	  ! pageHasHeader			&&
	  ! pageHasFooter			)
	{ skip= 1;	}


    if  ( skip )
	{ psAbortPage( ps, page, asLast );	}
    else{ psFinishPage( ps, page, asLast );	}

    return 0;
    }

static int docPsPrintStartPage(	void *				vps,
				const DocumentGeometry *	dgPage,
				DrawingContext *		dc,
				int				page )
    {
    PrintingState *	ps= (PrintingState *)vps;

    dc->dcCurrentTextAttributeSet= 0;
    dc->dcCurrentColorSet= 0;

    psRefreshNupSchema( ps, dgPage );

    psStartPage( ps, page );

#   if SHOW_PAGE_GRID
    docPsPageBoxes( dc, vps,
	&(dc->dcLayoutContext.lcDocument->bdProperties.dpGeometry) );
#   endif

    return 0;
    }

/************************************************************************/
/*									*/
/*  Leave a trace in the PDF document information.			*/
/*									*/
/************************************************************************/

static void psSaveInfo(		const char *		tag,
				const char *		info,
				PrintingState *		ps )
    {
    if  ( ! info )
	{ return;	}

    sioOutPrintf( ps->psSos, "  %s (", tag );
    psPrintString( ps->psSos, (unsigned char *)info, strlen( info ),
								ps->ps7Bits );
    sioOutPrintf( ps->psSos, ")\n" );
    }

static void psSaveDate(		const char *		tag,
				const struct tm *	tm,
				PrintingState *		ps )
    {
    char	scratch[40+1];

    if  ( tm->tm_mday == 0 )
	{ return;	}

    if  ( strftime( scratch, sizeof(scratch)- 1, "D:%Y%m%d%H%M", tm ) < 1 )
	{ LDEB(1); return;	}

    psSaveInfo( tag, scratch, ps );

    return;
    }

/************************************************************************/
/*									*/
/*  Print a range of pages in a document.				*/
/*									*/
/************************************************************************/

static int docPsPrintPageRange(	PrintingState *		ps,
				DrawingContext *	dc,
				BufferItem *		bodyBi,
				int			firstPage,
				int			lastPage,
				int			asLast )
    {
    const LayoutContext *	lc= &(dc->dcLayoutContext);
    BufferDocument *		bd= lc->lcDocument;
    int				i;
    LayoutPosition		lpBelow;

    docInitLayoutPosition( &lpBelow );

    for ( i= 0; i < bodyBi->biChildCount; i++ )
	{
	if  ( bodyBi->biChildren[i]->biBelowPosition.lpPage >= firstPage )
	    { break;	}
	}

    if  ( i >= bodyBi->biChildCount )
	{ LDEB(i); return -1; }

    docPsPrintStartPage( (void *)ps,
	    &(bodyBi->biChildren[i]->biSectDocumentGeometry), dc, firstPage );

    if  ( ! dc->dcPostponeHeadersFooters )
	{
	docDrawPageHeader( bodyBi->biChildren[i], (void *)ps, dc, firstPage );
	}

    if  ( docDrawShapesForTree( &(bd->bdBody),
				    bodyBi->biChildren[i],
				    (void *)ps, dc, 1, firstPage ) )
	{ LDEB(1);	}

    docDrawItem( &lpBelow, bodyBi, (void *)ps, dc );

    if  ( lastPage < 0 )
	{ lastPage= bodyBi->biBelowPosition.lpPage;	}

    for ( i= bodyBi->biChildCount- 1; i >= 0; i-- )
	{
	if  ( bodyBi->biChildren[i]->biTopPosition.lpPage <= lastPage )
	    { break;	}
	}

    if  ( i < 0 )
	{ LDEB(i); return -1;	}

    if  ( docDrawShapesForTree( &(bd->bdBody),
				    bodyBi->biChildren[i],
				    (void *)ps, dc, 0, lastPage ) )
	{ LDEB(1);	}

    if  ( ! dc->dcPostponeHeadersFooters )
	{
	docDrawPageFooter( bodyBi->biChildren[i], (void *)ps, dc, lastPage );
	}

    docPsFinishPage( (void *)ps, dc, bodyBi->biChildren[i],
						    lastPage, asLast );

    return 0;
    }

/************************************************************************/
/*									*/
/*  Print a document.							*/
/*									*/
/************************************************************************/

int docPsPrintDocument(	SimpleOutputStream *		sos,
			const char *			title,
			const char *			applicationName,
			const char *			applicationReference,
			const char *			fontDirectory,
			double				shadingMesh,
			const LayoutContext *		lc,
			const PrintGeometry *		pg )
    {
    BufferDocument *		bd= lc->lcDocument;
    DocumentProperties *	dp= &(bd->bdProperties);
    DocumentGeometry *		dg= &(dp->dpGeometry);
    BufferItem *		bodyBi= bd->bdBody.eiRoot;

    PostScriptTypeList		pstl;

    DrawingContext		dc;
    PrintingState		ps;

    int				firstPage= pg->pgFirstPage;
    int				lastPage= pg->pgLastPage;

    INIT_LAYOUT_EXTERNAL	initLayoutExternal= (INIT_LAYOUT_EXTERNAL)0;

    const PostScriptFontList *	psfl= lc->lcPostScriptFontList;

    psInitPostScriptFaceList( &pstl );
    pstl.pstlFontDirectory= fontDirectory;

    docInitDrawingContext( &dc );

    dc.dcSetColorRgb= docPsSetColorRgb;
    dc.dcSetFont= docPsSetFont;
    dc.dcDrawShape= docPsPrintDrawDrawingShape;

    dc.dcDrawTextLine= docPsPrintTextLine;
    dc.dcDrawOrnaments= docPsPrintOrnaments;
    dc.dcFinishPage= docPsFinishPage;
    dc.dcStartPage= docPsPrintStartPage;
    dc.dcInitLayoutExternal= initLayoutExternal;

    dc.dcLayoutContext= *lc;

    dc.dcFirstPage= pg->pgFirstPage;
    dc.dcLastPage= pg->pgLastPage;
    dc.dcDrawExternalItems= 1;
    dc.dcPostponeHeadersFooters= 0;

    if  ( pg->pgOmitHeadersOnEmptyPages )
	{ dc.dcPostponeHeadersFooters= 1;	}

    psInitPrintingState( &ps );
    ps.psSos= sos;
    ps.psUsePostScriptFilters= pg->pgUsePostScriptFilters;
    ps.psUsePostScriptIndexedImages= pg->pgUsePostScriptIndexedImages;
    ps.ps7Bits= pg->pg7Bits;

    ps.psPrintGeometry.pgEmbedFonts= pg->pgEmbedFonts;
    ps.psPrintGeometry.pgSkipEmptyPages= pg->pgSkipEmptyPages;
    ps.psPrintGeometry.pgSkipBlankPages= pg->pgSkipBlankPages;
    ps.psPrintGeometry.pgOmitHeadersOnEmptyPages=
					    pg->pgOmitHeadersOnEmptyPages;

    docInquireHeadersFooters( &(dc.dcDocHasPageHeaders),
				    &(dc.dcDocHasPageFooters), bodyBi );

    if  ( dp->dpTitle && dp->dpTitle[0] )
	{ title= (char *)dp->dpTitle;	}

    if  ( psSetNupSchema( &ps, dg, pg, dc.dcDocHasPageHeaders,
						    dc.dcDocHasPageFooters ) )
	{ LDEB(1); return -1;	}

    if  ( docPsPrintGetDocumentFonts( bd, &pstl, psfl ) )
	{ LDEB(1); return -1;	}

    psStartDSCDocument( &ps, &pstl,
			    title, applicationName, applicationReference );

    sioOutPrintf( sos, "%%%%BeginProlog\n" );

    psSetUtf8ShowImplementation( sos );
    psSetMvsImplementation( sos );

    docPsSaveTabLeaderProcedures( sos );
    psDefineBorderProcs( sos );

    psSetPdfmarkEmulation( sos );
    psSetRectfillEmulation( sos );
    psSetSelectfontEmulation( sos );

    psDefineEpsProcs( sos );

    if  ( pg->pgEmbedFonts )
	{ psIncludeFonts( sos, &pstl );	}
    psSelectFontProcedures( sos, &pstl, /*allFonts=*/ 0 );

    appMetaDefineProcsetPs( sos );
    psSetPatternImplementation( sos, shadingMesh );

    sioOutPrintf( sos, "%%%%EndProlog\n" );
    sioOutPrintf( sos, "%%%%BeginSetup\n" );

#   if 1
    sioOutPrintf( sos, "<< /PageSize [%d %d] >> setpagedevice\n",
	    ( ps.psPrintGeometry.pgSheetGeometry.dgPageWideTwips+ 19 )/ 20, 
	    ( ps.psPrintGeometry.pgSheetGeometry.dgPageHighTwips+ 19 )/ 20 );
#   endif

    if  ( ps.psUsePostScriptFilters )
	{ psImageQualityDistillerparams( sos );	}

    if  ( pg->pgCustomPsSetupFilename )
	{
	SimpleInputStream *	sis;

	sis= sioInStdioOpen( pg->pgCustomPsSetupFilename );
	if  ( ! sis )
	    { SXDEB(pg->pgCustomPsSetupFilename,sis);	}
	else{
	    unsigned char		buf[500];
	    int				got;

	    while( ( got= sioInReadBytes( sis, buf, sizeof(buf) ) ) > 0 )
		{ sioOutWriteBytes( sos, buf, got );	}

	    sioInClose( sis );
	    }
	}

    if  ( docHasDocumentInfo( dp ) )
	{
	char *			scratch;

	scratch= malloc( strlen( applicationName )+ 2+
					strlen( applicationReference )+ 2 );
	if  ( ! scratch )
	    { XDEB(scratch); return -1;	}
	sprintf( scratch, "%s: %s", applicationName, applicationReference );

	sioOutPrintf( sos, "[\n" );

	psSaveInfo( "/Title",		dp->dpTitle, &ps );
	psSaveInfo( "/Author",		dp->dpAuthor, &ps );
	psSaveInfo( "/Subject",		dp->dpSubject, &ps );
	psSaveInfo( "/Keywords",	dp->dpKeywords, &ps );
	psSaveInfo( "/Creator",		scratch, &ps );
	psSaveInfo( "/Producer",	scratch, &ps );

	psSaveDate( "/ModDate",		&(dp->dpRevtim), &ps );
	psSaveDate( "/CreationDate",	&(dp->dpCreatim), &ps );

	sioOutPrintf( sos, "/DOCINFO pdfmark\n\n" );

	free( scratch );
	}

    sioOutPrintf( sos, "%%%%EndSetup\n" );

    if  ( firstPage < 0 )
	{ firstPage= 0;	}

    if  ( pg->pgPrintOddSides		&&
	  pg->pgPrintEvenSides		&&
	  ! pg->pgPrintSheetsReverse	&&
	  ! pg->pgPrintBookletOrder	)
	{
	if  ( docPsPrintPageRange( &ps, &dc, bodyBi,
				    firstPage, lastPage, /* asLast */ 1 ) )
	    { LDEB(firstPage); return -1;	}
	}
    else{
	if  ( pg->pgPrintBookletOrder )
	    { LDEB(pg->pgPrintBookletOrder); }

	if  ( docPsPrintPageRange( &ps, &dc, bodyBi,
				    firstPage, lastPage, /* asLast */ 1 ) )
	    { LDEB(firstPage); return -1;	}
	}

    psCleanPrintingState( &ps );

    psCleanPostScriptFaceList( &pstl );

    docResetExternalItemLayout( bd );

    return 0;
    }

/************************************************************************/
/*									*/
/*  Booklet printing.							*/
/*									*/
/*  Document has n pages.						*/
/*  On the front (odd side) of sheet s left,right: page n-2s-1, 2s	*/
/*  On the back (even side) of sheet s left,right: page 2s-1, n-2s-2	*/
/*									*/
/************************************************************************/

