/************************************************************************/
/*									*/
/*  Layout of a document.						*/
/*									*/
/************************************************************************/

#   include	"tedConfig.h"

#   include	<stddef.h>
#   include	<stdio.h>
#   include	<stdlib.h>
#   include	<limits.h>

#   include	"docLayout.h"
#   include	"docPageGrid.h"
#   include	"docPixels.h"

#   include	<appDebugon.h>

/************************************************************************/
/*									*/
/*  Initialize a layout job.						*/
/*									*/
/************************************************************************/

void docInitLayoutJob(	LayoutJob *	lj )
    {
    lj->ljChangedRectanglePixels= (DocumentRectangle *)0;
    layoutInitContext( &(lj->ljContext) );
    lj->ljChangedItem= (BufferItem *)0;

    lj->ljBalancePage= -1;
    lj->ljBalanceY1= 0;

    lj->ljBodySectBi= (const BufferItem *)0;

    lj->ljStartScreenParagraph= (START_SCREEN_PARAGRAPH)0;
    lj->ljLayoutScreenLine= (LAYOUT_SCREEN_LINE)0;
    lj->ljAdjustScreenRectangle= (ADJUST_SCREEN_RECT)0;

    return;
    }

void docCleanLayoutJob(	LayoutJob *	lj )
    {
    return;
    }

/************************************************************************/
/*									*/
/*  Invalidate the layout of all paragraphs in the modified range.	*/
/*									*/
/************************************************************************/

static void docInvalidateChangedLayout(	BufferItem *	biParaStart,
					BufferItem *	biParaEnd )
    {
    BufferItem *	bi= biParaStart;

    while( bi )
	{
	docInvalidateParagraphLayout( bi );

	if  ( bi == biParaEnd )
	    { break;	}

	bi= docNextParagraph( bi );
	if  ( ! bi )
	    { XDEB(bi); }
	}

    return;
    }

/************************************************************************/
/*									*/
/*  Redo layout of the relevant part of the document after editing.	*/
/*									*/
/************************************************************************/

int docLayoutInvalidateRange(	DocumentSelection *	dsLayout,
				const DocumentTree *	ei,
				EditRange *		er )
    {
    DocumentPosition	dp;
    BufferItem *	startBi;
    BufferItem *	endBi;

    if  ( er->erHead.epParaNr == 0 )
	{
	LDEB(er->erHead.epParaNr);
	er->erHead.epParaNr= 1;
	}

    startBi= docGetParagraphByNumber( ei, er->erHead.epParaNr );
    endBi= docGetParagraphByNumber( ei, er->erTail.epParaNr );

    if  ( ! startBi )
	{
	/*LXDEB(er->erHead.epParaNr,startBi);*/
	if  ( docFirstPosition( &dp, ei->eiRoot ) )
	    { LDEB(er->erHead.epParaNr); return -1;	}

	startBi= dp.dpBi;
	}
    if  ( ! endBi )
	{
	/* LXDEB(er->erTail.epParaNr,endBi); */
	if  ( docLastPosition( &dp, ei->eiRoot ) )
	    { LDEB(er->erTail.epParaNr); return -1;		}

	endBi= dp.dpBi;
	}

    docInvalidateChangedLayout( startBi, endBi );

    docInitDocumentSelection( dsLayout );
    dsLayout->dsHead.dpBi= startBi;
    dsLayout->dsHead.dpStroff= 0;

    dsLayout->dsTail.dpBi= endBi;
    dsLayout->dsTail.dpStroff= docParaStrlen( endBi );

    return 0;
    }

/************************************************************************/
/*									*/
/*  Determine the frame when a formatting task is (re) started for a	*/
/*  buffer item.							*/
/*									*/
/*  2)  Calculate the frame in which the text is to be laid out.	*/
/*  3)  If the preceding paragraph ends on the same page where this	*/
/*	nodes begins, reserve space for the footnotes upto the		*/
/*	beginning of this block and subtract the height from the buttom	*/
/*	of the frame.							*/
/*									*/
/************************************************************************/

int docLayoutGetInitialFrame(		BlockFrame *		bf,
					const LayoutJob *	lj,
					const LayoutPosition *	lpHere,
					BufferItem *		bi )
    {
    const LayoutContext *	lc= &(lj->ljContext);
    BufferDocument *		bd= lc->lcDocument;
    const BufferItem *		prevParaBi= (const BufferItem *)0;

    /*  2  */
    docBlockFrameTwips( bf, bi, lj->ljBodySectBi, bd,
					lpHere->lpPage, lpHere->lpColumn );

    /*  3  */
    if  ( bi->biInExternalItem == DOCinBODY )
	{ prevParaBi= docPrevParagraph( bi );	}

    if  ( prevParaBi						&&
	  prevParaBi->biBelowPosition.lpPage >= lpHere->lpPage	)
	{
	DocumentPosition		dpHere;
	int				partHere;

	if  ( docFirstPosition( &dpHere, bi ) )
	    { LDEB(1); return -1;	}
	partHere= 0;

	if  ( docCollectFootnotesFromColumn( bf, &dpHere, partHere, bd,
					lpHere->lpPage, lpHere->lpColumn ) )
	    { LDEB(lpHere->lpPage); return -1;	}
	}

    return 0;
    }

/************************************************************************/
/*									*/
/*  Adjust the bottom of an item to changes inside.			*/
/*									*/
/************************************************************************/

void docLayoutSetItemBottom(	int *			pChanged,
				BufferItem *		bi,
				const LayoutPosition *	lp,
				const LayoutContext *	lc,
				DocumentRectangle *	drChanged )
    {
    AppDrawingData *	add= lc->lcAdd;
    int			changed= 0;

    int			oldY1Pixels;
    int			newY1Pixels;

    oldY1Pixels= BI_BELOW_PIXELS( add, bi );

    if  ( ! DOC_SAME_POSITION( &(bi->biBelowPosition), lp ) )
	{ bi->biBelowPosition= *lp; changed= 1; }

    newY1Pixels= LP_YPIXELS( add, lp );

    if  ( oldY1Pixels < newY1Pixels )
	{
	if  ( drChanged					&&
	      drChanged->drY1 < newY1Pixels -1	)
	    { drChanged->drY1=  newY1Pixels -1;	}
	}

    if  ( oldY1Pixels > newY1Pixels )
	{
	if  ( drChanged							&&
	      drChanged->drY1 < oldY1Pixels- 1	)
	    { drChanged->drY1=  oldY1Pixels- 1;	}
	}

    if  ( changed )
	{ *pChanged= changed;	}

    return;
    }

/************************************************************************/
/*									*/
/*  Get the pixel rectangle for a rectangle that lives on one page.	*/
/*									*/
/************************************************************************/

void docGetPixelRect(		DocumentRectangle *		drPixels,
				const LayoutContext *		lc,
				const DocumentRectangle *	drTwips,
				int				page )
    {
    const AppDrawingData *	add= lc->lcAdd;

    LayoutPosition		lp0;
    LayoutPosition		lp1;

    lp0.lpPage= lp1.lpPage= page;
    lp0.lpPageYTwips= drTwips->drY0;
    lp1.lpPageYTwips= drTwips->drY1;

    drPixels->drX0= X_PIXELS( add, drTwips->drX0 );
    drPixels->drX1= X_PIXELS( add, drTwips->drX1 );
    drPixels->drY0= LP_YPIXELS( add, &lp0 );
    drPixels->drY1= LP_YPIXELS( add, &lp1 );

    return;
    }


void docGetPixelRectForPos(	DocumentRectangle *		drPixels,
				const LayoutContext *		lc,
				int				x0Twips,
				int				x1Twips,
				const LayoutPosition *		lpTop,
				const LayoutPosition *		lpBottom )
    {
    const AppDrawingData *	add= lc->lcAdd;

    drPixels->drX0= X_PIXELS( add, x0Twips );
    drPixels->drX1= X_PIXELS( add, x1Twips );
    drPixels->drY0= LP_YPIXELS( add, lpTop );
    drPixels->drY1= LP_YPIXELS( add, lpBottom );

    return;
    }

void docGetPageRectPixels(	DocumentRectangle *		drPixels,
				const LayoutContext *		lc,
				int				page0,
				int				page1 )
    {
    const AppDrawingData *	add= lc->lcAdd;

    drPixels->drX0= 0;
    drPixels->drX1= INT_MAX;

    drPixels->drY0= add->addPageStepPixels* page0;
    drPixels->drY1= add->addPageStepPixels* ( page1+ 1 )- 1;

    return;
    }

void docPixelRectangleForPositions(
				DocumentRectangle *		drPixels,
				const PositionGeometry *	pgB,
				const PositionGeometry *	pgE,
				const LayoutContext *		lc )
    {
    DocumentRectangle		drPixelsB;
    DocumentRectangle		drPixelsE;

    docGetPixelRectForPos( &drPixelsB, lc,
			    pgB->pgXTwips, pgB->pgXTwips,
			    &(pgB->pgTopPosition),
			    &(pgB->pgBottomPosition) );

    drPixelsB.drX0= drPixelsB.drX1= pgB->pgXPixels; /* Do not trust XTwips yet */


    docGetPixelRectForPos( &drPixelsE, lc,
			    pgE->pgXTwips, pgE->pgXTwips,
			    &(pgE->pgTopPosition),
			    &(pgE->pgBottomPosition) );
    drPixelsE.drX0= drPixelsE.drX1= pgE->pgXPixels; /* Do not trust XTwips yet */

    geoUnionRectangle( drPixels, &drPixelsB, &drPixelsE );

    return;
    }
