/************************************************************************/
/*									*/
/*  Layout of a document. Layout of a series of paragraphs in a common	*/
/*  parent.								*/
/*									*/
/************************************************************************/

#   include	"tedConfig.h"

#   include	<stddef.h>
#   include	<stdio.h>
#   include	<stdlib.h>

#   include	"docLayout.h"
#   include	"docPageGrid.h"

#   include	<appDebugon.h>

/************************************************************************/
/*									*/
/*  Preparation for vertical alignment of tabe cells. It is also used	*/
/*  to position the contents of a frame.				*/
/*									*/
/************************************************************************/

void docRedoParaStripLayout(	const LayoutJob *		lj,
				BlockFrame *			bf,
				const LayoutPosition *		lpFrom,
				BufferItem *			cellBi,
				int				childFrom,
				int				childUpto )
    {
    const LayoutContext *	lc= &(lj->ljContext);
    int				stopCode= FORMATstopREADY;
    int				para;
    BufferItem *		paraBi0= cellBi->biChildren[childFrom];

    int				toNextPage= 0;

    ParagraphLayoutPosition	plpRedo;

    docInitParagraphLayoutPosition( &plpRedo );

    docStripLayoutStartChild( &plpRedo, childFrom );
    plpRedo.plpPos= *lpFrom;

    /*  1  */
    if  ( docLayoutStartParagraph( &toNextPage, lj, paraBi0, bf, &plpRedo ) )
	{ LDEB(1); return;	}

    if  ( toNextPage )
	{ LDEB(toNextPage); return;	}

    for ( para= childFrom; para < childUpto; para++ )
	{
	int		changed= 0;
	int		accepted;
	BufferItem *	paraBi= cellBi->biChildren[para];

	docStripLayoutStartChild( &plpRedo, para );

	paraBi->biTopPosition= plpRedo.plpPos;

	accepted= docLayoutParaLines( &stopCode,
					&(plpRedo.plpParagraphFrame),
					&(plpRedo.plpPos), &(plpRedo.pspLine),
					bf, lj, paraBi, plpRedo.pspPart );
	if  ( accepted < 0 )
	    { LDEB(accepted); break;	}

	docLayoutSetItemBottom( &changed, paraBi, &(plpRedo.plpPos),
					    lc, lj->ljChangedRectanglePixels );
	}

    return;
    }

/************************************************************************/

static int docLayoutChildInStrip(
			int *				pStopCode,
			ParagraphLayoutPosition *	plp,
			BlockFrame *			bf,
			const LayoutJob *		lj,
			int				cellTopInset,
			BufferItem *			childBi )
    {
    switch( childBi->biLevel )
	{
	case DOClevPARA:
	    return docLayoutParagraphInStrip( pStopCode, plp, bf, lj,
						    cellTopInset, childBi );
	case DOClevROW:
	    return docLayoutRowInStrip( pStopCode, plp, bf, lj,
						    cellTopInset, childBi );
	default:
	    LDEB(childBi->biLevel); return -1;
	}
    }

/************************************************************************/

static int docLayoutGetChildFrame(	int *			pFrameNumber,
					FrameProperties *	fp,
					const BufferDocument *	bd,
					const BufferItem *	childBi )
    {
    int		isFrame= 0;
    int		frameNumber;

    switch( childBi->biLevel )
	{
	case DOClevPARA:
	    frameNumber= childBi->biParaFrameNumber;
	    break;
	case DOClevROW:
	    frameNumber= childBi->biRowFrameNumber;
	    break;
	default:
	    LDEB(childBi->biLevel);
	    frameNumber= -1;
	    break;
	}

    if  ( frameNumber >= 0 )
	{
	docGetFramePropertiesByNumber( fp, &(bd->bdFramePropertyList),
							    frameNumber );
	isFrame= DOCisFRAME( fp );
	}

    *pFrameNumber= frameNumber;
    return isFrame;
    }

/************************************************************************/
/*									*/
/*  Layout as much of a series of paragraph as fits on the current	*/
/*  page. (Actually the column on the page).				*/
/*									*/
/*  a)  Do not format children that belong to different frames.		*/
/*									*/
/************************************************************************/

int docLayoutStripChildren(	int *				pStopCode,
				ParagraphLayoutJob *		plj,
				BlockFrame *			bfFlow,
				const LayoutJob *		lj,
				BufferItem *			cellBi )
    {
    const LayoutContext *	lc= &(lj->ljContext);
    ParagraphLayoutPosition *	plp= &(plj->pljPos);
    const int			childFrom= plp->pspChild;
    BufferItem *		childBi= cellBi->biChildren[childFrom];

    BlockFrame *		bf= bfFlow;

    const BufferDocument *	bd= lc->lcDocument;

    FrameProperties		fpBegin;
    int				frameBegin;
    int				wasFrame;

    FrameProperties		fpRow;
    int				frameRow= -1;
    int				rowIsFrame= 0;

    BufferItem *		rowBi;

    if  ( ! docIsCellItem( cellBi ) )
	{ SDEB(docLevelStr(cellBi->biLevel)); return -1;	}
    if  ( childFrom >= cellBi->biChildCount )
	{ LLDEB(childFrom,cellBi->biChildCount); return -1;	}

    rowBi= docGetRowItem( cellBi );
    if  ( rowBi )
	{
	DocumentPosition	dp;

	if  ( ! docFirstPosition( &dp, rowBi ) )
	    {
	    rowIsFrame= docLayoutGetChildFrame( &frameRow, &fpRow, bd, dp.dpBi );
	    }
	}

    /*  a  */
    childBi= cellBi->biChildren[plj->pljPos.pspChild];
    wasFrame= docLayoutGetChildFrame( &frameBegin, &fpBegin, bd, childBi );
    if  ( wasFrame && ( ! rowIsFrame || frameRow != frameBegin ) )
	{
	BlockFrame			bfTextFrame;
	ParagraphLayoutPosition		plpTextFrame;
	int				frameHeight= fpBegin.fpHighTwips;

	if  ( frameHeight < 0 )
	    { frameHeight= -frameHeight;	}

	plpTextFrame= *plp;
	plp= &plpTextFrame;

	docLayoutInitBlockFrame( &bfTextFrame );
	docLayoutSetTextFrame( &bfTextFrame, &(plp->plpPos),
						bf, &fpBegin, frameHeight );

	plp->plpPos.lpPageYTwips= bfTextFrame.bfContentRect.drY0;
	bf= &bfTextFrame;
	}

    while( plp->pspChild < plj->pljChildUpto )
	{
	int			stopCode= FORMATstopREADY;
	int			ret;

	FrameProperties		fpHere;
	int			frameHere;
	int			isFrame;

	childBi= cellBi->biChildren[plp->pspChild];
	isFrame= docLayoutGetChildFrame( &frameHere, &fpHere, bd, childBi );

	/*  a  */
	if  ( isFrame != wasFrame				||
	      ( wasFrame && ( frameHere != frameBegin )	)	)
	    {
	    if  ( wasFrame )
		{
		docLayoutFinishFrame( &fpBegin, bf, bfFlow, lj,
			&(plj->pljPos), cellBi, childFrom, plp->pspChild );

		plj->pljPos= *plp;
		}

	    *pStopCode= FORMATstopNEXT_FRAME;
	    break;
	    }

	ret= docLayoutChildInStrip( &stopCode, plp, bf, lj,
					    cellBi->biCellTopInset, childBi );
	if  ( ret < 0 )
	    { LDEB(ret); return -1;		}

	if  ( ret == FORMATstopPAGE_FULL )
	    { *pStopCode= stopCode; break;	}
	if  ( ret == FORMATstopPARTIAL )
	    { continue;				}
	if  ( ret == FORMATstopREADY )
	    { continue;				}

	LDEB(ret);
	}

    if  ( docLayoutStripDone( plp, plj ) )
	{ *pStopCode= FORMATstopREADY;	}

    return 0;
    }

/************************************************************************/
/*									*/
/*  Find a start point for the formatter to layout lines. A certain	*/
/*  work ahead is necessary to decide whether work on line layout can	*/
/*  be used or not. docCommitStripLayout() is used to decide what	*/
/*  work is final and what possibly is to be redone.			*/
/*									*/
/*  docCommitStripLayout() is called when a column is full. In	*/
/*  this case, everything that is not final yet needs to be moved to	*/
/*  the next page.							*/
/*									*/
/*  -   Insure that the constraints comming from Widow/Orphan control	*/
/*	are been obeyed.						*/
/*  -   Insure that paragraphs that should not be divided over pages	*/
/*	are not split.							*/
/*  -   Insure that paragraphs that should be with the next one as a	*/
/*	header really are with it. (Experimenting with Word shows that	*/
/*	paragraphs that should be with the next one should be		*/
/*	completely with it.)						*/
/*									*/
/*  When any of the above is violated, find a solution that respects	*/
/*  the requirements by pushing offending paragraphs or lines down.	*/
/*									*/
/*  docCommitStripLayout() shifts 'plp0' to the last position that is	*/
/*  not below 'plp1' that is not influenced by the layout of position	*/
/*  on 'plp1' or below. Everything before 'plp1' is formatted and	*/
/*  plp1->plpPos is the position where formatting would continue if	*/
/*  docCommitStripLayout() moved 'plp0' onto 'plp1'. (As it often does)	*/
/*									*/
/*  REGULAR APPROACH							*/
/*  ================							*/
/*  a)  Avoid superfluous work and the handling of this exception	*/
/*	below: Commit an explit page break.				*/
/*  b)	If the current position is a legal position for a page break..	*/
/*	break here.							*/
/*									*/
/*  'Keep With Next' and 'Widow/Orphan Contol'				*/
/*  ==========================================				*/
/*  1)  Find the place to begin. Anything on a previous page is final:	*/
/*	It will not be shifted to the next page. (Paragraph)		*/
/*  2)	Idem: Line. The line found here might very well be irrelevant	*/
/*	because a different paragraph to commit is found below.		*/
/*  3)	Commit all paragraphs before the current one that do not have	*/
/*	the 'Keep with Next' flag.					*/
/*	flag.								*/
/*  4)	If the paragraph to be committed is before the current one	*/
/*	commit the position we have found. This can be:			*/
/*	-   The first line on the current page (from 2)			*/
/*	    [ This case is however excluded by the exception code (0)].	*/
/*	-   The head of a series of KeepWithNext paragraphs. (from 3)	*/
/*  5)	When this paragraph is to be kept together, restart from its	*/
/*	beginning.							*/
/*  6)	When Widow/Orphan control is active.. Do not commit the first	*/
/*	line before the whole paragraph is committed.			*/
/*  7)	The special cases of the last line of a paragraph with widow	*/
/*	orphan control and a paragraph with three lines are covered	*/
/*	in docLayoutStripChildren() at the end of a paragraph.		*/
/*	So no exceptions have to be made here. Also note the		*/
/*	preparations in docAdjustParaLayout().				*/
/*									*/
/*  8)	And finally when no exception applies, start from where we are	*/
/*	now.								*/
/*									*/
/************************************************************************/

static void docCommitStripLayout_x(
				ParagraphLayoutPosition *	plp0,
				const ParagraphLayoutPosition *	plp1,
				int				advanceAnyway,
				int				page,
				int				column,
				const BufferItem *		cellBi )
    {
    const BufferItem *		paraBi0;
    const BufferItem *		paraBi1;
    const TextLine *		tl;

    int				para;

    int				line0;

    paraBi0= cellBi->biChildren[plp0->pspChild];
    paraBi1= (const BufferItem *)0;

    /*  a  */
    if  ( plp1->pspChild < cellBi->biChildCount )
	{
	int		line1;

	paraBi1= cellBi->biChildren[plp1->pspChild];

	line1= plp1->pspLine;

	if  ( line1- 1 >= 0				&&
	      line1- 1 < paraBi1->biParaLineCount	)
	    {
	    const TextLine *	tl= paraBi1->biParaLines+ line1- 1;

	    if  ( tl->tlFlags & TLflagPAGEBREAK )
		{
		*plp0= *plp1;
		return;
		}
	    }
	}

    /*  b  */
    if  ( paraBi1			&&
	  paraBi1->biParaKeepOnPage	&&
	  ! paraBi1->biParaKeepWithNext	&&
	  plp1->pspLine > 0		)
	{
	int		line= 1;

	if  ( paraBi1->biParaWidowControl )
	    { line++;	}

	if  ( plp1->pspLine >= line )
	    {
	    *plp0= *plp1;
	    return;
	    }
	}

    /*  1  */
    while( plp0->pspChild < plp1->pspChild )
	{
	if  ( paraBi0->biBelowPosition.lpPage >  page )
	    { break;	}
	if  ( paraBi0->biBelowPosition.lpPage == page		&&
	      paraBi0->biBelowPosition.lpColumn >= column	)
	    { break;	}

	docStripLayoutNextChild( plp0 );
	paraBi0= cellBi->biChildren[plp0->pspChild];
	}

    /*  2  */
    if  ( plp0->pspChild < plp1->pspChild )
	{ line0= paraBi0->biParaLineCount;	}
    else{ line0= plp1->pspLine;			}

    tl= paraBi0->biParaLines+ plp0->pspLine;
    while( plp0->pspLine < line0 )
	{
	if  ( tl->tlTopPosition.lpPage >  page )
	    { break;	}
	if  ( tl->tlTopPosition.lpPage == page	&&
	      tl->tlTopPosition.lpColumn >= column	)
	    { break;	}

	plp0->pspLine++; tl++;
	}

    /*  3  */
    for ( para= plp0->pspChild; para < plp1->pspChild; para++ )
	{
	const BufferItem *	paraBi= cellBi->biChildren[para];

	if  ( advanceAnyway || ! paraBi->biParaKeepWithNext )
	    {
	    docStripLayoutStartChild( plp0, para+ 1 );
	    paraBi0= cellBi->biChildren[plp0->pspChild];
	    }
	}

    /*  4  */
    if  ( plp0->pspChild < plp1->pspChild )
	{ return;	}

    if  ( plp0->pspChild != plp1->pspChild )
	{ LLLLDEB(page,column,plp0->pspChild,plp1->pspChild );	}

    if  ( plp0->pspChild >= cellBi->biChildCount )
	{ return;	}

    /*  5  */
    if  ( ! advanceAnyway			&&
	  ( paraBi0->biParaKeepOnPage	||
	    paraBi0->biParaKeepWithNext	)	)
	{
	docStripLayoutStartChild( plp0, plp0->pspChild );
	return;
	}

    /*  6  */
    if  ( ! advanceAnyway		&&
	  paraBi1			&&
	  paraBi1->biParaWidowControl	&&
	  plp1->pspLine == 1		&&
	  paraBi1->biParaLineCount >= 1	)
	{
	tl= paraBi1->biParaLines+ 0;

	if  ( tl->tlFirstParticule+ tl->tlParticuleCount <
					paraBi1->biParaParticuleCount	)
	    {
	    docStripLayoutStartChild( plp0, plp0->pspChild );
	    return;
	    }
	}

    /*  7  */

    /*  8  */
    *plp0= *plp1;

    return;
    }

/************************************************************************/
/*									*/
/*  Find out whether the fact that layout has proceeded to plj->pljPos	*/
/*  makes any layout final. If so.. Move pljPos0 past what has become	*/
/*  final.								*/
/*									*/
/************************************************************************/

# define docSetLayoutProgress( plpt, plpf ) \
    { \
    (plpt)->pspChild= (plpf)->pspChild; \
    (plpt)->pspChildAdvanced= (plpf)->pspChildAdvanced; \
    (plpt)->pspPart= (plpf)->pspPart; \
    (plpt)->pspLine= (plpf)->pspLine; \
    }

static int docCompareLayoutProgress(
			const ParagraphLayoutPosition *	plp0,
			const ParagraphLayoutPosition *	plp1 )
    {
    if  ( plp1->pspChild > plp0->pspChild )
	{ return  1;	}
    if  ( plp1->pspChild < plp0->pspChild )
	{ return -1;	}

    if  ( plp1->pspPart > plp0->pspPart )
	{ return  1;	}
    if  ( plp1->pspPart < plp0->pspPart )
	{ return -1;	}

    return 0;
    }

void docCommitStripLayout(
				int *				pAdvanced,
				int				advanceAnyway,
				ParagraphLayoutJob *		plj,
				int				page,
				int				column,
				const BufferItem *		cellBi )
    {
    int				advanced= 0;
    ParagraphLayoutPosition	plp0;

    plp0= plj->pljPos0;

    docCommitStripLayout_x( &plp0, &(plj->pljPos),
				    advanceAnyway, page, column, cellBi );

    advanced= ( docCompareLayoutProgress( &plp0, &(plj->pljPos0) ) < 0 );
    if  ( advanced )
	{
	docSetLayoutProgress( &(plj->pljPos0), &plp0 );
	docSetLayoutProgress( &(plj->pljPos), &plp0 );
	}
    else{
#	if 0
	int cmp= docCompareLayoutProgress( &(plj->pljPos0), &(plj->pljPos) );
	if  ( cmp >= 0 )
	    { LDEB(cmp);	}
#	endif
	/* Prevent loops: Be sure to advance */
	docSetLayoutProgress( &(plj->pljPos0), &(plj->pljPos) );
	}

    *pAdvanced= advanced;
    }

/************************************************************************/
/*									*/
/*  Find out where to start formatting a strip.				*/
/*  After some thought: That is the position that would be committed	*/
/*  if the strip were formatted upto the stat position.			*/
/*									*/
/************************************************************************/

void docFindStripLayoutOrigin(	ParagraphLayoutJob *		plj,
				int				page,
				int				column,
				const BufferItem *		cellBi )
    {
    int				advanceAnyway= 0;

    docCommitStripLayout_x( &(plj->pljPos0), &(plj->pljPos),
					advanceAnyway, page, column, cellBi );

    return;
    }

/************************************************************************/
/*									*/
/*  Format the lines in a series of paragraphs. On the way keep an	*/
/*  administration on where to restart formatting at a page break.	*/
/*									*/
/*  1)  Place as much as fits on the first page.			*/
/*  2)  While unformatted paragraphs remain: place some more on		*/
/*	subsequent pages.						*/
/*  3)  First place the footnotes on the page.				*/
/*  4)  Then find out where to restart the layout job on the next page	*/
/*	Widow/Orphan control etc can cause some of the text that we	*/
/*	already formatted to go to the next page.			*/
/*  5)  Skip to the next page.						*/
/*  6)  Determine available space on the next page.			*/
/*  7)  Place as much as fits on the next page.				*/
/*									*/
/************************************************************************/

int docLayoutParagraphs(	BufferItem *			cellBi,
				BlockFrame *			bf,
				const LayoutJob *		lj,
				ParagraphLayoutJob *		plj )
    {
    LayoutPosition		lpBefore;

    int				prevAdvanced= 1;
    int				advanceAnyway= 0;
    int				stopCode= FORMATstopREADY;
    int				prevStopCode= FORMATstopREADY;

    lpBefore= plj->pljPos.plpPos;

    /*  1  */
    if  ( docLayoutStripChildren( &stopCode, plj, bf, lj, cellBi ) )
	{ LDEB(1); return -1;	}

    /*  2  */
    while( stopCode != FORMATstopREADY		&&
	   stopCode != FORMATstopFRAME_FULL	)
	{
	int				advanced;
	const int			belowText= 0;
	LayoutPosition			lpBelowNotes;

	/*  3  */
	if  ( BF_HAS_FOOTNOTES( bf )					&&
	      ( cellBi->biInExternalItem == DOCinBODY		||
	        cellBi->biInExternalItem == DOCinENDNOTE	)	&&
	      ! bf->bfFootnotesPlaced					&&
	      docLayoutFootnotesForColumn( &lpBelowNotes,
			    &(plj->pljPos.plpPos), bf, belowText, lj )	)
	    { LDEB(1); return -1;	}

	/*  4  */
	docCommitStripLayout( &advanced, advanceAnyway, plj,
					lpBefore.lpPage, lpBefore.lpColumn,
					cellBi );

	/*  5  */
	if  ( stopCode == FORMATstopPAGE_FULL )
	    {
	    docLayoutToNextColumn( &(plj->pljPos.plpPos), bf, cellBi, lj );
	    }

	if  ( ! advanced )
	    {
	    if  ( ! prevAdvanced )
		{
		if  ( advanceAnyway )
		    {
		    LLLDEB(prevAdvanced,advanced,advanceAnyway);
		    LLDEB(prevStopCode,stopCode);
		    RECTDEB(&(plj->pljPos.plpParagraphFrame.pfParaContentRect));

		    docStripLayoutNextChild( &(plj->pljPos) );
		    docSetLayoutProgress( &(plj->pljPos0), &(plj->pljPos) );
		    advanced= 1;

		    if  ( docLayoutStripDone( &(plj->pljPos), plj ) )
			{ break;	}
		    }
		else{ advanceAnyway= 1;	}
		}
	    }

	prevAdvanced= advanced;

	/*  6  */
	docCellStripFrame( cellBi, bf, plj );

	/*  7  */
	lpBefore= plj->pljPos.plpPos;

	prevStopCode= stopCode;
	if  ( docLayoutStripChildren( &stopCode, plj, bf, lj, cellBi ) )
	    { LDEB(1); return -1;	}
	}

    if  ( cellBi->biChildCount > 0 )
	{ cellBi->biTopPosition= cellBi->biChildren[0]->biTopPosition; }

    return 0;
    }

/************************************************************************/
/*									*/
/*  Calculate the formatting frame for a paragraph layout.		*/
/*									*/
/************************************************************************/

void docCellStripFrame(		BufferItem *			cellBi,
				const BlockFrame *		bf,
				ParagraphLayoutJob *		plj )
					
    {
    BufferItem *		childBi;
    ParagraphFrame *		pf= &(plj->pljPos.plpParagraphFrame);

    childBi= cellBi->biChildren[plj->pljPos.pspChild];

    if  ( childBi->biLevel == DOClevPARA )
	{ docParagraphFrameTwips( pf, bf, childBi );	}
    else{ docCellFrameTwips( pf, bf, cellBi );		}

    return;
    }

