/************************************************************************/
/*									*/
/*  Paragraph Layout related administration.				*/
/*									*/
/************************************************************************/

#   include	"tedConfig.h"

#   include	<stddef.h>
#   include	<stdio.h>
#   include	<stdlib.h>
#   include	<limits.h>

#   include	<psFontMetrics.h>
#   include	<appMatchFont.h>
#   include	"docPageGrid.h"
#   include	"docLayout.h"
#   include	<appImage.h>

#   include	<appDebugon.h>

/************************************************************************/
/*									*/
/*  Calculate the height of a series of lines in a paragraph.		*/
/*									*/
/*  1)  The first paragraph in a cell must accomodate space for the	*/
/*	top border of the cells of the row and the cell padding.	*/
/*  2)  Allocate space for the fattest top border in the row.		*/
/*  3)  Allocate space for the row padding.				*/
/*  4)  Allocate space for the cell padding.				*/
/*  5)  Allocate space for the paragraph top border.			*/
/*									*/
/************************************************************************/

void docLayoutCalculateParaTopInset(
				const BufferDocument *	bd,
				BufferItem *		paraBi )
    {
    int				topInset= 0;
    int				nrAbove= -1;
    const BufferItem *		cellBi= paraBi->biParent;
    const BufferItem *		prevBi= (const BufferItem *)0;

    const NumberedPropertiesList *	bpl= &(bd->bdBorderPropertyList);
    const NumberedPropertiesList *	isl= &(bd->bdItemShadingList);

    int				cellMargin= 0;
    int				fillBefore= 0;

    if  ( paraBi->biNumberInParent > 0 )
	{ prevBi= cellBi->biChildren[paraBi->biNumberInParent- 1]; }

    topInset= paraBi->biParaSpaceBeforeTwips;

    /*  1  */
    if  ( paraBi->biParaTableNesting > 0	&&
	  paraBi->biNumberInParent == 0		)
	{
	const BufferItem *	rowBi= cellBi->biParent;

	const RowProperties *	rp= &(rowBi->biRowProperties);
	const CellProperties *	cp= rp->rpCells+ cellBi->biNumberInParent;

	if  ( ! docIsRowItem( rowBi ) )
	    { LDEB(1); cp= (const CellProperties *)0;	}

	/*  2  */
	cellMargin += rowBi->biRowTopInset;

	/*  3  */
	switch( rp->rpTopCellPaddingUnit )
	    {
	    case TRautoNONE:
		break;

	    case TRautoTWIPS:
		cellMargin= rp->rpTopCellPadding;
		break;

	    default:
		LDEB(rp->rpTopCellPaddingUnit);
		break;
	    }

	/*  4  */
	if  ( cp ) switch( cp->cpTopPaddingUnit )
	    {
	    case TRautoNONE:
		break;

	    case TRautoTWIPS:
		cellMargin= cp->cpTopPadding;
		break;

	    default:
		LDEB(cp->cpTopPaddingUnit);
		break;
	    }
	}

    if  ( paraBi->biParaTableNesting == 0	&&
	  paraBi->biNumberInParent == 0		&&
	  cellBi->biNumberInParent == 0		)
	{
	const BufferItem *	rowBi= cellBi->biParent;

	topInset += rowBi->biRowTopInset;
	}

    /*  5  */
    if  ( docBorderNumberIsBorder( bpl, paraBi->biParaTopBorderNumber ) )
	{
	if  ( ! prevBi							||
	      prevBi->biParaFrameNumber != paraBi->biParaFrameNumber	||
	      prevBi->biParaBottomBorderNumber !=
					paraBi->biParaTopBorderNumber	)
	    {
	    BorderProperties	bpTop;

	    docGetBorderPropertiesByNumber( &bpTop, bpl,
					    paraBi->biParaTopBorderNumber );

	    docAddBorderToInset( &topInset, &bpTop );
	    nrAbove= paraBi->biParaTopBorderNumber;
	    }

	if  ( prevBi							&&
	      prevBi->biParaFrameNumber == paraBi->biParaFrameNumber	&&
	      prevBi->biParaBottomBorderNumber ==
				    paraBi->biParaBottomBorderNumber	)
	    { fillBefore= 1;	}
	}

    if  ( prevBi							&&
	  docShadingNumberIsShading( isl, paraBi->biParaShadingNumber )	&&
	  docShadingNumberIsShading( isl, prevBi->biParaShadingNumber )	)
	{ fillBefore= 1;	}

    paraBi->biParaTopInset= cellMargin+ topInset;
    paraBi->biParaBorderNrAbove= nrAbove;

    if  ( fillBefore )
	{ paraBi->biParaFlags |=  PARAflagFILL_BEFORE;	}
    else{ paraBi->biParaFlags &= ~PARAflagFILL_BEFORE;	}

    return;
    }

void docLayoutCalculateParaBottomInset(
				    const BufferDocument *	bd,
				    BufferItem *		paraBi )
    {
    int				bottomInset= 0;
    int				nrBelow= -1;
    const BufferItem *		cellBi= paraBi->biParent;
    const BufferItem *		rowBi= cellBi->biParent;
    const BufferItem *		nextBi= (const BufferItem *)0;

    const NumberedPropertiesList *	bpl= &(bd->bdBorderPropertyList);
    const NumberedPropertiesList *	isl= &(bd->bdItemShadingList);

    int				cellMargin= 0;
    int				fillAfter= 0;

    if  ( paraBi->biNumberInParent < paraBi->biParent->biChildCount- 1 )
	{ nextBi= cellBi->biChildren[paraBi->biNumberInParent+ 1]; }

    if  ( paraBi->biParaTableNesting > 0			&&
	  cellBi->biNumberInParent <
			    rowBi->biRowProperties.rpCellCount	&&
	  paraBi->biNumberInParent == cellBi->biChildCount- 1	)
	{

	const RowProperties *	rp= &(rowBi->biRowProperties);
	const CellProperties *	cp= rp->rpCells+ cellBi->biNumberInParent;

	switch( rp->rpBottomCellPaddingUnit )
	    {
	    case TRautoNONE:
		break;

	    case TRautoTWIPS:
		cellMargin= rp->rpBottomCellPadding;
		break;

	    default:
		LDEB(rp->rpBottomCellPaddingUnit);
		break;
	    }

	switch( cp->cpBottomPaddingUnit )
	    {
	    case TRautoNONE:
		break;

	    case TRautoTWIPS:
		cellMargin= cp->cpBottomPadding;
		break;

	    default:
		LDEB(cp->cpTopPaddingUnit);
		break;
	    }
	}

    if  ( docBorderNumberIsBorder( bpl, paraBi->biParaBottomBorderNumber ) )
	{
	if  ( ! nextBi							||
	      nextBi->biParaFrameNumber != paraBi->biParaFrameNumber	||
	      nextBi->biParaTopBorderNumber !=
				    paraBi->biParaBottomBorderNumber	)
	    {
	    BorderProperties	bpBottom;

	    docGetBorderPropertiesByNumber( &bpBottom, bpl,
					    paraBi->biParaBottomBorderNumber );

	    docAddBorderToInset( &bottomInset, &bpBottom );
	    nrBelow= paraBi->biParaBottomBorderNumber;
	    }

	if  ( nextBi							&&
	      nextBi->biParaFrameNumber == paraBi->biParaFrameNumber	&&
	      nextBi->biParaBottomBorderNumber ==
				    paraBi->biParaBottomBorderNumber	)
	    { fillAfter= 1;	}
	}

    if  ( nextBi							&&
	  docShadingNumberIsShading( isl, paraBi->biParaShadingNumber )	&&
	  docShadingNumberIsShading( isl, nextBi->biParaShadingNumber )	)
	{ fillAfter= 1;	}

    bottomInset += paraBi->biParaSpaceAfterTwips;

    paraBi->biParaBottomInset= bottomInset+ cellMargin;
    paraBi->biParaBorderNrBelow= nrBelow;

    if  ( fillAfter )
	{ paraBi->biParaFlags |=  PARAflagFILL_AFTER;	}
    else{ paraBi->biParaFlags &= ~PARAflagFILL_AFTER;	}

    return;
    }

/************************************************************************/
/*									*/
/*  Determine the 'majority' font of a paragraph, and get the font	*/
/*  extents for that font.						*/
/*									*/
/*  1)  Note that subscript/superscript is NOT taken into account.	*/
/*									*/
/************************************************************************/

int docLayoutParagraphLineExtents(
				int *				pFontSize,
				BufferDocument *		bd,
				const PostScriptFontList *	psfl,
				BufferItem *			paraBi )
    {
    const TextParticule *	tp= paraBi->biParaParticules;

    int				sizeHalfPoints;
    int				y0= 0;
    int				y1= 0;
    int				i;

    static int *		counts;
    int *			fresh;

    int				found;
    int				foundCount;

    const NumberedPropertiesList *	tal= &(bd->bdTextAttributeList);
    int					count= tal->nplPagedList.plItemCount;

    fresh= (int *)realloc( counts, count* sizeof(int) );
    if  ( ! fresh )
	{ LXDEB(count,fresh); return -1;	}
    counts= fresh;

    for ( i= 0; i < count; i++ )
	{ counts[i]= 0;	}

    for ( i= 0; i < paraBi->biParaParticuleCount; tp++, i++ )
	{
	if  ( tp->tpKind != DOCkindSPAN		&&
	      tp->tpKind != DOCkindTAB		&&
	      tp->tpKind != DOCkindOBJECT	)
	    { continue;	}

	if  ( tp->tpTextAttrNr < 0			||
	      tp->tpTextAttrNr >= count	)
	    { LLDEB(tp->tpTextAttrNr,count); continue;	}

	counts[tp->tpTextAttrNr] += tp->tpStrlen+ 1;
	}

    found= -1;
    foundCount= 0;
    for ( i= 0; i < count; i++ )
	{
	if  ( counts[i] > foundCount )
	    { found= i; foundCount= counts[i];	}
	}

    if  ( found >= 0 )
	{
	DocumentProperties *	dp= &(bd->bdProperties);
	DocumentFontList *	dfl= &(dp->dpFontList);

	TextAttribute			ta;
	const AfmFontInfo *		afi;

	const int			vswap= 1;
	DocumentRectangle		drFontBBox;
	DocumentRectangle		drFontAscDesc;
	int				fontHigh;
	int				sizeTwips;

	utilGetTextAttributeByNumber( &ta, &(bd->bdTextAttributeList), found );

	afi= appGetFontInfoForAttribute( &ta, dfl, psfl );
	if  ( ! afi )
	    { XDEB(afi); return -1;	}

	sizeHalfPoints= ta.taFontSizeHalfPoints;
	sizeTwips= 10* sizeHalfPoints;

	psFontBBox( &drFontBBox, &drFontAscDesc, sizeTwips, vswap, afi );

	y0= drFontAscDesc.drY0;
	y1= drFontAscDesc.drY1;

	/*LINEDISTANCE: scale the position of the baseline based on the bbox */
	fontHigh= drFontBBox.drY1- drFontBBox.drY0;
	y0= ( drFontBBox.drY0* sizeTwips )/ fontHigh;
	y1= ( drFontBBox.drY1* sizeTwips )/ fontHigh;
	}
    else{
	/* LDEB(found); */
	sizeHalfPoints= 24;
	y0= -190;
	y1=   50;
	}

    paraBi->biParaMajorityFontSize= sizeHalfPoints;
    paraBi->biParaFontAscY0= y0;
    paraBi->biParaFontDescY1= y1;

    *pFontSize= 10* sizeHalfPoints;
    return 0;
    }

/************************************************************************/
/*									*/
/*  Scale an inserted object to fit where we want to place it.		*/
/*									*/
/*  1)  If the object fits on the page at its current scale, nothing	*/
/*	needs to be done. Apparently this was the case before as well,	*/
/*	as otherwise the ioScale[XY]Used would not have been equal to	*/
/*	ioScale[XY]Set.							*/
/*  2)  Otherwise scale the object to fit on the page. X and Y scale	*/
/*	are the same, irrespective of the original scaling.		*/
/*									*/
/************************************************************************/

void docLayoutScaleObjectToFitParagraphFrame(
					int *			pChanged,
					InsertedObject *	io,
					int			pageHigh,
					const ParagraphFrame *	pf )
    {
    const DocumentRectangle *	drParaContent= &(pf->pfParaContentRect);
    int			textWideTwips= drParaContent->drX1- drParaContent->drX0;

    int		objectWideTwips= ( io->ioScaleXSet* io->ioTwipsWide )/ 100.0;
    int		objectHighTwips= ( io->ioScaleYSet* io->ioTwipsHigh )/ 100.0;

    double	scaleX;
    double	scaleY;
    int		scaleMax;

    int		prevXScaleUsed= io->ioScaleXUsed;
    int		prevYScaleUsed= io->ioScaleYUsed;

    PictureProperties *	pip= &(io->ioPictureProperties);

    /*  1  */
    if  ( io->ioScaleXUsed == io->ioScaleXSet	&&
	  io->ioScaleYUsed == io->ioScaleYSet	&&
	  objectWideTwips <= textWideTwips	&&
	  objectHighTwips <= pageHigh		)
	{ *pChanged= 0; return;	}

    /*  2  */
    scaleX= (double)textWideTwips/ (double)io->ioTwipsWide;
    scaleY= (double)pageHigh/ (double)io->ioTwipsHigh;

    if  ( scaleY > scaleX )
	{ scaleY=  scaleX;	}
    scaleMax= (int)( 99* scaleY );

    io->ioScaleXUsed= io->ioScaleXSet;
    io->ioScaleYUsed= io->ioScaleYSet;

    pip->pipScaleXUsed= io->ioScaleXUsed;
    pip->pipScaleYUsed= io->ioScaleYUsed;

    if  ( scaleMax < io->ioScaleXUsed	||
	  scaleMax < io->ioScaleYUsed	)
	{
	if  ( io->ioScaleXUsed > scaleMax )
	    { io->ioScaleXUsed= scaleMax;	}
	if  ( io->ioScaleYUsed > scaleMax )
	    { io->ioScaleYUsed= scaleMax;	}

	if  ( io->ioScaleXUsed < 1 )
	    { io->ioScaleXUsed=  1;	}
	if  ( io->ioScaleYUsed < 1 )
	    { io->ioScaleYUsed=  1;	}

	pip->pipScaleXUsed= io->ioScaleXUsed;
	pip->pipScaleYUsed= io->ioScaleYUsed;
	}

    if  ( io->ioScaleXUsed != prevXScaleUsed	||
	  io->ioScaleYUsed != prevYScaleUsed	)
	{ *pChanged= 1;	}
    else{ *pChanged= 0;	}

    return;
    }

/************************************************************************/
/*									*/
/*  Invalidate the layout of all paragraphs in the modified range.	*/
/*									*/
/************************************************************************/

void docInvalidateParagraphLayout(	BufferItem *	paraBi )
    {
    paraBi->biParaLineCount= 0;

    paraBi->biParaFontAscY0= 0;
    paraBi->biParaFontDescY1= 0;
    paraBi->biParaMajorityFontSize= 0;
    }

/************************************************************************/
/*									*/
/*  Calculate the top inset after a table.				*/
/*									*/
/*  This routine is also used to calculate the inset that is used to	*/
/*  accomodate space for the bottom border of the previous row. This	*/
/*  inset is not used in real rows, but only in the immediate successor	*/
/*  of a real row.							*/
/*									*/
/*  1)  Reserve space for the bottom border of all cells.		*/
/*									*/
/************************************************************************/

void docLayoutCalculateAfterRowTopInset(	BufferItem *		belowBi,
						const BufferDocument *	bd )
    {
    int				col;
    const BufferItem *		rowBi;
    const CellProperties *	cp;

    if  ( belowBi->biNumberInParent == 0 )
	{ return;	}

    rowBi= belowBi->biParent->biChildren[belowBi->biNumberInParent- 1];
    if  ( ! docIsRowItem( rowBi ) )
	{ return;	}

    belowBi->biRowTopInset= 0;

    /*  1  */
    cp= rowBi->biRowCells;
    for ( col= 0; col < rowBi->biChildCount; cp++, col++ )
	{
	const int		atRowBottom= 1;
	int			useBelow= 0;
	BorderProperties	bpBottom;

	if  ( CELL_MERGED( cp ) )
	    { continue;	}

	docGetCellBottomBorder( &bpBottom, &useBelow, bd, rowBi,
							    col, atRowBottom );

	docStretchInsetForBorder( &(belowBi->biRowTopInset), &bpBottom );
	}

    return;
    }

/************************************************************************/
/*									*/
/*  Determine paragraph border and shading.				*/
/*									*/
/*  Experimentation with MS-Word revealed that MS-Word includes the	*/
/*  space before/after in the borders and shading between two		*/
/*  paragraphs if either:						*/
/*  a)  Both have the same border.					*/
/*  b)  Both have a shading.						*/
/*									*/
/************************************************************************/

void docGetParaOrnaments(
			BlockOrnaments *		ornaments,
			DocumentRectangle *		drOutside,
			DocumentRectangle *		drInside,
			const DocumentRectangle *	drPara,
			const BufferDocument *		bd,
			const BufferItem *		paraBi,
			int				atTop,
			int				atBottom )
    {
    const NumberedPropertiesList *	bpl= &(bd->bdBorderPropertyList);

    int				thick;
    int				space;

    *drOutside= *drPara;
    *drInside= *drPara;

    if  ( paraBi->biParaShadingNumber != 0 )
	{
	docGetItemShadingByNumber( &(ornaments->boShading),
						&(bd->bdItemShadingList),
						paraBi->biParaShadingNumber );

	PROPmaskADD( &(ornaments->boPropMask), ORNdrawSHADE );
	}

    docGetBorderPropertiesByNumber( &(ornaments->boLeftBorder), bpl,
					    paraBi->biParaLeftBorderNumber );
    docGetBorderPropertiesByNumber( &(ornaments->boRightBorder), bpl,
					    paraBi->biParaRightBorderNumber );

    if  ( atTop && paraBi->biParaBorderNrAbove >= 0 )
	{
	docGetBorderPropertiesByNumber( &(ornaments->boTopBorder), bpl,
					    paraBi->biParaBorderNrAbove );

	PROPmaskADD( &(ornaments->boPropMask), ORNdrawTOP_BORDER );

	thick= docBorderThick( &space, &(ornaments->boTopBorder) );
	/* No! The paragraph above covers the lower one like the 
	 * tiles on a roof.
	drOutside->drY0= drPara->drY0- space- thick;
	drInside->drY0= drPara->drY0- space;
	*/
	drOutside->drY0= drPara->drY0+ space;
	drInside->drY0= drPara->drY0+ space+ thick;
	}

    if  ( atBottom && paraBi->biParaBorderNrBelow >= 0 )
	{
	docGetBorderPropertiesByNumber( &(ornaments->boBottomBorder), bpl,
					    paraBi->biParaBorderNrBelow );

	PROPmaskADD( &(ornaments->boPropMask), ORNdrawBOTTOM_BORDER );

	thick= docBorderThick( &space, &(ornaments->boBottomBorder) );
	drInside->drY1= drPara->drY1+ space;
	drOutside->drY1= drPara->drY1+ space+ thick;
	}

    if  ( paraBi->biParaLeftBorderNumber != 0 )
	{
	docGetBorderPropertiesByNumber( &(ornaments->boLeftBorder), bpl,
					paraBi->biParaLeftBorderNumber );

	PROPmaskADD( &(ornaments->boPropMask), ORNdrawLEFT_BORDER );

	thick= docBorderThick( &space, &(ornaments->boLeftBorder) );
	drInside->drX0= drPara->drX0- space;
	drOutside->drX0= drPara->drX0- space- thick;
	}

    if  ( paraBi->biParaRightBorderNumber != 0 )
	{
	docGetBorderPropertiesByNumber( &(ornaments->boRightBorder), bpl,
					paraBi->biParaRightBorderNumber );

	PROPmaskADD( &(ornaments->boPropMask), ORNdrawRIGHT_BORDER );

	thick= docBorderThick( &space, &(ornaments->boRightBorder) );
	drInside->drX1= drPara->drX1+ space;
	drOutside->drX1= drPara->drX1+ space+ thick;
	}

    return;
    }

/************************************************************************/
/*									*/
/*  Initialisation for paragraph layout.				*/
/*									*/
/*  1)  Scale objects if necessary. If a scale was chenged, cause the	*/
/*	lines in the paragraph to be reformatted. There is no need to	*/
/*	completely invalidate the layout: Font related information is	*/
/*	unaffected.							*/
/*  2)  Calculate global line extent properties if necessary.		*/
/*									*/
/************************************************************************/

static int docStartParagraphLayout(
				const ParagraphFrame *		pf,
				const BlockFrame *		bf,
				BufferItem *			paraBi,
				const LayoutContext *		lc )
    {
    int				part;
    const TextParticule *	tp;
    int				fontSize= 0;
    int				pageHigh;

    pageHigh= bf->bfPageGeometry.dgPageHighTwips- 
			    bf->bfPageGeometry.dgTopMarginTwips-
			    bf->bfPageGeometry.dgBottomMarginTwips;

    /*  1  */
    tp= paraBi->biParaParticules;
    for ( part= 0; part < paraBi->biParaParticuleCount; tp++, part++ )
	{
	InsertedObject *	io;
	PictureProperties *	pip;
	int			fixed= 0;
	int			changed= 0;

	if  ( tp->tpKind != DOCkindOBJECT )
	    { continue;	}

	io= docGetObject( lc->lcDocument, tp->tpObjectNumber );
	if  ( ! io )
	    { LPDEB(tp->tpObjectNumber,io); continue;	}
	pip= &(io->ioPictureProperties);

	if  ( io->ioKind == DOCokMACPICT )
	    {
	    if  ( pip->pipTwipsWide == 0 )
		{ pip->pipTwipsWide= 20* pip->pip_xWinExt; fixed= 1;	}
	    if  ( pip->pipTwipsHigh == 0 )
		{ pip->pipTwipsHigh= 20* pip->pip_yWinExt; fixed= 1;	}
	    }

	if  ( io->ioKind == DOCokDRAWING_SHAPE )
	    {
	    const DrawingShape *	ds= io->ioDrawingShape;

	    if  ( ! ds )
		{ LXDEB(io->ioKind,io->ioDrawingShape);	}
	    else{
		const ShapeProperties *	sp= &(ds->dsShapeProperties);

		if  ( io->ioTwipsWide == 0 )
		    {
		    io->ioTwipsWide= sp->spRect.drX1- sp->spRect.drX0;
		    fixed= 1;
		    }
		if  ( io->ioTwipsHigh == 0 )
		    {
		    io->ioTwipsHigh= sp->spRect.drY1- sp->spRect.drY0;
		    fixed= 1;
		    }
		}
	    }

	if  ( ( pip->pipTwipsWide == 0		||
	        pip->pipTwipsHigh == 0		)	&&
	      ( io->ioKind == DOCokPICTPNGBLIP	||
	        io->ioKind == DOCokPICTJPEGBLIP	)	)
	    {
	    if  ( ! io->ioPrivate			&&
		  docGetBitmapForObject( io )		)
		{ LDEB(1);	}
	    else{
		AppBitmapImage *	abi= (AppBitmapImage *)io->ioPrivate;

		bmRectangleSizeTwips( &(pip->pipTwipsWide),
					&(pip->pipTwipsHigh),
					&(abi->abiBitmap),
					abi->abiBitmap.bdPixelsWide,
					abi->abiBitmap.bdPixelsHigh );
		}
	    }

	if  ( io->ioTwipsWide == 0 )
	    { io->ioTwipsWide= pip->pipTwipsWide; fixed= 1;	}
	if  ( io->ioTwipsHigh == 0 )
	    { io->ioTwipsHigh= pip->pipTwipsHigh; fixed= 1;	}

	docLayoutScaleObjectToFitParagraphFrame( &changed, io, pageHigh, pf );

	if  ( fixed || changed )
	    {
	    if  ( lc->lcCloseObject )
		{ (*lc->lcCloseObject)( lc->lcDocument, tp );	}

	    paraBi->biParaLineCount= 0;
	    }
	}

    /*  2  */
    if  ( paraBi->biParaMajorityFontSize == 0				&&
	  docLayoutParagraphLineExtents( &fontSize, lc->lcDocument,
				lc->lcPostScriptFontList, paraBi )	)
	{ LDEB(1); return -1;	}

    docLayoutCalculateParaTopInset( lc->lcDocument, paraBi );
    docLayoutCalculateParaBottomInset( lc->lcDocument, paraBi );

    return 0;
    }

/************************************************************************/
/*									*/
/*  Initialize the formatting of a paragraph by determining its frame	*/
/*									*/
/************************************************************************/

int docLayoutStartParagraph(	int *				pToNextColumn,
				const LayoutJob *		lj,
				BufferItem *			paraBi,
				const BlockFrame *		bf,
				ParagraphLayoutPosition *	plp )
    {
    const LayoutContext *	lc= &(lj->ljContext);
    const BufferDocument *	bd= lc->lcDocument;
    const DocumentGeometry *	dg;
    const BufferItem *		sectBi= paraBi;

    int				indentChanged= 0;

    sectBi= docGetSectItem( paraBi );
    if  ( ! sectBi )
	{ XDEB(sectBi); return -1;	}

    if  ( paraBi->biParaListOverride > 0				&&
	  docAdaptParagraphToListLevel( &indentChanged, paraBi, bd )	)
	{ LDEB(1);		}

    dg= &(sectBi->biSectDocumentGeometry);

    /*  1  */
    if  ( paraBi->biInExternalItem == DOCinBODY			&&
	  paraBi->biParaTableNesting == 0			&&
	  paraBi->biParaBreakKind != DOCibkNONE			&&
	  ! plp->plpPos.lpAtTopOfColumn				)
	{ *pToNextColumn= 1; return 0; }

    docParagraphFrameTwips( &(plp->plpParagraphFrame), bf, paraBi );

    if  ( docStartParagraphLayout( &(plp->plpParagraphFrame), bf, paraBi, lc ) )
	{ LDEB(1); return -1;	}

    if  ( lj->ljStartScreenParagraph					&&
	  (*lj->ljStartScreenParagraph)( paraBi,
				    &(plp->plpParagraphFrame), lc )	)
	{ LDEB(1); return -1;	}

    *pToNextColumn= 0;
    return 0;
    }

