/************************************************************************/
/*									*/
/*  Layout of a document.						*/
/*									*/
/*  Functionality to format notes and to subtract the space occupied by	*/
/*  notes and footnotes from what is left over prom the page (or the	*/
/*  column).								*/
/*									*/
/************************************************************************/

#   include	"tedConfig.h"

#   include	<stddef.h>
#   include	<stdio.h>
#   include	<stdlib.h>

#   include	"docLayout.h"
#   include	"docPageGrid.h"

#   include	<appDebugon.h>

#   define	SHOW_NOTE_LAYOUT	0	/* 0: nothing		*/
						/* 1: layout		*/
						/* 2: reservations	*/

/************************************************************************/
/*									*/
/*  Commit the space allocated for footnotes and the footnote separator	*/
/*									*/
/************************************************************************/

static void docLayoutCommitNotesReservation(
				NotesReservation *		nrTotal,
				const NotesReservation *	nr )
    {
    if  ( ! nrTotal->nrDfFirstFootnote		&&
	  nr->nrDfFirstFootnote			)
	{
	nrTotal->nrDfFirstFootnote= nr->nrDfFirstFootnote;
	nrTotal->nrFootnoteSectBi= nr->nrFootnoteSectBi;
	}

    if  ( nrTotal->nrFtnsepHeight == 0		&&
	  nr->nrFtnsepHeight > 0		)
	{ nrTotal->nrFtnsepHeight= nr->nrFtnsepHeight;	}

    nrTotal->nrFootnoteCount += nr->nrFootnoteCount;
    nrTotal->nrFootnoteHeight += nr->nrFootnoteHeight;

    return;
    }

/************************************************************************/
/*									*/
/*  Collect the sum of the height of footnotes between two positions	*/
/*  in the same paragraph. This routine and the subsequent one are	*/
/*  used to subtract the space claimed for footnotes on the current	*/
/*  page at the beginning of a formatting action.			*/
/*									*/
/************************************************************************/

static int docLayoutCollectOneFootnote( NotesReservation *	nr,
					int *			pHigh,
					BufferDocument *	bd,
					const BufferItem *	bodySectBi,
					const DocumentNote *	dn,
					DocumentField *		dfNote )
    {
    const DocumentTree *	ei;
    int				high;

    if  ( nr->nrFootnoteCount == 0 )
	{
	nr->nrDfFirstFootnote= dfNote;
	nr->nrFootnoteSectBi= bodySectBi;

	ei= docDocumentNoteSeparator( bd, DOCinFTNSEP );
	if  ( ! ei )
	    { LXDEB(DOCinFTNSEP,ei); return -1;	}
	if  ( ! ei->eiRoot )
	    { LXDEB(DOCinFTNSEP,ei->eiRoot);	}

	high= ei->eiY1UsedTwips- ei->eiY0UsedTwips;
	nr->nrFtnsepHeight= high;
	}

    ei= &(dn->dnDocumentTree);
    high= ei->eiY1UsedTwips- ei->eiY0UsedTwips;

    nr->nrFootnoteCount++;
    nr->nrFootnoteHeight += high;

    *pHigh= high;

    return 0;
    }

/************************************************************************/
/*									*/
/*  Collect the footnotes that belong to a range in a paragraph.	*/
/*  If the range plus the notes fit on the page, the caller will	*/
/*  convert the range into a line and place the notes on this page.	*/
/*									*/
/************************************************************************/

int docLayoutCollectParaFootnoteHeight(	NotesReservation *	nrTotal,
					int			referringPage,
					int			referringColumn,
					BufferDocument *	bd,
					const BufferItem *	bodySectBi,
					const BufferItem *	paraBi,
					int			partFrom,
					int			partUpto )
    {
    int			part;
    TextParticule *	tp;
    
    NotesReservation	nrLocal;

    if  ( paraBi->biInExternalItem != DOCinBODY )
	{ return 0;	}

    docInitNotesReservation( &nrLocal );

    if  ( partUpto > paraBi->biParaParticuleCount )
	{ LLDEB(partUpto,paraBi->biParaParticuleCount); return -1;	}

    tp= paraBi->biParaParticules+ partFrom;
    for ( part= partFrom; part < partUpto; tp++, part++ )
	{
	DocumentField *	df;
	DocumentNote *	dn;
	int		high;

	if  ( tp->tpKind != DOCkindFIELDSTART )
	    { continue;	}
	df= docGetFieldByNumber( &(bd->bdFieldList), tp->tpObjectNumber );
	if  ( ! df || df->dfKind != DOCfkCHFTN )
	    { continue;	}
	dn= docGetNoteOfField( df, bd );
	if  ( ! dn )
	    {
	    XDEB(dn);
	    if  ( docMakeNote( &dn, bd, df, bodySectBi,
						DOCinFOOTNOTE, DOCfkCHFTN ) )
		{ LDEB(1); return -1;	}
	    }

	dn->dnReferringPage= referringPage;
	dn->dnReferringColumn= referringColumn;

	if  ( dn->dnNoteProperties.npExternalItemKind != DOCinFOOTNOTE )
	    { continue;	}

	if  ( docLayoutCollectOneFootnote( &nrLocal, &high,
						    bd, bodySectBi, dn, df ) )
	    { LDEB(df->dfNoteIndex); return -1;	}

#	if SHOW_NOTE_LAYOUT > 1
	appDebug( "PAGE %3d:%d FOOTNOTE      --%5dtw note %d (Reserve)\n",
					    referringPage,
					    referringColumn,
					    high, df->dfNoteIndex );
#	endif
	}

    docLayoutCommitNotesReservation( nrTotal, &nrLocal );

#   if SHOW_NOTE_LAYOUT > 1
    if  ( nrTotal->nrFootnoteCount > 0 )
	{
	appDebug( "PAGE %3d:%d FOOTNOTES     --%5dtw %d notes (Reserved)\n",
					    referringPage,
					    referringColumn,
					    nrTotal->nrFootnoteHeight,
					    nrTotal->nrFootnoteCount );
	}
#   endif

    return 0;
    }

/************************************************************************/
/*									*/
/*  Collect the footnotes from a certain column above the beginning of	*/
/*  a layout action.							*/
/*									*/
/*  Usually, this is the column where we start a layout action.		*/
/*  On pages with different numbers of columns, it may be from the	*/
/*  beginning of the page.						*/
/*									*/
/************************************************************************/

static int docCollectFoornotesForItem(
			    int *			pReady,
			    NotesReservation *		nr,
			    BufferDocument *		bd,
			    const BufferItem *		bodySectBi,
			    const BufferItem *		bi,
			    const DocumentPosition *	dpUpto,
			    int				partUpto,
			    int				referringPage,
			    int				referringColumn )
    {
    int		i;

    if  ( bi == dpUpto->dpBi )
	{ *pReady= 1;	}

    /*  NO!! Bottom positions are not yet known if layout is 
        incomplete.
	Filter on the page of the lines inside the paragraphs instead.

    if  ( bi->biBelowPosition.lpPage < referringPage 		)
	{ goto ready;		}
    if  ( bi->biBelowPosition.lpPage == referringPage 		&&
	  bi->biBelowPosition.lpColumn < referringColumn	)
	{ goto ready;		}
    */

    if  ( bi->biTopPosition.lpPage > referringPage 		)
	{ goto ready;		}

    for ( i= 0; i < bi->biChildCount; i++ )
	{
	if  ( bi->biInExternalItem == DOCinBODY		&&
	      bi->biChildren[i]->biLevel == DOClevSECT	)
	    { bodySectBi= bi->biChildren[i];	}

	if  ( docCollectFoornotesForItem( pReady, nr, bd,
					    bodySectBi, bi->biChildren[i],
					    dpUpto, partUpto,
					    referringPage, referringColumn ) )
	    { LLDEB(referringPage,referringColumn); return -1;	}

	if  ( *pReady )
	    { goto ready;	}
	}

    if  ( bi->biLevel == DOClevPARA )
	{
	const TextLine *	tl= bi->biParaLines;

	for ( i= 0; i < bi->biParaLineCount; tl++, i++ )
	    {
	    int		from;
	    int		upto;

	    if  ( tl->tlTopPosition.lpPage < referringPage 	)
		{ continue;		}
	    if  ( tl->tlTopPosition.lpPage == referringPage 	&&
		  tl->tlTopPosition.lpColumn < referringColumn	)
		{ continue;		}

	    from= tl->tlFirstParticule;
	    upto= tl->tlFirstParticule+ tl->tlParticuleCount;

	    if  ( bi == dpUpto->dpBi && upto > partUpto )
		{ upto= partUpto;	}

	    if  ( upto > from )
		{
		/* Filtering should have been done by the caller */
		if  ( tl->tlTopPosition.lpPage > referringPage )
		    { LLDEB(tl->tlTopPosition.lpPage,referringPage); break; }
		if  ( tl->tlTopPosition.lpPage == referringPage &&
		      tl->tlTopPosition.lpColumn > referringColumn )
		    {
		    LLLDEB(referringPage,
			    tl->tlTopPosition.lpColumn,referringColumn);
		    break;
		    }

		if  ( docLayoutCollectParaFootnoteHeight( nr,
					    tl->tlTopPosition.lpPage,
					    tl->tlTopPosition.lpColumn,
					    bd, bodySectBi, bi, from, upto ) )
		    { LDEB(1); return -1;	}
		}

	    if  ( bi == dpUpto->dpBi && upto >= partUpto )
		{ /* Above: *pReady= 1; */ break;	}
	    }
	}

  ready:

    return 0;
    }

/************************************************************************/
/*									*/
/*  Collect footnotes that have been formatted earlier.			*/
/*									*/
/************************************************************************/

int docCollectFootnotesFromColumn(
			BlockFrame *			bf,
			const DocumentPosition *	dpHere,
			int				partHere,
			BufferDocument *		bd,
			int				referringPage,
			int				referringColumn )
    {
    NotesReservation		nr;
    int				ready= 0;

    docInitNotesReservation( &nr );

    /*  Does not work because part of the document might not be 
        formatted yet.
    if  ( docGetTopOfColumn( &dpPageTop, &linePageTop, &partPageTop, bd,
					referringPage, referringColumn ) )
	{ LLDEB(referringPage,referringColumn); return -1;	}
    */

    if  ( docCollectFoornotesForItem( &ready, &nr, bd,
			bd->bdBody.eiRoot->biChildren[0], bd->bdBody.eiRoot,
			dpHere, partHere, referringPage, referringColumn ) )
	{ LDEB(1); return -1;	}

    bf->bfNotesReservation= nr;

    return 0;
    }

/************************************************************************/
/*									*/
/*  Place the lines of a footnote on the page that refers to it.	*/
/*									*/
/*  2)  Get the separator between body and notes and calculate its	*/
/*	height.								*/
/*  3)  Start a new layout job directly below the section text.		*/
/*  4)  When the notes are to be placed at the bottom of the page, find	*/
/*	another start position.						*/
/*  5)  Skip the space reserved for the separator line between the	*/
/*	body text and the notes.					*/
/*  6)  Place the notes one below the other from the calculated top	*/
/*	position. As the notes are supposed to be small, their layout	*/
/*	is recalculated at little extra cost.				*/
/*  7)  Force the footnote separator to be reformatted later on.	*/
/*									*/
/************************************************************************/

int docLayoutFootnotesForColumn(	LayoutPosition *	lpBelowNotes,
					const LayoutPosition *	lpBelowText,
					const BlockFrame *	refBf,
					int			belowText,
					const LayoutJob *	refLj )
    {
    const LayoutContext *	lc= &(refLj->ljContext);
    BufferDocument *		bd= lc->lcDocument;
    const NotesReservation *	refNr= &(refBf->bfNotesReservation);
    const DocumentProperties *	dp= &(bd->bdProperties);
    const NotesProperties *	npFootnotes= &(dp->dpFootnoteProperties);

    LayoutJob			notesLj;
    LayoutPosition		lpHere;

    DocumentTree *		ei;
    DocumentTree *		eiNoteSep;
    int				notesDone;

    int				sepHigh= 0;

    DocumentField *		dfNote;
    DocumentNote *		dn;

    if  ( refBf->bfFootnotesPlaced )
	{ LDEB(refBf->bfFootnotesPlaced); return 0;	}

    /*  2  */
    eiNoteSep= docDocumentNoteSeparator( bd, DOCinFTNSEP );
    if  ( ! eiNoteSep )
	{ LXDEB(DOCinFTNSEP,eiNoteSep); return -1;	}
    if  ( ! eiNoteSep->eiRoot )
	{ LXDEB(DOCinFTNSEP,eiNoteSep->eiRoot);	}

    sepHigh= eiNoteSep->eiY1UsedTwips- eiNoteSep->eiY0UsedTwips;

    /*  3  */
    notesLj= *refLj;
    lpHere.lpPage= lpBelowText->lpPage;
    lpHere.lpColumn= lpBelowText->lpColumn;
    lpHere.lpPageYTwips= lpBelowText->lpPageYTwips;
    lpHere.lpAtTopOfColumn= 1; /* not really */

    /*  4  */
    if  ( ! belowText && npFootnotes->npPosition == FTN_POS_PAGE_BOTTOM )
	{
	int		high= sepHigh;

	dfNote= refNr->nrDfFirstFootnote;
	dn= docGetNoteOfField( dfNote, bd );
	notesDone= 0;

	for ( ;
	      dfNote && notesDone < refNr->nrFootnoteCount;
	      dfNote= docGetNextNoteInDocument( &dn, bd,
						    dfNote, DOCinFOOTNOTE ) )
	    {
	    ei= &(dn->dnDocumentTree);
	    if  ( ! ei->eiRoot )
		{ XDEB(ei->eiRoot); continue;	}

	    high += ei->eiY1UsedTwips- ei->eiY0UsedTwips;
	    notesDone++;
	    }

	lpHere.lpPageYTwips= refBf->bfContentRect.drY1- high;
	}

    /*  5  */
    lpHere.lpPageYTwips += sepHigh;

    /*  6  */
    dfNote= refNr->nrDfFirstFootnote;
    dn= docGetNoteOfField( dfNote, bd );
    notesDone= 0;
    for ( ;
          dfNote && notesDone < refNr->nrFootnoteCount;
	  dfNote= docGetNextNoteInDocument( &dn, bd, dfNote, DOCinFOOTNOTE ) )
	{
	int			y0Twips= lpHere.lpPageYTwips;

	/*  7  */
	eiNoteSep->eiPageFormattedFor= -1;
	eiNoteSep->eiColumnFormattedFor= -1;

	ei= &(dn->dnDocumentTree);
	if  ( ! ei->eiRoot )
	    { XDEB(ei->eiRoot); continue;	}

	if  ( ei->eiPageFormattedFor != lpBelowText->lpPage	||
	      ei->eiColumnFormattedFor != lpBelowText->lpColumn	||
	      ei->eiY0UsedTwips != y0Twips			)
	    {
	    DocumentTree *	eiBody;
	    BlockFrame		bfNote;
	    BufferItem *	bodySectBi;

	    if  ( docGetRootOfSelectionScope( &eiBody, &bodySectBi,
					    bd, &(dfNote->dfSelectionScope) ) )
		{ LDEB(1); return -1;	}

	    notesLj.ljBodySectBi= bodySectBi;

	    docLayoutInitBlockFrame( &bfNote );
	    if  ( docLayoutGetInitialFrame( &bfNote, &notesLj, &lpHere,
								ei->eiRoot ) )
		{ LDEB(1); return -1;	}

	    if  ( docLayoutItemImplementation( &lpHere, &lpHere,
					    ei->eiRoot, &bfNote, &notesLj ) )
		{ LDEB(1); return -1;	}

	    /*  HACK to avoid corruption of the height administration.
	        Eventually, this should be done differently.
	     */
	    if  ( lpHere.lpPage != lpBelowText->lpPage		||
		  lpHere.lpColumn != lpBelowText->lpColumn	)
		{
		if  ( refLj->ljBalancePage < 0 )
		    {
		    LLDEB(lpHere.lpPage,lpBelowText->lpPage);
		    LLDEB(lpHere.lpColumn,lpBelowText->lpColumn);
		    }
		}
	    else{
		ei->eiPageFormattedFor= lpBelowText->lpPage;
		ei->eiColumnFormattedFor= lpBelowText->lpColumn;
		ei->eiY0UsedTwips= y0Twips;
		ei->eiY1UsedTwips= lpHere.lpPageYTwips;
		}
	    }
	else{
	    lpHere.lpPageYTwips= ei->eiY1UsedTwips;
	    }

#	if SHOW_NOTE_LAYOUT > 0
	appDebug( "PAGE %3d:%d NOTE          ::%5dtw note %d (Layout)\n",
			    lpBelowText->lpPage,
			    lpBelowText->lpColumn,
			    ei->eiRoot->biBelowPosition.lpPageYTwips-
				ei->eiRoot->biTopPosition.lpPageYTwips,
			    dfNote->dfNoteIndex );
#       endif

	notesDone++;
	}

    *lpBelowNotes= lpHere;
    lpBelowNotes->lpAtTopOfColumn= lpBelowText->lpAtTopOfColumn;

    return 0;
    }

/************************************************************************/
/*									*/
/*  Place the notes belonging to a section immediately below its text.	*/
/*									*/
/*  1)  Are there any?							*/
/*  2)  Get separator.							*/
/*  2a) If there are footnotes on the same page, the footnotes come	*/
/*	first.								*/
/*  3)  Layout of separator.						*/
/*  4)  Layout of the endnotes of this section.				*/
/*  7)  Force the footnote separator to be reformatted later on.	*/
/*									*/
/************************************************************************/

int docLayoutEndnotesForSection(	LayoutPosition *	lpBelow,
					const LayoutPosition *	lpTop,
					int			sect,
					BlockFrame *		bf,
					LayoutJob *		lj )
    {
    const LayoutContext *	lc= &(lj->ljContext);
    BufferDocument *		bd= lc->lcDocument;
    const DocumentProperties *	dp= &(bd->bdProperties);
    const NotesProperties *	npFootnotes= &(dp->dpFootnoteProperties);

    DocumentNote *		dn;
    DocumentField *		dfFirstNote;

    DocumentTree *		ei;
    DocumentTree *		eiNoteSep;

    int				attempt;

    LayoutPosition		lpHere= *lpTop;

    /*  1  */
    if  ( sect >= 0 )
	{
	dfFirstNote= docGetFirstNoteOfSection( &dn, bd, sect, DOCinENDNOTE );
	}
    else{
	dfFirstNote= docGetFirstNoteOfDocument( &dn, bd, DOCinENDNOTE );
	}

    if  ( ! dfFirstNote )
	{ return 0;	}

    /*  2  */
    eiNoteSep= docDocumentNoteSeparator( bd, DOCinAFTNSEP );
    if  ( ! eiNoteSep )
	{ LXDEB(DOCinAFTNSEP,eiNoteSep); return -1;	}
    if  ( ! eiNoteSep->eiRoot )
	{ LXDEB(DOCinAFTNSEP,eiNoteSep->eiRoot);	}

    /*  2a  */
    if  ( BF_HAS_FOOTNOTES( bf )			&&
	  npFootnotes->npPosition == FTN_POS_BELOW_TEXT	)
	{
	const int	belowText= 1;

	if  ( docLayoutFootnotesForColumn( &lpHere, &lpHere, bf,
							    belowText, lj ) )
	    { LDEB(1); return -1;	}

	bf->bfFootnotesPlaced= 1;
	}

    for ( attempt= 0; attempt < 2; attempt++ )
	{
	int		retry= 0;
	DocumentField *	dfNote;

	if  ( eiNoteSep->eiRoot )
	    {
	    int		y0Twips= lpHere.lpPageYTwips;

	    /*  3  */
	    if  ( docLayoutItemImplementation( &lpHere, &lpHere,
						eiNoteSep->eiRoot, bf, lj ) )
		{ LDEB(1); return -1;	}

	    eiNoteSep->eiPageFormattedFor=
				    eiNoteSep->eiRoot->biTopPosition.lpPage;
	    eiNoteSep->eiColumnFormattedFor=
				    eiNoteSep->eiRoot->biTopPosition.lpColumn;
	    eiNoteSep->eiY0UsedTwips= y0Twips;
	    eiNoteSep->eiY1UsedTwips= lpHere.lpPageYTwips;

	    /*  7  */
	    eiNoteSep->eiPageFormattedFor= -1;
	    eiNoteSep->eiColumnFormattedFor= -1;
	    }

	/*  4  */
	for ( dfNote= dfFirstNote;
	      dfNote;
	      dfNote= docGetNextNoteInSection( &dn, bd, sect,
						    dfNote, DOCinENDNOTE ) )
	    {
	    int		y0Twips= lpHere.lpPageYTwips;

	    ei= &(dn->dnDocumentTree);
	    if  ( ! ei->eiRoot )
		{ XDEB(ei->eiRoot); continue;	}

	    if  ( docLayoutItemImplementation( &lpHere, &lpHere,
							ei->eiRoot, bf, lj ) )
		{ LDEB(1); return -1;	}

	    ei->eiPageFormattedFor= ei->eiRoot->biTopPosition.lpPage;
	    ei->eiColumnFormattedFor= ei->eiRoot->biTopPosition.lpColumn;
	    ei->eiY0UsedTwips= y0Twips;
	    ei->eiY1UsedTwips= lpHere.lpPageYTwips;

	    if  ( attempt == 0						&&
		  dfNote == dfFirstNote					&&
		  eiNoteSep->eiRoot					&&
		  eiNoteSep->eiRoot->biTopPosition.lpPage <
				    ei->eiRoot->biTopPosition.lpPage	)
		{
		lpHere= ei->eiRoot->biTopPosition;
		retry= 1;
		break;
		}
	    }

	if  ( ! retry )
	    { break;	}
	}

    *lpBelow= lpHere;
    return 0;
    }

int docLayoutEndnotesForDocument(	LayoutPosition *	lpBelow,
					const LayoutPosition *	lpTop,
					BlockFrame *		bf,
					LayoutJob *		lj )
    {
    return docLayoutEndnotesForSection( lpBelow, lpTop, -1, bf, lj );
    }

/************************************************************************/
/*									*/
/*  Determine the rectangle that the note separator would occupy on a	*/
/*  certain page.							*/
/*									*/
/************************************************************************/

int docNoteSeparatorRectangle(	DocumentRectangle *	drExtern,
				DocumentTree **		pEiNoteSep,
				int *			pY0Twips,
				const BufferItem *	bodySectBi,
				const DocumentNote *	dnFirstNote,
				int			extItKind,
				const LayoutContext *	lc )
    {
    const DocumentTree *	eiFirstNote= &(dnFirstNote->dnDocumentTree);
    DocumentTree *		eiNoteSep;

    int				page;
    int				column;

    int				y0Twips;
    int				y1Twips;

    int				high;

    BlockFrame			bfNoteSep;
    DocumentRectangle		drTwips;

    eiNoteSep= docDocumentNoteSeparator( lc->lcDocument, extItKind );
    if  ( ! eiNoteSep )
	{ LXDEB(extItKind,eiNoteSep); return -1;	}
    if  ( ! eiNoteSep->eiRoot )
	{ LXDEB(extItKind,eiNoteSep->eiRoot); return 1;	}

    page= eiFirstNote->eiRoot->biTopPosition.lpPage;
    column= eiFirstNote->eiRoot->biTopPosition.lpColumn;
    y1Twips= eiFirstNote->eiRoot->biTopPosition.lpPageYTwips;
    high= eiNoteSep->eiY1UsedTwips- eiNoteSep->eiY0UsedTwips;
    y0Twips= y1Twips- high;

    docLayoutInitBlockFrame( &bfNoteSep );
    docBlockFrameTwips( &bfNoteSep, eiFirstNote->eiRoot,
				bodySectBi, lc->lcDocument, page, column );

    drTwips.drX0= bfNoteSep.bfContentRect.drX0;
    drTwips.drX1= bfNoteSep.bfContentRect.drX1;
    drTwips.drY0= y0Twips;
    drTwips.drY1= y1Twips;

    docGetPixelRect( drExtern, lc, &drTwips, page );

    *pEiNoteSep= eiNoteSep;
    *pY0Twips= y0Twips;
    return 0;
    }

/************************************************************************/
/*									*/
/*  Commit the space allocated for footnotes and the footnote separator	*/
/*  in a frame.								*/
/*									*/
/************************************************************************/

void docLayoutReserveNoteHeight(	BlockFrame *			bf,
					const NotesReservation *	nr )
    {
    int			footnoteHeight;
    NotesReservation *	nrBf= &(bf->bfNotesReservation);

    docLayoutCommitNotesReservation( nrBf, nr );

    footnoteHeight= nrBf->nrFtnsepHeight+ nrBf->nrFootnoteHeight;

    if  ( bf->bfFlowRect.drY1 > bf->bfContentRect.drY1- footnoteHeight )
	{ bf->bfFlowRect.drY1=  bf->bfContentRect.drY1- footnoteHeight;	}

#   if SHOW_NOTE_LAYOUT > 1
    if  ( nrBf->nrFootnoteCount > 0 )
	{
	appDebug( "           NOTES         ,,%5dtw %d notes (Page %d:%d,First=%d)\n",
					footnoteHeight,
					nrBf->nrFootnoteCount,
					bf->bfPage, bf->bfColumn,
					nrBf->nrDfFirstFootnote->dfNoteIndex );
	}
#   endif

    return;
    }

int docSectNotesPrelayout(	int			sect,
				const BufferItem *	bodySectBi,
				LayoutJob *		lj )
    {
    const LayoutContext *	lc= &(lj->ljContext);
    BufferDocument *		bd= lc->lcDocument;

    DocumentField *		dfNote;
    DocumentNote *		dn;
    const int			extItKind= -1;

    if  ( sect >= 0 )
	{
	dfNote= docGetFirstNoteOfSection( &dn, bd, sect, extItKind );
	}
    else{
	dfNote= docGetFirstNoteOfDocument( &dn, bd, extItKind );
	}

    for ( ;
	  dfNote;
	  dfNote= docGetNextNoteInSection( &dn, bd, sect, dfNote, extItKind ) )
	{
	if  ( docExternalItemPrelayout( &(dn->dnDocumentTree), bodySectBi, lj ) )
	    { LDEB(1); return -1;	}
	}

    return 0;
    }
