/************************************************************************/
/*									*/
/*  Layout of a document.						*/
/*									*/
/************************************************************************/

#   include	"tedConfig.h"

#   include	<stddef.h>
#   include	<stdio.h>
#   include	<stdlib.h>
#   include	<limits.h>

#   include	"docLayout.h"

#   include	<appDebugon.h>

/************************************************************************/
/*									*/
/*  Place successive items, after the predecessor. This routine is	*/
/*  called for the tail of the document after a real formatting change.	*/
/*									*/
/*  Exceptions are covered by the caller: docAdjustParentLayout()	*/
/*  or by the calls to docLayoutItemImplementation(). The only added	*/
/*  knowledge here is that as long the stacking of items ends up in the	*/
/*  same position, subsequent siblings will not change either.		*/
/*  The only exception to this rule: A change to a table header is	*/
/*  covered by another exception: Children of a section are handled by	*/
/*  docLayoutSectItem()/docLayoutSectChildren().			*/
/*									*/
/************************************************************************/

static int docReLayoutStackedChildren(
				LayoutPosition *		lpBelow,
				const LayoutPosition *		lpTop,
				BufferItem *			parentBi,
				int				from,
				BlockFrame *			bf,
				LayoutJob *			lj )
    {
    const LayoutContext *	lc= &(lj->ljContext);
    int				i;
    int				changed;
    DocumentRectangle *		drChanged= lj->ljChangedRectanglePixels;

    LayoutPosition		lpHere= *lpTop;

    for ( i= from; i < parentBi->biChildCount; i++ )
	{
	BufferItem *	child= parentBi->biChildren[i];

	if  ( DOC_SAME_POSITION( &(child->biTopPosition), &lpHere ) )
	    {
	    const BufferItem *	last;

	    last= parentBi->biChildren[parentBi->biChildCount- 1];
	    lpHere= last->biBelowPosition;
	    break;
	    }

	child->biTopPosition= lpHere;

	if  ( docLayoutItemImplementation( &lpHere, &lpHere,
						    child, bf, lj ) )
	    { LDEB(i); return -1;	}

	docLayoutSetItemBottom( &changed, child, &lpHere, lc, drChanged );
	}

    *lpBelow= lpHere;
    return 0;
    }

/************************************************************************/
/*									*/
/*  Reformat the document body because a header/footer/note changed	*/
/*  size.								*/
/*									*/
/************************************************************************/

static int docRedoBodyLayout(		BufferItem *		bodyBi,
					const LayoutJob *	ljRef )
    {
    LayoutJob			bodyLj;

    if  ( bodyBi->biInExternalItem != DOCinBODY )
	{ LDEB(bodyBi->biInExternalItem);	}

    bodyLj= *ljRef;
    /* bodyLj.ljContext= ljRef->ljContext; */

    bodyLj.ljChangedItem= bodyBi;

    if  ( docLayoutItemAndParents( bodyBi, &bodyLj ) )
	{ LDEB(1); return -1;	}

    return 0;
    }

static int docRedoBodyItemLayout(	BufferItem *		bi,
					const LayoutPosition *	lpHere,
					const LayoutJob *	lj )

    {
    const LayoutContext *	lc= &(lj->ljContext);
    BufferDocument *		bd= lc->lcDocument;

    DocumentTree *		ei;
    BufferItem *		bodyBi;

    if  ( lj->ljChangedItem->biInExternalItem != bi->biInExternalItem )
	{
	LDEB(lj->ljChangedItem->biInExternalItem);
	LDEB(bi->biInExternalItem);
	return 0;
	}

    if  ( docGetRootForItem( &ei, &bodyBi, bd, bi ) )
	{ LDEB(bi->biInExternalItem); return -1; }
    if  ( ! bodyBi )
	{ bodyBi= bd->bdBody.eiRoot;	}

    if  ( lpHere->lpPageYTwips == ei->eiY1UsedTwips )
	{ return 0;	}

    if  ( docRedoBodyLayout( bodyBi, lj ) )
	{ LDEB(bi->biInExternalItem); return -1;	}

    return 0;
    }

/************************************************************************/
/*									*/
/*  Fixup routine that is called when the first child item does not	*/
/*  have the same top position as its parent.				*/
/*									*/
/*  This can happen when during a reformat of part of the document	*/
/*  the child moves to a different page. E.G: When a property change	*/
/*  of a paragraph makes it higher or lower.				*/
/*									*/
/************************************************************************/

static int docPsFixupParentGeometry(	BufferItem *		bi,
					BufferItem *		biParent )
    {
    LayoutPosition	lpBi= bi->biTopPosition;
    LayoutPosition	lpPa= biParent->biTopPosition;

    if  ( ! DOC_SAME_POSITION( &lpPa, &lpBi ) )
	{
#	if 0
	SSDEB(docLevelStr(biParent->biLevel),docLevelStr(bi->biLevel));
	LLDEB(lpPa.lpPage,lpBi.lpPage);
	LLDEB(lpPa.lpPageYTwips,lpBi.lpPageYTwips);
#	endif

	biParent->biTopPosition= lpBi;

	return 1;
	}

    return 0;
    }

/************************************************************************/
/*									*/
/*  Find the position in the document and the position on the page	*/
/*  to start adjusting the layout of a paragraph.			*/
/*									*/
/************************************************************************/

static int docAdjustParaParentLayout(	LayoutPosition *	lpHere,
					const BufferItem *	bi,
					const BufferItem *	parentBi )
    {
    int			from;
    ParagraphLayoutJob	plj;
    const int		line= 0;
    const int		part= 0;

    docInitParagraphLayoutJob( &plj );

    docBeginParagraphLayoutProgress( &plj,
				bi->biNumberInParent+ 1, line, part,
				parentBi->biChildCount, lpHere );

    docFindStripLayoutOrigin( &plj,
				lpHere->lpPage, lpHere->lpColumn, parentBi );

    if  ( bi->biParaBreakKind != DOCibkNONE	&&
	  bi->biParaTableNesting == 0	)
	{
	const BufferItem *	bbi= bi;

	while( bbi->biNumberInParent == 0 )
	    {
	    if  ( ! bbi->biParent )
		{ break;	}

	    bbi->biParent->biTopPosition= bi->biTopPosition;
	    bbi= bbi->biParent;
	    }
	}

    if  ( plj.pljPos0.pspChild >= bi->biNumberInParent+ 1 )
	{
	if  ( plj.pljPos0.pspChild > bi->biNumberInParent+ 1 )
	    { LLDEB(plj.pljPos0.pspChild,bi->biNumberInParent);	}

	from= bi->biNumberInParent+ 1;
	}
    else{
	from= plj.pljPos0.pspChild;

	if  ( from == 0 )
	    { *lpHere= parentBi->biChildren[from]->biTopPosition;	}
	else{ *lpHere= parentBi->biChildren[from- 1]->biBelowPosition;	}
	}

    docCleanParagraphLayoutJob( &plj );

    return from;
    }

/************************************************************************/
/*									*/
/*  Adjust the geometry of a parent item to changes in a child.		*/
/*									*/
/*  This actually is a full layout action of everything below the	*/
/*  recently reformatted part. There are two differences:		*/
/*  a)  Reformatting stops when a buffer item lands in the same		*/
/*	position where it was before the change that forced us to redo	*/
/*	part of the layout.						*/
/*  b)  We try not to recalculate the layout of the text inside the	*/
/*	individual text lines.						*/
/*									*/
/************************************************************************/

int docAdjustParentLayout(	const LayoutPosition *	lpTop,
				BufferItem *		bi,
				BlockFrame *		bf,
				LayoutJob *		lj )
    {
    const LayoutContext *	lc= &(lj->ljContext);
    BufferDocument *		bd= lc->lcDocument;
    const DocumentProperties *	dp= &(bd->bdProperties);
    const NotesProperties *	npEndnotes= &(dp->dpEndnoteProperties);

    DocumentRectangle *		drChanged= lj->ljChangedRectanglePixels;
    BufferItem *		parentBi;
    int				from;

    LayoutPosition		lpHere= *lpTop;

    parentBi= bi->biParent;

if(bf->bfPage!=lpHere.lpPage||
   bf->bfColumn!=lpHere.lpColumn)
  {
  SDEB("!!!!!!!!!!!!!!");
  LLDEB(bf->bfPage,lpHere.lpPage);
  LLDEB(bf->bfColumn,lpHere.lpColumn);
  }

    if  ( parentBi && parentBi->biLevel == DOClevCELL )
	{ from= docAdjustParaParentLayout( &lpHere, bi, parentBi );	}
    else{ from= bi->biNumberInParent+ 1;				}

    while( parentBi )
	{
	int		changed= 0;

	switch( parentBi->biLevel )
	    {
	    int		keepRowOnPage;

	    case DOClevBODY:
		if  ( bi->biNumberInParent == 0			&&
		      docPsFixupParentGeometry( bi, parentBi )	)
		    { from= 0; lpHere= bi->biTopPosition; }

		if  ( from <= parentBi->biChildCount- 1 )
		    {
		    docReLayoutStackedChildren( &lpHere, &lpHere,
						    parentBi, from, bf, lj );
		    }
		else{ lpHere= bi->biBelowPosition;		}

		if  ( parentBi->biInExternalItem == DOCinBODY	&&
		      npEndnotes->npPosition == FTN_POS_DOC_END	)
		    {
		    if  ( docLayoutEndnotesForDocument( &lpHere, &lpHere,
								    bf, lj ) )
			{ LDEB(1); return -1;	}
		    }

		break;

	    case DOClevSECT:
		if  ( bi->biNumberInParent == 0			&&
		      docPsFixupParentGeometry( bi, parentBi )	)
		    { from= 0; lpHere= bi->biTopPosition; }

		if  ( from <= parentBi->biChildCount- 1 )
		    {
		    docLayoutSectChildren( &lpHere, &lpHere,
						    parentBi, from, bf, lj );
		    }
		else{ lpHere= bi->biBelowPosition;		}

		if  ( docLayoutFinishSectItem( &lpHere, &lpHere,
							parentBi, bf, lj ) )
		    { LDEB(1); return -1;	}

		break;

	    parentAsGroup:
		if  ( bi->biNumberInParent == 0			&&
		      docPsFixupParentGeometry( bi, parentBi )	)
		    { from= 0; lpHere= bi->biTopPosition; }

		if  ( from <= parentBi->biChildCount- 1 )
		    {
		    docReLayoutStackedChildren( &lpHere, &lpHere,
						    parentBi, from, bf, lj );
		    }
		else{ lpHere= bi->biBelowPosition;		}

		break;

	    case DOClevCELL:
		if  ( bi->biNumberInParent == 0			&&
		      docPsFixupParentGeometry( bi, parentBi )	)
		    { from= 0; lpHere= bi->biTopPosition; }

		if  ( from <= parentBi->biChildCount- 1 )
		    {
		    if  ( docLayoutCellItem( &lpHere, &lpHere,
						    parentBi, from, bf, lj ) )
			{ LDEB(1); return -1;	}
		    }
		else{ lpHere= bi->biBelowPosition;		}

		break;

	    case DOClevROW:
		{
		int		stopCode= FORMATstopREADY;
		int		stayInThisColumn= 0;

		if  ( ! docIsRowItem( parentBi ) )
		    {
		    if  ( parentBi->biNumberInParent > 0 )
			{ docLayoutCalculateAfterRowTopInset( parentBi, bd ); }

		    goto parentAsGroup;
		    }

		keepRowOnPage= BI_ROW_IS_ONE_PAGE( parentBi );

		if  ( parentBi->biNumberInParent > 0	&&
		      parentBi->biParent		&&
		      keepRowOnPage			)
		    {
		    BufferItem *	pp= parentBi->biParent;
		    BufferItem *	ch;

		    ch= pp->biChildren[parentBi->biNumberInParent- 1];

		    parentBi->biTopPosition= ch->biBelowPosition;
		    }

		lpHere= parentBi->biTopPosition;

		if  ( bf->bfPage != lpHere.lpPage	||
		      bf->bfColumn != lpHere.lpColumn	)
		    {
		    if  ( docLayoutGetInitialFrame( bf, lj, &lpHere,
								parentBi ) )
			{ LDEB(1); return -1;	}
		    }

		if  ( docLayoutRowItem( &stopCode,
				    &lpHere, &lpHere,
				    parentBi, bf, stayInThisColumn, lj ) )
		    { LDEB(1); return -1;	}
		}

		break;

	    default:
		LDEB(parentBi->biLevel); return -1;
	    }

	docLayoutSetItemBottom( &changed, parentBi, &lpHere, lc, drChanged );

	if  ( ! changed )
	    { break;	}

	bi= parentBi; parentBi= bi->biParent;
	from= bi->biNumberInParent+ 1;
	}

    if  ( bi->biInExternalItem == DOCinBODY	&&
	  BF_HAS_FOOTNOTES( bf )		)
	{
	const int		belowText= 0;
	LayoutPosition		lpBelowNotes;

	if  ( docLayoutFootnotesForColumn( &lpBelowNotes, &lpHere, bf,
							    belowText, lj ) )
	    { LDEB(1); return -1;	}
	}

    if  ( ! parentBi )
	{
	switch( bi->biInExternalItem )
	    {
	    case DOCinBODY:
		if  ( lj->ljAdjustScreenRectangle )
		    { (*lj->ljAdjustScreenRectangle)( &lpHere, lj );	}
		break;

	    case DOCinFIRST_HEADER:
	    case DOCinLEFT_HEADER:
	    case DOCinRIGHT_HEADER:
	    case DOCinFIRST_FOOTER:
	    case DOCinLEFT_FOOTER:
	    case DOCinRIGHT_FOOTER:

		if  ( bi->biLevel != DOClevSECT )
		    { LLDEB(bi->biLevel,DOClevSECT); return -1;	}
		if  ( lj->ljChangedItem->biInExternalItem == DOCinBODY )
		    { break; }

		if  ( docRedoBodyItemLayout( bi, &lpHere, lj ) )
		    { LDEB(1); return -1;	}

		break;

	    case DOCinFOOTNOTE:
	    case DOCinENDNOTE:

		if  ( bi->biLevel != DOClevSECT )
		    { LLDEB(bi->biLevel,DOClevSECT); return -1;	}
		if  ( lj->ljChangedItem->biInExternalItem == DOCinBODY )
		    { break; }

		if  ( docRedoBodyItemLayout( bi, &lpHere, lj ) )
		    { LDEB(1); return -1;	}

		break;

	    case DOCinFTNSEP:
	    case DOCinFTNSEPC:
	    case DOCinFTNCN:
	    case DOCinAFTNSEP:
	    case DOCinAFTNSEPC:
	    case DOCinAFTNCN:

		if  ( bi->biLevel != DOClevSECT )
		    { LLDEB(bi->biLevel,DOClevSECT); return -1;	}
		if  ( lj->ljChangedItem->biInExternalItem == DOCinBODY )
		    { break; }

		if  ( docRedoBodyItemLayout( bi, &lpHere, lj ) )
		    { LDEB(1); return -1;	}

		break;

	    case DOCinSHPTXT:
		/*nothing*/
		break;

	    default:
		LDEB(bi->biInExternalItem);
		break;
	    }
	}

    return 0;
    }

