/************************************************************************/
/*									*/
/*  Pdfmark related PostScript generation.				*/
/*									*/
/************************************************************************/

#   include	"utilPsConfig.h"

#   include	<stddef.h>
#   include	<stdio.h>
#   include	<stdlib.h>
#   include	<string.h>
#   include	<time.h>

#   include	<psPrint.h>
#   include	<geo2DInteger.h>
#   include	<sioGeneral.h>

#   include	<appDebugon.h>

/************************************************************************/
/*									*/
/*  Include pdfmark/distiller related emulation in the PostScript	*/
/*  output.								*/
/*									*/
/************************************************************************/

static const char * appPsPdfmarkEmulation[]=
    {
    "%%pdfmark emulation",
    "/pdfmark where",
    "    { pop }",
    "    { userdict /pdfmark /cleartomark load put }",
    "ifelse",
    };

static const char * appPssetdistillerparamsEmulation[]=
    {
    "%%setdistillerparams emulation",
    "/setdistillerparams where",
    "    { pop }",
    "    { /setdistillerparams { pop } bind def }",
    "ifelse",
    };

void psSetPdfmarkEmulation(	SimpleOutputStream *	sos )
    {
    psDefineProcedure( sos, appPsPdfmarkEmulation,
		sizeof(appPsPdfmarkEmulation)/sizeof(const char *) );

    psDefineProcedure( sos, appPssetdistillerparamsEmulation,
		sizeof(appPssetdistillerparamsEmulation)/sizeof(const char *) );

    return;
    }

static const char * appPspsImageQualityDistillerparams[]=
    {
    "<<",
    "  /AutoFilterColorImages false",
    "  /ColorImageFilter /FlateEncode",
    "  /ColorACSImageDict <<",
    "    /QFactor 0.5",
    "    /Blend 1",
    "    /ColorTransform 1",
    "    /HSamples [1 1 1 1]",
    "    /VSamples [1 1 1 1] >>",
    ">> setdistillerparams",
    };

void psImageQualityDistillerparams(	SimpleOutputStream *	sos )
    {
    psDefineProcedure( sos, appPspsImageQualityDistillerparams,
	    sizeof(appPspsImageQualityDistillerparams)/sizeof(const char *) );
    }

/************************************************************************/
/*									*/
/*  Emit the bookmark part of an internal link.				*/
/*									*/
/*  As the link is internal, we save both the source of the and its	*/
/*  destination. This makes it possible to replace strange characters,	*/
/*  rather than to escape them to pass them through the PostScript	*/
/*  interpreter and the pdfmark translator.				*/
/*									*/
/************************************************************************/

static void utilPsEmitDestination(	SimpleOutputStream *	sos,
					const unsigned char *	destName,
					int			destSize )
    {
    int				i;
    const unsigned char *	s= destName;

    for ( i= 0; i < destSize; s++, i++ )
	{
	if  ( *s == '('	|| *s == ')'		||
	      *s == '<'	|| *s == '>'		||
	      *s == '['	|| *s == ']'		||
	      *s == '{'	|| *s == '}'		||
	      *s == '/'	|| *s == '%'		||
	      isspace( *s ) || ! isascii( *s )	)
	    {
	    sioOutPutByte( '_', sos );
	    continue;
	    }

	sioOutPutByte( *s, sos );
	}

    return;
    }

/************************************************************************/
/*									*/
/*  Insert the destination of a pdfmark jump in the Printout.		*/
/*									*/
/************************************************************************/

int psEmitDestPdfmark(		SimpleOutputStream *		sos,
				const AffineTransform2D *	at,
				int				lineTop,
				const char *			refName,
				int				refSize )
    {
    int			top;
    int			x= 0;
    int			y= lineTop;

    top= AT2_Y( x, y, at );

    sioOutPrintf( sos, "[ /Dest /" );

    utilPsEmitDestination( sos, (const unsigned char *)refName, refSize );
    sioOutPrintf( sos, "\n" );

    sioOutPrintf( sos, "  /View [ /XYZ null %d null ]\n", top );

    sioOutPrintf( sos, "/DEST pdfmark\n" );

    return 0;
    }

/************************************************************************/
/*									*/
/*  Write links to be picked up by the distiller.			*/
/*									*/
/************************************************************************/

/************************************************************************/
/*									*/
/*  Emit the bookmark part of an external link.				*/
/*									*/
/*  As the link is external, do not replace characters in the link as	*/
/*  we do for internal links and bookmarks: We mut trust the link as	*/
/*  we do not know that we are going to change the bookmark name.	*/
/*									*/
/************************************************************************/

static void utilPsUriLinkDestination(	SimpleOutputStream *	sos,
					const unsigned char *	fileName,
					int			fileSize,
					const unsigned char *	markName,
					int			markSize )
    {
    const int	sevenBits= 1;

    psPrintString( sos, fileName, fileSize, sevenBits );

    if  ( markSize > 0 )
	{
	sioOutPutByte( '#', sos );
	psPrintString( sos, markName, markSize, sevenBits );
	}
    }

static void utilPsWebLinkDestination(	SimpleOutputStream *	sos,
					const unsigned char *	fileName,
					int			fileSize,
					const unsigned char *	markName,
					int			markSize )
    {
    sioOutPrintf( sos, "  /Action << /Subtype /URI /URI (" );

    utilPsUriLinkDestination( sos, fileName, fileSize, markName, markSize );

    sioOutPrintf( sos, ") >>\n" );

    return;
    }

static void utilPsFileLinkDestMark(	SimpleOutputStream *	sos,
					const unsigned char *	fileName,
					int			fileSize,
					const unsigned char *	markName,
					int			markSize )
    {
    const int			sevenBits= 1;
    const unsigned char *	file;
    int				size;

    file= fileName;
    size= fileSize;

    if  ( size > 5 && ! strncmp( (const char *)file, "file:", 5 ) )
	{ file += 5; size -= 5; }
    else{
	while( size > 0 && isalpha( *file ) )
	    { file++; size--;	}

	if  ( size > 0 && *file == ':' )
	    {
	    utilPsWebLinkDestination( sos,
				fileName, fileSize, markName, markSize );
	    return;
	    }

	file= fileName;
	size= fileSize;
	}

    sioOutPrintf( sos, "  /Action /Launch /File (" );

    psPrintString( sos, file, size, sevenBits );

    sioOutPrintf( sos, ")\n" );

    if  ( markSize > 0 )
	{
	sioOutPrintf( sos, "  /URI (" );
	utilPsUriLinkDestination( sos, file, size, markName, markSize );
	sioOutPrintf( sos, ")\n" );
	}

    return;
    }

void psSourcePdfmark(		SimpleOutputStream *		sos,
				const DocumentRectangle *	drLink,
				const unsigned char *		fileName,
				int				fileSize,
				const unsigned char *		markName,
				int				markSize )
    {
    sioOutPrintf( sos, "[ /Rect [ %d %d %d %d ]\n",
				drLink->drX0, drLink->drY0,
				drLink->drX1, drLink->drY1 );
    sioOutPrintf( sos, "  /Border [ 0 0 0 ]\n" );

    if  ( fileSize == 0 )
	{
	if  ( ! markName )
	    { XDEB(markName);	}
	else{
	    sioOutPrintf( sos, "  /Dest /" );

	    utilPsEmitDestination( sos, markName, markSize );
	    sioOutPrintf( sos, "\n" );
	    }
	}
    else{
	utilPsFileLinkDestMark( sos,
				fileName, fileSize, markName, markSize );
	}

    sioOutPrintf( sos, "  /Subtype /Link\n" );
    sioOutPrintf( sos, "/ANN pdfmark\n" );

    return;
    }

