/************************************************************************/
/*									*/
/*  Mapping between Document fonts and PostScript fonts.		*/
/*									*/
/************************************************************************/

#   include	"docFontConfig.h"

#   include	<stdlib.h>
#   include	<stdio.h>
#   include	<string.h>
#   include	<math.h>

#   include	<appDebugon.h>

#   include	"utilDocFont.h"
#   include	"utilMatchFont.h"
#   include	"utilOfficeCharset.h"
#   include	"uniLegacyEncoding.h"

/************************************************************************/
/*									*/
/*  Map the RTF style faces to the actual faces of a PostScript font.	*/
/*									*/
/*  6)  Remember what the face numbers of the variants of the font.	*/
/*  7)  If no face number was found for plain, take an arbitrary one.	*/
/*  8)  For all variants without a face number, take the plian one.	*/
/*	(but see 7).							*/
/*									*/
/************************************************************************/

static void utilMatchPsFacesToDocFont(	DocumentFont *		df,
					const PsFontFamily *	aff )
    {
    int				fac;

    /*  6  */
    for ( fac= 0; fac < aff->affFaceCount; fac++ )
	{
	int	b= FONT_IS_BOLD( aff->affFaces[fac] );
	int	s= FONT_IS_SLANTED( aff->affFaces[fac] );

	df->dfPsFontInfo[FACE_INDEX(s,b)]= aff->affFaces[fac];
	}

    if  ( ! df->dfPsFontInfo[0] )
	{
	for ( fac= 0; fac < FONTface_COUNT; fac++ )
	    { 
	    if  ( df->dfPsFontInfo[fac] )
		{ df->dfPsFontInfo[0]= df->dfPsFontInfo[fac]; break; }
	    }
	}

    for ( fac= 0; fac < FONTface_COUNT; fac++ )
	{ 
	if  ( ! df->dfPsFontInfo[fac] )
	    { df->dfPsFontInfo[fac]= df->dfPsFontInfo[0]; }
	}

    return;
    }

/************************************************************************/

# define SHOW_MISSING 0
# if SHOW_MISSING

static int utilMatchFontShowMissing( int n, AfmFontInfo * afi )
    {
    SXDEB(afi->afiFullName,n);
    return 0;
    }

# endif

/************************************************************************/
/*									*/
/*  Decide whether a font can be used to render an office charset.	*/
/*									*/
/*  1)  Arbitrarily pick the first face of the family to check.		*/
/*  2)  If not unicodes are required, match without checking.		*/
/*									*/
/************************************************************************/

static int utilPsFontFamilySupportsUnicodes(
				    const PsFontFamily *	aff,
				    const IndexSet *		required )
    {
    int				rval= 1;
    AfmFontInfo *		afi= aff->affFaces[0]; /* 1 */

    IndexSet			missing;
    int				missingCount;

    utilInitIndexSet( &missing );

    if  ( aff->affFaceCount == 0 || ! afi )
	{ LXDEB(aff->affFaceCount,afi); rval= 0; goto ready;	}

    /*  2  */
    if  ( ! required )
	{ rval= 1; goto ready;	}

    if  ( afi->afiMetricsDeferred )
	{
	if  ( afi->afiResolveMetrics )
	    {
	    if  ( (*afi->afiResolveMetrics)( afi ) )
		{ SLDEB(afi->afiFullName,afi->afiMetricsDeferred);	}
	    }
	else{
	    SLDEB(afi->afiFullName,afi->afiMetricsDeferred);
	    }
	}

    missingCount= utilIndexSetRemoveAll( &missing, required,
						&(afi->afiUnicodesProvided) );
# if SHOW_MISSING
    if  ( missingCount > 0 )
	{
	IndexSet		xx;
	int			n;

	utilInitIndexSet( &xx );

	SLDEB(afi->afiFullName,missingCount);

	n= utilIndexSetIntersect( &xx, required, required );

	appDebug( "For %s %d/%d chars are missing\n",
				afi->afiFullName, missingCount, n );

	utilIndexSetForAll( &missing, utilMatchFontShowMissing, afi );
	}
# endif

    if  ( missingCount < 0 )
	{ LDEB(missingCount); rval= -1; goto ready;	}

    rval= ( missingCount == 0 );

  ready:

    utilCleanIndexSet( &missing );

    return rval;
    }

/************************************************************************/
/*									*/
/*  An half hearted attempt to derive the font style from the family	*/
/*  name.								*/
/*									*/
/************************************************************************/

int utilFontFamilyStyle(	const char *	fontFamilyName )
    {
    int		l= strlen( fontFamilyName );

    while( l > 0 )
	{
	int		past= -1;
	int		i;

	if  ( l == 5 && ! strncmp( fontFamilyName, "Arial", 5 ) )
	    { return DFstyleFSWISS;	}

	if  ( l == 9 && ! strncmp( fontFamilyName, "Helvetica", 9 ) )
	    { return DFstyleFSWISS;	}

	if  ( l == 5 && ! strncmp( fontFamilyName, "Times", 5 ) )
	    { return DFstyleFROMAN;	}

	if  ( l == 8 && ! strncmp( fontFamilyName, "Palatino", 8 ) )
	    { return DFstyleFROMAN;	}

	if  ( l == 6 && ! strncmp( fontFamilyName, "Symbol", 6 ) )
	    { return DFstyleFTECH;	}

	if  ( l == 7 && ! strncmp( fontFamilyName, "Courier", 7 ) )
	    { return DFstyleFMODERN;	}

	if  ( l == 5 && ! strncmp( fontFamilyName, "Ohlfs", 5 ) )
	    { return DFstyleFMODERN;	}

	if  ( l == 9 && ! strncmp( fontFamilyName, "Wingdings", 9 ) )
	    { return DFstyleFTECH;	}
	if  ( l == 8 && ! strncmp( fontFamilyName, "Webdings", 8 ) )
	    { return DFstyleFTECH;	}

	if  ( l == 7 && ! strncmp( fontFamilyName, "Corsiva", 7 ) )
	    { return DFstyleFSCRIPT;	}

	for ( i= 0; i < l- 1; i++ )
	    {
	    if  ( fontFamilyName[i] != ' ' && fontFamilyName[i+ 1] == ' ' )
		{ past= i+ 1;	}
	    }
	if  ( past <= 0 )
	    { break;	}
	l= past;

	for ( i= 0; i < l; i++ )
	    {
	    if  ( fontFamilyName[i] != ' ' )
		{ past= i+ 1;	}
	    }
	l= past;
	}

    return DFstyleFNIL;
    }

/************************************************************************/
/*									*/
/*  Get a font.								*/
/*									*/
/*  1)  Is it an open font? If so return its number.			*/
/*  2)  Claim memory.							*/
/*  3)  Obtain a list of the available fonts.				*/
/*  4)  Try to find an exact match based on the name.			*/
/*  4a) Try the first word.						*/
/*  5)  Try to map 'fnil' to 'Helvetica'.				*/
/*  6)  Try to map 'fswiss' to 'Helvetica'.				*/
/*  7)  Try to map 'froman' to 'Times'.					*/
/*  8)  Chose the first family that has a fixed width font for 'fmodern'*/
/*  9)  Try to map 'fscript' to 'ITC Zapf Chancery'.			*/
/*									*/
/************************************************************************/

static int utilGetPsFamilyByName(
				const PsFontFamily **		pPsf,
				const char *			name,
				const IndexSet *		unicodes,
				const PostScriptFontList *	psfl )
    {
    int				fam;
    const PsFontFamily *	aff;

    if  ( ! name )
	{ return -1;	}

    fam= psPostScriptFontListGetFamilyByName( psfl, name );
    if  ( fam < 0 )
	{ return -1;	}

    aff= psfl->psflFamilies[fam];
    if  ( aff								&&
	  aff->affFontFamilyName					&&
	  ( ! unicodes						||
		utilPsFontFamilySupportsUnicodes( aff, unicodes ) )	)
	{
	*pPsf= aff;
	return 0;
	}

    return -1;
    }

static int utilGetPsFamilyByModifiedName(
				const PsFontFamily **		pPsf,
				const char *			name,
				const IndexSet *		unicodes,
				const PostScriptFontList *	psfl )
    {
    if  ( ! name )
	{ return -1;	}

    if  ( ! utilGetPsFamilyByName( pPsf, name, unicodes, psfl ) )
	{ return 0;	}

    if  ( ! strncmp( name, "Times New Roman", 15 ) )
	{ name= "Times";	}

    if  ( ! strncmp( name, "Courier New", 11 ) )
	{ name= "Courier";	}

    if  ( ! strcmp( name, "Arial" ) )
	{ name= "Helvetica";	}
    if  ( ! strcmp( name, "Arial Narrow" ) )
	{ name= "Helvetica Narrow";	}

    if  ( ! strcmp( name, "Apple Chancery" ) )
	{ name= "ITC Zapf Chancery";	}

    if  ( ! strncmp( name, "Wingdings", 9 ) )
	{ name= "ITC Zapf Dingbats";	}
    if  ( ! strncmp( name, "Webdings", 8 ) )
	{ name= "ITC Zapf Dingbats";	}
    if  ( ! strcmp( name, "ZapfDingbats" ) )
	{ name= "ITC Zapf Dingbats";	}
    if  ( ! strcmp( name, "ZapfChancery" ) )
	{ name= "ITC Zapf Chancery";	}

    if  ( ! strcmp( name, "Lucida Console" ) )
	{ name= "Lucida Sans Typewriter";	}

    if  ( ! strcmp( name, "Bookman Old Style" ) )
	{ name= "ITC Bookman";	}

    if  ( ! strcmp( name, "Century Schoolbook" ) )
	{ name= "New Century Schoolbook";	}

    if  ( ! utilGetPsFamilyByName( pPsf, name, unicodes, psfl ) )
	{ return 0;	}

    if  ( ! strncmp( name, "ITC Zapf Dingbats", 17 ) )
	{ name= "Dingbats";	}

    if  ( ! utilGetPsFamilyByName( pPsf, name, unicodes, psfl ) )
	{ return 0;	}

    return -1;
    }

static int utilGetPsFamilyByStyle(
				const PsFontFamily **		pPsf,
				int				styleInt,
				const IndexSet *		unicodes,
				const PostScriptFontList *	psfl )
    {
    /*  5  */
    if  ( ! styleInt < 0 || styleInt == DFstyleFNIL )
	{
	if  ( ! utilGetPsFamilyByName( pPsf, "Helvetica", unicodes, psfl ) )
	    { return 0;	}
	}

    /*  7  */
    if  ( styleInt == DFstyleFROMAN )
	{
	if  ( ! utilGetPsFamilyByName( pPsf, "Times", unicodes, psfl ) )
	    { return 0;	}
	}

    /*  6  */
    if  ( styleInt == DFstyleFSWISS )
	{
	if  ( ! utilGetPsFamilyByName( pPsf, "Helvetica", unicodes, psfl ) )
	    { return 0;	}
	}

    /*  8  */
    if  ( styleInt == DFstyleFMODERN )
	{
	if  ( ! utilGetPsFamilyByName( pPsf, "Courier", unicodes, psfl ) )
	    { return 0;	}
	}

    /*  9, 10  */
    if  ( styleInt == DFstyleFSCRIPT || styleInt == DFstyleFDECOR )
	{
	if  ( ! utilGetPsFamilyByName( pPsf,
				"ITC Zapf Chancery", unicodes, psfl ) )
	    { return 0;	}
	}


    /*  10  */
    if  ( styleInt == DFstyleFTECH )
	{
	if  ( ! utilGetPsFamilyByName( pPsf, "Symbol", unicodes, psfl ) )
	    { return 0;	}
	}

    return -1;
    }

static int utilGetPsFamilyByStyleFromName(
				const PsFontFamily **		pPsf,
				const char *			name,
				const IndexSet *		unicodes,
				const PostScriptFontList *	psfl )
    {
    int			styleInt= utilFontFamilyStyle( name );

    if  ( styleInt < 0 )
	{ SLDEB(name,styleInt); return -1; }

    if  ( utilGetPsFamilyByStyle( pPsf, styleInt, unicodes, psfl ) )
	{ /*SSDEB(name,styleStr);*/ return -1;	}

    return 0;
    }

/************************************************************************/
/*									*/
/*  Find the PostScript font that best matches a document font.		*/
/*  NOTE that we match the families.					*/
/*									*/
/*  1)  Have a chached result? Ready!					*/
/*  2)  Determine character set.					*/
/*  3)  Find Document family: The name is used to resolve the font	*/
/*	name.								*/
/*  4)  Find the required encoding for the font.			*/
/*  5)  All kind of attempts to find a postscript font.			*/
/*									*/
/************************************************************************/

AfmFontInfo * utilGetPsFontInfoForAttribute(
				DocumentFont *			df,
				const TextAttribute *		ta,
				const PostScriptFontList *	psfl )
    {
    const PsFontFamily *		psf;
    AfmFontInfo *			afi;

    /*  1  */
    afi= df->dfPsFontInfo[FACE_INDEX(ta->taFontIsSlanted,ta->taFontIsBold)];
    if  ( afi )
	{ return afi;	}

    /*  5  */
    if  ( utilGetPsFamilyByName( &psf, df->dfName,
					&(df->dfUnicodesUsed), psfl )	&&
	  utilGetPsFamilyByModifiedName( &psf, df->dfName,
					&(df->dfUnicodesUsed), psfl )	&&
	  utilGetPsFamilyByStyle( &psf, df->dfStyleInt,
					&(df->dfUnicodesUsed), psfl )	&&
	  utilGetPsFamilyByStyleFromName( &psf, df->dfName,
					&(df->dfUnicodesUsed), psfl )	)
	{ return (AfmFontInfo *)0;	}

    utilMatchPsFacesToDocFont( df, psf );

    afi= df->dfPsFontInfo[FACE_INDEX(ta->taFontIsSlanted,ta->taFontIsBold)];
    if  ( ! afi )
	{ XDEB(afi);	}
    /*
    SLLDEB(df->dfName,df->dfCharset,psfl->psflFamilyCount);
    */
    return afi;
    }

/************************************************************************/
/*									*/
/*  Temporarily add the PostScript fonts that we have on the machine to	*/
/*  the font list of the document. As we only save the fonts that are	*/
/*  actually used in the document, the number of fonts in the document	*/
/*  will not grow every time it is opened in a different environment.	*/
/*									*/
/*  Just insert the font as an ANSI font: The exact details only count	*/
/*  once the font has been used and the document is saved.		*/
/*									*/
/************************************************************************/

int utilAddPsFontsToDocList(	DocumentFontList *		dfl,
				const PostScriptFontList *	psfl )
    {
    int				fam;

    for ( fam= 0; fam < psfl->psflFamilyCount; fam++ )
	{
	DocumentFont *		df;
	const PsFontFamily *	aff= psfl->psflFamilies[fam];

	int			styleInt;
	int			pitch= FONTpitchDEFAULT;

	styleInt= utilFontFamilyStyle( aff->affFontFamilyName );
	if  ( aff->affHasFixedWidth )
	    { pitch= FONTpitchFIXED;	}
	if  ( aff->affHasProportionalWidth )
	    { pitch= FONTpitchVARIABLE;	}

	df= utilAddFontToDocList( dfl, aff->affFontFamilyName,
							styleInt, pitch );
	if  ( ! df )
	    { SXDEB(aff->affFontFamilyName,df); return -1;	}

	utilMatchPsFacesToDocFont( df, aff );
	}

    return 0;
    }

typedef struct Base35Family
    {
    const char *	b35fName;
    int			b35fStyle;
    } Base35Family;

# if 0
    /*
    136 font set:
    http://www.adobe.com/products/postscript/pdfs/ps3fonts.pdf
    */

    { "Albertus",		DFstyleFROMAN	},
    { "Antique Olive",		DFstyleFSWISS	},
    { "Apple Chancery",		DFstyleFSCRIPT	},
    { "Arial",			DFstyleFSWISS	},
    { "Bodoni",			DFstyleFROMAN	},
    { "Bodoni Compressed",	DFstyleFROMAN	},
    { "Carta",			DFstyleFSWISS	},
    { "Chicago",		DFstyleFSWISS	},
    { "Clarendon",		DFstyleFROMAN	},
    { "Cooper",			DFstyleFROMAN	},
    { "Copperplate",		DFstyleFSWISS	},
    { "Coronet",		DFstyleFSCRIPT	},
    { "Eurostile",		DFstyleFSWISS	},
    { "Eurostile Extended",	DFstyleFSWISS	},
    { "Geneva",			DFstyleFSWISS	},
    { "Gill Sans",		DFstyleFSWISS	},
    { "Gill Sans Condensed",	DFstyleFSWISS	},
    { "Goudy",			DFstyleFROMAN	},
    { "Hoefler Text",		DFstyleFROMAN	},
    { "Hoefler Ornaments",	DFstyleFTECH	},
    { "Joanna",			DFstyleFROMAN	},
    { "Letter",			DFstyleFSWISS	},
    { "ITC Lubalin Graph",	DFstyleFROMAN	},
    { "Marigold",		DFstyleFSCRIPT	},
    { "Monaco",			DFstyleFSWISS	},
    { "ITC Mona Lisa Recut",	DFstyleFROMAN	},
    { "New York",		DFstyleFROMAN	},
    { "Optima",			DFstyleFSWISS	},
    { "Oxford",			DFstyleFSCRIPT	},
    { "Stempel Garamond",	DFstyleFROMAN	},
    { "Tekton",			DFstyleFSCRIPT	},
    { "Times New Roman",	DFstyleFROMAN	},
    { "Univers",		DFstyleFSWISS	},
    { "Univers Condensed",	DFstyleFSWISS	},
    { "Univers Extended",	DFstyleFSWISS	},
    { "Wingdings",		DFstyleFTECH	},
# endif

static const Base35Family UtilBase35Families[]=
{
    { "Courier",			DFstyleFMODERN	},
    { "Helvetica",			DFstyleFSWISS	},
    { "Helvetica Narrow",		DFstyleFSWISS	},
    { "ITC Avant Garde Gothic",		DFstyleFSWISS	},
    { "ITC Bookman",			DFstyleFROMAN	},
    { "ITC Zapf Chancery",		DFstyleFSCRIPT	},
    { "ITC Zapf Dingbats",		DFstyleFTECH	},
    { "New Century Schoolbook",		DFstyleFROMAN	},
    { "Palatino",			DFstyleFROMAN	},
    { "Symbol",				DFstyleFTECH	},
    { "Times",				DFstyleFROMAN	},
};
static const int UtilBase35FamilyCount=
			    sizeof(UtilBase35Families)/sizeof(Base35Family);

int utilAddBase35FontsToDocList(	DocumentFontList *	dfl )
    {
    int				fam;
    const Base35Family *	b35f= UtilBase35Families;

    for ( fam= 0; fam < UtilBase35FamilyCount; b35f++, fam++ )
	{
	DocumentFont *		df;
	int			pitch= FONTpitchVARIABLE;

	if  ( b35f->b35fStyle == DFstyleFMODERN )
	    { pitch= FONTpitchFIXED;	}

	df= utilAddFontToDocList( dfl, b35f->b35fName,
						b35f->b35fStyle, pitch );
	if  ( ! df )
	    { SXDEB(b35f->b35fName,df); return -1;	}
	}

    return 0;
    }

/************************************************************************/
/*									*/
/*  Return the mappings relating to a font.				*/
/*									*/
/*  1)  Dingbats are a bit from the past that we still have to support.	*/
/*  2)  For fonts without an encoding, assume the same encoding as the	*/
/*	document.							*/
/*  3)  But make an exception for yet another legacy font.		*/
/*									*/
/************************************************************************/

const char * utilGetEncodingName(	const char *	fontName,
					int		charset )
    {
    const OfficeCharset *	oc;
    int				charsetIdx;

    /*  1  */
    if  ( ! strcmp( fontName, "ITC Zapf Dingbats" )	||
	  ! strcmp( fontName, "ZapfDingbats" )		||
	  ! strcmp( fontName, "Dingbats" )		)
	{ return "DINGBATS"; }

    /*  3  */
    if  ( ! strcmp( fontName, "Symbol" ) )
	{ return "SYMBOL"; }

    oc= utilGetOfficeCharsetByCharset( &charsetIdx, charset );
    if  ( ! oc )
	{ return (const char *)0;	}

    return oc->ocEncodingName;
    }

/************************************************************************/
/*									*/
/*  Find the set of encodings that best covers the unicode values in	*/
/*  the document.							*/
/*									*/
/*  Use a naive implementation of the 'Greedy Set Cover' algorithm.	*/
/*									*/
/************************************************************************/

typedef struct FontCharset
    {
    DocumentFont *	fcFont;
    int			fcCharsetIdx;
    } FontCharset;

static int docFontSetUnicodeCharsetIdx(	int		unicode,
					void *		vfc )
    {
    const FontCharset *	fc= (const FontCharset *)vfc;

    if  ( utilIndexMappingPut( &(fc->fcFont->dfUnicodeToCharset),
						unicode, fc->fcCharsetIdx ) )
	{ LLDEB(unicode,fc->fcCharsetIdx); return -1;	}

    return 0;
    }

/************************************************************************/
/*									*/
/*  Find encodings that are needed for a font.				*/
/*									*/
/*  1)  Find the legacy encodings that intersect the collection of	*/
/*	characters that the document uses in this font. Also find the	*/
/*	charset that has the highest number of characters.		*/
/*  2)  Ignore the best character set and always use ANSI as the first	*/
/*	character set to use.						*/
/*  3)  Add the remaining legacy encodings to the set of fonts. The	*/
/*	ones with the highest number of characters in the font have	*/
/*	priority.							*/
/*  4)  Remember the biggest charset that has a particular unicode.	*/
/*	this administration is used to switch while saving the document.*/
/*									*/
/************************************************************************/

int docFontFindLegacyEncodings(	DocumentFont *	df )
    {
    int			rval= 0;

    int			csIdx;

    unsigned char	isUsed[CHARSETidx_COUNT];
    int			hitCount[CHARSETidx_COUNT];
    int			hitsLeft[CHARSETidx_COUNT];
    IndexSet		legacyUsed[CHARSETidx_COUNT];

    IndexSet		isBest;

    int			bestCount;
    int			bestIndex;

    df->dfCharsetIdxUsedCount= 0;

    utilInitIndexSet( &isBest );

    for ( csIdx= 0; csIdx < CHARSETidx_COUNT; csIdx++ )
	{
	utilInitIndexSet( &(legacyUsed[csIdx]) );
	hitCount[csIdx]= 0;
	hitsLeft[csIdx]= 0;
	isUsed[csIdx]= 0;
	df->dfCharsetIdxUsed[csIdx]= -1;
	}

    if  ( ! strcmp( df->dfName, "ITC Zapf Dingbats" )	||
	  ! strcmp( df->dfName, "Symbol" )		)
	{ goto superfluous;	}

    /*  1  */
    bestCount= 0; bestIndex= -1;
    for ( csIdx= 0; csIdx < CHARSETidx_COUNT; csIdx++ )
	{
	const	IndexSet *	legacySet;

	legacySet= utilOfficeCharsetGetCodeSet( csIdx );
	if  ( ! legacySet )
	    { continue;	}

	hitCount[csIdx]= utilIndexSetIntersect( &(legacyUsed[csIdx]),
					&(df->dfUnicodesUsed), legacySet );
	if  ( hitCount[csIdx] < 0 )
	    { LLDEB(csIdx,hitCount[csIdx]); rval= -1; goto ready;	}
	hitsLeft[csIdx]= hitCount[csIdx];

	if  ( bestCount < hitsLeft[csIdx] )
	    {
	    bestCount= hitsLeft[csIdx];
	    bestIndex= csIdx;
	    }
	}

    /*  2  */
    if  ( bestIndex > 0 )
	{
	int ansiCount= utilIndexSetRemoveAll( &isBest, &(legacyUsed[0]),
						    &(legacyUsed[bestIndex]) );
	if  ( ansiCount > 0 )
	    { bestIndex= 0;	}
	}

    /*  3  */
    while( bestIndex >= 0 )
	{
	const IndexSet *	subtract= &(legacyUsed[bestIndex]);

	df->dfCharsetIdxUsed[df->dfCharsetIdxUsedCount]= bestIndex;
	df->dfCharsetIdxUsedCount++;
	isUsed[bestIndex]= 1;

	bestCount= 0; bestIndex= -1;
	for ( csIdx= 0; csIdx < CHARSETidx_COUNT; csIdx++ )
	    {
	    if  ( isUsed[csIdx] || hitsLeft[csIdx] == 0 )
		{ continue;	}

	    hitsLeft[csIdx]= utilIndexSetRemoveAll( &(legacyUsed[csIdx]),
					    &(legacyUsed[csIdx]), subtract );
	    if  ( hitsLeft[csIdx] < 0 )
		{ LLDEB(csIdx,hitsLeft[csIdx]); rval= -1; goto ready;	}

	    if  ( bestCount < hitsLeft[csIdx] )
		{
		bestCount= hitsLeft[csIdx];
		bestIndex= csIdx;
		}
	    }
	}

    /*  4  */
    bestIndex= df->dfCharsetIdxUsedCount- 1;
    while( bestIndex >= 0 )
	{
	FontCharset	fc;

	fc.fcFont= df;
	fc.fcCharsetIdx= df->dfCharsetIdxUsed[bestIndex];

	if  ( utilIndexSetForAll( &(legacyUsed[fc.fcCharsetIdx]),
			    docFontSetUnicodeCharsetIdx, (void *)&fc ) < 0 )
	    { LDEB(fc.fcCharsetIdx); rval= -1; goto ready;	}

	bestIndex--;
	}

  superfluous:

    /*  Nothing found.. Take FONTcharsetDEFAULT */
    if  ( utilIndexSetGetNext( &(df->dfUnicodesUsed), -1 ) >= 0	&&
    	  df->dfCharsetIdxUsedCount == 0			)
	{
	if  ( ! utilGetOfficeCharsetByCharset( &(df->dfCharsetIdxUsed[0]),
							FONTcharsetDEFAULT ) )
	    { LDEB(1);	}

	df->dfCharsetIdxUsedCount++;
	}

  ready:

    utilCleanIndexSet( &isBest );

    for ( csIdx= 0; csIdx < CHARSETidx_COUNT; csIdx++ )
	{ utilCleanIndexSet( &(legacyUsed[csIdx]) );	}

    return rval;
    }
