/************************************************************************/
/*									*/
/*  Get/Move/Set Selections.						*/
/*									*/
/************************************************************************/

#   include	"docBufConfig.h"

#   include	<stdlib.h>
#   include	<string.h>
#   include	<stdio.h>

#   include	<appDebugon.h>

#   include	"docBuf.h"
#   include	"docParaString.h"

/************************************************************************/
/*									*/
/*  Move to the first/last position.					*/
/*									*/
/************************************************************************/

int docFirstPosition(	DocumentPosition *	dp,
			BufferItem *		bi )
    {
    while( bi )
	{
	if  ( bi->biLevel == DOClevPARA )
	    {
	    docSetDocumentPosition( dp, bi, docParaHeadStroff( bi ) );
	    return 0;
	    }

	if  ( bi->biChildCount == 0 )
	    { /*LDEB(bi->biChildCount);*/ return -1;	}

	bi= bi->biChildren[0];
	}

    /*XDEB(bi);*/ return -1;
    }

int docLastPosition(	DocumentPosition *	dp,
			BufferItem *		bi )
    {
    while( bi )
	{
	if  ( bi->biLevel == DOClevPARA )
	    {
	    docSetDocumentPosition( dp, bi, docParaTailStroff( bi ) );
	    return 0;
	    }

	if  ( bi->biChildCount == 0 )
	    { /*LDEB(bi->biChildCount);*/ return -1;	}

	bi= bi->biChildren[bi->biChildCount- 1];
	}

    /*XDEB(bi);*/ return -1;
    }

int docFirstDocumentPosition(	DocumentPosition *	dp,
				BufferDocument *	bd )
    {
    if  ( docFirstPosition( dp, bd->bdBody.eiRoot ) )
	{ return -1;	}

    return 0;
    }

int docLastDocumentPosition(	DocumentPosition *	dp,
				BufferDocument *	bd )
    {
    if  ( docLastPosition( dp, bd->bdBody.eiRoot ) )
	{ return -1;	}

    return 0;
    }

/************************************************************************/
/*									*/
/*  Keep out the read-only field at the Head of certain kinds of	*/
/*  paragraphs.								*/
/*									*/
/************************************************************************/

void docAvoidParaHeadField(	DocumentPosition *	dp,
				int *			pPart,
				const BufferDocument *	bd )
    {
    DocumentField *	dfHead= (DocumentField *)0;
    DocumentSelection	dsInsideHead;
    DocumentSelection	dsAroundHead;
    int			partBegin= -1;
    int			partEnd= -1;

    int	fieldKind= docParaHeadFieldKind( dp->dpBi, bd );
    if  ( fieldKind < 0 )
	{ return;	}

    if  ( docDelimitParaHeadField( &dfHead, &dsInsideHead, &dsAroundHead,
					&partBegin, &partEnd, dp->dpBi, bd ) )
	{ LDEB(1);	}
    else{
	if  ( dp->dpStroff < dsAroundHead.dsTail.dpStroff )
	    { dp->dpStroff=  dsAroundHead.dsTail.dpStroff;	}

	if  ( pPart && *pPart <= partBegin )
	    { *pPart= partBegin+ 1;	}
	}

    return;
    }

/************************************************************************/
/*									*/
/*  Move to the next/previous position.					*/
/*									*/
/************************************************************************/

# define docItemIsMerged( bi ) ( \
    (bi)->biParent && \
    docIsRowItem( (bi)->biParent ) && \
    (bi)->biNumberInParent < (bi)->biParent->biRowCellCount && \
    ( (bi)->biParent->biRowCells[(bi)->biNumberInParent].cpMergedWithAbove || \
    (bi)->biParent->biRowCells[(bi)->biNumberInParent].cpMergedWithLeft ) ) \

static BufferItem * docNextItem(	BufferItem *	bi,
					int		level )
    {
    for (;;)
	{
	if  ( ! bi->biParent )
	    { return (BufferItem *)0;	}

	if  ( bi->biNumberInParent < bi->biParent->biChildCount- 1 )
	    {
	    bi= bi->biParent->biChildren[bi->biNumberInParent+ 1];

	    if  ( docItemIsMerged( bi ) )
		{ continue;	}

	    while( bi->biLevel < level	&&
		   bi->biChildCount > 0	)
		{
		bi= bi->biChildren[0];

		if  ( docItemIsMerged( bi ) )
		    { break; }
		}

	    if  ( docItemIsMerged( bi ) )
		{ continue; }

	    if  ( bi->biLevel == level )
		{ return bi;	}
	    }
	else{ bi= bi->biParent;	}
	}
    }

BufferItem *	docNextParagraph(	BufferItem *	bi )
    { return docNextItem( bi, DOClevPARA );	}

BufferItem *	docNextSection(	BufferItem *	bi )
    { return docNextItem( bi, DOClevSECT );	}

static BufferItem * docPrevItem(	BufferItem *	bi,
					int		level )
    {
    for (;;)
	{
	if  ( ! bi->biParent )
	    { return (BufferItem *)0;	}

	if  ( bi->biNumberInParent > 0 )
	    {
	    bi= bi->biParent->biChildren[bi->biNumberInParent- 1];

	    if  ( docItemIsMerged( bi ) )
		{ continue;	}

	    while( bi->biLevel < level	&&
		   bi->biChildCount > 0	)
		{
		bi= bi->biChildren[bi->biChildCount- 1];

		if  ( docItemIsMerged( bi ) )
		    { break; }
		}

	    if  ( docItemIsMerged( bi ) )
		{ continue; }

	    if  ( bi->biLevel == level )
		{ return bi;	}
	    }
	else{ bi= bi->biParent;	}
	}
    }

BufferItem *	docPrevParagraph(	BufferItem *	bi )
    { return docPrevItem( bi, DOClevPARA );	}

BufferItem *	docPrevSection(		BufferItem *	bi )
    { return docPrevItem( bi, DOClevSECT );	}

/************************************************************************/
/*									*/
/*  Move to the Next/Previous word.					*/
/*									*/
/*  1)  Words start after a particule ending in a space or at the	*/
/*	beginning of a paragraph.					*/
/*									*/
/************************************************************************/

int docNextWord(	DocumentPosition *	dp )
    {
    BufferItem *		paraBi= dp->dpBi;
    int				stroff= dp->dpStroff;

    if  ( stroff == docParaStrlen( paraBi ) )
	{
	paraBi= docNextParagraph( paraBi );
	if  ( ! paraBi )
	    { return 1;	}

	stroff= 0;
	}

    stroff= docParaNextWord( paraBi, stroff );
    docSetDocumentPosition( dp, paraBi, stroff );

    return 0;
    }

int docPrevWord(	DocumentPosition *	dp )
    {
    BufferItem *		paraBi= dp->dpBi;
    int				stroff= dp->dpStroff;

    if  ( stroff == 0 )
	{
	paraBi= docPrevParagraph( paraBi );
	if  ( ! paraBi )
	    { return 1;	}

	stroff= docParaStrlen( paraBi );
	}

    stroff= docParaPrevWord( paraBi, stroff );
    docSetDocumentPosition( dp, paraBi, stroff );

    return 0;
    }


/************************************************************************/
/*									*/
/*  Move one line up.							*/
/*									*/
/*  1)  Go to previous line of the paragraph or to the last line of the	*/
/*	last paragraph before this one that has any lines.		*/
/*  2)  In a clumsy attempt to implement vertical movement in tables,	*/
/*	try to stay in the same column of a table.			*/
/*									*/
/************************************************************************/

int docLineUp(		TextLine **		pTl,
			int *			pLine,
			DocumentPosition *	dp,
			int			line )
    {
    TextLine *		tl;
    BufferItem *	bi= dp->dpBi;

    line--;

    /*  1  */
    while( bi )
	{
	if  ( bi->biParaLineCount == 0 )
	    { LLDEB(docNumberOfParagraph(bi),bi->biParaLineCount);	}

	if  ( bi->biLevel == DOClevPARA		&&
	      line < bi->biParaLineCount	&& /* against crashes */
	      line >= 0				)
	    {
	    tl= bi->biParaLines+ line;

	    dp->dpBi= bi;
	    dp->dpStroff= tl->tlStroff;

	    if  ( pTl )
		{ *pTl= tl;	}
	    if  ( pLine )
		{ *pLine= line;	}
	    return 0;
	    }

	/*  2  */
	if  ( bi->biLevel == DOClevPARA		&&
	      bi->biParaTableNesting > 0	&&
	      bi->biNumberInParent == 0		)
	    {
	    int			col;
	    int			row0;
	    int			row;
	    int			row1;

	    BufferItem *	parentBi;

	    if  ( docDelimitTable( bi, &parentBi, &col, &row0, &row, &row1 ) )
		{ LDEB(1); return -1;	}

	    if  ( row > row0 )
		{
		BufferItem *	rowBi= parentBi->biChildren[row-1];

		if  ( col < rowBi->biChildCount )
		    {
		    BufferItem *	cellBi= rowBi->biChildren[col];

		    bi= cellBi->biChildren[cellBi->biChildCount-1];
		    line= bi->biParaLineCount- 1;
		    continue;
		    }
		}
	    }

	bi= docPrevParagraph( bi );
	if  ( ! bi )
	    { break;	}
	line= bi->biParaLineCount- 1;
	}

    return -1;
    }

/************************************************************************/
/*									*/
/*  Move one line down.							*/
/*									*/
/*  1)  Go to the next line of the paragraph or to the first line of	*/
/*	the first paragraph after this one that has any lines.		*/
/*  2)  In a clumsy attempt to implement vertical movement in tables,	*/
/*	try to stay in the same column of a table.			*/
/*									*/
/************************************************************************/

int docLineDown(	TextLine **		pTl,
			int *			pLine,
			DocumentPosition *	dp,
			int			line )
    {
    TextLine *		tl;
    BufferItem *	bi= dp->dpBi;

    line++;

    /*  1  */
    while( bi )
	{
	if  ( bi->biLevel == DOClevPARA		&&
	      line < bi->biParaLineCount	&&
	      line >= 0				)   /*  against crashes  */
	    {
	    tl= bi->biParaLines+ line;

	    dp->dpBi= bi;
	    dp->dpStroff= tl->tlStroff;

	    if  ( pTl )
		{ *pTl= tl;	}
	    if  ( pLine )
		{ *pLine= line;	}
	    return 0;
	    }

	/*  2  */
	if  ( bi->biLevel == DOClevPARA					&&
	      bi->biParaTableNesting > 0				&&
	      bi->biNumberInParent == bi->biParent->biChildCount- 1	)
	    {
	    int			col;
	    int			row0;
	    int			row;
	    int			row1;

	    BufferItem *	parentBi;

	    if  ( docDelimitTable( bi, &parentBi, &col, &row0, &row, &row1 ) )
		{ LDEB(1); return -1;	}

	    if  ( row < row1 )
		{
		BufferItem *	rowBi= parentBi->biChildren[row+1];

		if  ( col < rowBi->biChildCount )
		    {
		    BufferItem *	cellBi= rowBi->biChildren[col];

		    bi= cellBi->biChildren[0];
		    line= 0;
		    continue;
		    }
		}
	    }

	bi= docNextParagraph( bi );
	line= 0;
	}

    return -1;
    }

/************************************************************************/
/*									*/
/*  Move to the beginning/end of a line					*/
/*									*/
/************************************************************************/

int docBeginOfLine(	DocumentPosition *	dp,
			int			wasAtHead )
    {
    BufferItem *	bi= dp->dpBi;

    int			line;
    TextLine *		tl;

    if  ( wasAtHead )
	{ return 0;	}

    tl= bi->biParaLines;
    for ( line= 0; line < bi->biParaLineCount; tl++, line++ )
	{
	int			stroff= tl->tlStroff;

	/*  1  */
	if  ( stroff+ tl->tlStrlen > dp->dpStroff	||
	      line == bi->biParaLineCount- 1		)
	    {
	    dp->dpBi= bi;
	    dp->dpStroff= stroff;

	    return 0;
	    }
	}

    LDEB(1); return 1;
    }

int docEndOfLine(	DocumentPosition *	dp,
			int			wasAtHead )
    {
    BufferItem *	bi= dp->dpBi;

    int			line;
    TextLine *		tl;

    tl= bi->biParaLines;
    for ( line= 0; line < bi->biParaLineCount; tl++, line++ )
	{
	int			stroff= tl->tlStroff;

	if  ( wasAtHead					&&
	      stroff+ tl->tlStrlen == dp->dpStroff	&&
	      line < bi->biParaLineCount- 1		)
	    { wasAtHead= 0; continue;	}

	if  ( stroff+ tl->tlStrlen >= dp->dpStroff )
	    {
	    dp->dpBi= bi;
	    dp->dpStroff= stroff+ tl->tlStrlen;

	    return 0;
	    }
	}

    LDEB(1); return 1;
    }

/************************************************************************/
/*									*/
/*  Go to the top/bottom of a certain page.				*/
/*									*/
/*  1)  Sections that start on an odd/even page may skip a page.	*/
/*									*/
/************************************************************************/

int docParaGetFirstInColumn(
			DocumentPosition *		dp,
			int *				pLineTop,
			int *				pPartTop,
			BufferItem *			paraBi,
			int				lineUpto,
			int				page,
			int				column )
    {
    int			i;
    const TextLine *	tl= paraBi->biParaLines;

    for ( i= 0; i < lineUpto; tl++, i++ )
	{
	if  ( tl->tlTopPosition.lpPage >= page )
	    {
	    if  ( tl->tlTopPosition.lpPage > page )
		{ break;	}
	    if  ( tl->tlTopPosition.lpColumn >= column )
		{ break;	}
	    }
	}

    if  ( paraBi->biParaLineCount > 0 && i >= paraBi->biParaLineCount )
	{
	tl= paraBi->biParaLines+ paraBi->biParaLineCount- 1;

	if  ( tl->tlFlags & TLflagPAGEBREAK )
	    {
	    BufferItem *	nextBi= docNextParagraph( paraBi );

	    if  ( nextBi->biParaLineCount > 0 )
		{ i= 0; paraBi= nextBi; tl= paraBi->biParaLines;	}
	    }
	}

    if  ( i >= paraBi->biParaLineCount		||
	  tl->tlTopPosition.lpPage != page	||
	  tl->tlTopPosition.lpColumn != column	)
	{
	SDEB(docExternalKindStr(paraBi->biInExternalItem));
	LLDEB(page,column);
	LDEB(paraBi->biBelowPosition.lpPage);
	LDEB(paraBi->biBelowPosition.lpColumn);
	/*docListItem(0,paraBi);*/
	LDEB(docNumberOfParagraph(paraBi));
	LLDEB(i,paraBi->biParaLineCount);
	return -1;
	}

    dp->dpBi= paraBi;
    dp->dpStroff= tl->tlStroff;

    *pLineTop= tl- paraBi->biParaLines;
    *pPartTop= tl->tlFirstParticule;

    return 0;
    }

int docGetFirstInColumnForItem(
			DocumentPosition *		dp,
			int *				pLineTop,
			int *				pPartTop,
			BufferItem *			bi,
			int				page,
			int				column )
    {
    int			i;

    while( bi && bi->biLevel != DOClevPARA )
	{
	/*  1  */
	if  ( bi->biTopPosition.lpPage > page )
	    {
	    /* LLDEB(bi->biTopPosition.lpPage,page); */
	    return 1;
	    }
	if  ( bi->biBelowPosition.lpPage < page )
	    {
	    /* LLDEB(bi->biTopPosition.lpPage,page); */
	    return 1;
	    }

	for ( i= 0; i < bi->biChildCount; i++ )
	    {
	    if  ( bi->biChildren[i]->biBelowPosition.lpPage >= page )
		{
		if  ( bi->biChildren[i]->biBelowPosition.lpPage > page )
		    { break;	}
		if  ( bi->biChildren[i]->biBelowPosition.lpColumn >= column )
		    { break;	}
		}
	    }

	if  ( i >= bi->biChildCount )
	    {
	    /*  NO! is possible e.g. when endnotes continue beyond the 
	        last page with body content.
	    for ( i= 0; i < bi->biChildCount; i++ )
		{ LLDEB(i,bi->biChildren[i]->biBelowPosition.lpPage); }
	    LLLDEB(bi->biTopPosition.lpPage,bi->biBelowPosition.lpPage,page);
	    return -1;
	    */
	    return 1;
	    }

	bi= bi->biChildren[i];
	}

    if  ( ! bi || bi->biLevel != DOClevPARA )
	{ XDEB(bi); return -1;	}

    return docParaGetFirstInColumn( dp, pLineTop, pPartTop,
				    bi, bi->biParaLineCount, page, column );
    }

int docGetTopOfColumn(	DocumentPosition *		dp,
			int *				pLineTop,
			int *				pPartTop,
			BufferDocument *		bd,
			int				page,
			int				column )
    {
    return docGetFirstInColumnForItem( dp, pLineTop, pPartTop,
					    bd->bdBody.eiRoot, page, column );
    }

int docGetLastInColumnForItem(	DocumentPosition *		dp,
				int *				pLineBot,
				int *				pPartBot,
				BufferItem *			bi,
				int				page,
				int				column )
    {
    int			i;
    const TextLine *	tl;

    while( bi && bi->biLevel != DOClevPARA )
	{
	if  ( bi->biTopPosition.lpPage > page )
	    { /*LLDEB(bi->biBelowPosition.lpPage,page);*/ return 1;	}
	if  ( bi->biBelowPosition.lpPage < page )
	    { /*LLDEB(bi->biBelowPosition.lpPage,page);*/ return 1;	}

	for ( i= bi->biChildCount- 1; i >= 0; i-- )
	    {
	    if  ( bi->biChildren[i]->biTopPosition.lpPage <= page )
		{
		if  ( bi->biChildren[i]->biTopPosition.lpPage < page )
		    { break;	}
		if  ( bi->biChildren[i]->biTopPosition.lpColumn <= column )
		    { break;	}
		}
	    }

	if  ( i < 0 )
	    {
	    /*
	    for ( i= 0; i < bi->biChildCount; i++ )
		{ LLDEB(i,bi->biChildren[i]->biTopPosition.lpPage); }
	    LLLDEB(bi->biTopPosition.lpPage,bi->biBelowPosition.lpPage,page);
	    */
	    return 1;
	    }

	bi= bi->biChildren[i];
	}

    if  ( ! bi || bi->biLevel != DOClevPARA )
	{ XDEB(bi); return -1;	}

    tl= bi->biParaLines+ bi->biParaLineCount- 1;
    for ( i= bi->biParaLineCount- 1; i >= 0; tl--, i-- )
	{
	if  ( tl->tlTopPosition.lpPage <= page )
	    {
	    if  ( tl->tlTopPosition.lpPage < page )
		{ break;	}
	    if  ( tl->tlTopPosition.lpColumn <= column )
		{ break;	}
	    }
	}

    if  ( i < 0					||
	  tl->tlTopPosition.lpPage != page	||
	  tl->tlTopPosition.lpColumn != column	)
	{
	SDEB(docExternalKindStr(bi->biInExternalItem));
	LLDEB(page,column);
	LDEB(bi->biBelowPosition.lpPage);
	LDEB(bi->biBelowPosition.lpColumn);
	/*docListItem(0,bi);*/
	LDEB(docNumberOfParagraph(bi));
	LLDEB(i,bi->biParaLineCount);
	return -1;
	}

    dp->dpBi= bi;
    dp->dpStroff= tl->tlStroff+ tl->tlStrlen;

    *pLineBot= tl- bi->biParaLines;
    *pPartBot= tl->tlFirstParticule+ tl->tlParticuleCount- 1;

    return 0;
    }

int docGetBottomOfColumn(
			DocumentPosition *		dp,
			int *				pPartBot,
			BufferDocument *		bd,
			int				page,
			int				column )
    {
    int		lineBot;

    return docGetLastInColumnForItem( dp, &lineBot, pPartBot,
					    bd->bdBody.eiRoot, page, column );
    }

/************************************************************************/
/*									*/
/*  Get buffer positions for a text line.				*/
/*									*/
/************************************************************************/

void docLineSelection(	DocumentSelection *	dsLine,
			int *			pPartLineBegin,
			int *			pPartLineEnd,
			const BufferItem *	bi,
			int			line )
    {
    const TextLine *	tl= bi->biParaLines+ line;

    if  ( bi->biLevel != DOClevPARA )
	{ LLDEB(bi->biLevel,DOClevPARA); return;	}
    if  ( line < 0 || line >= bi->biParaLineCount )
	{ LLDEB(line,bi->biParaLineCount); return;	}

    docInitDocumentSelection( dsLine );

    dsLine->dsHead.dpBi= (BufferItem *)bi;
    dsLine->dsHead.dpStroff= tl->tlStroff;

    dsLine->dsTail.dpBi= (BufferItem *)bi;
    dsLine->dsTail.dpStroff= tl->tlStroff+ tl->tlStrlen;

    dsLine->dsAnchor= dsLine->dsHead;
    dsLine->dsDirection= 1;

    dsLine->dsCol0= dsLine->dsCol1= -1;

    docSetSelectionScope( dsLine, bi );

    *pPartLineBegin= tl->tlFirstParticule;
    *pPartLineEnd= tl->tlFirstParticule+ tl->tlParticuleCount- 1;

    return;
    }

/************************************************************************/
/*									*/
/*  Delimit a 'Word' in the document.					*/
/*									*/
/*  A 'Word' is one of the following:					*/
/*  1)	A continguous stretch of text. It is not interrupted by either	*/
/*	white space or control particules. Any white space after the	*/
/*	word is included in the word.					*/
/*  2)	An object or an image.						*/
/*									*/
/************************************************************************/

void docWordSelection(	DocumentSelection *		dsWord,
			int *				pIsObject,
			const DocumentPosition *	dpAround )
    {
    TextParticule *	tp;
    const int		lastOne= 1;

    BufferItem *	paraBi= dpAround->dpBi;
    int			part;

    if  ( paraBi->biLevel != DOClevPARA )
	{ LLDEB(paraBi->biLevel,DOClevPARA); return;	}

    if  ( docFindParticuleOfPosition( &part, dpAround, lastOne ) )
	{ LDEB(dpAround->dpStroff); return;	}

    if  ( part == paraBi->biParaParticuleCount			&&
	  dpAround->dpStroff == docParaStrlen( dpAround->dpBi )	)
	{ part--;	}
    tp= paraBi->biParaParticules+ part;

    if  ( part < 0 || part >= paraBi->biParaParticuleCount )
	{ LLDEB(part,paraBi->biParaParticuleCount); return;	}

    while( part < paraBi->biParaParticuleCount- 1		&&
	   tp->tpStroff+ tp->tpStrlen <= dpAround->dpStroff	)
	{ tp++; part++;	}

    tp= paraBi->biParaParticules+ part;

    if  ( tp->tpStroff == dpAround->dpStroff	&&
	  part > 0				&&
	  tp[-1].tpKind == DOCkindOBJECT	)
	{
	docSetParaSelection( dsWord, paraBi, 1,
					tp[-1].tpStroff, tp[-1].tpStrlen );
	*pIsObject= 1;
	return;
	}

    if  ( tp->tpKind == DOCkindOBJECT )
	{
	docSetParaSelection( dsWord, paraBi, 1, tp->tpStroff, tp->tpStrlen );
	*pIsObject= 1;
	return;
	}
    else{
	int	partHead;
	int	partTail;
	int	stroffHead;
	int	stroffTail;

	partHead= docParaHeadOfWord( &stroffHead, paraBi, part );
	partTail= docParaTailOfWord( &stroffTail, paraBi, part );

	docSetParaSelection( dsWord, paraBi, 1,
					stroffHead, stroffTail- stroffHead );
	}

    return;
    }

/************************************************************************/
/*									*/
/*  Translate a TableRectangle to a DocumentSelection.			*/
/*									*/
/************************************************************************/

int docTableRectangleSelection(	DocumentSelection *	ds,
				BufferItem **		pSelParentBi,
				BufferDocument *	bd,
				const TableRectangle *	tr )
    {
    DocumentSelection	dsNew;

    BufferItem *	selSectBi;
    BufferItem *	selParentBi;
    BufferItem *	rowBi;
    BufferItem *	cellBi;

    DocumentTree *	ei;
    BufferItem *	bodySectBi;

    docInitDocumentSelection( &dsNew );

    /*******/

    selParentBi= docGetSelectionRoot( &ei, &bodySectBi, bd, ds );
    if  ( ! selParentBi )
	{ XDEB(selParentBi); return -1;	}

    if  ( selParentBi->biLevel == DOClevPARA )
	{ selParentBi= selParentBi->biParent;	}
    if  ( selParentBi->biLevel == DOClevCELL )
	{ selParentBi= selParentBi->biParent;	}
    if  ( selParentBi->biLevel == DOClevROW )
	{ selParentBi= selParentBi->biParent;	}

    selSectBi= docGetSectItem( selParentBi );
    if  ( ! selSectBi )
	{ XDEB(selSectBi); return -1;	}

    /*******/

    if  ( tr->trRow0 < 0					||
	  tr->trRow0 >= selParentBi->biChildCount		)
	{ LLDEB(tr->trRow0,selParentBi->biChildCount); return -1;	}
    rowBi= selParentBi->biChildren[tr->trRow0];

    if  ( tr->trCol0 < 0			||
	  tr->trCol0 >= rowBi->biChildCount	)
	{ LLDEB(tr->trCol0,rowBi->biChildCount); return -1;	}
    cellBi= rowBi->biChildren[tr->trCol0];

    if  ( docFirstPosition( &dsNew.dsHead, cellBi ) )
	{ LDEB(0); return -1;	}

    /*******/

    if  ( tr->trRow1 < 0				||
	  tr->trRow1 >= selParentBi->biChildCount	)
	{ LLDEB(tr->trRow0,selParentBi->biChildCount); return -1;	}
    rowBi= selParentBi->biChildren[tr->trRow1];

    if  ( tr->trCol1 < 0			||
	  tr->trCol1 >= rowBi->biChildCount	)
	{ LLDEB(tr->trCol1,rowBi->biChildCount); return -1;	}
    cellBi= rowBi->biChildren[tr->trCol1];

    if  ( docLastPosition( &dsNew.dsTail, cellBi ) )
	{ LDEB(0); return -1;	}

    /*******/

    dsNew.dsCol0= tr->trCol0;
    dsNew.dsCol1= tr->trCol1;
    dsNew.dsDirection= 1;
    dsNew.dsAnchor= dsNew.dsHead;

    if  ( tr->trRow1 < tr->trRow0					||
	  ( tr->trRow1 == tr->trRow0 && tr->trCol1 < tr->trCol0 )	)
	{
	dsNew.dsDirection= -1;
	dsNew.dsAnchor= dsNew.dsTail;
	}

    docSetSelectionScope( &dsNew, selSectBi );

    *ds= dsNew;
    *pSelParentBi= selParentBi;

    return 0;
    }

/************************************************************************/
/*									*/
/*  Implementation of a 'Select All' menu option.			*/
/*  Extend selection to whole item tree.				*/
/*									*/
/************************************************************************/

int docSelectWholeTree(		DocumentSelection *	ds,
				BufferDocument *	bd )
    {
    BufferItem *		selRootBi= (BufferItem *)0;

    DocumentTree *		ei;
    BufferItem *		bodySectBi;

    if  ( ds->dsHead.dpBi && ds->dsTail.dpBi )
	{ selRootBi= docGetSelectionRoot( &ei, &bodySectBi, bd, ds );	}

    if  ( ! selRootBi )
	{ selRootBi= bd->bdBody.eiRoot;	}

    while( selRootBi && selRootBi->biParent )
	{ selRootBi= selRootBi->biParent; }
    if  ( ! selRootBi )
	{ XDEB(selRootBi); return -1;	}

    if  ( docFirstPosition( &(ds->dsHead), selRootBi ) )
	{ LDEB(1); return -1;	}
    if  ( docLastPosition( &(ds->dsTail), selRootBi ) )
	{ LDEB(1); return -1;	}

    ds->dsAnchor= ds->dsHead;
    ds->dsDirection= 1;

    docSetSelectionScope( ds, ds->dsHead.dpBi );

    return 0;
    }

/************************************************************************/
/*									*/
/*  Change the selection to cover whole sections.			*/
/*									*/
/*  Depending on the 'direction' argument:				*/
/*	<  0:	The previous one.					*/
/*	== 0:	The current one.					*/
/*	>  0:	The next one.						*/
/*									*/
/************************************************************************/

int docSelectWholeSection(	DocumentSelection *	ds,
				BufferDocument *	bd,
				int			direction )
    {
    BufferItem *		selSectBi;

    DocumentTree *		ei;
    BufferItem *		bodySectBi;

    selSectBi= docGetSelectionRoot( &ei, &bodySectBi, bd, ds );
    if  ( ! selSectBi )
	{ XDEB(selSectBi); return -1;	}
    selSectBi= docGetSectItem( selSectBi );
    if  ( ! selSectBi )
	{ XDEB(selSectBi); return -1;	}

    if  ( direction > 0 )
	{
	if  ( docLastPosition( &(ds->dsHead), selSectBi ) )
	    { LDEB(1); return -1;	}
	if  ( docNextPosition( &(ds->dsHead) ) )
	    { return 1;	}

	ds->dsTail= ds->dsHead;

	selSectBi= docGetSelectionRoot( &ei, &bodySectBi, bd, ds );
	if  ( ! selSectBi )
	    { XDEB(selSectBi); return -1;	}
	}

    if  ( direction < 0 )
	{
	if  ( docFirstPosition( &(ds->dsHead), selSectBi ) )
	    { LDEB(1); return -1;	}
	if  ( docPrevPosition( &(ds->dsHead) ) )
	    { return 1;	}

	ds->dsTail= ds->dsHead;

	selSectBi= docGetSelectionRoot( &ei, &bodySectBi, bd, ds );
	if  ( ! selSectBi )
	    { XDEB(selSectBi); return -1;	}
	}

    selSectBi= docGetSectItem( selSectBi );
    if  ( ! selSectBi )
	{ XDEB(selSectBi); return -1;	}

    if  ( docFirstPosition( &(ds->dsHead), selSectBi ) )
	{ LDEB(1); return -1;	}
    if  ( docLastPosition( &(ds->dsTail), selSectBi ) )
	{ LDEB(1); return -1;	}

    if  ( direction >= 0 )
	{ ds->dsAnchor= ds->dsHead;	}
    else{ ds->dsAnchor= ds->dsTail;	}

    if  ( direction >= 0 )
	{ ds->dsDirection=  1;	}
    else{ ds->dsDirection= -1;	}

    docSetSelectionScope( ds, ds->dsHead.dpBi );

    return 0;
    }

/************************************************************************/
/*									*/
/*  Is the selection exactly an object?					*/
/*									*/
/************************************************************************/

int docGetObjectSelection(	const DocumentSelection *	ds,
				const BufferDocument *		bd,
				int *				pPart,
				DocumentPosition *		dpObject,
				InsertedObject **		pIo )
    {
    if  ( ds->dsHead.dpBi			&&
	  ds->dsTail.dpBi == ds->dsHead.dpBi	)
	{
	BufferItem *		paraBi= ds->dsHead.dpBi;
	int			part;
	TextParticule *		tp;

	const int		lastOne= 1;

	if  ( docFindParticuleOfPosition( &part, &(ds->dsHead), lastOne ) )
	    { LDEB(ds->dsHead.dpStroff); return -1;	}

	tp= paraBi->biParaParticules+ part;

	if  ( tp->tpKind == DOCkindOBJECT			&&
	      ds->dsHead.dpStroff == tp->tpStroff		&&
	      ds->dsTail.dpStroff == tp->tpStroff+ tp->tpStrlen	)
	    {
	    *pPart= part;
	    *dpObject= ds->dsHead;
	    *pIo= docGetObject( bd, tp->tpObjectNumber );

	    return 0;
	    }
	}

    return 1;
    }

/************************************************************************/
/*									*/
/*  Find the line number for a certain string particule Number.		*/
/*									*/
/*  NOTE: This does not expect the paragraph to be formatted. We just	*/
/*	fail because there are no lines. We do not crash however.	*/
/*									*/
/************************************************************************/

int docFindLineOfPosition(	int *				pLine,
				const DocumentPosition *	dp,
				int				lastOne )
    {
    const BufferItem *	paraBi= dp->dpBi;
    int			line= 0;

    const TextLine *	tl;

    line= 0; tl= paraBi->biParaLines+ line;
    while( line < paraBi->biParaLineCount			&&
	   tl->tlStroff+ tl->tlStrlen < dp->dpStroff	)
	{ line++; tl++;	}

    if  ( line >= paraBi->biParaLineCount )
	{
	const int checkGeometry= 1;

	LLDEB(line,paraBi->biParaLineCount);
	LLDEB(dp->dpStroff,paraBi->biParaParticuleCount);
	docListItem( 0, paraBi, checkGeometry );
	return -1;
	}

    while( lastOne					&&
	   line < paraBi->biParaLineCount- 1		&&
	   tl->tlStroff+ tl->tlStrlen == dp->dpStroff	)
	{ line++; tl++;	}

    *pLine= line;

    return 0;
    }

/************************************************************************/
/*									*/
/*  Select the whole frame around a position. Make no difficulties if	*/
/*  the position is not in a frame: simply select the neghbourhood that	*/
/*  is not a frame. (Usually the whole cell.)				*/
/*									*/
/************************************************************************/

int docSelectFrameOfPosition(	DocumentSelection *		ds,
				const DocumentPosition *	dp )
    {
    BufferItem *	cellBi;
    int			para0;
    int			para1;
    int			frameNumber;

    DocumentPosition	dpBegin;
    DocumentPosition	dpEnd;

    const int		direction= 1;
    const int		col0= -1;
    const int		col1= -1;

    docInitDocumentPosition( &dpBegin );
    docInitDocumentPosition( &dpEnd );

    if  ( ! dp->dpBi )
	{ XDEB(dp->dpBi); return 1;	}

    frameNumber= dp->dpBi->biParaFrameNumber;
    cellBi= dp->dpBi->biParent;
    para0= para1= dp->dpBi->biNumberInParent;

    while( para0 > 0 )
	{
	if  ( cellBi->biChildren[para0- 1]->biParaFrameNumber != frameNumber )
	    { break;	}
	para0--;
	}

    while( para1 < cellBi->biChildCount- 1 )
	{
	if  ( cellBi->biChildren[para1+ 1]->biParaFrameNumber != frameNumber )
	    { break;	}
	para1++;
	}

    if  ( docFirstPosition( &dpBegin, cellBi->biChildren[para0] ) )
	{ LDEB(1); return 1;	}
    if  ( docLastPosition( &dpEnd, cellBi->biChildren[para1] ) )
	{ LDEB(1); return 1;	}

    docSetRangeSelection( ds, &dpBegin, &dpEnd, direction, col0, col1 );

    return 0;
    }

/************************************************************************/

BufferItem * docGetBodySectBiOfScope(	const SelectionScope *	ss,
					const BufferDocument *	bd )
    {
    BufferItem *		sectBi;
    int				sect;

    if  ( ss->ssInExternalItem == DOCinBODY )
	{ sect= ss->ssSectNr;		}
    else{ sect= ss->ssOwnerSectNr;	}

    sectBi= bd->bdBody.eiRoot->biChildren[sect];

    if  ( sectBi->biLevel != DOClevSECT )
	{ LDEB(sectBi->biLevel); return (BufferItem *)0;	}

    return sectBi;
    }

const BufferItem * docGetBodySectBi(	BufferItem *		bi,
					const BufferDocument *	bd )
    {
    int				sect;

    bi= docGetSectItem( bi );
    if  ( ! bi )
	{ XDEB(bi); return (BufferItem *)0;	}

    if  ( bi->biInExternalItem == DOCinBODY )
	{ sect= bi->biNumberInParent;			}
    else{ sect= bi->biSectSelectionScope.ssOwnerSectNr;	}

    return bd->bdBody.eiRoot->biChildren[sect];
    }

