/************************************************************************/
/*									*/
/*  Save a BufferDocument into an RTF file.				*/
/*									*/
/************************************************************************/

#   include	"docBufConfig.h"

#   include	<stdlib.h>
#   include	<string.h>
#   include	<stdio.h>
#   include	<ctype.h>

#   include	<appDebugon.h>

#   include	"docRtfReadWrite.h"
#   include	"docRtf.h"

/************************************************************************/
/*									*/
/*  Forward declaration..						*/
/*									*/
/************************************************************************/

static int docRtfSaveItem(	RtfWritingContext *		rwc,
				const BufferItem *		bi,
				const DocumentSelection *	ds );

/************************************************************************/

static int docRtfSavePropertiesOfRow(
				RtfWritingContext *		rwc,
				const RowProperties *		rp,
				const DocumentSelection *	ds )
    {
    int			col0= -1;
    int			col1= -1;

    if  ( ds )
	{ col0= ds->dsCol0; col1= ds->dsCol1; 	}

    docRtfSaveRowProperties( rwc, rp, col0, col1 );

    if  ( docCopyRowProperties( &(rwc->rwcRowProperties), rp,
					(const DocumentAttributeMap *)0 ) )
	{ LDEB(1); return -1;	}

    return 0;
    }

/************************************************************************/
/*									*/
/*  Enter/Leave a table.						*/
/*									*/
/************************************************************************/

static int docRtfPushTable(	RtfWritingContext *		rwc,
				const BufferItem *		rowBi,
				const DocumentSelection *	ds )
    {
    PropertyMask	ppChgMask;
    PropertyMask	ppUpdMask;

    docRtfWriteNextLine( rwc );
    docRtfWriteDestinationBegin( rwc, "" );

    rwc->rwcTableNesting++;

    if  ( docRtfPushAttribute( rwc ) )
	{ LDEB(1); return -1;	}

    utilPropMaskClear( &ppChgMask );

    utilPropMaskClear( &ppUpdMask );
    utilPropMaskFill( &ppUpdMask, PPprop_FULL_COUNT );

    if  ( docUpdParaProperties( &ppChgMask,
			    &(rwc->rwcOutsideTableParagraphProperties),
			    &ppUpdMask, &(rwc->rwcParagraphProperties),
			    (const DocumentAttributeMap *)0 ) )
	{ LDEB(1); return -1;	}

    docRtfSavePropertiesOfRow( rwc, &(rowBi->biRowProperties), ds );

    return 0;
    }

int docRtfPopTable(	RtfWritingContext *	rwc )
    {
    PropertyMask	ppChgMask;
    PropertyMask	ppUpdMask;

    docRtfWriteDestinationEnd( rwc );
    docRtfWriteNextLine( rwc );

    rwc->rwcTableNesting--;

    docCleanRowProperties( &(rwc->rwcRowProperties) );
    docInitRowProperties( &(rwc->rwcRowProperties) );

    if  ( docRtfPopAttribute( rwc ) )
	{ LDEB(1); return -1;	}

    utilPropMaskClear( &ppChgMask );

    utilPropMaskClear( &ppUpdMask );
    utilPropMaskFill( &ppUpdMask, PPprop_FULL_COUNT );

    if  ( docUpdParaProperties( &ppChgMask,
			    &(rwc->rwcParagraphProperties), &ppUpdMask,
			    &(rwc->rwcOutsideTableParagraphProperties),
			    (const DocumentAttributeMap *)0 ) )
	{ LDEB(1); return -1;	}

    return 0;
    }

/************************************************************************/
/*									*/
/*  Save an inidividual header, footer, or footnote.			*/
/*									*/
/************************************************************************/

int docRtfSaveExternalItem(		RtfWritingContext *	rwc,
					const char *		tag,
					BufferItem *		sectBi,
					int			evenIfAbsent,
					int			forcePar )
    {
    int				i;

    int				savedTableNesting= rwc->rwcTableNesting;

    if  ( ! sectBi )
	{
	if  ( evenIfAbsent )
	    {
	    docRtfWriteDestinationBegin( rwc, tag );
	    docRtfWriteDestinationEnd( rwc );
	    }

	return 0;
	}

    if  ( sectBi->biLevel != DOClevSECT )
	{ SDEB(docLevelStr(sectBi->biLevel)); return -1;	}

    docRtfWriteDestinationBegin( rwc, tag );
    if  ( docRtfPushAttribute( rwc ) )
	{ LDEB(1); return -1;	}

    docCleanParagraphProperties( &(rwc->rwcParagraphProperties) );
    docInitParagraphProperties( &(rwc->rwcParagraphProperties) );

    rwc->rwcTableNesting= 0;

    docRtfWriteTag( rwc, "\\pard" );
    docRtfWriteSwitchToPlain( rwc );
    docRtfWriteNextLine( rwc );

    for ( i= 0; i < sectBi->biChildCount; i++ )
	{
	if  ( docRtfSaveItem( rwc, sectBi->biChildren[i],
						(DocumentSelection *)0 ) )
	    { LDEB(i); return -1;	}
	}

    /********************************************************************/
    /*  MS-Word does not pick up paragraph properties of headers and	*/
    /*  footers without a \par						*/
    /********************************************************************/
    if  ( forcePar )
	{
	DocumentPosition	dp;

	if  ( ! docLastPosition( &dp, sectBi )		&&
	      dp.dpBi->biParaTableNesting == 0		)
	    { docRtfWriteTag( rwc, "\\par" );	}
	}

    docCleanParagraphProperties( &(rwc->rwcParagraphProperties) );
    docInitParagraphProperties( &(rwc->rwcParagraphProperties) );
    utilInitTextAttribute( &(rwc->rwcTextAttribute) );
    rwc->rwcTextCharset= FONTcharsetANSI;

    while( rwc->rwcTableNesting > 0 )
	{
	if  ( docRtfPopTable( rwc ) )
	    { LDEB(rwc->rwcTableNesting);	}
	}

    docRtfWriteDestinationEnd( rwc );
    docRtfWriteNextLine( rwc );

    if  ( docRtfPopAttribute( rwc ) )
	{ LDEB(1); return -1;	}
    rwc->rwcTableNesting= savedTableNesting;

    return 0;
    }

/************************************************************************/
/*									*/
/*  Save a Row level item.						*/
/*									*/
/************************************************************************/

static int docRtfSaveRowItem(	RtfWritingContext *		rwc,
				const BufferItem *		rowBi,
				const DocumentSelection *	ds )
    {
    int			col;
    int			tableNesting= docTableNesting( rowBi );

    const char *	cellTag= "\\cell";
    const char *	rowTag= "\\row";

    int			confinedToCell;

    if  ( tableNesting > 1 )
	{
	cellTag= "\\nestcell";
	rowTag= "\\nestrow";
	}

    confinedToCell= 0;
    if  ( ds && ds->dsHead.dpBi->biParent == ds->dsTail.dpBi->biParent )
	{ confinedToCell= 1;	}

    if  ( tableNesting == 1 && ! confinedToCell )
	{
	if  ( ! docEqualRows( &(rowBi->biRowProperties),
						&(rwc->rwcRowProperties) ) )
	    {
	    if  ( rwc->rwcTableNesting == 1 && docRtfPopTable( rwc ) )
		{ LDEB(1);	}
	    }

	if  ( rwc->rwcTableNesting == 0		&&
	      docRtfPushTable( rwc, rowBi, ds )	)
	    { LDEB(1);	}
	}

    for ( col= 0; col < rowBi->biChildCount; col++ )
	{
	int		child;
	BufferItem *	cellBi= rowBi->biChildren[col];

	if  ( ds && docCompareItemPositions( cellBi, ds->dsHead.dpBi ) < 0 )
	    { continue;	}

	if  ( ds && docCompareItemPositions( cellBi, ds->dsTail.dpBi ) > 0 )
	    { continue;	}

	if  ( ! ds						||
	      ( ( ds->dsCol0 < 0 || col >= ds->dsCol0 )	&&
		( ds->dsCol1 < 0 || col <= ds->dsCol1 )	)	)
	    {
	    int		afterPara= 0;

	    for ( child= 0; child < cellBi->biChildCount; child++ )
		{
		BufferItem *	childBi= cellBi->biChildren[child];

		if  ( docRtfSaveItem( rwc, childBi, ds ) )
		    { LLDEB(col,child); return -1;	}

		afterPara= ( childBi->biLevel == DOClevPARA );
		}

	    if  ( ! afterPara )
		{ docRtfSaveParaTableNesting( rwc, tableNesting );	}

	    docRtfWriteTag( rwc, cellTag );
	    if  ( col != rowBi->biChildCount- 1 )
		{ docRtfWriteNextLine( rwc );	}
	    }
	}

    if  ( tableNesting > 1 )
	{
	int			nip= rowBi->biNumberInParent;
	const BufferItem *	cellBi= rowBi->biParent;

	docRtfWriteNextLine( rwc );
	docRtfWriteDestinationBegin( rwc, "\\*\\nesttableprops" );

	docRtfSavePropertiesOfRow( rwc, &(rowBi->biRowProperties), ds );

	docRtfWriteTag( rwc, rowTag );
	docRtfWriteNextLine( rwc );

	docRtfWriteDestinationEnd( rwc );
	docRtfWriteNextLine( rwc );

	if  ( nip == cellBi->biChildCount- 1				||
	      ( nip < cellBi->biChildCount- 1			&&
	        cellBi->biChildren[nip+1]->biLevel != DOClevROW	)	)
	    {
	    docRtfSavePropertiesOfRow( rwc,
			    &(cellBi->biParent->biRowProperties), ds );
	    }
	}
    else{
	docRtfWriteTag( rwc, rowTag );
	docRtfWriteNextLine( rwc );
	}

    return 0;
    }

static int docSaveSectionProperties(	RtfWritingContext *		rwc,
					const DocumentProperties *	dp,
					const DocumentSelection *	ds,
					const SectionProperties *	sp )
    {
    SectionProperties		spDef;
    PropertyMask		updMask;

    docInitSectionProperties( &spDef );

    utilPropMaskClear( &updMask );
    utilPropMaskFill( &updMask, DGprop_COUNT );
    PROPmaskUNSET( &updMask, DGpropHEADER_POSITION );
    PROPmaskUNSET( &updMask, DGpropFOOTER_POSITION );

    utilUpdDocumentGeometry( &(spDef.spDocumentGeometry),
				    &(dp->dpGeometry), &updMask, &updMask );

    utilPropMaskClear( &updMask );
    utilPropMaskFill( &updMask, SPprop_COUNT );

    docSectPropertyDifference( &updMask, &spDef, sp, &updMask );

    docCleanSectionProperties( &spDef );
    if  ( ds )
	{ PROPmaskUNSET( &updMask, SPpropTITLEPG );	}

    docRtfWriteNextLine( rwc );
    docRtfWriteTag( rwc, "\\sectd" );

    docRtfSaveSectionProperties( rwc, &updMask, sp );

    return 0;
    }

/************************************************************************/
/*									*/
/*  Save a buffer item, hierarchically descending to the leaves of the	*/
/*  document tree.							*/
/*									*/
/************************************************************************/

static int docRtfSaveGroupItem(	RtfWritingContext *		rwc,
				const BufferItem *		bi,
				const DocumentSelection *	ds )
    {
    int				i;

    for ( i= 0; i < bi->biChildCount; i++ )
	{
	if  ( docRtfSaveItem( rwc, bi->biChildren[i], ds ) )
	    { LDEB(i); return -1;	}
	}

    return 0;
    }

static int docRtfSaveItem(	RtfWritingContext *		rwc,
				const BufferItem *		bi,
				const DocumentSelection *	ds )
    {
    const DocumentProperties *	dp= &(rwc->rwcBd->bdProperties);

    if  ( ds )
	{
	if  ( docCompareItemPositions( bi, ds->dsHead.dpBi ) < 0	)
	    { return 0;	}

	if  ( docCompareItemPositions( bi, ds->dsTail.dpBi ) > 0	)
	    { return 0;	}
	}

    switch( bi->biLevel )
	{
	case DOClevBODY:
	case DOClevCELL:
	    if  ( docRtfSaveGroupItem( rwc, bi, ds ) )
		{ LDEB(1); return -1;	}
	    break;

	case DOClevSECT:

	    if  ( docSaveSectionProperties( rwc, dp, ds,
						&(bi->biSectProperties) ) )
		{ LDEB(1); return -1;	}

	    if  ( ! ds							&&
		  bi->biInExternalItem == DOCinBODY			&&
		  docRtfSaveSectHeadersFooters( rwc, bi ) )
		{ LDEB(1); return -1;	}

	    if  ( docRtfSaveGroupItem( rwc, bi, ds ) )
		{ LDEB(1); return -1;	}

	    while( rwc->rwcTableNesting > 0 )
		{
		if  ( docRtfPopTable( rwc ) )
		    { LDEB(rwc->rwcTableNesting);	}
		}

	    if  ( bi->biParent						&&
		  bi->biNumberInParent < bi->biParent->biChildCount- 1	)
		{ docRtfWriteTag( rwc, "\\sect" );	}

	    docRtfWriteNextLine( rwc );
	    break;

	case DOClevROW:
	    if  ( docIsRowItem( bi ) )
		{
		if  ( docRtfSaveRowItem( rwc, bi, ds ) )
		    { LDEB(1); return -1;	}
		}
	    else{
		if  ( rwc->rwcTableNesting > 0 && docRtfPopTable( rwc ) )
		    { LDEB(rwc->rwcTableNesting);	}

		if  ( docRtfSaveGroupItem( rwc, bi, ds ) )
		    { LDEB(1); return -1;	}
		}
	    break;

	case DOClevPARA:
	    if  ( docRtfSaveParaItem( rwc, bi, ds ) )
		{ LDEB(1); return -1;	}
	    break;

	default:
	    LDEB(bi->biLevel); return -1;
	}

    return 0;
    }

int docRtfSaveDocument(		SimpleOutputStream *		sos,
				BufferDocument *		bd,
				const DocumentSelection *	ds,
				int				flags )
    {
    int				rval= 0;
    const BufferItem *		bi= bd->bdBody.eiRoot;

    RtfWritingContext		rwc;

    docRtfInitWritingContext( &rwc );

    if  ( ds )
	{
	DocumentTree *	ei;
	BufferItem *	bodySectBi;

	bi= docGetSelectionRoot( &ei, &bodySectBi, bd, ds );
	if  ( ! bi )
	    { XDEB(bi); rval= -1; goto ready;	}
	}

    rwc.rwcBd= bd;
    rwc.rwcSosOut= sos;
    rwc.rwcSaveFlags= flags;

    if  ( docRtfWriteBuildFontAdmin( &rwc ) )
	{ LDEB(1); rval= -1; goto ready;	}

    docRtfWriteDestinationBegin( &rwc, "\\rtf1\\ansi" );
    docRtfWriteNextLine( &rwc );

    if  ( docRtfSaveDocumentProperties( &rwc, &(bd->bdProperties) ) )
	{ LDEB(1); rval= -1; goto ready;	}

    if  ( docRtfSaveDocNotesSeparators( &rwc, bd ) )
	{ LDEB(1); rval= -1; goto ready;	}

    if  ( docRtfSaveItem( &rwc, bi, ds ) )
	{ LDEB(bi->biLevel); rval= -1; goto ready; }

    while( rwc.rwcTableNesting > 0 )
	{
	if  ( docRtfPopTable( &rwc ) )
	    { LDEB(rwc.rwcTableNesting);	}
	}

    docRtfWriteDestinationEnd( &rwc );
    docRtfWriteNextLine( &rwc );

  ready:
    docRtfCleanWritingContext( &rwc );

    return rval;
    }

