/************************************************************************/
/*									*/
/*  Read the various document tables of an RTF text file into a		*/
/*  BufferDocument.							*/
/*									*/
/************************************************************************/

#   include	"docBufConfig.h"

#   include	<stdlib.h>
#   include	<string.h>
#   include	<stdio.h>
#   include	<ctype.h>

#   include	<appDebugon.h>

#   include	<appUnit.h>
#   include	"docRtf.h"

/************************************************************************/
/*									*/
/*  Consume the 'stylesheet' of a document				*/
/*									*/
/************************************************************************/

static int docRtfStyleName(	RtfReadingContext *	rrc,
				const char *		name,
				int			len )
    {
    RtfReadingState *		rrs= rrc->rrcState;
    int				mindTable= 1;
    BufferDocument *		bd= rrc->rrcBd;

    DocumentStyle *		ds;

    int				size;
    const int			removeSemicolon= 1;

    if  ( docRtfSaveDocEncodedText( rrc, name, len ) )
	{ LDEB(len); return -1;	}

    ds= &(rrc->rrcDocumentStyle);
    switch( ds->dsLevel )
	{
	case DOClevSPAN:
	    break;

	case 0: case -1:
	    LDEB(rrc->rrcDocumentStyle.dsLevel);
	    ds->dsStyleNumber= 0;
	    ds->dsLevel= DOClevPARA;
	    break;

	case DOClevPARA:
	    ds->dsStyleNumber= rrs->rrsParagraphProperties.ppStyle;
	    break;

	case DOClevROW:
	    /* NO! They have trowd and the sheet owns the style
	    ds->dsStyleNumber= rrc->rrcRowProperties.rpRowStyle;
	    */
	    break;

	case DOClevSECT:
	    ds->dsStyleNumber= rrc->rrcSectionProperties.spStyle;
	    break;

	default:
	    LDEB(rrc->rrcDocumentStyle.dsLevel); return -1;
	}

    ds= docInsertStyle( &(rrc->rrcBd->bdStyleSheet),
					rrc->rrcDocumentStyle.dsStyleNumber );
    if  ( ! ds )
	{ XDEB(ds); return -1;	}

    {
    int sav= ds->dsStyleNumber;

    if  ( docCopyStyle( ds, &(rrc->rrcDocumentStyle) ) )
	{ LDEB(1); return -1;	}

    ds->dsStyleNumber= sav;
    }

    if  ( docRtfStoreSavedText( &(ds->dsName), &size, rrc, removeSemicolon ) )
	{ LDEB(len); return -1;	}

    docCopySectionProperties( &(ds->dsSectionProperties),
					    &(rrc->rrcSectionProperties) );

    if  ( docRtfSetParaProperties( &(ds->dsParagraphProperties), bd,
							    mindTable, rrs ) )
	{ LDEB(1);	}

    if  ( rrs->rrsTextShadingChanged )
	{
	docRtfRefreshTextShading( rrc, rrs );
	PROPmaskADD( &(rrc->rrcDocumentStyle.dsTextMask), TApropSHADING );
	}

    ds->dsTextAttribute= rrs->rrsTextAttribute;

    docCleanDocumentStyle( &(rrc->rrcDocumentStyle) );
    docInitDocumentStyle( &(rrc->rrcDocumentStyle) );

    docRtfResetParagraphProperties( rrs );
    docRtfResetTextAttribute( rrs, bd );

    return 0;
    }

static int docRtfStyleGroup(		const RtfControlWord *	rcw,
					int			arg,
					RtfReadingContext *	rrc )
    {
    RtfReadingState *	rrs= rrc->rrcState;

    switch( rcw->rcwID )
	{
	case DOClevSPAN:
	    rrc->rrcDocumentStyle.dsStyleNumber= arg;
	    rrc->rrcDocumentStyle.dsStyleNumber= arg;
	    rrc->rrcDocumentStyle.dsLevel= DOClevSPAN;
	    break;

	case DOClevSECT:
	    rrc->rrcSectionProperties.spStyle= arg;
	    rrc->rrcDocumentStyle.dsStyleNumber= arg;
	    rrc->rrcDocumentStyle.dsLevel= DOClevSECT;
	    break;

	case DOClevROW:
	    rrc->rrcRowProperties.rpRowStyle= arg;
	    rrc->rrcDocumentStyle.dsStyleNumber= arg;
	    rrc->rrcDocumentStyle.dsLevel= DOClevROW;
	    break;

	case DOClevPARA:
	    rrs->rrsParagraphProperties.ppStyle= arg;
	    rrc->rrcDocumentStyle.dsStyleNumber= arg;
	    rrc->rrcDocumentStyle.dsLevel= DOClevPARA;
	    break;

	default:
	    LDEB(rcw->rcwID);
	}

    if  ( docRtfReadGroup( rcw, 0, 0, rrc,
		    (RtfControlWord *)0, docRtfStyleName, (RtfCommitGroup)0 ) )
	{ SLDEB(rcw->rcwWord,arg); return -1;	}

    rrc->rrcDocumentStyle.dsLevel= DOClevANY;

    return 0;
    }

/************************************************************************/
/*									*/
/*  Inside a stylesheet.. These tags are handled differently.		*/
/*									*/
/************************************************************************/

static RtfControlWord	docRtfStylesheetGroups[]=
    {
	RTF_DEST_XX( "cs",	DOClevSPAN,	docRtfStyleGroup ),
	RTF_DEST_XX( "ds",	DOClevSECT,	docRtfStyleGroup ),
	RTF_DEST_XX( "ts",	DOClevROW,	docRtfStyleGroup ),
	RTF_DEST_XX( "tsrowd",	DOClevROW,	docRtfStyleGroup ),
	RTF_DEST_XX( "s",	DOClevPARA,	docRtfStyleGroup ),

	{ (char *)0, 0, 0 }
    };

/************************************************************************/
/*									*/
/*  Consume a group introduced by a control word that is not a		*/
/*  destination.							*/
/*									*/
/************************************************************************/

static int docRtfReadWordGroup(	RtfReadingContext *	rrc,
				int			gotArg,
				int			arg,
				const char *		controlWord,
				const RtfControlWord *	groupWords,
				RtfAddTextParticule	addParticule )
    {
    const RtfControlWord *	rcw;

    rcw= docRtfFindPropertyWord( controlWord );
    if  ( rcw )
	{
	if  ( docRtfReadGroupX( rcw, rcw, gotArg, arg, rrc, groupWords,
					addParticule, (RtfCommitGroup)0 ) )
	    { SDEB(rcw->rcwWord); return -1;	}
	}
    else{
	if  ( rrc->rrcComplainUnknown )
	    { LSDEB(rrc->rrcCurrentLine,controlWord);	}

	if  ( docRtfReadUnknownGroup( rrc ) )
	    { LSDEB(rrc->rrcCurrentLine,controlWord); return -1;	}
	}

    return 0;
    }

/************************************************************************/
/*									*/
/*  Consume the stylesheet in a document.				*/
/*									*/
/*  Because <styledef> is optional in the stylesheet. (For the first	*/
/*  style at least.) The normal mechanism of the parser does not work	*/
/*  for the stylesheet. Do things by hand.				*/
/*									*/
/************************************************************************/

int docRtfStylesheet(		const RtfControlWord *	rcw,
				int			arg,
				RtfReadingContext *	rrc )
    {
    RtfReadingState *		rrs= rrc->rrcState;
    int				res;

    char			controlWord[TEDszRTFCONTROL+1];
    int				gotArg;
    int				c;

    docCleanDocumentStyle( &(rrc->rrcDocumentStyle) );
    docInitDocumentStyle( &(rrc->rrcDocumentStyle) );

    res= docRtfFindControl( rrc, &c, controlWord, &gotArg, &arg );
    if  ( res < 0 )
	{ LDEB(res); return -1;	}

    rrc->rrcDocumentStyle.dsLevel= DOClevPARA;
    rrs->rrsParagraphProperties.ppStyle= 0;

    for (;;)
	{
	rrs= rrc->rrcState;

	switch( res )
	    {
	    case RTFfiCLOSE:
		break;

	    case RTFfiCTRLGROUP:
		rcw= docRtfFindWord( controlWord, docRtfStylesheetGroups );
		if  ( ! rcw )
		    {
		    if  ( docRtfReadWordGroup( rrc, gotArg, arg, controlWord,
					(RtfControlWord *)0, docRtfStyleName ) )
			{ SDEB(controlWord); return -1;	}
		    }
		else{
		  groupFound:
		    res= docRtfApplyControlWord( rcw, gotArg, arg, rrc );
		    if  ( res < 0 )
			{ LDEB(res); SDEB(controlWord); return -1;	}
		    }

		res= docRtfFindControl( rrc, &c, controlWord, &gotArg, &arg );
		if  ( res < 0 )
		    { LDEB(res); return -1;	}
		continue;

	    case RTFfiSTARGROUP:
		rcw= docRtfFindWord( controlWord, docRtfStylesheetGroups );
		if  ( rcw )
		    { goto groupFound; }

		rrc->rrcInIgnoredGroup++;

		if  ( docRtfReadUnknownGroup( rrc ) )
		    { LDEB(1); rrc->rrcInIgnoredGroup--; return -1;	}

		rrc->rrcInIgnoredGroup--;

		res= docRtfFindControl( rrc, &c, controlWord, &gotArg, &arg );
		if  ( res < 0 )
		    { LDEB(res); return -1;	}
		continue;

	    case RTFfiTEXTGROUP:
		{
		const RtfControlWord *	rcwApplyFirst= (RtfControlWord *)0;
		int			argApplyFirst= -1;
		int			gotArgApplyFirst= 0;

		if  ( docRtfReadGroupX( (const RtfControlWord *)0,
			rcwApplyFirst, gotArgApplyFirst, argApplyFirst,
			rrc, docRtfStylesheetGroups,
			docRtfStyleName, (RtfCommitGroup)0 ) )
		    { LDEB(1); return -1;	}

		res= docRtfFindControl( rrc, &c, controlWord, &gotArg, &arg );
		if  ( res < 0 )
		    { LDEB(res); return -1;	}
		}
		continue;

	    default:
		LDEB(res); return -1;
	    }

	break;
	}

    return 0;
    }

/************************************************************************/
/*									*/
/*  Write a Style Sheet.						*/
/*									*/
/************************************************************************/

void docRtfWriteStyleSheet(	RtfWritingContext *		rwc,
				const DocumentStyleSheet *	dss )
    {
    int				i;
    const DocumentStyle *	ds;

    docRtfWriteDestinationBegin( rwc, "\\stylesheet" );
    docRtfWriteNextLine( rwc );

    ds= dss->dssStyles;
    for ( i= 0; i < dss->dssStyleCount; ds++, i++ )
	{
	if  ( ds->dsStyleNumber < 0 )
	    { continue;	}

	switch( ds->dsLevel )
	    {
	    case DOClevSPAN:
		docRtfWriteArgDestinationBegin( rwc, "\\*\\cs",
							ds->dsStyleNumber );
		docRtfSaveTextAttribute( rwc,
				&(ds->dsTextMask), &(ds->dsTextAttribute) );
		break;

	    case DOClevPARA:
		if  ( ds->dsStyleNumber > 0 )
		    {
		    docRtfWriteArgDestinationBegin( rwc, "\\s",
							ds->dsStyleNumber );
		    }
		else{
		    docRtfWriteDestinationBegin( rwc, "" );
		    }

		docRtfSaveParagraphProperties( rwc, &(ds->dsParaMask),
						&(ds->dsParagraphProperties) );
		docRtfSaveTextAttribute( rwc,
				&(ds->dsTextMask), &(ds->dsTextAttribute) );
		break;

	    case DOClevSECT:
		docRtfWriteArgDestinationBegin( rwc, "\\ds",
							ds->dsStyleNumber );

		docRtfSaveSectionProperties( rwc,
			    &(ds->dsSectMask), &(ds->dsSectionProperties) );
		docRtfSaveParagraphProperties( rwc, &(ds->dsParaMask),
					    &(ds->dsParagraphProperties) );
		docRtfSaveTextAttribute( rwc,
			    &(ds->dsTextMask), &(ds->dsTextAttribute) );
		break;

	    case DOClevROW:
		docRtfWriteArgDestinationBegin( rwc, "\\*\\ts",
							ds->dsStyleNumber );

	    default:
		LDEB(ds->dsLevel); continue;
	    }

	if  ( ds->dsAdditive )
	    { docRtfWriteTag( rwc, "\\additive" ); }
	if  ( ds->dsBasedOn >= 0 )
	    { docRtfWriteArgTag( rwc, "\\sbasedon", ds->dsBasedOn ); }
	if  ( ds->dsNext >= 0 )
	    { docRtfWriteArgTag( rwc, "\\snext", ds->dsNext ); }
	if  ( ds->dsAutoupd )
	    { docRtfWriteTag( rwc, "\\sautoupd" ); }
	if  ( ds->dsHidden )
	    { docRtfWriteTag( rwc, "\\shidden" ); }
	if  ( ds->dsLocked )
	    { docRtfWriteTag( rwc, "\\slocked" ); }
	if  ( ds->dsSemiHidden )
	    { docRtfWriteTag( rwc, "\\ssemihidden" ); }
	if  ( ds->dsPersonal )
	    { docRtfWriteTag( rwc, "\\spersonal" ); }
	if  ( ds->dsLink >= 0 )
	    { docRtfWriteTag( rwc, "\\slink" ); }

	docRtfWriteDocEncodedString( rwc, ds->dsName, strlen( ds->dsName ) );
	docRtfWriteSemicolon( rwc );

	docRtfWriteDestinationEnd( rwc );
	docRtfWriteNextLine( rwc );
	}

    docRtfWriteDestinationEnd( rwc );
    docRtfWriteNextLine( rwc );

    return;
    }

/************************************************************************/
/*									*/
/*  Handle a style property when reading RTF.				*/
/*									*/
/************************************************************************/

int docRtfRememberStyleProperty(	const RtfControlWord *	rcw,
					int			arg,
					RtfReadingContext *	rrc )
    {
    DocumentStyle *	ds= &(rrc->rrcDocumentStyle);

    switch( rcw->rcwID )
	{
	/* \s \cs \ds tags but \s and \ds handeled with the para/sect style */
	case DSpropSTYLE_NUMBER:
	    ds->dsStyleNumber= arg;
	    /* Exceptional trick */
	    ds->dsLevel= rcw->rcwEnumValue;
	    break;

	case DSpropADDITIVE:
	    ds->dsAdditive= arg != 0;
	    break;
	case DSpropBASED_ON:
	    ds->dsBasedOn= arg;
	    break;
	case DSpropNEXT:
	    ds->dsNext= arg;
	    break;
	case DSpropLINK:
	    ds->dsLink= arg;
	    break;
	case DSpropAUTO_UPD:
	    ds->dsAutoupd= arg != 0;
	    break;
	case DSpropHIDDEN:
	    ds->dsHidden= arg != 0;
	    break;
	case DSpropLOCKED:
	    ds->dsLocked= arg != 0;
	    break;
	case DSpropSEMI_HIDDEN:
	    ds->dsSemiHidden= arg != 0;
	    break;
	case DSpropPERSONAL:
	    ds->dsPersonal= arg != 0;
	    break;

	case DSpropIGNORED:
	    break;

	default:
	    SDEB(rcw->rcwWord);
	    return -1;
	}

    return 0;
    }
