/************************************************************************/
/*									*/
/*  Read footnotes, headers, footers, textboxes.			*/
/*									*/
/************************************************************************/

#   include	"docBufConfig.h"

#   include	<stdlib.h>
#   include	<string.h>
#   include	<stdio.h>
#   include	<ctype.h>

#   include	<appDebugon.h>

#   include	"docRtf.h"

/************************************************************************/
/*									*/
/*  Consume Headers, Footers, Notes &c: Separate item trees that are.	*/
/*  embedded in the document.						*/
/*									*/
/*  1)  Save the current position.					*/
/*  2)  Make the header/note &c.					*/
/*  3)  Consume its contents.						*/
/*  4)  Make sure that no bookmarks protrude beyond the end of the	*/
/*	external item.							*/
/*  5)  If the external item happens to be empty, insert an empty	*/
/*	paragraph to avoid problems later on.				*/
/*  6)  Restore the current position.					*/
/*									*/
/************************************************************************/

int docRtfReadExternalItem(	const RtfControlWord *	rcw,
				DocumentTree *		ei,
				int *			pExtItKind,
				RtfReadingContext *	rrc,
				int			ignoreEmpty,
				const SelectionScope *	ss )
    {
    int				rval= 0;
    RtfReadingState		internRrs;

    DocumentTree *		savedEi;
    BufferItem *		savedBi;
    int				savedLevel;
    SelectionScope		savedSelectionScope;
    struct RtfFieldStackLevel *	savedFieldStackLevel;
    int				savedLastFieldNumber;
    RowProperties		savedRowProperties;

    /*  1  */
    savedEi= rrc->rrcEi;
    savedBi= rrc->rrcBi;
    savedLevel= rrc->rrcLevel;
    savedSelectionScope= rrc->rrcSelectionScope;
    savedFieldStackLevel= rrc->rrcFieldStack;
    savedLastFieldNumber= rrc->rrcLastFieldNumber;
    savedRowProperties= rrc->rrcRowProperties;

    docInitRowProperties( &(rrc->rrcRowProperties) );

    docRtfPushReadingState( rrc, &internRrs );

    docRtfResetParagraphProperties( &internRrs );
    docRtfResetTextAttribute( &internRrs, rrc->rrcBd );

    if  ( ! savedBi )
	{ XDEB(savedBi); rval= -1; goto ready;	}

    /*  2  */
    if  ( ! ei->eiRoot							&&
	  docMakeExternalItem( rrc->rrcBd, ei,
				    ss, &(rrc->rrcSectionProperties) )	)
	{ LDEB(1); rval= -1; goto ready;	}

    /*  3  */
    rrc->rrcEi= ei;
    rrc->rrcBi= ei->eiRoot;
    rrc->rrcLevel= DOClevSECT;
    rrc->rrcSelectionScope= *ss;
    rrc->rrcFieldStack= (struct RtfFieldStackLevel *)0;
    rrc->rrcLastFieldNumber= -1;

    if  ( docRtfReadGroup( rcw, 0, 0, rrc,
			    docRtfDocumentGroups,
			    docRtfTextParticule, docRtfFinishCurrentItem ) )
	{ LDEB(1); rval= -1;	}

    /*  5  */
    {
    DocumentPosition	dp;

    if  ( docLastPosition( &dp, ei->eiRoot ) )
	{
	if  ( ignoreEmpty )
	    {
	    docCleanExternalItem( rrc->rrcBd, ei );
	    docInitExternalItem( ei );
	    }
	else{
	    const int	textAttributeNumber= 0;

	    if  ( ! docInsertEmptyParagraph( rrc->rrcBd, ei->eiRoot,
						textAttributeNumber ) )
		{ LDEB(textAttributeNumber);	}
	    }
	}
    else{
	/********************************************************/
	/*  Delete empty paragraph caused by final \par		*/
	/********************************************************/
	if  ( docParaStrlen( dp.dpBi ) ==  0		&&
	      dp.dpBi->biParaTableNesting == 0		&&
	      docNumberOfParagraph( dp.dpBi ) > 1	&&
	      dp.dpBi->biParaParticuleCount == 1	)
	    {
	    BufferItem *	bi= dp.dpBi;

	    while( bi->biParent && bi->biParent->biChildCount == 1 )
		{ bi= bi->biParent;	}

	    docDeleteItem( rrc->rrcBd, ei, bi );
	    }
	}
    }

    /*  4  */
    if  ( docRtfPopScopeFromFieldStack( rrc ) )
	{ LDEB(1); rval= -1; goto ready;	}

    *pExtItKind= rrc->rrcSelectionScope.ssInExternalItem;

  ready:
    /*  6  */

    docCleanRowProperties( &(rrc->rrcRowProperties) );
    rrc->rrcRowProperties= savedRowProperties;

    rrc->rrcLastFieldNumber= savedLastFieldNumber;
    rrc->rrcFieldStack= savedFieldStackLevel;
    rrc->rrcSelectionScope= savedSelectionScope;
    rrc->rrcLevel= savedLevel;
    rrc->rrcBi= savedBi;
    rrc->rrcEi= savedEi;

    docRtfPopReadingState( rrc );

    return rval;
    }

int docRtfReadExtIt(		const RtfControlWord *	rcw,
				int			arg,
				RtfReadingContext *	rrc )
    {
    BufferItem *	sectBi;
    BufferDocument *	bd= rrc->rrcBd;
    DocumentTree *	ei;

    SelectionScope	ss;
    int			extItKind;
    int			ignoreEmpty= 0;

    docInitSelectionScope( &ss );
    ss.ssSectNr= 0;

    switch( rcw->rcwID )
	{
	case DOCinFIRST_HEADER:
	case DOCinLEFT_HEADER:
	case DOCinRIGHT_HEADER:

	case DOCinFIRST_FOOTER:
	case DOCinLEFT_FOOTER:
	case DOCinRIGHT_FOOTER:

	    sectBi= docRtfGetSectItem( rrc );
	    if  ( ! sectBi )
		{ XDEB(sectBi); return -1;	}

	    ei= docSectionHeaderFooter( sectBi, rcw->rcwID );
	    if  ( ! ei )
		{ LXDEB(rcw->rcwID,ei); return -1;	}

	    ss.ssOwnerSectNr= sectBi->biNumberInParent;
	    ignoreEmpty= 1;
	    break;

	case DOCinFTNSEP:
	case DOCinFTNSEPC:
	case DOCinFTNCN:
	case DOCinAFTNSEP:
	case DOCinAFTNSEPC:
	case DOCinAFTNCN:

	    if  ( ! bd )
		{ XDEB(bd); return -1;	}

	    ei= docDocumentNoteSeparator( bd, rcw->rcwID );
	    if  ( ! ei )
		{ LXDEB(rcw->rcwID,ei); return -1;	}

	    ignoreEmpty= 1;
	    break;

	default:
	    SDEB(rcw->rcwWord); return -1;
	}

    if  ( ei->eiRoot )
	{
	docCleanExternalItem( bd, ei );
	docInitExternalItem( ei );
	ei->eiRoot= (BufferItem *)0;
	}

    ss.ssInExternalItem= rcw->rcwID;
    ss.ssOwnerNumber= -1;

    if  ( docRtfReadExternalItem( rcw, ei, &extItKind, rrc, ignoreEmpty, &ss ) )
	{ SDEB(rcw->rcwWord); return -1;	}

    return 0;
    }

/************************************************************************/

static int docExtractFixedTextNote(	DocumentNote *		dn,
					BufferDocument *	bd,
					BufferItem *		ownerBi,
					int			fixedStroff,
					int			fixedStrlen,
					const char *		fieldinst,
					int			fieldsize )

    {
    DocumentSelection	dsField;
    BufferItem *	noteBi;
    const int		part0= 0;

    if  ( utilSetMemoryBuffer( &(dn->dnNoteProperties.npFixedText),
		    docParaString( ownerBi,  fixedStroff ), fixedStrlen ) )
	{ LDEB(fixedStrlen);	}

    if  ( docFirstPosition( &(dsField.dsHead),
					dn->dnDocumentTree.eiRoot ) )
	{ LDEB(1); return -1;	}
    noteBi= dsField.dsHead.dpBi;
    docSetDocumentPosition( &(dsField.dsTail), noteBi, fixedStrlen );

    if  ( docParaStrlen( noteBi ) >= fixedStrlen			&&
	  ! memcmp( docParaString( noteBi, 0 ),
	    docParaString( ownerBi, fixedStroff ), fixedStrlen )	)
	{
	int			part1;
	TextParticule *		tp1;
	DocumentField *		df;

	if  ( docFindParticuleOfPosition( &part1, &(dsField.dsTail), 0 ) )
	    { LDEB(fixedStroff); return -1; }
	tp1= noteBi->biParaParticules+ part1;

	if  ( tp1->tpStroff+ tp1->tpStrlen > fixedStrlen )
	    {
	    if  ( docSplitTextParticule( &tp1, (TextParticule **)0,
					    noteBi, part1, fixedStrlen ) )
		{ LDEB(part1); return -1;	}
	    }

	part1++;

	df= docMakeField( bd, &(dn->dnDocumentTree), &dsField,
		    part0, part1, tp1->tpTextAttrNr, tp1->tpTextAttrNr );
	if  ( ! df )
	    { XDEB(df); return -1;	}

	df->dfKind= DOCfkCHFTN;
	if  ( docSetFieldInst( df, fieldinst, fieldsize ) )
	    { LDEB(1);	}
	}

    return 0;
    }

/************************************************************************/
/*									*/
/*  Consume a footnote.							*/
/*									*/
/************************************************************************/

int docRtfReadFootnote(		const RtfControlWord *	rcw,
				int			arg,
				RtfReadingContext *	rrc )
    {
    int			rval= 0;
    BufferDocument *	bd= rrc->rrcBd;

    DocumentField *	dfNote;
    DocumentNote *	dn;
    int			noteIndex;

    BufferItem *	ownerBi;
    BufferItem *	sectBi;
    SelectionScope	ss;

    int			externalItemKind;
    int			autoNumber= 0;
    const int		ignoreEmpty= 0;

    int			fixedStroff= 0;
    int			fixedStrlen= 0;

    const char *	fieldinst= " -CHFTN ";
    int			fieldsize= 8;

    docInitSelectionScope( &ss );

    ownerBi= docRtfGetParaItem( rrc );
    if  ( ! ownerBi )
	{ XDEB(ownerBi); rval= -1; goto ready;	}
    sectBi= docGetSectItem( ownerBi );
    if  ( ! sectBi )
	{ XDEB(sectBi); rval= -1; goto ready;	}

    if  ( rrc->rrcAfterNoteref )
	{ autoNumber= 1;	}
    else{
	const char *		fieldRslt= (const char *)0;

	DocumentPosition	dp;
	int			part0;

	dfNote= docRtfSpecialField( DOCfkCHFTN,
				    fieldinst, fieldsize, fieldRslt, rrc );
	if  ( ! dfNote )
	    { SDEB(rcw->rcwWord); rval= -1; goto ready; }

	docSetDocumentPosition( &dp, ownerBi,
					dfNote->dfHeadPosition.epStroff );
	if  ( docFindParticuleOfPosition( &part0, &dp, 0 ) )
	    { LDEB(dfNote->dfHeadPosition.epStroff); rval= -1; goto ready; }

	if  ( ownerBi->biParaParticules[part0   ].tpKind == DOCkindSPAN	&&
	      ownerBi->biParaParticules[part0+ 1].tpKind == DOCkindFIELDSTART )
	    {
	    fixedStroff= ownerBi->biParaParticules[part0   ].tpStroff;
	    fixedStrlen= ownerBi->biParaParticules[part0   ].tpStrlen;

	    dfNote->dfHeadPosition.epStroff= fixedStroff;

	    ownerBi->biParaParticules[part0+ 1].tpKind= DOCkindSPAN;
	    ownerBi->biParaParticules[part0+ 1].tpObjectNumber= -1;
	    ownerBi->biParaParticules[part0+ 1].tpStroff= fixedStroff;
	    ownerBi->biParaParticules[part0+ 1].tpStrlen= fixedStrlen;

	    ownerBi->biParaParticules[part0   ].tpKind= DOCkindFIELDSTART;
	    ownerBi->biParaParticules[part0   ].tpObjectNumber=
						    dfNote->dfFieldNumber;
	    ownerBi->biParaParticules[part0   ].tpStrlen= 0;
	    }

	autoNumber= 0;
	}

    dfNote= docGetFieldByNumber( &(bd->bdFieldList), rrc->rrcLastFieldNumber );
    if  ( ! dfNote )
	{ LPDEB(rrc->rrcLastFieldNumber,dfNote); return -1;	}

    noteIndex= docInsertNote( &dn, bd, dfNote, autoNumber );
    if  ( noteIndex < 0 )
	{ LDEB(noteIndex); rval= -1; goto ready;	}

    rrc->rrcAfterNoteref= 0;
    rrc->rrcAtParaHead= 0;

    ss.ssInExternalItem= DOCinFOOTNOTE;
    ss.ssSectNr= 0;
    ss.ssOwnerSectNr= sectBi->biNumberInParent;
    ss.ssOwnerNumber= dfNote->dfFieldNumber;

    if  ( docRtfReadExternalItem( rcw, &(dn->dnDocumentTree), &externalItemKind,
						rrc, ignoreEmpty, &ss ) )
	{ SDEB(rcw->rcwWord); rval= -1; goto ready;	}

    dn->dnNoteProperties.npAutoNumber= autoNumber;
    dn->dnNoteProperties.npExternalItemKind= externalItemKind;
    docSetExternalItemKind( dn->dnDocumentTree.eiRoot, externalItemKind );

    if  ( ! autoNumber							&&
	  fixedStrlen > 0						&&
	  docExtractFixedTextNote( dn, bd, ownerBi,
					fixedStroff, fixedStrlen,
					fieldinst, fieldsize )		)
	{ LDEB(1); rval= -1; goto ready;	}

  ready:

    return rval;
    }

