/************************************************************************/
/*									*/
/*  Manage item hierarchy from the rtf parser.				*/
/*									*/
/************************************************************************/

#   include	"docBufConfig.h"

#   include	<stdlib.h>
#   include	<string.h>
#   include	<stdio.h>
#   include	<ctype.h>
#   include	<charnames.h>

#   include	<appDebugon.h>

#   include	<appUnit.h>

#   include	"docRtf.h"
#   include	"docEdit.h"

/************************************************************************/
/*									*/
/*  Apply the row properties that have been collected to a row.		*/
/*									*/
/************************************************************************/

static int docRtfSetRowProperties(	RtfReadingContext *	rrc,
					BufferItem *		rowBi )
    {
    PropertyMask		rpDoneMask;

    rrc->rrcRowProperties.rpShadingNumber= docItemShadingNumber(
				&(rrc->rrcBd->bdItemShadingList),
				&(rrc->rrcRowShading) );

    rrc->rrcRowProperties.rpFrameNumber= docFramePropertiesNumber(
				&(rrc->rrcBd->bdFramePropertyList),
				&(rrc->rrcRowFrameProperties) );

    utilPropMaskClear( &rpDoneMask );

    if  ( docChangeRowProperties( &rpDoneMask, rowBi,
					    &(rrc->rrcRowPropertyMask),
					    &(rrc->rrcRowProperties) ) )
	{ LDEB(1); return -1;	}

    return 0;
    }

/************************************************************************/
/*									*/
/*  Adjust the level in the hierarchy for docRtfReadGroup().		*/
/*									*/
/*  1)  Empty paragraphs get one empty particule to have a height.	*/
/*									*/
/************************************************************************/

static int docRtfCheckParagraph(	RtfReadingContext *	rrc,
					BufferItem *		paraBi )
    {
    int			addEmptyParticule= 0;

    if  ( paraBi->biParaParticuleCount == 0 )
	{ addEmptyParticule= 1;	}
    else{
	const TextParticule *	tp;

	tp= paraBi->biParaParticules+ paraBi->biParaParticuleCount- 1;

	if  ( tp->tpKind == DOCkindLINEBREAK	||
	      tp->tpKind == DOCkindPAGEBREAK	||
	      tp->tpKind == DOCkindCOLUMNBREAK	)
	    { addEmptyParticule= 1;	}

	if  ( tp->tpKind == DOCkindFIELDEND )
	    {
	    const DocumentFieldList *	dfl= &(rrc->rrcBd->bdFieldList);
	    const DocumentField *	df;

	    df= docGetFieldByNumber( dfl, tp->tpObjectNumber );
	    if  ( df->dfKind == DOCfkCHFTN )
		{ addEmptyParticule= 1;	}
	    }
	}

    if  ( addEmptyParticule )
	{
	RtfReadingState *	rrs= rrc->rrcState;

	const int		part= paraBi->biParaParticuleCount;
	const int		stroff= docParaStrlen( paraBi );
	const int		count= 0;

	TextAttribute		ta;
	int			textAttrNr;

	docPlainTextAttribute( &ta, rrc->rrcBd );

	if  ( rrs )
	    {
	    if  ( rrs->rrsTextShadingChanged )
		{ docRtfRefreshTextShading( rrc, rrs );	}

	    ta= rrs->rrsTextAttribute;
	    }

	textAttrNr= docTextAttributeNumber( rrc->rrcBd, &ta );
	if  ( textAttrNr < 0 )
	    { LDEB(textAttrNr); return -1;	}

	if  ( ! docInsertTextParticule( paraBi, part,
				    stroff, count, DOCkindSPAN, textAttrNr ) )
	    { LDEB(1); return -1;	}

	rrc->rrcAfterNoteref= 0;
	rrc->rrcAtParaHead= 0;
	}

    return 0;
    }

/************************************************************************/
/*									*/
/*  Finish a section in the document.					*/
/*									*/
/************************************************************************/

static int docRtfCloseSect(	RtfReadingContext *	rrc,
				BufferItem *		sectBi )
    {
    int		changed= 0;

    docRenumberNotes( &changed, rrc->rrcBd );

    if  ( sectBi->biInExternalItem == DOCinBODY	)
	{
	PropertyMask		spDoneMask;
	PropertyMask		spUpdMask;

	utilPropMaskClear( &spDoneMask );

	utilPropMaskClear( &spUpdMask );
	utilPropMaskFill( &spUpdMask, SPprop_COUNT );

	if  ( docUpdSectProperties( &spDoneMask, &(sectBi->biSectProperties),
			    &spUpdMask, &(rrc->rrcSectionProperties) ) )
	    { LDEB(1); return -1;	}
	}

    return 0;
    }

/************************************************************************/
/*									*/
/*  Finish a row in the document.					*/
/*									*/
/*  In this order: (1) can make that (2) is needed.			*/
/*  1)  To avoid crashes, make sure that the row does not have more	*/
/*	children than columns: Split it until this is the case.		*/
/*  2)  To avoid crashes, make sure that the row does not have more	*/
/*	columns than children: Remove the surplus.			*/
/*									*/
/************************************************************************/

static int docRtfCloseRow(	RtfReadingContext *	rrc,
				BufferItem *		rowBi )
    {
    if  ( rowBi->biRowForTable )
	{
	if  ( docRtfSetRowProperties( rrc, rowBi ) )
	    { LDEB(1); return -1;	}
	}
    else{
	docCleanRowProperties( &(rowBi->biRowProperties) );
	docInitRowProperties( &(rowBi->biRowProperties) );
	}

    if  ( docIsRowItem( rowBi )				&&
	  rowBi->biRowCellCount != rowBi->biChildCount	)
	{
	LLDEB(rowBi->biRowCellCount,rowBi->biChildCount);
	LDEB(rrc->rrcRowProperties.rpCellCount);
	docListItem(0,rowBi,0);

	/*  1  */
	while( rowBi->biRowCellCount < rowBi->biChildCount )
	    {
	    BufferItem *	insBi;
	    BufferItem *	aftBi;

	    LLDEB(rowBi->biRowCellCount,rowBi->biChildCount);

	    if  ( docSplitGroupItem( rrc->rrcBd, &insBi, &aftBi,
				    rowBi,
				    rowBi->biRowCellCount, DOClevROW ) )
		{ LDEB(1); return -1;	}

	    rrc->rrcBi= rowBi= aftBi;
	    }

	/*  2  */
	if  ( rowBi->biRowCellCount > rowBi->biChildCount )
	    {
	    docDeleteColumnsFromRow( &(rowBi->biRowProperties),
				rowBi->biChildCount,
				rowBi->biRowCellCount- rowBi->biChildCount );
	    }
	}

    return 0;
    }

/************************************************************************/
/*									*/
/*  Close a cell.							*/
/*									*/
/************************************************************************/

static int docRtfCloseCell(	RtfReadingContext *	rrc,
				BufferItem *		cellBi )
    {
    if  ( cellBi->biChildCount == 0 )
	{ LDEB(cellBi->biChildCount); return 0;	}

    return 0;
    }

/************************************************************************/

static void docRtfSetForRow(	BufferItem *	bi )
    {
    bi= docGetRowLevelItem( bi );
    while( bi )
	{
	if  ( ! bi->biRowForTable )
	    { bi->biRowForTable= 1;	}

	bi= docGetRowLevelItem( bi->biParent );
	}
    }

/************************************************************************/
/*									*/
/*  Close a paragraph in the document.					*/
/*									*/
/*  1)  Give up on syntactically incorrect rtf files (more '}' than	*/
/*	'{') in the hope that is we avoid the crash, the user can	*/
/*	decide what to do.						*/
/*									*/
/************************************************************************/

static int docRtfCloseParagraph(	RtfReadingContext *	rrc,
					BufferItem *		paraBi )
    {
    RtfReadingState *		rrs= rrc->rrcState;
    BufferDocument *		bd= rrc->rrcBd;
    const DocumentProperties *	dp= &(bd->bdProperties);

    int				paraNr;
    int				textAttributeNumber;
    const int			mindTable= 1;

    ListNumberTreeNode *	root;

    /* 1  */
    if  ( ! rrs )
	{ return 0;	}

    if  ( docRtfCheckParagraph( rrc, paraBi ) )
	{ LDEB(1); return -1;	}

    textAttributeNumber= paraBi->biParaParticules->tpTextAttrNr;

    paraNr= docNumberOfParagraph( paraBi );

    if  ( docRtfPopParaFromFieldStack( rrc, paraNr ) )
	{ LDEB(1); return -1;	}

    if  ( docRtfSetParaProperties( &(paraBi->biParaProperties),
							bd, mindTable, rrs ) )
	{ LDEB(1); return -1;	}

    rrs->rrsParagraphBreakOverride= -1;

    if  ( paraBi->biParaListOverride > 0 )
	{
	DocumentField *			dfHead= (DocumentField *)0;
	DocumentSelection		dsInsideHead;
	DocumentSelection		dsAroundHead;
	int				partBegin= -1;
	int				partEnd= -1;

	const ListOverrideTable *	lot;

	lot= &(dp->dpListOverrideTable);

	if  ( paraBi->biParaListOverride >= lot->lotOverrideCount )
	    {
	    LLDEB(paraBi->biParaListOverride,lot->lotOverrideCount);
	    return -1;
	    }

	if  ( docDelimitParaHeadField( &dfHead, &dsInsideHead, &dsAroundHead,
					&partBegin, &partEnd, paraBi, bd ) )
	    {
	    if  ( docInsertParaHeadField( &dfHead, &dsInsideHead, &dsAroundHead,
			&partBegin, &partEnd, paraBi, bd, rrc->rrcEi,
			DOCfkLISTTEXT, textAttributeNumber ) )
		{
		LDEB(paraBi->biParaListOverride); return -1;
		}
	    }

	root= docGetListNumberTree( &(rrc->rrcEi->eiListNumberTrees),
						paraBi->biParaListOverride );
	if  ( ! root )
	    { LXDEB(paraBi->biParaListOverride,root); return -1;	}

	if  ( docListNumberTreeInsertParagraph( root,
				    paraBi->biParaListLevel, paraNr ) )
	    { LDEB(paraNr);	}
	}

    if  ( paraBi->biParaOutlineLevel < PPoutlineBODYTEXT )
	{
	if  ( docListNumberTreeInsertParagraph(
				    &(rrc->rrcEi->eiOutlineTree),
				    paraBi->biParaOutlineLevel, paraNr ) )
	    { LDEB(paraNr);	}
	}

    if  ( paraBi->biParaTableNesting > 0 )
	{ docRtfSetForRow( paraBi );	}
    else{
	BufferItem *	rowBi= docGetRowLevelItem( paraBi );

	if  ( rowBi->biRowForTable )
	    {
	    if  ( rowBi->biChildCount == 1 )
		{
		rowBi->biRowForTable= 0;
		}
	    else{
		LDEB(rowBi->biChildCount);
		LLDEB(paraBi->biParaTableNesting,rowBi->biRowForTable);
		}
	    }
	}

    return 0;
    }

/************************************************************************/
/*									*/
/*  Close an item.							*/
/*									*/
/************************************************************************/

static int docRtfCloseItem(	RtfReadingContext *	rrc,
				BufferItem *		bi )
    {
    switch( bi->biLevel )
	{
	case DOClevSECT:
	    if  ( docRtfCloseSect( rrc, bi ) )
		{ LDEB(1); return -1;	}
	    break;

	case DOClevROW:
	    if  ( docRtfCloseRow( rrc, bi ) )
		{ LDEB(1); return -1;	}
	    break;

	case DOClevCELL:
	    if  ( docRtfCloseCell( rrc, bi ) )
		{ LDEB(1); return -1;	}
	    break;

	case DOClevPARA:
	    if  ( docRtfCloseParagraph( rrc, bi ) )
		{ LDEB(1); return -1;	}
	    break;

	default:
	    break;
	}

    return 0;
    }

/************************************************************************/
/*									*/
/*  Pop an item from the 'stack' of items that is built by the parser.	*/
/*									*/
/************************************************************************/

static int docRtfPopItem(		RtfReadingContext *	rrc,
					BufferItem *		bi )
    {
    while( rrc->rrcBi )
	{
	if  ( docRtfCloseItem( rrc, rrc->rrcBi ) )
	    { LDEB(1); return -1;	}

	if  ( rrc->rrcBi == bi )
	    {
	    rrc->rrcBi= rrc->rrcBi->biParent;
	    rrc->rrcLevel= rrc->rrcBi->biLevel;
	    return 0;
	    }

	rrc->rrcBi= rrc->rrcBi->biParent;
	}

    XDEB(rrc->rrcBi); return -1;
    }

/************************************************************************/
/*									*/
/*  Handle tags that relate to the document hierarchy.			*/
/*									*/
/************************************************************************/

int docRtfHierarchy(	const RtfControlWord *	rcw,
			int			arg,
			RtfReadingContext *	rrc )
    {
    BufferItem *	levelBi= (BufferItem *)0;
    int			paraInCell;

    switch( rcw->rcwID )
	{
	case DOClevSECT:
	    /*  1  */
	    if  ( rrc->rrcBi )
		{ levelBi= docGetSectItem( rrc->rrcBi );	}
	    if  ( ! levelBi )
		{
		if  ( ! docRtfGetParaItem( rrc ) )
		    { SDEB(rcw->rcwWord); return -1; }
		}
	    if  ( ! rrc->rrcBi )
		{ SXDEB(rcw->rcwWord,rrc->rrcBi); return -1;	}

	    levelBi= docGetSectItem( rrc->rrcBi );
	    if  ( ! levelBi )
		{ SXDEB(rcw->rcwWord,levelBi); return -1;	}
	    if  ( docRtfPopItem( rrc, levelBi ) )
		{ SDEB(rcw->rcwWord); return -1;	}

	    break;

	case DOClevROW:
	case DOClevNESTROW:
	    /*  1  */
	    if  ( rrc->rrcBi )
		{ levelBi= docGetRowLevelItem( rrc->rrcBi );	}
	    /* ignore stray \row tags rather than fail */
	    if  ( ! levelBi )
		{ LSDEB(rrc->rrcCurrentLine,rcw->rcwWord); return 0;	}
	    if  ( docRtfPopItem( rrc, levelBi ) )
		{ SDEB(rcw->rcwWord); return -1;	}
	    break;

	case DOClevCELL:
	case DOClevNESTCELL:
	    /*  1  */
	    if  ( rrc->rrcBi )
		{ levelBi= docGetCellItem( rrc->rrcBi );	}

	    paraInCell= 1;
	    if ( rcw->rcwID == DOClevCELL		&&
		 rrc->rrcBi->biLevel == DOClevCELL	)
		{
		int	cc= rrc->rrcBi->biChildCount;

		if  ( cc > 0						  &&
		      rrc->rrcBi->biChildren[cc-1]->biLevel != DOClevPARA )
		      { paraInCell= 0;	}
		}

	    if  ( ! levelBi						||
		  ( rrc->rrcBi->biLevel <= DOClevCELL && paraInCell )	)
		{
		BufferItem *	paraBi= docRtfGetParaItem( rrc );

		if  ( ! paraBi )
		    { SDEB(rcw->rcwWord); return -1; }

		/* cope with: {\intbl etc\par}\cell */
		if  ( paraBi->biParaTableNesting == 0 )
		    {
		    docSetParaTableNesting( paraBi );
		    docRtfSetForRow( paraBi );
		    if  ( rrc->rrcState )
			{
			rrc->rrcState->rrsParagraphProperties.ppTableNesting= 
						    paraBi->biParaTableNesting;
			}
		    }
		}
	    if  ( ! rrc->rrcBi )
		{ SXDEB(rcw->rcwWord,rrc->rrcBi); return -1;	}

	    /* MS-Word does this: The first clue about a table is \cell */
	    if  ( rrc->rrcBi->biLevel == DOClevPARA	&&
		  rrc->rrcBi->biParaTableNesting == 0	)
		{
		/*LDEB(rrc->rrcBi->biParaTableNesting);*/
		rrc->rrcBi->biParaTableNesting= 1;
		docRtfSetForRow( rrc->rrcBi );
		if  ( rrc->rrcState )
		    {
		    rrc->rrcState->rrsParagraphProperties.ppTableNesting= 
						rrc->rrcBi->biParaTableNesting;
		    }
		}

	    levelBi= docGetCellItem( rrc->rrcBi );
	    if  ( ! levelBi )
		{ SXDEB(rcw->rcwWord,levelBi); return -1;	}
	    if  ( docRtfPopItem( rrc, levelBi ) )
		{ SDEB(rcw->rcwWord); return -1;	}
	    break;

	case DOClevPARA:
	    levelBi= docRtfGetParaItem( rrc );
	    if  ( ! levelBi )
		{ SDEB(rcw->rcwWord); return -1; }

	    if  ( docRtfPopItem( rrc, levelBi ) )
		{ SDEB(rcw->rcwWord); return -1;	}
	    break;

	case DOClevNONESTTABLES:
	    /* should be a destination */
	    break;

	default:
	    SLDEB(rcw->rcwWord,rcw->rcwID); break;
	}

    return 0;
    }

/************************************************************************/

static int docRtfStartParagraph(	RtfReadingContext *	rrc )
    {
    BufferItem *		paraBi;
    RtfReadingState *		rrs= rrc->rrcState;
    const int			mindTable= 1;

    paraBi= docInsertItem( rrc->rrcBd, rrc->rrcBi, -1, DOClevPARA );
    if  ( ! paraBi )
	{ SXDEB(docLevelStr(rrc->rrcLevel),paraBi); return -1; }
    rrc->rrcBi= paraBi;

    if  ( docRtfSetParaProperties( &(paraBi->biParaProperties),
						rrc->rrcBd, mindTable, rrs ) )
	{ LDEB(1); return -1;	}

    /*  Silly old Jade  */
    if  ( paraBi->biParaLeftIndentTwips < 0	 )
	{ paraBi->biParaLeftIndentTwips= 0;	}
    if  ( paraBi->biParaFirstIndentTwips+ paraBi->biParaLeftIndentTwips < 0 )
	{ paraBi->biParaFirstIndentTwips= -paraBi->biParaLeftIndentTwips; }

    return 0;
    }

/************************************************************************/
/*									*/
/*  Start/Close a section.						*/
/*									*/
/************************************************************************/

static int docRtfStartSect(	RtfReadingContext *	rrc )
    {
    BufferItem *		sectBi;

    sectBi= docInsertItem( rrc->rrcBd, rrc->rrcBi, -1, DOClevSECT );
    if  ( ! sectBi )
	{ SXDEB(docLevelStr(rrc->rrcLevel),sectBi); return -1; }
    rrc->rrcBi= sectBi;

    if  ( sectBi->biInExternalItem == DOCinBODY )
	{
	rrc->rrcSelectionScope= sectBi->biSectSelectionScope;

	if  ( docCopySectionProperties( &(sectBi->biSectProperties),
					 &(rrc->rrcSectionProperties) ) )
	    { LDEB(1); return -1;	}

	if  ( sectBi->biNumberInParent > 0 )
	    {
	    const BufferItem *	prevSectBi;

	    prevSectBi= sectBi->biParent->biChildren[
					sectBi->biNumberInParent- 1];

	    if  ( docCopySectHeadersFooters( sectBi, rrc->rrcBd,
						prevSectBi, rrc->rrcBd ) )
		{ LDEB(1); return -1;	}
	    }
	}

    return 0;
    }

/************************************************************************/
/*									*/
/*  Start a row.							*/
/*									*/
/************************************************************************/

static int docRtfStartRow(	RtfReadingContext *	rrc,
				int			forTable )
    {
    BufferItem *		rowBi;

    rowBi= docInsertItem( rrc->rrcBd, rrc->rrcBi, -1, DOClevROW );
    if  ( ! rowBi )
	{ SXDEB(docLevelStr(rrc->rrcLevel),rowBi); return -1; }
    rowBi->biRowForTable= forTable;
    rrc->rrcBi= rowBi;

    if  ( docRtfSetRowProperties( rrc, rowBi ) )
	{ LDEB(1); return -1;	}

    return 0;
    }

static int docRtfStartCell(	RtfReadingContext *	rrc )
    {
    BufferItem *		cellBi;

    cellBi= docInsertItem( rrc->rrcBd, rrc->rrcBi, -1, DOClevCELL );
    if  ( ! cellBi )
	{ SXDEB(docLevelStr(rrc->rrcLevel),cellBi); return -1; }
    rrc->rrcBi= cellBi;

    return 0;
    }

/************************************************************************/
/*									*/
/*  Finish the item that is currently under construction.		*/
/*									*/
/************************************************************************/

int docRtfFinishCurrentItem(	const RtfControlWord *	rcw,
				RtfReadingContext *	rrc )
    {
    while(  rrc->rrcBi )
	{
	if  ( docRtfCloseItem( rrc, rrc->rrcBi ) )
	    { LDEB(1); return -1;	}

	rrc->rrcBi= rrc->rrcBi->biParent;
	}

    return 0;
    }

/************************************************************************/

static int docRtfStartItem(	int *			pRowNesting,
				RtfReadingContext *	rrc )
    {
    RtfReadingState *	rrs= rrc->rrcState;
    int			tableNestingTo;
    int			rowNestingTo;

    tableNestingTo= rrs->rrsParagraphProperties.ppTableNesting;
    rowNestingTo= tableNestingTo;
    if  ( rowNestingTo == 0 )
	{ rowNestingTo= 1;	}

    switch( rrc->rrcBi->biLevel )
	{
	case DOClevBODY:
	    if  ( docRtfStartSect( rrc ) )
		{ LDEB(1); return -1;	}
	    break;

	case DOClevSECT:
	    if  ( docRtfStartRow( rrc, tableNestingTo > 0 ) )
		{ LDEB(1); return -1;	}
	    (*pRowNesting)++;
	    break;

	case DOClevROW:
	    if  ( docRtfStartCell( rrc ) )
		{ LDEB(1); return -1;	}
	    break;

	case DOClevCELL:
	    if  ( tableNestingTo > 0				&&
		  rrc->rrcBi->biParent->biLevel == DOClevROW	&&
		  ! rrc->rrcBi->biParent->biRowForTable		)
		{
		if  ( docRtfPopItem( rrc, rrc->rrcBi->biParent ) )
		    { LDEB(1); return -1;	}

		(*pRowNesting)--;
		break;
		}

	    if  ( rowNestingTo > *pRowNesting )
		{
		if  ( *pRowNesting < 1 )
		    { LDEB(*pRowNesting); return -1;	}

		if  ( docRtfStartRow( rrc, tableNestingTo > 0 ) )
		    { LDEB(1); return -1;	}

		(*pRowNesting)++;
		break;
		}

	    if  ( rowNestingTo < *pRowNesting )
		{ LLDEB(rowNestingTo,*pRowNesting); return -1;	}

	    if  ( docRtfStartParagraph( rrc ) )
		{ LDEB(1); return -1;	}
	    break;

	default:
	    SDEB(docLevelStr(rrc->rrcBi->biLevel));
	    return -1;
	}

    return 0;
    }

/************************************************************************/
/*									*/
/*  Return the current paragraph. If there is none, make one.		*/
/*									*/
/************************************************************************/

BufferItem * docRtfGetParaItem(	RtfReadingContext *	rrc )
    {
    int			rowNesting;

    if  ( ! rrc->rrcBi )
	{ XDEB(rrc->rrcBi); return (BufferItem *)0;	}

    rowNesting= docRowNesting( rrc->rrcBi );

    while( rrc->rrcBi->biLevel != DOClevPARA )
	{
	if  ( docRtfStartItem( &rowNesting, rrc ) )
	    { LDEB(1); return (BufferItem *)0;	}
	}

    return rrc->rrcBi;
    }

/************************************************************************/
/*									*/
/*  Return the current section. If there is none, make one.		*/
/*									*/
/************************************************************************/

BufferItem * docRtfGetSectItem(	RtfReadingContext *	rrc )
    {
    BufferItem *	sectBi= docGetSectItem( rrc->rrcBi );

    if  ( ! sectBi )
	{
	int	rowNesting= 0;

	while( rrc->rrcBi->biLevel != DOClevSECT )
	    {
	    if  ( docRtfStartItem( &rowNesting, rrc ) )
		{ LDEB(1); return (BufferItem *)0;	}
	    }

	sectBi= rrc->rrcBi;
	}

    if  ( ! sectBi || sectBi->biLevel != DOClevSECT )
	{ XDEB(sectBi); return (BufferItem *)0;	}

    return sectBi;
    }

