/************************************************************************/
/*									*/
/*  Read the various document tables of an RTF text file into a		*/
/*  BufferDocument.							*/
/*									*/
/************************************************************************/

#   include	"docBufConfig.h"

#   include	<stdlib.h>
#   include	<string.h>
#   include	<stdio.h>
#   include	<ctype.h>

#   include	<appDebugon.h>

#   include	<appUnit.h>
#   include	<utilOfficeCharset.h>
#   include	"docRtf.h"

/************************************************************************/

void docRtfRestartFont(		RtfReadingContext *	rrc )
    {
    docCleanDocumentFont( &(rrc->rrcCurrentFont) );
    docInitDocumentFont( &(rrc->rrcCurrentFont) );
    docRtfInitEncodedFont( &(rrc->rrcCurrentEncodedFont) );
    if  ( rrc->rrcBd )
	{
	rrc->rrcCurrentEncodedFont.ecCharset=
				rrc->rrcBd->bdProperties.dpDocumentCharset;
	}

    return ;
    }

/************************************************************************/
/*									*/
/*  Consume a font table.						*/
/*									*/
/************************************************************************/

static int docRtfCommitFaltName(	const RtfControlWord *	rcw,
					RtfReadingContext *	rrc )
    {
    const int	removeSemicolon= 0;
    int		size= 0;

    if  ( docRtfStoreSavedText( &(rrc->rrcCurrentFont.dfAltName), &size,
						    rrc, removeSemicolon ) )
	{ LDEB(1); return -1; }

    return 0;
    }

static int docRtfCommitFontPanoseText(	const RtfControlWord *	rcw,
					RtfReadingContext *	rrc )
    {
    RtfReadingState *	rrs= rrc->rrcState;

    if  ( rrs->rrsSavedText.mbSize != FONTlenPANOSE )
	{
	LLDEB(rrs->rrsSavedText.mbSize,FONTlenPANOSE);
	utilEmptyMemoryBuffer( &(rrs->rrsSavedText) );
	return 0;
	}

    memcpy( rrc->rrcCurrentFont.dfPanose,
			rrs->rrsSavedText.mbBytes,
			rrs->rrsSavedText.mbSize );
    utilEmptyMemoryBuffer( &(rrs->rrsSavedText) );

    rrc->rrcCurrentFont.dfPanose[FONTlenPANOSE]= '\0';

    return 0;
    }

static int docRtfAddCurrentFontToList(	RtfReadingContext *	rrc )
    {
    int		rval= 0;
    const int	removeSemicolon= 1;
    int		size= 0;

    if  ( docRtfStoreSavedText( &(rrc->rrcCurrentFont.dfName), &size,
						    rrc, removeSemicolon ) )
	{ LDEB(1); rval= -1; goto ready; }

    if  ( rrc->rrcCurrentFont.dfName && rrc->rrcCurrentFont.dfName[0] == '@' )
	{
	if  ( docFontSetFamilyName( &(rrc->rrcCurrentFont),
					    rrc->rrcCurrentFont.dfName+ 1 ) )
	    { LDEB(1); return -1;	}
	}

    if  ( rrc->rrcCurrentFont.dfName )
	{
	DocumentProperties *	dp= &(rrc->rrcBd->bdProperties);
	DocumentFontList *	dfl= &(dp->dpFontList);
	DocumentFont *		df;

	EncodedFont *		efIn= &(rrc->rrcCurrentEncodedFont);
	EncodedFont *		efOut;

	int			cs;

	if  ( ! strcmp( rrc->rrcCurrentFont.dfName, "ZapfDingbats" ) )
	    {
	    if  ( docFontSetFamilyName( &(rrc->rrcCurrentFont),
							"ITC Zapf Dingbats" ) )
		{ LDEB(1); return -1;	}
	    }
	if  ( ! strcmp( rrc->rrcCurrentFont.dfName, "ZapfChancery" ) )
	    {
	    if  ( docFontSetFamilyName( &(rrc->rrcCurrentFont),
							"ITC Zapf Chancery" ) )
		{ LDEB(1); return -1;	}
	    }

	cs= utilRemoveCharsetFromFontName( &(rrc->rrcCurrentFont),
							    efIn->ecCharset );

	efIn->ecBufFontNumber= docGetFontByName( dfl,
						rrc->rrcCurrentFont.dfName );
	if  ( efIn->ecBufFontNumber < 0 )
	    {
	    SLDEB(rrc->rrcCurrentFont.dfName,efIn->ecBufFontNumber);
	    rval= -1; goto ready;
	    }

	df= docFontListGetFontByNumber( dfl, efIn->ecBufFontNumber );
	if  ( ! df )
	    { SXDEB(rrc->rrcCurrentFont.dfName,df); rval= -1; goto ready; }

	if  ( docCopyDocumentFont( df, &(rrc->rrcCurrentFont) ) )
	    { LDEB(1); rval= -1; goto ready;	}
	df->dfDocFontNumber= efIn->ecBufFontNumber;
	df->dfUsed= 1;

	efOut= (EncodedFont *)utilPagedListClaimItem(
			&(rrc->rrcEncodedFontList), efIn->ecFileFontNumber );
	if  ( ! efOut )
	    { LXDEB(efIn->ecFileFontNumber,efOut); rval= -1; goto ready; }
	*efOut= *efIn;
	}

  ready:
    docRtfRestartFont( rrc );

    return rval;
    }

static int docRtfFontName(	RtfReadingContext *	rrc,
				const char *		name,
				int			len )
    {
    if  ( docRtfSaveDocEncodedText( rrc, name, len ) )
	{ LDEB(len); return -1;	}

    if  ( docRtfAddCurrentFontToList( rrc ) )
	{ LDEB(len); return -1;	}

    return 0;
    }

static int docRtfCommitCurrentFont(	const RtfControlWord *	rcw,
					RtfReadingContext *	rrc )
    { return docRtfAddCurrentFontToList( rrc );	}

int docRtfFontProperty(		const RtfControlWord *	rcw,
				int			arg,
				RtfReadingContext *	rrc )
    {
    switch( rcw->rcwID )
	{
	case DFpropFAMILY_STYLE:
	    rrc->rrcCurrentFont.dfStyleInt= arg;
	    break;

	case DFpropCHARSET:
	    rrc->rrcCurrentEncodedFont.ecCharset= arg;
	    break;

	case DFpropCODEPAGE:
	    rrc->rrcCurrentEncodedFont.ecCodepage= arg;
	    break;

	case DFpropPITCH:
	    rrc->rrcCurrentFont.dfPitch= arg;
	    break;

	default:
	    SLDEB(rcw->rcwWord,arg);
	}

    return 0;
    }

static RtfControlWord	docRtfFontGroupGroups[]=
    {
    RTF_TEXT_GROUP( "panose",	DFpropPANOSE,	docRtfCommitFontPanoseText ),
    RTF_TEXT_GROUP( "falt",	DFpropALT_NAME,	docRtfCommitFaltName ),

    { (char *)0, 0, 0 }
    };

static int docRtfFontGroup(	const RtfControlWord *	rcw,
				int			arg,
				RtfReadingContext *	rrc )
    {
    docRtfRestartFont( rrc );

    rrc->rrcCurrentEncodedFont.ecFileFontNumber= arg;

    if  ( docRtfReadGroup( rcw, 0, 0, rrc,
					    docRtfFontGroupGroups,
					    docRtfSaveDocEncodedText,
					    docRtfCommitCurrentFont ) )
	{ SLDEB(rcw->rcwWord,arg); return -1;	}

    return 0;
    }

static int docRtfThemeFontGroup(	const RtfControlWord *	rcw,
					int			arg,
					RtfReadingContext *	rrc )
    {
    docRtfRestartFont( rrc );

    if  ( docRtfReadGroup( rcw, 0, 0, rrc,
					    docRtfFontGroupGroups,
					    docRtfSaveDocEncodedText,
					    docRtfCommitCurrentFont ) )
	{ SLDEB(rcw->rcwWord,arg); return -1;	}

    return 0;
    }

static RtfControlWord	docRtfFontTableGroups[]=
    {
	RTF_DEST_XX( "f",	RTFidF,	docRtfFontGroup ),
	RTF_DEST_XX( "flomajor",RTFidF,	docRtfThemeFontGroup ),
	RTF_DEST_XX( "fhimajor",RTFidF,	docRtfThemeFontGroup ),
	RTF_DEST_XX( "fdbmajor",RTFidF,	docRtfThemeFontGroup ),
	RTF_DEST_XX( "fbimajor",RTFidF,	docRtfThemeFontGroup ),
	RTF_DEST_XX( "flominor",RTFidF,	docRtfThemeFontGroup ),
	RTF_DEST_XX( "fhiminor",RTFidF,	docRtfThemeFontGroup ),
	RTF_DEST_XX( "fdbminor",RTFidF,	docRtfThemeFontGroup ),
	RTF_DEST_XX( "fbiminor",RTFidF,	docRtfThemeFontGroup ),

	{ (char *)0, 0, 0 }
    };

static int docRtfCommitFontTable(	const RtfControlWord *	rcw,
					RtfReadingContext *	rrc )
    {
    BufferDocument *		bd= rrc->rrcBd;
    DocumentProperties *	dp= &(bd->bdProperties);

    int				charset;

    if  ( rrc->rrcDefaultFont >= 0 )
	{
	docRtfReadMapFont( rrc, &(dp->dpDefaultFont), &charset,
						    rrc->rrcDefaultFont );
	}

    if  ( rrc->rrcDefaultFontDbch >= 0 )
	{
	docRtfReadMapFont( rrc, &(dp->dpDefaultFontDbch), &charset,
						    rrc->rrcDefaultFontDbch );
	}

    if  ( rrc->rrcDefaultFontLoch >= 0 )
	{
	docRtfReadMapFont( rrc, &(dp->dpDefaultFontLoch), &charset,
						    rrc->rrcDefaultFontLoch );
	}

    if  ( rrc->rrcDefaultFontHich >= 0 )
	{
	docRtfReadMapFont( rrc, &(dp->dpDefaultFontHich), &charset,
						    rrc->rrcDefaultFontHich );
	}

    if  ( rrc->rrcDefaultFontBi >= 0 )
	{
	docRtfReadMapFont( rrc, &(dp->dpDefaultFontBi), &charset,
						    rrc->rrcDefaultFontBi );
	}

    return 0;
    }

int docRtfFontTable(	const RtfControlWord *	rcw,
			int			arg,
			RtfReadingContext *	rrc )
    {
    if  ( docRtfReadGroup( rcw, 0, 0, rrc,
						docRtfFontTableGroups,
						docRtfFontName,
						docRtfCommitFontTable ) )
	{ SLDEB(rcw->rcwWord,arg); return -1;	}

    return 0;
    }

/************************************************************************/

static const char * docRtfFontFamilyStyleTag(	int styleInt )
    {
    switch( styleInt )
	{
	case DFstyleFNIL:	return "\\fnil";
	case DFstyleFROMAN:	return "\\froman";
	case DFstyleFSWISS:	return "\\fswiss";
	case DFstyleFMODERN:	return "\\fmodern";
	case DFstyleFSCRIPT:	return "\\fscript";
	case DFstyleFDECOR:	return "\\fdecor";
	case DFstyleFTECH:	return "\\ftech";
	case DFstyleFBIDI:	return "\\fbidi";

	default:
	    LDEB(styleInt); return "\\fnil";
	}
    }

/************************************************************************/
/*									*/
/*  Write a font table.							*/
/*									*/
/************************************************************************/

static void docRtfWriteEncodedFont(	RtfWritingContext *	rwc,
					const DocumentFont *	df,
					const char *		suffix,
					int			fontnum,
					int			charset )
    {
    docRtfWriteArgDestinationBegin( rwc, "\\f", fontnum );

    docRtfWriteTag( rwc, docRtfFontFamilyStyleTag( df->dfStyleInt ) );

    if  ( charset != FONTcharsetDEFAULT )
	{ docRtfWriteArgTag( rwc, "\\fcharset", charset ); }

    if  ( df->dfPitch != FONTpitchDEFAULT )
	{ docRtfWriteArgTag( rwc, "\\fprq", df->dfPitch ); }

    if  ( df->dfPanose[0] )
	{
	const int	addSemicolon= 0;

	docRtfWriteDocEncodedStringDestination( rwc, "\\*\\panose",
		    df->dfPanose, strlen( df->dfPanose ), addSemicolon );
	}

    if  ( df->dfName )
	{
	const char *	name= df->dfName;

	if  ( ! strcmp( name, "ITC Zapf Dingbats" ) )
	    { name= "ZapfDingbats";	}
	if  ( ! strcmp( name, "ITC Zapf Chancery" ) )
	    { name= "ZapfChancery";	}

	docRtfWriteDocEncodedString( rwc, name, strlen( name ) );

	if  ( suffix )
	    { docRtfWriteDocEncodedString( rwc, suffix, strlen( suffix ) ); }
	}

    if  ( df->dfAltName )
	{
	const int	addSemicolon= 0;

	docRtfWriteDocEncodedStringDestination( rwc, "\\*\\falt",
		    df->dfAltName, strlen( df->dfAltName ), addSemicolon );
	}

    docRtfWriteSemicolon( rwc );

    docRtfWriteDestinationEnd( rwc );

    return;
    }

static const EncodedFont * dowRtfWriteGetEncodedFont(
				RtfWritingContext *		rwc,
				const DocumentFont *		df,
				int				csIdx )
    {
    const EncodedFont *		ef;
    int				key;

    key= CHARSETidx_COUNT* df->dfDocFontNumber+ csIdx;
    ef= (const EncodedFont *)utilPagedListGetItemByNumber(
					&(rwc->rwcEncodedFontList), key );
    if  ( ! ef )
	{ LLLXDEB(df->dfDocFontNumber,csIdx,key,ef);	}

    return ef;
    }

int docRtfWriteGetCharset(	RtfWritingContext *		rwc,
				int *				pCharset,
				const DocumentFont *		df,
				int				symbol )
    {
    int			csIdx;
    const EncodedFont *	ef;
    int			charset= FONTcharsetDEFAULT;
    int			fontNumber= -1;

    csIdx= utilIndexMappingGet( &(df->dfUnicodeToCharset), symbol );
    if  ( csIdx < 0 )
	{ csIdx= df->dfCharsetIdxUsed[0];	}

    ef= dowRtfWriteGetEncodedFont( rwc, df, csIdx );
    if  ( ef )
	{
	charset= ef->ecCharset;
	fontNumber= ef->ecFileFontNumber;
	}
    else{ LXDEB(csIdx,ef);		}

    *pCharset= charset;
    return fontNumber;
    }

void docRtfWriteFontTable(	RtfWritingContext *		rwc,
				const DocumentFontList *	dfl )
    {
    int				fnt;

    docRtfWriteDestinationBegin( rwc, "\\fonttbl" );
    docRtfWriteNextLine( rwc );

    for ( fnt= 0; fnt < dfl->dflFontCount; fnt++ )
	{
	const DocumentFont *	df= docFontListGetFontByNumber( dfl, fnt );
	int			cs;

	if  ( ! df || ! df->dfName )
	    { continue;	}
	if  ( ! df->dfUsed )
	    { continue;	}

	for  ( cs= 0; cs < df->dfCharsetIdxUsedCount; cs++ )
	    {
	    const OfficeCharset *	oc;
	    const EncodedFont *		ef;
	    const char *		suffix= (const char *)0;

	    ef= dowRtfWriteGetEncodedFont( rwc, df, df->dfCharsetIdxUsed[cs] );
	    if  ( ! ef )
		{ LXDEB(cs,ef); continue;	}

	    oc= utilGetOfficeCharsetByIndex( df->dfCharsetIdxUsed[cs] );
	    if  ( ! oc )
		{ LXDEB(df->dfCharsetIdxUsed[cs],oc); continue; }

	    if  ( oc->ocEmitSuffix )
		{
		suffix= oc->ocOfficeFontnameSuffix;
		}

	    docRtfWriteEncodedFont( rwc, df, suffix, ef->ecFileFontNumber,
							oc->ocOfficeCharset );

	    if  ( fnt+ 1 < dfl->dflFontCount		||
		  cs+ 1 < df->dfCharsetIdxUsedCount	)
		{ docRtfWriteNextLine( rwc );	}
	    }
	}

    docRtfWriteDestinationEnd( rwc );
    docRtfWriteNextLine( rwc );

    return;
    }

/************************************************************************/

int docRtfWriteBuildFontAdmin(	RtfWritingContext *		rwc )
    {
    BufferDocument *		bd= rwc->rwcBd;
    DocumentProperties *	dp= &(bd->bdProperties);
    DocumentFontList *		dfl= &(dp->dpFontList);

    int				f;
    int				fileFontNumber= 0;

    utilDocFontListClearCharsUsed( dfl );

    if  ( docGetCharsUsed( bd ) )
	{ LDEB(1); return -1;	}

    for ( f= 0; f < dfl->dflFontCount; f++ )
	{
	DocumentFont *	df= docFontListGetFontByNumber( dfl, f );

	if  ( ! df )
	    { continue;	}

	if  ( docFontFindLegacyEncodings( df ) )
	    { LSDEB(f,df->dfName); return -1;	}

	if  ( df->dfCharsetIdxUsedCount > 0 )
	    {
	    int		cs;

	    for ( cs= 0; cs < df->dfCharsetIdxUsedCount; cs++ )
		{
		EncodedFont		efIn;
		EncodedFont *		efOut;
		const OfficeCharset *	oc;
		int			key;

		oc= utilGetOfficeCharsetByIndex( df->dfCharsetIdxUsed[cs] );

		docRtfInitEncodedFont( &efIn );

		efIn.ecFileFontNumber= fileFontNumber++;
		efIn.ecBufFontNumber= df->dfDocFontNumber;
		efIn.ecCharset= oc->ocOfficeCharset;
		efIn.ecCharsetIdx= df->dfCharsetIdxUsed[cs];

		key= CHARSETidx_COUNT* efIn.ecBufFontNumber+
						    df->dfCharsetIdxUsed[cs];

		efOut= (EncodedFont *)utilPagedListClaimItem(
					    &(rwc->rwcEncodedFontList), key );
		if  ( ! efOut )
		    { LXDEB(key,efOut); return -1;	}
		*efOut= efIn;
		}
	    }
	}

    return 0;
    }

/************************************************************************/

void docRtfInitEncodedFont(	EncodedFont *	ef )
    {
    ef->ecFileFontNumber= -1;
    ef->ecBufFontNumber= -1;
    ef->ecCharset= FONTcharsetANSI;
    ef->ecCharsetIdx= -1;
    ef->ecCodepage= -1;
    }

/************************************************************************/

int docRtfReadMapFont(	const RtfReadingContext *	rrc,
			int *				pDocFontNum,
			int *				pCharset,
			int				fileFontNum )
    {
    EncodedFont *	ef;

    if  ( fileFontNum < 0 )
	{ LDEB(fileFontNum); return -1;	}

    ef= (EncodedFont *)utilPagedListGetItemByNumber(
				    &(rrc->rrcEncodedFontList), fileFontNum );
    if  ( ! ef )
	{ return 1;	}

    if  ( ef->ecCodepage >= 0 )
	{ LDEB(ef->ecCodepage);	}

    *pDocFontNum= ef->ecBufFontNumber;
    *pCharset= ef->ecCharset;

    return 0;

    }
