/************************************************************************/
/*									*/
/*  Read/Write document properties to/from RTF.				*/
/*									*/
/************************************************************************/

#   include	"docBufConfig.h"

#   include	<stdlib.h>
#   include	<string.h>
#   include	<stdio.h>
#   include	<ctype.h>

#   include	<appDebugon.h>

#   include	<utilOfficeCharset.h>
#   include	"docRtf.h"

/************************************************************************/
/*									*/
/*  Remember Document Properties.					*/
/*									*/
/************************************************************************/

/************************************************************************/
/*									*/
/*  Remember the code page for the document.				*/
/*  = All values from the Word 2003 spec explicitly included.		*/
/*  = String values checked against GNU iconv 'iconv (GNU libc) 2.3.6'	*/
/*  = Check strings against iconv -l					*/
/*									*/
/************************************************************************/

static int docRtfSetAnsicpg(	TextConverter *		tc,
				int			ansicpg )
    {
    char			scratch[20];

    switch( ansicpg )
	{
	case 437: case 819: case 850: case 852:
	case 460: case 862: case 863: case 864:
	case 465: case 866: case 874: case 932:
	case 936: case 949: case 950:
	case 1361:
	    sprintf( scratch, "CP%d", ansicpg );
	    utilTextConverterSetNativeEncodingName( tc, scratch );
	    return 0;

	case 708:
	    utilTextConverterSetNativeEncodingName( tc, "ASMO-708" );
	    return 0;
	case 709:
	    utilTextConverterSetNativeEncodingName( tc, "ASMO_449" );
	    return 0;

	case 1250: case 1251: case 1252: case 1253:
	case 1254: case 1255: case 1256: case 1257:
	case 1258:
	    /* sprintf( scratch, "WINDOWS-%d", ansicpg ); */ /* GNU? */
	    sprintf( scratch, "CP%d", ansicpg );
	    utilTextConverterSetNativeEncodingName( tc, scratch );
	    return 0;

	case 10000:
	    utilTextConverterSetNativeEncodingName( tc, "MACINTOSH" );
	    return 0;

	case 710:
	case 711:
	case 720:
	default:
	    LDEB(ansicpg);
	    utilTextConverterSetNativeEncodingName( tc, "WINDOWS-1252" );
	    return -1;
	}
    }

/************************************************************************/
/*									*/
/*  Remember the document charset. It is translated to a code page	*/
/*  where possible.							*/
/*									*/
/************************************************************************/

static void docRtfReadSetDocumentCharset(
				RtfReadingContext *	rrc,
				int			arg )
    {
    DocumentProperties *	dp= &(rrc->rrcBd->bdProperties);
    TextConverter *		tc= &(rrc->rrcTextConverters.rtcRtfConverter);

    switch( arg )
	{
	case DOCcharsetANSI:
	    if  ( ! docRtfSetAnsicpg( tc, 1252 ) )
		{ dp->dpDocumentCharset= arg;	}
	    break;

	case DOCcharsetPC:
	    if  ( ! docRtfSetAnsicpg( tc, 437 ) )
		{ dp->dpDocumentCharset= arg;	}
	    break;

	case DOCcharsetPCA:
	    if  ( ! docRtfSetAnsicpg( tc, 850 ) )
		{ dp->dpDocumentCharset= arg;	}
	    break;

	case DOCcharsetMAC:
	    if  ( ! docRtfSetAnsicpg( tc, 10000 ) )
		{ dp->dpDocumentCharset= arg;	}
	    break;

	default:
	    LDEB(arg);
	    break;
	}
    }

int docRtfRememberDocProperty(	const RtfControlWord *	rcw,
				int			arg,
				RtfReadingContext *	rrc )
    {
    DocumentProperties *	dp= &(rrc->rrcBd->bdProperties);
    SectionProperties *		sp= &(rrc->rrcSectionProperties);

    int				charset= -1;

    switch( rcw->rcwID )
	{
	case DGpropLEFT_MARGIN:
	    dp->dpGeometry.dgLeftMarginTwips= arg;
	    sp->spDocumentGeometry.dgLeftMarginTwips= arg;
	    rrc->rrcGotDocGeometry= 1;
	    break;
	case DGpropTOP_MARGIN:
	    dp->dpGeometry.dgTopMarginTwips= arg;
	    sp->spDocumentGeometry.dgTopMarginTwips= arg;
	    rrc->rrcGotDocGeometry= 1;
	    break;
	case DGpropRIGHT_MARGIN:
	    dp->dpGeometry.dgRightMarginTwips= arg;
	    sp->spDocumentGeometry.dgRightMarginTwips= arg;
	    rrc->rrcGotDocGeometry= 1;
	    break;
	case DGpropBOTTOM_MARGIN:
	    dp->dpGeometry.dgBottomMarginTwips= arg;
	    sp->spDocumentGeometry.dgBottomMarginTwips= arg;
	    break;
	case DGpropPAGE_WIDTH:
	    dp->dpGeometry.dgPageWideTwips= arg;
	    sp->spDocumentGeometry.dgPageWideTwips= arg;
	    rrc->rrcGotDocGeometry= 1;
	    break;
	case DGpropPAGE_HEIGHT:
	    dp->dpGeometry.dgPageHighTwips= arg;
	    sp->spDocumentGeometry.dgPageHighTwips= arg;
	    rrc->rrcGotDocGeometry= 1;
	    break;
	case DGpropGUTTER:
	    dp->dpGeometry.dgGutterTwips= arg;
	    sp->spDocumentGeometry.dgGutterTwips= arg;
	    rrc->rrcGotDocGeometry= 1;
	    break;
	case DGpropMARGMIR:
	    dp->dpGeometry.dgMirrorMargins= arg != 0;
	    sp->spDocumentGeometry.dgMirrorMargins= arg != 0;
	    rrc->rrcGotDocGeometry= 1;
	    break;

	case DPpropFACING_PAGES:
	    dp->dpHasFacingPages= arg != 0;
	    break;
	case DPpropGUTTER_HORIZONTAL:
	    dp->dpGutterHorizontal= arg != 0;
	    break;
	case DPpropWIDOWCTRL:
	    dp->dpWidowControl= arg != 0;
	    break;
	case DPpropTWO_ON_ONE:
	    dp->dpTwoOnOne= arg != 0;
	    break;
	case DPpropDOCTEMP:
	    dp->dpIsDocumentTemplate= arg != 0;
	    break;

	case DPpropSTART_PAGE:
	    dp->dpStartPageNumber= arg;
	    break;
	case DPpropDEFTAB:
	    dp->dpTabIntervalTwips= arg;
	    break;

	case DPpropDEFF:
	    rrc->rrcDefaultFont= arg;
	    docRtfReadMapFont( rrc, &(dp->dpDefaultFont), &charset,
						    rrc->rrcDefaultFont );
	    break;
	case DPpropSTSHFDBCH:
	    rrc->rrcDefaultFontDbch= arg;
	    docRtfReadMapFont( rrc, &(dp->dpDefaultFontDbch), &charset,
						    rrc->rrcDefaultFontDbch );
	    break;
	case DPpropSTSHFLOCH:
	    rrc->rrcDefaultFontLoch= arg;
	    docRtfReadMapFont( rrc, &(dp->dpDefaultFontLoch), &charset,
						    rrc->rrcDefaultFontLoch );
	    break;
	case DPpropSTSHFHICH:
	    rrc->rrcDefaultFontHich= arg;
	    docRtfReadMapFont( rrc, &(dp->dpDefaultFontHich), &charset,
						    rrc->rrcDefaultFontHich );
	    break;
	case DPpropSTSHFBI:
	    rrc->rrcDefaultFontBi= arg;
	    docRtfReadMapFont( rrc, &(dp->dpDefaultFontBi), &charset,
						    rrc->rrcDefaultFontBi );
	    break;

	case DPpropNOTETYPES:
	    if  ( arg < 0 || arg > 2 )
		{ LDEB(arg); break;	}
	    dp->dpFootEndNoteType= arg;
	    break;

	case DPpropFOOTNOTE_STARTNR:
	    dp->dpFootnoteProperties.npStartNumber= arg;
	    break;
	case DPpropENDNOTE_STARTNR:
	    dp->dpEndnoteProperties.npStartNumber= arg;
	    break;

	case DPpropFOOTNOTE_POSITION:
	    dp->dpFootnoteProperties.npPosition= arg;
	    break;

	case DPpropENDNOTE_POSITION:
	    dp->dpEndnoteProperties.npPosition= arg;
	    break;

	case DPpropFOOTNOTE_RESTART:
	    dp->dpFootnoteProperties.npRestart= arg;
	    break;

	case DPpropENDNOTE_RESTART:
	    dp->dpEndnoteProperties.npRestart= arg;
	    break;

	case DPpropFOOTNOTE_STYLE:
	    dp->dpFootnoteProperties.npNumberStyle= arg;
	    break;

	case DPpropENDNOTE_STYLE:
	    dp->dpEndnoteProperties.npNumberStyle= arg;
	    break;

	case DPpropDOC_CHARSET:
	    docRtfReadSetDocumentCharset( rrc, arg );
	    break;

	case DPpropANSICPG:
	    if  ( ! docRtfSetAnsicpg(
			    &(rrc->rrcTextConverters.rtcRtfConverter), arg ) )
		{ dp->dpAnsiCodepage= arg;	}
	    break;

	case DPpropTOP_BORDER:
	    dp->dpTopBorder= rrc->rrcBorderProperties;
	    docInitBorderProperties( &(rrc->rrcBorderProperties) );
	    break;

	case DPpropBOTTOM_BORDER:
	    dp->dpBottomBorder= rrc->rrcBorderProperties;
	    docInitBorderProperties( &(rrc->rrcBorderProperties) );
	    break;

	case DPpropLEFT_BORDER:
	    dp->dpLeftBorder= rrc->rrcBorderProperties;
	    docInitBorderProperties( &(rrc->rrcBorderProperties) );
	    break;

	case DPpropRIGHT_BORDER:
	    dp->dpRightBorder= rrc->rrcBorderProperties;
	    docInitBorderProperties( &(rrc->rrcBorderProperties) );
	    break;

	case DPpropHEAD_BORDER:
	    dp->dpHeadBorder= rrc->rrcBorderProperties;
	    docInitBorderProperties( &(rrc->rrcBorderProperties) );
	    break;

	case DPpropFOOT_BORDER:
	    dp->dpFootBorder= rrc->rrcBorderProperties;
	    docInitBorderProperties( &(rrc->rrcBorderProperties) );
	    break;

	default:
	    break;
	}
    
    return 0;
    }

/************************************************************************/
/*									*/
/*  Save Document Properties to RTF.					*/
/*									*/
/************************************************************************/

static void docRtfSaveInfo(	RtfWritingContext *	rwc,
				const char *		tag,
				const char *		info )
    {
    const int		addSemicolon= 0;

    if  ( ! info )
	{ return;	}

    docRtfWriteDocEncodedStringDestination( rwc, tag,
				    info, strlen( info ), addSemicolon );

    docRtfWriteNextLine( rwc );

    return;
    }

static void docRtfSaveDate(	RtfWritingContext *	rwc,
				const char *		tag,
				const struct tm *	tm )
    {
    if  ( tm->tm_mday == 0 )
	{ return;	}

    docRtfWriteDestinationBegin( rwc, tag );

    docRtfWriteArgTag( rwc, "\\yr",	tm->tm_year+ 1900 );
    docRtfWriteArgTag( rwc, "\\mo",	tm->tm_mon+ 1 );
    docRtfWriteArgTag( rwc, "\\dy",	tm->tm_mday );
    docRtfWriteArgTag( rwc, "\\hr",	tm->tm_hour );
    docRtfWriteArgTag( rwc, "\\min",	tm->tm_min );
    docRtfWriteArgTag( rwc, "\\sec",	tm->tm_sec );

    docRtfWriteDestinationEnd( rwc );
    docRtfWriteNextLine( rwc );

    return;
    }

int docRtfSaveDocumentProperties(	RtfWritingContext *		rwc,
					const DocumentProperties *	dp )
    {
    const DocumentGeometry *	dg= &(dp->dpGeometry);

    /* NO! We always write ansi 1252 documents
    int				ansicpg= dp->dpAnsiCodepage;

    if  ( ansicpg >= 0 )
	{
	if  ( docRtfSetAnsicpg( &(rwc->rwcTextConverters.rtcRtfConverter),
								    ansicpg ) )
	    { LDEB(ansicpg);	}
	}
    */

    if  ( dp->dpDefaultFont >= 0				&&
	  dp->dpDefaultFont < dp->dpFontList.dflFontCount	)
	{ docRtfWriteArgTag( rwc, "\\deff", dp->dpDefaultFont );	}

    if  ( dp->dpDefaultFontDbch >= 0			&&
	  dp->dpDefaultFontDbch < dp->dpFontList.dflFontCount	)
	{
	docRtfWriteArgTag( rwc, "\\stshfdbch", dp->dpDefaultFontDbch );
	}

    if  ( dp->dpDefaultFontLoch >= 0			&&
	  dp->dpDefaultFontLoch < dp->dpFontList.dflFontCount	)
	{
	docRtfWriteArgTag( rwc, "\\stshfloch", dp->dpDefaultFontLoch );
	}

    if  ( dp->dpDefaultFontHich >= 0			&&
	  dp->dpDefaultFontHich < dp->dpFontList.dflFontCount	)
	{
	docRtfWriteArgTag( rwc, "\\stshfhich", dp->dpDefaultFontHich );
	}

    if  ( dp->dpDefaultFontBi >= 0			&&
	  dp->dpDefaultFontBi < dp->dpFontList.dflFontCount	)
	{ docRtfWriteArgTag( rwc, "\\stshfbi", dp->dpDefaultFontBi ); }

    /* Strictly spoken, this is not a document property */
    docRtfWriteArgTag( rwc, "\\uc", 1 );

    if  ( dp->dpFontList.dflFontCount > 0 )
	{
	docRtfWriteNextLine( rwc );
	docRtfWriteFontTable( rwc, &(dp->dpFontList) );
	}

    if  ( dp->dpColorCount > 0 )
	{
	docRtfWriteNextLine( rwc );
	docRtfWriteColorTable( rwc, dp );
	}

    if  ( rwc->rwcBd->bdStyleSheet.dssStyleCount > 0 )
	{
	docRtfWriteNextLine( rwc );
	docRtfWriteStyleSheet( rwc, &(rwc->rwcBd->bdStyleSheet) );
	}

    if  ( dp->dpListTable.dltListCount > 0 )
	{
	docRtfWriteNextLine( rwc );
	docRtfWriteListTable( rwc, &(dp->dpListTable) );
	}

    if  ( dp->dpListOverrideTable.lotOverrideCount > 0 )
	{
	docRtfWriteNextLine( rwc );
	docRtfWriteListOverrideTable( rwc, &(dp->dpListOverrideTable) );
	}

    if  ( dp->dpAuthorCount > 0 )
	{
	docRtfWriteNextLine( rwc );
	docRtfWriteRevisorTable( rwc, dp );
	}

    docRtfSaveInfo( rwc, "\\*\\generator", dp->dpGeneratorWrite );

    if  ( docHasDocumentInfo( dp ) )
	{
	docRtfWriteNextLine( rwc );
	docRtfWriteDestinationBegin( rwc, "\\info" );
	docRtfWriteNextLine( rwc );

	docRtfSaveInfo( rwc, "\\title",		dp->dpTitle );
	docRtfSaveInfo( rwc, "\\author",	dp->dpAuthor );
	docRtfSaveInfo( rwc, "\\company",	dp->dpCompany );
	docRtfSaveInfo( rwc, "\\subject",	dp->dpSubject );
	docRtfSaveInfo( rwc, "\\keywords",	dp->dpKeywords );
	docRtfSaveInfo( rwc, "\\doccomm",	dp->dpDoccomm );
	docRtfSaveInfo( rwc, "\\hlinkbase",	dp->dpHlinkbase );

	docRtfSaveDate( rwc, "\\creatim",	&(dp->dpCreatim) );
	docRtfSaveDate( rwc, "\\revtim",	&(dp->dpRevtim) );
	docRtfSaveDate( rwc, "\\printim",	&(dp->dpPrintim) );

	docRtfWriteDestinationEnd( rwc );
	}

    docRtfWriteNextLine( rwc );

    docRtfWriteArgTag( rwc, "\\paperw", dg->dgPageWideTwips );
    docRtfWriteArgTag( rwc, "\\paperh", dg->dgPageHighTwips );

    docRtfWriteArgTag( rwc, "\\margl", dg->dgLeftMarginTwips );
    docRtfWriteArgTag( rwc, "\\margr", dg->dgRightMarginTwips );
    docRtfWriteArgTag( rwc, "\\margt", dg->dgTopMarginTwips );
    docRtfWriteArgTag( rwc, "\\margb", dg->dgBottomMarginTwips );

    if  ( dg->dgGutterTwips != 0 )
	{ docRtfWriteArgTag( rwc, "\\gutter", dg->dgGutterTwips ); }
    if  ( dg->dgMirrorMargins )
	{ docRtfWriteTag( rwc, "\\margmirror" ); }

    if  ( dp->dpHasFacingPages )
	{ docRtfWriteTag( rwc, "\\facingp" );	}
    if  ( dp->dpGutterHorizontal )
	{ docRtfWriteTag( rwc, "\\gutterprl" );	}

    if  ( dp->dpWidowControl )
	{ docRtfWriteTag( rwc, "\\widowctrl" );	}

    if  ( dp->dpTwoOnOne )
	{ docRtfWriteTag( rwc, "\\twoonone" );	}
    if  ( dp->dpIsDocumentTemplate )
	{ docRtfWriteTag( rwc, "\\doctemp" );	}

    if  ( dp->dpTabIntervalTwips > 0 && dp->dpTabIntervalTwips != 720 )
	{ docRtfWriteArgTag( rwc, "\\deftab", dp->dpTabIntervalTwips ); }

    docRtfWriteNextLine( rwc );

    switch( dp->dpFootnoteProperties.npPosition )
	{
	case FTN_POS_SECT_END:
	    docRtfWriteTag( rwc, "\\endnotes" );
	    break;
	case FTN_POS_DOC_END:
	    docRtfWriteTag( rwc, "\\enddoc" );
	    break;
	case FTN_POS_BELOW_TEXT:
	    docRtfWriteTag( rwc, "\\ftntj" );
	    break;
	case FTN_POS_PAGE_BOTTOM:
	    docRtfWriteTag( rwc, "\\ftnbj" );
	    break;
	default:
	    LDEB(dp->dpFootnoteProperties.npPosition); break;
	}

    switch( dp->dpEndnoteProperties.npPosition )
	{
	case FTN_POS_SECT_END:
	    docRtfWriteTag( rwc, "\\aendnotes" );
	    break;
	case FTN_POS_DOC_END:
	    docRtfWriteTag( rwc, "\\aenddoc" );
	    break;
	case FTN_POS_BELOW_TEXT:
	    docRtfWriteTag( rwc, "\\aftntj" );
	    break;
	case FTN_POS_PAGE_BOTTOM:
	    docRtfWriteTag( rwc, "\\aftnbj" );
	    break;
	default:
	    LDEB(dp->dpEndnoteProperties.npPosition); break;
	}

    if  ( dp->dpFootnoteProperties.npStartNumber != 1 )
	{
	docRtfWriteArgTag( rwc, "\\ftnstart",
				    dp->dpFootnoteProperties.npStartNumber );
	}
    if  ( dp->dpEndnoteProperties.npStartNumber != 1 )
	{
	docRtfWriteArgTag( rwc, "\\aftnstart",
				    dp->dpEndnoteProperties.npStartNumber );
	}

    switch( dp->dpFootnoteProperties.npRestart )
	{
	case FTN_RST_CONTINUOUS:
	    docRtfWriteTag( rwc, "\\ftnrstcont" );
	    break;
	case FTN_RST_PER_SECTION:
	    docRtfWriteTag( rwc, "\\ftnrestart" );
	    break;
	case FTN_RST_PER_PAGE:
	    docRtfWriteTag( rwc, "\\ftnrstpg" );
	    break;
	default:
	    LDEB(dp->dpFootnoteProperties.npRestart); break;
	}

    switch( dp->dpEndnoteProperties.npRestart )
	{
	case FTN_RST_CONTINUOUS:
	    docRtfWriteTag( rwc, "\\aftnrstcont" );
	    break;
	case FTN_RST_PER_SECTION:
	    docRtfWriteTag( rwc, "\\aftnrestart" );
	    break;
	/* No!
	case FTN_RST_PER_PAGE:
	    docRtfWriteTag( rwc, "\\aftnrstpg" );
	    break;
	*/
	default:
	    LDEB(dp->dpEndnoteProperties.npRestart); break;
	}


    switch( dp->dpFootnoteProperties.npNumberStyle )
	{
	case FTNstyleNAR:
	    docRtfWriteTag( rwc, "\\ftnnar" );
	    break;
	case FTNstyleNALC:
	    docRtfWriteTag( rwc, "\\ftnnalc" );
	    break;
	case FTNstyleNAUC:
	    docRtfWriteTag( rwc, "\\ftnnauc" );
	    break;
	case FTNstyleNRLC:
	    docRtfWriteTag( rwc, "\\ftnnrlc" );
	    break;
	case FTNstyleNRUC:
	    docRtfWriteTag( rwc, "\\ftnnruc" );
	    break;
	case FTNstyleNCHI:
	    docRtfWriteTag( rwc, "\\ftnnchi" );
	    break;
	default:
	    LDEB(dp->dpFootnoteProperties.npNumberStyle); break;
	}

    switch( dp->dpEndnoteProperties.npNumberStyle )
	{
	case FTNstyleNAR:
	    docRtfWriteTag( rwc, "\\aftnnar" );
	    break;
	case FTNstyleNALC:
	    docRtfWriteTag( rwc, "\\aftnnalc" );
	    break;
	case FTNstyleNAUC:
	    docRtfWriteTag( rwc, "\\aftnnauc" );
	    break;
	case FTNstyleNRLC:
	    docRtfWriteTag( rwc, "\\aftnnrlc" );
	    break;
	case FTNstyleNRUC:
	    docRtfWriteTag( rwc, "\\aftnnruc" );
	    break;
	case FTNstyleNCHI:
	    docRtfWriteTag( rwc, "\\aftnnchi" );
	    break;
	default:
	    LDEB(dp->dpEndnoteProperties.npNumberStyle); break;
	}

    if  ( dp->dpTopBorder.bpStyle != DOCbsNONE )
	{ docRtfSaveBorder( rwc, "\\pgbrdrt", &(dp->dpTopBorder) ); }
    if  ( dp->dpBottomBorder.bpStyle != DOCbsNONE )
	{ docRtfSaveBorder( rwc, "\\pgbrdrb", &(dp->dpBottomBorder) ); }
    if  ( dp->dpLeftBorder.bpStyle != DOCbsNONE )
	{ docRtfSaveBorder( rwc, "\\pgbrdrl", &(dp->dpLeftBorder) ); }
    if  ( dp->dpRightBorder.bpStyle != DOCbsNONE )
	{ docRtfSaveBorder( rwc, "\\pgbrdrr", &(dp->dpRightBorder) ); }
    if  ( dp->dpHeadBorder.bpStyle != DOCbsNONE )
	{ docRtfSaveBorder( rwc, "\\pgbrdrhead", &(dp->dpHeadBorder) ); }
    if  ( dp->dpFootBorder.bpStyle != DOCbsNONE )
	{ docRtfSaveBorder( rwc, "\\pgbrdrfoot", &(dp->dpFootBorder) ); }

    docRtfWriteNextLine( rwc );

    return 0;
    }

/************************************************************************/
/*									*/
/*  Save the various separators relating to notes.			*/
/*									*/
/************************************************************************/

int docRtfSaveDocNotesSeparators(	RtfWritingContext *	rwc,
					const BufferDocument *	bd )
    {
    const int			evenIfAbsent= 0;
    const int			forcePar= 0;

    if  ( docRtfSaveExternalItem( rwc, "\\ftnsep", bd->bdEiFtnsep.eiRoot,
						    evenIfAbsent, forcePar ) )
	{ LDEB(1); return -1;	}

    if  ( docRtfSaveExternalItem( rwc, "\\ftnsepc", bd->bdEiFtnsepc.eiRoot,
						    evenIfAbsent, forcePar ) )
	{ LDEB(1); return -1;	}

    if  ( docRtfSaveExternalItem( rwc, "\\ftncn", bd->bdEiFtncn.eiRoot,
						    evenIfAbsent, forcePar ) )
	{ LDEB(1); return -1;	}

    /******/

    if  ( docRtfSaveExternalItem( rwc, "\\aftnsep", bd->bdEiAftnsep.eiRoot,
						    evenIfAbsent, forcePar ) )
	{ LDEB(1); return -1;	}

    if  ( docRtfSaveExternalItem( rwc, "\\aftnsepc", bd->bdEiAftnsepc.eiRoot,
						    evenIfAbsent, forcePar ) )
	{ LDEB(1); return -1;	}

    if  ( docRtfSaveExternalItem( rwc, "\\aftncn", bd->bdEiAftncn.eiRoot,
						    evenIfAbsent, forcePar ) )
	{ LDEB(1); return -1;	}

    return 0;
    }

/************************************************************************/
/*									*/
/*  Remember fields from dates and times.				*/
/*									*/
/************************************************************************/

int docRtfReadTimeField(	const RtfControlWord *	rcw,
				int			arg,
				RtfReadingContext *	rrc )
    {
    switch( rcw->rcwID )
	{
	case RTFidHR:
	    rrc->rrcTm.tm_hour= arg;
	    break;
	case RTFidMIN:
	    rrc->rrcTm.tm_min= arg;
	    break;
	case RTFidSEC:
	    rrc->rrcTm.tm_sec= arg;
	    break;
	case RTFidDY:
	    rrc->rrcTm.tm_mday= arg;
	    break;
	case RTFidMO:
	    rrc->rrcTm.tm_mon= arg- 1;
	    break;
	case RTFidYR:
	    rrc->rrcTm.tm_year= arg- 1900;
	    break;
	}

    return 0;
    }

