/************************************************************************/
/*									*/
/*  Buffer administration routines.					*/
/*									*/
/************************************************************************/

#   include	"docBufConfig.h"

#   include	<stdlib.h>
#   include	<string.h>
#   include	<stdio.h>

#   include	<appDebugon.h>

#   include	<appUnit.h>
#   include	"docRowProperties.h"

/************************************************************************/
/*									*/
/*  Initialise a BufferItem.						*/
/*									*/
/************************************************************************/

void docCleanInitRowProperties(	RowProperties *	rp )
    {
    docCleanRowProperties( rp );
    docInitRowProperties( rp );
    }

void docCleanRowProperties(	RowProperties *	rp )
    {
    if  ( rp->rpCells )
	{ free( rp->rpCells );	}
    }

void docInitRowProperties(	RowProperties *	rp )
    {
    rp->rpCellCount= 0;
    rp->rpCells= (CellProperties *)0;

    rp->rpHalfGapWidthTwips= 0;
    rp->rpHeightTwips= 0;
    rp->rpLeftIndentTwips= 0;

    rp->rpIsTableHeader= 0;
    rp->rpKeepOnOnePage= 0;
    rp->rp_Keepfollow= 0;
    rp->rpAutofit= 0;

    rp->rpTopBorderNumber= 0;
    rp->rpBottomBorderNumber= 0;
    rp->rpLeftBorderNumber= 0;
    rp->rpRightBorderNumber= 0;
    rp->rpHorizontalBorderNumber= 0;
    rp->rpVerticalBorderNumber= 0;

    rp->rpShadingNumber= 0;
    rp->rpFrameNumber= 0;

    rp->rpAlignment= DOCiaLEFT;

    /**/
    rp->rpPreferredWidth= 0;
    rp->rpPreferredWidthUnit= TRautoNONE;

    /**/
    rp->rpLeftDefaultCellSpacing= 0;
    rp->rpRightDefaultCellSpacing= 0;
    rp->rpTopDefaultCellSpacing= 0;
    rp->rpBottomDefaultCellSpacing= 0;

    rp->rpLeftDefaultCellSpacingUnit= TRautoNONE;
    rp->rpRightDefaultCellSpacingUnit= TRautoNONE;
    rp->rpTopDefaultCellSpacingUnit= TRautoNONE;
    rp->rpBottomDefaultCellSpacingUnit= TRautoNONE;

    /**/
    rp->rpLeftCellPadding= 0;
    rp->rpRightCellPadding= 0;
    rp->rpTopCellPadding= 0;
    rp->rpBottomCellPadding= 0;

    rp->rpLeftCellPaddingUnit= TRautoNONE;
    rp->rpRightCellPaddingUnit= TRautoNONE;
    rp->rpTopCellPaddingUnit= TRautoNONE;
    rp->rpBottomCellPaddingUnit= TRautoNONE;

    /**/
    rp->rpCellWidthBefore= 0;
    rp->rpCellWidthAfter= 0;

    rp->rpCellWidthBeforeUnit= TRautoNONE;
    rp->rpCellWidthAfterUnit= TRautoNONE;

    /**/
    rp->rpRowNumber= -1;
    rp->rpRowBandNumber= -1;
    rp->rpRowStyle= -1;

    rp->rpIsLastRow= 0;
    rp->rpAuthor= -1;

    return;
    }

int docInsertRowColumn(	RowProperties *			rp,
			int				n,
			const CellProperties *		cp,
			const DocumentAttributeMap *	dam )
    {
    CellProperties *	fresh;

    fresh= (CellProperties *)realloc( rp->rpCells,
			(rp->rpCellCount+ 1)* sizeof(CellProperties) );
    if  ( ! fresh )
	{ LXDEB(rp->rpCellCount,fresh); return -1;	}
    rp->rpCells= fresh;

    docInitCellProperties( fresh+ rp->rpCellCount );

    if  ( n < 0 )
	{ n= rp->rpCellCount;	}
    else{
	int		i;

	for ( i= rp->rpCellCount; i > n; i-- )
	    { docCopyCellProperties( fresh+ i, fresh+ i- 1, dam );	}
	}

    docCopyCellProperties( fresh+ n, cp, dam );

    rp->rpCellCount++;

    return 0;
    }

/************************************************************************/
/*									*/
/*  Set the width of a range of columns. Space might have been stolen	*/
/*  from a victim column that has become narrower. Also adjust its	*/
/*  width.								*/
/*									*/
/************************************************************************/

void docRowPropertiesSetWidth(		RowProperties *		rp,
					int			col0,
					int			col1,
					int			wide,
					int			victim,
					int			victimWide )
    {
    CellProperties *	cp;
    int			col;

    int			ox0= rp->rpLeftIndentTwips;
    int			nx0= rp->rpLeftIndentTwips;

    cp= rp->rpCells;
    for ( col= 0; col < col0; cp++, col++ )
	{
	int		ox1= cp->cpRightBoundaryTwips;
	int		w= cp->cpRightBoundaryTwips- ox0;

	if  ( col == victim )
	    { w= victimWide;	}

	ox0= ox1;
	nx0= cp->cpRightBoundaryTwips= nx0+ w;
	}

    for ( col= col0; col <= col1; cp++, col++ )
	{
	int		ox1= cp->cpRightBoundaryTwips;

	ox0= ox1;
	nx0= cp->cpRightBoundaryTwips= nx0+ wide;
	}

    for ( col= col1+ 1; col < rp->rpCellCount; cp++, col++ )
	{
	int		ox1= cp->cpRightBoundaryTwips;
	int		w= cp->cpRightBoundaryTwips- ox0;

	if  ( col == victim )
	    { w= victimWide;	}

	ox0= ox1;
	nx0= cp->cpRightBoundaryTwips= nx0+ w;
	}

    return;
    }

/************************************************************************/
/*									*/
/*  Delete a number of columns from row properties.			*/
/*									*/
/************************************************************************/

int docDeleteColumnsFromRow(	RowProperties *		rp,
				int			col0,
				int			count )
    {
    int		col1= col0+ count- 1;
    int		left;
    int		wide;
    int		col;

    if  ( col0+ count > rp->rpCellCount )
	{ LLLDEB(col0,count,rp->rpCellCount); return -1;	}

    if  ( col0 == 0 )
	{ left= rp->rpLeftIndentTwips;				}
    else{ left= rp->rpCells[col0-1].cpRightBoundaryTwips;	}

    wide= rp->rpCells[col1].cpRightBoundaryTwips- left;

    rp->rpCellCount -= count;

    for ( col= col0; col < rp->rpCellCount; col++ )
	{
	docCleanCellProperties( &(rp->rpCells[col]) );
	rp->rpCells[col]= rp->rpCells[col+ count];
	rp->rpCells[col].cpRightBoundaryTwips -= wide;
	}

    return 0;
    }

/************************************************************************/
/*									*/
/*  Copy row properties. As this routine is not called very often,	*/
/*  just call the 'update' routine.					*/
/*									*/
/************************************************************************/

int docCopyRowProperties(	RowProperties *			rpTo,
				const RowProperties *		rpFrom,
				const DocumentAttributeMap *	dam )
    {
    PropertyMask		rpDoneMask;
    PropertyMask		rpSetMask;

    utilPropMaskClear( &rpDoneMask );
    utilPropMaskClear( &rpSetMask );

    utilPropMaskFill( &rpSetMask, RPprop_FULL_COUNT );

    if  ( docUpdRowProperties( &rpDoneMask, rpTo, &rpSetMask, rpFrom, dam ) )
	{ LDEB(1); return -1;	}

    return 0;
    }

/************************************************************************/
/*									*/
/*  1)  Are the columns in two RowProperties 'the same' (Do they	*/
/*	align?)								*/
/*  2)  All internal borders equal?					*/
/*  3)  All column properties identical?				*/
/*									*/
/************************************************************************/

/*  1  */
int docApproximatelyAlignedColumns(	const RowProperties *	rp1,
					const RowProperties *	rp2 )
    {
    CellProperties *	cp1;
    CellProperties *	cp2;
    int			i;

    const int		D= 40;

    if  ( rp1->rpCellCount != rp2->rpCellCount )
	{ return 0;	}

    /* No!
    if  ( rp1->rpHalfGapWidthTwips != rp2->rpHalfGapWidthTwips )
	{ return 0;	}
    */

    if  ( rp1->rpLeftIndentTwips > rp2->rpLeftIndentTwips+ D	||
	  rp1->rpLeftIndentTwips < rp2->rpLeftIndentTwips- D	)
	{ return 0;	}

    cp1= rp1->rpCells;
    cp2= rp2->rpCells;
    for ( i= 0; i < rp1->rpCellCount; cp2++, cp1++, i++ )
	{
	if  ( cp1->cpRightBoundaryTwips > cp2->cpRightBoundaryTwips+ D	||
	      cp1->cpRightBoundaryTwips < cp2->cpRightBoundaryTwips- D	)
	    { return 0;	}
	}

    return 1;
    }

/*  2  */
int docEqualRowBorders(	const RowProperties *	rp1,
			const RowProperties *	rp2 )
    {
    PropertyMask	rpChgMask;
    PropertyMask	rpUpdMask;

    utilPropMaskClear( &rpUpdMask );
    PROPmaskADD( &rpUpdMask, RPpropTOP_BORDER );
    PROPmaskADD( &rpUpdMask, RPpropBOTTOM_BORDER );
    PROPmaskADD( &rpUpdMask, RPpropLEFT_BORDER );
    PROPmaskADD( &rpUpdMask, RPpropRIGHT_BORDER );
    PROPmaskADD( &rpUpdMask, RPpropHORIZ_BORDER );
    PROPmaskADD( &rpUpdMask, RPpropVERT_BORDER );
    utilPropMaskClear( &rpChgMask );

    docRowPropertyDifference( &rpChgMask, rp1, rp2, &rpUpdMask,
					    (const DocumentAttributeMap *)0 );

    if  ( ! utilPropMaskIsEmpty( &rpChgMask ) )
	{ return 0;	}

    return 1;
    }

/*  3  */
int docEqualRows(	const RowProperties *	rp1,
			const RowProperties *	rp2 )
    {
    PropertyMask	rpChgMask;
    PropertyMask	rpUpdMask;

    utilPropMaskFill( &rpUpdMask, RPprop_FULL_COUNT );
    utilPropMaskClear( &rpChgMask );

    docRowPropertyDifference( &rpChgMask, rp1, rp2, &rpUpdMask,
					    (const DocumentAttributeMap *)0 );

    if  ( ! utilPropMaskIsEmpty( &rpChgMask ) )
	{ return 0;	}

    return 1;
    }

/************************************************************************/
/*									*/
/*  Change row properties and tell what has been changed.		*/
/*									*/
/************************************************************************/

int docUpdRowProperties(	PropertyMask *			pRpDonePask,
				RowProperties *			rpTo,
				const PropertyMask *		rpUpdMask,
				const RowProperties *		rpFrom,
				const DocumentAttributeMap *	dam )
    {
    PropertyMask		rpDoneMask;

    utilPropMaskClear( &rpDoneMask );

    if  ( PROPmaskISSET( rpUpdMask, RPpropCOLUMNS ) )
	{
	int			col;
	CellProperties *	cpTo;
	const CellProperties *	cpFrom;

	while( rpTo->rpCellCount > rpFrom->rpCellCount )
	    {
	    docCleanCellProperties( &(rpTo->rpCells[rpTo->rpCellCount-1]) );
	    PROPmaskADD( &rpDoneMask, RPpropCOLUMNS );
	    rpTo->rpCellCount--;
	    }

	cpTo= rpTo->rpCells;
	cpFrom= rpFrom->rpCells;
	for ( col= 0; col < rpTo->rpCellCount; cpTo++, cpFrom++, col++ )
	    {
	    PropertyMask	cpDoneMask;
	    PropertyMask	cpUpdMask;

	    utilPropMaskFill( &cpUpdMask, CLprop_COUNT );
	    utilPropMaskClear( &cpDoneMask );

	    if  ( docUpdCellProperties( &cpDoneMask, cpTo,
				    &cpUpdMask, cpFrom, dam ) )
		{ LDEB(1); return -1;	}

	    if  ( ! utilPropMaskIsEmpty( &cpDoneMask ) )
		{ PROPmaskADD( &rpDoneMask, RPpropCOLUMNS );	}
	    }

	while( rpTo->rpCellCount < rpFrom->rpCellCount )
	    {
	    if  ( docInsertRowColumn( rpTo, rpTo->rpCellCount, cpFrom, dam ) )
		{ LDEB(rpTo->rpCellCount); return -1;	}

	    PROPmaskADD( &rpDoneMask, RPpropCOLUMNS );
	    cpFrom++;
	    }
	}

    if  ( PROPmaskISSET( rpUpdMask, RPpropGAP_WIDTH ) )
	{
	if  ( rpTo->rpHalfGapWidthTwips != rpFrom->rpHalfGapWidthTwips )
	    {
	    rpTo->rpHalfGapWidthTwips= rpFrom->rpHalfGapWidthTwips;
	    PROPmaskADD( &rpDoneMask, RPpropGAP_WIDTH );
	    }
	}

    if  ( PROPmaskISSET( rpUpdMask, RPpropLEFT_INDENT ) )
	{
	if  ( rpTo->rpLeftIndentTwips != rpFrom->rpLeftIndentTwips )
	    {
	    rpTo->rpLeftIndentTwips= rpFrom->rpLeftIndentTwips;
	    PROPmaskADD( &rpDoneMask, RPpropLEFT_INDENT );
	    }
	}

    if  ( PROPmaskISSET( rpUpdMask, RPpropHEIGHT ) )
	{
	if  ( rpTo->rpHeightTwips != rpFrom->rpHeightTwips )
	    {
	    rpTo->rpHeightTwips= rpFrom->rpHeightTwips;
	    PROPmaskADD( &rpDoneMask, RPpropHEIGHT );
	    }
	}

    if  ( PROPmaskISSET( rpUpdMask, RPpropTOP_BORDER ) )
	{
	int	fromNumber= rpFrom->rpTopBorderNumber;

	if  ( fromNumber >= 0 && dam && dam->damBorderMap )
	    { fromNumber= dam->damBorderMap[fromNumber];	}

	if  ( rpTo->rpTopBorderNumber != fromNumber )
	    {
	    rpTo->rpTopBorderNumber= fromNumber;
	    PROPmaskADD( &rpDoneMask, RPpropTOP_BORDER );
	    }
	}

    if  ( PROPmaskISSET( rpUpdMask, RPpropBOTTOM_BORDER ) )
	{
	int	fromNumber= rpFrom->rpBottomBorderNumber;

	if  ( fromNumber >= 0 && dam && dam->damBorderMap )
	    { fromNumber= dam->damBorderMap[fromNumber];	}

	if  ( rpTo->rpBottomBorderNumber != fromNumber )
	    {
	    rpTo->rpBottomBorderNumber= fromNumber;
	    PROPmaskADD( &rpDoneMask, RPpropBOTTOM_BORDER );
	    }
	}

    if  ( PROPmaskISSET( rpUpdMask, RPpropLEFT_BORDER ) )
	{
	int	fromNumber= rpFrom->rpLeftBorderNumber;

	if  ( fromNumber >= 0 && dam && dam->damBorderMap )
	    { fromNumber= dam->damBorderMap[fromNumber];	}

	if  ( rpTo->rpLeftBorderNumber != fromNumber )
	    {
	    rpTo->rpLeftBorderNumber= fromNumber;
	    PROPmaskADD( &rpDoneMask, RPpropLEFT_BORDER );
	    }
	}

    if  ( PROPmaskISSET( rpUpdMask, RPpropRIGHT_BORDER ) )
	{
	int	fromNumber= rpFrom->rpRightBorderNumber;

	if  ( fromNumber >= 0 && dam && dam->damBorderMap )
	    { fromNumber= dam->damBorderMap[fromNumber];	}

	if  ( rpTo->rpRightBorderNumber != fromNumber )
	    {
	    rpTo->rpRightBorderNumber= fromNumber;
	    PROPmaskADD( &rpDoneMask, RPpropRIGHT_BORDER );
	    }
	}

    if  ( PROPmaskISSET( rpUpdMask, RPpropHORIZ_BORDER ) )
	{
	int	fromNumber= rpFrom->rpHorizontalBorderNumber;

	if  ( fromNumber >= 0 && dam && dam->damBorderMap )
	    { fromNumber= dam->damBorderMap[fromNumber];	}

	if  ( rpTo->rpHorizontalBorderNumber != fromNumber )
	    {
	    rpTo->rpHorizontalBorderNumber= fromNumber;
	    PROPmaskADD( &rpDoneMask, RPpropHORIZ_BORDER );
	    }
	}

    if  ( PROPmaskISSET( rpUpdMask, RPpropVERT_BORDER ) )
	{
	int	fromNumber= rpFrom->rpVerticalBorderNumber;

	if  ( fromNumber >= 0 && dam && dam->damBorderMap )
	    { fromNumber= dam->damBorderMap[fromNumber];	}

	if  ( rpTo->rpVerticalBorderNumber != fromNumber )
	    {
	    rpTo->rpVerticalBorderNumber= fromNumber;
	    PROPmaskADD( &rpDoneMask, RPpropVERT_BORDER );
	    }
	}

    if  ( PROPmaskISSET( rpUpdMask, RPpropSHADING ) )
	{
	int	fromNumber= rpFrom->rpShadingNumber;

	if  ( fromNumber >= 0 && dam && dam->damShadingMap )
	    { fromNumber= dam->damShadingMap[fromNumber];	}

	if  ( rpTo->rpShadingNumber != fromNumber )
	    {
	    rpTo->rpShadingNumber= fromNumber;
	    PROPmaskADD( &rpDoneMask, RPpropSHADING );
	    }
	}

    if  ( PROPmaskISSET( rpUpdMask, RPpropFRAME ) )
	{
	int	fromNumber= rpFrom->rpFrameNumber;

	if  ( fromNumber >= 0 && dam && dam->damFrameMap )
	    { fromNumber= dam->damFrameMap[fromNumber];	}

	if  ( rpTo->rpFrameNumber != fromNumber )
	    {
	    rpTo->rpFrameNumber= fromNumber;
	    PROPmaskADD( &rpDoneMask, RPpropFRAME );
	    }
	}

    if  ( PROPmaskISSET( rpUpdMask, RPpropALIGNMENT ) )
	{
	if  ( rpTo->rpAlignment != rpFrom->rpAlignment )
	    {
	    rpTo->rpAlignment= rpFrom->rpAlignment;
	    PROPmaskADD( &rpDoneMask, RPpropALIGNMENT );
	    }
	}

    if  ( PROPmaskISSET( rpUpdMask, RPpropIS_TABLE_HEADER ) )
	{
	if  ( rpTo->rpIsTableHeader != rpFrom->rpIsTableHeader )
	    {
	    rpTo->rpIsTableHeader= rpFrom->rpIsTableHeader;
	    PROPmaskADD( &rpDoneMask, RPpropIS_TABLE_HEADER );
	    }
	}

    if  ( PROPmaskISSET( rpUpdMask, RPpropKEEP_ON_ONE_PAGE ) )
	{
	if  ( rpTo->rpKeepOnOnePage != rpFrom->rpKeepOnOnePage )
	    {
	    rpTo->rpKeepOnOnePage= rpFrom->rpKeepOnOnePage;
	    PROPmaskADD( &rpDoneMask, RPpropKEEP_ON_ONE_PAGE );
	    }
	}

    if  ( PROPmaskISSET( rpUpdMask, RPprop_KEEPFOLLOW ) )
	{
	if  ( rpTo->rp_Keepfollow != rpFrom->rp_Keepfollow )
	    {
	    rpTo->rp_Keepfollow= rpFrom->rp_Keepfollow;
	    PROPmaskADD( &rpDoneMask, RPprop_KEEPFOLLOW );
	    }
	}

    if  ( PROPmaskISSET( rpUpdMask, RPpropAUTOFIT ) )
	{
	if  ( rpTo->rpAutofit != rpFrom->rpAutofit )
	    {
	    rpTo->rpAutofit= rpFrom->rpAutofit;
	    PROPmaskADD( &rpDoneMask, RPpropAUTOFIT );
	    }
	}

    /**/
    if  ( PROPmaskISSET( rpUpdMask, RPpropTRW_WIDTH ) )
	{
	if  ( rpTo->rpPreferredWidth != rpFrom->rpPreferredWidth )
	    {
	    rpTo->rpPreferredWidth= rpFrom->rpPreferredWidth;
	    PROPmaskADD( &rpDoneMask, RPpropTRW_WIDTH );
	    }
	}

    if  ( PROPmaskISSET( rpUpdMask, RPpropTRFTS_WIDTH ) )
	{
	if  ( rpTo->rpPreferredWidthUnit != rpFrom->rpPreferredWidthUnit )
	    {
	    rpTo->rpPreferredWidthUnit= rpFrom->rpPreferredWidthUnit;
	    PROPmaskADD( &rpDoneMask, RPpropTRFTS_WIDTH );
	    }
	}

    /**/
    if  ( PROPmaskISSET( rpUpdMask, RPpropTRSPDL ) )
	{
	if  ( rpTo->rpLeftDefaultCellSpacing !=
				      rpFrom->rpLeftDefaultCellSpacing )
	    {
	    rpTo->rpLeftDefaultCellSpacing= rpFrom->rpLeftDefaultCellSpacing;
	    PROPmaskADD( &rpDoneMask, RPpropTRSPDL );
	    }
	}

    if  ( PROPmaskISSET( rpUpdMask, RPpropTRSPDR ) )
	{
	if  ( rpTo->rpRightDefaultCellSpacing !=
				    rpFrom->rpRightDefaultCellSpacing )
	    {
	    rpTo->rpRightDefaultCellSpacing= rpFrom->rpRightDefaultCellSpacing;
	    PROPmaskADD( &rpDoneMask, RPpropTRSPDR );
	    }
	}

    if  ( PROPmaskISSET( rpUpdMask, RPpropTRSPDT ) )
	{
	if  ( rpTo->rpTopDefaultCellSpacing !=
				    rpFrom->rpTopDefaultCellSpacing )
	    {
	    rpTo->rpTopDefaultCellSpacing= rpFrom->rpTopDefaultCellSpacing;
	    PROPmaskADD( &rpDoneMask, RPpropTRSPDT );
	    }
	}

    if  ( PROPmaskISSET( rpUpdMask, RPpropTRSPDB ) )
	{
	if  ( rpTo->rpBottomDefaultCellSpacing !=
				    rpFrom->rpBottomDefaultCellSpacing )
	    {
	    rpTo->rpBottomDefaultCellSpacing= rpFrom->rpBottomDefaultCellSpacing;
	    PROPmaskADD( &rpDoneMask, RPpropTRSPDB );
	    }
	}

    /**/
    if  ( PROPmaskISSET( rpUpdMask, RPpropTRSPDFL ) )
	{
	if  ( rpTo->rpLeftDefaultCellSpacingUnit !=
				      rpFrom->rpLeftDefaultCellSpacingUnit )
	    {
	    rpTo->rpLeftDefaultCellSpacingUnit=
				    rpFrom->rpLeftDefaultCellSpacingUnit;
	    PROPmaskADD( &rpDoneMask, RPpropTRSPDFL );
	    }
	}

    if  ( PROPmaskISSET( rpUpdMask, RPpropTRSPDFR ) )
	{
	if  ( rpTo->rpRightDefaultCellSpacingUnit !=
				    rpFrom->rpRightDefaultCellSpacingUnit )
	    {
	    rpTo->rpRightDefaultCellSpacingUnit=
				    rpFrom->rpRightDefaultCellSpacingUnit;
	    PROPmaskADD( &rpDoneMask, RPpropTRSPDFR );
	    }
	}

    if  ( PROPmaskISSET( rpUpdMask, RPpropTRSPDFT ) )
	{
	if  ( rpTo->rpTopDefaultCellSpacingUnit !=
				    rpFrom->rpTopDefaultCellSpacingUnit )
	    {
	    rpTo->rpTopDefaultCellSpacingUnit=
				    rpFrom->rpTopDefaultCellSpacingUnit;
	    PROPmaskADD( &rpDoneMask, RPpropTRSPDFT );
	    }
	}

    if  ( PROPmaskISSET( rpUpdMask, RPpropTRSPDFB ) )
	{
	if  ( rpTo->rpBottomDefaultCellSpacingUnit !=
				    rpFrom->rpBottomDefaultCellSpacingUnit )
	    {
	    rpTo->rpBottomDefaultCellSpacingUnit=
				    rpFrom->rpBottomDefaultCellSpacingUnit;
	    PROPmaskADD( &rpDoneMask, RPpropTRSPDFB );
	    }
	}

    /**/
    if  ( PROPmaskISSET( rpUpdMask, RPpropTRPADDL ) )
	{
	if  ( rpTo->rpLeftCellPadding !=
				      rpFrom->rpLeftCellPadding )
	    {
	    rpTo->rpLeftCellPadding= rpFrom->rpLeftCellPadding;
	    PROPmaskADD( &rpDoneMask, RPpropTRPADDL );
	    }
	}

    if  ( PROPmaskISSET( rpUpdMask, RPpropTRPADDR ) )
	{
	if  ( rpTo->rpRightCellPadding !=
				    rpFrom->rpRightCellPadding )
	    {
	    rpTo->rpRightCellPadding= rpFrom->rpRightCellPadding;
	    PROPmaskADD( &rpDoneMask, RPpropTRPADDR );
	    }
	}

    if  ( PROPmaskISSET( rpUpdMask, RPpropTRPADDT ) )
	{
	if  ( rpTo->rpTopCellPadding !=
				    rpFrom->rpTopCellPadding )
	    {
	    rpTo->rpTopCellPadding= rpFrom->rpTopCellPadding;
	    PROPmaskADD( &rpDoneMask, RPpropTRPADDT );
	    }
	}

    if  ( PROPmaskISSET( rpUpdMask, RPpropTRPADDB ) )
	{
	if  ( rpTo->rpBottomCellPadding !=
				    rpFrom->rpBottomCellPadding )
	    {
	    rpTo->rpBottomCellPadding= rpFrom->rpBottomCellPadding;
	    PROPmaskADD( &rpDoneMask, RPpropTRPADDB );
	    }
	}

    /**/
    if  ( PROPmaskISSET( rpUpdMask, RPpropTRPADDFL ) )
	{
	if  ( rpTo->rpLeftCellPaddingUnit !=
				      rpFrom->rpLeftCellPaddingUnit )
	    {
	    rpTo->rpLeftCellPaddingUnit=
				    rpFrom->rpLeftCellPaddingUnit;
	    PROPmaskADD( &rpDoneMask, RPpropTRPADDFL );
	    }
	}

    if  ( PROPmaskISSET( rpUpdMask, RPpropTRPADDFR ) )
	{
	if  ( rpTo->rpRightCellPaddingUnit !=
				    rpFrom->rpRightCellPaddingUnit )
	    {
	    rpTo->rpRightCellPaddingUnit=
				    rpFrom->rpRightCellPaddingUnit;
	    PROPmaskADD( &rpDoneMask, RPpropTRPADDFR );
	    }
	}

    if  ( PROPmaskISSET( rpUpdMask, RPpropTRPADDFT ) )
	{
	if  ( rpTo->rpTopCellPaddingUnit !=
				    rpFrom->rpTopCellPaddingUnit )
	    {
	    rpTo->rpTopCellPaddingUnit=
				    rpFrom->rpTopCellPaddingUnit;
	    PROPmaskADD( &rpDoneMask, RPpropTRPADDFT );
	    }
	}

    if  ( PROPmaskISSET( rpUpdMask, RPpropTRPADDFB ) )
	{
	if  ( rpTo->rpBottomCellPaddingUnit !=
				    rpFrom->rpBottomCellPaddingUnit )
	    {
	    rpTo->rpBottomCellPaddingUnit=
				    rpFrom->rpBottomCellPaddingUnit;
	    PROPmaskADD( &rpDoneMask, RPpropTRPADDFB );
	    }
	}

    /**/
    if  ( PROPmaskISSET( rpUpdMask, RPpropTRW_WIDTHB ) )
	{
	if  ( rpTo->rpCellWidthBefore != rpFrom->rpCellWidthBefore )
	    {
	    rpTo->rpCellWidthBefore= rpFrom->rpCellWidthBefore;
	    PROPmaskADD( &rpDoneMask, RPpropTRW_WIDTHB );
	    }
	}

    if  ( PROPmaskISSET( rpUpdMask, RPpropTRW_WIDTHA ) )
	{
	if  ( rpTo->rpCellWidthAfter != rpFrom->rpCellWidthAfter )
	    {
	    rpTo->rpCellWidthAfter= rpFrom->rpCellWidthAfter;
	    PROPmaskADD( &rpDoneMask, RPpropTRW_WIDTHA );
	    }
	}

    /**/
    if  ( PROPmaskISSET( rpUpdMask, RPpropTRFTS_WIDTHB ) )
	{
	if  ( rpTo->rpCellWidthBeforeUnit != rpFrom->rpCellWidthBeforeUnit )
	    {
	    rpTo->rpCellWidthBeforeUnit= rpFrom->rpCellWidthBeforeUnit;
	    PROPmaskADD( &rpDoneMask, RPpropTRFTS_WIDTHB );
	    }
	}

    if  ( PROPmaskISSET( rpUpdMask, RPpropTRFTS_WIDTHA ) )
	{
	if  ( rpTo->rpCellWidthAfterUnit != rpFrom->rpCellWidthAfterUnit )
	    {
	    rpTo->rpCellWidthAfterUnit= rpFrom->rpCellWidthAfterUnit;
	    PROPmaskADD( &rpDoneMask, RPpropTRFTS_WIDTHA );
	    }
	}

    /**/
    if  ( PROPmaskISSET( rpUpdMask, RPpropROW_NUMBER ) )
	{
	if  ( rpTo->rpRowNumber != rpFrom->rpRowNumber )
	    {
	    rpTo->rpRowNumber= rpFrom->rpRowNumber;
	    PROPmaskADD( &rpDoneMask, RPpropROW_NUMBER );
	    }
	}

    if  ( PROPmaskISSET( rpUpdMask, RPpropROW_BAND_NUMBER ) )
	{
	if  ( rpTo->rpRowBandNumber != rpFrom->rpRowBandNumber )
	    {
	    rpTo->rpRowBandNumber= rpFrom->rpRowBandNumber;
	    PROPmaskADD( &rpDoneMask, RPpropROW_BAND_NUMBER );
	    }
	}

    if  ( PROPmaskISSET( rpUpdMask, RPpropROW_STYLE ) )
	{
	if  ( rpTo->rpRowStyle != rpFrom->rpRowStyle )
	    {
	    rpTo->rpRowStyle= rpFrom->rpRowStyle;
	    PROPmaskADD( &rpDoneMask, RPpropROW_STYLE );
	    }
	}

    /**/
    if  ( PROPmaskISSET( rpUpdMask, RPpropIS_LAST_ROW ) )
	{
	if  ( rpTo->rpIsLastRow != rpFrom->rpIsLastRow )
	    {
	    rpTo->rpIsLastRow= rpFrom->rpIsLastRow;
	    PROPmaskADD( &rpDoneMask, RPpropIS_LAST_ROW );
	    }
	}

    /**/
    if  ( PROPmaskISSET( rpUpdMask, RPpropTRAUTH ) )
	{
	if  ( rpTo->rpAuthor != rpFrom->rpAuthor )
	    {
	    rpTo->rpAuthor= rpFrom->rpAuthor;
	    PROPmaskADD( &rpDoneMask, RPpropTRAUTH );
	    }
	}

    /**/

    *pRpDonePask= rpDoneMask; return 0;
    }

/************************************************************************/
/*									*/
/*  Compare row properties.						*/
/*									*/
/************************************************************************/

void docRowPropertyDifference(	PropertyMask *			pRpDiffPask,
				const RowProperties *		rp1,
				const RowProperties *		rp2,
				const PropertyMask *		rpUpdMask,
				const DocumentAttributeMap *	dam )
    {
    PropertyMask		rpDiffMask;

    utilPropMaskClear( &rpDiffMask );

    if  ( PROPmaskISSET( rpUpdMask, RPpropCOLUMNS ) )
	{
	if  ( rp1->rpCellCount != rp2->rpCellCount )
	    { PROPmaskADD( &rpDiffMask, RPpropCOLUMNS );	}
	else{
	    int				col;
	    const CellProperties *	cp1;
	    const CellProperties *	cp2;

	    cp1= rp1->rpCells;
	    cp2= rp2->rpCells;
	    for ( col= 0; col < rp1->rpCellCount; cp1++, cp2++, col++ )
		{
		PropertyMask	cpDiffMask;
		PropertyMask	cpUpdMask;

		utilPropMaskFill( &cpUpdMask, CLprop_COUNT );
		utilPropMaskClear( &cpDiffMask );

		docCellPropertyDifference( &cpDiffMask, cp1, cp2, &cpUpdMask,
									dam );

		if  ( ! utilPropMaskIsEmpty( &cpDiffMask ) )
		    {
		    PROPmaskADD( &rpDiffMask, RPpropCOLUMNS );
		    break;
		    }
		}
	    }
	}

    if  ( PROPmaskISSET( rpUpdMask, RPpropGAP_WIDTH ) )
	{
	if  ( rp1->rpHalfGapWidthTwips != rp2->rpHalfGapWidthTwips )
	    { PROPmaskADD( &rpDiffMask, RPpropGAP_WIDTH );	}
	}

    if  ( PROPmaskISSET( rpUpdMask, RPpropLEFT_INDENT ) )
	{
	if  ( rp1->rpLeftIndentTwips != rp2->rpLeftIndentTwips )
	    { PROPmaskADD( &rpDiffMask, RPpropLEFT_INDENT );	}
	}

    if  ( PROPmaskISSET( rpUpdMask, RPpropHEIGHT ) )
	{
	if  ( rp1->rpHeightTwips != rp2->rpHeightTwips )
	    { PROPmaskADD( &rpDiffMask, RPpropHEIGHT );	}
	}

    /**/
    if  ( PROPmaskISSET( rpUpdMask, RPpropTOP_BORDER ) )
	{
	int	fromNumber= rp2->rpTopBorderNumber;

	if  ( fromNumber >= 0 && dam && dam->damBorderMap )
	    { fromNumber= dam->damBorderMap[fromNumber];	}

	if  ( rp1->rpTopBorderNumber != fromNumber )
	    { PROPmaskADD( &rpDiffMask, RPpropTOP_BORDER ); }
	}

    if  ( PROPmaskISSET( rpUpdMask, RPpropBOTTOM_BORDER ) )
	{
	int	fromNumber= rp2->rpBottomBorderNumber;

	if  ( fromNumber >= 0 && dam && dam->damBorderMap )
	    { fromNumber= dam->damBorderMap[fromNumber];	}

	if  ( rp1->rpBottomBorderNumber != fromNumber )
	    { PROPmaskADD( &rpDiffMask, RPpropBOTTOM_BORDER ); }
	}

    if  ( PROPmaskISSET( rpUpdMask, RPpropLEFT_BORDER ) )
	{
	int	fromNumber= rp2->rpLeftBorderNumber;

	if  ( fromNumber >= 0 && dam && dam->damBorderMap )
	    { fromNumber= dam->damBorderMap[fromNumber];	}

	if  ( rp1->rpLeftBorderNumber != fromNumber )
	    { PROPmaskADD( &rpDiffMask, RPpropLEFT_BORDER ); }
	}

    if  ( PROPmaskISSET( rpUpdMask, RPpropRIGHT_BORDER ) )
	{
	int	fromNumber= rp2->rpRightBorderNumber;

	if  ( fromNumber >= 0 && dam && dam->damBorderMap )
	    { fromNumber= dam->damBorderMap[fromNumber];	}

	if  ( rp1->rpRightBorderNumber != fromNumber )
	    { PROPmaskADD( &rpDiffMask, RPpropRIGHT_BORDER ); }
	}

    if  ( PROPmaskISSET( rpUpdMask, RPpropHORIZ_BORDER ) )
	{
	int	fromNumber= rp2->rpHorizontalBorderNumber;

	if  ( fromNumber >= 0 && dam && dam->damBorderMap )
	    { fromNumber= dam->damBorderMap[fromNumber];	}

	if  ( rp1->rpHorizontalBorderNumber != fromNumber )
	    { PROPmaskADD( &rpDiffMask, RPpropHORIZ_BORDER ); }
	}

    if  ( PROPmaskISSET( rpUpdMask, RPpropVERT_BORDER ) )
	{
	int	fromNumber= rp2->rpVerticalBorderNumber;

	if  ( fromNumber >= 0 && dam && dam->damBorderMap )
	    { fromNumber= dam->damBorderMap[fromNumber];	}

	if  ( rp1->rpVerticalBorderNumber != fromNumber )
	    { PROPmaskADD( &rpDiffMask, RPpropVERT_BORDER ); }
	}

    /**/
    if  ( PROPmaskISSET( rpUpdMask, RPpropSHADING ) )
	{
	int	fromNumber= rp2->rpShadingNumber;

	if  ( fromNumber >= 0 && dam && dam->damShadingMap )
	    { fromNumber= dam->damShadingMap[fromNumber];	}

	if  ( rp1->rpShadingNumber != fromNumber )
	    { PROPmaskADD( &rpDiffMask, RPpropSHADING );	}
	}

    if  ( PROPmaskISSET( rpUpdMask, RPpropFRAME ) )
	{
	int	fromNumber= rp2->rpFrameNumber;

	if  ( fromNumber >= 0 && dam && dam->damFrameMap )
	    { fromNumber= dam->damFrameMap[fromNumber];	}

	if  ( rp1->rpFrameNumber != fromNumber )
	    { PROPmaskADD( &rpDiffMask, RPpropFRAME );	}
	}

    /**/
    if  ( PROPmaskISSET( rpUpdMask, RPpropALIGNMENT ) )
	{
	if  ( rp1->rpAlignment != rp2->rpAlignment )
	    { PROPmaskADD( &rpDiffMask, RPpropALIGNMENT ); }
	}

    if  ( PROPmaskISSET( rpUpdMask, RPpropIS_TABLE_HEADER ) )
	{
	if  ( rp1->rpIsTableHeader != rp2->rpIsTableHeader )
	    { PROPmaskADD( &rpDiffMask, RPpropIS_TABLE_HEADER ); }
	}

    if  ( PROPmaskISSET( rpUpdMask, RPpropKEEP_ON_ONE_PAGE ) )
	{
	if  ( rp1->rpKeepOnOnePage != rp2->rpKeepOnOnePage )
	    { PROPmaskADD( &rpDiffMask, RPpropKEEP_ON_ONE_PAGE ); }
	}

    if  ( PROPmaskISSET( rpUpdMask, RPprop_KEEPFOLLOW ) )
	{
	if  ( rp1->rp_Keepfollow != rp2->rp_Keepfollow )
	    { PROPmaskADD( &rpDiffMask, RPprop_KEEPFOLLOW ); }
	}

    if  ( PROPmaskISSET( rpUpdMask, RPpropAUTOFIT ) )
	{
	if  ( rp1->rpAutofit != rp2->rpAutofit )
	    { PROPmaskADD( &rpDiffMask, RPpropAUTOFIT ); }
	}

    /**/
    if  ( PROPmaskISSET( rpUpdMask, RPpropTRW_WIDTH ) )
	{
	if  ( rp1->rpPreferredWidth != rp2->rpPreferredWidth )
	    { PROPmaskADD( &rpDiffMask, RPpropTRW_WIDTH );	}
	}

    if  ( PROPmaskISSET( rpUpdMask, RPpropTRFTS_WIDTH ) )
	{
	if  ( rp1->rpPreferredWidthUnit != rp2->rpPreferredWidthUnit )
	    { PROPmaskADD( &rpDiffMask, RPpropTRFTS_WIDTH );	}
	}

    /**/
    if  ( PROPmaskISSET( rpUpdMask, RPpropTRSPDL ) )
	{
	if  ( rp1->rpLeftDefaultCellSpacing !=
				      rp2->rpLeftDefaultCellSpacing )
	    { PROPmaskADD( &rpDiffMask, RPpropTRSPDL ); }
	}

    if  ( PROPmaskISSET( rpUpdMask, RPpropTRSPDR ) )
	{
	if  ( rp1->rpRightDefaultCellSpacing !=
				    rp2->rpRightDefaultCellSpacing )
	    { PROPmaskADD( &rpDiffMask, RPpropTRSPDR ); }
	}

    if  ( PROPmaskISSET( rpUpdMask, RPpropTRSPDT ) )
	{
	if  ( rp1->rpTopDefaultCellSpacing !=
				    rp2->rpTopDefaultCellSpacing )
	    { PROPmaskADD( &rpDiffMask, RPpropTRSPDT ); }
	}

    if  ( PROPmaskISSET( rpUpdMask, RPpropTRSPDB ) )
	{
	if  ( rp1->rpBottomDefaultCellSpacing !=
				    rp2->rpBottomDefaultCellSpacing )
	    { PROPmaskADD( &rpDiffMask, RPpropTRSPDB ); }
	}

    /**/
    if  ( PROPmaskISSET( rpUpdMask, RPpropTRSPDFL ) )
	{
	if  ( rp1->rpLeftDefaultCellSpacingUnit !=
				      rp2->rpLeftDefaultCellSpacingUnit )
	    { PROPmaskADD( &rpDiffMask, RPpropTRSPDFL ); }
	}

    if  ( PROPmaskISSET( rpUpdMask, RPpropTRSPDFR ) )
	{
	if  ( rp1->rpRightDefaultCellSpacingUnit !=
				    rp2->rpRightDefaultCellSpacingUnit )
	    { PROPmaskADD( &rpDiffMask, RPpropTRSPDFR ); }
	}

    if  ( PROPmaskISSET( rpUpdMask, RPpropTRSPDFT ) )
	{
	if  ( rp1->rpTopDefaultCellSpacingUnit !=
				    rp2->rpTopDefaultCellSpacingUnit )
	    { PROPmaskADD( &rpDiffMask, RPpropTRSPDFT ); }
	}

    if  ( PROPmaskISSET( rpUpdMask, RPpropTRSPDFB ) )
	{
	if  ( rp1->rpBottomDefaultCellSpacingUnit !=
				    rp2->rpBottomDefaultCellSpacingUnit )
	    { PROPmaskADD( &rpDiffMask, RPpropTRSPDFB ); }
	}

    /**/
    if  ( PROPmaskISSET( rpUpdMask, RPpropTRPADDL ) )
	{
	if  ( rp1->rpLeftCellPadding !=
				      rp2->rpLeftCellPadding )
	    { PROPmaskADD( &rpDiffMask, RPpropTRPADDL ); }
	}

    if  ( PROPmaskISSET( rpUpdMask, RPpropTRPADDR ) )
	{
	if  ( rp1->rpRightCellPadding !=
				    rp2->rpRightCellPadding )
	    { PROPmaskADD( &rpDiffMask, RPpropTRPADDR ); }
	}

    if  ( PROPmaskISSET( rpUpdMask, RPpropTRPADDT ) )
	{
	if  ( rp1->rpTopCellPadding !=
				    rp2->rpTopCellPadding )
	    { PROPmaskADD( &rpDiffMask, RPpropTRPADDT ); }
	}

    if  ( PROPmaskISSET( rpUpdMask, RPpropTRPADDB ) )
	{
	if  ( rp1->rpBottomCellPadding !=
				    rp2->rpBottomCellPadding )
	    { PROPmaskADD( &rpDiffMask, RPpropTRPADDB ); }
	}

    /**/
    if  ( PROPmaskISSET( rpUpdMask, RPpropTRPADDFL ) )
	{
	if  ( rp1->rpLeftCellPaddingUnit !=
				      rp2->rpLeftCellPaddingUnit )
	    { PROPmaskADD( &rpDiffMask, RPpropTRPADDFL ); }
	}

    if  ( PROPmaskISSET( rpUpdMask, RPpropTRPADDFR ) )
	{
	if  ( rp1->rpRightCellPaddingUnit !=
				    rp2->rpRightCellPaddingUnit )
	    { PROPmaskADD( &rpDiffMask, RPpropTRPADDFR ); }
	}

    if  ( PROPmaskISSET( rpUpdMask, RPpropTRPADDFT ) )
	{
	if  ( rp1->rpTopCellPaddingUnit !=
				    rp2->rpTopCellPaddingUnit )
	    { PROPmaskADD( &rpDiffMask, RPpropTRPADDFT ); }
	}

    if  ( PROPmaskISSET( rpUpdMask, RPpropTRPADDFB ) )
	{
	if  ( rp1->rpBottomCellPaddingUnit !=
				    rp2->rpBottomCellPaddingUnit )
	    { PROPmaskADD( &rpDiffMask, RPpropTRPADDFB ); }
	}

    /**/
    if  ( PROPmaskISSET( rpUpdMask, RPpropTRW_WIDTHB ) )
	{
	if  ( rp1->rpCellWidthBefore != rp2->rpCellWidthBefore )
	    { PROPmaskADD( &rpDiffMask, RPpropTRW_WIDTHB );	}
	}

    if  ( PROPmaskISSET( rpUpdMask, RPpropTRW_WIDTHA ) )
	{
	if  ( rp1->rpCellWidthAfter != rp2->rpCellWidthAfter )
	    { PROPmaskADD( &rpDiffMask, RPpropTRW_WIDTHA );	}
	}

    /**/
    if  ( PROPmaskISSET( rpUpdMask, RPpropTRFTS_WIDTHB ) )
	{
	if  ( rp1->rpCellWidthBeforeUnit != rp2->rpCellWidthBeforeUnit )
	    { PROPmaskADD( &rpDiffMask, RPpropTRFTS_WIDTHB );	}
	}

    if  ( PROPmaskISSET( rpUpdMask, RPpropTRFTS_WIDTHA ) )
	{
	if  ( rp1->rpCellWidthAfterUnit != rp2->rpCellWidthAfterUnit )
	    { PROPmaskADD( &rpDiffMask, RPpropTRFTS_WIDTHA );	}
	}

    /**/

    *pRpDiffPask= rpDiffMask; return;
    }

int docSetRowProperty(		RowProperties *		rp,
				int			prop,
				int			arg )
    {
    switch( prop )
	{
	case RPpropCOLUMNS:
	    LDEB(prop); return -1;
	    break;

	case RPpropGAP_WIDTH:
	    rp->rpHalfGapWidthTwips= arg;
	    break;

	case RPpropLEFT_INDENT:
	    rp->rpLeftIndentTwips= arg;
	    break;

	case RPpropHEIGHT:
	    rp->rpHeightTwips= arg;
	    break;

	case RPpropTOP_BORDER:
	    rp->rpTopBorderNumber= arg;
	    break;
	case RPpropBOTTOM_BORDER:
	    rp->rpBottomBorderNumber= arg;
	    break;
	case RPpropLEFT_BORDER:
	    rp->rpLeftBorderNumber= arg;
	    break;
	case RPpropRIGHT_BORDER:
	    rp->rpRightBorderNumber= arg;
	    break;
	case RPpropHORIZ_BORDER:
	    rp->rpHorizontalBorderNumber= arg;
	    break;
	case RPpropVERT_BORDER:
	    rp->rpVerticalBorderNumber= arg;
	    break;

	case RPpropSHADING:
	    rp->rpShadingNumber= arg;
	    break;
	case RPpropFRAME:
	    rp->rpFrameNumber= arg;
	    break;

	/**/
	case RPpropALIGNMENT:
	    rp->rpAlignment= arg;
	    break;

	case RPpropIS_TABLE_HEADER:
	    rp->rpIsTableHeader= ( arg != 0 );
	    break;
	case RPpropKEEP_ON_ONE_PAGE:
	    rp->rpKeepOnOnePage= ( arg != 0 );
	    break;
	case RPprop_KEEPFOLLOW:
	    rp->rp_Keepfollow= ( arg != 0 );
	    break;
	case RPpropAUTOFIT:
	    rp->rpAutofit= ( arg != 0 );
	    break;

	/**/
	case RPpropTRW_WIDTH:
	    rp->rpPreferredWidth= arg;
	    break;
	case RPpropTRFTS_WIDTH:
	    rp->rpPreferredWidthUnit= arg;
	    break;

	/**/
	case RPpropTRSPDL:
	    rp->rpLeftDefaultCellSpacing= arg;
	    break;
	case RPpropTRSPDR:
	    rp->rpRightDefaultCellSpacing= arg;
	    break;
	case RPpropTRSPDT:
	    rp->rpTopDefaultCellSpacing= arg;
	    break;
	case RPpropTRSPDB:
	    rp->rpBottomDefaultCellSpacing= arg;
	    break;

	case RPpropTRSPDFL:
	    rp->rpLeftDefaultCellSpacingUnit= arg;
	    break;
	case RPpropTRSPDFR:
	    rp->rpRightDefaultCellSpacingUnit= arg;
	    break;
	case RPpropTRSPDFT:
	    rp->rpTopDefaultCellSpacingUnit= arg;
	    break;
	case RPpropTRSPDFB:
	    rp->rpBottomDefaultCellSpacingUnit= arg;
	    break;

	/**/
	case RPpropTRPADDL:
	    rp->rpLeftCellPadding= arg;
	    break;
	case RPpropTRPADDR:
	    rp->rpRightCellPadding= arg;
	    break;
	case RPpropTRPADDT:
	    rp->rpTopCellPadding= arg;
	    break;
	case RPpropTRPADDB:
	    rp->rpBottomCellPadding= arg;
	    break;

	case RPpropTRPADDFL:
	    rp->rpLeftCellPaddingUnit= arg;
	    break;
	case RPpropTRPADDFR:
	    rp->rpRightCellPaddingUnit= arg;
	    break;
	case RPpropTRPADDFT:
	    rp->rpTopCellPaddingUnit= arg;
	    break;
	case RPpropTRPADDFB:
	    rp->rpBottomCellPaddingUnit= arg;
	    break;

	/**/
	case RPpropTRW_WIDTHB:
	    rp->rpCellWidthBefore= arg;
	    break;
	case RPpropTRW_WIDTHA:
	    rp->rpCellWidthAfter= arg;
	    break;

	case RPpropTRFTS_WIDTHB:
	    rp->rpCellWidthBeforeUnit= arg;
	    break;
	case RPpropTRFTS_WIDTHA:
	    rp->rpCellWidthAfterUnit= arg;
	    break;

	case RPpropAUTOFORMAT_BORDERS:
	    rp->rpAutoformatBorders= ( arg != 0 );
	    break;
	case RPpropAUTOFORMAT_SHADING:
	    rp->rpAutoformatShading= ( arg != 0 );
	    break;
	case RPpropAUTOFORMAT_FONT:
	    rp->rpAutoformatFont= ( arg != 0 );
	    break;
	case RPpropAUTOFORMAT_COLOR:
	    rp->rpAutoformatColor= ( arg != 0 );
	    break;
	case RPpropAUTOFORMAT_APPLY_BEST_FIT:
	    rp->rpAutoformatApplyBestFit= ( arg != 0 );
	    break;
	case RPpropAUTOFORMAT_FIRST_ROW:
	    rp->rpAutoformatFirstRow= ( arg != 0 );
	    break;
	case RPpropAUTOFORMAT_LAST_ROW:
	    rp->rpAutoformatLastRow= ( arg != 0 );
	    break;
	case RPpropAUTOFORMAT_FIRST_COLUMN:
	    rp->rpAutoformatFirstColumn= ( arg != 0 );
	    break;
	case RPpropAUTOFORMAT_LAST_COLUMN:
	    rp->rpAutoformatLastColumn= ( arg != 0 );
	    break;

	case RPpropROW_NUMBER:
	    rp->rpRowNumber= arg;
	    break;
	case RPpropROW_BAND_NUMBER:
	    rp->rpRowBandNumber= arg;
	    break;
	case RPpropROW_STYLE:
	    rp->rpRowStyle= arg;
	    break;

	case RPpropIS_LAST_ROW:
	    rp->rpIsLastRow= ( arg != 0 );
	    break;

	case RPpropTRAUTH:
	    rp->rpAuthor= arg;
	    break;

	default:
	    LDEB(prop); return -1;
	}

    return 0;
    }

