/************************************************************************/
/*									*/
/*  Buffer administration routines relating to the text particules in a	*/
/*  text paragraph.							*/
/*									*/
/************************************************************************/

#   include	"docBufConfig.h"

#   include	<stdlib.h>
#   include	<string.h>
#   include	<stdio.h>

#   include	<appDebugon.h>

#   include	<uniUtf8.h>

#   include	"docBuf.h"
#   include	"docParaString.h"
#   include	"docEdit.h"

/*
# define DEB_PARTICULES
*/

/************************************************************************/

static void docInitTextParticule(	TextParticule *		tp )
    {
    tp->tpStroff= 0;
    tp->tpStrlen= 0;
    tp->tpKind= DOCkindUNKNOWN;
    tp->tpFlags= 0;
    tp->tpTwipsWide= 0;	
    tp->tpXContentXPixels= 0;
    tp->tpObjectNumber= -1;
    tp->tpTextAttrNr= 0;

    return;
    }

/************************************************************************/
/*									*/
/*  Split a text particule.						*/
/*									*/
/*  E.G. To change the attributes of part of its text or to insert	*/
/*  something in the middle.						*/
/*									*/
/************************************************************************/

int docSplitTextParticule(	TextParticule **		pTpPart,
				TextParticule **		pTpNext,
				BufferItem *			paraBi,
				int				part,
				int				stroff )
    {
    TextParticule	tpScratch;
    TextParticule *	tpPart;
    TextParticule *	tpNext;

    if  ( part < 0 || part >= paraBi->biParaParticuleCount )
	{ LLDEB(part,paraBi->biParaParticuleCount); return -1;	}

    tpPart= paraBi->biParaParticules+ part;
    tpScratch= *tpPart;

    if  ( tpPart->tpKind != DOCkindSPAN			||
	  stroff <= tpPart->tpStroff			||
	  stroff >= tpPart->tpStroff+ tpPart->tpStrlen	)
	{
	SLLLDEB(docKindStr(tpPart->tpKind),stroff,tpPart->tpStroff,tpPart->tpStrlen);
	docListItem(0,paraBi,0);
	return -1;
	}

    tpPart= docInsertTextParticule( paraBi, part,
					    tpScratch.tpStroff,
					    stroff- tpScratch.tpStroff,
					    tpScratch.tpKind,
					    tpScratch.tpTextAttrNr );
    if  ( ! tpPart )
	{ LXDEB(part,tpPart); return -1;	}

    tpNext= tpPart+ 1;

    tpNext->tpStroff= stroff;
    tpNext->tpStrlen= ( tpScratch.tpStroff+ tpScratch.tpStrlen )- stroff;

    if  ( pTpPart )
	{ *pTpPart= tpPart;	}
    if  ( pTpNext )
	{ *pTpNext= tpNext;	}

    return 0;
    }

/************************************************************************/
/*									*/
/*  Add a Particule to a paragraph.					*/
/*									*/
/************************************************************************/

static TextParticule *	docInsertParticules(	BufferItem *	paraBi,
						int		part,
						int		count )
    {
    int			i;
    TextParticule *	tp;

    tp= (TextParticule *)realloc( paraBi->biParaParticules,
	    ( paraBi->biParaParticuleCount+ count ) * sizeof( TextParticule ) );
    if  ( ! tp )
	{ LLDEB(paraBi->biParaParticuleCount,tp); return tp; }
    paraBi->biParaParticules= tp;

    if  ( part < 0 )
	{ part= paraBi->biParaParticuleCount;	}
    else{
	int	tail= paraBi->biParaParticuleCount- part;

	tp= paraBi->biParaParticules+ paraBi->biParaParticuleCount+ count- 1;

	for ( i= 0; i < tail; tp--, i++ )
	    { tp[0]= tp[-count];	}
	}

    tp= paraBi->biParaParticules+ part;
    for ( i= 0; i < count; tp++, i++ )
	{
	tp->tpTextAttrNr= -1;

	docInitTextParticule( tp );
	tp->tpKind= DOCkindSPAN;

	paraBi->biParaParticuleCount++;
	}

    return paraBi->biParaParticules+ part;
    }

/************************************************************************/
/*									*/
/*  Insert a text particule.						*/
/*									*/
/************************************************************************/

TextParticule *	docInsertTextParticule(	BufferItem *	paraBi,
					int		part,
					int		off,
					int		len,
					int		kind,
					int		textAttributeNumber )
    {
    const int		count= 1;
    TextParticule *	tp;

    tp= docInsertParticules( paraBi, part, count );
    if  ( ! tp )
	{ XDEB(tp); return tp;	}

    docInitTextParticule( tp );
    tp->tpStroff= off;
    tp->tpStrlen= len;
    tp->tpKind= kind;
    tp->tpTextAttrNr= textAttributeNumber;

    return tp;
    }

/************************************************************************/
/*									*/
/*  Delete a series of particules.					*/
/*									*/
/************************************************************************/

void docDeleteParticules(	BufferItem *	paraBi,
				int		first,
				int		count )
    {
    if  ( first > paraBi->biParaParticuleCount )
	{
	LLDEB(first,paraBi->biParaParticuleCount);
	first= paraBi->biParaParticuleCount;
	}

    if  ( first+ count > paraBi->biParaParticuleCount )
	{
	LLDEB(first+count,paraBi->biParaParticuleCount);
	count= paraBi->biParaParticuleCount- first;
	}

    if  ( count <= 0 )
	{ LDEB(count); return;	}

    paraBi->biParaParticuleCount -= count;

    while( first < paraBi->biParaParticuleCount )
	{
	paraBi->biParaParticules[first]= paraBi->biParaParticules[first+ count];
	first++;
	}

    return;
    }

/************************************************************************/
/*									*/
/*  Enter a field while copying. This assumes that copying is done in	*/
/*  document order.							*/
/*									*/
/************************************************************************/

static DocumentField * docStartFieldCopy(
				DocumentCopyJob *		dcj,
				const EditPosition *		epStart,
				int				obnrFrom )
    {
    EditOperation *		eo= dcj->dcjEditOperation;
    BufferDocument *		bdTo= eo->eoBd;
    const DocumentFieldList *	dflFrom= &(dcj->dcjBdFrom->bdFieldList);
    const int			fieldCount= dflFrom->dflPagedList.plItemCount;

    DocumentField *		dfFrom;
    DocumentField *		dfTo;

    dfFrom= docGetFieldByNumber( dflFrom, obnrFrom );

    dfTo= docClaimFieldCopy( &(bdTo->bdFieldList), dfFrom,
					&(dcj->dcjSelectionScope), epStart );
    if  ( ! dfTo )
	{ XDEB(dfTo); return (DocumentField *)0;	}

    if  ( obnrFrom < 0 || obnrFrom >= fieldCount )
	{ LDEB(obnrFrom); }
    else{
	dcj->dcjFieldMap[obnrFrom]= dfTo->dfFieldNumber;
	}

    if  ( docPushFieldOnCopyStack( dcj, dfTo ) )
	{ LDEB(1); return (DocumentField *)0;	}

    if  ( dfTo->dfKind >= DOC_FieldKindCount	)
	{ LDEB(dfTo->dfKind);	}
    else{
	eo->eoFieldUpdate |=
		DOC_FieldKinds[dfTo->dfKind].fkiCalculateWhen;
	}

    if  ( dfTo->dfKind == DOCfkHYPERLINK && dcj->dcjRefFileName )
	{
	docMakeHyperlinkRelative( dfTo,
			    dcj->dcjRefFileName, dcj->dcjRefFileSize );
	}

    if  ( dfTo->dfKind == DOCfkBOOKMARK )
	{
	char		newName[DOCmaxBOOKMARK+ 1];
	const char *	markName;
	int		markSize;

	if  ( docFieldGetBookmark( dfFrom, &markName, &markSize ) )
	    { LDEB(1); newName[0]= 'x'; newName[1]= '\0'; markSize= 1;	}
	else{
	    if  ( markSize > DOCmaxBOOKMARK )
		{ markSize= DOCmaxBOOKMARK;	}
	    strncpy( newName, markName, markSize )[markSize]= '\0';
	    }

	if  ( docMakeBookmarkUnique( bdTo, newName, markSize ) )
	    { LDEB(1); return (DocumentField *)0;	}

	if  ( docFieldSetBookmark( dfTo, newName, markSize ) )
	    { LDEB(markSize); return (DocumentField *)0;	}
	}

    if  ( dfFrom->dfKind == DOCfkCHFTN && dfFrom->dfNoteIndex >= 0 )
	{
	if  ( docCopyNote( dcj, dfTo, dfFrom ) )
	    { LDEB(1); return (DocumentField *)0;	}

	dcj->dcjNotesCopied++;
	}

    return dfTo;
    }

/************************************************************************/
/*									*/
/*  Leave a field while copying. This assumes that copying is done in	*/
/*  document order.							*/
/*									*/
/************************************************************************/

static int docFinishFieldCopy(	DocumentCopyJob *		dcj,
				const EditPosition *		epEnd,
				int				obnrFrom )
    {
    EditOperation *		eo= dcj->dcjEditOperation;
    BufferDocument *		bdTo= eo->eoBd;
    const DocumentFieldList *	dflFrom= &(dcj->dcjBdFrom->bdFieldList);
    const int			fieldCount= dflFrom->dflPagedList.plItemCount;

    DocumentField *		dfTo;
    int				obnrTo;

    FieldCopyStackLevel *	fcsl= dcj->dcjFieldStack;

    if  ( obnrFrom < 0				||
	  obnrFrom >= fieldCount	)
	{ LLDEB(obnrFrom,fieldCount); return -1;	}

    if  ( dcj->dcjFieldMap[obnrFrom] < 0 )
	{
	LLDEB(obnrFrom,fieldCount);
	SDEB(docFieldKindStr(docGetFieldKindByNumber(dflFrom,obnrFrom)));
	return -1;
	}

    obnrTo= dcj->dcjFieldMap[obnrFrom];
    dfTo= docGetFieldByNumber( &(bdTo->bdFieldList), obnrTo );
    docSetFieldEnd( dfTo, epEnd );

    if  ( ! fcsl )
	{ XDEB(fcsl);	}
    else{
	if  ( fcsl->fcslPrev )
	    {
	    if  ( docAddChildToField( fcsl->fcslField,
						fcsl->fcslPrev->fcslField ) )
		{ LDEB(1); return -1;	}
	    }
	else{
	    if  ( docExternalItemAddRootField( dcj->dcjEiTo, fcsl->fcslField ) )
		{ LDEB(1); return -1;	}
	    }

	dcj->dcjFieldStack= fcsl->fcslPrev;
	free( fcsl );
	}

    return obnrTo;
    }

/************************************************************************/
/*									*/
/*  Copy particule data from one paragraph to another.			*/
/*									*/
/*  9)  Not needed, the document is marked as changed, so this can wait	*/
/*	until it is saved.						*/
/*									*/
/************************************************************************/

static int docCopyParticuleData(	DocumentCopyJob *	dcj,
					BufferItem *		biTo,
					const BufferItem *	biFrom,
					TextParticule *		tpTo,
					const TextParticule *	tpFrom )
    {
    EditOperation *		eo= dcj->dcjEditOperation;
    BufferDocument *		bdTo= eo->eoBd;
    int				paraNr= docNumberOfParagraph( biTo );

    switch( tpFrom->tpKind )
	{
	case DOCkindSPAN:
	case DOCkindTAB:
	case DOCkindLINEBREAK:
	case DOCkindPAGEBREAK:
	case DOCkindCOLUMNBREAK:
	    break;

	case DOCkindOBJECT:
	    {
	    const InsertedObject *	ioFrom;
	    InsertedObject *		ioTo;
	    int				nr;

	    ioFrom= docGetObject( dcj->dcjBdFrom, tpFrom->tpObjectNumber );
	    if  ( ! ioFrom )
		{ LPDEB(tpFrom->tpObjectNumber,ioFrom); return -1;	}

	    ioTo= docClaimObjectCopy( bdTo, &nr, ioFrom );
	    if  ( ! ioTo )
		{ XDEB(ioTo); return -1;	}
	    tpTo->tpObjectNumber= nr;

	    ioFrom= docGetObject( dcj->dcjBdFrom, tpFrom->tpObjectNumber );
	    if  ( ! ioFrom )
		{ LPDEB(tpFrom->tpObjectNumber,ioFrom); return -1;	}

	    if  ( ioFrom->ioKind == DOCokDRAWING_SHAPE )
		{
		if  ( ! ioFrom->ioDrawingShape )
		    { XDEB(ioFrom->ioDrawingShape);	}
		else{
		    ioTo->ioDrawingShape= docCopyDrawingShape( dcj,
						    ioFrom->ioDrawingShape );
		    if  ( ! ioTo->ioDrawingShape )
			{ XDEB(ioTo->ioDrawingShape);	}
		    }
		}

	    /*  9
	    if  ( ioTo->ioBliptag == 0 )
		{ ioTo->ioBliptag= appGetTimestamp();	}
	    */
	    }
	    break;

	case DOCkindFIELDEND:
	    {
	    EditPosition		ep;

	    ep.epParaNr= paraNr;
	    ep.epStroff= tpTo->tpStroff;

	    if  ( dcj->dcjCopyFields )
		{
		tpTo->tpObjectNumber= docFinishFieldCopy( dcj, &ep,
						tpFrom->tpObjectNumber );
		}
	    else{ tpTo->tpObjectNumber= tpFrom->tpObjectNumber;	}
	    if  ( tpTo->tpObjectNumber < 0 )
		{ LDEB(tpTo->tpObjectNumber);	}
	    }
	    break;

	case DOCkindFIELDSTART:
	    if  ( dcj->dcjCopyFields )
		{
		EditPosition		ep;
		DocumentField *		dfTo;

		ep.epParaNr= paraNr;
		ep.epStroff= tpTo->tpStroff;

		dfTo= docStartFieldCopy( dcj, &ep, tpFrom->tpObjectNumber );
		if  ( ! dfTo )
		    { XDEB(dfTo); return -1;	}
		tpTo->tpObjectNumber= dfTo->dfFieldNumber;
		}
	    else{ tpTo->tpObjectNumber= tpFrom->tpObjectNumber;	}
	    break;

	default:
	    LDEB(tpFrom->tpKind); return -1;
	}

    return 0;
    }

/************************************************************************/
/*									*/
/*  Copy particules plus contents from one paragraph to another.	*/
/*									*/
/*  This might be a copy to a different document, so font numbers and	*/
/*  attribute numbers need to be recalculated.				*/
/*									*/
/*  1)  Find out where to insert.					*/
/*	(At the end of the paragraph or before particule[partTo].	*/
/*  2)  Find position and length of the source string.			*/
/*  3)  Find out whether to replace the empty particule that we use to	*/
/*	have at least one particule in a paragraph.			*/
/*  4)  Insert the desired number of particules in the correct		*/
/*	position. Note that that is one less in a paragraph that was	*/
/*	originally empty.						*/
/*  5)  Insert the new text into the paragraph string.			*/
/*  6)  Shift the particules in the tail of the target paragraph to	*/
/*	make place for the new text string bytes. The possible		*/
/*	overwritten 'empty' particule should not be shifted.		*/
/*  7)  Patch the particules in the hole created above to correct	*/
/*	values derived from the source particules.			*/
/*									*/
/************************************************************************/

int docCopyParticules(	DocumentCopyJob *		dcj,
			BufferItem *			biTo,
			const BufferItem *		biFrom,
			int				partTo,
			int				partFrom,
			int				countFrom,
			int *				pParticulesInserted,
			int *				pCharactersCopied )
    {
    EditOperation *		eo= dcj->dcjEditOperation;

    TextParticule *		tpTo;
    const TextParticule *	tpFrom;

    int				stroffTo;
    int				stroffShift;

    int				i;

    int				replaceEmpty= 0;

    int				stroffFrom;
    int				strlenFrom;

    int				notesCopied= 0;

    int				paraNrTo= docNumberOfParagraph( biTo );

    /*  1  */
    if  ( partTo > biTo->biParaParticuleCount )
	{
	LLDEB(partTo,biTo->biParaParticuleCount);
	partTo= biTo->biParaParticuleCount;
	}

    if  ( partTo == biTo->biParaParticuleCount )
	{ stroffTo= docParaStrlen( biTo );				}
    else{ stroffTo= biTo->biParaParticules[partTo].tpStroff;	}

    /*  2  */
    tpFrom= biFrom->biParaParticules+ partFrom;
    stroffFrom= tpFrom->tpStroff;
    strlenFrom= tpFrom[countFrom- 1].tpStroff+ tpFrom[countFrom- 1].tpStrlen-
							    tpFrom->tpStroff;

    /*  3  */
    if  ( docParaStrlen( biTo ) == 0 && biTo->biParaParticuleCount == 1 )
	{
	if  ( partTo < 0 || partTo > 1 )
	    { LDEB(partTo);	}
	if  ( partTo != 0 )
	    { partTo= 0;	}

	replaceEmpty= 1;
	}

    /*  4  */
    tpTo= docInsertParticules( biTo, partTo, countFrom- replaceEmpty );
    if  ( ! tpTo )
	{ XDEB(tpTo); return -1;	}

    /*  5  */
    if  ( docParaStringReplace( &stroffShift, biTo, stroffTo, stroffTo,
		    (char *)docParaString( biFrom, stroffFrom ), strlenFrom ) )
	{
	LDEB(strlenFrom);
	docDeleteParticules( biTo, partTo, countFrom- replaceEmpty );
	return -1;
	}

    /*  6  */
    if  ( docEditShiftParticuleOffsets( eo, biTo, paraNrTo,
						partTo+ countFrom,
						biTo->biParaParticuleCount,
						stroffTo, stroffShift ) )
	{ LDEB(stroffShift);	}

    /*  7  */
    tpTo= biTo->biParaParticules+ partTo;
    for ( i= 0; i < countFrom; tpTo++, tpFrom++, i++ )
	{
	int		textAttributeNumberTo;

	textAttributeNumberTo= docMapTextAttributeNumber( dcj,
							tpFrom->tpTextAttrNr );
	if  ( textAttributeNumberTo < 0 )
	    { LDEB(textAttributeNumberTo); return -1;	}

	tpTo->tpStroff= stroffTo;
	tpTo->tpStrlen= tpFrom->tpStrlen;
	tpTo->tpKind= tpFrom->tpKind;
	tpTo->tpTextAttrNr= textAttributeNumberTo;

	if  ( docCopyParticuleData( dcj, biTo, biFrom, tpTo, tpFrom ) )
	    { LLDEB(partTo,i); return -1;	}

	stroffTo += tpTo->tpStrlen;
	}

    dcj->dcjNotesCopied += notesCopied;
    *pParticulesInserted += countFrom- replaceEmpty;
    *pCharactersCopied += stroffShift;

    return 0;
    }

int docCopyParticuleAndData(		TextParticule **	pTpTo,
					DocumentCopyJob *	dcj,
					BufferItem *		paraBiTo,
					int			partTo,
					int			stroffTo,
					int			strlenTo,
					const BufferItem *	paraBiFrom,
					const TextParticule *	tpFrom )
    {
    TextParticule	tpSaved;
    int			textAttributeNumberTo;

    TextParticule *	tpTo;

    tpSaved= *tpFrom;

    textAttributeNumberTo= docMapTextAttributeNumber( dcj,
					tpSaved.tpTextAttrNr );

    if  ( textAttributeNumberTo < 0 )
	{ LDEB(textAttributeNumberTo); return -1;	}

    if  ( partTo < 0 )
	{ partTo= paraBiTo->biParaParticuleCount;	}

    tpTo= docInsertTextParticule( paraBiTo, partTo,
					    stroffTo, strlenTo,
					    tpSaved.tpKind,
					    textAttributeNumberTo );
    if  ( ! tpTo )
	{ LXDEB(partTo,tpTo); return -1;	}

    if  ( docCopyParticuleData( dcj, paraBiTo, paraBiFrom, tpTo, &tpSaved ) )
	{
	docDeleteParticules( paraBiTo, partTo, 1 );
	LDEB(partTo); return -1;
	}

    *pTpTo= tpTo;
    return 0;
    }

/************************************************************************/

TextParticule * docMakeSpecialParticule(
				BufferItem *			paraBi,
				int				n,
				int				stroff,
				int				kind,
				int				textAttrNr )
    {
    int			stroffShift= 0;

    unsigned char	bytes[7];
    int			count;
    int			sym= '?';

    TextParticule *	tp;

    switch( kind )
	{
	case DOCkindTAB:	sym= ' '; break;
	case DOCkindLINEBREAK:	sym= ' '; break;
	case DOCkindPAGEBREAK:	sym= ' '; break;
	case DOCkindCOLUMNBREAK:sym= ' '; break;

	case DOCkindOPT_HYPH:	sym= '-'; break;
	case DOCkindCHFTNSEP:	sym= '='; break;
	case DOCkindCHFTNSEPC:	sym= '='; break;

	case DOCkindFIELDSTART:	sym= '{'; break;
	case DOCkindFIELDEND:	sym= '}'; break;
	default:		sym= '@'; break;
	}

    count= uniPutUtf8( bytes, sym );
    if  ( count < 1 )
	{ LDEB(count); return (TextParticule *)0;	}
    bytes[count]= '\0';

    if  ( ! docIsParaItem( paraBi ) )
	{ LDEB(paraBi->biLevel); return (TextParticule *)0;	}

    if  ( docParaStringReplace( &stroffShift, paraBi, stroff, stroff,
						    (char *)bytes, count ) )
	{ LDEB(docParaStrlen(paraBi)); return (TextParticule *)0;	}

    tp= docInsertTextParticule( paraBi, n, stroff, count,
						kind, textAttrNr );
    if  ( ! tp )
	{ LDEB(paraBi->biParaParticuleCount); return (TextParticule *)0; }

    return tp;
    }

/************************************************************************/
/*									*/
/*  Insert a length 1 particule that is used to represent tabs, breaks,	*/
/*  or objects like images in the paragraph and in the string of the	*/
/*  paragraph.								*/
/*									*/
/************************************************************************/

int docSaveSpecialParticule(	BufferDocument *		bd,
				BufferItem *			paraBi,
				const TextAttribute *		ta,
				int				kind )
    {
    int			n= -1;
    int			stroff= docParaStrlen( paraBi );
    int			textAttrNr;

    textAttrNr= docTextAttributeNumber( bd, ta );
    if  ( textAttrNr < 0 )
	{ LDEB(textAttrNr); return -1;	}

    if  ( ! docMakeSpecialParticule( paraBi, n, stroff, kind, textAttrNr ) )
	{ LDEB(paraBi->biParaParticuleCount); return -1;	}

    return 0;
    }

/************************************************************************/
/*									*/
/*  Insert a zero length particule that is used for internal		*/
/*  administration such as delimiting text fields.			*/
/*									*/
/*  Also used to insert the one and only particule in an empty		*/
/*  paragraph.								*/
/*									*/
/*  1)  Paranoia: can be called directly from file readers.		*/
/*									*/
/************************************************************************/

int docInsertAdminParticule(		BufferDocument *	bd,
					BufferItem *		paraBi,
					int			n,
					int			stroff,
					int			objectNumber,
					int			kind,
					const TextAttribute *	ta )
    {
    TextParticule *	tp;
    int			textAttrNr;

    /*  1  */
    if  ( ! bd || ! paraBi )
	{ XXDEB(bd,paraBi); return -1;	}

    textAttrNr= docTextAttributeNumber( bd, ta );
    if  ( textAttrNr < 0 )
	{ LDEB(textAttrNr); return -1;	}

    tp= docMakeSpecialParticule( paraBi, n, stroff, kind, textAttrNr );
    if  ( ! tp )
	{ XDEB(tp); return -1;	}

    tp->tpObjectNumber= objectNumber;

    return 0;
    }

/************************************************************************/
/*									*/
/*  Make sure the a paragraph does not end in an explicit break.	*/
/*									*/
/************************************************************************/

int docCheckNoBreakAsLast(	EditOperation *		eo,
				BufferItem *		paraBi )
    {
    const int			part= paraBi->biParaParticuleCount;
    const int			stroff= docParaStrlen( paraBi );
    const int			len= 0;

    const TextParticule *	tp= paraBi->biParaParticules+ part- 1;

    if  ( tp->tpKind != DOCkindLINEBREAK	&&
	  tp->tpKind != DOCkindPAGEBREAK	&&
	  tp->tpKind != DOCkindCOLUMNBREAK	)
	{ return 0;	}

    tp= docInsertTextParticule( paraBi, part, stroff, len,
				    DOCkindSPAN, tp->tpTextAttrNr );
    if  ( ! tp )
	{ XDEB(tp); return -1;	}

    docEditIncludeItemInReformatRange( eo, paraBi );

    return 0;
    }

/************************************************************************/
/*									*/
/*  Shift the particules in a paragraph after an insertion or a		*/
/*  deletion.								*/
/*									*/
/************************************************************************/

static void docShiftParticuleOffset(	BufferDocument *	bd,
					const BufferItem *	paraBi,
					TextParticule *		tp,
					int			stroffShift )
    {
    if  ( tp->tpKind == DOCkindFIELDSTART )
	{
	DocumentField *	df;

	if  ( tp->tpObjectNumber < 0 )
	    { LDEB(tp->tpObjectNumber); docListItem(0,paraBi,0); }
	else{
	    df= docGetFieldByNumber( &(bd->bdFieldList), tp->tpObjectNumber );
	    df->dfHeadPosition.epStroff += stroffShift;
	    }
	}

    if  ( tp->tpKind == DOCkindFIELDEND )
	{
	DocumentField *	df;

	if  ( tp->tpObjectNumber < 0 )
	    { LDEB(tp->tpObjectNumber); docListItem(0,paraBi,0); }
	else{
	    df= docGetFieldByNumber( &(bd->bdFieldList), tp->tpObjectNumber );
	    if  ( ! df )
		{ LPDEB(tp->tpObjectNumber,df);			}
	    else{ df->dfTailPosition.epStroff += stroffShift;	}
	    }
	}

    tp->tpStroff += stroffShift;

    return;
    }

int docShiftParticuleOffsets(	BufferDocument *	bd,
				BufferItem *		paraBi,
				int			partFrom,
				int			partUpto,
				int			stroffShift )
    {
    int			part;
    TextParticule *	tp;

    if  ( stroffShift < 0 )
	{
	tp= paraBi->biParaParticules+ partFrom;
	for ( part= partFrom; part < partUpto; tp++, part++ )
	    { docShiftParticuleOffset( bd, paraBi, tp, stroffShift );	}
	}

    if  ( stroffShift > 0 )
	{
	tp= paraBi->biParaParticules+ partUpto- 1;
	for ( part= partUpto- 1; part >= partFrom; tp--, part-- )
	    { docShiftParticuleOffset( bd, paraBi, tp, stroffShift );	}
	}

    return 0;
    }

/************************************************************************/
/*									*/
/*  Find the particule number for a certain string position.		*/
/*									*/
/*  NOTE: This does not expect the paragraph to be formatted.		*/
/*									*/
/************************************************************************/

int docFindParticuleOfPosition(	int *				pPart,
				const DocumentPosition *	dp,
				int				lastOne )
    {
    const BufferItem *		paraBi= dp->dpBi;
    int				stroff= dp->dpStroff;

    int				l= 0;
    int				r= paraBi->biParaParticuleCount;
    int				m= ( l+ r )/ 2;

    const TextParticule *	tp= paraBi->biParaParticules+ m;

    if  ( r <= 0 )
	{ LDEB(r); return -1;	}

    if  ( lastOne )
	{
	/* Binary search for last: tp->tpStroff <= stroff */
	while( l < m )
	    {
	    if  ( tp->tpStroff <= stroff )
		{ l= m;	}
	    else{ r= m;	}

	    m= ( l+ r )/ 2; tp= paraBi->biParaParticules+ m;
	    }

	if  ( stroff < tp->tpStroff )
	    { m--;	}
	}
    else{
	/* Binary search for first: tp->tpStroff+ tp->tpStrlen >= stroff */
	while( l < m )
	    {
	    if  ( tp->tpStroff+ tp->tpStrlen < stroff )
		{ l= m;	}
	    else{ r= m;	}

	    m= ( l+ r )/ 2; tp= paraBi->biParaParticules+ m;
	    }

	if  ( stroff > tp->tpStroff+ tp->tpStrlen )
	    { m++;	}
	}

    if  ( m < 0 || m >= paraBi->biParaParticuleCount )
	{
	const int checkGeometry= 0;

	LLLLDEB(stroff,m,lastOne,paraBi->biParaParticuleCount);
	docListItem( 0, paraBi, checkGeometry );
	return -1;
	}

    *pPart= m;

    return 0;
    }

/************************************************************************/
/*									*/
/*  Free a Paragraph level node.					*/
/*									*/
/************************************************************************/

void docCleanParaItem(	BufferDocument *	bd,
			DocumentTree *		ei,
			BufferItem *		bi )
    {
    const DocumentFieldList *	dfl= &(bd->bdFieldList);
    const int			fieldCount= dfl->dflPagedList.plItemCount;
    int				i;
    TextParticule *		tp;

    tp= bi->biParaParticules;
    for ( i= 0; i < bi->biParaParticuleCount; tp++, i++ )
	{
	switch( tp->tpKind )
	    {
	    case DOCkindFIELDSTART:
	    case DOCkindFIELDEND:
		{
		DocumentField *	df;

		if  ( tp->tpObjectNumber < 0		||
		      tp->tpObjectNumber >= fieldCount	)
		    {
		    LLDEB(tp->tpObjectNumber,fieldCount);
		    continue;
		    }
		df= docGetFieldByNumber( &(bd->bdFieldList),
						    tp->tpObjectNumber );
		if  ( df )
		    {
		    if  ( docDeleteFieldFromParent( ei, df ) )
			{ LDEB(df->dfFieldNumber);	}

		    docDeleteFieldFromDocument( bd, df );
		    }
		}
		break;

	    case DOCkindOBJECT:
		{
		InsertedObject *	io;

		io=  docGetObject( bd, tp->tpObjectNumber );
		if  ( ! io )
		    { LPDEB(tp->tpObjectNumber,io);	}
		else{
		    if  ( io->ioKind == DOCokDRAWING_SHAPE )
			{ docDeleteDrawingShape( bd, io->ioDrawingShape ); }
		    }
		}
		break;
	    }
	}

    utilCleanMemoryBuffer( &(bi->biParaStringBuffer) );
    if  ( bi->biParaParticules )
	{ free( bi->biParaParticules );	}
    if  ( bi->biParaLines )
	{ free( bi->biParaLines );	}

    docCleanParagraphProperties( &(bi->biParaProperties) );

    return;
    }

void docInitParaItem(	BufferItem *		bi )
    {
    utilInitMemoryBuffer( &(bi->biParaStringBuffer) );

    bi->biParaParticuleCount= 0;
    bi->biParaParticules= (TextParticule *)0;

    bi->biParaLineCount= 0;
    bi->biParaLines= (TextLine *)0;

    bi->biParaFontAscY0= 0;
    bi->biParaFontDescY1= 0;
    bi->biParaMajorityFontSize= 0;
				/********************************/
				/*  Also used to find out	*/
				/*  whether the line extents	*/
				/*  must be recalculated.	*/
				/********************************/

    bi->biParaBorderNrAbove= -1;
    bi->biParaBorderNrBelow= -1;

    bi->biParaTopInset= 0;
    bi->biParaBottomInset= 0;
    bi->biParaFlags= 0;

    docInitParagraphProperties( &(bi->biParaProperties) );
    }

/************************************************************************/
/*									*/
/*  Copy a paragraph to make a new one.					*/
/*									*/
/*  1)  Copy paragraph item.						*/
/*  2)  Open a hole in references to the item to possibly receive some	*/
/*	to the fresh item.						*/
/*  3)  Do not copy a possible list text field from the original.	*/
/*  4)  Copy the contents of the original or insert a dummy particule.	*/
/*  5)  Finally adapt the properties of the target paragraph to those	*/
/*	of the original.						*/
/*									*/
/************************************************************************/

BufferItem * docCopyParaItem(	DocumentCopyJob *	dcj,
					const SelectionScope *	ssRoot,
					BufferItem *		biCellTo,
					int			n,
					const BufferItem *	biParaFrom )
    {
    EditOperation *	eo= dcj->dcjEditOperation;
    BufferDocument *	bdTo= eo->eoBd;
    BufferItem *	biParaTo;

    const int		partTo= 0;
    int			partFrom= 0;

    PropertyMask	ppChgMask;
    PropertyMask	ppUpdMask;

    /*  1  */
    biParaTo= docInsertItem( bdTo, biCellTo, n, DOClevPARA );
    if  ( ! biParaTo )
	{ XDEB(biParaTo); return biParaTo;	}

    docSetParaTableNesting( biParaTo );

    /*  2  */
    if  ( eo )
	{
	int		paraNr;
	const int	stroff= 0;
	const int	sectShift= 0;
	const int	paraShift= 1;

	paraNr= docNumberOfParagraph( biParaTo );

	docEditShiftReferences( eo, ssRoot, paraNr, stroff,
					    sectShift, paraShift, -stroff );
	}

    /*  3  */
    if  ( biParaFrom->biParaListOverride > 0 )
	{
	DocumentField *		dfHead= (DocumentField *)0;
	DocumentSelection	dsInsideHead;
	DocumentSelection	dsAroundHead;
	int			bulletPartBegin= -1;
	int			bulletPartEnd= -1;

	if  ( docDelimitParaHeadField( &dfHead, &dsInsideHead, &dsAroundHead,
					&bulletPartBegin, &bulletPartEnd,
					biParaFrom, dcj->dcjBdFrom ) )
	    { LDEB(1);	}

	if  ( partFrom <= bulletPartEnd )
	    { partFrom= bulletPartEnd+ 1;	}
	}

    /*  4  */
    if  ( partFrom < biParaFrom->biParaParticuleCount )
	{
	int		particulesInserted= 0;
	int		charactersCopied= 0;

	if  ( docCopyParticules( dcj, biParaTo, biParaFrom, partTo,
			partFrom, biParaFrom->biParaParticuleCount- partFrom,
			&particulesInserted, &charactersCopied ) )
	    {
	    LDEB(biParaFrom->biParaParticuleCount);
	    docDeleteItem( eo->eoBd, eo->eoEi, biParaTo );
	    return (BufferItem *)0;
	    }
	}
    else{
	int			textAttributeNumberTo;
	const TextParticule *	tpFrom;

	tpFrom= biParaFrom->biParaParticules+
				    biParaFrom->biParaParticuleCount- 1;

	textAttributeNumberTo= docMapTextAttributeNumber( dcj,
							tpFrom->tpTextAttrNr );
	if  ( textAttributeNumberTo < 0 )
	    { LDEB(textAttributeNumberTo); return (BufferItem *)0;	}

	if  ( ! docInsertTextParticule( biParaTo, 0, 0, 0,
					DOCkindSPAN, textAttributeNumberTo ) )
	    { LDEB(1); return (BufferItem *)0;	}
	}

    /*  5  */
    utilPropMaskClear( &ppChgMask );

    utilPropMaskClear( &ppUpdMask );
    utilPropMaskFill( &ppUpdMask, PPprop_FULL_COUNT );
    PROPmaskUNSET( &ppUpdMask, PPpropTABLE_NESTING );

    if  ( docEditUpdParaProperties( eo, &ppChgMask, biParaTo, &ppUpdMask,
					&(biParaFrom->biParaProperties),
					&(dcj->dcjAttributeMap) ) )
	{ LDEB(1); return (BufferItem *)0;	}

    eo->eoParagraphsInserted++;
    if  ( biParaTo->biParaListOverride > 0 )
	{ dcj->dcjBulletsCopied++;	}

    return biParaTo;
    }

