/************************************************************************/
/*									*/
/*  Layout of a table row in the document.				*/
/*									*/
/************************************************************************/

#   include	"docBufConfig.h"

#   include	<stddef.h>
#   include	<stdio.h>
#   include	<stdlib.h>

#   include	"docPageGrid.h"

#   include	<appDebugon.h>

/************************************************************************/
/*									*/
/*  Draw part of the grid originating from a table row.			*/
/*									*/
/*  3)  Is this cell shaded?						*/
/*  6)  If a cell has a right boder, and the next one a left border,	*/
/*	MS-Word only draws the right border of the leftmost cell.	*/
/*	( The same applies in the vertical direction. )			*/
/*  7)  Base the decisions on whether to draw a table grid on the same	*/
/*	criteria.							*/
/*  8)  Draw cell top border.						*/
/*  9)  Draw cell left border.						*/
/* 10)  Draw cell right border.						*/
/* 11)  Draw cell bottom border.					*/
/*									*/
/* Experimentation with MS-Word reveals:				*/
/* a)	The the top border is drawn inside the cell and that the	*/
/*	formatter allocates space for it.				*/
/* b)	That the left and right borders are half in the cell. Typically	*/
/*	covering part of the horizontal gap that the table reserves.	*/
/* c)	That the bottom border is completely below the cell. So inside	*/
/*	the table it comes at the same location as the top border of	*/
/*	the next row.							*/
/*									*/
/************************************************************************/

static int docCellLeftBorderDrawn(
				BorderProperties *		bpLeft,
				int *				pProp,
				DocumentRectangle *		drOutside,
				DocumentRectangle *		drInside,
				const DocumentRectangle *	drCell,
				const BufferDocument *		bd,
				const BufferItem *		rowBi,
				int				col,
				int				drawTableGrid )
    {
    int				rval= 0;
    const CellProperties *	cp= rowBi->biRowCells+ col;
    const NumberedPropertiesList *	bpl= &(bd->bdBorderPropertyList);

    int				leftHasBorder= 0;
    int				hasLeftBorder= 0;

    int				thick= 0;
    int				space;

    /*  6  */
    if  ( col > 0							&&
	  docBorderNumberIsBorder( bpl, cp[-1].cpRightBorderNumber )	)
	{ leftHasBorder= 1;	}

    docGetBorderPropertiesByNumber( bpLeft, bpl, cp->cpLeftBorderNumber );
    hasLeftBorder= DOCisBORDER( bpLeft );

    if  ( leftHasBorder						&&
	  cp[-1].cpRightBorderNumber != cp->cpLeftBorderNumber	)
	{
	docGetBorderPropertiesByNumber( bpLeft, bpl,
					    cp[-1].cpRightBorderNumber );
	thick= docBorderThick( &space, bpLeft );

	drOutside->drX0= drCell->drX0+ 1- thick/ 2;
	drInside->drX0= drOutside->drX0+ thick;
	}
    else{
	/*  9  */
	if  ( hasLeftBorder )
	    {
	    *pProp= ORNdrawLEFT_BORDER; rval= 1;

	    thick= docBorderThick( &space, bpLeft );

	    drOutside->drX0= drCell->drX0- thick/ 2;
	    drInside->drX0= drOutside->drX0+ thick;
	    }
	}

    /*  7  */
    if  ( drawTableGrid )
	{
	if  ( ! hasLeftBorder && ! leftHasBorder )
	    { *pProp= ORNdrawLEFT_GRID; rval= 1;	}
	}

    return rval;
    }

static int docCellRightBorderDrawn(
				BorderProperties *		bpRight,
				int *				pProp,
				DocumentRectangle *		drOutside,
				DocumentRectangle *		drInside,
				const DocumentRectangle *	drCell,
				const BufferDocument *		bd,
				const BufferItem *		rowBi,
				int				col,
				int				drawTableGrid )
    {
    int				rval= 0;
    const CellProperties *	cp= rowBi->biRowCells+ col;
    const NumberedPropertiesList *	bpl= &(bd->bdBorderPropertyList);

    int				thick;
    int				space;

    int				drawCellRight= 1;

    int				rightHasBorder= 0;
    int				hasRightBorder= 0;

    if  ( col < rowBi->biChildCount- 1	&&
	  cp[1].cpMergedWithLeft	)
	{ drawCellRight= 0;	}

    /*  6  */
    if  ( col < rowBi->biChildCount- 1					&&
	  docBorderNumberIsBorder( bpl, cp[1].cpLeftBorderNumber )	)
	{ rightHasBorder= 1;	}

    docGetBorderPropertiesByNumber( bpRight, bpl, cp->cpRightBorderNumber );
    hasRightBorder= DOCisBORDER( bpRight );

    if  ( ! hasRightBorder && rightHasBorder )
	{
	docGetBorderPropertiesByNumber( bpRight, bpl,
						cp[1].cpLeftBorderNumber );
	}

    /*  10  */
    if  ( drawCellRight && hasRightBorder )
	{
	*pProp= ORNdrawRIGHT_BORDER; rval= 1;

	thick= docBorderThick( &space, bpRight );
	drInside->drX1= drCell->drX1- thick/ 2;
	drOutside->drX1= drInside->drX1+ thick;
	}

    /*  7  */
    if  ( drawTableGrid )
	{
	if  ( ! hasRightBorder && ! rightHasBorder )
	    { *pProp= ORNdrawRIGHT_GRID; rval= 1;	}
	}

    return rval;
    }

static int docCellTopBorderDrawn( BorderProperties *		bpTop,
				int *				pProp,
				DocumentRectangle *		drOutside,
				DocumentRectangle *		drInside,
				const DocumentRectangle *	drCell,
				const BufferDocument *		bd,
				const BufferItem *		rowBi,
				int				col,
				int				atRowTop,
				int				drawTableGrid )
    {
    int				rval= 0;
    int				useAbove= 0;

    docGetCellTopBorder( bpTop, &useAbove, bd, rowBi, col, atRowTop );

    if  ( useAbove )
	{
	int	thick;
	int	space;

	thick= docBorderThick( &space, bpTop );

	drInside->drY0= drCell->drY0+ thick;
	}

    /*  8  */
    if  ( ! useAbove && DOCisBORDER( bpTop ) )
	{
	int	thick;
	int	space;

	*pProp= ORNdrawTOP_BORDER; rval= 1;

	thick= docBorderThick( &space, bpTop );

	/* border inside cell */
	drInside->drY0= drCell->drY0+ 1+ thick;
	}

    /*  7  */
    if  ( drawTableGrid )
	{
	if  ( ! DOCisBORDER( bpTop )		&&
	      ! useAbove			)
	    {
	    *pProp= ORNdrawTOP_GRID; rval= 1;
	    /* inside cell, width determined by screen drawer */
	    }
	}

    return rval;
    }

static int docCellBottomBorderDrawn(
				BorderProperties *		bpBottom,
				int *				pProp,
				DocumentRectangle *		drOutside,
				DocumentRectangle *		drInside,
				const DocumentRectangle *	drCell,
				const BufferDocument *		bd,
				const BufferItem *		rowBi,
				int				col,
				int				atRowBottom,
				int				drawTableGrid )
    {
    int				rval= 0;

    int				thick;
    int				space;

    int				useBelow= 0;

    docGetCellBottomBorder( bpBottom, &useBelow,
					bd, rowBi, col, atRowBottom );

    /*  11  */
    if  ( DOCisBORDER( bpBottom ) )
	{
	*pProp= ORNdrawBOTTOM_BORDER; rval= 1;

	thick= docBorderThick( &space, bpBottom );
	drOutside->drY1= drCell->drY1+ thick;
	}

    /*  7  */
    if  ( drawTableGrid )
	{
	if  ( ! DOCisBORDER( bpBottom )		&&
	      ! useBelow			)
	    { *pProp= ORNdrawBOTTOM_GRID; rval= 1;	}
	}

    return rval;
    }

void docGetCellOrnaments(
			BlockOrnaments *		cellOrnaments,
			DocumentRectangle *		drOutside,
			DocumentRectangle *		drInside,
			const DocumentRectangle *	drCell,
			const BufferDocument *		bd,
			const BufferItem *		rowBi,
			int				col,
			int				atRowTop,
			int				atRowBottom,
			int				drawTableGrid )
    {
    const CellProperties *	cp= rowBi->biRowCells+ col;
    const RowProperties *	rp= &(rowBi->biRowProperties);
    int				shadingNumber;

    int				prop= 0;

    *drOutside= *drCell;
    *drInside= *drCell;

    /*  3  */
    shadingNumber= cp->cpShadingNumber;
    if  ( shadingNumber == 0 )
	{ shadingNumber= rp->rpShadingNumber;	}

    if  ( shadingNumber != 0 )
	{
	docGetItemShadingByNumber( &(cellOrnaments->boShading),
						&(bd->bdItemShadingList),
						shadingNumber );

	PROPmaskADD( &(cellOrnaments->boPropMask), ORNdrawSHADE );
	}

    if  ( docCellTopBorderDrawn( &(cellOrnaments->boTopBorder), &prop,
		    drOutside, drInside, drCell,
		    bd, rowBi, col, atRowTop, drawTableGrid ) )
	{ PROPmaskADD( &(cellOrnaments->boPropMask), prop );	}

    if  ( docCellLeftBorderDrawn( &(cellOrnaments->boLeftBorder), &prop,
		    drOutside, drInside, drCell,
		    bd, rowBi, col, drawTableGrid ) )
	{ PROPmaskADD( &(cellOrnaments->boPropMask), prop );	}

    if  ( docCellRightBorderDrawn( &(cellOrnaments->boRightBorder), &prop,
		    drOutside, drInside, drCell,
		    bd, rowBi, col, drawTableGrid ) )
	{ PROPmaskADD( &(cellOrnaments->boPropMask), prop );	}

    if  ( docCellBottomBorderDrawn( &(cellOrnaments->boBottomBorder), &prop,
		    drOutside, drInside, drCell,
		    bd, rowBi, col, atRowBottom, drawTableGrid ) )
	{ PROPmaskADD( &(cellOrnaments->boPropMask), prop );	}

    return;
    }

/************************************************************************/
/*									*/
/*  Decide what top border to use for a table cell (If any)		*/
/*									*/
/*  1)  Get the table row above the current row.			*/
/*  2)  If a cell has a bottom border, and the next one a top border,	*/
/*	MS-Word only draws the bottom border of the topmost cell.	*/
/*  3)  By default use the cells top border.				*/
/*  4)  Use the table top border at the top of a page.			*/
/*									*/
/************************************************************************/

void docGetCellTopBorder(	BorderProperties *		pBpTop,
				int *				pUseAbove,
				const BufferDocument *		bd,
				const BufferItem *		rowBi,
				int				col,
				int				atRowTop )
    {
    const NumberedPropertiesList *	bpl= &(bd->bdBorderPropertyList);
    const BufferItem *		parentBi= rowBi->biParent;
    const CellProperties *	cp= &(rowBi->biRowCells[col]);

    const BufferItem *		rowBiAbove= (const BufferItem *)0;

    const CellProperties *	aboveCpC= (const CellProperties *)0;
    int				useAbove= 0;

    BorderProperties		bpAbove;
    BorderProperties		bpTop;

    /*  1  */
    if  ( rowBi->biNumberInParent > rowBi->biRowTableFirst )
	{
	if  ( rowBi->biNumberInParent <= 0 )
	    { LDEB(rowBi->biNumberInParent);	}
	else{ rowBiAbove= parentBi->biChildren[rowBi->biNumberInParent- 1]; }
	}

    if  ( rowBiAbove )
	{
	int		csp= 1;
	int		cspa= 1;
	int		l, r;
	int		aboveCol;

	if  ( col == 0 )
	    { l= rowBi->biRowLeftIndentTwips;	}
	else{ l= cp[-1].cpRightBoundaryTwips;	}
	r= cp->cpRightBoundaryTwips;
	if  ( cp->cpLeftInMergedRow )
	    { r= docGetCellRight( &csp, rowBi, col );	}

	aboveCol= docGetMatchingCell( &cspa, rowBiAbove, l, r );
	if  ( aboveCol >= 0 )
	    { aboveCpC= rowBiAbove->biRowCells+ aboveCol;	}
	}

    /*  2  */
    if  ( aboveCpC							&&
	  aboveCpC->cpBottomBorderNumber != cp->cpTopBorderNumber	)
	{
	docGetBorderPropertiesByNumber( &bpAbove, bpl,
					    aboveCpC->cpBottomBorderNumber );
	if  ( DOCisBORDER( &bpAbove ) )
	    { useAbove= 1;	}
	}

    /*  3  */
    docGetBorderPropertiesByNumber( &bpTop, bpl, cp->cpTopBorderNumber );

    /*  2  */
    if  ( useAbove )
	{ bpTop= bpAbove;	}

    /*  4  */
    if  ( rowBi->biNumberInParent > rowBi->biRowTableFirst	&&
          ! atRowTop						)
	{
	docGetBorderPropertiesByNumber( &bpTop,
					bpl, rowBi->biRowTopBorderNumber );
	useAbove= 0;
	}

    *pBpTop= bpTop;
    *pUseAbove= useAbove;

    return;
    }

/************************************************************************/
/*									*/
/*  Decide what bottom border to use for a table cell (If any)		*/
/*									*/
/*  0)  Paranoia checks: Bugs that I actually once programmed.		*/
/*  1)  Get the table row below the current row.			*/
/*  2)  By default use the cells bottom border.				*/
/*  3)  If a cell has a bottom boder, and the next one a top border,	*/
/*	MS-Word only draws the bottom border of the topmost cell.	*/
/*  4)  Use the table bottom border at the bottom of a page.		*/
/*									*/
/************************************************************************/

void docGetCellBottomBorder(	BorderProperties *		pBpBottom,
				int *				pUseBelow,
				const BufferDocument *		bd,
				const BufferItem *		rowBi,
				int				col,
				int				atRowBottom )
    {
    const NumberedPropertiesList *	bpl= &(bd->bdBorderPropertyList);
    const BufferItem *		parentBi= rowBi->biParent;
    const CellProperties *	cp= &(rowBi->biRowCells[col]);

    const BufferItem *		rowBiBelow= (const BufferItem *)0;

    const CellProperties *	belowCpC= (const CellProperties *)0;
    int				useBelow= 0;

    BorderProperties		bpBelow;
    BorderProperties 		bpBottom;

    /*  0  */
    if  ( col >= rowBi->biRowCellCount )
	{ LLDEB(col,rowBi->biRowCellCount);	}
    if  ( rowBi->biRowTablePast < 0				||
	  rowBi->biRowTablePast > rowBi->biParent->biChildCount	)
	{ LLDEB(rowBi->biRowTablePast,rowBi->biParent->biChildCount); }

    /*  1  */
    if  ( rowBi->biNumberInParent < rowBi->biRowTablePast- 1 )
	{ rowBiBelow= parentBi->biChildren[rowBi->biNumberInParent+ 1];	}

    if  ( rowBiBelow )
	{
	int		csp= 1;
	int		cspb= 1;
	int		l, r;
	int		belowCol;

	if  ( col == 0 )
	    { l= rowBi->biRowLeftIndentTwips;	}
	else{ l= cp[-1].cpRightBoundaryTwips;	}
	r= cp->cpRightBoundaryTwips;
	if  ( cp->cpLeftInMergedRow )
	    { r= docGetCellRight( &csp, rowBi, col );	}

	belowCol= docGetMatchingCell( &cspb, rowBiBelow, l, r );
	if  ( belowCol >= 0 )
	    { belowCpC= rowBiBelow->biRowCells+ belowCol;	}
	}

    /*  2  */
    docGetBorderPropertiesByNumber( &bpBottom,
					    bpl, cp->cpBottomBorderNumber );

    /*  3  */
    if  ( belowCpC )
	{
	docGetBorderPropertiesByNumber( &bpBelow, bpl,
					    belowCpC->cpBottomBorderNumber );

	if  ( ! DOCisBORDER( &bpBottom )		&&
	      belowCpC					&&
	      DOCisBORDER( &bpBelow )	)
	    { useBelow= 1;	}
	}

    /*  4  */
    if  ( rowBi->biNumberInParent < rowBi->biRowTablePast- 1	&&
	  ! atRowBottom						)
	{
	docGetBorderPropertiesByNumber( &bpBottom,
					bpl, rowBi->biRowBottomBorderNumber );
	useBelow= 0;
	}

    *pBpBottom= bpBottom;
    *pUseBelow= useBelow;

    return;
    }

/************************************************************************/
/*									*/
/*  Calculate layout insets for a (table) row.				*/
/*									*/
/*  This routine is also used to calculate the inset that is used to	*/
/*  accomodate space for the bottom border of the previous row. This	*/
/*  inset is not only used in real rows, but also in the immediate	*/
/*  successor of a real row.						*/
/*									*/
/*  1)  Reserve space for the top border of all cells.			*/
/*									*/
/************************************************************************/

void docLayoutCalculateRowTopInset(	int *			pInset,
					const BufferDocument *	bd,
					const BufferItem *	rowBi,
					int			atRowTop )
    {
    int				col;
    const CellProperties *	cp;

    *pInset= 0;

    /*  1  */
    cp= rowBi->biRowCells;
    for ( col= 0; col < rowBi->biChildCount; cp++, col++ )
	{
	int			useAbove= 0;
	BorderProperties	bpTop;

	if  ( CELL_MERGED( cp ) )
	    { continue;	}

	docGetCellTopBorder( &bpTop, &useAbove, bd, rowBi, col, atRowTop );

	docStretchInsetForBorder( pInset, &bpTop );
	}

    return;
    }

