/************************************************************************/
/*									*/
/*  Buffer administration routines. Functionality related to the item	*/
/*  tree.								*/
/*									*/
/************************************************************************/

#   include	"docBufConfig.h"

#   include	<stdlib.h>
#   include	<string.h>
#   include	<stdio.h>

#   include	<appDebugon.h>

#   include	"docBuf.h"
#   include	"docEdit.h"

#   define	VALIDATE_TREE	0

#if 0

Schema to illustrate the numbering of paragraphs and the 
administration that is kept for that purpose
=========================================================

bi->biLeftParagraphs is the number of paragraphs to the left 
of this item in its immediate parent plus the number of paragraph
descendants. I.E: For items with children, the number of paragraphs 
in the children is included in bi->biLeftParagraphs. A paragraph 
includes itself in its bi->biLeftParagraphs.

Numbers of the paragraphs and the value of bi->biLeftParagraphs:

    1     2     3     4     5     6     7     8     9    10    11    12
    +     +     +     +     +     +     +     +     +     +     +     +
    1     2     3     4     1     2     3     4     5     1     2     3
    |     |     |     |     |     |     |     |     |     |     |     |
    4-----+-----+-----+     9-----+-----+-----+-----+     12----+-----+
    |                       |                             |
    12----------------------+-----------------------------+
    |
    *

Deleting 8 yeilds:

    1     2     3     4     5     6     7           8     9    10    11
    +     +     +     +     +     +     +           +     +     +     +
    1     2     3     4     1     2     3           4     1     2     3
    |     |     |     |     |     |     |           |     |     |     |
    4-----+-----+-----+     8-----+-----+-----+-----+     11----+-----+
    |                       |                             |
    11----------------------+-----------------------------+
    |
    *

So after deleting/inserting paragraphs: Descend to the root of the 
tree. In every parent adapt bi->biLeftParagraphs of all direct 
children to the right of the child that we come from. Set
bi->biLeftParagraphs of the parent to the value in its right most 
child. Continue with the parent of the parent until we have reached 
the root.

NOTE: I am educated as a biologist. My trees have their root at the 
    bottom, not at the top like those of computer scientists that 
    turn the world upside down. (I must admit that the metaphor 
    of parents and descendants has the computer science orientation.)

#endif

/************************************************************************/
/*									*/
/*  Free a BufferItem.							*/
/*									*/
/************************************************************************/

void docCleanItem(	BufferDocument *	bd,
			DocumentTree *		ei,
			BufferItem *		bi )
    {
    int				i;

    for ( i= bi->biChildCount- 1; i >= 0; i-- )
	{ docFreeItem( bd, ei, bi->biChildren[i] ); }
    if  ( bi->biChildren )
	{ free( bi->biChildren );	}

    switch( bi->biLevel )
	{
	case DOClevBODY:
	    break;

	case DOClevSECT:
	    docCleanExternalItem( bd, &(bi->biSectFirstPageHeader) );
	    docCleanExternalItem( bd, &(bi->biSectLeftPageHeader) );
	    docCleanExternalItem( bd, &(bi->biSectRightPageHeader) );

	    docCleanExternalItem( bd, &(bi->biSectFirstPageFooter) );
	    docCleanExternalItem( bd, &(bi->biSectLeftPageFooter) );
	    docCleanExternalItem( bd, &(bi->biSectRightPageFooter) );

	    docCleanSectionProperties( &(bi->biSectProperties) );

	    break;

	case DOClevCELL:
	    break;

	case DOClevROW:
	    docCleanRowProperties( &(bi->biRowProperties) );
	    break;

	case DOClevPARA:
	    docCleanParaItem( bd, ei, bi );
	    break;

	default:
	    /*FALLTHROUGH*/
	case DOClevOUT:
	    LDEB(bi->biLevel);
	    break;
	}

    bi->biLevel= DOClevOUT;
    }

void docFreeItem(	BufferDocument *	bd,
			DocumentTree *		ei,
			BufferItem *		bi )
    {
    docCleanItem( bd, ei, bi );
    free( bi );
    }

/************************************************************************/
/*									*/
/*  Initialise a BufferItem.						*/
/*									*/
/************************************************************************/

void docInitItem(	BufferItem *		bi,
			BufferItem *		parent,
			const BufferDocument *	bd,
			int			numberInParent,
			int			level,
			int			inExternalItem )
    {
    bi->biChildren= (BufferItem **)0;
    bi->biChildCount= 0;
    bi->biLeftParagraphs= 0;

    switch( level )
	{
	case DOClevBODY:
	    break;

	case DOClevSECT:
	    docInitExternalItem( &(bi->biSectFirstPageHeader) );
	    docInitExternalItem( &(bi->biSectLeftPageHeader) );
	    docInitExternalItem( &(bi->biSectRightPageHeader) );

	    docInitExternalItem( &(bi->biSectFirstPageFooter) );
	    docInitExternalItem( &(bi->biSectLeftPageFooter) );
	    docInitExternalItem( &(bi->biSectRightPageFooter) );

	    docInitSectionProperties( &(bi->biSectProperties) );

	    if  ( bd )
		{ bi->biSectDocumentGeometry= bd->bdProperties.dpGeometry; }

	    docInitSelectionScope( &(bi->biSectSelectionScope) );

	    bi->biSectSelectionScope.ssInExternalItem= inExternalItem;
	    bi->biSectSelectionScope.ssSectNr= numberInParent;

	    bi->biSectExternalUseForSectBi= (const BufferItem *)0;
	    bi->biSectExternalUseForPage= -1;
	    bi->biSectExternalUseForColumn= -1;

	    break;

	case DOClevCELL:
	    bi->biCellTopInset= 0;
	    bi->biCellIsMergedWithBelow= 0;
	    bi->biCellMergedCellTopRow= -1;
	    bi->biCellMergedCellTopCol= -1;
	    break;

	case DOClevROW:
	    bi->biRowTableHeaderRow= -1;
	    bi->biRowTableFirst= -1;
	    bi->biRowTablePast= -1;
	    bi->biRowPrecededByHeader= 0;
	    bi->biRowForTable= 0;

	    bi->biRowTopInset= 0;

	    docInitRowProperties( &(bi->biRowProperties) );

	    docInitLayoutPosition( &(bi->biRowBelowAllPosition) );
	    docInitLayoutPosition( &(bi->biRowAboveHeaderPosition) );
	    break;

	case DOClevPARA:
	    docInitParaItem( bi );
	    break;

	default:
	    bi->biLevel= DOClevOUT;
	    bi->biParent= (BufferItem *)0;
	    LDEB(level); return;
	}

    bi->biLevel= level;
    bi->biInExternalItem= inExternalItem;
    bi->biParent= parent;
    bi->biNumberInParent= numberInParent;

    docInitLayoutPosition( &(bi->biTopPosition) );
    docInitLayoutPosition( &(bi->biBelowPosition) );

    return;
    }

/************************************************************************/
/*									*/
/*  1)  Delete a series of items.					*/
/*  2)  Delete an item from its parent.					*/
/*									*/
/************************************************************************/

static void docSectSetSelectionScopes(		BufferItem *	sectBi )
    {
    int		n= sectBi->biNumberInParent;

    sectBi->biSectSelectionScope.ssSectNr= n;

    if  ( sectBi->biSectFirstPageHeader.eiRoot )
	{
	sectBi->biSectFirstPageHeader.eiRoot->
			    biSectSelectionScope.ssOwnerSectNr= n;
	}
    if  ( sectBi->biSectLeftPageHeader.eiRoot )
	{
	sectBi->biSectLeftPageHeader.eiRoot->
			    biSectSelectionScope.ssOwnerSectNr= n;
	}
    if  ( sectBi->biSectRightPageHeader.eiRoot )
	{
	sectBi->biSectRightPageHeader.eiRoot->
			    biSectSelectionScope.ssOwnerSectNr= n;
	}

    if  ( sectBi->biSectFirstPageFooter.eiRoot )
	{
	sectBi->biSectFirstPageFooter.eiRoot->
			    biSectSelectionScope.ssOwnerSectNr= n;
	}
    if  ( sectBi->biSectLeftPageFooter.eiRoot )
	{
	sectBi->biSectLeftPageFooter.eiRoot->
			    biSectSelectionScope.ssOwnerSectNr= n;
	}
    if  ( sectBi->biSectRightPageFooter.eiRoot )
	{
	sectBi->biSectLeftPageFooter.eiRoot->
			    biSectSelectionScope.ssOwnerSectNr= n;
	}

    return;
    }

/************************************************************************/
/*									*/
/*  Paragraphs have been deleted from a group item.			*/
/*  Administration in the child array has been done. Fix		*/
/*  biLeftParagraphs in the item itself and its parents.		*/
/*									*/
/************************************************************************/

static void docParagraphsDeleted(	BufferItem *	bi,
					int		paragraphsDeleted )
    {
    while( bi->biParent )
	{
	int		first;
	int		f;

	first= bi->biNumberInParent;
	bi= bi->biParent;

	for ( f= first; f < bi->biChildCount; f++ )
	    { bi->biChildren[f]->biLeftParagraphs -= paragraphsDeleted;	}
	}

    bi->biLeftParagraphs -= paragraphsDeleted;

    return;
    }

/*  1  */
void docDeleteItems(	BufferDocument *	bd,
			DocumentTree *		ei,
			BufferItem *		bi,
			int			first,
			int			count )
    {
    int		n;
    int		f;
    int		c;
    int		paragraphsDeleted= 0;

#   if VALIDATE_TREE
    if  ( docCheckItem( bd->bdBody.eiRoot ) )
	{ LDEB(2); docListItem( 0, bd->bdBody.eiRoot ); abort(); }
#   endif

    if  ( first > bi->biChildCount )
	{
	LLDEB(first,bi->biChildCount);
	first= bi->biChildCount;
	}

    if  ( first+ count > bi->biChildCount )
	{
	LLDEB(first+count,bi->biChildCount);
	count= bi->biChildCount- first;
	}

    if  ( count <= 0 )
	{ LDEB(count); return;	}

    if  ( count > 0 )
	{
	f= 0;
	if  ( first > 0 )
	    { f= bi->biChildren[first- 1]->biLeftParagraphs;	}

	c= bi->biChildren[first+ count- 1]->biLeftParagraphs;

	paragraphsDeleted= c- f;
	}

    n= first+ count- 1;
    while( n >= first )
	{ docFreeItem( bd, ei, bi->biChildren[n] ); n--; }

    bi->biChildCount -= count;

    f= first;
    while( f < bi->biChildCount )
	{
	bi->biChildren[f]= bi->biChildren[f+ count];
	bi->biChildren[f]->biNumberInParent= f;
	bi->biChildren[f]->biLeftParagraphs -= paragraphsDeleted;

	if  ( bi->biChildren[f]->biLevel == DOClevSECT )
	    { docSectSetSelectionScopes( bi->biChildren[f] ); }

	f++;
	}

    docParagraphsDeleted( bi, paragraphsDeleted );

#   if VALIDATE_TREE
    if  ( docCheckItem( bd->bdBody.eiRoot ) )
	{ LDEB(2); docListItem( 0, bd->bdBody.eiRoot ); abort(); }
#   endif

    return;
    }

/*  2  */
void docDeleteItem(	BufferDocument *	bd,
			DocumentTree *		ei,
			BufferItem *		bi )
    {
    if  ( bi->biParent )
	{
	docDeleteItems( bd, ei, bi->biParent, bi->biNumberInParent, 1 );
	}
    else{ docFreeItem( bd, ei, bi );					}
    }

/************************************************************************/
/*									*/
/*  Delete the Document Tree Node that is the root of a document Tree	*/
/*  This is NOT the way to delete a tree node. Use			*/
/*  docCleanExternalItem() to do so.					*/
/*									*/
/************************************************************************/

void docDeleteExternalItem(	BufferDocument *	bd,
				DocumentTree *		ei )
    { docFreeItem( bd, ei, ei->eiRoot );	}

/************************************************************************/
/*									*/
/*  Paragraphs have been inserted into a group item.			*/
/*  Administration in the child array has been done. Fix		*/
/*  biLeftParagraphs in the item itself and its parents.		*/
/*									*/
/************************************************************************/

static void docParagraphsInserted(	BufferItem *	bi,
					int		paragraphsInserted )
    {
    while( bi->biParent )
	{
	int		first;
	int		f;

	first= bi->biNumberInParent;
	bi= bi->biParent;

	for ( f= first; f < bi->biChildCount; f++ )
	    { bi->biChildren[f]->biLeftParagraphs += paragraphsInserted; }
	}

    bi->biLeftParagraphs += paragraphsInserted;

    return;
    }

/************************************************************************/

int docValidChildLevel(		int		parentLevel,
				int		childLevel )
    {
    switch( parentLevel )
	{
	case DOClevBODY:
	    if  ( childLevel != DOClevSECT )
		{ return 0;	}
	    break;

	case DOClevSECT:
	    if  ( childLevel != DOClevROW )
		{ return 0; }
	    break;

	case DOClevROW:
	    if  ( childLevel != DOClevCELL )
		{ return 0; }
	    break;

	case DOClevCELL:
	    if  ( childLevel != DOClevPARA && childLevel != DOClevROW )
		{ return 0; }
	    break;

	default:
	    LLDEB(parentLevel,childLevel); return 0;
	}

    return 1;
    }

/************************************************************************/
/*									*/
/*  Add a new child to a parent.					*/
/*									*/
/************************************************************************/

BufferItem * docInsertItem(	const BufferDocument *	bd,
				BufferItem *		parent,
				int			n,
				ItemLevel		level )
    {
    BufferItem *	rval= (BufferItem *)0;
    BufferItem *	newBi= (BufferItem *)0;

    int			i;

    int			newSize;

    BufferItem **	freshChildren;

    int			paragraphsInserted;

#   if VALIDATE_TREE
    if  ( docCheckItem( bd->bdBody.eiRoot ) )
	{ LDEB(2); docListItem( 0, bd->bdBody.eiRoot ); abort(); }
#   endif

    if  ( ! docValidChildLevel( parent->biLevel, level ) )
	{ SSDEB(docLevelStr(parent->biLevel),docLevelStr(level)); goto ready; }

    if  ( n == -1 )
	{ n= parent->biChildCount;	}

    newSize= parent->biChildCount;

    if  ( newSize % 10 )
	{ newSize ++;		}
    else{ newSize += 10;	}

    newSize *= sizeof(BufferItem *);

    freshChildren= (BufferItem **)realloc( parent->biChildren, newSize );
    if  ( ! freshChildren )
	{ LXDEB(newSize,freshChildren); goto ready; }
    parent->biChildren= freshChildren;

    newBi= (BufferItem *)malloc( sizeof(BufferItem) );
    if  ( ! newBi )
	{ XDEB(newBi); goto ready;	}

    docInitItem( newBi, parent, bd, n, level, parent->biInExternalItem );

    if  ( n == 0 )
	{ newBi->biTopPosition= parent->biTopPosition;			}
    else{ newBi->biTopPosition= freshChildren[n-1]->biBelowPosition;	}

    for ( i= parent->biChildCount; i > n; i-- )
	{
	freshChildren[i]= freshChildren[i-1];

	freshChildren[i]->biNumberInParent= i;

	if  ( freshChildren[i]->biLevel == DOClevSECT )
	    { docSectSetSelectionScopes( freshChildren[i] ); }
	}

    freshChildren[n]= newBi;
    parent->biChildCount++;

    rval= newBi; newBi= (BufferItem *)0; /* steal */

    if  ( level == DOClevPARA )
	{
	paragraphsInserted= 1;

	docSetParaTableNesting( rval );

	if  ( n > 0 )
	    { rval->biLeftParagraphs= freshChildren[n-1]->biLeftParagraphs; }

	for ( i= n; i < parent->biChildCount; i++ )
	    { parent->biChildren[i]->biLeftParagraphs += paragraphsInserted; }

	docParagraphsInserted( parent, paragraphsInserted );
	}
    else{
	if  ( n > 0 )
	    { rval->biLeftParagraphs= freshChildren[n-1]->biLeftParagraphs; }
	}

#   if VALIDATE_TREE
    if  ( docCheckItem( bd->bdBody.eiRoot ) )
	{ LDEB(2); docListItem( 0, bd->bdBody.eiRoot ); abort(); }
#   endif

  ready:

    if  ( newBi )
	{ free( newBi );	}

    return rval;
    }

/************************************************************************/
/*									*/
/*  Make an empty paragraph: Needed at several locations.		*/
/*									*/
/************************************************************************/

BufferItem * docInsertEmptyParagraph(
				BufferDocument *	bd,
				BufferItem *		bi,
				int			textAttributeNumber )
    {
    if  ( bi->biLevel < DOClevSECT )
	{ LDEB(bi->biLevel); return (BufferItem *)0;	}

    if  ( bi->biLevel < DOClevROW )
	{
	bi= docInsertItem( bd, bi, -1, DOClevROW );
	if  ( ! bi )
	    { XDEB(bi); return (BufferItem *)0;   }
	}

    if  ( bi->biLevel < DOClevCELL )
	{
	bi= docInsertItem( bd, bi, -1, DOClevCELL );
	if  ( ! bi )
	    { XDEB(bi); return (BufferItem *)0;   }
	}

    if  ( bi->biLevel < DOClevPARA )
	{
	bi= docInsertItem( bd, bi, -1, DOClevPARA );
	if  ( ! bi )
	    { XDEB(bi); return (BufferItem *)0;   }
	}
    else{
	bi= docInsertItem( bd, bi->biParent, -1, DOClevPARA );
	if  ( ! bi )
	    { XDEB(bi); return (BufferItem *)0;   }
	}

    if  ( ! docInsertTextParticule( bi, 0, 0, 0,
				    DOCkindSPAN, textAttributeNumber ) )
	{ LDEB(1); return (BufferItem *)0;	}

    return bi;
    }

/************************************************************************/

static BufferItem * docCopyXItem(	DocumentCopyJob *	dcj,
					const SelectionScope *	ssRoot,
					BufferItem *		parentBiTo,
					int			n,
					const BufferItem *	biFrom );

static BufferItem * docCopyCellItem(
				DocumentCopyJob *	dcj,
				const SelectionScope *	ssRoot,
				BufferItem *		biRowTo,
				int			col,
				const BufferItem *	biCellFrom )
    {
    EditOperation *	eo= dcj->dcjEditOperation;
    BufferDocument *	bdTo= eo->eoBd;

    BufferItem *	biCellTo;
    int			para;

    biCellTo= docInsertItem( bdTo, biRowTo, col, DOClevCELL );

    if  ( ! biCellTo )
	{ XDEB(biCellTo); return (BufferItem *)0;	}

    for ( para= 0; para < biCellFrom->biChildCount; para++ )
	{
	BufferItem *	biParaFrom= biCellFrom->biChildren[para];

	if  ( ! docCopyXItem( dcj, ssRoot, biCellTo, para, biParaFrom ) )
	    { LDEB(para); return (BufferItem *)0;	}
	}

    return biCellTo;
    }

static BufferItem * docCopyRowItem(	DocumentCopyJob *	dcj,
					const SelectionScope *	ssRoot,
					BufferItem *		sectBiTo,
					int			n,
					const BufferItem *	rowBiFrom )
    {
    EditOperation *	eo= dcj->dcjEditOperation;
    BufferDocument *	bdTo= eo->eoBd;

    BufferItem *	rowBiTo;
    int			col;

    rowBiTo= docInsertItem( bdTo, sectBiTo, n, DOClevROW );
    if  ( ! rowBiTo )
	{ XDEB(rowBiTo); return rowBiTo;	}

    if  ( docCopyRowProperties( &(rowBiTo->biRowProperties),
		    &(rowBiFrom->biRowProperties), &(dcj->dcjAttributeMap ) ) )
	{ LDEB(1); return (BufferItem *)0;	}

    for ( col= 0; col < rowBiFrom->biChildCount; col++ )
	{
	BufferItem *	biCellTo;

	biCellTo= docCopyXItem( dcj, ssRoot, rowBiTo, col,
						rowBiFrom->biChildren[col] );
	if  ( ! biCellTo )
	    { XDEB(biCellTo); return (BufferItem *)0;	}
	}

    return rowBiTo;
    }

/************************************************************************/
/*									*/
/*  Copy a section.							*/
/*									*/
/************************************************************************/

int docCopySectChildren(	DocumentCopyJob *	dcj,
				BufferItem *		sectBiTo,
				const BufferItem *	sectBiFrom )
    {
    int		row;

    for ( row= 0; row < sectBiFrom->biChildCount; row++ )
	{
	BufferItem *	rowBiFrom= sectBiFrom->biChildren[row];
	BufferItem *	rowBiTo;

	rowBiTo= docCopyXItem( dcj, &(sectBiTo->biSectSelectionScope),
						sectBiTo, row, rowBiFrom );
	if  ( ! rowBiTo )
	    { XDEB(rowBiTo); return -1;	}
	}

    return 0;
    }

static BufferItem * docCopySectItem(	DocumentCopyJob *	dcj,
					const SelectionScope *	ssRoot,
					BufferItem *		parentBiTo,
					int			n,
					const BufferItem *	sectBiFrom )
    {
    EditOperation *	eo= dcj->dcjEditOperation;
    BufferDocument *	bdTo= eo->eoBd;

    BufferItem *	rval= (BufferItem *)0;
    BufferItem *	sectBiTo= (BufferItem *)0;

    if  ( parentBiTo )
	{
	if  ( parentBiTo->biInExternalItem != ssRoot->ssInExternalItem )
	    {
	    LDEB(parentBiTo->biInExternalItem);
	    LDEB(ssRoot->ssInExternalItem);
	    goto ready;
	    }

	sectBiTo= docInsertItem( bdTo, parentBiTo, n, DOClevSECT );
	if  ( ! sectBiTo )
	    { XDEB(sectBiTo); goto ready;	}
	}
    else{
	n= 0;
	sectBiTo= (BufferItem *)malloc( sizeof(BufferItem) );

	if  ( ! sectBiTo )
	    { XDEB(sectBiTo); goto ready;	}

	docInitItem( sectBiTo, (BufferItem *)0, bdTo, n,
				DOClevSECT, ssRoot->ssInExternalItem );
	}

    sectBiTo->biSectSelectionScope= *ssRoot;

    if  ( docCopySectionProperties( &(sectBiTo->biSectProperties),
					&(sectBiFrom->biSectProperties) ) )
	{ LDEB(1); goto ready;	}

    if  ( docCopySectChildren( dcj, sectBiTo, sectBiFrom ) )
	{ LDEB(1); goto ready;	}

    if  ( ! dcj->dcjInExternalItem && parentBiTo )
	{
	DocumentPosition		dp;

	if  ( docLastPosition( &dp, sectBiTo ) )
	    { LDEB(1);	}
	else{
	    int		paraNr;
	    const int	stroff= 0;
	    const int	sectShift= 1;
	    const int	paraShift= 0;

	    paraNr= docNumberOfParagraph( dp.dpBi )+ 1;

	    docEditShiftReferences( eo, ssRoot, paraNr, stroff,
					    sectShift, paraShift, -stroff );
	    }
	}

    rval= sectBiTo; sectBiTo= (BufferItem *)0; /* steal */

  ready:

    if  ( sectBiTo )
	{ docDeleteItem( eo->eoBd, eo->eoEi, sectBiTo );	}

    return rval;
    }

static BufferItem * docCopyXItem(	DocumentCopyJob *	dcj,
					const SelectionScope *	ssRoot,
					BufferItem *		parentBiTo,
					int			n,
					const BufferItem *	biFrom )
    {
    switch( biFrom->biLevel )
	{
	case DOClevPARA:
	    return docCopyParaItem( dcj, ssRoot, parentBiTo, n, biFrom );
	case DOClevCELL:
	    return docCopyCellItem( dcj, ssRoot, parentBiTo, n, biFrom );
	case DOClevROW:
	    return docCopyRowItem(  dcj, ssRoot, parentBiTo, n, biFrom );
	case DOClevSECT:
	    return docCopySectItem( dcj, ssRoot, parentBiTo, n, biFrom );
	default:
	    LDEB(biFrom->biLevel); return (BufferItem *)0;
	}
    }

BufferItem * docCopyItem(	DocumentCopyJob *	dcj,
				BufferItem *		parentBiTo,
				int			n,
				const BufferItem *	biFrom )
    {
    EditOperation *		eo= dcj->dcjEditOperation;
    const SelectionScope *	ssRoot= &(eo->eoSelectionScope);

    return docCopyXItem( dcj, ssRoot, parentBiTo, n, biFrom );
    }

int docCopyDocumentTree( 	DocumentCopyJob *	dcj,
				DocumentTree *		eiTo,
				const SelectionScope *	ssRoot,
				const BufferItem *	sectBiFrom )
    {
    int				rval= 0;
    FieldCopyStackLevel *	fcsl= dcj->dcjFieldStack;
    SelectionScope		ssSave= dcj->dcjSelectionScope;
    DocumentTree *		eiSave= dcj->dcjEiTo;

    dcj->dcjSelectionScope= *ssRoot;
    dcj->dcjEiTo= eiTo;
    dcj->dcjFieldStack= (FieldCopyStackLevel *)0;
    dcj->dcjInExternalItem++;

    eiTo->eiRoot= docCopySectItem( dcj, ssRoot, (BufferItem *)0, 0,
							    sectBiFrom );
    if  ( ! eiTo->eiRoot )
	{ XDEB(eiTo->eiRoot); rval= -1;	}

    if  ( dcj->dcjFieldStack )
	{ XDEB(dcj->dcjFieldStack);	}

    dcj->dcjInExternalItem--;
    dcj->dcjFieldStack= fcsl;
    dcj->dcjEiTo= eiSave;
    dcj->dcjSelectionScope= ssSave;

    return rval;
    }

/************************************************************************/
/*									*/
/*  Insert a new row in a table.					*/
/*									*/
/************************************************************************/

BufferItem * docInsertRowItem(	BufferDocument *	bd,
				BufferItem *		sectBi,
				int			n,
				const RowProperties *	rp,
				int			textAttributeNumber )
    {
    int				col;

    BufferItem *		rval= (BufferItem *)0;
    BufferItem *		rowBi= (BufferItem *)0;

    rowBi= docInsertItem( bd, sectBi, n, DOClevROW );
    if  ( ! rowBi )
	{ XDEB(rowBi); goto ready;	}

    if  ( docCopyRowProperties( &(rowBi->biRowProperties), rp,
					(const DocumentAttributeMap *)0 ) )
	{ LDEB(1); goto ready; }

    for ( col= 0; col < rp->rpCellCount; col++ )
	{
	BufferItem *	paraBi;

	paraBi= docInsertEmptyParagraph( bd, rowBi, textAttributeNumber );
	if  ( ! paraBi )
	    { XDEB(paraBi); goto ready; }

	docSetParaTableNesting( paraBi );
	}

    rval= rowBi; rowBi= (BufferItem *)0; /* steal */

  ready:

    if  ( rowBi )
	{ docDeleteItem( bd, (DocumentTree *)0, rowBi );	}

    return rval;
    }

/************************************************************************/
/*									*/
/*  Split an item with childen.						*/
/*									*/
/*  1)  See whether there is a parent of the given level that can be	*/
/*	split.								*/
/*									*/
/************************************************************************/

static int docSplitGroupItemLow(	BufferDocument *	bd,
					BufferItem **		pNewBi,
					BufferItem *		oldBi,
					int			n )
    {
    BufferItem *	newBi;
    int			i;
    int			prev;

    newBi= docInsertItem( bd, oldBi->biParent,
				oldBi->biNumberInParent, oldBi->biLevel );
    if  ( ! newBi )
	{ XDEB(newBi); return -1;	}

    switch( oldBi->biLevel )
	{
	case DOClevSECT:
	    if  ( docCopySectionProperties( &newBi->biSectProperties,
					    &oldBi->biSectProperties ) )
		{ LDEB(1); return -1;	}

		if  ( oldBi->biInExternalItem == DOCinBODY		&&
		      docCopySectHeadersFooters( newBi, bd, oldBi, bd ) )
		    { LDEB(1); return -1;	}
	    break;

	case DOClevCELL:
	    break;

	case DOClevROW:
	    if  ( docCopyRowProperties( &(newBi->biRowProperties),
				&(oldBi->biRowProperties),
				(const DocumentAttributeMap *)0 ) )
		{ LDEB(1); return -1;	}
	    break;

	default:
	    LDEB(oldBi->biLevel); return -1;
	}

    newBi->biChildren= (BufferItem **)malloc( n* sizeof(BufferItem *) );
    if  ( ! newBi->biChildren )
	{ XDEB(newBi->biChildren); return -1;	}

    for ( i= 0; i < n; i++ )
	{
	newBi->biChildren[i]= oldBi->biChildren[i];
	newBi->biChildren[i]->biParent= newBi;
	}

    prev= 0;
    if  ( newBi->biNumberInParent > 0 )
	{
	prev= newBi->biParent->biChildren[newBi->biNumberInParent-1]->
							biLeftParagraphs;
	}

    if  ( n == 0 )
	{ newBi->biLeftParagraphs= prev; }
    else{
	newBi->biLeftParagraphs= prev+ newBi->biChildren[n-1]->biLeftParagraphs;
	}

    newBi->biChildCount= n;
    oldBi->biChildCount -= n;

    prev= 0;
    for ( i= 0; i < oldBi->biChildCount; i++ )
	{
	BufferItem *	child= oldBi->biChildren[i+ n];

	oldBi->biChildren[i]= child;
	child->biNumberInParent -= n;

	if  ( oldBi->biChildren[i]->biLevel == DOClevPARA )
	    { prev++;	}
	else{
	    if  ( child->biChildCount > 0 )
		{
		prev += child->biChildren[child->biChildCount-1]->
							    biLeftParagraphs;
		}
	    }

	oldBi->biChildren[i]->biLeftParagraphs= prev;
	}

    *pNewBi= newBi; return 0;
    }

/************************************************************************/
/*									*/
/*  Split a 'group item' I.E. an item that has children rather than	*/
/*  its own text.							*/
/*									*/
/*  The goal is to make sure that the split before position n in	*/
/*  parentBi is between two BufferItems with level level.		*/
/*									*/
/************************************************************************/

int docSplitGroupItem(	BufferDocument *	bd,
			BufferItem **		pNewBi,
			BufferItem **		pParentBi,
			BufferItem *		parentBi,
			int			n,
			int			level )
    {
    BufferItem *	newBi= (BufferItem *)0;
    BufferItem *	splitBi;

#   if VALIDATE_TREE
    SDEB(docLevelStr(parentBi->biLevel));
    if  ( docCheckItem( bd->bdBody.eiRoot ) )
	{ LDEB(2); docListItem( 0, bd->bdBody.eiRoot ); abort(); }
#   endif

    splitBi= parentBi;
    while( splitBi && splitBi->biLevel != level- 1 )
	{ splitBi= splitBi->biParent; }
    if  ( ! splitBi )
	{
	SSXDEB(docLevelStr(parentBi->biLevel),docLevelStr(level),splitBi);
	return -1;
	}

    if  ( n < 0 || n >= parentBi->biChildCount )
	{ LLDEB(n,parentBi->biChildCount); return -1;	}

    for (;;)
	{
	if  ( n > 0 || parentBi->biLevel == level )
	    {
	    if  ( docSplitGroupItemLow( bd, &newBi, parentBi, n ) )
		{ LDEB(n); return -1;	}

	    if  ( parentBi->biLevel == level )
		{ break;	}
	    }

	n= parentBi->biNumberInParent;
	parentBi= parentBi->biParent;
	}

#   if VALIDATE_TREE
    SDEB(docLevelStr(parentBi->biLevel));
    if  ( docCheckItem( bd->bdBody.eiRoot ) )
	{ LDEB(2); docListItem( 0, bd->bdBody.eiRoot ); abort(); }
#   endif

    *pNewBi= newBi;
    *pParentBi= parentBi;
    return 0;
    }

/************************************************************************/
/*									*/
/*  Merge the children of two items into the first one.			*/
/*									*/
/************************************************************************/

int docMergeGroupItems(		BufferItem *	to,
				BufferItem *	from )
    {
    BufferItem **	freshChildren;
    int			f;
    int			t;
    int			left;
    int			prev;
    int			paragraphsMoved;

    if  ( to == from )
	{ XXDEB(to,from); return -1;	}
    if  ( to->biLevel != from->biLevel )
	{ LLDEB(to->biLevel,from->biLevel); return -1;	}

    if  ( from->biChildCount == 0 )
	{ return 0;	}

#   if VALIDATE_TREE
    SSDEB(docLevelStr(to->biLevel),docLevelStr(from->biLevel));
    if  ( docCheckRootItem( to ) )
	{ LDEB(2); docListRootItem( 0, to ); abort(); }
    if  ( docCheckRootItem( from ) )
	{ LDEB(2); docListRootItem( 0, from ); abort(); }
#   endif

    freshChildren= realloc( to->biChildren,
	    ( to->biChildCount+ from->biChildCount )* sizeof(BufferItem *) );
    if  ( ! freshChildren )
	{
	LLXDEB(to->biChildCount,from->biChildCount,freshChildren);
	return -1;
	}
    to->biChildren= freshChildren;

    if  ( from->biParent && from->biNumberInParent > 0 )
	{
	BufferItem *	prevBi;

	prevBi= from->biParent->biChildren[from->biNumberInParent- 1];
	paragraphsMoved= from->biLeftParagraphs- prevBi->biLeftParagraphs;
	}
    else{
	paragraphsMoved= from->biLeftParagraphs;
	}

    left= 0; prev= 0;
    t= to->biChildCount;
    if  ( t > 0 )
	{ left= to->biChildren[t- 1]->biLeftParagraphs; }
    for ( f= 0; f < from->biChildCount; t++, f++ )
	{
	int	count;

	freshChildren[t]= from->biChildren[f];
	freshChildren[t]->biParent= to;
	freshChildren[t]->biNumberInParent= t;

	count= freshChildren[t]->biLeftParagraphs- prev;
	prev= freshChildren[t]->biLeftParagraphs;
	left += count;
	freshChildren[t]->biLeftParagraphs= left;

	if  ( freshChildren[t]->biLevel == DOClevSECT )
	    { docSectSetSelectionScopes( freshChildren[t] ); }
	}

    to->biChildCount += from->biChildCount;
    from->biChildCount= 0;

    docParagraphsInserted( to, paragraphsMoved );
    docParagraphsDeleted( from, paragraphsMoved );

#   if VALIDATE_TREE
    SSDEB(docLevelStr(to->biLevel),docLevelStr(from->biLevel));
    if  ( docCheckRootItem( to ) )
	{ LDEB(2); docListRootItem( 0, to ); abort(); }
    if  ( docCheckRootItem( from ) )
	{ LDEB(2); docListRootItem( 0, from ); abort(); }
#   endif

    return 0;
    }

/************************************************************************/
/*									*/
/*  Return the number of a paragraph.					*/
/*									*/
/************************************************************************/

int docNumberOfParagraph(	const BufferItem *	bi )
    {
    int		n= 0;

    if  ( bi->biLevel != DOClevPARA )
	{ SDEB(docLevelStr(bi->biLevel)); return -1;	}

    n= bi->biLeftParagraphs;

    while( bi->biParent )
	{
	bi= bi->biParent;

	if  ( bi->biParent && bi->biNumberInParent > 0 )
	    {
	    if  ( bi->biNumberInParent >= bi->biParent->biChildCount )
		{ LLDEB(bi->biNumberInParent,bi->biParent->biChildCount); }

	    n += bi->biParent->biChildren[
				bi->biNumberInParent- 1]->biLeftParagraphs;
	    }
	}

    return n;
    }

BufferItem * docGetParagraphByNumber(	const DocumentTree *	ei,
					int			paraNr )
    {
    BufferItem *	bi= ei->eiRoot;

    if  ( paraNr < 1 )
	{ LDEB(paraNr); return (BufferItem *)0;	}

    while( bi->biChildCount > 0 )
	{
	int		i;

	for ( i= 0; i < bi->biChildCount; i++ )
	    {
	    if  ( bi->biChildren[i]->biLeftParagraphs >= paraNr )
		{ break;	}
	    }

	if  ( i >= bi->biChildCount )
	    {
	    /* LLSDEB(paraNr,bi->biChildCount,docLevelStr(bi->biLevel)); */
	    return (BufferItem *)0;
	    }

	if  ( i > 0 )
	    { paraNr -= bi->biChildren[i-1]->biLeftParagraphs;	}

	bi= bi->biChildren[i];
	}

    if  ( bi->biLevel != DOClevPARA )
	{ SDEB(docLevelStr(bi->biLevel)); return (BufferItem *)0;	}

    if  ( paraNr != 1 )
	{ LDEB(paraNr); return (BufferItem *)0; }

    return bi;
    }

/************************************************************************/
/*									*/
/*  Return the nearest parent of a BufferItem that is a real row.	*/
/*  candidate row, cell.						*/
/*									*/
/************************************************************************/

BufferItem * docGetRowItem(		BufferItem *		bi )
    {
    while( bi && ! docIsRowItem( bi ) )
	{ bi= bi->biParent;	}

    return bi;
    }

BufferItem * docGetRowLevelItem(	BufferItem *		bi )
    {
    while( bi					&&
	   bi->biLevel != DOClevROW		)
	{ bi= bi->biParent;	}

    return bi;
    }

BufferItem * docGetCellItem(		BufferItem *		bi )
    {
    while( bi					&&
	   bi->biLevel != DOClevCELL		)
	{ bi= bi->biParent;	}

    return bi;
    }

/************************************************************************/
/*									*/
/*  Return the nearest parent of a BufferItem that is a section.	*/
/*									*/
/************************************************************************/

BufferItem * docGetSectItem(		BufferItem *		bi )
    {
    while( bi					&&
	   bi->biLevel != DOClevSECT		)
	{ bi= bi->biParent;	}

    return bi;
    }

/************************************************************************/
/*									*/
/*  Calculate table nesting: Follow the path to the root and count	*/
/*  the number of (table) rows that we traverse.			*/
/*									*/
/************************************************************************/

int docTableNesting(		const BufferItem *	bi )
    {
    int			tableNesting= 0;

    while( bi )
	{
	if  ( docIsRowItem( bi ) )
	    { tableNesting++;	}

	bi= bi->biParent;
	}

    return tableNesting;
    }

int docRowNesting(		const BufferItem *	bi )
    {
    int			rowNesting= 0;

    while( bi )
	{
	if  ( bi->biLevel == DOClevROW )
	    { rowNesting++;	}

	bi= bi->biParent;
	}

    return rowNesting;
    }

void docSetParaTableNesting(		BufferItem *	paraBi )
    {
    if  ( paraBi->biLevel != DOClevPARA )
	{ SDEB(docLevelStr(paraBi->biLevel)); return;	}

    paraBi->biParaTableNesting= docTableNesting( paraBi );
    }

/************************************************************************/
/*									*/
/*  Change the kind of external item for a node and all its children.	*/
/*									*/
/************************************************************************/

void docSetExternalItemKind(	BufferItem *		bi,
				int			extItKind )
    {
    int		i;

    bi->biInExternalItem= extItKind;

    if  ( bi->biLevel == DOClevSECT )
	{ bi->biSectSelectionScope.ssInExternalItem= extItKind; }

    for ( i= 0; i < bi->biChildCount; i++ )
	{ docSetExternalItemKind( bi->biChildren[i], extItKind );	}

    return;
    }

