/************************************************************************/
/*									*/
/*  Save a BufferDocument into an HTML file.				*/
/*  Depending on the parameters, this is either an HTML file with	*/
/*  a directory for the images, or a MHTML (rfc2112,rfc2557) aggregate.	*/
/*  RFC 2557 was never validated.					*/
/*									*/
/************************************************************************/

#   include	"docBufConfig.h"

#   include	<stdlib.h>
#   include	<string.h>
#   include	<stdio.h>
#   include	<ctype.h>

#   include	<bitmap.h>

#   include	<appSystem.h>
#   include	<uniLegacyEncoding.h>

#   include	<sioGeneral.h>
#   include	<sioStdio.h>
#   include	<sioMemory.h>
#   include	<sioHex.h>
#   include	<sioBase64.h>

#   include	"docBuf.h"
#   include	"docPageGrid.h"

#   include	<appDebugon.h>

#   include	<charnames.h>

#   define	TWIPS_TO_SIZE(x)	(((x)+9)/18)

#   define	IDlenTAIL		200

#   define	USE_PNG			0
#   define	USE_GIF			1

/************************************************************************/
/*									*/
/*  Information used when writing HTML.					*/
/*									*/
/************************************************************************/

typedef struct HtmlWritingContext
    {
    SimpleOutputStream *	hwcSos;
    BufferDocument *		hwcBd;

    int				hwcAsMimeAggregate;

    const char *		hwcFilename;
    const char *		hwcMimeBoundary;

    int				hwcBaselength;
    int				hwcRelativeOffset;
    int				hwcInHyperlink;
    int				hwcInBookmark;
    int				hwcInPageref;
    int				hwcBytesInLink;
    TextAttribute		hwcDefaultAttribute;
    ParagraphProperties		hwcParagraphProperties;

    int				hwcColumn;

    int				hwcImageCount;
    int				hwcNoteRefCount;
    int				hwcNoteDefCount;

    char *			hwcNameScratch;
    int				hwcWriteImageFromObjectData;
    bmWriteBitmap		hwcWriteThisBitmap;

    char			hwcImageMimeType[40+1];
    char			hwcContentIdTail[IDlenTAIL+1];

    int				hwcCurrentAttributeNumber;
    } HtmlWritingContext;

static int docHtmlSaveItem(	const BufferItem *		bi,
				const BufferItem *		bodySectBi,
				HtmlWritingContext *		hwc );

static void docInitHtmlWritingContext(	HtmlWritingContext *	hwc )
    {
    hwc->hwcSos= (SimpleOutputStream *)0;
    hwc->hwcBd= (BufferDocument *)0;

    hwc->hwcAsMimeAggregate= 0;

    hwc->hwcFilename= (const char *)0;
    hwc->hwcMimeBoundary= (const char *)0;

    hwc->hwcBaselength= 0;
    hwc->hwcRelativeOffset= 0;
    hwc->hwcInHyperlink= 0;
    hwc->hwcInBookmark= 0;
    hwc->hwcInPageref= 0;
    hwc->hwcBytesInLink= 0;

    docPlainTextAttribute( &(hwc->hwcDefaultAttribute), hwc->hwcBd );
    hwc->hwcDefaultAttribute.taFontSizeHalfPoints= 24;
    hwc->hwcDefaultAttribute.taFontNumber= -1;

    docInitParagraphProperties( &(hwc->hwcParagraphProperties) );

    hwc->hwcColumn= 0;

    hwc->hwcImageCount= 0;
    hwc->hwcNoteRefCount= 0;
    hwc->hwcNoteDefCount= 0;

    hwc->hwcNameScratch= (char *)0;
    hwc->hwcWriteImageFromObjectData= 0;
    hwc->hwcWriteThisBitmap= (bmWriteBitmap)0;

    hwc->hwcImageMimeType[0]= '\0';
    hwc->hwcContentIdTail[0]= '\0';

    hwc->hwcCurrentAttributeNumber= -1;
    return;
    }

static void docCleanHtmlWritingContext(	HtmlWritingContext *	hwc )
    {
    docCleanParagraphProperties( &(hwc->hwcParagraphProperties) );

    if  ( hwc->hwcNameScratch )
	{ free( hwc->hwcNameScratch );	}

    return;
    }

/************************************************************************/
/*									*/
/*  Make the bitmap for an image.					*/
/*  Make a name for an image.						*/
/*									*/
/************************************************************************/

static void docHtmlSetImageNameAndType(	HtmlWritingContext *		hwc,
					const PictureProperties *	pip,
					const char *			ext,
					const char *			mime )
    {
    if  ( hwc->hwcAsMimeAggregate || hwc->hwcBaselength == 0 )
	{ sprintf( hwc->hwcNameScratch, "%08lx.%s", pip->pipBliptag, ext ); }
    else{
	sprintf( hwc->hwcNameScratch, "%.*s.img/%08lx.%s",
		hwc->hwcBaselength, hwc->hwcFilename, pip->pipBliptag, ext );
	}

    strcpy( hwc->hwcImageMimeType, mime );

    return;
    }

static int docHtmlMakeNameForImage(	HtmlWritingContext *	hwc,
					InsertedObject *	io )
    {
    char *		fresh;
    AppBitmapImage *	abi;
    int			siz;

    PictureProperties *	pip= &(io->ioPictureProperties);

    siz= hwc->hwcBaselength+ 50;
    fresh= (char *)realloc( hwc->hwcNameScratch, siz+ 1 );
    if  ( ! fresh )
	{ LXDEB(siz,fresh); return -1;  }
    hwc->hwcNameScratch= fresh;

    if  ( pip->pipBliptag == 0 )
	{ pip->pipBliptag= appGetTimestamp();	}

    abi= (AppBitmapImage *)io->ioPrivate;
    if  ( ! abi )
	{ return 1;	}

    if  ( io->ioKind == DOCokPICTJPEGBLIP )
	{
	docHtmlSetImageNameAndType( hwc, pip, "jpg", "image/jpeg" );
	hwc->hwcWriteImageFromObjectData= 1;
	hwc->hwcWriteThisBitmap= bmJpegWriteJfif; /* ignored */
	return 0;
	}

    if  ( io->ioKind == DOCokPICTPNGBLIP )
	{
	docHtmlSetImageNameAndType( hwc, pip, "png", "image/png" );
	hwc->hwcWriteImageFromObjectData= 1;
	hwc->hwcWriteThisBitmap= bmPngWritePng; /* ignored */
	return 0;
	}

    if  ( bmCanWriteGifFile( &(abi->abiBitmap), 89 )	&&
	  bmCanWriteJpegFile( &(abi->abiBitmap), 0 )	)
	{ return 1;	}

#   if  USE_PNG
    if  ( abi->abiBitmap.bdColorEncoding == BMcoRGB8PALETTE )
	{
	docHtmlSetImageNameAndType( hwc, pip, "png", "image/png" );
	hwc->hwcWriteImageFromObjectData= 0;
	hwc->hwcWriteThisBitmap= bmPngWritePng;
	return 0;
	}
#   endif

#   if  USE_GIF
    if  ( ! bmCanWriteGifFile( &(abi->abiBitmap), 89 ) )
	{
	docHtmlSetImageNameAndType( hwc, pip, "gif", "image/gif" );
	hwc->hwcWriteImageFromObjectData= 0;
	hwc->hwcWriteThisBitmap= bmGifWriteGif;
	return 0;
	}
#   endif

    if  ( ! bmCanWriteJpegFile( &(abi->abiBitmap), 0 ) )
	{
	docHtmlSetImageNameAndType( hwc, pip, "jpg", "image/jpeg" );
	hwc->hwcWriteImageFromObjectData= 0;
	hwc->hwcWriteThisBitmap= bmJpegWriteJfif;
	return 0;
	}

    return 1;
    }

/************************************************************************/
/*									*/
/*  Save a tag with an argument.					*/
/*									*/
/************************************************************************/

static void docHtmlPutString(		const char *		s,
					HtmlWritingContext *	hwc )
    {
    SimpleOutputStream *	sos= hwc->hwcSos;
    int				col= hwc->hwcColumn;

    /*  mime  */
    if  ( col == 0 && s[0] == '-' )
	{ sioOutPutByte( ' ', sos ); col += 1; }

    sioOutPutString( s, sos ); col += strlen( s );

    hwc->hwcColumn= col; return;
    }

static void docHtmlNewLine(	HtmlWritingContext *	hwc )
    {
    SimpleOutputStream *	sos= hwc->hwcSos;

    if  ( hwc->hwcAsMimeAggregate )
	{ sioOutPutByte( '\r', sos );	}

    sioOutPutByte( '\n', sos );
    hwc->hwcColumn= 0;
    }

static void docHtmlWriteTagStringArg(	const char *		tag,
					const char *		arg,
					HtmlWritingContext *	hwc )
    {
    if  ( hwc->hwcColumn > 1				&&
	  hwc->hwcColumn+ 1+ strlen( tag )+ 1+ 3 > 76	)
	{ docHtmlNewLine( hwc );		}
    else{ docHtmlPutString( " ", hwc );	}

    docHtmlPutString( tag, hwc );
    docHtmlPutString( "=", hwc );
    docHtmlPutString( arg, hwc );

    return;
    }

static void docHtmlWriteTagIntArg(	const char *		tag,
					int			arg,
					HtmlWritingContext *	hwc )
    {
    char	scratch[20];

    sprintf( scratch, "%d", arg );

    docHtmlWriteTagStringArg( tag, scratch, hwc );

    return;
    }

static void docHtmlEscapeCharacters(	const char *		ss,
					int			len,
					HtmlWritingContext *	hwc )
    {
    SimpleOutputStream *	sos= hwc->hwcSos;
    const unsigned char *	us= (const unsigned char *)ss;

    while( len > 0 )
	{
	const char *	escaped= (const char *)0;

	if  ( *us == '-' && hwc->hwcColumn == 0 )
	    { escaped= "&#045;";			}
	else{
	    if  ( *us == '<'	||
		  *us == '>'	||
		  *us == '&'	||
		  *us == '\''	||
		  *us == '"'	)
		{ escaped= utilHtmlCharEntities[*us];	}
	    }

	if  ( escaped )
	    {
	    sioOutPutString( escaped, sos );
	    hwc->hwcColumn += strlen( escaped );
	    }
	else{
	    sioOutPutByte( *us, sos );
	    hwc->hwcColumn++;
	    }

	us++; len--;
	}

    return;
    }

static void docHtmlEscapeString(	const char *		s,
					HtmlWritingContext *	hwc )
    {
    docHtmlEscapeCharacters( s, strlen( s ), hwc );

    return;
    }

# if 0
static int docHtmlFontSize(	int	halfPoints )
    {
    if  ( halfPoints/2 < 6 )	{ return 1;	}
    if  ( halfPoints/2 < 9 )	{ return 2;	}

    if  ( halfPoints/2 > 24 )	{ return 7;	}
    if  ( halfPoints/2 > 19 )	{ return 6;	}
    if  ( halfPoints/2 > 15 )	{ return 5;	}
    if  ( halfPoints/2 > 11 )	{ return 4;	}

    return 3;
    }
# endif

/************************************************************************/
/*									*/
/*  Translate an RTF font designation to an HTML/CSS1 style one.	*/
/*									*/
/************************************************************************/

static int docHtmlFontFamilyIndicator(	char *		target,
					int		maxSize,
					int		styleInt,
					const char *	familyName )
    {
    const char *	genericFamily= (const char *)0;
    const char *	hintFamily= (const char *)0;

    int			genericSize= 0;
    int			hintSize= 0;
    int			nameSize= 0;
    int			size= 0;

    int			commas= 0;

    switch( styleInt )
	{
	case DFstyleFROMAN:
	    genericFamily= "serif";
	    hintFamily= "Times\",\"Times New Roman";
	    break;

	case DFstyleFNIL:
	case DFstyleFSWISS:
	    genericFamily= "sans-serif";
	    hintFamily= "Helvetica\",\"Arial";
	    break;

	case DFstyleFMODERN:
	    genericFamily= "monospace";
	    hintFamily= "Courier";
	    break;

	case DFstyleFSCRIPT:
	case DFstyleFDECOR:
	    genericFamily= "cursive";
	    hintFamily= "Zapf-Chancery";
	    break;

	case DFstyleFTECH:
	    hintFamily= "Symbol";
	    break;
	}

    if  ( genericFamily )
	{
	commas++;
	genericSize= strlen( genericFamily ); /* keyword -> no quotes */
	}
    if  ( hintFamily )
	{
	commas++;
	hintSize= 1+ strlen( hintFamily )+ 1;
	}
    nameSize= 1+ strlen( familyName )+ 1;

    size= nameSize+ genericSize+ hintSize+ commas;
    if  ( size > maxSize )
	{ LLLLDEB(nameSize,genericSize,hintSize,maxSize); return -1; }

    *(target++)= '"';
    strcpy( target, familyName ); target += nameSize- 2;
    *(target++)= '"';

    if  ( hintFamily )
	{
	*(target++)= ',';
	*(target++)= '"';
	strcpy( target, hintFamily ); target += hintSize- 2;
	*(target++)= '"';
	}

    if  ( genericFamily )
	{
	*(target++)= ',';
	strcpy( target, genericFamily ); target += genericSize;
	}

    *(target  )= '\0';
    return size;
    }

/************************************************************************/
/*									*/
/*  Change attributes.							*/
/*									*/
/************************************************************************/


static void docHtmlChangeAttributes(	HtmlWritingContext *		hwc,
					int				taNr )
    {
    if  ( taNr == hwc->hwcCurrentAttributeNumber )
	{ return;	}

    if  ( hwc->hwcCurrentAttributeNumber >= 0 )
	{
	docHtmlPutString( "</span>", hwc );
	}

    if  ( taNr >= 0 )
	{
	char	scratch[20];

	sprintf( scratch, "t%d", taNr );

	docHtmlPutString( "<span", hwc );
	docHtmlWriteTagStringArg( "class", scratch, hwc );
	docHtmlPutString( ">", hwc );
	}

    hwc->hwcCurrentAttributeNumber= taNr;
    }

static int docHtmlSaveImgTag(	int				w,
				int				h,
				HtmlWritingContext *		hwc )
    {
    SimpleOutputStream *	sos= hwc->hwcSos;

    docHtmlPutString( "<IMG", hwc );

    if  ( hwc->hwcColumn > 10					&&
	  hwc->hwcColumn+ 6+ strlen( hwc->hwcNameScratch ) > 76	)
	{ docHtmlNewLine( hwc );		}
    else{ docHtmlPutString( " ", hwc );	}

    docHtmlPutString( "SRC=\"", hwc );

    if  ( hwc->hwcAsMimeAggregate )
	{
	docHtmlPutString( "cid:", hwc );
	sioOutPutString( hwc->hwcNameScratch, sos );
	sioOutPutByte( '.', sos );
	sioOutPutString( hwc->hwcContentIdTail, sos );
	}
    else{
	docHtmlPutString( hwc->hwcNameScratch+ hwc->hwcRelativeOffset,
								hwc );
	}

    docHtmlPutString( "\"", hwc );

    docHtmlWriteTagIntArg( "WIDTH", w, hwc );
    docHtmlWriteTagIntArg( "HEIGHT", h, hwc );

    docHtmlPutString( " ALT=\"&lt;IMG&gt;\">", hwc );
    docHtmlNewLine( hwc );

    return 0;
    }

static int docHtmlSavePicture(	InsertedObject *	io,
				int *			pDone,
				HtmlWritingContext *	hwc )
    {
    int				res;

    const AppBitmapImage *	abi;
    const BitmapDescription *	bd;

    int				w;
    int				h;
    int				d;

    if  ( ! io->ioPrivate )
	{
	if  ( docGetBitmapForObject( io ) )
	    { LDEB(1); return 1;	}
	}

    abi= (AppBitmapImage *)io->ioPrivate;
    if  ( ! abi )
	{ return 0;	}
    bd= &(abi->abiBitmap);

    res= docHtmlMakeNameForImage( hwc, io );
    if  ( res < 0 )
	{ LDEB(res); return -1;	}
    if  ( res > 0 )
	{ return 0;	}

    w= TWIPS_TO_SIZE( ( io->ioScaleXSet* io->ioTwipsWide )/100 );
    h= TWIPS_TO_SIZE( ( io->ioScaleYSet* io->ioTwipsHigh )/100 );

    if  ( w < 1 ) { w= 1;	}
    if  ( h < 1 ) { h= 1;	}

    d= ( 100* bd->bdPixelsWide- 100* w )/ bd->bdPixelsWide;
    if  ( d < 0 )
	{ d= -d;	}
    if  ( d <= 15 )
	{ w= bd->bdPixelsWide;	}

    d= ( 100* bd->bdPixelsHigh- 100* h )/ bd->bdPixelsHigh;
    if  ( d < 0 )
	{ d= -d;	}
    if  ( d <= 15 )
	{ h= bd->bdPixelsHigh;	}

    if  ( docHtmlSaveImgTag( w, h, hwc ) )
	{ SDEB(hwc->hwcNameScratch); return -1;	}

    hwc->hwcImageCount++;

    *pDone= 1; return 0;
    }

static int docHtmlStartAnchor(	HtmlWritingContext *		hwc,
				int				isNote,
				const char *			fileName,
				int				fileSize,
				const char *			markName,
				int				markSize,
				const char *			refName,
				int				refSize,
				const char *			title,
				int				titleSize )
    {
    SimpleOutputStream *	sos= hwc->hwcSos;

    int				afterSpace;
    int				needed= 0;

    docHtmlPutString( "<A", hwc );
    if  ( isNote )
	{ docHtmlPutString( " style=\"text-decoration:none\"", hwc ); }
    afterSpace= 0;

    if  ( fileSize > 0 || markSize > 0 )
	{
	needed += 1+ 6+ fileSize+ markSize+ 1;
	
	if  ( markSize > 0 )
	    { needed += 1;	}
	}

    if  ( refSize > 0 )
	{ needed += 1+ 6+ refSize+ 1;	}

    if  ( hwc->hwcColumn > 5		&&
	  hwc->hwcColumn+ needed > 76	)
	{ docHtmlNewLine( hwc ); afterSpace= 1;		}

    if  ( fileSize > 0 || markSize > 0 )
	{
	if  ( ! afterSpace )
	    { docHtmlPutString( " ", hwc ); }

	docHtmlPutString( "HREF=\"", hwc );

	if  ( fileSize > 0 )
	    {
	    while( fileSize > 0 )
		{
		sioOutPutByte( *fileName, sos );
		fileName++; fileSize--; hwc->hwcColumn++;
		}
	    }

	if  ( markName && markSize > 0 )
	    {
	    sioOutPutByte( '#', sos );

	    while( markSize > 0 )
		{
		sioOutPutByte( *markName, sos );
		markName++; markSize--; hwc->hwcColumn++;
		}
	    }

	docHtmlPutString( "\"", hwc );
	afterSpace= 0;
	}

    if  ( refSize > 0 )
	{
	if  ( ! afterSpace )
	    { docHtmlPutString( " ", hwc ); }

	docHtmlPutString( "NAME=\"", hwc );

	while( refSize > 0 )
	    {
	    sioOutPutByte( *refName, sos );
	    refName++; refSize--; hwc->hwcColumn++;
	    }

	docHtmlPutString( "\"", hwc );
	afterSpace= 0;
	}

    if  ( titleSize > 0 )
	{
	if  ( ! afterSpace )
	    { docHtmlPutString( " ", hwc ); }

	docHtmlPutString( "TITLE=\"", hwc );

	while( titleSize > 0 )
	    {
	    sioOutPutByte( *title, sos );
	    title++; titleSize--; hwc->hwcColumn++;
	    }

	docHtmlPutString( "\"", hwc );
	afterSpace= 0;
	}

    docHtmlPutString( ">", hwc );

    return 0;
    }

static int docHtmlStartNote(	const DocumentField *		df,
				HtmlWritingContext *		hwc,
				const BufferItem *		bi,
				int				attNr )
    {
    const char *	fileName= (const char *)0;
    int			fileSize= 0;
    const char *	markName= (const char *)0;
    int			markSize= 0;
    const char *	refName= (const char *)0;
    int			refSize= 0;
    const char *	title= (const char *)0;
    int			titleSize= 0;

    char		ref[25+1];
    char		def[25+1];

    const int		isNote= 1;

    if  ( bi->biInExternalItem == DOCinBODY )
	{
	DocumentNote *	dn;

	sprintf( ref, "_NREF_%d", hwc->hwcNoteRefCount+ 1 );
	sprintf( def, "_NDEF_%d", hwc->hwcNoteRefCount+ 1 );

	markName= def;
	refName=  ref;

	dn= docGetNoteOfField( df, hwc->hwcBd );
	if  ( dn )
	    {
	    DocumentPosition	dpLast;

	    if  ( ! docLastPosition( &dpLast, dn->dnDocumentTree.eiRoot ) )
		{
		int	paraNr= docNumberOfParagraph( dpLast.dpBi );

		if  ( paraNr == 1 )
		    {
		    title= (char *)docParaString( dpLast.dpBi, 0 );
		    titleSize= docParaStrlen( dpLast.dpBi );
		    }
		}
	    }
	}
    else{
	sprintf( ref, "_NREF_%d", hwc->hwcNoteDefCount+ 1 );
	sprintf( def, "_NDEF_%d", hwc->hwcNoteDefCount+ 1 );

	markName= ref;
	refName=  def;
	}

    markSize= strlen( markName );
    refSize= strlen( refName );

    docHtmlChangeAttributes( hwc, -1 );

    docHtmlStartAnchor( hwc, isNote, fileName, fileSize,
			markName, markSize,
			refName, refSize,
			title, titleSize );

    docHtmlChangeAttributes( hwc, attNr );

    hwc->hwcBytesInLink= 0;

    return 0;
    }

static int docHtmlStartField(	const DocumentField *		df,
				HtmlWritingContext *		hwc,
				const BufferItem *		bi,
				int				attNr )
    {
    const char *	fileName= (const char *)0;
    int			fileSize= 0;
    const char *	markName= (const char *)0;
    int			markSize= 0;
    const char *	refName= (const char *)0;
    int			refSize= 0;
    const char *	title= (const char *)0;
    int			titleSize= 0;

    switch( df->dfKind )
	{
	case DOCfkCHFTN:
	    hwc->hwcInHyperlink++;
	    if  ( ! hwc->hwcInBookmark && hwc->hwcInHyperlink == 1 )
		{ docHtmlStartNote( df, hwc, bi, attNr );	}
	    break;

	case DOCfkHYPERLINK:
	    hwc->hwcInHyperlink++;
	    if  ( ! hwc->hwcInBookmark && hwc->hwcInHyperlink == 1 )
		{
		if  ( ! docFieldGetHyperlink( df,
				&fileName, &fileSize, &markName, &markSize ) )
		    {
		    const int	isNote= 0;

		    docHtmlChangeAttributes( hwc, -1 );

		    docHtmlStartAnchor( hwc, isNote,
					    fileName, fileSize,
					    markName, markSize,
					    refName, refSize,
					    title, titleSize );

		    docHtmlChangeAttributes( hwc, attNr );

		    hwc->hwcBytesInLink= 0;
		    }
		}
	    break;

	case DOCfkBOOKMARK:
	    hwc->hwcInBookmark++;
	    if  ( ! hwc->hwcInHyperlink && hwc->hwcInBookmark == 1 )
		{
		if  ( ! docFieldGetBookmark( df, &refName, &refSize ) )
		    {
		    const int	isNote= 0;

		    docHtmlChangeAttributes( hwc, -1 );

		    docHtmlStartAnchor( hwc, isNote,
					    fileName, fileSize,
					    markName, markSize,
					    refName, refSize,
					    title, titleSize );

		    docHtmlChangeAttributes( hwc, attNr );
		    }
		}
	    break;

	case DOCfkPAGEREF:
	    hwc->hwcInPageref++;
	    break;

	default:
	    break;
	}

    return 0;
    }

static int docHtmlFinishField(	const DocumentField *		df,
				HtmlWritingContext *		hwc )
    {
    switch( df->dfKind )
	{
	case DOCfkCHFTN:
	case DOCfkHYPERLINK:
	    if  ( ! hwc->hwcInBookmark && hwc->hwcInHyperlink == 1 )
		{
		docHtmlChangeAttributes( hwc, -1 );

		docHtmlPutString( "</A>", hwc );
		}
	    hwc->hwcInHyperlink--;
	    if  ( hwc->hwcInHyperlink < 0 )
		{ hwc->hwcInHyperlink= 0;	}
	    break;

	case DOCfkBOOKMARK:
	    if  ( ! hwc->hwcInHyperlink && hwc->hwcInBookmark == 1 )
		{
		docHtmlChangeAttributes( hwc, -1 );

		docHtmlPutString( "</A>", hwc );
		}
	    hwc->hwcInBookmark--;
	    if  ( hwc->hwcInBookmark < 0 )
		{ hwc->hwcInBookmark= 0;	}
	    break;

	case DOCfkPAGEREF:
	    hwc->hwcInPageref--;
	    break;

	default:
	    break;
	}
    
    return 0;
    }

static int docHtmlSaveParticules( const BufferItem *		bi,
				int				part,
				int				partUpto,
				HtmlWritingContext *		hwc )
    {
    int					done= 0;

    int					afterSpace= 0;

    TextParticule *			tp= bi->biParaParticules+ part;
    int					stroff= tp->tpStroff;
    unsigned char *			s= docParaString( bi, stroff );

    int					pictureDone;
    InsertedObject *			io;

    TextAttribute			ta;

    const DocumentField *		df;
    const FieldKindInformation *	fki;

    int					len;

    const BufferDocument *		bd= hwc->hwcBd;

    while( part < partUpto )
	{
	utilGetTextAttributeByNumber( &ta, &(bd->bdTextAttributeList),
						tp->tpTextAttrNr );

	switch( tp->tpKind )
	    {
	    case DOCkindTAB:
		docHtmlPutString( " ", hwc );
		afterSpace= 1;
		s++; stroff++;
		break;

	    case DOCkindSPAN:
		docHtmlChangeAttributes( hwc, tp->tpTextAttrNr );

		if  ( afterSpace && hwc->hwcColumn+ tp->tpStrlen > 76 )
		    { docHtmlNewLine( hwc );	}

		len= tp->tpStroff+ tp->tpStrlen- stroff;

		if  ( ! hwc->hwcInHyperlink	||
		      ! hwc->hwcInPageref	||
		      hwc->hwcBytesInLink == 0	)
		    {
		    docHtmlEscapeCharacters( (char *)s, len, hwc );

		    if  ( hwc->hwcInHyperlink )
			{ hwc->hwcBytesInLink += len;	}
		    }

		s += len;
		stroff += len;

		afterSpace= 0;
		if  ( tp->tpStrlen > 0 && s[-1] == ' ' )
		    { afterSpace= 1;	}
		break;

	    case DOCkindOBJECT:
		pictureDone= 0;
		io= docGetObject( bd, tp->tpObjectNumber );
		if  ( ! io )
		    { LXDEB(tp->tpObjectNumber,io); return -1;	}

		if  (   io->ioKind == DOCokPICTWMETAFILE		||
			io->ioKind == DOCokPICTPNGBLIP			||
			io->ioKind == DOCokPICTJPEGBLIP			||
			io->ioKind == DOCokMACPICT			||
		      ( io->ioKind == DOCokOLEOBJECT 		&&
		        io->ioResultKind == DOCokPICTWMETAFILE	)	)
		    {
		    if  ( docHtmlSavePicture( io, &pictureDone, hwc ) )
			{ XDEB(io);	}
		    }

		if  ( ! pictureDone )
		    { docHtmlPutString( " ", hwc );	}

		afterSpace= 0; s++; stroff++;
		break;

	    case DOCkindFIELDSTART:
		df= docGetFieldByNumber( &(bd->bdFieldList),
						    tp->tpObjectNumber );
		fki= DOC_FieldKinds+ df->dfKind;

		if  ( fki->fkiIsFieldInRtf		&&
		      fki->fkiLevel == DOClevSPAN	)
		    {
		    docHtmlStartField( df, hwc, bi, tp->tpTextAttrNr );
		    }

		if  ( df->dfKind == DOCfkBOOKMARK )
		    {
		    docHtmlStartField( df, hwc, bi, tp->tpTextAttrNr );
		    }

		if  ( df->dfKind == DOCfkCHFTN )
		    {
		    int		count;
		    int		closed;
		    char	scratch[20+1];

		    count= docCountParticulesInField( bi, &closed,
							    part, partUpto );

		    docHtmlStartField( df, hwc, bi, tp->tpTextAttrNr );

		    if  ( bi->biInExternalItem == DOCinBODY )
			{
			sprintf( scratch, "%d", hwc->hwcNoteRefCount+ 1 );
			hwc->hwcNoteRefCount++;
			}
		    else{
			sprintf( scratch, "%d", hwc->hwcNoteDefCount+ 1 );
			hwc->hwcNoteDefCount++;
			}

		    docHtmlPutString( scratch, hwc );

		    if  ( hwc->hwcInHyperlink )
			{ hwc->hwcBytesInLink += strlen( scratch );	}

		    done= count+ 1;
		    stroff= tp[done].tpStroff; s= docParaString( bi, stroff );
		    break;
		    }

		s += tp->tpStrlen; stroff += tp->tpStrlen; /* += 0 */
		done= 1;
		break;

	    case DOCkindFIELDEND:
		df= docGetFieldByNumber( &(bd->bdFieldList),
						    tp->tpObjectNumber );
		fki= DOC_FieldKinds+ df->dfKind;

		if  ( df->dfKind == DOCfkCHFTN )
		    {
		    docHtmlFinishField( df, hwc );
		    }

		if  ( df->dfKind == DOCfkBOOKMARK )
		    { docHtmlFinishField( df, hwc ); }

		if  ( fki->fkiIsFieldInRtf		&&
		      fki->fkiLevel == DOClevSPAN	)
		    { docHtmlFinishField( df, hwc ); }

		s += tp->tpStrlen; stroff += tp->tpStrlen; /* += 0 */
		done= 1;
		break;

	    case DOCkindLINEBREAK:
	    case DOCkindPAGEBREAK:
	    case DOCkindCOLUMNBREAK:
		docHtmlPutString( "<BR>", hwc );
		docHtmlNewLine( hwc );
		afterSpace= 0;
		s += tp->tpStrlen; stroff += tp->tpStrlen; /* += 1 */
		break;

	    default:
		LDEB(tp->tpKind);
		s += tp->tpStrlen; stroff += tp->tpStrlen;
		break;
	    }

	done++; part++; tp++;
	}

    docHtmlChangeAttributes( hwc, -1 );

    return done;
    }

/************************************************************************/
/*									*/
/*  The body of a paragraph is written as a <div> or <td> division.	*/
/*									*/
/************************************************************************/

static void docHtmlEmitBorderStyle(
			    const char *		whatBorder,
			    const BorderProperties *	bp,
			    HtmlWritingContext *	hwc )
    {
    SimpleOutputStream *	sos= hwc->hwcSos;

    docHtmlPutString( whatBorder, hwc );
    sioOutPutByte( ':', sos ); hwc->hwcColumn++;

    /* width */
    sioOutPrintf( sos, " %dpt", ( bp->bpPenWideTwips+ 10 )/ 20 );
    hwc->hwcColumn += 5;

    /* style */
    docHtmlPutString( " solid", hwc );

    /* color */
    if  ( bp->bpColor == 0 )
	{
	docHtmlPutString( " black", hwc );
	}
    else{
	const BufferDocument *		bd= hwc->hwcBd;
	const DocumentProperties *	dp= &(bd->bdProperties);

	const RGB8Color *		rgb8= dp->dpColors+ bp->bpColor;

	char				scratch[30];

	sprintf( scratch, " #%02x%02x%02x",
					rgb8->rgb8Red,
					rgb8->rgb8Green,
					rgb8->rgb8Blue );

	docHtmlPutString( scratch, hwc );
	}

    sioOutPutByte( ';', sos ); hwc->hwcColumn++;

    return;
    }

static void docHtmlEmitBorderStyleByNumber(
			    const char *		whatBorder,
			    int				num,
			    HtmlWritingContext *	hwc )
    {
    const BufferDocument *	bd= hwc->hwcBd;
    const NumberedPropertiesList *	bpl= &(bd->bdBorderPropertyList);

    BorderProperties		bp;

    docGetBorderPropertiesByNumber( &bp, bpl, num );

    if  ( ! DOCisBORDER( &bp ) )
	{ return;	}

    docHtmlEmitBorderStyle( whatBorder, &bp, hwc );
    }

static void docHtmlEmitBackgroundProperty(
				const ItemShading *		is,
				HtmlWritingContext *		hwc )
    {
    const BufferDocument *	bd= hwc->hwcBd;

    int				isFilled= 0;
    RGB8Color			rgb8;

    char			scratch[39];

    if  ( is->isPattern == DOCspSOLID )
	{
	if  ( docGetSolidRgbShadeOfItem( &isFilled, &rgb8, bd, is ) )
	    { LDEB(1);	}

	if  ( isFilled )
	    {
	    sprintf( scratch, "\"#%02x%02x%02x\"",
			    rgb8.rgb8Red, rgb8.rgb8Green, rgb8.rgb8Blue );

	    docHtmlWriteTagStringArg( "BGCOLOR", scratch, hwc );
	    }
	}

    return;
    }

static int docHtmlUseBackgroundStyle( 
				const ItemShading *		is,
				HtmlWritingContext *		hwc )
    {
    const BufferDocument *	bd= hwc->hwcBd;

    int				isFilled= 0;
    RGB8Color			rgb8;

    if  ( is->isPattern == DOCspSOLID )
	{
	if  ( docGetSolidRgbShadeOfItem( &isFilled, &rgb8, bd, is ) )
	    { LDEB(1);	}
	if  ( isFilled )
	    { return 1;	}
	}

    return 0;
    }

static void docHtmlEmitBackgroundStyle(
				const ItemShading *		is,
				HtmlWritingContext *		hwc )
    {
    const BufferDocument *	bd= hwc->hwcBd;

    int				isFilled= 0;
    RGB8Color			rgb8;

    char			scratch[50];

    if  ( is->isPattern == DOCspSOLID )
	{
	if  ( docGetSolidRgbShadeOfItem( &isFilled, &rgb8, bd, is ) )
	    { LDEB(1);	}

	if  ( isFilled )
	    {
	    sprintf( scratch, "background-color: #%02x%02x%02x;",
			    rgb8.rgb8Red, rgb8.rgb8Green, rgb8.rgb8Blue );

	    docHtmlPutString( scratch, hwc );
	    }
	}

    return;
    }


static void docHtmlStartParagraphBody(	const BufferItem *	paraBi,
					const char *		tag,
					int			fontHeight,
					HtmlWritingContext *	hwc )
    {
    const BufferDocument *		bd= hwc->hwcBd;
    const NumberedPropertiesList *	bpl= &(bd->bdBorderPropertyList);
    const NumberedPropertiesList *	isl= &(bd->bdItemShadingList);

    SimpleOutputStream *	sos= hwc->hwcSos;
    int				useStyle= 0;

    ItemShading			is;

    if  ( paraBi->biParaLeftIndentTwips <= -100	||
	  paraBi->biParaLeftIndentTwips >=  100	)
	{ useStyle++;	}

    if  ( ( paraBi->biParaFirstIndentTwips <= -100	||
	    paraBi->biParaFirstIndentTwips >=  100	)	)
	{ useStyle++;	}

    if  ( docBorderNumberIsBorder( bpl, paraBi->biParaTopBorderNumber ) )
	{ useStyle++;	}
    if  ( docBorderNumberIsBorder( bpl, paraBi->biParaLeftBorderNumber ) )
	{ useStyle++;	}
    if  ( docBorderNumberIsBorder( bpl, paraBi->biParaRightBorderNumber ) )
	{ useStyle++;	}
    if  ( docBorderNumberIsBorder( bpl, paraBi->biParaBottomBorderNumber ) )
	{ useStyle++;	}

    docGetItemShadingByNumber( &is, isl, paraBi->biParaShadingNumber );

    if  ( docHtmlUseBackgroundStyle( &is, hwc ) )
	{ useStyle++;	}

    sioOutPutByte( '<', sos ); hwc->hwcColumn++;
    docHtmlPutString( tag, hwc );

    if  ( ! useStyle )
	{
	switch( paraBi->biParaAlignment )
	    {
	    case DOCiaLEFT:
		break;
	    case DOCiaRIGHT:
		docHtmlPutString( " ALIGN=\"RIGHT\"", hwc );
		break;

	    case DOCiaCENTERED:
		docHtmlPutString( " ALIGN=\"CENTER\"", hwc );
		break;

	    case DOCiaJUSTIFIED:
		docHtmlPutString( " ALIGN=\"JUSTIFY\"", hwc );
		break;

	    default:
		LDEB(paraBi->biParaAlignment);
		break;
	    }
	}

    /* No!
    if  ( ! useStyle && paraBi->biParaShading.isPattern == DOCspSOLID )
	{ docHtmlEmitBackgroundProperty( &(paraBi->biParaShading), hwc ); }
    */

    if  ( useStyle > 0 )
	{
	docHtmlPutString( " style=\"", hwc );

	if  ( paraBi->biParaLeftIndentTwips <= -100	||
	      paraBi->biParaLeftIndentTwips >=  100	)
	    {
	    docHtmlPutString( "padding-left:", hwc );
	    sioOutPrintf( sos, "%dpt;", paraBi->biParaLeftIndentTwips/ 20 );
	    hwc->hwcColumn += 6;
	    }

	if  ( ( paraBi->biParaFirstIndentTwips <= -100	||
		paraBi->biParaFirstIndentTwips >=  100	)	)
	    {
	    docHtmlPutString( "text-indent:", hwc );
	    sioOutPrintf( sos, "%dpt;", paraBi->biParaFirstIndentTwips/ 20 );
	    hwc->hwcColumn += 6;
	    }

	if  ( is.isPattern == DOCspSOLID )
	    { docHtmlEmitBackgroundStyle( &is, hwc ); }

	docHtmlEmitBorderStyleByNumber( "border-top",
					paraBi->biParaTopBorderNumber, hwc );
	docHtmlEmitBorderStyleByNumber( "border-left",
					paraBi->biParaLeftBorderNumber, hwc );
	docHtmlEmitBorderStyleByNumber( "border-right",
					paraBi->biParaRightBorderNumber, hwc );
	docHtmlEmitBorderStyleByNumber( "border-bottom",
					paraBi->biParaBottomBorderNumber, hwc );

	switch( paraBi->biParaAlignment )
	    {
	    case DOCiaLEFT:
		break;
	    case DOCiaRIGHT:
		docHtmlPutString( " text-align:right;", hwc );
		break;

	    case DOCiaCENTERED:
		docHtmlPutString( " text-align:center;", hwc );
		break;

	    case DOCiaJUSTIFIED:
		docHtmlPutString( " text-align:justify;", hwc );
		break;

	    default:
		LDEB(paraBi->biParaAlignment);
		break;
	    }

	docHtmlPutString( "\"", hwc );
	}

    docHtmlPutString( ">", hwc );

    if  ( paraBi->biParaSpaceBeforeTwips > fontHeight/ 2 )
	{
	docHtmlPutString( "&nbsp;<BR>", hwc );
	docHtmlNewLine( hwc );
	}

    return;
    }

/************************************************************************/
/*									*/
/*  Save a 'Paragraph'							*/
/*									*/
/*  But not as a <P>...</P>.						*/
/*									*/
/************************************************************************/

static int docHtmlSaveParaItem(	const BufferItem *		paraBi,
				HtmlWritingContext *		hwc )
    {
    TextParticule *		tp;
    unsigned char *		s;

    int				part= 0;
    int				stroff= 0;

    TextAttribute		ta;
    int				fontHeight;

    PropertyMask		ppChgMask;
    PropertyMask		ppUpdMask;

    const BufferDocument *	bd= hwc->hwcBd;

    if  ( paraBi->biParaParticuleCount == 0		||
	  docParaStrlen( paraBi ) == 0			)
	{
	docHtmlPutString( "<div>&nbsp;</div>", hwc );
	docHtmlNewLine( hwc );
	return 0;
	}

    part= 0;
    stroff= 0;
    tp= paraBi->biParaParticules+ part;
    s= docParaString( paraBi, stroff );

    utilGetTextAttributeByNumber( &ta, &(bd->bdTextAttributeList),
							    tp->tpTextAttrNr );

    fontHeight= 10* ta.taFontSizeHalfPoints;

    docHtmlStartParagraphBody( paraBi, "div", fontHeight, hwc );

    if  ( docHtmlSaveParticules( paraBi, part,
				paraBi->biParaParticuleCount, hwc ) < 0 )
	{ LDEB(part); return -1;	}

    if  ( paraBi->biParaSpaceAfterTwips > fontHeight/ 2 )
	{ docHtmlPutString( "<BR>&nbsp;</div>", hwc );	}
    else{
	docHtmlPutString( "</div>", hwc );		
	}
    docHtmlNewLine( hwc );

    utilPropMaskClear( &ppChgMask );

    utilPropMaskClear( &ppUpdMask );
    utilPropMaskFill( &ppUpdMask, PPprop_FULL_COUNT );

    if  ( docUpdParaProperties( &ppChgMask, &(hwc->hwcParagraphProperties),
		    &ppUpdMask, &(paraBi->biParaProperties),
		    (const DocumentAttributeMap *)0 ) )
	{ LDEB(1);	}

    return 0;
    }

static int docHtmlStartTable(	const BufferItem *		rowBi,
				const BufferItem *		bodySectBi,
				HtmlWritingContext *		hwc )
    {
    int				cellPadding;
    int				width;

    BlockFrame			bf;
    ParagraphFrame		pf;

    int				tableX0;
    int				tableX1;
    int				parentX0;
    int				parentX1;

    const int			page= 0;
    const int			column= 0;

    char			scratch[100];

    docLayoutInitBlockFrame( &bf );
    docBlockFrameTwips( &bf, (BufferItem *)rowBi, bodySectBi,
						    hwc->hwcBd, page, column );

    docCellFrameTwips( &pf, &bf, rowBi->biChildren[0] );
    tableX0= pf.pfCellRect.drX0;
    if  ( rowBi->biChildCount > 1 )
	{
	docCellFrameTwips( &pf, &bf,
			rowBi->biChildren[rowBi->biChildCount- 1] );
	}
    tableX1= pf.pfCellRect.drX1;

    if  ( rowBi->biParent->biLevel == DOClevCELL )
	{
	docCellFrameTwips( &pf, &bf, rowBi->biParent );
	parentX0= pf.pfCellRect.drX0;
	parentX1= pf.pfCellRect.drX1;
	}
    else{
	parentX0= bf.bfContentRect.drX0;
	parentX1= bf.bfContentRect.drX1;
	}

    width= ( 100* tableX1- 100* tableX0+ 50 )/ ( parentX1- parentX0 );
    sprintf( scratch, "<table style=\"width:%d%%\" cellspacing=0", width );
    docHtmlPutString( scratch, hwc );

    cellPadding= TWIPS_TO_SIZE( rowBi->biRowHalfGapWidthTwips )- 4;
    if  ( cellPadding < 1 )
	{ docHtmlWriteTagIntArg( "cellpadding", 0, hwc ); }
    if  ( cellPadding > 1 )
	{ docHtmlWriteTagIntArg( "cellpadding", cellPadding, hwc ); }

    docHtmlPutString( ">", hwc );

    return 0;
    }

static int docHtmlFinishTable(	HtmlWritingContext *		hwc )
    {
    docHtmlPutString( "</table>", hwc );
    docHtmlNewLine( hwc );
    return 0;
    }

/************************************************************************/
/*									*/
/*  Save the children of an item.					*/
/*									*/
/************************************************************************/

static int docHtmlSaveItemChildren(
				const BufferItem *		bi,
				const BufferItem *		bodySectBi,
				HtmlWritingContext *		hwc )
    {
    int		intbl= 0;
    int		child;

    for ( child= 0; child < bi->biChildCount; child++ )
	{
	const BufferItem *	childBi= bi->biChildren[child];

	if  ( docIsRowItem( childBi ) )
	    {
	    if  ( intbl && child > 0 )
		{
		if  ( childBi->biRowTableFirst == child )
		    { docHtmlFinishTable( hwc ); intbl= 0; }
		}

	    if  ( ! intbl )
		{ docHtmlStartTable( childBi, bodySectBi, hwc ); intbl= 1; }
	    }
	else{
	    if  ( intbl )
		{ docHtmlFinishTable( hwc ); intbl= 0; }
	    }

	if  ( docHtmlSaveItem( childBi, bodySectBi, hwc ) )
	    { LDEB(child); return -1;	}
	}

    if  ( intbl )
	{ docHtmlFinishTable( hwc ); intbl= 0; }

    return 0;
    }

static int docHtmlSaveRowItem(	const BufferItem *		rowBi,
				const BufferItem *		bodySectBi,
				HtmlWritingContext *		hwc )
    {
    int					i;
    const BufferDocument *		bd= hwc->hwcBd;
    const NumberedPropertiesList *	bpl= &(bd->bdBorderPropertyList);
    const NumberedPropertiesList *	isl= &(bd->bdItemShadingList);
    const CellProperties *		cp;
    int					high;

    docHtmlPutString( "<tr VALIGN=\"TOP\"", hwc );

    high= rowBi->biRowHeightTwips;
    if  ( high < 0 )
	{ high= -high;	}
    high= TWIPS_TO_SIZE( high );
    if  ( high > 0 )
	{ docHtmlWriteTagIntArg( "height", high, hwc );	}

    docHtmlPutString( ">", hwc );

    cp= rowBi->biRowCells;
    for ( i= 0; i < rowBi->biChildCount; cp++, i++ )
	{
	BufferItem *			cellBi= rowBi->biChildren[i];
	int				wide;

	const int			page= 0;
	const int			column= 0;

	BlockFrame			bf;
	ParagraphFrame			pf;

	int				useStyle= 0;

	int				rowspan= 1;
	int				colspan= 1;

	ItemShading			is;

	if  ( CELL_MERGED( cp ) )
	    { continue;	}

	docTableDetermineCellspans( &rowspan, &colspan, rowBi, i );

	docBlockFrameTwips( &bf, cellBi->biChildren[0],
						bodySectBi, bd, page, column );

	docCellFrameTwips( &pf, &bf, cellBi );
	wide= TWIPS_TO_SIZE( pf.pfCellContentRect.drX1- pf.pfCellContentRect.drX0 );

	if  ( docBorderNumberIsBorder( bpl, cp->cpTopBorderNumber ) )
	    { useStyle++;	}
	if  ( docBorderNumberIsBorder( bpl, cp->cpLeftBorderNumber ) )
	    { useStyle++;	}
	if  ( docBorderNumberIsBorder( bpl, cp->cpRightBorderNumber ) )
	    { useStyle++;	}
	if  ( docBorderNumberIsBorder( bpl, cp->cpBottomBorderNumber ) )
	    { useStyle++;	}

	docHtmlPutString( "<td", hwc );
	docHtmlWriteTagIntArg( "WIDTH", wide, hwc );

	if  ( colspan > 1 )
	    { docHtmlWriteTagIntArg( "COLSPAN", colspan, hwc );	}
	if  ( rowspan > 1 )
	    { docHtmlWriteTagIntArg( "ROWSPAN", rowspan, hwc );	}

	switch( cp->cpValign )
	    {
	    case DOCtvaTOP:
		/* docHtmlWriteTagStringArg( "valign, "top", hwc ); */
		break;
	    case DOCtvaCENTERED:
		docHtmlWriteTagStringArg( "valign", "middle", hwc );
		break;
	    case DOCtvaBOTTOM:
		docHtmlWriteTagStringArg( "valign", "bottom", hwc );
		break;

	    default:
		LDEB(cp->cpValign);
		break;
	    }

	docGetItemShadingByNumber( &is, isl, cp->cpShadingNumber );

	if  ( ! useStyle && is.isPattern == DOCspSOLID )
	    { docHtmlEmitBackgroundProperty( &is, hwc );	}

	if  ( useStyle )
	    {
	    docHtmlPutString( " style=\"", hwc );

	    if  ( is.isPattern == DOCspSOLID )
		{ docHtmlEmitBackgroundStyle( &is, hwc );	}

	    docHtmlEmitBorderStyleByNumber( "border-top",
					    cp->cpTopBorderNumber, hwc );
	    docHtmlEmitBorderStyleByNumber( "border-left",
					    cp->cpLeftBorderNumber, hwc );
	    docHtmlEmitBorderStyleByNumber( "border-right",
					    cp->cpRightBorderNumber, hwc );
	    docHtmlEmitBorderStyleByNumber( "border-bottom",
					    cp->cpBottomBorderNumber, hwc );

	    docHtmlPutString( "\"", hwc );
	    }

	docHtmlPutString( ">", hwc );
	docHtmlNewLine( hwc );

	if  ( docHtmlSaveItemChildren( cellBi, bodySectBi, hwc ) )
	    { LDEB(1);	}

	docHtmlPutString( "</td>", hwc );
	if  ( i < rowBi->biChildCount- 1 )
	    { docHtmlNewLine( hwc );	}
	}

    docHtmlPutString( "</tr>", hwc );
    docHtmlNewLine( hwc );

    return 0;
    }

static int docHtmlSaveItem(	const BufferItem *		bi,
				const BufferItem *		bodySectBi,
				HtmlWritingContext *		hwc )
    {
    switch( bi->biLevel )
	{
	case DOClevSECT:
	case DOClevBODY:
	case DOClevCELL:
	rowAsGroup:
	    if  ( docHtmlSaveItemChildren( bi, bodySectBi, hwc ) )
		{ LDEB(1);	}
	    break;

	case DOClevROW:
	    if  ( ! docIsRowItem( bi ) )
		{ goto rowAsGroup;	}

	    if  ( docHtmlSaveRowItem( bi, bodySectBi, hwc ) )
		{ LDEB(1); return -1;	}
	    break;

	case DOClevPARA:
	    if  ( docHtmlSaveParaItem( bi, hwc ) )
		{ LDEB(1); return -1;	}
	    break;

	default:
	    LDEB(bi->biLevel); return -1;
	}

    return 0;
    }

/************************************************************************/
/*									*/
/*  Start to output an image						*/
/*									*/
/************************************************************************/

static SimpleOutputStream * docHtmlStartImage(	HtmlWritingContext *	hwc )
    {
    SimpleOutputStream *	sos= hwc->hwcSos;
    SimpleOutputStream *	sosImage;

    if  ( hwc->hwcAsMimeAggregate )
	{
	sioOutPutByte( '-', sos ); sioOutPutByte( '-', sos );
	sioOutPutString( hwc->hwcMimeBoundary, sos );
	sioOutPutByte( '\r', sos ); sioOutPutByte( '\n', sos );

	sioOutPutString( "Content-Type: ", sos );
	sioOutPutString( hwc->hwcImageMimeType, sos );
	sioOutPutByte( '\r', sos ); sioOutPutByte( '\n', sos );

	sioOutPutString( "Content-Id: <", sos );
	sioOutPutString( hwc->hwcNameScratch, sos );
	sioOutPutByte( '.', sos );
	sioOutPutString( hwc->hwcContentIdTail, sos );
	sioOutPutByte( '>', sos );
	sioOutPutByte( '\r', sos ); sioOutPutByte( '\n', sos );

	sioOutPutString( "Content-Transfer-Encoding: ", sos );
	sioOutPutString( "base64", sos );
	sioOutPutByte( '\r', sos ); sioOutPutByte( '\n', sos );

	sioOutPutString( "Content-Disposition: inline;", sos );
	sioOutPutByte( '\r', sos ); sioOutPutByte( '\n', sos );
	sioOutPutString( " filename=\"", sos );
	sioOutPutString( hwc->hwcNameScratch, sos );
	sioOutPutByte( '"', sos );
	sioOutPutByte( '\r', sos ); sioOutPutByte( '\n', sos );

	sioOutPutByte( '\r', sos ); sioOutPutByte( '\n', sos );

	sosImage= sioOutBase64Open( sos );
	if  ( ! sosImage )
	    { SXDEB(hwc->hwcNameScratch,sosImage); return sosImage; }
	}
    else{
	sosImage= sioOutStdioOpen( hwc->hwcNameScratch );
	if  ( ! sosImage )
	    { SXDEB(hwc->hwcNameScratch,sosImage); return sosImage; }
	}

    return sosImage;
    }

/************************************************************************/
/*									*/
/*  Save the images in the document.					*/
/*									*/
/************************************************************************/

static int docHtmlSaveImageBytes(	HtmlWritingContext *	hwc,
					InsertedObject *	io )
    {
    int				rval= 0;

    int				res;
    SimpleOutputStream *	sosImage= (SimpleOutputStream *)0;
    SimpleInputStream *		sisMem= (SimpleInputStream *)0;
    SimpleInputStream *		sisHex= (SimpleInputStream *)0;
    AppBitmapImage *		abi;

    res= docHtmlMakeNameForImage( hwc, io );
    if  ( res < 0 )
	{ LDEB(res); rval= -1; goto ready;	}
    if  ( res > 0 )
	{ rval= 0; goto ready;	}

    abi= (AppBitmapImage *)io->ioPrivate;

    sosImage= docHtmlStartImage( hwc );
    if  ( ! sosImage )
	{ XDEB(sosImage); rval= -1; goto ready; }

    if  ( hwc->hwcWriteImageFromObjectData )
	{
	int		done;
	unsigned char	buf[1024];

	sisMem= sioInMemoryOpen( &(io->ioObjectData) );
	if  ( ! sisMem )
	    { XDEB(sisMem); rval= -1; goto ready;	}

	sisMem= sioInMemoryOpen( &(io->ioObjectData) );
	if  ( ! sisMem )
	    { XDEB(sisMem); rval= -1; goto ready;	}
	sisHex= sioInHexOpen( sisMem );
	if  ( ! sisHex )
	    { XDEB(sisHex); rval= -1; goto ready;	}

	while( ( done= sioInReadBytes( sisHex, buf, sizeof(buf) ) ) > 0 )
	    {
	    if  ( sioOutWriteBytes( sosImage, buf, done ) != done )
		{ LDEB(done);	}
	    }
	}
    else{
	if  ( (*hwc->hwcWriteThisBitmap)( &(abi->abiBitmap), abi->abiBuffer,
								sosImage ) )
	    { LDEB(1); rval= -1; goto ready;	}
	}

  ready:

    if  ( sisHex )
	{ sioInClose( sisHex );	}
    if  ( sisMem )
	{ sioInClose( sisMem );	}

    if  ( sosImage )
	{ sioOutClose( sosImage );	}

    return rval;
    }

static int docHtmlSaveImages(	const BufferItem *		bi,
				HtmlWritingContext *		hwc )
    {
    int			i;
    TextParticule *	tp;

    switch( bi->biLevel )
	{
	case DOClevSECT:
	case DOClevBODY:
	case DOClevCELL:
	case DOClevROW:
	    for ( i= 0; i < bi->biChildCount; i++ )
		{
		if  ( docHtmlSaveImages( bi->biChildren[i], hwc ) )
		    { LDEB(i); return -1;	}
		}

	    return 0;

	case DOClevPARA:
	    break;

	default:
	    LDEB(bi->biLevel); return -1;
	}

    tp= bi->biParaParticules;
    for ( i= 0; i < bi->biParaParticuleCount; tp++, i++ )
	{
	InsertedObject *	io;

	if  ( tp->tpKind != DOCkindOBJECT )
	    { continue;	}

	io= docGetObject( hwc->hwcBd, tp->tpObjectNumber );
	if  ( ! io )
	    { LXDEB(tp->tpObjectNumber,io); return -1;	}

	if  (   io->ioKind == DOCokPICTWMETAFILE		||
		io->ioKind == DOCokPICTPNGBLIP			||
		io->ioKind == DOCokPICTJPEGBLIP			||
		io->ioKind == DOCokMACPICT			||
	      ( io->ioKind == DOCokOLEOBJECT 		&&
		io->ioResultKind == DOCokPICTWMETAFILE	)	)
	    {
	    if  ( docHtmlSaveImageBytes( hwc, io ) )
		{ LDEB(1); return -1;	}
	    }
	}

    return 0;
    }

/************************************************************************/
/*									*/
/*  Save styles corresponding to the different text attributes in the	*/
/*  document.								*/
/*									*/
/*  1)  Silly netscape really makes fonts too small. Add a little.	*/
/*									*/
/************************************************************************/

static void docHtmlSaveTextAttributeStyle(
				    const TextAttribute *	ta,
				    int				n,
				    void *			through )
    {
    HtmlWritingContext *		hwc= (HtmlWritingContext *)through;
    char				scratch[150+ 1];
    int					size;
    int					fontSize;

    const BufferDocument *		bd= hwc->hwcBd;
    const DocumentProperties *		dp= &(bd->bdProperties);
    const DocumentFontList *		dfl= &(dp->dpFontList);
    const DocumentFont *		df;

    const NumberedPropertiesList *		bpl= &(bd->bdBorderPropertyList);
    const NumberedPropertiesList *		isl= &(bd->bdItemShadingList);

    /*  1  */
    fontSize= ta->taFontSizeHalfPoints;
    fontSize= ( 6* ta->taFontSizeHalfPoints )/ 5;

    df= docFontListGetFontByNumber( dfl, ta->taFontNumber );
    if  ( ! df )
	{ LXDEB(ta->taFontNumber,df); return;	}

    sprintf( scratch, "span.t%d, div.t%d", n, n );
    docHtmlPutString( scratch, hwc );
    docHtmlNewLine( hwc );
    docHtmlPutString( "  {", hwc );
    docHtmlNewLine( hwc );

    if  ( ta->taTextColorNumber > 0 )
	{
	const RGB8Color *	rgb8= dp->dpColors+ ta->taTextColorNumber;

	sprintf( scratch, "#%02x%02x%02x",
					rgb8->rgb8Red,
					rgb8->rgb8Green,
					rgb8->rgb8Blue );

	docHtmlPutString( "  color: ", hwc );
	docHtmlPutString( scratch, hwc );
	docHtmlPutString( ";", hwc );
	docHtmlNewLine( hwc );
	}

    size= docHtmlFontFamilyIndicator( scratch, sizeof(scratch)- 1,
						df->dfStyleInt, df->dfName );
    if  ( size < 0 )
	{ LDEB(size);	}
    if  ( size > 0 )
	{
	docHtmlPutString( "  font-family: ", hwc );
	docHtmlPutString( scratch, hwc );
	docHtmlPutString( ";", hwc );
	docHtmlNewLine( hwc );
	}

    if  ( ta->taFontIsBold )
	{
	docHtmlPutString( "  font-weight: bold", hwc );
	docHtmlPutString( ";", hwc );
	docHtmlNewLine( hwc );
	}

    if  ( ta->taFontIsSlanted )
	{
	docHtmlPutString( "  font-style: italic", hwc );
	docHtmlPutString( ";", hwc );
	docHtmlNewLine( hwc );
	}

    if  ( ta->taTextIsUnderlined )
	{
	docHtmlPutString( "  text-decoration: underline", hwc );
	docHtmlPutString( ";", hwc );
	docHtmlNewLine( hwc );
	}

    if  ( ta->taHasStrikethrough )
	{
	docHtmlPutString( "  text-decoration: line-through", hwc );
	docHtmlPutString( ";", hwc );
	docHtmlNewLine( hwc );
	}

    if  ( ta->taSmallCaps )
	{
	docHtmlPutString( "  font-variant: small-caps", hwc );
	docHtmlPutString( ";", hwc );
	docHtmlNewLine( hwc );
	}

    if  ( ta->taSuperSub == DOCfontSUPERSCRIPT )
	{
	docHtmlPutString( "  vertical-align: super", hwc );
	docHtmlPutString( ";", hwc );
	docHtmlNewLine( hwc );

	fontSize= SUPERSUB_SIZE( fontSize );
	}

    if  ( ta->taSuperSub == DOCfontSUBSCRIPT )
	{
	docHtmlPutString( "  vertical-align: sub", hwc );
	docHtmlPutString( ";", hwc );
	docHtmlNewLine( hwc );

	fontSize= SUPERSUB_SIZE( fontSize );
	}

    if  ( docBorderNumberIsBorder( bpl, ta->taBorderNumber ) )
	{
	docHtmlEmitBorderStyleByNumber( "border", ta->taBorderNumber, hwc );
	}

    if  ( docShadingNumberIsShading( isl, ta->taShadingNumber ) )
	{
	ItemShading		is;

	docGetItemShadingByNumber( &is, isl, ta->taShadingNumber );

	if  ( docHtmlUseBackgroundStyle( &is, hwc ) )
	    { docHtmlEmitBackgroundStyle( &is, hwc );	}
	}

    sprintf( scratch, "%d%spt", fontSize/ 2, fontSize % 2?".5":"" );
    docHtmlPutString( "  font-size: ", hwc );
    docHtmlPutString( scratch, hwc );
    docHtmlPutString( ";", hwc );
    docHtmlNewLine( hwc );

    docHtmlPutString( "  }", hwc );
    docHtmlNewLine( hwc );

    return;
    }

static void docHtmlSaveTextAttributeStyles(	HtmlWritingContext *	hwc )
    {
    const BufferDocument *	bd= hwc->hwcBd;
    const DocumentProperties *	dp= &(bd->bdProperties);
    const DocumentGeometry *	dg= &(dp->dpGeometry);

    char			scratch[40];

    docHtmlPutString( "<style type=\"text/css\">", hwc );
    docHtmlNewLine( hwc );

    docHtmlPutString( "body.ted", hwc ); docHtmlNewLine( hwc );
    docHtmlPutString( "  {", hwc ); docHtmlNewLine( hwc );
    docHtmlPutString( "  background-color: #ffffff;", hwc );
    docHtmlNewLine( hwc );
    docHtmlPutString( "  color: #000000;", hwc );
    docHtmlNewLine( hwc );

    if  ( dg->dgTopMarginTwips > 300 )
	{
	sprintf( scratch, "margin-top: %dpt;", dg->dgTopMarginTwips/ 30 );
	docHtmlPutString( scratch, hwc );
	docHtmlNewLine( hwc );
	}

    if  ( dg->dgLeftMarginTwips > 300 )
	{
	sprintf( scratch, "margin-left: %dpt;", dg->dgLeftMarginTwips/ 30 );
	docHtmlPutString( scratch, hwc );
	docHtmlNewLine( hwc );
	}

    if  ( dg->dgRightMarginTwips > 300 )
	{
	sprintf( scratch, "margin-right: %dpt;", dg->dgRightMarginTwips/ 30 );
	docHtmlPutString( scratch, hwc );
	docHtmlNewLine( hwc );
	}

    if  ( dg->dgBottomMarginTwips > 300 )
	{
	sprintf( scratch, "margin-bottom: %dpt;", dg->dgBottomMarginTwips/ 30 );
	docHtmlPutString( scratch, hwc );
	docHtmlNewLine( hwc );
	}

    docHtmlPutString( "  }", hwc );
    docHtmlNewLine( hwc );
    docHtmlNewLine( hwc );

    utilForAllTextAttributes( &(bd->bdTextAttributeList),
			    docHtmlSaveTextAttributeStyle, (void *)hwc );

    docHtmlPutString( "</style>", hwc );
    docHtmlNewLine( hwc );

    return;
    }

/************************************************************************/
/*									*/
/*  Save a document to HTML, or to a html mail mime aggregate.		*/
/*									*/
/************************************************************************/

int docHtmlSaveDocument(	SimpleOutputStream *	sos,
				BufferDocument *	bd,
				int			asMimeAggr,
				const char *		mimeBoundary,
				const char *		filename )
    {
    HtmlWritingContext		hwc;

    const BufferItem *		bodyBi= bd->bdBody.eiRoot;
    const DocumentProperties *	dp= &(bd->bdProperties);

    docInitHtmlWritingContext( &hwc );

    hwc.hwcSos= sos;
    hwc.hwcBd= bd;
    hwc.hwcAsMimeAggregate= asMimeAggr;
    hwc.hwcMimeBoundary= mimeBoundary;

    if  ( filename )
	{
	const char *	s;

	hwc.hwcFilename= filename;
	s= strrchr( filename, '.' );
	if  ( s )
	    { hwc.hwcBaselength= s- filename;		}
	else{ hwc.hwcBaselength= strlen( filename );	}

	s= strrchr( filename, '/' );
	if  ( s )
	    { hwc.hwcRelativeOffset= s- filename+ 1;	}
	else{ hwc.hwcRelativeOffset= 0;			}
	}

    if  ( hwc.hwcAsMimeAggregate )
	{
	if  ( appMakeUniqueString( hwc.hwcContentIdTail, IDlenTAIL ) )
	    { LDEB(IDlenTAIL); return -1;	}

	sioOutPutByte( '-', sos ); sioOutPutByte( '-', sos );
	sioOutPutString( hwc.hwcMimeBoundary, sos );
	sioOutPutByte( '\r', sos ); sioOutPutByte( '\n', sos );
	sioOutPutString( "Content-Type: text/html", sos );
	sioOutPutByte( '\r', sos ); sioOutPutByte( '\n', sos );
	sioOutPutString( "Content-Transfer-Encoding: 8bit", sos );
	sioOutPutByte( '\r', sos ); sioOutPutByte( '\n', sos );

	sioOutPutByte( '\r', sos ); sioOutPutByte( '\n', sos );
	}

    docHtmlPutString( "<!DOCTYPE HTML PUBLIC \"-//W3C//DTD HTML 4.01//EN\">",
									&hwc );
    docHtmlNewLine( &hwc );
    docHtmlPutString( "<HTML>", &hwc );
    docHtmlNewLine( &hwc );

    docHtmlPutString( "<HEAD>", &hwc );
    docHtmlNewLine( &hwc );
    docHtmlPutString( "<meta http-equiv=\"Content-Type\" content=\"text/html; charset=UTF-8\">",
									&hwc );
    docHtmlNewLine( &hwc );

    if  ( dp->dpTitle )
	{
	docHtmlPutString( "<TITLE>", &hwc );
	docHtmlEscapeString( dp->dpTitle, &hwc );
	docHtmlPutString( "</TITLE>", &hwc );
	docHtmlNewLine( &hwc );
	}

    if  ( dp->dpSubject )
	{
	docHtmlPutString(
		    "<META NAME=\"description\" CONTENT=\"", &hwc );
	docHtmlEscapeString( dp->dpSubject, &hwc );
	docHtmlPutString( "\">", &hwc );
	docHtmlNewLine( &hwc );
	}

    if  ( dp->dpKeywords )
	{
	docHtmlPutString(
		    "<META NAME=\"keywords\" CONTENT=\"", &hwc );
	docHtmlEscapeString( dp->dpKeywords, &hwc );
	docHtmlPutString( "\">", &hwc );
	docHtmlNewLine( &hwc );
	}

    if  ( dp->dpDoccomm )
	{
	docHtmlPutString(
		    "<META NAME=\"comment\" CONTENT=\"", &hwc );
	docHtmlEscapeString( dp->dpDoccomm, &hwc );
	docHtmlPutString( "\">", &hwc );
	docHtmlNewLine( &hwc );
	}

    if  ( dp->dpAuthor )
	{
	docHtmlPutString(
		    "<META NAME=\"author\" CONTENT=\"", &hwc );
	docHtmlEscapeString( dp->dpAuthor, &hwc );
	docHtmlPutString( "\">", &hwc );
	docHtmlNewLine( &hwc );
	}

    if  ( dp->dpHlinkbase )
	{
	docHtmlPutString( "<BASE HREF=\"", &hwc );
	docHtmlEscapeString( dp->dpHlinkbase, &hwc );
	docHtmlPutString( "\">", &hwc );
	docHtmlNewLine( &hwc );
	}

    docHtmlSaveTextAttributeStyles( &hwc );

    docHtmlPutString( "</HEAD>", &hwc );
    docHtmlNewLine( &hwc );

    docHtmlPutString( "<body class=\"ted\" >", &hwc );
    docHtmlNewLine( &hwc );

    if  ( docHtmlSaveItem( bodyBi, bodyBi->biChildren[0], &hwc ) )
	{ LDEB(bodyBi->biLevel); return -1; }

    if  ( hwc.hwcNoteRefCount > 0 )
	{
	DocumentField *	dfNote;
	DocumentNote *	dn;

	docHtmlPutString( "<HR>", &hwc );

	for ( dfNote= docGetFirstNoteOfDocument( &dn, bd, -1 );
	      dfNote;
	      dfNote= docGetNextNoteInDocument( &dn, bd, dfNote, -1 ) )
	    {
	    DocumentTree *	ei;

	    ei= &(dn->dnDocumentTree);
	    if  ( ! ei->eiRoot )
		{ XDEB(ei->eiRoot); continue;	}

	    if  ( docHtmlSaveItem( ei->eiRoot, (const BufferItem *)0, &hwc ) )
		{ XDEB(ei->eiRoot); return -1; }
	    }

	if  ( hwc.hwcNoteDefCount != hwc.hwcNoteRefCount )
	    { LLDEB(hwc.hwcNoteDefCount,hwc.hwcNoteRefCount);	}
	}

    docHtmlPutString( "</body></HTML>", &hwc );
    docHtmlNewLine( &hwc );

    if  ( ! hwc.hwcAsMimeAggregate	&&
	    hwc.hwcImageCount > 0	)
	{
	strncpy( hwc.hwcNameScratch, hwc.hwcFilename, hwc.hwcBaselength );
	strcpy( hwc.hwcNameScratch+ hwc.hwcBaselength, ".img" );

	if  ( appTestDirectory( hwc.hwcNameScratch )	&&
	      appMakeDirectory( hwc.hwcNameScratch )	)
	    { SDEB(hwc.hwcNameScratch); return -1;	}
	}

    if  ( hwc.hwcImageCount > 0			&&
	  docHtmlSaveImages( bodyBi, &hwc )	)
	{ LDEB(hwc.hwcImageCount); return -1;	}

    if  ( hwc.hwcAsMimeAggregate )
	{
	sioOutPutByte( '-', sos ); sioOutPutByte( '-', sos );
	sioOutPutString( hwc.hwcMimeBoundary, sos );
	sioOutPutByte( '-', sos ); sioOutPutByte( '-', sos );
	sioOutPutByte( '\r', sos ); sioOutPutByte( '\n', sos );
	}

    docCleanHtmlWritingContext( &hwc );

    return 0;
    }
